/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  flow_control_example.c
*
* @purpose   OpEN FlowControl example.
*
* @component OpEN
*
* @create    06/08/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_flow_control.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the FlowControl Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the Flow Control mode of the system: %s 1 \n", name);
  printf("Test 2: Set the Flow Control mode of the system: %s 2 <mode>\n", name);
  printf("Test 3: Get the Flow Control mode of the interafce: %s 3 <ifNum>\n", name);
  printf("Test 4: Set the Flow Control mode of the interafce: %s 4 <ifNum> <mode>\n", name);
  printf("Test 5: Get the Flow Control operational mode of the interafce: %s 5 <ifNum>\n", name);
  printf("Test 6: Get the pause frames receive count of the interafce: %s 6 <ifNum> \n", name);
  printf("Test 7: Get the pause frames transmit count of the interafce: %s 7 <ifNum> \n", name);

  return;
}

/*********************************************************************
* @purpose  Get the Flow Control mode of the system.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    mode           @b{(output)} Flow control mode
*
* @returns  none
*
* @end
*********************************************************************/
void flowControlModeGet(openapiClientHandle_t *client_handle, 
                        OPEN_FLOWCONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiFlowControlModeGet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Flow Control mode of the system."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Flow Control mode of the system is %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Set the Flow Control mode of the system.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    mode          @b{(input)}  Flow control mode
*
* @returns  none
*
* @end
*********************************************************************/
void flowControlModeSet(openapiClientHandle_t *client_handle, 
                        OPEN_FLOWCONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiFlowControlModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Flow Control mode of the system."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Flow Control mode of the system set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the Flow Control mode of the interafce.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
* @param    mode            @b{(output)} Flow control mode
*
* @returns  none
*
* @end
*********************************************************************/
void flowControlIfModeGet(openapiClientHandle_t *client_handle, 
                          uint32_t ifNum, OPEN_FLOWCONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiFlowControlIfModeGet(client_handle, ifNum, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Flow Control mode of the "
            "interafce. (result = %d)\n", result);
  }
  else
  {
    printf("Flow Control mode of the interface is %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Set the Flow Control mode of the interafce.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    mode           @b{(input)}  Flow control mode
*
* @returns  none
*
* @end
*********************************************************************/
void flowControlIfModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                          OPEN_FLOWCONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiFlowControlIfModeSet(client_handle, ifNum, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Flow Control mode of the "
           "interafce. (result = %d)\n", result);
  }
  else
  {
    printf("Flow Control mode of the interface set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the Flow Control operational mode of the interafce.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
* @param    mode            @b{(output)} Flow control operational mode
*
* @returns  none
*
* @end
*********************************************************************/
void flowControlIfOperModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                              OPEN_FLOWCONTROL_OPER_t *mode)
{
  open_error_t result;

  if ((result = openapiFlowControlIfOperModeGet(client_handle, ifNum, mode)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Flow Control operational mode of"
           " the interafce. (result = %d)\n", result);
  }
  else
  {
    printf("Flow Control operational mode of the interface is %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Get the pause frames receive count of the interafce.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface
* @param    rxCount          @b{(output)} Pause frames receive count
*
* @returns  none
*
* @end
*********************************************************************/
void flowControlIfPauseFramesRxGet(openapiClientHandle_t *client_handle, 
                                   uint32_t ifNum, uint32_t *rxCount)
{
  open_error_t result;

  if ((result = openapiFlowControlIfPauseFramesRxGet(client_handle, ifNum, rxCount))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the pause frames receive count of the "
           "interafce. (result = %d)\n", result);
  }
  else
  {
    printf("Pause frames receive count of the interafce is %u.\n", *rxCount);
  }
  return;
}

/*********************************************************************
* @purpose  Get the pause frames transmit count of the interafce.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface
* @param    txCount          @b{(output)} Pause frames transmit count
*
* @returns  none
*
* @end
*********************************************************************/
void flowControlIfPauseFramesTxGet(openapiClientHandle_t *client_handle, 
                                   uint32_t ifNum, uint32_t *txCount)
{
  open_error_t result;

  if ((result = openapiFlowControlIfPauseFramesTxGet(client_handle, ifNum, txCount))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the pause frames transmit count of the"
           " interafce. (result = %d)\n", result);
  }
  else
  {
    printf("Pause frames transmit count of the interafce is %u.\n", *txCount);
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for FlowControl
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum, ifNum, rxCount, txCount;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_FLOWCONTROL_t mode;
  OPEN_FLOWCONTROL_OPER_t operMode;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);
  if (argc > 2)
  {
    ifNum   = atoi(argv[2]);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("FlowControl example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting FlowControl API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 2)
      {
        flowControlModeGet(&client_handle, &mode);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        flowControlModeSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        flowControlIfModeGet(&client_handle, ifNum, &mode);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 4)
      {
        flowControlIfModeSet(&client_handle, ifNum, atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        flowControlIfOperModeGet(&client_handle, ifNum, &operMode);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        flowControlIfPauseFramesRxGet(&client_handle, ifNum, &rxCount);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        flowControlIfPauseFramesTxGet(&client_handle, ifNum, &txCount);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping FlowControl API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}


