/*********************************************************************
*
*  Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  green_ethernet_example.c
*
* @purpose   Green ethernet configuration OpEN APIs example 
*
* @component OpEN
*
* @comments
*
* @create    05/19/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_system.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the green ethernet Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printGreenEthernetAppMenu()
{
  printf("Usage:  green_ethernet_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: testGreenEthrIsValidIntfType: green_ethernet_example 1 <interface>\n");
  printf("Test 2: testGreenEthrValidIntfFirstGet: green_ethernet_example 2\n");
  printf("Test 3: testGreenEthrValidIntfNextGet: green_ethernet_example 3 <interface>\n");
  printf("Test 4: testGreenEthrIsIntfEEECapable: green_ethernet_example 4 <interface>\n");
  printf("Test 5: testGreenEthrEnergyDetectModeSet: green_ethernet_example 5 <interface> <mode>\n");
  printf("Test 6: testGreenEthrShrtRchAutoModeSet: green_ethernet_example 6 <interface> <mode>\n");
  printf("Test 7: testGreenEthrShrtRchForceModeSet: green_ethernet_example 7 <interface> <mode>\n");
  printf("Test 8: testGreenEthrIsIntfEEECapable: green_ethernet_example 8 <interface>\n");
  printf("Test 9: testGreenEthrEnergyDetectAdminModeGet: green_ethernet_example 9 <interface>\n");
  printf("Test 10: testGreenEthrShrtRchAutoAdminModeGet: green_ethernet_example 10 <interface>\n");
  printf("Test 11: testGreenEthrShrtRchForceAdminModeGet: green_ethernet_example 11 <interface>\n");
  printf("Test 12: testGreenEthrEnergyDetectOprStatusGet: green_ethernet_example 12 <interface>\n");
  printf("Test 13: testGreenEthrShrtRchOprStatusGet: green_ethernet_example 13 <interface>\n");
  printf("Test 14: testGreenEthrGlobalShortReachModeGet: green_ethernet_example 14\n");
  printf("Test 15: testGreenEthrGlobalEnergyDetectModeGet: green_ethernet_example 15\n");
  printf("Test 16: testGreenEthrGlobalShortReachModeSet: green_ethernet_example 16 <mode>\n");
  printf("Test 17: testGreenEthrGlobalEnergyDetectModeSet: green_ethernet_example 17 <mode>\n");

  return;
}



/*********************************************************************
* @purpose  Determine if the interface is valid 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void testGreenEthrIsValidIntfType(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_BOOL_t valid;

  if ((result = openapiGreenEthrIsValidIntfType(clientHandle, intIfNum, &valid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get valid interface. (result = %d)\n", result);
  }
  else
  {
    printf("Interface valid: %d. (result = %d)\n", valid, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the first interface that is valid for green Ethernet 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void testGreenEthrValidIntfFirstGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t intIfNum;

  if ((result = openapiGreenEthrValidIntfFirstGet(clientHandle, &intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get first interface. (result = %d)\n", result);
  }
  else
  {
    printf("Valid interface: %d. (result = %d)\n", intIfNum, result);
  }

  return;
}


/*********************************************************************
* @purpose  Get the next interface that is valid for green Ethernet
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    preIntIf         @b{(input)} previous internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrValidIntfNextGet(openapiClientHandle_t *clientHandle, uint32_t preIntIf)
{
  open_error_t result;
  uint32_t intIfNum;

  if ((result = openapiGreenEthrValidIntfNextGet(clientHandle, preIntIf, &intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get next interface. (result = %d)\n", result);
  }
  else
  {
    printf("Next interface: %d. (result = %d)\n", intIfNum, result);
  }

  return;
}

/*********************************************************************
* @purpose  Determine if the interface is EEE capable 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void testGreenEthrIsIntfEEECapable(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_BOOL_t valid;

  if ((result = openapiGreenEthrIsIntfEEECapable(clientHandle, intIfNum, &valid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get EEE capable. (result = %d)\n", result);
  }
  else
  {
    printf("EEE capable: %d. (result = %d)\n", valid, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the energy-detect mode 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEnergyDetectModeSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t mode)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if (1 == mode)
  {
    adminMode = OPEN_ENABLE;
  }
  else if (0 == mode)
  {
    adminMode = OPEN_DISABLE;
  }
  else
  {
    printf("Invalid input params\n");
    return;
  }

  if ((result = openapiGreenEthrEnergyDetectModeSet(clientHandle, intIfNum, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set energy-detect mode. (result = %d)\n", result);
  }
  else
  {
    printf("energy-detect mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the short-reach auto mode 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrShrtRchAutoModeSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t mode)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if (1 == mode)
  {
    adminMode = OPEN_ENABLE;
  }
  else if (0 == mode)
  {
    adminMode = OPEN_DISABLE;
  }
  else
  {
    printf("Invalid input params\n");
    return;
  }

  if ((result = openapiGreenEthrShrtRchAutoModeSet(clientHandle, intIfNum, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set short-reach auto mode. (result = %d)\n", result);
  }
  else
  {
    printf("short-reach auto mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}
/*********************************************************************
* @purpose  Set the short-reach force mode 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrShrtRchForceModeSet(openapiClientHandle_t *clientHandle, uint32_t intIfNum, uint32_t mode)
{
  open_error_t result;
  OPEN_BOOL_t adminMode;

  if (1 == mode)
  {
    adminMode = OPEN_TRUE;
  }
  else if (0 == mode)
  {
    adminMode = OPEN_FALSE;
  }
  else
  {
    printf("Invalid input params\n");
    return;
  }

  if ((result = openapiGreenEthrShrtRchForceModeSet(clientHandle, intIfNum, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set short-reach force mode. (result = %d)\n", result);
  }
  else
  {
    printf("short-reach force mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the energy detect mode 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEnergyDetectAdminModeGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if ((result = openapiGreenEthrEnergyDetectAdminModeGet(clientHandle, intIfNum, &adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get energy detect mode. (result = %d)\n", result);
  }
  else
  {
    printf("Energy detect administrative mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}
/*********************************************************************
* @purpose  Get the short-reach auto mode 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrShrtRchAutoAdminModeGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_BOOL_t adminMode;

  if ((result = openapiGreenEthrShrtRchAutoAdminModeGet(clientHandle, intIfNum, &adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to short-reach auto mode. (result = %d)\n", result);
  }
  else
  {
    printf("short-reach auto mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}
/*********************************************************************
* @purpose  Get the short-reach force mode 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrShrtRchForceAdminModeGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_BOOL_t adminMode;

  if ((result = openapiGreenEthrShrtRchForceAdminModeGet(clientHandle, intIfNum, &adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to short-reach force mode. (result = %d)\n", result);
  }
  else
  {
    printf("short-reach force mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}
/*********************************************************************
* @purpose  Get the energy-detect operational status 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEnergyDetectOprStatusGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_BOOL_t adminMode;
  OPEN_GREEN_ETH_OPER_STATUS_t reason;

  if ((result = openapiGreenEthrEnergyDetectOprStatusGet(clientHandle, intIfNum, &adminMode, &reason)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to energy-detect operational status. (result = %d)\n", result);
  }
  else
  {
    printf("energy-detect operational status: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the short-reach operational status 
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrShrtRchOprStatusGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  OPEN_BOOL_t adminMode;
  OPEN_GREEN_ETH_OPER_STATUS_t reason;

  if ((result = openapiGreenEthrShrtRchOprStatusGet(clientHandle, intIfNum, &adminMode, &reason)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to short-reach operational status. (result = %d)\n", result);
  }
  else
  {
    printf("short-reach operational status: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the green ethernet global short reach mode 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void testGreenEthrGlobalShortReachModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if ((result = openapiGreenEthrGlobalShortReachModeGet(clientHandle, &adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global short reach mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global short reach mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the green ethernet global short reach mode 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    adminMode        @b{(output)}  admin mode
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void testGreenEthrGlobalShortReachModeSet(openapiClientHandle_t *clientHandle, uint32_t mode)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if (1 == mode)
  {
    adminMode = OPEN_ENABLE;
  }
  else if (0 == mode)
  {
    adminMode = OPEN_DISABLE;
  }
  else
  {
    printf("Invalid input params\n");
    return;
  }

  if ((result = openapiGreenEthrGlobalShortReachModeSet(clientHandle, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global short reach mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global short reach mode set: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the green ethernet global energy detect mode 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void testGreenEthrGlobalEnergyDetectModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if ((result = openapiGreenEthrGlobalEnergyDetectModeGet(clientHandle, &adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global energy detect mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global energy detect mode: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the green ethernet global energy detect mode 
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    adminMode        @b{(output)}  admin mode
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void testGreenEthrGlobalEnergyDetectModeSet(openapiClientHandle_t *clientHandle, uint32_t mode)
{
  open_error_t result;
  OPEN_CONTROL_t adminMode;

  if (1 == mode)
  {
    adminMode = OPEN_ENABLE;
  }
  else if (0 == mode)
  {
    adminMode = OPEN_DISABLE;
  }
  else
  {
    printf("Invalid input params\n");
    return;
  }
  if ((result = openapiGreenEthrGlobalEnergyDetectModeSet(clientHandle, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set energy detect mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global energy detect mode set: %d. (result = %d)\n", adminMode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Sanity of green ethernet OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note
* 
* @end
*********************************************************************/
void greenEthernetOpENAPISanity(openapiClientHandle_t *clientHandle)
{
  return;
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  open_revision_data_t openApiVersion;
  char switch_os_revision_string[OPENAPI_OS_REVISION_STR_SIZE];
  uint32_t arg1, arg2;

  if (argc < 2)
  {
    printGreenEthernetAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("green_ethernet_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting green ethernet API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("Open API Version = %d.%d.%d.%d\n", 
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("Open API Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrIsValidIntfType(&clientHandle, arg1);
      break;
    case 2:
      if (argc != 2)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      testGreenEthrValidIntfFirstGet(&clientHandle);
      break;
    case 3:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrValidIntfNextGet(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrIsIntfEEECapable(&clientHandle, arg1);
      break;
    case 5:
      if (argc != 4)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      testGreenEthrEnergyDetectModeSet(&clientHandle, arg1, arg2);
      break;
    case 6:
      if (argc != 4)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      testGreenEthrShrtRchAutoModeSet(&clientHandle, arg1, arg2);
      break;
    case 7:
      if (argc != 4)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      testGreenEthrShrtRchForceModeSet(&clientHandle, arg1, arg2);
      break;
    case 8:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrIsIntfEEECapable(&clientHandle, arg1);
      break;
    case 9:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEnergyDetectAdminModeGet(&clientHandle, arg1);
      break;
    case 10:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrShrtRchAutoAdminModeGet(&clientHandle, arg1);
      break;
    case 11:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrShrtRchForceAdminModeGet(&clientHandle, arg1);
      break;
    case 12:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEnergyDetectOprStatusGet(&clientHandle, arg1);
      break;
    case 13:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrShrtRchOprStatusGet(&clientHandle, arg1);
      break;
    case 14:
      if (argc != 2)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      testGreenEthrGlobalShortReachModeGet(&clientHandle);
      break;
    case 15:
      if (argc != 2)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      testGreenEthrGlobalEnergyDetectModeGet(&clientHandle);
      break;
    case 16:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrGlobalShortReachModeSet(&clientHandle, arg1);
      break;
    case 17:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrGlobalEnergyDetectModeSet(&clientHandle, arg1);
      break;

    default:
      printGreenEthernetAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping green ethernet API example application");
        
  (void) openapiClientTearDown(&clientHandle);          
  return 0;
}
