/*********************************************************************
*
*  Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  green_pwr_sav_history_example.c
*
* @purpose   Green ethernet configuration OpEN APIs example 
*
* @component OpEN
*
* @comments
*
* @create    05/19/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_system.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the green pwr save history Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printGreenEthernetAppMenu()
{
  printf("Usage:  green_ethernet_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: testGreenEthrEeeLpiHistoryIntfSampleIntervalSet: green_pwr_sav_history_example 1 <interval>\n");
  printf("Test 2: testGreenEthrEeeLpiHistoryIntfSampleIntervalGet: green_pwr_sav_history_example 2\n");
  printf("Test 3: testGreenEthrEeeLpiHistoryIntfMaxSamplesSet: green_pwr_sav_history_example 3 <samples>\n");
  printf("Test 4: testGreenEthrEeeLpiHistoryIntfMaxSamplesGet: green_pwr_sav_history_example 4\n");
  printf("Test 5: testGreenEthrPwrSavHistoryIntfStatisticsClear: green_pwr_sav_history_example 5 <interface>\n");
  printf("Test 6: testx2GreenEthrPwrSavHistoryIntfEntriesGet: green_pwr_sav_history_example 6 <interface>\n");

  return;
}

/*********************************************************************
* @purpose Set Green Mode Power Saving History Sampling Interval
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    sampleInterval  @b{(input)}  Sampling Interval
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeLpiHistoryIntfSampleIntervalSet(openapiClientHandle_t *clientHandle, uint32_t sampleInterval)
{
  open_error_t result;

  if ((result = openapiGreenEthrEeeLpiHistoryIntfSampleIntervalSet(clientHandle, sampleInterval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set green mode power saving history sampling interval. (result = %d)\n", result);
  }
  else
  {
    printf("Set Green Mode Power Saving History Sampling Interval. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose Get green mode power saving history sampling interval
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeLpiHistoryIntfSampleIntervalGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t sampleInterval;

  if ((result = openapiGreenEthrEeeLpiHistoryIntfSampleIntervalGet(clientHandle, &sampleInterval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get green mode power saving history sampling interval. (result = %d)\n", result);
  }
  else
  {
    printf("Get green mode power saving history sampling interval: %d. (result = %d)\n",sampleInterval, result);
  }

  return;
}

/*********************************************************************
* @purpose Set green mode power saving history max samples to keep
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    maxSamples       @b{(input)} maxSamples
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeLpiHistoryIntfMaxSamplesSet(openapiClientHandle_t *clientHandle, uint32_t maxSamples)
{
  open_error_t result;

  if ((result = openapiGreenEthrEeeLpiHistoryIntfMaxSamplesSet(clientHandle, maxSamples)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set green mode power saving history max samples to keep. (result = %d)\n", result);
  }
  else
  {
    printf("Set green mode power saving history max samples to keep. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose Get green mode power saving history max samples to keep
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrEeeLpiHistoryIntfMaxSamplesGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t maxSamples;

  if ((result = openapiGreenEthrEeeLpiHistoryIntfMaxSamplesGet(clientHandle, &maxSamples)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get green mode power saving history max samples to keep. (result = %d)\n", result);
  }
  else
  {
    printf("Get green mode power saving history max samples to keep: %d (result = %d)\n", maxSamples, result);
  }

  return;
}

/*********************************************************************
* @purpose Clear Green Mode Power Saving History Statistics for the given interface
* 
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
* 
* @returns  none
*   
* @
* 
* @end
*********************************************************************/
void testGreenEthrPwrSavHistoryIntfStatisticsClear(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;

  if ((result = openapiGreenEthrPwrSavHistoryIntfStatisticsClear(clientHandle, intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear Green Mode Power Saving History Statistics. (result = %d)\n", result);
  }
  else
  {
    printf("Clear Green Mode Power Saving History Statistics. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get valid entries of green mode power savings history
*           on given interface - latest sample first
*
* @param    clientHandle     @b{(input)} client handle from registration API
* @param    intIfNum         @b{(input)} internal interface number
*
* @returns  none
*
* @notes    To get the time Since the sample was recorded, subtract the entry->sampleTime from 
*           the system up time
*
* @end
*********************************************************************/
void testx2GreenEthrPwrSavHistoryIntfEntriesGet(openapiClientHandle_t *clientHandle, uint32_t intIfNum)
{
  open_error_t result;
  openGreenPwrHistoryIntfEntry_t entry;

  if ((result = openapix2GreenEthrPwrSavHistoryIntfFirstEntryGet(clientHandle, intIfNum, &entry)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get first valid entry of green mode power savings history. (result = %d)\n", result);
  }
  else
  {
    printf("First valid entry of green mode power savings history:\n");
    printf("sampleNo %d\n", entry.sampleNo);
    printf("sampleTime %d\n", entry.sampleTime);
    printf("percentLpiTime %d\n", entry.percentLpiTime);
    printf("percentLpiTimeTotal %d\n", entry.percentLpiTimeTotal);
    printf("nextEntry %p\n", entry.nextEntry);

    while (1)
    {
      if (entry.nextEntry)
      {
        if ((result = openapix2GreenEthrPwrSavHistoryIntfNextEntryGet(clientHandle, &entry)) != OPEN_E_NONE)
        {
          printf("Bad return code trying to get next valid entry of green mode power savings history. (result = %d)\n", result);
        }
        else
        {
          printf("\nNext valid entry of green mode power savings history:\n");
          printf("sampleNo %d\n", entry.sampleNo);
          printf("sampleTime %d\n", entry.sampleTime);
          printf("percentLpiTime %d\n", entry.percentLpiTime);
          printf("percentLpiTimeTotal %d\n", entry.percentLpiTimeTotal);
          printf("nextEntry %p\n", entry.nextEntry);
        }
      }
      else
      {
        printf("No more entries\n");
        break;
      }
    }
  }
  return;
}

/*********************************************************************
* @purpose  Sanity of green pwr save history OpEN APIs
*
* @param    clientHandle     @b{(input)}   Client handle from registration API
* 
* @returns  none
* 
* @note
* 
* @end
*********************************************************************/
void pwrSaveOpENAPISanity(openapiClientHandle_t *clientHandle)
{

  return;
}


/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  open_revision_data_t openApiVersion;
  char switch_os_revision_string[OPENAPI_OS_REVISION_STR_SIZE];
  uint32_t arg1;

  if (argc < 2)
  {
    printGreenEthernetAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("green_ethernet_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting green pwr save history API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("Open API Version = %d.%d.%d.%d\n", 
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("Open API Version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeLpiHistoryIntfSampleIntervalSet(&clientHandle, arg1);
      break;
    case 2:
      if (argc != 2)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      testGreenEthrEeeLpiHistoryIntfSampleIntervalGet(&clientHandle);
      break;
    case 3:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrEeeLpiHistoryIntfMaxSamplesSet(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 2)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      testGreenEthrEeeLpiHistoryIntfMaxSamplesGet(&clientHandle);
      break;
    case 5:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testGreenEthrPwrSavHistoryIntfStatisticsClear(&clientHandle, arg1);
      break;
    case 6:
      if (argc != 3)
      {
        printGreenEthernetAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      testx2GreenEthrPwrSavHistoryIntfEntriesGet(&clientHandle, arg1);
      break;

    default:
      printGreenEthernetAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping green ethernet API example application");
        
  (void) openapiClientTearDown(&clientHandle);          
  return 0;
}
