/*********************************************************************
*
*  Copyright 2024 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  interface_app_example.c
*
* @purpose   OpEN interface_app example.
*
* @component OpEN
*
* @create    02/20/2024
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_interface_app.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the interface_app Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get interface time range information: %s 1 <intIfNum>\n", name);
  printf("Test 2: Set interface time range information: %s 2 <intIfNum><timeRange>\n", name);
  printf("Test 3: Run API sanity checks: %s 3 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *client_handle)
{
  open_error_t result;

  uint32_t intIfNum=1;
  char buffer[128];
  open_buffdesc buf;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;
  open_buffdesc badBufdescLen;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  badBufdescLen.pstart = buffer;
  badBufdescLen.size = sizeof(buffer);
  buf.pstart = buffer;
  buf.size = 32;
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  printf("Testing interface_app OpEN APIs sanity:\n");
  printf("Testing openapiInterfaceAppTimeRangeGet():\n");
  result = openapiInterfaceAppTimeRangeGet(NULL, intIfNum, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiInterfaceAppTimeRangeGet(client_handle, 0, NULL);
  printf("Incorrect argument 2:(result = %d)\n", result);
  result = openapiInterfaceAppTimeRangeGet(client_handle, intIfNum, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiInterfaceAppTimeRangeGet(client_handle, intIfNum, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiInterfaceAppTimeRangeGet(client_handle, intIfNum, &badBufdescLen);
  printf("buffdesc greater size argument 3:(result = %d)\n", result);
  result = openapiInterfaceAppTimeRangeGet(client_handle, intIfNum, &buf);
  printf("All valid arguments :(result = %d)\n", result);

  printf("Testing openapiInterfaceAppTimeRangeSet():\n");
  result = openapiInterfaceAppTimeRangeSet(NULL, intIfNum, &buf);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiInterfaceAppTimeRangeSet(client_handle, 0, NULL);
  printf("Incorrect argument 2:(result = %d)\n", result);
  result = openapiInterfaceAppTimeRangeSet(client_handle, intIfNum, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiInterfaceAppTimeRangeSet(client_handle, intIfNum, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiInterfaceAppTimeRangeSet(client_handle, intIfNum, &badBufdescLen);
  printf("buffdesc incorrect size argument 3:(result = %d)\n", result);
  return;
}


/*********************************************************************
* @purpose  Get interface time range information.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    pTimeRange     @b{(output)} Time range name

*
* @returns  none
*
* @end
*********************************************************************/
void
interfaceAppTimeRangeGet(openapiClientHandle_t *client_handle,
                         uint32_t intIfNum, open_buffdesc *pTimeRange)
{
  open_error_t result;
  if (OPEN_E_NONE != (result = openapiInterfaceAppTimeRangeGet(client_handle, 
          intIfNum, pTimeRange)))
  {
    printf("Bad return code trying to get interface time range information. "
        "(result = %d)\n", result);
  }
  else
  {
    printf("Time range information for interface:%d is %s.\n",
        intIfNum,(char*)pTimeRange->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Set interface time range information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    pTimeRange       @b{(input)}  Time range name

*
* @returns  none
*
* @end
*********************************************************************/
void 
interfaceAppTimeRangeSet(openapiClientHandle_t *client_handle,
                         uint32_t intIfNum, open_buffdesc *pTimeRange)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiInterfaceAppTimeRangeSet(client_handle,
          intIfNum, pTimeRange)))
  {
    printf("Bad return code trying to set interface time range information. "
        "(result = %d)\n", result);
  }
  else
  {
    printf("Time range information %s is successfully set for interface:%d.\n",
        (char*)pTimeRange->pstart, intIfNum);
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for interface_app
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  char buffer[128];
  open_buffdesc timeRange;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("interface_app example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting interface_app API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (3 == argc)
      {
        memset(buffer, 0, sizeof(buffer));
        timeRange.pstart = &buffer; 
        timeRange.size = sizeof(buffer);
        interfaceAppTimeRangeGet(&client_handle, atoi(argv[2]), &timeRange);
        show_help = 0;
      }
      break;
    case 2:
      if (4 == argc)
      {
        memset(buffer, 0, sizeof(buffer));
        strncpy(buffer, argv[3], strlen(argv[3]));
        timeRange.pstart = &buffer; 
        timeRange.size = strlen(buffer);
        /* To clear time range information, empty string with size as 1 is passed*/
        if(0 == timeRange.size)
        {
          timeRange.size = 1;
        }
        interfaceAppTimeRangeSet(&client_handle, atoi(argv[2]), &timeRange);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 2)
      {
        runSanity(&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping interface_app API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
