/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  iphelper_example.c
*
* @purpose   OpEN IpHelper example.
*
* @component OpEN
*
* @create    02/28/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_iphelper.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the IpHelper Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the Maximum Hop count configured for DHCP relay agent for the given VRF: %s 1 <vrfName>\n", name);
  printf("Test 2: Set the Maximum Hop count for DHCP relay agent for the given VRF: %s 2 <vrfName> <maxHopCount>\n", name);
  printf("Test 3: Get the Minimum Wait time configured for DHCP relay agent for the given VRF: %s 3 <vrfName>\n", name);
  printf("Test 4: Set the Minimum Wait time for DHCP relay agent for the given VRF: %s 4 <vrfName> <minWaitTime>\n", name);
  printf("Test 5: Get the administrative mode of IP-Helper for the given VRF: %s 5 <vrfName>\n", name);
  printf("Test 6: Set the administrative mode of IP-Helper for the given VRF: %s 6 <vrfName> <mode>\n", name);
  printf("Test 7: Get the Relay Agent Circuit ID option mode for the given VRF: %s 7 <vrfName>\n", name);
  printf("Test 8: Set the Relay Agent Circuit ID option mode for the given VRF: %s 8 <vrfName> <circuitIDOptionMode>\n", name);
  printf("Test 9: Get the Relay Agent global option check mode for the given VRF: %s 9 <vrfName>\n", name);
  printf("Test 10: Set the Relay Agent global option check mode for the given VRF: %s 10 <vrfName> <cidOptionCheckMode>\n", name);
  printf("Test 11: Get the IP Helper statistics for the given VRF: %s 11 <vrfName>\n", name);
  printf("Test 12: Get the Relay Agent global Server Override mode for the given VRF: %s 12 <vrfName>\n", name);
  printf("Test 13: Set the Relay Agent global Server Override mode for the given VRF: %s 13 <vrfName> <mode>\n", name);
  printf("Test 14: Get the Relay Agent Server Override mode for the given interface: %s 14 <ifNum>\n", name);
  printf("Test 15: Set the Relay Agent Server Override mode for the given interface: %s 15 <ifNum> <mode>\n", name);
  printf("Test 16: Get the global Source interface of DHCP relay agent for the given VRF: %s 16 <vrfName>\n", name);
  printf("Test 17: Set the global Source interface of DHCP relay agent for the given VRF: %s 17 <vrfName> <ifNum>\n", name);
  printf("Test 18: Clear the global Source interface of DHCP relay agent for the given VRF: %s 18 <vrfName>\n", name);
  printf("Test 19: Get the interface level Source interface for DHCP relay agent: %s 19 <ifNum>\n", name);
  printf("Test 20: Set the interface level Source interface for DHCP relay agent: %s 20 <ifNum> <srcIfNum>\n", name);
  printf("Test 21: Clear the interface level Source interface for DHCP relay agent: %s 21 <ifNum>\n", name);
  printf("Test 22: Remove all IP helper addresses configured on an interface: %s 22 <ifNum>\n", name);
  printf("Test 23: Get the Relay Agent Circuit ID Option mode for the given interface: %s 23 <ifNum>\n", name);
  printf("Test 24: Set the Relay Agent Circuit ID Option mode for the given interface: %s 24 <ifNum> <circuitIDintfOptionMode>\n", name);
  printf("Test 25: Get the Relay Agent Circuit ID option check mode for the given interface: %s 25 <ifNum>\n", name);
  printf("Test 26: Set the Relay Agent Circuit ID option check mode for the given interface: %s 26 <ifNum> <cidOptionCheckMode>\n", name);
  printf("Test 27: Clear IP helper statistics for the given VRF: %s 27 <vrfName>\n", name);
  printf("Test 28: Get the first relay entry along with server's VRF name and interface info: %s 28 <vrfName> <ifNum>\n", name);
  printf("Test 29: Get the next relay entry along with server's VRF name, interface and UDP-port info: %s 29 <vrfName> <ifNum> <udpPort>\n", name);

  return;
}

/*********************************************************************
* @purpose  Get the Maximum Hop count configured for DHCP relay agent for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayMaxHopCountGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t maxHopCount = 0; /* The maximum hop count */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiBootpDhcpRelayMaxHopCountGet(client_handle, &vrfNameBufd, &maxHopCount);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe Maximum Hop count configured for DHCP relay agent for the given VRF is %u.\n", maxHopCount);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the Maximum Hop count configured for DHCP relay agent for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the Maximum Hop count for DHCP relay agent for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
* @param    maxHopCount    @b{(input)}  The maximum hop count
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayMaxHopCountSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t maxHopCount)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiBootpDhcpRelayMaxHopCountSet(client_handle, &vrfNameBufd, maxHopCount);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the Maximum Hop count for DHCP relay agent for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the Maximum Hop count for DHCP relay agent for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the Minimum Wait time configured for DHCP relay agent for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayMinWaitTimeGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t minWaitTime = 0; /* The minimum wait time */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiBootpDhcpRelayMinWaitTimeGet(client_handle, &vrfNameBufd, &minWaitTime);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe Minimum Wait time configured for DHCP relay agent for the given VRF is %u.\n", minWaitTime);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the Minimum Wait time configured for DHCP relay agent for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the Minimum Wait time for DHCP relay agent for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
* @param    minWaitTime    @b{(input)}  The minimum wait time
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayMinWaitTimeSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t minWaitTime)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiBootpDhcpRelayMinWaitTimeSet(client_handle, &vrfNameBufd, minWaitTime);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the Minimum Wait time for DHCP relay agent for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the Minimum Wait time for DHCP relay agent for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the administrative mode of IP-Helper for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ipHelperAdminModeGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t adminMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIpHelperAdminModeGet(client_handle, &vrfNameBufd, &adminMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe administrative mode of IP-Helper for given VRF is %u (1-Enabled, 0-Disabled).\n", adminMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the administrative mode of IP-Helper for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the administrative mode of IP-Helper for the given VRF.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    *vrfName      @b{(input)}  VRF name
* @param    modeVal       @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void ipHelperAdminModeSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t modeVal)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (1 < modeVal)
  {
    printf("\nERROR: Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == modeVal)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiIpHelperAdminModeSet(client_handle, &vrfNameBufd, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the administrative mode of IP-Helper for the given VRF.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the administrative mode of IP-Helper for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the Relay Agent Circuit ID option mode for the given VRF.
*
* @param    client_handle  @b{(input)} Client handle from registration API
* @param    *vrfName       @b{(input)} VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayCircuitIdOptionModeGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t circuitIDOptionMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiBootpDhcpRelayCircuitIdOptionModeGet(client_handle, &vrfNameBufd, &circuitIDOptionMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe Relay Agent Circuit ID option mode for the given VRF is %u (1-Enabled,0-Disabled).\n", circuitIDOptionMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the Relay Agent Circuit ID option mode for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the Relay Agent Circuit ID option mode for the given VRF.
*
* @param    client_handle  @b{(input)} Client handle from registration API
* @param    *vrfName       @b{(input)} VRF name
* @param    modeVal        @b{(input)} OPEN_ENABLE(1) or OPEN_DISABLE(0)
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayCircuitIdOptionModeSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t modeVal)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (1 < modeVal)
  {
    printf("\nERROR: Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == modeVal)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiBootpDhcpRelayCircuitIdOptionModeSet(client_handle, &vrfNameBufd, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the Relay Agent Circuit ID option mode for the given VRF.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the Relay Agent Circuit ID option mode for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the Relay Agent global option check mode for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayCIDOptionCheckModeGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t cidOptionCheckMode = OPEN_DISABLE; /* OPEN_DISABLE/OPEN_ENABLE */
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiBootpDhcpRelayCIDOptionCheckModeGet(client_handle, &vrfNameBufd, &cidOptionCheckMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe Relay Agent global option check mode for the given VRF is %u (1-Enabled, 0-Disabled).\n", cidOptionCheckMode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the Relay Agent global option check mode for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the Relay Agent global option check mode for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
* @param    mode           @b{(input)}  OPEN_ENABLE or OPEN_DISABLE
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayCIDOptionCheckModeSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_CONTROL_t openMode = OPEN_DISABLE;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (1 < mode)
  {
    printf("\nERROR: Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == mode)
  {
    openMode = OPEN_ENABLE;
  }

  result = openapiBootpDhcpRelayCIDOptionCheckModeSet(client_handle, &vrfNameBufd, openMode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the Relay Agent global option check mode for the given VRF.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the Relay Agent global option check mode for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the IP Helper statistics for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ipHelperStatisticsGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  OPEN_IPHELPER_STATS_t ihStats; /* IP helper stats */


  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  memset(&ihStats, 0, sizeof(ihStats));
  result = openapiIpHelperStatisticsGet(client_handle, &vrfNameBufd, &ihStats);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nFollowing are the IP Helper statistics for the given VRF:\n");
         printf("\n\t Number of valid DHCP client messages received is %u.",     ihStats.dhcpClientMsgsReceived);
         printf("\n\t Number of DHCP client packets relayed to a server is %u.", ihStats.dhcpClientMsgsRelayed); 
         printf("\n\t Number of messages received from a DHCP server %u",        ihStats.dhcpServerMsgsReceived);                
         printf("\n\t Number of DHCP server messages relayed to client is %u.",  ihStats.dhcpServerMsgsRelayed);   
         printf("\n\t Number of UDP client messages received (including DHCP) is %u.",        ihStats.udpClientMsgsReceived);
         printf("\n\t Number of UDP client messages relayed (including DHCP) is %u.",         ihStats.udpClientMsgsRelayed);   
         printf("\n\t Number of DHCP client messages arrived with hops > max allowed is %u.", ihStats.tooManyHops);   
         printf("\n\t Number of DHCP client messages arrived with secs field lower than min allowed is %u.",      ihStats.tooEarly);                              
         printf("\n\t Number of DHCP client messages received with giaddr already set to our own address is %u.", ihStats.spoofedGiaddr);
         printf("\n\t Number of packets intercepted whose TTL is <= 1 is %u.",         ihStats.ttlExpired);
         printf("\n\t Number of times server lookup matched a discard entry is %u.\n", ihStats.matchDiscardEntry);   
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the IP Helper statistics for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the Relay Agent global Server Override mode for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayServerOverrideModeGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  OPEN_BOOL_t mode = OPEN_FALSE;  /* OPEN_DISABLE/OPEN_ENABLE */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiBootpDhcpRelayServerOverrideModeGet(client_handle, &vrfNameBufd, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe Relay Agent global Server Override mode for the given VRF is %u (1-Enabled, 0-Disabled).\n", mode);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the Relay Agent global Server Override mode for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the Relay Agent global Server Override mode for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
* @param    modeFlag       @b{(input)}  OPEN_TRUE(1) or OPEN_FALSE(0)
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayServerOverrideModeSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t modeFlag)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_BOOL_t mode = OPEN_FALSE;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  if (1 < modeFlag)
  {
    printf("\nERROR: Invalid mode value. Expected 0(for Disable) or 1(for Enable).\n");
    return;
  }

  if (1 == modeFlag)
  {
    mode = OPEN_TRUE;
  }

  result = openapiBootpDhcpRelayServerOverrideModeSet(client_handle, &vrfNameBufd, mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the Relay Agent global Server Override mode for the given VRF.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the Relay Agent global Server Override mode for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the Relay Agent Server Override mode for the given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayIntfServerOverrideGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_DHCP_RELAY_INTF_VALUE_t mode = OPEN_RELAY_INTF_NOT_CONFIGURED;

  result = openapiBootpDhcpRelayIntfServerOverrideGet(client_handle, ifNum, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe Relay Agent global Server Override mode for the given interface is %u (2-NotConfigured, 1-Enabled, 0-Disabled).\n", mode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the Relay Agent Server Override mode for the given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the Relay Agent Server Override mode for the given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
* @param    mode           @b{(input)}  OPEN_RELAY_INTF_DISABLE or OPEN_RELAY_INTF_ENABLE or OPEN_RELAY_INTF_NOT_CONFIGURED
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayIntfServerOverrideSet(openapiClientHandle_t *client_handle, uint32_t ifNum, OPEN_DHCP_RELAY_INTF_VALUE_t mode)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiBootpDhcpRelayIntfServerOverrideSet(client_handle, ifNum, mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the Relay Agent Server Override mode for the given interface.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the Relay Agent Server Override mode for the given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the global Source interface of DHCP relay agent for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelaySourceInterfaceGet(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t ifNum = 0; /* The source interface */

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiBootpDhcpRelaySourceInterfaceGet(client_handle, &vrfNameBufd, &ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe global Source interface of the DHCP relay agent for the given VRF is %u.\n", ifNum);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the global Source interface of DHCP relay agent for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the global Source interface of DHCP relay agent for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
* @param    ifNum          @b{(input)}  The source interface
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelaySourceInterfaceSet(openapiClientHandle_t *client_handle, char *vrfName, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiBootpDhcpRelaySourceInterfaceSet(client_handle, &vrfNameBufd, ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the global Source interface of DHCP relay agent for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the global Source interface of DHCP relay agent for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Clear the global Source interface of DHCP relay agent for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelaySourceInterfaceClear(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiBootpDhcpRelaySourceInterfaceClear(client_handle, &vrfNameBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccesssfully cleared the global Source interface of DHCP relay agent for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to clear the global Source interface of DHCP relay agent for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the interface level Source interface for DHCP relay agent.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayIntfSourceInterfaceGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  uint32_t srcIfNum = 0; /* The source interface */

  result = openapiBootpDhcpRelayIntfSourceInterfaceGet(client_handle, ifNum, &srcIfNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe interface level source-interface for DHCP relay agent of given interface is %u.\n", srcIfNum);
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to get the interface level Source interface for DHCP relay agent. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the interface level Source interface for DHCP relay agent.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
* @param    srcIfNum       @b{(input)}  The source interface
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayIntfSourceInterfaceSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t srcIfNum)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiBootpDhcpRelayIntfSourceInterfaceSet(client_handle, ifNum, srcIfNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the interface level source-interface for DHCP relay agent of given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to set the interface level Source interface for DHCP relay agent. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Clear the interface level Source interface for DHCP relay agent.
*
* @param    client_handle  @b{(input)} Client handle from registration API
* @param    ifNum          @b{(input)} Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayIntfSourceInterfaceClear(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiBootpDhcpRelayIntfSourceInterfaceClear(client_handle, ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nCleared the interface level source-interface for DHCP relay agent of given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to clear the interface level source-interface for DHCP relay agent. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Remove all IP helper addresses configured on an interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ipHelperAddressesRemove(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiIpHelperAddressesRemove(client_handle, ifNum);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully removed all IP helper addresses configured on given interface.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to remove all IP helper addresses configured on an interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the Relay Agent Circuit ID Option mode for the given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayrtrCIDOptionModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_DHCP_RELAY_INTF_VALUE_t mode = OPEN_RELAY_INTF_NOT_CONFIGURED;

  result = openapiBootpDhcpRelayrtrCIDOptionModeGet(client_handle, ifNum, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe Relay Agent global Circuit ID Option mode for the given interface is %u (2-NotConfigured, 1-Enabled, 0-Disabled).\n", mode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the Relay Agent Circuit ID Option mode for the given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the Relay Agent Circuit ID Option mode for the given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
* @param    mode           @b{(input)}  OPEN_RELAY_INTF_DISABLE(0) or OPEN_RELAY_INTF_ENABLE(1) or OPEN_RELAY_INTF_NOT_CONFIGURED(2)
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayrtrCIDOptionModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiBootpDhcpRelayrtrCIDOptionModeSet(client_handle, ifNum, mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the Relay Agent Circuit ID option mode for the given interface.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the Relay Agent Circuit ID Option mode for the given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the Relay Agent Circuit ID option check mode for the given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayrtrCIDOptionCheckModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum)
{
  open_error_t result = OPEN_E_FAIL;
  OPEN_DHCP_RELAY_INTF_VALUE_t mode = OPEN_RELAY_INTF_NOT_CONFIGURED;

  result = openapiBootpDhcpRelayrtrCIDOptionCheckModeGet(client_handle, ifNum, &mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nThe Relay Agent Circuit ID option check mode for the given interface is %u (2-NotConfigured, 1-Enabled, 0-Disabled).\n", mode);
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to get the Relay Agent Circuit ID option check mode for the given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Set the Relay Agent Circuit ID option check mode for the given interface.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    ifNum          @b{(input)}  Internal interface number
* @param    mode           @b{(input)}  OPEN_RELAY_INTF_DISABLE(0) or OPEN_RELAY_INTF_ENABLE(1) or OPEN_RELAY_INTF_NOT_CONFIGURED(2)
*
* @returns  none
*
* @end
*********************************************************************/
void bootpDhcpRelayrtrCIDOptionCheckModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, uint32_t mode)
{
  open_error_t result = OPEN_E_FAIL;

  result = openapiBootpDhcpRelayrtrCIDOptionCheckModeSet(client_handle, ifNum, mode);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccessfully set the Relay Agent Circuit ID option check mode for the given interface.\n");
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the Relay Agent Circuit ID option check mode for the given interface. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Clear IP helper statistics for the given VRF.
*
* @param    client_handle  @b{(input)}  Client handle from registration API
* @param    *vrfName       @b{(input)}  VRF name
*
* @returns  none
*
* @end
*********************************************************************/
void ipHelperStatisticsClear(openapiClientHandle_t *client_handle, char *vrfName)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("ERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  result = openapiIpHelperStatisticsClear(client_handle, &vrfNameBufd);
  switch(result)
  {
    case OPEN_E_NONE:
         printf("\nSuccesssfully cleared IP helper statistics for the given VRF.\n");
         break;

    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;

    default:
         printf("\nERROR: Bad return code trying to clear IP helper statistics for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the first relay entry along with server's VRF name config
*
* @param    client_handle  @b{(input)} Client handle from registration API
* @param    vrfName        @b{(input)} VRF name
* @param    intIfNum       @b{(input)} internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void ihIpHelperAddressFirst(openapiClientHandle_t *client_handle, char *vrfName, uint32_t intIfNum)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint16_t udpPort = 0;
  uint32_t serverAddr = 0;
  OPEN_BOOL_t discard = OPEN_FALSE;
  OPEN_BOOL_t isServerVrfSet = OPEN_FALSE;
  open_buffdesc serverVrfNameBufd;
  char serverVrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t hitCount = 0;
  char ipStr[16];

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  memset(serverVrfNameStr, 0, sizeof(serverVrfNameStr));
  serverVrfNameBufd.pstart = serverVrfNameStr;
  serverVrfNameBufd.size = strlen(serverVrfNameStr) + 1;


  result = openapiIhIpHelperAddressFirst(client_handle, &vrfNameBufd, &intIfNum, &udpPort,
                                         &serverAddr, &discard, &isServerVrfSet,
                                         &serverVrfNameBufd, &hitCount);
  switch(result)
  {
    case OPEN_E_NONE:
         memset(ipStr, 0, sizeof(ipStr));
         inet_ntop(AF_INET, &serverAddr, ipStr, 16);
         printf("\nSuccessfully got the first entry for the given VRF/interface. Below are details of the entry:\n");
         printf("\nVRF Name: [%s]", vrfNameStr);
         printf("\ninternal interface number: %u", intIfNum);
         printf("\nUDP port number: %u", udpPort);
         printf("\nIPv4 address of server: [%s]", ipStr);
         printf("\nIs this a Discard entry [1-True/0-False]: %u",discard);
         printf("\nIs the server's VRF name configured [1-True/0-False]: %u", isServerVrfSet);
         printf("\nVRF Name of the server if configured: [%s]", serverVrfNameStr);
         printf("\nNumber of times this entry has been used to relay a packet: %u.\n\n", hitCount); 
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the Relay Agent Circuit ID option mode for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}

/*********************************************************************
* @purpose  Get the next relay entry along with server's VRF name config
*
* @param    client_handle  @b{(input)} Client handle from registration API
* @param    vrfName        @b{(input)} VRF name
* @param    intIfNum       @b{(input)} internal interface number
* @param    udpPort        @b{(input)} UDP port number
*
* @returns  none
*
* @end
*********************************************************************/
void ihIpHelperAddressNext( openapiClientHandle_t *client_handle, char *vrfName, uint32_t intIfNum, uint16_t udpPort)
{
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc vrfNameBufd;
  char vrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t serverAddr = 0;
  OPEN_BOOL_t discard = OPEN_FALSE;
  OPEN_BOOL_t isServerVrfSet = OPEN_FALSE;
  open_buffdesc serverVrfNameBufd;
  char serverVrfNameStr[OPEN_VRF_MAX_NAME_LEN + 1] = "";
  uint32_t hitCount = 0;
  char ipStr[16];

  memset(vrfNameStr, 0, sizeof(vrfNameStr));
  if (OPEN_VRF_MAX_NAME_LEN < strlen(vrfName))
  {
    printf("\nERROR: Invalid VRF name string.\n");
    return;
  }
  strncpy(vrfNameStr, vrfName, sizeof(vrfNameStr) - 1);
  vrfNameBufd.pstart = vrfNameStr;
  vrfNameBufd.size = strlen(vrfNameStr) + 1;

  memset(serverVrfNameStr, 0, sizeof(serverVrfNameStr));
  serverVrfNameBufd.pstart = serverVrfNameStr;
  serverVrfNameBufd.size = strlen(serverVrfNameStr) + 1;


  result = openapiIhIpHelperAddressNext(client_handle, &vrfNameBufd, &intIfNum, &udpPort,
                                        &serverAddr, &discard, &isServerVrfSet,
                                        &serverVrfNameBufd, &hitCount);
  switch(result)
  {
    case OPEN_E_NONE:
         memset(ipStr, 0, sizeof(ipStr));
         inet_ntop(AF_INET, &serverAddr, ipStr, 16);
         printf("\nSuccessfully got the next entry for the given VRF/interface and UDP-port. Below are details of the entry:\n");
         printf("\nVRF Name: [%s]", vrfNameStr);
         printf("\ninternal interface number: %u", intIfNum);
         printf("\nUDP port number: %u", udpPort);
         printf("\nIPv4 address of server: [%s]", ipStr);
         printf("\nIs this a Discard entry [1-True/0-False]: %u",discard);
         printf("\nIs the server's VRF name configured [1-True/0-False]: %u", isServerVrfSet);
         printf("\nVRF Name of the server if configured: [%s]", serverVrfNameStr);
         printf("\nNumber of times this entry has been used to relay a packet: %u.\n\n", hitCount); 
         break;
    case OPEN_E_PARAM:
         printf("\nERROR: Invalid argument passed.\n");
         break;
    default:
         printf("\nERROR: Bad return code trying to set the Relay Agent Circuit ID option mode for the given VRF. (result = %d)\n", result);
         break;
  }
  return;
}




/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for IpHelper
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  uint32_t value = 0;
  uint32_t ifNum = 0;
  uint32_t mode = 0;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("IpHelper example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting IpHelper API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1: /* Test 1: Get the Maximum Hop count configured for DHCP relay agent for the given VRF: iphelper_example 1 <vrfName> */
      if (argc == 3)
      {
        bootpDhcpRelayMaxHopCountGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 2: /* Test 2: Set the Maximum Hop count for DHCP relay agent for the given VRF: iphelper_example 2 <vrfName> <maxHopCount> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        bootpDhcpRelayMaxHopCountSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 3: /* Test 3: Get the Minimum Wait time configured for DHCP relay agent for the given VRF: iphelper_example 3 <vrfName> */
      if (argc == 3)
      {
        bootpDhcpRelayMinWaitTimeGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 4: /* Test 4: Set the Minimum Wait time for DHCP relay agent for the given VRF: iphelper_example 4 <vrfName> <minWaitTime> */
      if (argc == 4)
      {
        value = atoi(argv[3]);
        bootpDhcpRelayMinWaitTimeSet(&client_handle, argv[2], value);
        show_help = 0;
      }
      break;
    case 5: /* Test 5: Get the administrative mode of IP-Helper for the given VRF: iphelper_example 5 <vrfName> */
      if (argc == 3)
      {
        ipHelperAdminModeGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 6: /* Test 6: Set the administrative mode of IP-Helper for the given VRF: iphelper_example 6 <vrfName> <mode> */
      if (argc == 4)
      {
        mode = atoi(argv[3]);
        ipHelperAdminModeSet(&client_handle, argv[2], mode);
        show_help = 0;
      }
      break;
    case 7: /* Test 7: Get the Relay Agent Circuit ID option mode for the given VRF: iphelper_example 7 <vrfName> */
      if (argc == 3)
      {
        bootpDhcpRelayCircuitIdOptionModeGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 8: /* Test 8: Set the Relay Agent Circuit ID option mode for the given VRF: iphelper_example 8 <vrfName> <circuitIDOptionMode> */
      if (argc == 4)
      {
        mode = atoi(argv[3]);
        bootpDhcpRelayCircuitIdOptionModeSet(&client_handle, argv[2], mode);
        show_help = 0;
      }
      break;
    case 9: /* Test 9: Get the Relay Agent global option check mode for the given VRF: iphelper_example 9 <vrfName> */
      if (argc == 3)
      {
        bootpDhcpRelayCIDOptionCheckModeGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 10: /* Test 10: Set the Relay Agent global option check mode for the given VRF: iphelper_example 10 <vrfName> <cidOptionCheckMode> */
      if (argc == 4)
      {
        mode = atoi(argv[3]);
        bootpDhcpRelayCIDOptionCheckModeSet(&client_handle, argv[2], mode);
        show_help = 0;
      }
      break;
    case 11: /* Test 11: Get the IP Helper statistics for the given VRF: iphelper_example 11 <vrfName> */
      if (argc == 3)
      {
        ipHelperStatisticsGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 12: /* Test 12: Get the Relay Agent global Server Override mode for the given VRF: iphelper_example 12 <vrfName> */
      if (argc == 3)
      {
        bootpDhcpRelayServerOverrideModeGet(&client_handle, argv[2]); 
        show_help = 0;
      }
      break;
    case 13: /* Test 13: Set the Relay Agent global Server Override mode for the given VRF: iphelper_example 13 <vrfName> <mode> */
      if (argc == 4)
      {
        mode = atoi(argv[3]);
        bootpDhcpRelayServerOverrideModeSet(&client_handle, argv[2], mode);
        show_help = 0;
      }
      break;
    case 14: /* Test 14: Get the Relay Agent Server Override mode for the given interface: iphelper_example 14 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        bootpDhcpRelayIntfServerOverrideGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 15: /* Test 15: Set the Relay Agent Server Override mode for the given interface: iphelper_example 15 <ifNum> <mode> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        mode = atoi(argv[3]);
        bootpDhcpRelayIntfServerOverrideSet(&client_handle, ifNum, mode);
        show_help = 0;
      }
      break;
    case 16: /* Test 16: Get the global Source interface of DHCP relay agent for the given VRF: iphelper_example 16 <vrfName> */
      if (argc == 3)
      {
        bootpDhcpRelaySourceInterfaceGet(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 17: /* Test 17: Set the global Source interface of DHCP relay agent for the given VRF: iphelper_example 17 <vrfName> <ifNum> */
      if (argc == 4)
      {
        ifNum = atoi(argv[3]);
        bootpDhcpRelaySourceInterfaceSet(&client_handle, argv[2], ifNum);
        show_help = 0;
      }
      break;
    case 18: /* Test 18: Clear the global Source interface of DHCP relay agent for the given VRF: iphelper_example 18 <vrfName> */
      if (argc == 3)
      {
        bootpDhcpRelaySourceInterfaceClear(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 19: /* Test 19: Get the interface level Source interface for DHCP relay agent: iphelper_example 19 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        bootpDhcpRelayIntfSourceInterfaceGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 20: /* Test 20: Set the interface level Source interface for DHCP relay agent: iphelper_example 20 <ifNum> <srcIfNum> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        value = atoi(argv[3]);
        bootpDhcpRelayIntfSourceInterfaceSet(&client_handle, ifNum, value);
        show_help = 0;
      }
      break;
    case 21: /* Test 21: Clear the interface level Source interface for DHCP relay agent: iphelper_example 21 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        bootpDhcpRelayIntfSourceInterfaceClear(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 22: /* Test 22: Remove all IP helper addresses configured on an interface: iphelper_example 22 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        ipHelperAddressesRemove(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 23: /* Test 23: Get the Relay Agent Circuit ID Option mode for the given interface: iphelper_example 23 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        bootpDhcpRelayrtrCIDOptionModeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 24: /* Test 24: Set the Relay Agent Circuit ID Option mode for the given interface: iphelper_example 24 <ifNum> <circuitIDintfOptionMode> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        mode = atoi(argv[3]);
        bootpDhcpRelayrtrCIDOptionModeSet(&client_handle, ifNum, mode);
        show_help = 0;
      }
      break;
    case 25: /* Test 25: Get the Relay Agent Circuit ID option check mode for the given interface: iphelper_example 25 <ifNum> */
      if (argc == 3)
      {
        ifNum = atoi(argv[2]);
        bootpDhcpRelayrtrCIDOptionCheckModeGet(&client_handle, ifNum);
        show_help = 0;
      }
      break;
    case 26: /* Test 26: Set the Relay Agent Circuit ID option check mode for the given interface: iphelper_example 26 <ifNum> <cidOptionCheckMode> */
      if (argc == 4)
      {
        ifNum = atoi(argv[2]);
        mode = atoi(argv[3]);
        bootpDhcpRelayrtrCIDOptionCheckModeSet(&client_handle, ifNum, mode);
        show_help = 0;
      }
      break;
    case 27: /* Test 27: Clear IP helper statistics for the given VRF: iphelper_example 27 <vrfName> */ 
      if (argc == 3)
      {
        ipHelperStatisticsClear(&client_handle, argv[2]);
        show_help = 0;
      }
      break;

    case 28: /* Test 28: Get the first relay entry along with server's VRF name config: iphelper_example 28 <vrfName> <ifNum> */
      if (argc == 4)
      {
        ifNum = atoi(argv[3]);
        ihIpHelperAddressFirst(&client_handle, argv[2], ifNum);
        show_help = 0;
      }
      break;

    case 29: /* Test 29: Get the next relay entry along with server's VRF name, interface and UDP-port info: iphelper_example 29 <vrfName> <ifNum> <udpPort> */
      if (argc == 5)
      {
        ifNum = atoi(argv[3]);
        value = atoi(argv[4]);
        ihIpHelperAddressNext(&client_handle, argv[2], ifNum, value);
        show_help = 0;
      }
      break;


    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping IpHelper API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
