/*********************************************************************
*
*  Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ipRouteTable_example.c
*
* @purpose   Switch IP Route Table APIs Example.
*
* @component OPEN
*
* @create  25/05/2016
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "openapi_ip.h"
#include "openapi_ip_route_table.h"
#include "openapi_common.h"
#include "proc_util.h"

void errorMessage(open_error_t rc, char *message)
{
  switch (rc)
  {
    case OPEN_E_FAIL:
      strcpy(message, "Operation Fail.");
      break;
    case OPEN_E_UNAVAIL:
      strcpy(message, "Parameter/feature is not supported.");
      break;
    case OPEN_E_PARAM:
      strcpy(message, "Invalid parameter.");
      break;
    default:
      strcpy(message, "Internal error.");
      break;
  }
  return;
}

void ipRouteTableExample(openapiClientHandle_t clientHandle)
{
  open_error_t rc = OPEN_E_NONE;
  char message[100] = {'\0'};
  uint32_t routeDest = 0;
  uint32_t routeMask = 0;
  OPEN_RTO_PROTOCOL_INDICES_t routeProto = OPEN_RTO_RESERVED;
  uint32_t routePref = 0;
  uint32_t metric = 0;
  uint32_t routeAge = 0;
  OPEN_BOOL_t hwFailure = OPEN_FALSE;
  uint8_t bufNextHopIntf[1024] = {'\0'};
  open_buffdesc nextHopIntf = {.pstart = bufNextHopIntf, .size = 1024};
  uint8_t bufNextHopAddr[1024] = {'\0'};
  open_buffdesc nextHopAddr = {.pstart = bufNextHopAddr, .size = 1024};

  rc = openapiIpRouteEntryGetNext(&clientHandle, &routeDest, &routeMask, &routeProto, &routePref);
  if (rc != OPEN_E_NONE)
  {
    memset(message, 0, sizeof(message));
    (void) errorMessage(rc, message);
    printf("INFORMATION: Cannot get IP Route Table entry. Error = %s\n", message);
  }
  else
  {
     printf("Route destination IP address(integer) is: %d\n", routeDest);
     printf("Route mask is(integer): %d\n", routeMask);
     printf("Route protocol(integer) is: %d\n", routeProto);
     printf("Route preference is: %d\n", routePref);

    rc = openapiIpRouteMetricGet(&clientHandle, &routeDest, &routeMask, &routeProto, &routePref, &metric);
    if (rc != OPEN_E_NONE)
    {
      memset(message, 0, sizeof(message));
      (void) errorMessage(rc, message);
      printf("INFORMATION: Cannot get route metric. Error = %s\n", message);
    }
    else
    {
      printf("Route metric is: %d \n", metric);
    }

    rc = openapiIpRouteAgeGet(&clientHandle, &routeDest, &routeMask, &routeProto, &routePref, &routeAge);
    if (rc != OPEN_E_NONE)
    {
      memset(message, 0, sizeof(message));
      (void) errorMessage(rc, message);
      printf("INFORMATION: Cannot get route age. Error = %s\n", message);
    }
    else
    {
      printf("Route age is: %d \n", routeAge);
    }

    rc = openapiIpRouteHwFailureGet(&clientHandle, &routeDest, &routeMask, &routeProto, &routePref, &hwFailure);
    if (rc != OPEN_E_NONE)
    {
      memset(message, 0, sizeof(message));
      (void) errorMessage(rc, message);
      printf("INFORMATION: Cannot get hardware failure status. Error = %s\n", message);
    }
    else
    {
      printf("Hardware failure status is: %d \n", hwFailure);
    }

    rc = openapiIpRouteNextHopIntfGet(&clientHandle, &routeDest, &routeMask, &routeProto, &routePref, &nextHopIntf);
    if (rc != OPEN_E_NONE)
    {
      memset(message, 0, sizeof(message));
      (void) errorMessage(rc, message);
      printf("INFORMATION: Cannot get next hop interfaces. Error = %s\n", message);
    }
    else
    {
      printf("Next hop interfaces: %s \n", bufNextHopIntf);
    }

    rc = openapiIpRouteNextHopAddrGet(&clientHandle, &routeDest, &routeMask, &routeProto, &routePref, &nextHopAddr);
    if (rc != OPEN_E_NONE)
    {
      memset(message, 0, sizeof(message));
      (void) errorMessage(rc, message);
      printf("INFORMATION: Cannot get next hop IP addresses. Error = %s\n", message);
    }
    else
    {
      printf("Next hop IP addresses: %s \n", bufNextHopAddr);
    }
  }
}

/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         Interface OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_FAIL;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("ipRouteTable_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting result = %d.\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }
  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf("Network OS version retrieve error\n");
  }

  (void) ipRouteTableExample(clientHandle);
  printf("\n");

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping ipRouteTable API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
