
/*********************************************************************
*
*  Copyright 2016-2024 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  lldp_example.c
*
* @purpose   LLDP OpEN API Examples.
*
* @component OpEN
*
* @comments
*
* @create    19/10/2012
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_lldp.h"

#define STRNCAT(_dst, _src) \
  strncat((_dst), (_src), sizeof((_dst))-strlen((_dst))-1)

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the LLDP Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printLldpAppMenu()
{
  printf("Usage: lldp_example <test#> <interface> <arg1> <arg2> ... \n");
  printf("Test 1: Set LLDP Interface Transmit Mode: lldp_example 1 <interface> <tramsmit mode> \n");
  printf("Test 2: Get LLDP Interface Transmit Mode: lldp_example 2 <interface> \n");
  printf("Test 3: Set LLDP Interface Receive Mode: lldp_example 3 <interface> <receive mode> \n");
  printf("Test 4: Get LLDP Interface Receive Mode: lldp_example 4 <interface> \n");
  printf("Test 5: Set LLDP Interface TLV Transmit Mode: lldp_example 5 <interface> <TLV type> <transmit mode> \n");
  printf("Test 6: Get LLDP Interface TLV Transmit Mode: lldp_example 6 <interface> \n");
  printf("Test 7: Set LLDP Interface Notification Mode: lldp_example 7 <interface> <notification mode> \n");
  printf("Test 8: Get LLDP Interface Notification Mode: lldp_example 8 <interface> \n");
  printf("Test 9: Get LLDP Remote Device details: lldp_example 9 <interface> \n");
  printf("Test 10: Get LLDP Local Device details: lldp_example 10 <interface> \n");
  printf("Test 11: Get LLDP-MED interface administrative mode: lldp_example 11 <interface> \n");
  printf("Test 12: Set LLDP-MED interface administrative mode: lldp_example 12 <interface> <mode> \n");
  printf("Test 13: Get LLDP-MED interface config notification mode: lldp_example 13 <interface> \n");
  printf("Test 14: Set LLDP-MED interface config notification mode: lldp_example 14 <interface> <mode> \n");
  printf("Test 15: Get LLDP-MED interface TLVs enabled for transmision: lldp_example 15 <interface> \n");
  printf("Test 16: Set LLDP-MED interface TLVs for transmision: lldp_example 16 <interface> \n");
  printf("Test 17: Get LLDP-MED Remote Device data: lldp_example 17 <interface> \n");
  printf("Test 18: Gets the 802.1AB global transmit interval: lldp_example 18 \n");
  printf("Test 19: Sets the 802.1AB global transmit interval: lldp_example 19 <interval>\n");
  printf("Test 20: Gets the 802.1AB global transmit hold muliplier: lldp_example 20 \n");
  printf("Test 21: Sets the 802.1AB global transmit hold muliplier: lldp_example 21 <holdCount>\n");
  printf("Test 22: Gets the 802.1AB global transmit reinit delay: lldp_example 22 \n");
  printf("Test 23: Sets the 802.1AB global transmit reinit delay: lldp_example 23 <delay>\n");
  printf("Test 24: Gets the 802.1AB global notification interval limit: lldp_example 24 \n");
  printf("Test 25: Sets the 802.1AB global notification interval limit: lldp_example 25 <interval>\n");
  printf("Test 26: Gets the fast start repeat count: lldp_example 26 \n");
  printf("Test 27: Sets the fast start repeat count: lldp_example 27 <count>\n");
  printf("Test 28: Gets the 802.1AB mgmt addr transmit mode for the specified interface: lldp_example 28 <interface>\n");
  printf("Test 29: Sets the 802.1AB mgmt addr transmit mode for the specified interface: lldp_example 29 <interface> <mode>\n");
  printf("Test 30: Gets the vlan id associated with a particular policy type: lldp_example 30 <interface> <policyAppType> \n");
  printf("Test 31: Gets the tagged bit associated with a particular policy type: lldp_example 31 <interface> <policyAppType> \n");
  printf("Test 32: Gets the priority associated with a particular policy type: lldp_example 32 <interface> <policyAppType> \n");
  printf("Test 33: Gets the DSCP associated with a particular policy type: lldp_example 33 <ifNum> <policyAppType> \n");
  printf("Test 34: Gets the first interface enable for LLDP receives: lldp_example 34 \n");
  printf("Test 35: Gets the next valid interface enable for LLDP receives: lldp_example 35 <ifNum> \n");
  printf("Test 36: Get remote port's auto negotiation capability: lldp_example 36 <ifNum> <remIndex> <timestamp>\n");
  printf("Test 37: Get remote port's auto negotiation enabled status: lldp_example 37 <ifNum> <remIndex> <timestamp>\n");
  printf("Test 38: Get remote port's auto negotiation advertized capability as a string: lldp_example 38 <ifNum> <remIndex> <timestamp>\n");
  printf("Test 39: Get remote port's auto negotiation operational MAU Type: lldp_example 39 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 40: Get remote port's auto negotiation operational MAU in string format: lldp_example 40 <mauType> \n");
  printf("Test 41: Get remote device's hardware revision: lldp_example 41 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 42: Get remote device's firmware revision: lldp_example 42 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 43: Get remote device's software revision: lldp_example 43 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 44: Get remote device's serial number: lldp_example 44 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 45: Get remote device's asset ID: lldp_example 45 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 46: Get remote device's PoE device type connected to this port: lldp_example 46 <ifNum> <remIndex> <timestamp>\n");
  printf("Test 47: Get remote ports PSE power value in tenths of watts: lldp_example 47 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 48: Get remote ports PSE power source: lldp_example 48 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 49: Get remote ports PSE power priority: lldp_example 49 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 50: Get the location info received on this port for the given location subtype: lldp_example 50 <ifNum> <remIndex> <timestamp> <subType> \n");
  printf("Test 51: Get the vlan id associated with a particular policy type: lldp_example 51 <ifNum> <remIndex> <timestamp> <policyAppType> \n");
  printf("Test 52: Get the priority associated with a particular policy type: lldp_example 52 <ifNum> <remIndex> <timestamp> <policyAppType> \n");
  printf("Test 53: Get the DSCP associated with a particular policy type: lldp_example 53 <ifNum> <remIndex> <timestamp> <policyAppType> \n");
  printf("Test 54: Get the unknown bit associated with a particular policy type: lldp_example 54 <ifNum> <remIndex> <timestamp> <policyAppType> \n");
  printf("Test 55: Get the tagged bit associated with a particular policy type: lldp_example 55 <ifNum> <remIndex> <timestamp> <policyAppType> \n");
  printf("Test 56: Get the next unknown TLV entry for the indicated remote data entry: lldp_example 56 <ifNum> <remIndex> <timestamp> <tlvType> \n");
  printf("Test 57: Determine if unknown TLV entry exists: lldp_example 57 <ifNum> <remIndex> <timestamp> <tlvType> \n");
  printf("Test 58: Get the TLV info for the specified remote entry: lldp_example 58 <ifNum> <remIndex> <timestamp> <tlvType> \n");
  printf("Test 59: Get the remote device's manufacturer's name: lldp_example 59 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 60: Get the remote device's model name: lldp_example 60 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 61: Get the remote port's PD power requirement: lldp_example 61 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 62: Get the remote port's PD power source: lldp_example 62 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 63: Get the remote port's PD power priority: lldp_example 63 <ifNum> <remIndex> <timestamp> \n");
  printf("Test 64: Get the organizationally defined OUI for specified entry: lldp_example 64 <ifNum> <remIndex> <timestamp> <infoIndex> \n");
  printf("Test 65: Get the organizationally defined subtype for specified entry: lldp_example 65 <ifNum> <remIndex> <timestamp> <infoIndex> \n");
  printf("Test 66: Get a string representation for Chassis ID Subtype: lldp_example 66 <subtype> \n");
  printf("Test 67: Get a string representation for port ID Subtype: lldp_example 67 <subtype> \n");
  printf("Test 68: Get the port's auto negotiation capability: lldp_example 68 <ifNum>\n");
  printf("Test 69: Get port's auto negotiation enabled status: lldp_example 69 <ifNum>\n");
  printf("Test 70: Get port's auto negotiation advertized capability as a string: lldp_example 70 <ifNum>\n");
  printf("Test 71: Get port's auto negotiation operational MAU Type: lldp_example 71 <ifNum>\n");
  printf("Test 72: Get local device's MED Classification: lldp_example 72\n");
  printf("Test 73: Test LLDP OpEN APIs sanity: lldp_example 73 <interface> \n");

  return;
}

/*********************************************************************
* @purpose  Sets LLDP PDU transmit mode on the specified interface.
*
* @param    clientHandle   @b{(input)}   client handle from registration API
* @param    ifNum          @b{(input)}   ID of the interface
* @param    mode           @b{(input)}   LLDP transmit mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void lldpIntfTransmitModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiLldpIntfTransmitModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP Interface Transmit Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Transmit Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets LLDP PDU transmit mode on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfTransmitModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t lldpTxMode;
  
  if ((result = openapiLldpIntfTransmitModeGet(clientHandle, ifNum, &lldpTxMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP Interface Transmit Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Transmit Mode for interface %u: %u \n", ifNum, lldpTxMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets LLDP PDU reception mode on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
* @param    mode           @b{(input)}  LLDP receive mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void lldpIntfReceiveModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiLldpIntfReceiveModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP Interface Receive Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Receive Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets LLDP PDU reception mode on the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfReceiveModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t lldpRxMode;
  
  if ((result = openapiLldpIntfReceiveModeGet(clientHandle, ifNum, &lldpRxMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP Interface Receive Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Receive Mode for interface %u: %u \n", ifNum, lldpRxMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the LLDP optional TLVs transmit mode for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
* @param    tlvType          @b{(input)}   LLDP TLV of type OPEN_LLDP_TLV_TYPE_t
* @param    tlvTransmitMode  @b{(input)}   TLV transmit mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void lldpIntfTLVTransmitModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, 
                                OPEN_LLDP_TLV_TYPE_t tlvType, OPEN_CONTROL_t tlvTransmitMode)
{
  open_error_t result;
  
  if ((result = openapiLldpIntfTLVTransmitModeSet(clientHandle, ifNum, tlvType, tlvTransmitMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP Interface TLV Transmit Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface TLV Transmit Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the LLDP optional TLVs transmit mode for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfTLVTransmitModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t lldpTLVTxMode;
  uint32_t index;
  
  for (index = 0; index < OPEN_LLDP_LAST_TLV; index++)
  {
    if ((result = openapiLldpIntfTLVTransmitModeGet(clientHandle, ifNum, index, &lldpTLVTxMode)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get LLDP Interface TLV Transmit Mode. (result = %d)\n", result);
      return;
    }
    else
    {
      printf("Interface: %u, TLV type: %u, Transmit Mode: %u \n", ifNum, index, lldpTLVTxMode);
    }
  }
  return;
}

/*********************************************************************
* @purpose  Sets the LLDP Notification mode for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
* @param    mode           @b{(input)}  LLDP Notification mode
*
* @returns  none
*
* @notes  Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void lldpIntfNotificationModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;
  
  if ((result = openapiLldpIntfNotificationModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP Interface Notification Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Notification Mode set successfully \n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the LLDP Notification mode for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfNotificationModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t lldpNotMode;
  
  if ((result = openapiLldpIntfNotificationModeGet(clientHandle, ifNum, &lldpNotMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP Interface Notification Mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP Interface Notification Mode for interface %u: %u \n", ifNum, lldpNotMode);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the LLDP Remote Device details for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfRemoteDeviceDetailsGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_t chassisIdSubtype;
  OPEN_LLDP_PORT_ID_SUBTYPE_t portIdSubtype;
  open_buffdesc chassisId, portId;
  open_buffdesc sysName, sysDesc, portDesc;
  open_buffdesc sysCapSupp, sysCapEnabled;
  char *str = NULL;
  open_buffdesc mgmtAddress;
  OPEN_LLDP_IANA_ADDR_FAMILY_t addressFamily = 0;
  OPEN_LLDP_MAN_ADDR_IF_SUBTYPE_t subType;
  uint32_t remIndex = 0;
  uint32_t timeStamp = 0;
  uint32_t ttl = 0;
  uint8_t addrLen;
  uint32_t strLen = 0;
  uint32_t intfId = 0;
  uint16_t portVlan = 0;
  uint16_t vlanId = 0;
  char *vlanNameStr = NULL;
  open_buffdesc vlanName;
  uint32_t vlanNameLen = 0;
  uint32_t infoIndex = 0, nextInfoIndex = 0;
  uint16_t maxFrameSize = 0;
  uint32_t aggregationId = 0;
  OPEN_BOOL_t aggregationCapable = OPEN_FALSE;
  OPEN_BOOL_t aggregationMember = OPEN_FALSE;

  if ((result = openapiLldpMgmtStrLenGet(clientHandle, &strLen)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(strLen)) == NULL) 
  {
    printf("Could not allocate memory.\n");
    return;
  }

  printf("LLDP Remote Device details for interface %u\n", ifNum);
  printf("============================================\n\n");

  while(openapiLldpIntfRemoteDeviceNextGet(clientHandle, ifNum, 
                                           &remIndex, &timeStamp) == OPEN_E_NONE)
  {
    memset(str, 0, strLen);
    chassisId.pstart = str;
    chassisId.size = strLen;

    if ((result = openapiLldpIntfRemoteDeviceChassisIdGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                          &chassisIdSubtype, &chassisId)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("Remote Identifier: %u\n", remIndex);
      printf("Chassis Id Subtype: %u\n", chassisIdSubtype);
      printf("Chassis Id: %s\n\n", str);
    }

    memset(str, 0, strLen);
    portId.pstart = str;
    portId.size = strLen;

    if ((result = openapiLldpIntfRemoteDevicePortIdGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                       &portIdSubtype, &portId)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("Port Id Subtype: %u\n", portIdSubtype);
      printf("Port Id: %s\n\n", str);
    }

    memset(str, 0, strLen);
    sysName.pstart = str;
    sysName.size = strLen;

    if ((result = openapiLldpIntfRemoteDeviceSystemNameGet(clientHandle, ifNum, remIndex, 
                                                           timeStamp, &sysName)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("System name: %s\n\n", str);
    }

    memset(str, 0, strLen);
    sysDesc.pstart = str;
    sysDesc.size = strLen;

    if ((result = openapiLldpIntfRemoteDeviceSystemDescGet(clientHandle, ifNum, remIndex, 
                                                           timeStamp, &sysDesc)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("System Description: %s\n\n", str);
    }

    memset(str, 0, strLen);
    portDesc.pstart = str;
    portDesc.size = strLen;

    if ((result = openapiLldpIntfRemoteDevicePortDescGet(clientHandle, ifNum, remIndex, 
                                                         timeStamp, &portDesc)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("Port Description: %s\n\n", str);
    }

    memset(str, 0, strLen);
    sysCapSupp.pstart = str;
    sysCapSupp.size = strLen;

    if ((result = openapiLldpIntfRemoteDeviceSysCapSuppGet(clientHandle, ifNum, remIndex, 
                                                           timeStamp, &sysCapSupp)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("System Capabilities Supported: %s\n\n", str);
    }

    memset(str, 0, strLen);
    sysCapEnabled.pstart = str;
    sysCapEnabled.size = strLen;

    if ((result = openapiLldpIntfRemoteDeviceSysCapEnabledGet(clientHandle, ifNum, remIndex, 
                                                              timeStamp, &sysCapEnabled)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("System Capabilities Enabled: %s\n\n", str);
    }

    memset(str, 0, strLen);
    mgmtAddress.pstart = str;
    mgmtAddress.size = strLen;
    addrLen = 0;

    while (openapiLldpIntfRemoteDeviceEntryNextGet(clientHandle, ifNum, remIndex, 
                                                   timeStamp, &addressFamily, 
                                                   &mgmtAddress, 
                                                   &addrLen, &subType, &intfId)
           == OPEN_E_NONE)
    {
      printf("Management Address Type: %u\n", addressFamily);
      printf("Management Address: %s\n", str);
      printf("Interface subtype : %u\n", subType);
      printf("Interface ID : %u\n\n", intfId);
    }

    if ((result = openapiLldpIntfRemoteDeviceTTLGet(clientHandle, ifNum, remIndex, 
                                                    timeStamp, &ttl)) != OPEN_E_NONE)
    {
      printf("Bad return code. (result = %d)\n\n", result);
    }
    else
    {
      printf("LLDP Neighbor TTL: %u\n\n", ttl);
    }
     
    portVlan = 0;
    if ((result = openapiLldpIntfRemoteDevicePortVlanGet(clientHandle, ifNum, remIndex, 
                                                         timeStamp, &portVlan)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to PVID. (result = %d)\n\n", result);
    }
    else
    {
      printf("Port VLAN: %u\n\n", portVlan);
    }

    while (OPEN_E_NONE == openapiLldpRemoteDeviceVlanInfoIndexGet(clientHandle, ifNum,
                                                                  remIndex, timeStamp,
                                                                  infoIndex, &nextInfoIndex))
    {
      if ((result = openapiVlanNameLenGet(clientHandle, &vlanNameLen)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get VLAN Name Length. (result = %d)\n", result);
      }
      else
      {
        vlanNameStr = (char*)malloc(vlanNameLen);
      
        if (NULL == vlanNameStr)
        {
          printf("  Could not allocate memory for VLAN name.\n");
        }
        else
        {
          memset(vlanNameStr, 0, vlanNameLen);
          vlanName.pstart = vlanNameStr;
          vlanName.size = vlanNameLen;
          if ((result = openapiLldpIntfRemoteDeviceVlanIdAndNameGet(clientHandle, ifNum,
                                                                    remIndex, timeStamp,
                                                                    nextInfoIndex,
                                                                    &vlanId, &vlanName)) != OPEN_E_NONE)
          {
            printf("Unable to get the VLAN Id and name. (result = %d)\n", result);
          }
          else
          {
            printf("VLAN: %d\n", vlanId);
            printf("VLAN Name: %s\n\n", vlanNameStr);
          }
          infoIndex = nextInfoIndex;
          free(vlanNameStr);
        }
      }
    }

    maxFrameSize = 0;
    if ((result = openapiLldpIntfRemoteDeviceMaxFrameSizeGet(clientHandle, ifNum, remIndex, 
                                                             timeStamp, &maxFrameSize)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to Maximum Frame Size. (result = %d)\n\n", result);
    }
    else
    {
      printf("Maximum Frame Size: %u\n\n", maxFrameSize);
    }

    aggregationCapable = OPEN_FALSE;
    aggregationMember = OPEN_FALSE;
    aggregationId = 0;
    if ((result = openapiLldpIntfRemotePortLinkAggregationInfoGet(clientHandle, ifNum, remIndex, 
                                                                 timeStamp, &aggregationCapable,
                                                                 &aggregationMember, &aggregationId)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to Link Aggregation information. (result = %d)\n\n", result);
    }
    else
    {
      printf("Link Aggregation Information:\n");
      if (OPEN_TRUE == aggregationCapable)
      {
        printf("  Aggregation Capable: Yes\n");
      }
      else
      {
        printf("  Aggregation Capable: No\n");
      }
      if (OPEN_TRUE == aggregationMember)
      {
        printf("  Aggregation Status: Yes\n");
      }
      else
      {
        printf("  Aggregation Status: No\n");
      }
      printf("  Aggregation ID: %u\n\n", aggregationId);
    }
  }
  free(str);
  return;
}

/*********************************************************************
* @purpose  Gets the LLDP Remote Device for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfRemoteDevicesGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  uint32_t currIfNum, nextIfNum;

  if (ifNum == 0)
  {
    currIfNum = ifNum;
    while (openapiLldpIntfNextGet(clientHandle, currIfNum, &nextIfNum) == OPEN_E_NONE)
    {
      lldpIntfRemoteDeviceDetailsGet(clientHandle, nextIfNum);
      currIfNum = nextIfNum;
    }
  }
  else
  {
    lldpIntfRemoteDeviceDetailsGet(clientHandle, ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the LLDP Local Device details for the specified interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfLocalDeviceGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_t chassisIdSubtype;
  OPEN_LLDP_PORT_ID_SUBTYPE_t portIdSubtype;
  open_buffdesc chassisId, portId;
  open_buffdesc sysName, sysDesc, portDesc;
  open_buffdesc sysCapSupp, sysCapEnabled;
  open_buffdesc mgmtAddress;
  OPEN_LLDP_IANA_ADDR_FAMILY_t addressFamily;
  char *str = NULL;
  uint32_t strLen = 0, i = 0;
  uint16_t portVlan = 0;
  OPEN_VLAN_LIST_t vlanList;
  char *vlanNameStr = NULL;
  open_buffdesc vlanName;
  uint32_t vlanNameLen = 0;
  uint16_t maxFrameSize = 0;
  uint32_t aggregationId = 0;
  OPEN_BOOL_t aggregationCapable = OPEN_FALSE;
  OPEN_BOOL_t aggregationMember = OPEN_FALSE;

  if ((result = openapiLldpMgmtStrLenGet(clientHandle, &strLen)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(strLen)) == NULL) 
  {
    printf("Could not allocate memory.\n");
    return;
  }

  printf("LLDP Local Device details for interface %u\n", ifNum);
  printf("===========================================\n\n");

  memset(str, 0, strLen);
  chassisId.pstart = str;
  chassisId.size = strLen;
    
  if ((result = openapiLldpIntfLocalDeviceChassisIdGet(clientHandle, ifNum, &chassisIdSubtype, 
                                                       &chassisId)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("Chassis Id Subtype: %u\n", chassisIdSubtype);
    printf("Chassis Id: %s\n\n", str);
  }

  memset(str, 0, strLen);
  portId.pstart = str;
  portId.size = strLen;

  if ((result = openapiLldpIntfLocalDevicePortIdGet(clientHandle, ifNum, &portIdSubtype, 
                                                    &portId)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("Port Id Subtype: %u\n", portIdSubtype);
    printf("Port Id: %s\n\n", str);
  }

  memset(str, 0, strLen);
  sysName.pstart = str;
  sysName.size = strLen;

  if ((result = openapiLldpIntfLocalDeviceSystemNameGet(clientHandle, ifNum, &sysName)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("System name: %s\n\n", str);
  }

  memset(str, 0, strLen);
  sysDesc.pstart = str;
  sysDesc.size = strLen;

  if ((result = openapiLldpIntfLocalDeviceSystemDescGet(clientHandle, ifNum, &sysDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("System Description: %s\n\n", str);
  }

  memset(str, 0, strLen);
  portDesc.pstart = str;
  portDesc.size = strLen;

  if ((result = openapiLldpIntfLocalDevicePortDescGet(clientHandle, ifNum, &portDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("Port Description: %s\n\n", str);
  }

  memset(str, 0, strLen);
  sysCapSupp.pstart = str;
  sysCapSupp.size = strLen;

  if ((result = openapiLldpIntfLocalDeviceSysCapSuppGet(clientHandle, ifNum, &sysCapSupp)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("System Capabilities Supported: %s\n\n", str);
  }

  memset(str, 0, strLen);
  sysCapEnabled.pstart = str;
  sysCapEnabled.size = strLen;

  if ((result = openapiLldpIntfLocalDeviceSysCapEnabledGet(clientHandle, ifNum, &sysCapEnabled)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("System Capabilities Enabled: %s\n\n", str);
  }

  memset(str, 0, strLen);
  mgmtAddress.pstart = str;
  mgmtAddress.size = strLen;

  if ((result = openapiLldpIntfLocalDeviceMgmtAddressGet(clientHandle, ifNum, 
                                                         &addressFamily, &mgmtAddress)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n\n", result);
  }
  else
  {
    printf("Management Address Type: %u\n", addressFamily);
    printf("Management Address: %s\n\n", str);
  }

  free(str);

  portVlan = 0;
  if ((result = openapiLldpIntfLocalDevicePortVlanGet(clientHandle, ifNum, 
                                                      &portVlan)) != OPEN_E_NONE)
  {
    printf("Unable to get the Port VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("Port VLAN: %u\n\n", portVlan);
  }

  memset(&vlanList, 0, sizeof(vlanList));
  if ((result = openapiLldpIntfLocalDeviceVlanGet(clientHandle, ifNum, 
                                                  &vlanList)) != OPEN_E_NONE)
  {
    printf("Unable to get the VLAN. (result = %d)\n", result);
  }
  else
  {
    printf("VLANs list retrieved:\r\n");
    for (i = 0; i < vlanList.numEntries; i++)
    {
      printf("VLAN: %d\n", vlanList.ids[i]);
      if ((result = openapiVlanNameLenGet(clientHandle, &vlanNameLen)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get VLAN Name Length. (result = %d)\n", result);
      }
      else
      {
        vlanNameStr = (char*)malloc(vlanNameLen);
      
        if (NULL == vlanNameStr)
        {
          printf("  Could not allocate memory for VLAN name.\n");
        }
        else
        {
          memset(vlanNameStr, 0, vlanNameLen);
          vlanName.pstart = vlanNameStr;
          vlanName.size = vlanNameLen;
          if ((result = openapiLldpIntfLocalDeviceVlanNameGet(clientHandle, ifNum,
                                                              vlanList.ids[i], 
                                                              &vlanName)) != OPEN_E_NONE)
          {
            printf("Unable to get the VLAN name. (result = %d)\n", result);
          }
          else
          {
            printf("VLAN Name: %s\n\n", vlanNameStr);
          }
          free(vlanNameStr);
        }
      }
    }

    maxFrameSize = 0;
    if ((result = openapiLldpIntfLocalDeviceMaxFrameSizeGet(clientHandle, ifNum,
                                                            &maxFrameSize)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to Maximum Frame Size. (result = %d)\n\n", result);
    }
    else
    {
      printf("Maximum Frame Size: %u\n\n", maxFrameSize);
    }

    aggregationCapable = OPEN_FALSE;
    aggregationMember = OPEN_FALSE;
    aggregationId = 0;
    if ((result = openapiLldpIntfLocalPortLinkAggregationInfoGet(clientHandle, ifNum, &aggregationCapable,
                                                                 &aggregationMember, &aggregationId)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to Link Aggregation information. (result = %d)\n\n", result);
    }
    else
    {
      printf("Link Aggregation Information:\n");
      if (OPEN_TRUE == aggregationCapable)
      {
        printf("  Aggregation Capable: Yes\n");
      }
      else
      {
        printf("  Aggregation Capable: No\n");
      }
      if (OPEN_TRUE == aggregationMember)
      {
        printf("  Aggregation Status: Yes\n");
      }
      else
      {
        printf("  Aggregation Status: No\n");
      }
      printf("  Aggregation ID: %u\n\n", aggregationId);
    }
    printf("\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the interface LLDP-MED administrative mode.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedIntfModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  if ((result = openapiLldpMedIntfModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP-MED interface mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP-MED interface mode for interface %u: %u \n", ifNum, mode);
  }

  return;
}

/*********************************************************************
* @purpose  Set LLDP-MED administrative mode on LLDP valid interfaces.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  an internal interface number or 
*                          @ref OPEN_ALL_INTERFACES for all interfaces
* @param    mode           @b{(input)}  LLDP mode
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    Administrative mode is set on all LLDP valid interfaces, if 
*           @ref OPEN_ALL_INTERFACES is passed in for the 'ifNum' parameter.

*
* @end
*********************************************************************/
void lldpMedIntfModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiLldpMedIntfModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP-MED mode on interface. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP-MED mode set to %d on interface. (result = %d)\n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the interface LLDP-MED configuration notification mode.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedIntfConfigNotificationModeGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_BOOL_t mode;

  if ((result = openapiLldpMedIntfConfigNotificationModeGet(clientHandle, ifNum, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP-MED interface configuration notification mode. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP-MED interface configuration notification mode for interface %u: %u \n", ifNum, mode);
  }

  return;
}

/*********************************************************************
* @purpose  Set LLDP-MED configuration notification mode on LLDP valid interfaces.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  an internal interface number or 
*                          @ref OPEN_ALL_INTERFACES for all interfaces
* @param    mode           @b{(input)}  notification mode
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    Administrative mode is set on all LLDP valid interfaces, if 
*           @ref OPEN_ALL_INTERFACES is passed in for the 'ifNum' parameter.

*
* @end
*********************************************************************/
void lldpMedIntfConfigNotificationModeSet(openapiClientHandle_t *clientHandle, uint32_t ifNum, OPEN_BOOL_t mode)
{
  open_error_t result;

  if ((result = openapiLldpMedIntfConfigNotificationModeSet(clientHandle, ifNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set LLDP-MED configuration notification mode on interface. (result = %d)\n", result);
  }
  else
  {
    printf("LLDP-MED configuration notification mode set to %d on interface. (result = %d)\n", mode, result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the interface LLDP-MED TLVs enabled for transmision.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  internal interface number
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedInterfaceTransmitTlvGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_LLDP_MED_CAPABILITIES_t cap;
  char str[256];

  memset(&cap, 0, sizeof(cap));
  if ((result = openapiLldpMedIntfTransmitTlvGet(clientHandle, ifNum, &cap)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP-MED interface TLVs enabled for transmision. (result = %d)\n", result);
  }
  else
  {
    memset(&str, 0, sizeof(str));
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_CAPABILITIES_BITMASK)
    {
      STRNCAT(str, "0-CAPABILITIES ");
    }
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_NETWORKPOLICY_BITMASK)
    {
      STRNCAT(str, "1-NETWORKPOLICY ");
    }
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_LOCATION_BITMASK)
    {
      STRNCAT(str, "2-LOCATION ");
    }
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_EXT_PSE_BITMASK)
    {
      STRNCAT(str, "3-EXTENDEDPSE ");
    }
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_EXT_PD_BITMASK)
    {
      STRNCAT(str, "4-EXTENDEDPD ");
    }
    if (cap.bitmap[1] & OPEN_LLDP_MED_CAP_INVENTORY_BITMASK)
    {
      STRNCAT(str, "5-INVENTORY");
    }
    str[sizeof(str) - 1] = '\0';

    printf("LLDP-MED interface TLVs enabled for transmision for interface %u: %s. \n", ifNum, str);
  }

  return;
}

/*********************************************************************
* @purpose  Set LLDP-MED TLVs for transmision on LLDP valid interfaces.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  an internal interface number or 
*                          @ref OPEN_ALL_INTERFACES for all interfaces
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    Administrative mode is set on all LLDP valid interfaces, if 
*           @ref OPEN_ALL_INTERFACES is passed in for the 'ifNum' parameter. 
*
* @end
*********************************************************************/
void lldpMedInterfaceTransmitTlvSet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_LLDP_MED_CAPABILITIES_t current_cap;
  OPEN_LLDP_MED_CAPABILITIES_t test_cap;

  memset(&current_cap, 0, sizeof(current_cap));
  /* not all platforms support the same set of capabilities, test setting to no capabilities is safe */
  memset(&test_cap, 0, sizeof(test_cap));

  if ((result = openapiLldpMedIntfTransmitTlvGet(clientHandle, ifNum, &current_cap)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get current LLDP-MED TLVs for transmision on interface. (result = %d)\n", result);
  }
  else
  {
    if ((result = openapiLldpMedIntfTransmitTlvSet(clientHandle, ifNum, &test_cap)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to set LLDP-MED TLVs for transmision on interface. (result = %d)\n", result);
    }
    else
    {
      if ((result = openapiLldpMedIntfTransmitTlvSet(clientHandle, ifNum, &current_cap)) != OPEN_E_NONE)
      {
        printf("Bad return code trying to set LLDP-MED TLVs for transmision on interface. (result = %d)\n", result);
      }
      else
      {
        printf("Setting LLDP-MED TLVs for transmision on interface successful.\n");
      }
    }
  }

  return;
}

/*********************************************************************
* @purpose  Get the remote device information for the given interface.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface number
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemoteDeviceInfoGet(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  uint32_t remIndex = 0;
  uint32_t timeStamp = 0;
  open_buffdesc modelName;
  char *str = NULL;
  uint32_t strLen = 0;
  OPEN_LLDP_MED_CAPABILITIES_t supportedCapabilities;
  OPEN_LLDP_MED_CAPABILITIES_t enabledCapabilities;
  uint32_t deviceClass;

  if ((result = openapiLldpMedRemoteModelNameSizeGet(clientHandle, &strLen)) != OPEN_E_NONE)
  {
    printf("Bad return code from openapiLldpMedRemoteModelNameSizeGet(). (result = %d)\n", result);
    return;
  }

  if ((str = malloc(strLen)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  printf("LLDP-MED Remote Device details for interface %u\n", ifNum);
  printf("=================================================\n\n");

  while(openapiLldpIntfRemoteDeviceNextGet(clientHandle, ifNum, &remIndex, &timeStamp) == OPEN_E_NONE)
  {
    if ((result = openapiLldpMedRemoteCapabilitiesGet(clientHandle, ifNum, remIndex, timeStamp,
                                                      &supportedCapabilities, &enabledCapabilities)) != OPEN_E_NONE)
    {
      printf("Bad return code from openapiLldpMedRemoteCapabilitiesGet(). (result = %d)\n\n", result);
    }
    else
    {
      printf("Supported capabilities bitmap: 0x%02x%02x\n", supportedCapabilities.bitmap[0], supportedCapabilities.bitmap[1]);
      printf("Enabled capabilities bitmap:   0x%02x%02x\n", enabledCapabilities.bitmap[0], enabledCapabilities.bitmap[1]);
    }

    if ((result = openapiLldpMedRemoteDeviceClassGet(clientHandle, ifNum, remIndex, timeStamp,
                                                     &deviceClass)) != OPEN_E_NONE)
    {
      printf("Bad return code from openapiLldpMedRemoteDeviceClassGet(). (result = %d)\n\n", result);
    }
    else
    {
      printf("Device class: %u\n", deviceClass);
    }

    memset(str, 0, strLen);
    modelName.pstart = str;
    modelName.size = strLen;

    if ((result = openapiLldpMedRemoteModelNameGet(clientHandle, ifNum, remIndex, timeStamp,
                                                   &modelName)) != OPEN_E_NONE)
    {
      printf("Bad return code from openapiLldpMedRemoteModelNameGet(). (result = %d)\n\n", result);
    }
    else
    {
      printf("Model name: %s\n\n", str);
    }
  }

  free(str);
  return;
}

/*********************************************************************
* @purpose  Gets the 802.1AB global transmit interval.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    interval       @b{(output)} Interval in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void lldpTxIntervalGet(openapiClientHandle_t *client_handle, uint32_t *interval)
{
  open_error_t result;

  if ((result = openapiLldpTxIntervalGet(client_handle, interval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1AB global transmit interval."
           " (result = %d)\n", result);
  }
  else
  {
    printf("802.1AB global transmit interval is %u.\n", *interval);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the 802.1AB global transmit interval.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    interval       @b{(input)}  Interval in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void lldpTxIntervalSet(openapiClientHandle_t *client_handle, uint32_t interval)
{
  open_error_t result;

  if ((result = openapiLldpTxIntervalSet(client_handle, interval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1AB global transmit interval."
           " (result = %d)\n", result);
  }
  else
  {
    printf("802.1AB global transmit interval set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the 802.1AB global transmit hold muliplier.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    holdCount       @b{(output)} Hold multiplier
*
* @returns  none
*
* @end
*********************************************************************/
void lldpTxHoldGet(openapiClientHandle_t *client_handle, uint32_t *holdCount)
{
  open_error_t result;

  if ((result = openapiLldpTxHoldGet(client_handle, holdCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1AB global transmit hold "
           "muliplier. (result = %d)\n", result);
  }
  else
  {
    printf("802.1AB global transmit hold muliplier is %u.\n", *holdCount);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the 802.1AB global transmit hold muliplier.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    holdCount       @b{(input)}  Hold multiplier
*
* @returns  none
*
* @end
*********************************************************************/
void lldpTxHoldSet(openapiClientHandle_t *client_handle, uint32_t holdCount)
{
  open_error_t result;

  if ((result = openapiLldpTxHoldSet(client_handle, holdCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1AB global transmit hold "
           "muliplier. (result = %d)\n", result);
  }
  else
  {
    printf("802.1AB global transmit hold muliplier set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the 802.1AB global transmit reinit delay.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    delay           @b{(output)} Delay in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void lldpTxReinitDelayGet(openapiClientHandle_t *client_handle, uint32_t *delay)
{
  open_error_t result;

  if ((result = openapiLldpTxReinitDelayGet(client_handle, delay)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1AB global transmit reinit "
           "delay. (result = %d)\n", result);
  }
  else
  {
    printf("802.1AB global transmit reinit delay is %u.\n", *delay);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the 802.1AB global transmit reinit delay.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    delay         @b{(input)}  Delay in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void lldpTxReinitDelaySet(openapiClientHandle_t *client_handle, uint32_t delay)
{
  open_error_t result;

  if ((result = openapiLldpTxReinitDelaySet(client_handle, delay)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1AB global transmit reinit "
           "delay. (result = %d)\n", result);
  }
  else
  {
    printf("802.1AB global transmit reinit delay set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the 802.1AB global notification interval limit.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    interval       @b{(output)} Interval in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void lldpNotificationIntervalGet(openapiClientHandle_t *client_handle, 
                                 uint32_t *interval)
{
  open_error_t result;

  if ((result = openapiLldpNotificationIntervalGet(client_handle, interval))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1AB global notification "
           "interval limit. (result = %d)\n", result);
  }
  else
  {
    printf("802.1AB global notification interval limit is %u.\n", *interval);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the 802.1AB global notification interval limit.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    interval       @b{(input)}  Interval in seconds
*
* @returns  none
*
* @end
*********************************************************************/
void lldpNotificationIntervalSet(openapiClientHandle_t *client_handle, 
                                 uint32_t interval)
{
  open_error_t result;

  if ((result = openapiLldpNotificationIntervalSet(client_handle, interval))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1AB global notification"
           " interval limit. (result = %d)\n", result);
  }
  else
  {
    printf("802.1AB global notification interval limit set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the fast start repeat count.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    count          @b{(output)} Fast start repeat count
*
* @returns  none
*
* @end
*********************************************************************/
void lldpXMedFastStartRepeatCountGet(openapiClientHandle_t *client_handle, 
                                     uint32_t *count)
{
  open_error_t result;

  if ((result = openapiLldpXMedFastStartRepeatCountGet(client_handle, 
                                                       count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the fast start repeat count. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Fast start repeat count value is %u.\n", *count);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the fast start repeat count.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    count          @b{(input)}  Fast start repeat count
*
* @returns  none
*
* @end
*********************************************************************/
void lldpXMedFastStartRepeatCountSet(openapiClientHandle_t *client_handle, 
                                     uint32_t count)
{
  open_error_t result;

  if ((result = openapiLldpXMedFastStartRepeatCountSet(client_handle, count))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the fast start repeat count."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Fast start repeat count set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the 802.1AB mgmt addr transmit mode for the specified interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    mode           @b{(output)} Transmit mgmt addr TLV mode
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfTxMgmtAddrModeGet(openapiClientHandle_t *client_handle, 
                               uint32_t ifNum, OPEN_BOOL_t *mode)
{
  open_error_t result;

  if ((result = openapiLldpIntfTxMgmtAddrModeGet(client_handle, ifNum, mode))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the 802.1AB mgmt addr transmit"
           " mode for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("802.1AB mgmt addr transmit mode is %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the 802.1AB mgmt addr transmit mode for the specified interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    mode           @b{(input)}  Transmit mgmt addr TLV mode
*
* @returns  none
*
* @end
*********************************************************************/
void lldpIntfTxMgmtAddrModeSet(openapiClientHandle_t *client_handle, 
                               uint32_t ifNum, OPEN_BOOL_t mode)
{
  open_error_t result;

  if ((result = openapiLldpIntfTxMgmtAddrModeSet(client_handle, ifNum, mode))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the 802.1AB mgmt addr transmit mode"
           " for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("802.1AB mgmt addr transmit mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the vlan id associated with a particular policy type.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface
* @param    policyAppType    @b{(input)}  Policy application type
* @param    vlanId           @b{(output)} Vlan Id
*
* @returns  none
*
* @end
*********************************************************************/
void lldpXMedLocMediaPolicyVlanIDGet(openapiClientHandle_t *client_handle,
                                     uint32_t ifNum,
                                     OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                     uint32_t *vlanId)
{
  open_error_t result;

  if ((result = openapiLldpXMedLocMediaPolicyVlanIDGet(client_handle, ifNum, 
                                                       policyAppType, vlanId))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the vlan id associated with a"
           " particular policy type. (result = %d)\n", result);
  }
  else
  {
    printf("Vlan id associated with a given policy type is %u.\n", *vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the tagged bit associated with a particular policy type.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface
* @param    policyAppType    @b{(input)}  Policy application type
* @param    taggedStatus     @b{(output)} Tagged Status
*
* @returns  none
*
* @end
*********************************************************************/
void lldpXMedLocMediaPolicyTaggedGet(openapiClientHandle_t *client_handle,
                                     uint32_t ifNum,
                                     OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                     OPEN_BOOL_t *taggedStatus)
{
  open_error_t result;

  if ((result = openapiLldpXMedLocMediaPolicyTaggedGet(client_handle, ifNum, 
                                                       policyAppType, taggedStatus))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the tagged bit associated with a "
           "particular policy type. (result = %d)\n", result);
  }
  else
  {
    printf("Tagged status for the given policy type is %u.\n", *taggedStatus);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the priority associated with a particular policy type.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface
* @param    policyAppType    @b{(input)}  Policy application type
* @param    priority         @b{(output)} Priority
*
* @returns  none
*
* @end
*********************************************************************/
void lldpXMedLocMediaPolicyPriorityGet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum,
                                       OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                       uint32_t *priority)
{
  open_error_t result;

  if ((result = openapiLldpXMedLocMediaPolicyPriorityGet(client_handle, ifNum,
                                                         policyAppType, priority))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the priority associated with a "
           "particular policy type. (result = %d)\n", result);
  }
  else
  {
    printf("Priority associated with a given policy type is %u.\n", *priority);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the DSCP associated with a particular policy type.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface
* @param    policyAppType    @b{(input)}  Policy application type
* @param    dscp             @b{(output)} Dscp vlaue
*
* @returns  none
*
* @end
*********************************************************************/
void lldpXMedLocMediaPolicyDscpGet(openapiClientHandle_t *client_handle, 
                                   uint32_t ifNum,
                                   OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                   uint32_t *dscp)
{
  open_error_t result;

  if ((result = openapiLldpXMedLocMediaPolicyDscpGet(client_handle, ifNum,
                                                     policyAppType, dscp)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the DSCP associated with a"
           " particular policy type. (result = %d)\n", result);
  }
  else
  {
    printf("DSCP associated with given policy type is %u.\n", *dscp);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the first interface enable for LLDP receives.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(output)} ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpRemoteIntfFirstGet(openapiClientHandle_t *clientHandle, uint32_t *ifNum)
{
  open_error_t result;
  
  if ((result = openapiLldpRemoteIntfFirstGet (clientHandle, ifNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP first interface. (result = %d)\n",
           result);
  }
  else
  {
    printf("First interface enabled for LLDP receives is %u. \n", *ifNum);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the first interface enable for LLDP receives.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
* @param    nextIfNum      @b{(output)} Next ID of the valid interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpRemoteIntfNextGet(openapiClientHandle_t *clientHandle, uint32_t ifNum,
                           uint32_t *nextIfNum)
{
  open_error_t result;
  
  if ((result = openapiLldpRemoteIntfNextGet (clientHandle, ifNum, nextIfNum)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get LLDP next valid interface.(result = %d)\n",
           result);
  }
  else
  {
    printf("Next valid interface enabled for LLDP receives is %u. \n", *nextIfNum);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote port's auto negotiation capability.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    isCapable      @b{(output)} Remote port auto negotiation capability status
*
* @returns  none
*
* @end
*********************************************************************/
void lldpdot3RemPortAutoNegSupportedGet(openapiClientHandle_t *client_handle, 
                                        uint32_t ifNum, uint32_t remIndex, 
                                        uint32_t timestamp, OPEN_BOOL_t *isCapable)
{
  open_error_t result;

  if ((result = openapiLldpdot3RemPortAutoNegSupportedGet(client_handle, ifNum, 
                                                          remIndex, timestamp, 
                                                          isCapable)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote port's auto negotiation "
           "capability. (result = %d)\n", result);
  }
  else
  {
    printf("Remote port's auto negotiation capability is %u.\n", *isCapable);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote port's auto negotiation enabled status.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    isEnabled     @b{(output)} Remote port auto negotiation enabled status
*
* @returns  none
*
* @end
*********************************************************************/
void lldpdot3RemPortAutoNegEnabledGet(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, uint32_t remIndex, 
                                      uint32_t timestamp, OPEN_BOOL_t *isEnabled)
{
  open_error_t result;

  if ((result = openapiLldpdot3RemPortAutoNegEnabledGet(client_handle, ifNum, 
                                                        remIndex, timestamp, 
                                                        isEnabled)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote port's auto negotiation enabled"
           " status. (result = %d)\n", result);
  }
  else
  {
    printf("Remote port's auto negotiation enabled status is %u.\n", *isEnabled);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote port's auto negotiation advertized capability as a string.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    advrtCap      @b{(output)} Remote port auto negotiation advertized 
*                                      capability
*
* @returns  none
*
* @end
*********************************************************************/
void lldpdot3RemPortAutoNegAdvertizedCapGet(openapiClientHandle_t *client_handle, 
                                            uint32_t ifNum, uint32_t remIndex, 
                                            uint32_t timestamp, 
                                            open_buffdesc *advrtCap)
{
  open_error_t result;
  unsigned short advertizedCap = 0;

  if ((result = openapiLldpdot3RemPortAutoNegAdvertizedCapGet(client_handle, ifNum,
                                                              remIndex, timestamp,
                                                              advrtCap)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote port's auto negotiation "
           "advertized capability as a string. (result = %d)\n", result);
  }
  else
  {
    memcpy((void *)&advertizedCap, advrtCap->pstart, sizeof(unsigned short));  
    printf("Remote port's auto negotiation advertized capability is %d.\n",
           advertizedCap);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote port's auto negotiation operational MAU Type.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    type          @b{(output)} MAU type
*
* @returns  none
*
* @end
*********************************************************************/
void lldpdot3RemPortAutoNegOperMauTypeGet(openapiClientHandle_t *client_handle, 
                                          uint32_t ifNum, uint32_t remIndex, 
                                          uint32_t timestamp, uint32_t *type)
{
  open_error_t result;

  if ((result = openapiLldpdot3RemPortAutoNegOperMauTypeGet(client_handle, ifNum,
                                                            remIndex, timestamp, 
                                                            type)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote port's auto negotiation "
           "operational MAU Type. (result = %d)\n", result);
  }
  else
  {
    printf("Remote port's auto negotiation operational MAU Type is %u.\n",
           *type);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote port's auto negotiation operational MAU in string format.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mauType         @b{(input)}  MAU Type
* @param    mauString       @b{(output)} MAU type in string
*
* @returns  none
*
* @end
*********************************************************************/
void lldpdot3RemPortAutoNegOperMauStringGet(openapiClientHandle_t *client_handle, 
                                            uint32_t mauType, 
                                            open_buffdesc *mauString)
{
  open_error_t result;

  if ((result = openapiLldpdot3RemPortAutoNegOperMauStringGet(client_handle, 
                                                              mauType, mauString))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote port's auto negotiation "
           "operational MAU in string format. (result = %d)\n", result);
  }
  else
  {
    printf("Remote port's auto negotiation operational MAU is %s.\n",
           (char *)mauString->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote device's hardware revision.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    hardwareRev    @b{(output)} Remote device hardware revision
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemHardwareRevGet(openapiClientHandle_t *client_handle, 
                              uint32_t ifNum, uint32_t remIndex, 
                              uint32_t timestamp, open_buffdesc *hardwareRev)
{
  open_error_t result;

  if ((result = openapiLldpMedRemHardwareRevGet(client_handle, ifNum, remIndex, 
                                                timestamp, hardwareRev)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote device's hardware revision."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Remote device's hardware revision is %s.\n", (char *)hardwareRev->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote device's firmware revision.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    firmwareRev   @b{(output)} Remote device firmware revision
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemFirmwareRevGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                              uint32_t remIndex, uint32_t timestamp, 
                              open_buffdesc *firmwareRev)
{
  open_error_t result;

  if ((result = openapiLldpMedRemFirmwareRevGet(client_handle, ifNum, remIndex,
                                                timestamp, firmwareRev)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote device's firmware revision."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Remote device's firmware revision is %s.\n", (char *)firmwareRev->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote device's software revision.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    softwareRev   @b{(output)} Remote device software revision
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemSoftwareRevGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                              uint32_t remIndex, uint32_t timestamp, 
                              open_buffdesc *softwareRev)
{
  open_error_t result;

  if ((result = openapiLldpMedRemSoftwareRevGet(client_handle, ifNum, remIndex,
                                                timestamp, softwareRev)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote device's software revision."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Remote device's software revision is %s.\n", (char *)softwareRev->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote device's serial number.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    serialNum      @b{(output)} Remote device serial number
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemSerialNumGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                            uint32_t remIndex, uint32_t timestamp, 
                            open_buffdesc *serialNum)
{
  open_error_t result;

  if ((result = openapiLldpMedRemSerialNumGet(client_handle, ifNum, remIndex, 
                                              timestamp, serialNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote device's serial number."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Remote device's serial number is %s.\n", (char *)serialNum->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote device's asset ID.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    assetId       @b{(output)} Remote device asset ID
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemAssetIdGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                          uint32_t remIndex, uint32_t timestamp, 
                          open_buffdesc *assetId)
{
  open_error_t result;

  if ((result = openapiLldpMedRemAssetIdGet(client_handle, ifNum, remIndex, 
                                            timestamp, assetId)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote device's asset ID."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Remote device's asset ID is %s.\n", (char *)assetId->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote device's PoE device type connected to this port.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    deviceType    @b{(output)} Remote device POE device type
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemPoeDeviceTypeGet(openapiClientHandle_t *client_handle, 
                                uint32_t ifNum, uint32_t remIndex, 
                                uint32_t timestamp, uint32_t *deviceType)
{
  open_error_t result;

  if ((result = openapiLldpMedRemPoeDeviceTypeGet(client_handle, ifNum, remIndex, 
                                                  timestamp, deviceType)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote device's PoE device type connected"
           " to this port. (result = %d)\n", result);
  }
  else
  {
    printf("Remote device's PoE device type is %u.\n", *deviceType);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote ports PSE power value in tenths of watts.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    powerAv       @b{(output)} Available power
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemPoePSEPowerAvGet(openapiClientHandle_t *client_handle, 
                                uint32_t ifNum, uint32_t remIndex, 
                                uint32_t timestamp, uint32_t *powerAv)
{
  open_error_t result;

  if ((result = openapiLldpMedRemPoePSEPowerAvGet(client_handle, ifNum, 
                                                  remIndex, timestamp, powerAv))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote ports PSE power value in tenths"
           " of watts. (result = %d)\n", result);
  }
  else
  {
    printf("Remote ports PSE power value in tenths of watts is %u.\n", *powerAv);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote ports PSE power source.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    powerSrcType  @b{(output)} Power soruce type
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemPoePSEPowerSrcGet(openapiClientHandle_t *client_handle, 
                                 uint32_t ifNum, uint32_t remIndex, 
                                 uint32_t timestamp, uint32_t *powerSrcType)
{
  open_error_t result;

  if ((result = openapiLldpMedRemPoePSEPowerSrcGet(client_handle, ifNum, 
                                                   remIndex, timestamp, 
                                                   powerSrcType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote ports PSE power source."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Remote ports PSE power source is %u.\n", *powerSrcType);
  }
  return;
}

/*********************************************************************
* @purpose  Get remote ports PSE power priority.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    powerPri      @b{(output)} Power priority
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemPoePSEPowerPriGet(openapiClientHandle_t *client_handle, 
                                 uint32_t ifNum, uint32_t remIndex, 
                                 uint32_t timestamp, uint32_t *powerPri)
{
  open_error_t result;

  if ((result = openapiLldpMedRemPoePSEPowerPriGet(client_handle, ifNum, 
                                                   remIndex, timestamp, powerPri))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get remote ports PSE power priority."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Remote ports PSE power priority is %u.\n", *powerPri);
  }
  return;
}

/*********************************************************************
* @purpose  Get the location info received on this port for the given location 
*           subtype.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    subType        @b{(input)}  Loaction subtype
* @param    locInfo        @b{(output)} Location information
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemLocationInfoGet(openapiClientHandle_t *client_handle, 
                               uint32_t ifNum, uint32_t remIndex, 
                               uint32_t timestamp, uint32_t subType, 
                               open_buffdesc *locInfo)
{
  open_error_t result;

  if ((result = openapiLldpMedRemLocationInfoGet(client_handle, ifNum, remIndex,
                                                 timestamp, subType, locInfo)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the location info received on this"
           " port for the given location subtype. (result = %d)\n", result);
  }
  else
  {
    printf("Location info received on this port is %s.\n", (char *)locInfo->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get the vlan id associated with a particular policy type.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    policyAppType  @b{(input)}  Type of application policy
* @param    vlanId         @b{(output)} Location information
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemMediaPolicyVlanIdGet(openapiClientHandle_t *client_handle, 
                                    uint32_t ifNum, uint32_t remIndex, 
                                    uint32_t timestamp, 
                                    OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                    uint32_t *vlanId)
{
  open_error_t result;

  if ((result = openapiLldpMedRemMediaPolicyVlanIdGet(client_handle, ifNum, 
                                                      remIndex, timestamp, 
                                                      policyAppType, vlanId)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the vlan id associated with a"
           " particular policy type. (result = %d)\n", result);
  }
  else
  {
    printf("Vlan id associated with a given policy type is %u.\n", *vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Get the priority associated with a particular policy type.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    policyAppType  @b{(input)}  Type of application policy
* @param    priority       @b{(output)} Priority
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemMediaPolicyPriorityGet(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, uint32_t remIndex, 
                                      uint32_t timestamp, 
                                      OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                      uint32_t *priority)
{
  open_error_t result;

  if ((result = openapiLldpMedRemMediaPolicyPriorityGet(client_handle, ifNum, 
                                                        remIndex, timestamp, 
                                                        policyAppType, priority)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the priority associated with a "
           "particular policy type. (result = %d)\n", result);
  }
  else
  {
    printf("Priority associated with a given policy type is %u.\n", *priority);
  }
  return;
}

/*********************************************************************
* @purpose  Get the DSCP associated with a particular policy type.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    policyAppType @b{(input)}  Type of application policy
* @param    dscp          @b{(output)} DSCP value
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemMediaPolicyDscpGet(openapiClientHandle_t *client_handle, 
                                  uint32_t ifNum, uint32_t remIndex, 
                                  uint32_t timestamp, 
                                  OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                  uint32_t *dscp)
{
  open_error_t result;

  if ((result = openapiLldpMedRemMediaPolicyDscpGet(client_handle, ifNum, 
                                                    remIndex, timestamp, 
                                                    policyAppType, dscp)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the DSCP associated with a "
           "particular policy type. (result = %d)\n", result);
  }
  else
  {
    printf("Dscp associated with a given policy type is %u.\n", *dscp);
  }
  return;
}

/*********************************************************************
* @purpose  Get the unknown bit associated with a particular policy type.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    policyAppType @b{(input)}  Type of application policy
* @param    unknown       @b{(output)} Unknown bit status
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemMediaPolicyUnknownGet(openapiClientHandle_t *client_handle, 
                                     uint32_t ifNum, uint32_t remIndex, 
                                     uint32_t timestamp, 
                                     OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                     OPEN_BOOL_t *unknown)
{
  open_error_t result;

  if ((result = openapiLldpMedRemMediaPolicyUnknownGet(client_handle, ifNum, 
                                                       remIndex, timestamp, 
                                                       policyAppType, unknown)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the unknown bit associated "
           "with a particular policy type. (result = %d)\n", result);
  }
  else
  {
    printf("Unknown bit associated with a given policy type is %u.\n", *unknown);
  }
  return;
}

/*********************************************************************
* @purpose  Get the tagged bit associated with a particular policy type.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    policyAppType  @b{(input)}  Type of application policy
* @param    tagged         @b{(output)} Tagged bit status
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemMediaPolicyTaggedGet(openapiClientHandle_t *client_handle, 
                                    uint32_t ifNum, uint32_t remIndex, 
                                    uint32_t timestamp, 
                                    OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType, 
                                    OPEN_BOOL_t *tagged)
{
  open_error_t result;

  if ((result = openapiLldpMedRemMediaPolicyTaggedGet(client_handle, ifNum, 
                                                      remIndex, timestamp, 
                                                      policyAppType, tagged))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the tagged bit associated with "
           "a particular policy type. (result = %d)\n", result);
  }
  else
  {
    printf("Tagged bit associated with a given policy type is %u.\n", *tagged);
  }
  return;
}

/*********************************************************************
* @purpose  Get the next unknown TLV entry for the indicated remote data entry.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    tlvType        @b{(input)}  TLV type
* @param    nextTlvType    @b{(output)} Next TLV type
*
* @returns  none
*
* @end
*********************************************************************/
void lldpRemUnknownTLVEntryNextGet(openapiClientHandle_t *client_handle, 
                                   uint32_t ifNum, uint32_t remIndex, 
                                   uint32_t timestamp, uint32_t tlvType, 
                                   uint32_t *nextTlvType)
{
  open_error_t result;

  if ((result = openapiLldpRemUnknownTLVEntryNextGet(client_handle, ifNum, 
                                                     remIndex, timestamp, 
                                                     tlvType, nextTlvType)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the next unknown TLV entry for "
           "the indicated remote data entry. (result = %d)\n", result);
  }
  else
  {
    printf("Next unknown TLV entry for the indicated remote data entry is %u.\n",
           *nextTlvType);
  }
  return;
}

/*********************************************************************
* @purpose  Determine if unknown TLV entry exists.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    tlvType        @b{(input)}  TLV type
* @param    isExists       @b{(output)} Entry existtance status
*
* @returns  none
*
* @end
*********************************************************************/
void lldpRemUnknownTLVEntryIsExists(openapiClientHandle_t *client_handle, 
                                    uint32_t ifNum, uint32_t remIndex, 
                                    uint32_t timestamp, uint32_t tlvType, 
                                    OPEN_BOOL_t *isExists)
{
  open_error_t result;

  if ((result = openapiLldpRemUnknownTLVEntryIsExists(client_handle, ifNum, 
                                                      remIndex, timestamp, tlvType,
                                                      isExists)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to determine if unknown TLV entry exists."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Unknwon TLV entry existance status is %u.\n", *isExists);
  }
  return;
}

/*********************************************************************
* @purpose  Get the TLV info for the specified remote entry.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    tlvType        @b{(input)}  TLV type
* @param    info           @b{(output)} Organization defined info
*
* @returns  none
*
* @end
*********************************************************************/
void lldpRemUnknownTLVInfoGet(openapiClientHandle_t *client_handle, 
                              uint32_t ifNum, uint32_t remIndex, 
                              uint32_t timestamp, uint32_t tlvType, 
                              open_buffdesc *info)
{
  open_error_t result;

  if ((result = openapiLldpRemUnknownTLVInfoGet(client_handle, ifNum, 
                                                remIndex, timestamp, tlvType,
                                                info)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the TLV info for the specified"
           " remote entry. (result = %d)\n", result);
  }
  else
  {
    printf("TLV info for the specified remote entry is %s.\n",(char *) info->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get the remote device's manufacturer's name.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    mfgName        @b{(output)} Manufacturer's name as a string
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemMfgNameGet(openapiClientHandle_t *client_handle, 
                          uint32_t ifNum, uint32_t remIndex, 
                          uint32_t timestamp, open_buffdesc *mfgName)
{
  open_error_t result;

  if ((result = openapiLldpMedRemMfgNameGet(client_handle, ifNum, remIndex,
                                            timestamp, mfgName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the remote device's manufacturer's"
           " name. (result = %d)\n", result);
  }
  else
  {
    printf("Remote device's manufacturer's name is %s.\n",(char *) mfgName->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get the remote device's model name.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    modelName      @b{(output)} Model name as a string
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemModelNameGet(openapiClientHandle_t *client_handle, 
                            uint32_t ifNum, uint32_t remIndex, 
                            uint32_t timestamp, open_buffdesc *modelName)
{
  open_error_t result;

  if ((result = openapiLldpMedRemModelNameGet(client_handle, ifNum, 
                                              remIndex, timestamp, modelName))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the remote device's model name."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Remote device's model name is %s.\n",(char *) modelName->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get the remote port's PD power requirement.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    powerReq      @b{(output)} Power requirement
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemPoePDPowerReqGet(openapiClientHandle_t *client_handle, 
                                uint32_t ifNum, uint32_t remIndex, 
                                uint32_t timestamp, uint32_t *powerReq)
{
  open_error_t result;

  if ((result = openapiLldpMedRemPoePDPowerReqGet(client_handle, ifNum, 
                                                  remIndex, timestamp, powerReq))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the remote port's PD power requirement."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Remote port's PD power requirement is %u.\n", *powerReq);
  }
  return;
}

/*********************************************************************
* @purpose  Get the remote port's PD power source.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    powerSrc       @b{(output)} Power source
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemPoePDPowerSrcGet(openapiClientHandle_t *client_handle, 
                                uint32_t ifNum, uint32_t remIndex, 
                                uint32_t timestamp, uint32_t *powerSrc)
{
  open_error_t result;

  if ((result = openapiLldpMedRemPoePDPowerSrcGet(client_handle, ifNum, 
                                                  remIndex, timestamp, powerSrc))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the remote port's PD power source."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Remote port's PD power source is %u.\n", *powerSrc);
  }
  return;
}

/*********************************************************************
* @purpose  Get the remote port's PD power priority.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    powerPri       @b{(output)} Power priority
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedRemPoePDPowerPriGet(openapiClientHandle_t *client_handle, 
                                uint32_t ifNum, uint32_t remIndex, 
                                uint32_t timestamp, uint32_t *powerPri)
{
  open_error_t result;

  if ((result = openapiLldpMedRemPoePDPowerPriGet(client_handle, ifNum, 
                                                  remIndex, timestamp, powerPri))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the remote port's PD power priority."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Remote port's PD power priority is %u.\n", *powerPri);
  }
  return;
}

/*********************************************************************
* @purpose  Get the organizationally defined OUI for specified entry.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    remIndex       @b{(input)}  Index associated with this remote entry
* @param    timestamp      @b{(input)}  Time Stamp associated with this remote entry
* @param    infoIndex      @b{(input)}  Info index
* @param    oui            @b{(output)} Organization unique identifier in XX:XX:XX format
*
* @returns  none
*
* @end
*********************************************************************/
void lldpRemOrgDefOUIGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                         uint32_t remIndex, uint32_t timestamp, uint32_t infoIndex,
                         open_buffdesc *oui)
{
  open_error_t result;

  if ((result = openapiLldpRemOrgDefOUIGet(client_handle, ifNum, remIndex, 
                                           timestamp, infoIndex, oui)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the organizationally defined OUI"
           " for specified entry. (result = %d)\n", result);
  }
  else
  {
    printf("Organizationally defined OUI for specified entry is %s.\n",
          (char *) oui->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get the organizationally defined subtype for specified entry.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    remIndex      @b{(input)}  Index associated with this remote entry
* @param    timestamp     @b{(input)}  Time Stamp associated with this remote entry
* @param    infoIndex     @b{(input)}  Info index
* @param    subtype       @b{(output)} Organization defined subtype
*
* @returns  none
*
* @end
*********************************************************************/
void lldpRemOrgDefSubtypeGet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                             uint32_t remIndex, uint32_t timestamp, 
                             uint32_t infoIndex, uint32_t *subtype)
{
  open_error_t result;

  if ((result = openapiLldpRemOrgDefSubtypeGet(client_handle, ifNum, remIndex, 
                                               timestamp, infoIndex, subtype)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the organizationally defined subtype "
           "for specified entry. (result = %d)\n", result);
  }
  else
  {
    printf("Organizationally defined subtype for specified entry is %u.\n",
           *subtype);
  }
  return;
}

/*********************************************************************
* @purpose  Get a string representation for Chassis ID Subtype.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    subtype         @b{(input)}  Chassis ID subtype
* @param    subtypeStr      @b{(output)} Subtype string
*
* @returns  none
*
* @end
*********************************************************************/
void lldpChassisIdSubtypeStringGet(openapiClientHandle_t *client_handle, 
                                   OPEN_LLDP_CHASSIS_ID_SUBTYPE_t subtype, open_buffdesc *subtypeStr)
{
  open_error_t result;

  if ((result = openapiLldpChassisIdSubtypeStringGet(client_handle, subtype, 
                                                     subtypeStr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get a string representation for Chassis"
           " ID Subtype. (result = %d)\n", result);
  }
  else
  {
    printf("String representation for Chassis ID Subtype is %s.\n", 
          (char *) subtypeStr->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get a string representation for port ID Subtype.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    subtype          @b{(input)}  Port ID subtype
* @param    subtypeStr       @b{(output)} Subtype string
*
* @returns  none
*
* @end
*********************************************************************/
void lldpPortIdSubtypeStringGet(openapiClientHandle_t *client_handle, 
                                uint32_t subtype, open_buffdesc *subtypeStr)
{
  open_error_t result;

  if ((result = openapiLldpPortIdSubtypeStringGet(client_handle, subtype, 
                                                  subtypeStr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get a string representation for port ID "
           "Subtype. (result = %d)\n", result);
  }
  else
  {
    printf("String representation for port ID Subtype is %s.\n",
           (char *) subtypeStr->pstart);
  }
  return;
}

/*********************************************************************
* @purpose  Get the port's auto negotiation capability.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
* @param    isCapable       @b{(output)} Interface auto negotiation capability 
*                                        status
*
* @returns  none
*
* @end
*********************************************************************/
void lldpdot3LocPortAutoNegSupportedGet(openapiClientHandle_t *client_handle, 
                                        uint32_t ifNum, OPEN_BOOL_t *isCapable)
{
  open_error_t result;

  if ((result = openapiLldpdot3LocPortAutoNegSupportedGet(client_handle, ifNum,
                                                          isCapable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the port's auto negotiation"
           " capability. (result = %d)\n", result);
  }
  else
  {
    printf("Port's auto negotiation capability is %u.\n", *isCapable);
  }
  return;
}

/*********************************************************************
* @purpose  Get port's auto negotiation enabled status.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
* @param    isEnabled       @b{(output)} Interface auto negotiation enabled status
*
* @returns  none
*
* @end
*********************************************************************/
void lldpdot3LocPortAutoNegEnabledGet(openapiClientHandle_t *client_handle, 
                                      uint32_t ifNum, OPEN_BOOL_t *isEnabled)
{
  open_error_t result;

  if ((result = openapiLldpdot3LocPortAutoNegEnabledGet(client_handle, ifNum, 
                                                        isEnabled)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get port's auto negotiation enabled status."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Port's auto negotiation enabled status is %u.\n", *isEnabled);
  }
  return;
}

/*********************************************************************
* @purpose  Get port's auto negotiation advertized capability as a string.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    advrtCap       @b{(output)} Interface auto negotiation advertized
*                                       capability
*
* @returns  none
*
* @end
*********************************************************************/
void lldpdot3LocPortAutoNegAdvertizedCapGet(openapiClientHandle_t *client_handle, 
                                            uint32_t ifNum, open_buffdesc *advrtCap)
{
  open_error_t result;
  unsigned short advertizedCap = 0; 

  if ((result = openapiLldpdot3LocPortAutoNegAdvertizedCapGet(client_handle, ifNum,
                                                              advrtCap)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to get port's auto negotiation advertized"
           " capability as a string. (result = %d)\n", result);
  }
  else
  {
    memcpy((void *)&advertizedCap, advrtCap->pstart, sizeof(unsigned short));
    printf("Port's auto negotiation advertized capability is %d.\n", advertizedCap);
  }

  return;
}

/*********************************************************************
* @purpose  Get port's auto negotiation operational MAU Type.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    type           @b{(output)} MAU type
*
* @returns  none
*
* @end
*********************************************************************/
void lldpdot3LocPortAutoNegOperMauTypeGet(openapiClientHandle_t *client_handle, 
                                          uint32_t ifNum, uint32_t *type)
{
  open_error_t result;

  if ((result = openapiLldpdot3LocPortAutoNegOperMauTypeGet(client_handle, ifNum,
                                                            type)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get port's auto negotiation operational"
           " MAU Type. (result = %d)\n", result);
  }
  else
  {
    printf("Port's auto negotiation operational MAU Type is %u.\n", *type);
  }
  return;
}

/*********************************************************************
* @purpose  Get local device's MED Classification
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    deviceClass       @b{(output)}  Device class
*
* @returns  none
*
* @end
*********************************************************************/
void lldpMedLocDeviceClassGet(openapiClientHandle_t *client_handle, 
                              uint32_t *deviceClass)
{
  open_error_t result;

  if ((result = openapiLldpMedLocDeviceClassGet(client_handle, deviceClass)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get local device's MED Classification"
           " (result = %d)\n", result);
  }
  else
  {
    printf("Local device's MED Classification is %u.\n", *deviceClass);
  }
  return;
}

/*********************************************************************
* @purpose  This function sanity checks all LLDP OpEN APIs. All LLDP OpEN APIs are 
*           called with possible NULL and invalid parameters to check the API robustness.
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  ID of the interface
*
* @returns  none
*
* @end
*********************************************************************/
void lldpOpENAPIsTestSanity(openapiClientHandle_t *clientHandle, uint32_t ifNum)
{
  open_error_t result;
  OPEN_CONTROL_t controlMode;
  uint32_t invalidControlMode = 10;
  uint32_t lldpMgmtStrLen, strLen, ttl;
  uint32_t invalidTLVType = 10;
  uint32_t nextIfNum, remIndex, timeStamp;
  OPEN_LLDP_CHASSIS_ID_SUBTYPE_t chassisIdSubtype;
  open_buffdesc buffDesc;
  char *str;
  OPEN_LLDP_PORT_ID_SUBTYPE_t portIdSubtype;
  OPEN_LLDP_IANA_ADDR_FAMILY_t  addressFamily;
  uint8_t addrLen;
  OPEN_BOOL_t boolMode;
  OPEN_LLDP_MED_CAPABILITIES_t cap;
  uint32_t count;
  uint32_t priority;
  uint32_t dscp;
  uint32_t interval;
  uint32_t holdCount;
  uint32_t vlanId;
  uint32_t delay;
  OPEN_BOOL_t mode;
  OPEN_LLDPXMED_POLICY_APPTYPE_VALUE_t policyAppType = 1;
  OPEN_BOOL_t taggedStatus;

  printf("Testing LLDP OpEN APIs sanity:\n\n");

  if ((result = openapiLldpMgmtStrLenGet(clientHandle, &lldpMgmtStrLen)) != OPEN_E_NONE)
  {
    printf("Bad return code. (result = %d)\n", result);
    return;
  }

  if ((str = (char*)malloc(lldpMgmtStrLen)) == NULL) 
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, lldpMgmtStrLen);

  /* openapiLldpMgmtStrLenGet() */
  printf("Testing openapiLldpMgmtStrLenGet():\n");

  result = openapiLldpMgmtStrLenGet(NULL, &strLen);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMgmtStrLenGet(clientHandle, NULL);
  printf("NULL String Length:(result = %d)\n", result);

  printf("openapiLldpMgmtStrLenGet() sanity successful\n\n");

  /* openapiLldpIntfTransmitModeSet() */
  printf("Testing openapiLldpIntfTransmitModeSet():\n");

  result = openapiLldpIntfTransmitModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfTransmitModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfTransmitModeSet() sanity successful\n\n");

  /* openapiLldpIntfTransmitModeGet() */
  printf("Testing openapiLldpIntfTransmitModeGet():\n");

  result = openapiLldpIntfTransmitModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfTransmitModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfTransmitModeGet() sanity successful\n\n");

  /* openapiLldpIntfReceiveModeSet() */
  printf("Testing openapiLldpIntfReceiveModeSet():\n");

  result = openapiLldpIntfReceiveModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfReceiveModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfReceiveModeSet() sanity successful\n\n");

  /* openapiLldpIntfReceiveModeGet() */
  printf("Testing openapiLldpIntfReceiveModeGet():\n");

  result = openapiLldpIntfReceiveModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfReceiveModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfReceiveModeGet() sanity successful\n\n");

  /* openapiLldpIntfTLVTransmitModeSet() */
  printf("Testing openapiLldpIntfTLVTransmitModeSet():\n");

  result = openapiLldpIntfTLVTransmitModeSet(NULL, ifNum, OPEN_LLDP_SYS_DESC_TLV, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfTLVTransmitModeSet(clientHandle, ifNum, invalidTLVType, OPEN_ENABLE);
  printf("Invalid TLV type:(result = %d)\n", result);

  result = openapiLldpIntfTLVTransmitModeSet(clientHandle, ifNum, OPEN_LLDP_SYS_DESC_TLV, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfTLVTransmitModeSet() sanity successful\n\n");

  /* openapiLldpIntfTLVTransmitModeGet() */
  printf("Testing openapiLldpIntfTLVTransmitModeGet():\n");

  result = openapiLldpIntfTLVTransmitModeGet(NULL, ifNum, OPEN_LLDP_SYS_DESC_TLV, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfTLVTransmitModeGet(clientHandle, ifNum, OPEN_LLDP_SYS_DESC_TLV, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfTLVTransmitModeGet() sanity successful\n\n");

  /* openapiLldpIntfNotificationModeSet() */
  printf("Testing openapiLldpIntfNotificationModeSet():\n");

  result = openapiLldpIntfNotificationModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfNotificationModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfNotificationModeSet() sanity successful\n\n");

  /* openapiLldpIntfNotificationModeGet() */
  printf("Testing openapiLldpIntfNotificationModeGet():\n");

  result = openapiLldpIntfNotificationModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfNotificationModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiLldpIntfNotificationModeGet() sanity successful\n\n");

  /* openapiLldpIntfNextGet() */
  printf("Testing openapiLldpIntfNextGet():\n");

  result = openapiLldpIntfNextGet(NULL, ifNum, &nextIfNum);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfNextGet(clientHandle, ifNum, NULL);
  printf("NULL Next Interface Number:(result = %d)\n", result);

  printf("openapiLldpIntfNextGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceNextGet() */
  printf("Testing openapiLldpIntfRemoteDeviceNextGet():\n");

  result = openapiLldpIntfRemoteDeviceNextGet(NULL, ifNum, &remIndex, &timeStamp);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceNextGet(clientHandle, ifNum, NULL, &timeStamp);
  printf("NULL Remote Index:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceNextGet(clientHandle, ifNum, &remIndex, NULL);
  printf("NULL Time Stamp:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceNextGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceChassisIdGet() */
  printf("Testing openapiLldpIntfRemoteDeviceChassisIdGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceChassisIdGet(NULL, ifNum, remIndex, timeStamp,
                                                   &chassisIdSubtype, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceChassisIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                   NULL, &buffDesc);
  printf("NULL Chassis Id Subtype:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceChassisIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                   &chassisIdSubtype, NULL);
  printf("NULL Chassis Id:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceChassisIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                   &chassisIdSubtype, &buffDesc);
  printf("NULL Chassis Id buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceChassisIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                   &chassisIdSubtype, &buffDesc);
  printf("NULL Chassis Id buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceChassisIdGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDevicePortIdGet() */
  printf("Testing openapiLldpIntfRemoteDevicePortIdGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDevicePortIdGet(NULL, ifNum, remIndex, timeStamp,
                                                &portIdSubtype, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDevicePortIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                NULL, &buffDesc);
  printf("NULL Port Id Subtype:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDevicePortIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                &portIdSubtype, NULL);
  printf("NULL Port Id:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDevicePortIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                &portIdSubtype, &buffDesc);
  printf("NULL Port Id buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDevicePortIdGet(clientHandle, ifNum, remIndex, timeStamp,
                                                &portIdSubtype, &buffDesc);
  printf("NULL Port Id buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDevicePortIdGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceSystemNameGet() */
  printf("Testing openapiLldpIntfRemoteDeviceSystemNameGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceSystemNameGet(NULL, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceSystemNameGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL System Name:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceSystemNameGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL System Name buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceSystemNameGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL System Name buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDevicePortIdGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceSystemDescGet() */
  printf("Testing openapiLldpIntfRemoteDeviceSystemDescGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceSystemDescGet(NULL, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceSystemDescGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL System Description:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceSystemDescGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL System Description buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceSystemDescGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL System Description buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceSystemDescGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDevicePortDescGet() */
  printf("Testing openapiLldpIntfRemoteDevicePortDescGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDevicePortDescGet(NULL, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDevicePortDescGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL Port Description:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDevicePortDescGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Port Description buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDevicePortDescGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Port Description buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceSystemDescGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceSysCapSuppGet() */
  printf("Testing openapiLldpIntfRemoteDeviceSysCapSuppGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceSysCapSuppGet(NULL, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceSysCapSuppGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL Supported Sys Capabilities:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceSysCapSuppGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Supported Sys Capabilities buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceSysCapSuppGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Supported Sys Capabilities buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceSysCapSuppGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceSysCapEnabledGet() */
  printf("Testing openapiLldpIntfRemoteDeviceSysCapEnabledGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceSysCapEnabledGet(NULL, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceSysCapEnabledGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL Enabled Sys Capabilities:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceSysCapEnabledGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Enabled Sys Capabilities buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceSysCapEnabledGet(clientHandle, ifNum, remIndex, timeStamp, &buffDesc);
  printf("NULL Enabled Sys Capabilities buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceSysCapEnabledGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceMgmtAddressNextGet() */
  printf("Testing openapiLldpIntfRemoteDeviceMgmtAddressNextGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfRemoteDeviceMgmtAddressNextGet(NULL, ifNum, remIndex, timeStamp, 
                                                         &addressFamily, &buffDesc, &addrLen);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceMgmtAddressNextGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                         NULL, &buffDesc, &addrLen);
  printf("NULL Address Family:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceMgmtAddressNextGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                         &addressFamily, &buffDesc, NULL);
  printf("NULL Address Length:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfRemoteDeviceMgmtAddressNextGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                         &addressFamily, &buffDesc, &addrLen);
  printf("NULL Address buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfRemoteDeviceMgmtAddressNextGet(clientHandle, ifNum, remIndex, timeStamp, 
                                                         &addressFamily, &buffDesc, &addrLen);
  printf("NULL Address buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceMgmtAddressNextGet() sanity successful\n\n");

  /* openapiLldpIntfRemoteDeviceTTLGet() */
  printf("Testing openapiLldpIntfRemoteDeviceTTLGet():\n");

  result = openapiLldpIntfRemoteDeviceTTLGet(NULL, ifNum, remIndex, timeStamp, &ttl);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfRemoteDeviceTTLGet(clientHandle, ifNum, remIndex, timeStamp, NULL);
  printf("NULL TTL:(result = %d)\n", result);

  printf("openapiLldpIntfRemoteDeviceTTLGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceChassisIdGet() */
  printf("Testing openapiLldpIntfLocalDeviceChassisIdGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceChassisIdGet(NULL, ifNum, &chassisIdSubtype, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceChassisIdGet(clientHandle, ifNum, NULL, &buffDesc);
  printf("NULL Chassis Id Subtype:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceChassisIdGet(clientHandle, ifNum, &chassisIdSubtype, NULL);
  printf("NULL Chassis Id:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceChassisIdGet(clientHandle, ifNum, &chassisIdSubtype, &buffDesc);
  printf("NULL Chassis Id buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceChassisIdGet(clientHandle, ifNum, &chassisIdSubtype, &buffDesc);
  printf("NULL Chassis Id buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceChassisIdGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDevicePortIdGet() */
  printf("Testing openapiLldpIntfLocalDevicePortIdGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDevicePortIdGet(NULL, ifNum, &portIdSubtype, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDevicePortIdGet(clientHandle, ifNum, NULL, &buffDesc);
  printf("NULL Port Id Subtype:(result = %d)\n", result);

  result = openapiLldpIntfLocalDevicePortIdGet(clientHandle, ifNum, &portIdSubtype, NULL);
  printf("NULL Port Id:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDevicePortIdGet(clientHandle, ifNum, &portIdSubtype, &buffDesc);
  printf("NULL Port Id buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDevicePortIdGet(clientHandle, ifNum, &portIdSubtype, &buffDesc);
  printf("NULL Port Id buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDevicePortIdGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceSystemNameGet() */
  printf("Testing openapiLldpIntfLocalDeviceSystemNameGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceSystemNameGet(NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceSystemNameGet(clientHandle, ifNum, NULL);
  printf("NULL System Name:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceSystemNameGet(clientHandle, ifNum, &buffDesc);
  printf("NULL System Name buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceSystemNameGet(clientHandle, ifNum, &buffDesc);
  printf("NULL System Name buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceSystemNameGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceSystemDescGet() */
  printf("Testing openapiLldpIntfLocalDeviceSystemDescGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceSystemDescGet(NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceSystemDescGet(clientHandle, ifNum, NULL);
  printf("NULL System Description:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceSystemDescGet(clientHandle, ifNum, &buffDesc);
  printf("NULL System Description buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceSystemDescGet(clientHandle, ifNum, &buffDesc);
  printf("NULL System Description buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceSystemDescGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDevicePortDescGet() */
  printf("Testing openapiLldpIntfLocalDevicePortDescGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDevicePortDescGet(NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDevicePortDescGet(clientHandle, ifNum, NULL);
  printf("NULL Port Description:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDevicePortDescGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Port Description buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDevicePortDescGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Port Description buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDevicePortDescGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceSysCapSuppGet() */
  printf("Testing openapiLldpIntfLocalDeviceSysCapSuppGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceSysCapSuppGet(NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceSysCapSuppGet(clientHandle, ifNum, NULL);
  printf("NULL Supported Sys Capabilities:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceSysCapSuppGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Supported Sys Capabilities buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceSysCapSuppGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Supported Sys Capabilities buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceSysCapSuppGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceSysCapEnabledGet() */
  printf("Testing openapiLldpIntfLocalDeviceSysCapEnabledGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceSysCapEnabledGet(NULL, ifNum, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceSysCapEnabledGet(clientHandle, ifNum, NULL);
  printf("NULL Enabled Sys Capabilities:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceSysCapEnabledGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Enabled Sys Capabilities buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceSysCapEnabledGet(clientHandle, ifNum, &buffDesc);
  printf("NULL Enabled Sys Capabilities buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceSysCapEnabledGet() sanity successful\n\n");

  /* openapiLldpIntfLocalDeviceMgmtAddressGet() */
  printf("Testing openapiLldpIntfLocalDeviceMgmtAddressGet():\n");

  buffDesc.pstart = str;
  buffDesc.size = lldpMgmtStrLen;

  result = openapiLldpIntfLocalDeviceMgmtAddressGet(NULL, ifNum, &addressFamily, &buffDesc);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpIntfLocalDeviceMgmtAddressGet(clientHandle, ifNum, NULL, &buffDesc);
  printf("NULL Address Family:(result = %d)\n", result);

  buffDesc.pstart = NULL;
  result = openapiLldpIntfLocalDeviceMgmtAddressGet(clientHandle, ifNum, &addressFamily, &buffDesc);
  printf("NULL Address buffer:(result = %d)\n", result);

  buffDesc.pstart = str;
  buffDesc.size = 0;

  result = openapiLldpIntfLocalDeviceMgmtAddressGet(clientHandle, ifNum, &addressFamily, &buffDesc);
  printf("NULL Address buffer length:(result = %d)\n", result);

  printf("openapiLldpIntfLocalDeviceMgmtAddressGet() sanity successful\n\n");

  /* openapiLldpMedIntfModeGet() */
  printf("Testing openapiLldpMedIntfModeGet():\n");

  result = openapiLldpMedIntfModeGet(NULL, ifNum, &controlMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfModeGet(clientHandle, ifNum, NULL);
  printf("NULL Control Mode:(result = %d)\n", result);

  printf("openapiLldpMedIntfModeGet() sanity successful\n\n");

  /* openapiLldpMedIntfModeSet() on an LLDP-MED valid interface */
  printf("Testing openapiLldpMedIntfModeSet() on an LLDP-MED valid interface:\n");

  result = openapiLldpMedIntfModeSet(NULL, ifNum, OPEN_ENABLE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Control Mode:(result = %d)\n", result);

  printf("openapiLldpMedIntfModeSet() sanity successful on an LLDP-MED valid interface\n\n");

  /* openapiLldpMedIntfConfigNotificationModeGet() */
  printf("Testing openapiLldpMedIntfConfigNotificationModeGet():\n");

  result = openapiLldpMedIntfConfigNotificationModeGet(NULL, ifNum, &boolMode);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfConfigNotificationModeGet(clientHandle, ifNum, NULL);
  printf("NULL Bool Mode:(result = %d)\n", result);

  printf("openapiLldpMedIntfConfigNotificationModeGet() sanity successful\n\n");

  /* openapiLldpMedIntfConfigNotificationModeSet() on an LLDP-MED valid interface */
  printf("Testing openapiLldpMedIntfConfigNotificationModeSet() on an LLDP-MED valid interface:\n");

  result = openapiLldpMedIntfConfigNotificationModeSet(NULL, ifNum, OPEN_TRUE);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfConfigNotificationModeSet(clientHandle, ifNum, invalidControlMode);
  printf("Invalid Bool Mode:(result = %d)\n", result);

  printf("openapiLldpMedIntfConfigNotificationModeSet() sanity successful on an LLDP-MED valid interface\n\n");

  /* openapiLldpMedIntfTransmitTlvGet() */
  printf("Testing openapiLldpMedIntfTransmitTlvGet():\n");

  result = openapiLldpMedIntfTransmitTlvGet(NULL, ifNum, &cap);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfTransmitTlvGet(clientHandle, ifNum, NULL);
  printf("NULL TLV:(result = %d)\n", result);

  printf("openapiLldpMedIntfTransmitTlvGet() sanity successful\n\n");

  /* openapiLldpMedIntfTransmitTlvSet() on an LLDP-MED valid interface */
  printf("Testing openapiLldpMedIntfTransmitTlvSet() on an LLDP-MED valid interface:\n");

  result = openapiLldpMedIntfTransmitTlvSet(NULL, ifNum, &cap);
  printf("NULL Client Handle:(result = %d)\n", result);

  result = openapiLldpMedIntfTransmitTlvSet(clientHandle, ifNum, NULL);
  printf("NULL TLV:(result = %d)\n", result);

  printf("openapiLldpMedIntfTransmitTlvSet() sanity successful on an LLDP-MED valid interface\n\n");

  free(str);
  
  printf("Testing openapiLldpTxIntervalGet():\n");
  result = openapiLldpTxIntervalGet(NULL, &interval);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpTxIntervalGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiLldpTxIntervalSet():\n");
  result = openapiLldpTxIntervalSet(NULL, interval);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpTxIntervalSet(clientHandle, OPEN_LLDP_TX_INTERVAL_MIN - 1);
  printf("less than minimum value argument 2:(result = %d)\n", result);
  result = openapiLldpTxIntervalSet(clientHandle, OPEN_LLDP_TX_INTERVAL_MAX + 1);
  printf("greater than maximum value argument 2:(result = %d)\n", result);

  printf("Testing openapiLldpTxHoldGet():\n");
  result = openapiLldpTxHoldGet(NULL, &holdCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpTxHoldGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiLldpTxHoldSet():\n");
  result = openapiLldpTxHoldSet(NULL, holdCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpTxHoldSet(clientHandle, OPEN_LLDP_TX_HOLD_MULTIPLIER_MIN - 1);
  printf("less than minimum value argument 2:(result = %d)\n", result);
  result = openapiLldpTxHoldSet(clientHandle, OPEN_LLDP_TX_HOLD_MULTIPLIER_MAX + 1);
  printf("greater than maximum value argument 2:(result = %d)\n", result);

  printf("Testing openapiLldpTxReinitDelayGet():\n");
  result = openapiLldpTxReinitDelayGet(NULL, &delay);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpTxReinitDelayGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiLldpTxReinitDelaySet():\n");
  result = openapiLldpTxReinitDelaySet(NULL, delay);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpTxReinitDelaySet(clientHandle, OPEN_LLDP_REINIT_DELAY_MIN - 1);
  printf("less than minimum value argument 2:(result = %d)\n", result);
  result = openapiLldpTxReinitDelaySet(clientHandle, OPEN_LLDP_REINIT_DELAY_MAX + 1);
  printf("greater than maximum value argument 2:(result = %d)\n", result);

  printf("Testing openapiLldpNotificationIntervalGet():\n");
  result = openapiLldpNotificationIntervalGet(NULL, &interval);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpNotificationIntervalGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiLldpNotificationIntervalSet():\n");
  result = openapiLldpNotificationIntervalSet(NULL, interval);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpNotificationIntervalSet(clientHandle, OPEN_LLDP_NOTIFY_INTERVAL_MIN - 1);
  printf("less than minimum value argument 2:(result = %d)\n", result);
  result = openapiLldpNotificationIntervalSet(clientHandle, OPEN_LLDP_NOTIFY_INTERVAL_MAX + 1);
  printf("greater than maximum value argument 2:(result = %d)\n", result);

  printf("Testing openapiLldpXMedFastStartRepeatCountGet():\n");
  result = openapiLldpXMedFastStartRepeatCountGet(NULL, &count);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpXMedFastStartRepeatCountGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiLldpXMedFastStartRepeatCountSet():\n");
  result = openapiLldpXMedFastStartRepeatCountSet(NULL, count);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpXMedFastStartRepeatCountSet(clientHandle, OPEN_LLDP_MED_FASTSTART_REPEAT_COUNT_MIN - 1);
  printf("less than minimum value argument 2:(result = %d)\n", result);
  result = openapiLldpXMedFastStartRepeatCountSet(clientHandle, OPEN_LLDP_MED_FASTSTART_REPEAT_COUNT_MAX + 1);
  printf("greater than maximum value argument 2:(result = %d)\n", result);

  printf("Testing openapiLldpIntfTxMgmtAddrModeGet():\n");
  result = openapiLldpIntfTxMgmtAddrModeGet(NULL, ifNum, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpIntfTxMgmtAddrModeGet(clientHandle, ifNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiLldpXMedLocMediaPolicyVlanIDGet():\n");
  result = openapiLldpXMedLocMediaPolicyVlanIDGet(NULL, ifNum, policyAppType, &vlanId);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpXMedLocMediaPolicyVlanIDGet(clientHandle, ifNum, policyAppType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiLldpXMedLocMediaPolicyTaggedGet():\n");
  result = openapiLldpXMedLocMediaPolicyTaggedGet(NULL, ifNum, policyAppType, &taggedStatus);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpXMedLocMediaPolicyTaggedGet(clientHandle, ifNum, policyAppType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiLldpXMedLocMediaPolicyPriorityGet():\n");
  result = openapiLldpXMedLocMediaPolicyPriorityGet(NULL, ifNum, policyAppType, &priority);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpXMedLocMediaPolicyPriorityGet(clientHandle, ifNum, policyAppType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiLldpXMedLocMediaPolicyDscpGet():\n");
  result = openapiLldpXMedLocMediaPolicyDscpGet(NULL, ifNum, policyAppType, &dscp);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiLldpXMedLocMediaPolicyDscpGet(clientHandle, ifNum, policyAppType, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for LLDP.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t ifNum, testNum, arg1, arg2, val, remIndex, timestamp, policyAppType;
  open_buffdesc switch_os_revision, bufDesc;
  char switch_os_revision_string[100];
  char buf[256] = {0};
  OPEN_BOOL_t status;

  bufDesc.pstart = buf;
  bufDesc.size   = 256;

  if (argc < 2)
  {
    printLldpAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);
  
  if (argc > 2)
  {
    ifNum = atoi(argv[2]);
  }

  if ((argc > 4) && (testNum > 35) && (testNum < 66))
  {
    ifNum     = atoi(argv[2]);
    remIndex  = atoi(argv[3]);
    timestamp = atoi(argv[4]);

    if (argc > 5)
    {
      policyAppType = atoi(argv[5]);  
    }
  }
  
  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("lldp_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting LLDP API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");
  /* exercise various OPEN API LLDP functions */

  switch (testNum)
  {
    case 1:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      lldpIntfTransmitModeSet(&clientHandle, ifNum, arg1);
      break;

    case 2:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfTransmitModeGet(&clientHandle, ifNum);
      break;

    case 3:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      lldpIntfReceiveModeSet(&clientHandle, ifNum, arg1);
      break;

    case 4:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfReceiveModeGet(&clientHandle, ifNum);
      break;
    
    case 5:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      arg2 = atoi(argv[4]);
      lldpIntfTLVTransmitModeSet(&clientHandle, ifNum, arg1, arg2);
      break;

    case 6:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfTLVTransmitModeGet(&clientHandle, ifNum);
      break;
  
    case 7:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      lldpIntfNotificationModeSet(&clientHandle, ifNum, arg1);
      break;
  
    case 8:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfNotificationModeGet(&clientHandle, ifNum);
      break;
  
    case 9:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfRemoteDevicesGet(&clientHandle, ifNum);
      break;

    case 10:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfLocalDeviceGet(&clientHandle, ifNum);
      break;

    case 11:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedIntfModeGet(&clientHandle, ifNum);
      break;

    case 12:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      lldpMedIntfModeSet(&clientHandle, ifNum, arg1);
      break;

    case 13:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedIntfConfigNotificationModeGet(&clientHandle, ifNum);
      break;

    case 14:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      lldpMedIntfConfigNotificationModeSet(&clientHandle, ifNum, arg1);
      break;

    case 15:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedInterfaceTransmitTlvGet(&clientHandle, ifNum);
      break;
    
    case 16:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedInterfaceTransmitTlvSet(&clientHandle, ifNum);
      break;

    case 17:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemoteDeviceInfoGet(&clientHandle, ifNum);
      break;

    case 18:
      if (argc != 2)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpTxIntervalGet(&clientHandle, &val);
      break;
    case 19:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpTxIntervalSet(&clientHandle, atoi(argv[2]));
      break;
    case 20:
      if (argc != 2)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpTxHoldGet(&clientHandle, &val);
      break;
    case 21:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpTxHoldSet(&clientHandle, atoi(argv[2]));
      break;
    case 22:
      if (argc != 2)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpTxReinitDelayGet(&clientHandle, &val);
      break;
    case 23:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpTxReinitDelaySet(&clientHandle, atoi(argv[2]));
      break;
    case 24:
      if (argc != 2)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpNotificationIntervalGet(&clientHandle, &val);
      break;
    case 25:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpNotificationIntervalSet(&clientHandle, atoi(argv[2]));
      break;
    case 26:
      if (argc != 2)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpXMedFastStartRepeatCountGet(&clientHandle, &val);
      break;
    case 27:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpXMedFastStartRepeatCountSet(&clientHandle, atoi(argv[2]));
      break;
    case 28:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfTxMgmtAddrModeGet(&clientHandle, ifNum, &status);
      break;
    case 29:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpIntfTxMgmtAddrModeSet(&clientHandle, ifNum, atoi(argv[3]));
      break;
    case 30:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpXMedLocMediaPolicyVlanIDGet(&clientHandle, ifNum, atoi(argv[3]),
                                      &val);
      break;
    case 31:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpXMedLocMediaPolicyTaggedGet(&clientHandle, ifNum, atoi(argv[3]), 
                                      &status);
      break;
    case 32:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpXMedLocMediaPolicyPriorityGet(&clientHandle, ifNum, atoi(argv[3]),
                                        &val);
      break;
    case 33:
      if (argc != 4)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpXMedLocMediaPolicyDscpGet(&clientHandle, ifNum, atoi(argv[3]), 
                                    &val);
      break;
    case 34:
      if (argc != 2)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpRemoteIntfFirstGet(&clientHandle, &val);
      break;
    case 35:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpRemoteIntfNextGet(&clientHandle, atoi(argv[2]),&val);
      break;
    case 36:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpdot3RemPortAutoNegSupportedGet(&clientHandle, ifNum, remIndex, 
                                         timestamp, &status);
      break;
    case 37:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpdot3RemPortAutoNegEnabledGet(&clientHandle, ifNum, remIndex,
                                       timestamp, &status);
      break;
    case 38:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpdot3RemPortAutoNegAdvertizedCapGet(&clientHandle, ifNum, 
                                             remIndex, timestamp, &bufDesc);
      break;
    case 39:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpdot3RemPortAutoNegOperMauTypeGet(&clientHandle, ifNum, 
                                           remIndex, timestamp, &val);
      break;
    case 40:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpdot3RemPortAutoNegOperMauStringGet(&clientHandle, atoi(argv[2]), 
                                             &bufDesc);
      break;
    case 41:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemHardwareRevGet(&clientHandle, ifNum, remIndex, timestamp,
                               &bufDesc);
      break;
    case 42:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemFirmwareRevGet(&clientHandle, ifNum, remIndex, timestamp, &bufDesc);
      break;
    case 43:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemSoftwareRevGet(&clientHandle, ifNum, remIndex, timestamp, &bufDesc);
      break;
    case 44:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemSerialNumGet(&clientHandle, ifNum, remIndex, timestamp, &bufDesc);
      break;
    case 45:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemAssetIdGet(&clientHandle, ifNum, remIndex, timestamp, &bufDesc);
      break;
    case 46:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemPoeDeviceTypeGet(&clientHandle, ifNum, remIndex, timestamp, 
                                 &val);
      break;
    case 47:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemPoePSEPowerAvGet(&clientHandle, ifNum, remIndex, timestamp, 
                                 &val);
      break;
    case 48:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemPoePSEPowerSrcGet(&clientHandle, ifNum, remIndex, timestamp, 
                                  &val);
      break;
    case 49:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemPoePSEPowerPriGet(&clientHandle, ifNum, remIndex, timestamp, &val);
      break;
    case 50:
      if (argc != 6)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemLocationInfoGet(&clientHandle, ifNum, remIndex, timestamp, 
                                atoi(argv[5]), &bufDesc);
      break;
    case 51:
      if (argc != 6)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemMediaPolicyVlanIdGet(&clientHandle, ifNum, remIndex, timestamp, 
                                     policyAppType, &val);
      break;
    case 52:
      if (argc != 6)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemMediaPolicyPriorityGet(&clientHandle, ifNum, remIndex, timestamp,
                                       policyAppType, &val);
      break;
    case 53:
      if (argc != 6)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemMediaPolicyDscpGet(&clientHandle, ifNum, remIndex, timestamp, 
                                   policyAppType, &val);
      break;
    case 54:
      if (argc != 6)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemMediaPolicyUnknownGet(&clientHandle, ifNum, remIndex, timestamp,
                                      policyAppType, &status);
      break;
    case 55:
      if (argc != 6)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemMediaPolicyTaggedGet(&clientHandle, ifNum, remIndex, timestamp, 
                                     policyAppType, &status);
      break;
    case 56:
      if (argc != 6)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpRemUnknownTLVEntryNextGet(&clientHandle, ifNum, remIndex, timestamp, 
                                    atoi(argv[5]), &val);
      break;
    case 57:
      if (argc != 6)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpRemUnknownTLVEntryIsExists(&clientHandle, ifNum, remIndex, timestamp, 
                                     atoi(argv[5]), &status);
      break;
    case 58:
      if (argc != 6)
      {
        printLldpAppMenu();
        exit(1);
      }
      char buf1[512] = {0};

      bufDesc.pstart = buf1;
      bufDesc.size   = 512;
      lldpRemUnknownTLVInfoGet(&clientHandle, ifNum, remIndex, timestamp, 
                               atoi(argv[5]), &bufDesc);
      break;
    case 59:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemMfgNameGet(&clientHandle, ifNum, remIndex, timestamp, &bufDesc);
      break;
    case 60:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemModelNameGet(&clientHandle, ifNum, remIndex, timestamp, &bufDesc);
      break;
    case 61:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemPoePDPowerReqGet(&clientHandle, ifNum, remIndex, timestamp, 
                                 &val);
      break;
    case 62:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemPoePDPowerSrcGet(&clientHandle, ifNum, remIndex, timestamp, 
                                 &val);
      break;
    case 63:
      if (argc != 5)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedRemPoePDPowerPriGet(&clientHandle, ifNum, remIndex, timestamp, 
                                 &val);
      break;
    case 64:
      if (argc != 6)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpRemOrgDefOUIGet(&clientHandle, ifNum, remIndex, timestamp, atoi(argv[5]), 
                          &bufDesc);
      break;
    case 65:
      if (argc != 6)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpRemOrgDefSubtypeGet(&clientHandle, ifNum, remIndex, timestamp, 
                              atoi(argv[5]), &val);
      break;
    case 66:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpChassisIdSubtypeStringGet(&clientHandle, atoi(argv[2]), &bufDesc);
      break;
    case 67:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpPortIdSubtypeStringGet(&clientHandle, atoi(argv[2]), &bufDesc);
      break;
    case 68:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpdot3LocPortAutoNegSupportedGet(&clientHandle, ifNum, &status);
      break;
    case 69:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpdot3LocPortAutoNegEnabledGet(&clientHandle, ifNum, &status);
      break;
    case 70:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpdot3LocPortAutoNegAdvertizedCapGet(&clientHandle, ifNum, &bufDesc);
      break;
    case 71:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpdot3LocPortAutoNegOperMauTypeGet(&clientHandle, ifNum, &val);
      break;
    case 72:
      if (argc != 2)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpMedLocDeviceClassGet (&clientHandle, &val);
      break;
 
    case 73:
      if (argc != 3)
      {
        printLldpAppMenu();
        exit(1);
      }
      lldpOpENAPIsTestSanity(&clientHandle, ifNum);
      break;

    default:
      printLldpAppMenu();
      break;
  }
                                          
  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping LLDP API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
