/*********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  mvr_example.c
*
* @purpose   OpEN MVR example.
*
* @component OpEN
*
* @create    11/30/2022
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_mvr.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the MVR Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Gets the MVR admin mode: %s 1 \n", name);
  printf("Test 2: Sets the MVR admin mode: %s 2 <adminMode>\n", name);
  printf("Test 3: Gets the MVR query time: %s 3 \n", name);
  printf("Test 4: Sets the MVR query time: %s 4 <queryTime>\n", name);
  printf("Test 5: Gets the MVR mode type: %s 5 \n", name);
  printf("Test 6: Sets the MVR mode type: %s 6 <modeType>\n", name);
  printf("Test 7: Gets the MVR multicast vlan: %s 7 \n", name);
  printf("Test 8: Sets the MVR multicast vlan: %s 8 <multicastVlan>\n", name);
  printf("Test 9: Check if receivers participate in specific vlan: %s 9 <vlanId> \n", name);
  printf("Test 10: Gets the MVR interface mode: %s 10 <ifNum> \n", name);
  printf("Test 11: Sets the MVR interface mode: %s 11 <ifNum> <intfMode>\n", name);
  printf("Test 12: Gets the interface MVR type of a specified interface: %s 12 <ifNum> \n", name);
  printf("Test 13: Sets the interface MVR type of a specified interface: %s 13 <ifNum> <intfMvrType>\n", name);
  printf("Test 14: Gets the immediate leave mode of a specified interface: %s 14 <ifNum> \n", name);
  printf("Test 15: Sets the immediate leave mode of a specified interface: %s 15 <ifNum> <immediateMode>\n", name);
  printf("Test 16: Add static group IP address: %s 16 <groupIp>\n", name);
  printf("Test 17: Delete static group IP address: %s 17 <groupIp>\n", name);
  printf("Test 18: Add specified interface to the group: %s 18 <ifNum> <groupIp>\n", name);
  printf("Test 19: Delete specified interface from the group: %s 19 <ifNum> <groupIp>\n", name);
  printf("Test 20: Display MVR group entries: %s 20\n", name);
  printf("Test 21: List MVR members: %s 21\n", name);
  printf("Test 22: Get next MVR interface: %s 22 <ifNum>\n", name);
  printf("Test 23: Get the number of the group entries: %s 23 \n", name);
  printf("Test 24: Get the group IDs registered on specified interface: %s 24 <ifNum>\n", name);
  printf("Test 25: Get the group IPs statically registered on specified interface: %s 25 <ifNum>\n", name);
  printf("Test 26: Get the global PDU counter values: %s 26 \n", name);
  printf("Test 27: Get the maximum number of multicast group entries: %s 27 \n", name);

  return;
}

/*********************************************************************
* @purpose  Gets the MVR admin mode.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    adminMode        @b{(output)} MVR admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void mvrAdminModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *adminMode)
{
  open_error_t result;

  if ((result = openapiMvrAdminModeGet(client_handle, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the MVR admin mode. (result = %d)\n", 
           result);
  }
  else
  {
    printf("MVR admin mode is %u.\n", *adminMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the MVR admin mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    adminMode           @b{(input)}  MVR admin mode
*
* @returns  none
*
* @end
*********************************************************************/
void mvrAdminModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t adminMode)
{
  open_error_t result;

  if ((result = openapiMvrAdminModeSet(client_handle, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the MVR admin mode. (result = %d)\n", 
           result);
  }
  else
  {
    printf("MVR admin mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the MVR query time.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    queryTime       @b{(output)} MVR query time
*
* @returns  none
*
* @end
*********************************************************************/
void mvrQueryTimeGet(openapiClientHandle_t *client_handle, uint32_t *queryTime)
{
  open_error_t result;

  if ((result = openapiMvrQueryTimeGet(client_handle, queryTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the MVR query time. (result = %d)\n",
           result);
  }
  else
  {
    printf("MVR query time is %u.\n", *queryTime);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the MVR query time.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    queryTime      @b{(input)}  MVR query time
*
* @returns  none
*
* @end
*********************************************************************/
void mvrQueryTimeSet(openapiClientHandle_t *client_handle, uint32_t queryTime)
{
  open_error_t result;

  if ((result = openapiMvrQueryTimeSet(client_handle, queryTime)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MVR query time. (result = %d)\n", 
            result);
  }
  else
  {
    printf("MVR query time set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the MVR mode type.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    modeType        @b{(output)} MVR mode type
*
* @returns  none
*
* @end
*********************************************************************/
void mvrModeTypeGet(openapiClientHandle_t *client_handle, 
                    OPEN_MVR_MODE_TYPE_t *modeType)
{
  open_error_t result;

  if ((result = openapiMvrModeTypeGet(client_handle, modeType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the MVR mode type. (result = %d)\n",
           result);
  }
  else
  {
    printf("MVR mode type is %u.\n", *modeType);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the MVR mode type.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    modeType       @b{(input)}  MVR mode type
*
* @returns  none
*
* @end
*********************************************************************/
void mvrModeTypeSet(openapiClientHandle_t *client_handle, 
                    OPEN_MVR_MODE_TYPE_t modeType)
{
  open_error_t result;

  if ((result = openapiMvrModeTypeSet(client_handle, modeType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the MVR mode type. (result = %d)\n", 
           result);
  }
  else
  {
    printf("MVR mode type set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the MVR multicast vlan.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    multicastVlan   @b{(output)} MVR multicast vlan
*
* @returns  none
*
* @end
*********************************************************************/
void mvrMulticastVlanGet(openapiClientHandle_t *client_handle, 
                         uint32_t *multicastVlan)
{
  open_error_t result;

  if ((result = openapiMvrMulticastVlanGet(client_handle, multicastVlan)) !=
       OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the MVR multicast vlan. (result = %d)\n",
            result);
  }
  else
  {
    printf("MVR multicast vlan is %u.\n", *multicastVlan);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the MVR multicast vlan.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    multicastVlan   @b{(input)}  MVR multicast vlan
*
* @returns  none
*
* @end
*********************************************************************/
void mvrMulticastVlanSet(openapiClientHandle_t *client_handle, 
                         uint32_t multicastVlan)
{
  open_error_t result;

  if ((result = openapiMvrMulticastVlanSet(client_handle, multicastVlan)) !=
       OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the MVR multicast vlan. (result = %d)\n",
            result);
  }
  else
  {
    printf("MVR multicast vlan set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Check if receivers participate in specific vlan.
*
* @param    client_handle               @b{(input)}  client handle 
*                                                    from registration API
* @param    vlanId                      @b{(input)}  MVR multicast vlan
* @param    rxVlanParticipationStatus   @b{(output)} Receiver's vlan 
*                                                    participation status
*
* @returns  none
*
* @end
*********************************************************************/
void mvrIsReceiversInVlan(openapiClientHandle_t *client_handle, uint32_t vlanId,
                          OPEN_BOOL_t *rxVlanParticipationStatus)
{
  open_error_t result;

  if ((result = openapiMvrIsReceiversInVlan(client_handle, vlanId, 
                                            rxVlanParticipationStatus)) !=
       OPEN_E_NONE)
  {
    printf("Bad return code trying to check if receivers participate" 
           " in specific vlan. (result = %d)\n", result);
  }
  else
  {
    printf("Receivers %s participated in vlan %u.\n", 
            (*rxVlanParticipationStatus == OPEN_TRUE )? "" : "not",
            vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  Gets the MVR interface mode.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    intfMode       @b{(output)} MVR interface mode
*
* @returns  none
*
* @end
*********************************************************************/
void mvrIntfModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                    OPEN_CONTROL_t *intfMode)
{
  open_error_t result;

  if ((result = openapiMvrIntfModeGet(client_handle, ifNum, intfMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the MVR interface mode. (result = %d)\n",
            result);
  }
  else
  {
    printf("MVR interface mode is %u.\n", *intfMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the MVR interface mode.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    intfMode       @b{(input)}  MVR interface mode
*
* @returns  none
*
* @end
*********************************************************************/
void mvrIntfModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                    OPEN_CONTROL_t intfMode)
{
  open_error_t result;

  if ((result = openapiMvrIntfModeSet(client_handle, ifNum, intfMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the MVR interface mode. (result = %d)\n",
            result);
  }
  else
  {
    printf("MVR interface mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the interface MVR type of a specified interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    intfMvrType    @b{(output)} Interface MVR type
*
* @returns  none
*
* @end
*********************************************************************/
void mvrIntfMvrTypeGet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                       OPEN_MVR_INTF_TYPE_t *intfMvrType)
{
  open_error_t result;

  if ((result = openapiMvrIntfMvrTypeGet(client_handle, ifNum, intfMvrType)) !=
       OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the interface MVR type of a"
           " specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("MVR interfaece type is %u.\n", *intfMvrType);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the interface MVR type of a specified interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    intfMvrType    @b{(input)}  Interface MVR type
*
* @returns  none
*
* @end
*********************************************************************/
void mvrIntfMvrTypeSet(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                       OPEN_MVR_INTF_TYPE_t intfMvrType)
{
  open_error_t result;

  if ((result = openapiMvrIntfMvrTypeSet(client_handle, ifNum, intfMvrType)) != 
       OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the interface MVR type"
           " of a specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("MVR interface type set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets the immediate leave mode of a specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
* @param    immediateMode   @b{(output)} Immediate leave mode
*
* @returns  none
*
* @end
*********************************************************************/
void mvrIntfImmediateModeGet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                             OPEN_CONTROL_t *immediateMode)
{
  open_error_t result;

  if ((result = openapiMvrIntfImmediateModeGet(client_handle, ifNum, 
                                               immediateMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the immediate leave mode "
           "of a specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("MVR immediate mode is %u.\n", *immediateMode);
  }
  return;
}

/*********************************************************************
* @purpose  Sets the immediate leave mode of a specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
* @param    immediateMode   @b{(input)}  Immediate leave mode
*
* @returns  none
*
* @end
*********************************************************************/
void mvrIntfImmediateModeSet(openapiClientHandle_t *client_handle, uint32_t ifNum,
                             OPEN_CONTROL_t immediateMode)
{
  open_error_t result;

  if ((result = openapiMvrIntfImmediateModeSet(client_handle, ifNum, 
                                               immediateMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the immediate leave mode of a"
           " specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("MVR immediate mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Add static group IP address.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    groupIp        @b{(input)}  Group IP address
*
* @returns  none
*
* @end
*********************************************************************/
void mvrGroupIpAdd(openapiClientHandle_t *client_handle, char * groupIp)
{
  open_error_t result;
  uint32_t inetIpAddr = 0;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, groupIp, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;
  if ((result = openapiInetAton(client_handle, &ipBuffdesc, &inetIpAddr)) != 
       OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert internet address string "
           "to a 32 bit integer. result = %d.\n", result);
    return;
  }

  if ((result = openapiMvrGroupIpAdd(client_handle, inetIpAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add static group IP address. (result = %d)\n",
           result);
  }
  else
  {
    printf("Group IP added successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Delete static group IP address.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    groupIp        @b{(input)}  Group IP address
*
* @returns  none
*
* @end
*********************************************************************/
void mvrGroupIpDelete(openapiClientHandle_t *client_handle, char * groupIp)
{
  open_error_t result;
  uint32_t inetIpAddr = 0;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, groupIp, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;
  if ((result = openapiInetAton(client_handle, &ipBuffdesc, &inetIpAddr)) != 
       OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert internet address string "
           "to a 32 bit integer. result = %d.\n", result);
    return;
  }

  if ((result = openapiMvrGroupIpDelete(client_handle, inetIpAddr)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete static group IP address."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Group IP deleted successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Add specified interface to the group.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    groupIp       @b{(input)}  Group IP address
*
* @returns  none
*
* @end
*********************************************************************/
void mvrIntfGroupIpAdd(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                       char * groupIp)
{
  open_error_t result;
  uint32_t inetIpAddr = 0;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, groupIp, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;
  if ((result = openapiInetAton(client_handle, &ipBuffdesc, &inetIpAddr)) != 
       OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert internet address string "
           "to a 32 bit integer. result = %d.\n", result);
    return;
  }

  if ((result = openapiMvrIntfGroupIpAdd(client_handle, ifNum, inetIpAddr)) !=
       OPEN_E_NONE)
  {
    printf("Bad return code trying to add specified interface to the group."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Addition of specific interface to the group was successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Delete specified interface from the group.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    groupIp       @b{(input)}  Group IP address
*
* @returns  none
*
* @end
*********************************************************************/
void mvrIntfGroupIpDelete(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                          char* groupIp)
{
  open_error_t result;
  uint32_t inetIpAddr = 0;
  open_buffdesc ipBuffdesc;
  char str[40];

  memset(str, 0, sizeof(str));
  strncpy(str, groupIp, sizeof(str) - 1);
  ipBuffdesc.pstart = str;
  ipBuffdesc.size = strlen(str) + 1;
  if ((result = openapiInetAton(client_handle, &ipBuffdesc, &inetIpAddr)) != 
       OPEN_E_NONE)
  {
    printf("ERROR: Bad return code trying to convert internet address string "
           "to a 32 bit integer. result = %d.\n", result);
    return;
  }

  if ((result = openapiMvrIntfGroupIpDelete(client_handle, ifNum, inetIpAddr)) !=
       OPEN_E_NONE)
  {
    printf("Bad return code trying to delete specified interface to the group."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Deletion of specific interface to the group was successful.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Display group entries
*
* @param    client_handle @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void mvrGroupEntriesDisplay(openapiClientHandle_t *client_handle)
{
  uint32_t groupId = 0, nextGroupId = 0;
  open_error_t result;
  uint32_t groupIp = 0;
  open_buffdesc ipDesc;
  char ipDescStr[OPEN_IP_ADDR_DISP_LENGTH];
  open_inet_addr_t openAddr;
 
  ipDesc.pstart = ipDescStr;
  ipDesc.size = sizeof(ipDescStr);
  openAddr.family = OPEN_AF_INET;
  printf ("\nGroup Id  IP Address\n");
  printf ("--------  ----------------\n");
  do
  {
    if ((result = openapiMvrGroupEntryGetNext(client_handle, groupId, 
         &nextGroupId)) == OPEN_E_NONE)
    {
      if (OPEN_E_NONE == openapiMvrGroupIpGet(client_handle, nextGroupId, 
          &groupIp))
      {
        openAddr.addr.ipv4 = groupIp;
        (void)openapiOpenIPtoStringGet(client_handle, openAddr, &ipDesc);
        printf ("%-8u  %-15s\n", nextGroupId, (char *)ipDesc.pstart);
      }
      groupId = nextGroupId;
    } 
  } while (result == OPEN_E_NONE); 
  return;
}

/*********************************************************************
* @purpose  List MVR members
*
* @param    client_handle   @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void mvrListMembers(openapiClientHandle_t *client_handle)
{
  uint32_t groupIp = 0, nextGroupIp = 0, groupId = 0;
  open_buffdesc ipDesc;
  char ipDescStr[OPEN_IP_ADDR_DISP_LENGTH];
  open_inet_addr_t openAddr;
  OPEN_BOOL_t status = OPEN_FALSE;
  uint32_t intfNum = 0;
  OPEN_MVR_ENTRY_TYPE_t type;

  ipDesc.pstart = ipDescStr;
  ipDesc.size = sizeof(ipDescStr);
  openAddr.family = OPEN_AF_INET;
 
  if (openapiMvrEntryGetNext(client_handle, groupIp, &nextGroupIp) == 
      OPEN_E_NONE)
  {
    printf ("\nMVR Group IP        Status              Members\n");
    printf ("---------------     ---------------     -----------------------"
            "--------------\n");
    while (nextGroupIp != 0)
    {
      if (OPEN_E_NONE == openapiMvrGroupIdGet(client_handle, nextGroupIp, 
                                              &groupId))
      {
        openAddr.addr.ipv4 = nextGroupIp;
        memset (ipDescStr, 0, sizeof(ipDescStr));
        (void) openapiOpenIPtoStringGet(client_handle, openAddr, &ipDesc);
        (void) openapiMvrGroupStatusGet(client_handle, groupId, &status);
        printf ("%-17.15s   %-17.15s   ", (char *)ipDesc.pstart,
                status == OPEN_FALSE ? "INACTIVE" : "ACTIVE");

        while (OPEN_E_NONE == openapiMvrIntfGroupGetNext (client_handle, groupId,
                                                          intfNum, &intfNum))
        {
          if (OPEN_E_NONE == openapiMvrIntfEntryTypeGet (client_handle, intfNum,
                                                         groupId, &type))
          {
            if (OPEN_MVR_STATIC_TYPE == type)
            {
              printf ("%u(s) ", intfNum);  
            }
            else
            {
              printf ("%u(d) ", intfNum);  
            }
          }
        }
        printf ("\n");
        intfNum = 0;
      }
      if (openapiMvrEntryGetNext(client_handle, nextGroupIp, &nextGroupIp) != 
          OPEN_E_NONE)
      {
        nextGroupIp = 0;  
      }
    }
  }
  else
  {
    printf ("No MVR members.\n");  
  }

  return;
}

/*********************************************************************
* @purpose  Get next MVR interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface
* @param    nextIfNum           @b{(output)} Next MVR interface

*
* @returns  none
*
* @end
*********************************************************************/
void mvrIntfGetNext(openapiClientHandle_t *client_handle, uint32_t ifNum, 
                    uint32_t *nextIfNum)
{

  if (openapiMvrIntfGetNext(client_handle, ifNum, nextIfNum) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get next MVR interface.\n");
  }
  else
  {
    printf ("Next MVR interface is %u.\n", *nextIfNum);  
  }

  return;
}

/*********************************************************************
* @purpose  Get the number of the group entries.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    groupEntries        @b{(output)} Number of the group entries

*
* @returns  none
*
* @end
*********************************************************************/
void mvrGroupNumGet(openapiClientHandle_t *client_handle, uint32_t *groupEntries)
{
  open_error_t result;

  if ((result = openapiMvrGroupNumGet(client_handle, groupEntries)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of the group entries. (result = %d)\n", result);
  }
  else
  {
    printf("Number of group entries is %u.\n", *groupEntries);
  }
  return;
}

/*********************************************************************
* @purpose  Get group IDs registered on specified interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
*
* @returns  none
*
* @end
*********************************************************************/
void mvrIntfGroupIdGet(openapiClientHandle_t *client_handle, 
                       uint32_t ifNum)
{
  uint32_t groupId = 0;

  if (openapiMvrIntfGroupIdGetNext(client_handle, ifNum, groupId,
                                   &groupId) == OPEN_E_NONE)
  {
    printf ("Group ID's configured for given interface : ");  
    do
    {
      printf ("%u ", groupId);  
    } while (openapiMvrIntfGroupIdGetNext(client_handle, ifNum, groupId,
                                          &groupId) == OPEN_E_NONE);
    printf ("\n");    
  }
  else
  {
    printf("No Group configured for the entered interface.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get group IPs statically registered on specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
*
* @returns  none
*
* @end
*********************************************************************/
void mvrIntfStaticGroupIpGet(openapiClientHandle_t *client_handle, 
                             uint32_t ifNum)
{
  uint32_t groupIp = 0;  
  open_buffdesc ipDesc;
  char ipDescStr[OPEN_IP_ADDR_DISP_LENGTH] = {0};
  open_inet_addr_t openAddr;

  ipDesc.pstart = ipDescStr;
  ipDesc.size = sizeof(ipDescStr);
  openAddr.family = OPEN_AF_INET;

  if (openapiMvrIntfStaticGroupIpGetNext(client_handle, ifNum, groupIp,
                                         &groupIp) == OPEN_E_NONE)
  {
    printf ("Group IP for interface : ");  
    do
    {
      openAddr.addr.ipv4 = groupIp;
      memset (ipDescStr, 0, sizeof(ipDescStr));
      (void) openapiOpenIPtoStringGet(client_handle, openAddr, &ipDesc);
      printf ("%s ", (char *)ipDesc.pstart);
    } while (openapiMvrIntfStaticGroupIpGetNext(client_handle, ifNum, 
                                                groupIp, &groupIp) == OPEN_E_NONE);
    printf ("\n");
  }
  else
  {
    printf("No group IP configured.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the global PDU counter values.
*
* @param    client_handle   @b{(input)}  client handle from registration API
*
* @returns  none
*
* @end
*********************************************************************/
void mvrPduCountersGet(openapiClientHandle_t *client_handle)
{
  uint32_t counterValue = 0;
  open_error_t result;
  
  if ((result = openapiMvrPduCounterGet (client_handle, OPEN_MVR_PKT_IGMP_QUERY_RX_COUNTER, 
                                         &counterValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global PDU counter values. (result = %d)\n", result);
  }
  else
  {
    printf ("IGMP Query Received %u\n", counterValue);
  }

  counterValue = 0;
  if ((result = openapiMvrPduCounterGet (client_handle, OPEN_MVR_PKT_IGMP_REPORT_V1_RX_COUNTER, 
                                         &counterValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global PDU counter values. (result = %d)\n", result);
  }
  else
  {
    printf ("IGMP Report V1 Received %u\n", counterValue);
  }

  counterValue = 0;
  if ((result = openapiMvrPduCounterGet (client_handle, OPEN_MVR_PKT_IGMP_REPORT_V2_RX_COUNTER, 
                                         &counterValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global PDU counter values. (result = %d)\n", result);
  }
  else
  {
    printf ("IGMP Report V2 Received %u\n", counterValue);
  }

  counterValue = 0;
  if ((result = openapiMvrPduCounterGet (client_handle, OPEN_MVR_PKT_IGMP_LEAVE_RX_COUNTER, 
                                         &counterValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global PDU counter values. (result = %d)\n", result);
  }
  else
  {
    printf ("IGMP Leave Received %u\n", counterValue);
  }

  counterValue = 0;
  if ((result = openapiMvrPduCounterGet (client_handle, OPEN_MVR_PKT_IGMP_QUERY_TX_COUNTER, 
                                         &counterValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global PDU counter values. (result = %d)\n", result);
  }
  else
  {
    printf ("IGMP Query Transmitted %u\n", counterValue);
  }

  counterValue = 0;
  if ((result = openapiMvrPduCounterGet (client_handle, OPEN_MVR_PKT_IGMP_REPORT_V1_TX_COUNTER, 
                                         &counterValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global PDU counter values. (result = %d)\n", result);
  }
  else
  {
    printf ("IGMP Report V1 Transmitted %u\n", counterValue);
  }

  counterValue = 0;
  if ((result = openapiMvrPduCounterGet (client_handle, OPEN_MVR_PKT_IGMP_REPORT_V2_TX_COUNTER, 
                                         &counterValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global PDU counter values. (result = %d)\n", result);
  }
  else
  {
    printf ("IGMP Report V2 Transmitted %u\n", counterValue);
  }

  counterValue = 0;
  if ((result = openapiMvrPduCounterGet (client_handle, OPEN_MVR_PKT_IGMP_LEAVE_TX_COUNTER, 
                                         &counterValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global PDU counter values. (result = %d)\n", result);
  }
  else
  {
    printf ("IGMP Leave Transmitted %u\n", counterValue);
  }

  counterValue = 0;
  if ((result = openapiMvrPduCounterGet (client_handle, OPEN_MVR_PKT_RX_FAILURE_COUNTER,
                                         &counterValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global PDU counter values. (result = %d)\n", result);
  }
  else
  {
    printf ("IGMP Packet Receive Failures %u\n", counterValue);
  }

  counterValue = 0;
  if ((result = openapiMvrPduCounterGet (client_handle, OPEN_MVR_PKT_TX_FAILURE_COUNTER, 
                                         &counterValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global PDU counter values. (result = %d)\n", result);
  }
  else
  {
    printf ("IGMP Packet Transmit Failures %u\n", counterValue);
  }

  return;
}

/*********************************************************************
* @purpose  Get the maximum number of multicast group entries.
*
* @param    client_handle       @b{(input)}  client handle from 
*                                            registration API.
* @param    groupEntries        @b{(output)} Number of maximum multicast
*                                            group entries.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrMaxGroupNumGet(openapiClientHandle_t *client_handle,
                       uint32_t *groupEntries)
{
  open_error_t result;

  if ((result = openapiMvrMaxGroupCountGet(client_handle, 
                                           groupEntries)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the maximum number of multicast "
           "group entries. (result = %d)\n", result);
  }
  else
  {
    printf("Maximum number of multicast group entries is %u.\n", *groupEntries);
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for Mvr
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t  show_help = 1, value = 0;
  OPEN_CONTROL_t val = 0;
  OPEN_MVR_MODE_TYPE_t modeType;
  OPEN_BOOL_t boolVal;
  OPEN_MVR_INTF_TYPE_t intfMvrType;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("MVR example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting MVR API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 2)
      {
        mvrAdminModeGet(&client_handle, &val);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        mvrAdminModeSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 2)
      {
        mvrQueryTimeGet(&client_handle, &value);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        mvrQueryTimeSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 2)
      {
        mvrModeTypeGet(&client_handle, &modeType);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        mvrModeTypeSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 2)
      {
        mvrMulticastVlanGet(&client_handle, &value);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        mvrMulticastVlanSet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        mvrIsReceiversInVlan(&client_handle, atoi(argv[2]), &boolVal);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 3)
      {
        mvrIntfModeGet(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 4)
      {
        mvrIntfModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 3)
      {
        mvrIntfMvrTypeGet(&client_handle, atoi(argv[2]), &intfMvrType);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 4)
      {
        mvrIntfMvrTypeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 3)
      {
        mvrIntfImmediateModeGet(&client_handle, atoi(argv[2]), &val);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 4)
      {
        mvrIntfImmediateModeSet(&client_handle, atoi(argv[2]), atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 3)
      {
        mvrGroupIpAdd(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 3)
      {
        mvrGroupIpDelete(&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 4)
      {
        mvrIntfGroupIpAdd(&client_handle, atoi(argv[2]), argv[3]);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 4)
      {
        mvrIntfGroupIpDelete(&client_handle, atoi(argv[2]), argv[3]);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 2)
      {
        mvrGroupEntriesDisplay(&client_handle);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 2)
      {
        mvrListMembers(&client_handle);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 3)
      {
        mvrIntfGetNext(&client_handle, atoi(argv[2]), &value);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 2)
      {
        mvrGroupNumGet(&client_handle, &value);
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 3)
      {
        mvrIntfGroupIdGet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 3)
      {
        mvrIntfStaticGroupIpGet(&client_handle, atoi(argv[2]));
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 2)
      {
        mvrPduCountersGet(&client_handle);
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 2)
      {
        mvrMaxGroupNumGet(&client_handle, &value);
        show_help = 0;
      }
      break;
   default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping MVR API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}

