/*********************************************************************
*
* Copyright 2021-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ntp_example.c
*
* @purpose   NTP OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    10/13/2021
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_ntp.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the NTP Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printNtpAppMenu()
{
  printf("Usage: ntp_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Add NTP server: ntp_example 1 <ntp-server> \n");
  printf("Test 2: Delete NTP server: ntp_example 2 <ntp-server> \n");
  printf("Test 3: Get NTP server info: ntp_example 3 \n");
  printf("Test 4: Set NTP mode: ntp_example 4 <ntp mode> \n");
  printf("Test 5: Get NTP info: ntp_example 5 \n");
  printf("Test 6: Set NTP authentication mode: ntp_example 6 <ntp auth mode> \n");
  printf("Test 7: Set NTP broadcast delay: ntp_example 7 <broadcast delay> \n");
  printf("Test 8: Set NTP source interface: ntp_example 8 <interface number> \n");
  printf("Test 9: Set NTP VRF name: ntp_example 9 <vrf-name> \n");
  printf("Test 10: Set NTP server authentication key: ntp_example 10 <ntp-server> <key-number> \n");
  printf("Test 11: Set NTP server max poll interval: ntp_example 11 <ntp-server> <max-poll> \n");
  printf("Test 12: Set NTP server min poll interval: ntp_example 12 <ntp-server> <min-poll> \n");
  printf("Test 13: Set NTP server prefer status: ntp_example 13 <ntp-server> <prefer-status> \n");
  printf("Test 14: Set NTP server burst status: ntp_example 14 <ntp-server> <burst-status> \n");
  printf("Test 15: Set NTP server iburst status: ntp_example 15 <ntp-server> <iburst-status> \n");
  printf("Test 16: Set NTP server version: ntp_example 16 <ntp-server> <version> \n");
  printf("Test 17: Add NTP authentication table key: ntp_example 17 <key-number> <key-value> <isEncrypted> <auth alg> \n");
  printf("Test 18: Delete NTP authentication table key: ntp_example 18 <key-number> \n");
  printf("Test 19: Set NTP authentication table key value: ntp_example 19 <key-number> <key-value> <isEncrypted> \n");
  printf("Test 20: Set NTP authentication table algorithm: ntp_example 20 <key-number> <auth alg> \n");
  printf("Test 21: Set NTP autentication key trusted status: ntp_example 21 <key-number> <trusted status> \n");
  printf("Test 22: Get NTP authentication table info: ntp_example 22 \n");
  printf("Test 23: ntp_example OpEN APIs sanity: ntp_example 23 \n");
  printf("Test 24: Set NTP broadcast client mode: ntp_example 24 <mode> \n");
  printf("Test 25: Get NTP software information: ntp_example 25 \n");
  printf("Test 26: Get NTP packet statistics: ntp_example 26  \n");
  printf("Test 27: Get NTP association details: ntp_example 27 \n");
  printf("Test 28: Get NTP association statistics: ntp_example 28 \n");

  return;
}

/*********************************************************************
* @purpose  Adds an NTP server.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    ntpServerAddr   @b{(input)}   NTP server address
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void ntpServerAdd(openapiClientHandle_t *clientHandle, char *ntpServerAddr)
{
  open_error_t result;
  open_buffdesc ntpServer;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, ntpServerAddr, (sizeof(str) - 1));
  ntpServer.pstart = str;
  ntpServer.size = strlen(str);

  if ((result = openapiNtpServerAdd(clientHandle, &ntpServer)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add NTP server. (result = %d)\n", result);
  }
  else
  {
    printf("NTP server is added successfully. (result = %d)\n", result); 
  }
  
  return; 
}

/*********************************************************************
* @purpose  Delete the NTP server.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    ntpServerAddr   @b{(input)}   NTP server address
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void ntpServerDelete(openapiClientHandle_t *clientHandle, char *ntpServerAddr)
{
  open_error_t result;
  open_buffdesc ntpServer;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, ntpServerAddr, (sizeof(str) - 1));
  ntpServer.pstart = str;
  ntpServer.size = strlen(str);

  if ((result = openapiNtpServerDelete(clientHandle, &ntpServer)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete NTP server. (result = %d)\n", result);
  }
  else
  {
    printf("NTP server is deleted successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the NTP servers details.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void ntpServerInfoGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t maxNtpServers = 0;
  uint32_t ntpServerMaxLen = 0;
  uint32_t stratum = 0;
  uint32_t ntpServerRefIdStrLen = 0;
  OPEN_NTP_SERVER_MODE_t ntpServerMode;
  uint32_t currNtpServers = 0, val = 0;
  open_buffdesc ntpServer, activeNtpServer, referenceId, ntpNextServer;
  OPEN_NTP_ADDRESS_TYPE_t addressType;
  char *pNtpServInfo, *str1, *str2, *pNtpNxtServInfo;
  uint16_t temp = 0;
  OPEN_BOOL_t status; 

  if ((result = openapiNtpServersMaxSupportedGet(clientHandle, &maxNtpServers)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get maximum number of NTP servers supported. (result = %d)\n", result);
    return;
  }
  else
  {
    printf("Maximum no. of NTP servers: %d\n", maxNtpServers);
  }

  if ((result = openapiNtpServerCurrentEntriesGet(clientHandle, &currNtpServers)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get current NTP servers supported. (result = %d)\n", result);
    return;
  } 
  else
  {
    printf("Current NTP servers: %d\n", currNtpServers);
  }

  if ((result = openapiNtpServerMaxAddrLenGet(clientHandle, &ntpServerMaxLen)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get maximum length of NTP server. (result = %d)\n", result);
    return;
  }
  else
  {
    printf(" Server Max Address Length: %d\n", ntpServerMaxLen); 
  }

  if ((result = openapiNtpServerReferenceStringLengthGet(clientHandle, &ntpServerRefIdStrLen)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get maximum length of NTP server reference ID. (result = %d)\n", result);
    return;
  }
  else
  {
    printf(" Server Max Reference ID length: %d\n", ntpServerRefIdStrLen); 
  }

  if ((pNtpServInfo = (char *)malloc(ntpServerMaxLen+1)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  if ((str1 = (char *)malloc(ntpServerMaxLen+1)) == NULL)
  {
    printf("Could not allocate memory.\n");
    free(pNtpServInfo);
    return;
  }

  if ((str2 = (char *)malloc(ntpServerRefIdStrLen+1)) == NULL)
  {
    printf("Could not allocate memory.\n");
    free(pNtpServInfo);
    free(str1);
    return;
  }

  memset(pNtpServInfo, 0, ntpServerMaxLen+1);
  ntpServer.pstart = pNtpServInfo;
  ntpServer.size = ntpServerMaxLen+1;

  memset(str1, 0, ntpServerMaxLen+1);
  activeNtpServer.pstart = str1;
  activeNtpServer.size = ntpServerMaxLen+1;

  memset(str2, 0, ntpServerRefIdStrLen+1);
  referenceId.pstart = str2;
  referenceId.size = ntpServerRefIdStrLen+1;

  if ((pNtpNxtServInfo = (char *)malloc(ntpServerMaxLen+1)) == NULL)
  {
    printf("Could not allocate memory.\n");
    free(pNtpServInfo);
    free(str1);
    free(str2);
    return;
  }

  memset(pNtpNxtServInfo, 0, ntpServerMaxLen+1);
  ntpNextServer.pstart = pNtpNxtServInfo;
  ntpNextServer.size = ntpServerMaxLen+1;

  printf("\n");

  printf("NTP active server details: \n");
  printf("---------------------------\n");

  printf("NTP server Address: ");
  if ((result = openapiActiveNtpServerIpAddressGet(clientHandle, &activeNtpServer)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get active NTP server address. (result = %d)\n", result);
  }
  else
  {
    printf("%s \n", (char *) activeNtpServer.pstart);
  }

  printf("NTP server Type: ");
  if ((result = openapiActiveNtpServerAddressTypeGet(clientHandle, &addressType)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get active NTP server type. (result = %d)\n", result);
  }
  else
  {
    switch(addressType)
    {
      case OPEN_NTP_ADDRESS_UNKNOWN:
        printf("Unknown \n");
        break;
      case OPEN_NTP_ADDRESS_IPV4:
        printf("IPV4 \n");
        break;
      case OPEN_NTP_ADDRESS_DNS:
        printf("DNS \n");
        break;
      case OPEN_NTP_ADDRESS_IPV6:
        printf("IPV6 \n");
        break;
      default:
        break;
    }
  }

  printf("Server stratum: ");
  if ((result = openapiActiveNtpServerStratumGet(clientHandle, &stratum)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get active NTP server stratum. (result = %d)\n", result);
  }
  else
  {
    printf("%d \n", stratum);
  }

  printf("Server Reference ID: ");
  if ((result = openapiActiveNtpServerReferenceIdGet(clientHandle, &referenceId)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get active NTP server reference ID. (result = %d)\n", result);
  }
  else
  {
    printf("%s \n", (char *) referenceId.pstart);
  }

  printf("Server Mode: ");
  if ((result = openapiActiveNtpServerModeGet(clientHandle, &ntpServerMode)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get active NTP server Mode. (result = %d)\n", result);
  }
  else
  {
    switch(ntpServerMode)
    {
      case OPEN_NTP_SERVER_MODE_RESERVED:
        printf("Reserved \n");
        break;
      case OPEN_NTP_SERVER_MODE_SYMMETRIC_ACTIVE:
        printf("Symmetric Active \n");
        break;
      case OPEN_NTP_SERVER_MODE_SYMMETRIC_PASSIVE:
        printf("Symmetric Passive \n");
        break;
      case OPEN_NTP_SERVER_MODE_CLIENT:
        printf("Client \n");
        break;
      case OPEN_NTP_SERVER_MODE_SERVER:
        printf("Server \n");
        break;
      case OPEN_NTP_SERVER_MODE_BROADCAST:
        printf("Broadcast \n");
        break;
      case OPEN_NTP_SERVER_MODE_RESERVED_CONTROL:
        printf("Reserved Control \n");
        break;
      case OPEN_NTP_SERVER_MODE_RESERVED_PRIVATE:
        printf("Reserved Private \n");
        break;
      default:
      break;
    }
  }

  printf("\n");
  printf("NTP servers details: \n");
  printf("---------------------------\n");

  if ((result = openapiNtpServerFirstGet(clientHandle, &ntpNextServer)) != OPEN_E_NONE)
  {
    printf("  Bad return code while getting first NTP server. (result = %d)\n", result);
    free(pNtpServInfo);
    free(str1);
    free(str2);
    free(pNtpNxtServInfo);
    return;
  }
  do
  {
    memset(ntpServer.pstart, 0, ntpServerMaxLen+1);
    strncpy(ntpServer.pstart, ntpNextServer.pstart, ntpServerMaxLen);
    ntpServer.size = strnlen(ntpServer.pstart, ntpServerMaxLen);
    memset(ntpNextServer.pstart, 0, ntpServerMaxLen+1);
    ntpNextServer.size = ntpServerMaxLen+1;
    
    printf("NTP server: %s\n",(char *)ntpServer.pstart);

    printf("Address Type: ");
    if ((result = openapiNtpServerAddressTypeGet(clientHandle, &ntpServer, &addressType)) != OPEN_E_NONE)
    {
      printf("Address type get failed. (result = %d) \n", result);
    }
    else
    {
      switch(addressType)
      {
        case OPEN_NTP_ADDRESS_UNKNOWN:
          printf("Unknown \n");
          break;
        case OPEN_NTP_ADDRESS_IPV4:
          printf("IPV4 \n");
          break;
        case OPEN_NTP_ADDRESS_DNS:
          printf("DNS \n");
          break;
        case OPEN_NTP_ADDRESS_IPV6:
          printf("IPV6 \n");
          break;
        default:
          break;
      }
    }
    
    printf("Authentication key: ");
    if ((result = openapiNtpServerTableAuthKeyGet(clientHandle, &ntpServer, &val)) != OPEN_E_NONE)
    { 
      printf("Unable to get NTP server authentication key. (result = %d) \n", result);
    }
    else
    {
      printf("%u \n", val);
    }

    printf("Max poll interval: ");
    if ((result = openapiNtpServerTableMaxPollGet(clientHandle, &ntpServer, &temp)) != OPEN_E_NONE)
    {
      printf("Unable to get NTP server max poll. (result = %d) \n", result);
    }
    else
    {
      printf("%d \n", temp);
    }

    printf("Min poll interval: ");
    if ((result = openapiNtpServerTableMinPollGet(clientHandle, &ntpServer, &temp)) != OPEN_E_NONE)
    {
      printf("Unable to get NTP server min poll. (result = %d) \n", result);
    }
    else
    {
      printf("%d \n", temp);
    }

    printf("Prefer: ");
    if ((result = openapiNtpServerTablePreferGet(clientHandle, &ntpServer, &status)) != OPEN_E_NONE)
    {
      printf("Unable to get NTP server prefer status. (result = %d) \n", result);
    }
    else
    {
      switch(status)
      {
        case OPEN_TRUE:
          printf("Yes \n");
          break;
        case OPEN_FALSE:
          printf("No \n");
          break;
        default:
          break;
      }
    }

    printf("Burst: ");
    if ((result = openapiNtpServerTableBurstGet(clientHandle, &ntpServer, &status)) != OPEN_E_NONE)
    {
      printf("Unable to get NTP server burst status. (result = %d) \n", result);
    }
    else
    {
      switch(status)
      {
        case OPEN_TRUE:
          printf("Yes \n");
          break;
        case OPEN_FALSE:
          printf("No \n");
          break;
        default:
          break;
      }
    }

    printf("IBurst: ");
    if ((result = openapiNtpServerTableIBurstGet(clientHandle, &ntpServer, &status)) != OPEN_E_NONE)
    {
      printf("Unable to get NTP server iburst status. (result = %d) \n", result);
    }
    else
    {
      switch(status)
      {
        case OPEN_TRUE:
          printf("Yes \n");
          break;
        case OPEN_FALSE:
          printf("No \n");
          break;
        default:
          break;
      }
    }

    printf("Version: ");
    if ((result = openapiNtpServerTableVersionGet(clientHandle, &ntpServer, &temp)) != OPEN_E_NONE)
    {
      printf("Unable to get NTP server version. (result = %d) \n", result);
    }
    else
    {
      printf("%d \n", temp);
    }

    printf("\n");
  }while(openapiNtpServerNextGet(clientHandle, &ntpServer, &ntpNextServer) == OPEN_E_NONE);

  free(pNtpServInfo);
  free(str1);
  free(str2);
  free(pNtpNxtServInfo);
  return; 
}

/*********************************************************************
* @purpose  Set the NTP mode.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   NTP mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void ntpModeSet(openapiClientHandle_t *clientHandle, OPEN_NTP_MODE_t mode)
{
  open_error_t result;

  if ((result = openapiNtpModeSet(clientHandle, mode)) != OPEN_E_NONE) 
  {
    printf("Bad return code trying to set NTP mode. (result = %d)\n", result);
  }
  else
  {
    printf("NTP mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose Set the ntp broadcast client mode of the NTP application. 
*
* @param    clientHandle     @b{(input)}   client handle from registration API
* @param    mode             @b{(input)}   NTP broadcast client mode
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void ntpBroadcastClientModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiNtpBroadcastClientModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set NTP broadcast client mode. (result = %d)\n", result);
  }
  else
  {
    printf("NTP broadcast client mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the current authentication mode for this application
*
* @param    clientHandle   @b{(input)}  Client handle from 
*                                        registration API
* @param    mode            @b{(input)} OPEN_TRUE to enable,
*                                       OPEN_FALSE to disable
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void ntpAuthenticationModeSet(openapiClientHandle_t *clientHandle,
                              OPEN_BOOL_t mode)
{
  open_error_t result;

  if ((result = openapiNtpAuthenticationModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set NTP authentication mode. (result = %d)\n", result);
  }
  else
  {
    printf("NTP authentication mode is set successfully.\n");
  }

  return;
}

/*********************************************************************
* @purpose  Set the Broadcast delay for the NTP application.
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    delay           @b{(input)} Broadcast delay in milliseconds 
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void ntpBroadcastDelaySet(openapiClientHandle_t *clientHandle,
                                 uint32_t delay)
{
  open_error_t result;

  if ((result = openapiNtpBroadcastDelaySet(clientHandle, delay)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set NTP broadcast delay. (result = %d)\n", result);
  }
  else
  {
    printf("NTP broadcast delay is set successfully.\n");
  }

  return;
}

/**********************************************************************
* @purpose  Set the Source Interface of the NTP application.
*
* @param    clientHandle   @b{(input)}  Client handle from 
*                                         registration API
* @param    intIfNum        @b{(input)} Source Interface number
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
*********************************************************************/
void ntpSourceInterfaceSet(openapiClientHandle_t *clientHandle,
                                          uint32_t intIfNum)
{
  open_error_t result;

  if ((result = openapiNtpSourceInterfaceSet(clientHandle, intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set NTP source interface. (result = %d)\n", result);
  }
  else
  {
    printf("NTP source interface is set successfully.\n");
  }

  return;
}

/**********************************************************************
* @purpose  Set the VRF name of the NTP server.
*
* @param    clientHandle   @b{(input)}  Client handle from 
*                                        registration API
* @param    vrfName         @b{(input)} VRF Name 
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpVrfNameSet(openapiClientHandle_t *clientHandle,
                                  char *vrfName)
{
  open_error_t result;
  open_buffdesc vrfNameBuff;
  char str[OPEN_VRF_MAX_NAME+1] = {0};

  strncpy(str, vrfName, (sizeof(str) - 1));
  vrfNameBuff.pstart = str;
  vrfNameBuff.size = strlen(str);

  if ((result = openapiNtpVrfNameSet(clientHandle, &vrfNameBuff)) != OPEN_E_NONE)
  { 
    printf("Bad return code trying to set NTP VRF name. (result = %d)\n", result);
  }
  else
  { 
    printf("NTP VRF name is set successfully.\n");
  }
  
  return;
}

/*********************************************************************
* @purpose  Set the authentication key for the specified
*           NTP server table entry
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServerAddr   @b{(input)} NTP server
* @param    keyIndex        @b{(input)} index of authenticate table entry.
*
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpServerTableAuthKeySet(openapiClientHandle_t *clientHandle,
                                char *ntpServerAddr, uint32_t keyIndex)
{
  open_buffdesc ntpServer;
  char str[100];
  open_error_t result;
 
  memset(str, 0, sizeof(str));
  strncpy(str, ntpServerAddr, (sizeof(str) - 1));
  ntpServer.pstart = str;
  ntpServer.size = strlen(str);

  if ((result = openapiNtpServerTableAuthKeySet(clientHandle, &ntpServer, keyIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set NTP server authentication key. (result = %d)\n", result);
  }
  else
  {
    printf("NTP authentication key is set successfully. (result = %d)\n", result);
  }

  return;

}

/*********************************************************************
* @purpose  Set the server maximum poll interval in seconds as a power of two
*
* @param    clientHandle   @b{(input)}  Client handle from 
*                                        registration API
* @param    ntpServerAddr   @b{(input)} NTP server
* @param    interval        @b{(input)} power of two of poll interval  
*
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpServerTableMaxPollSet(openapiClientHandle_t *clientHandle,
                                             char *ntpServerAddr, uint16_t interval)
{
  open_error_t result;
  open_buffdesc ntpServer;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, ntpServerAddr, (sizeof(str) - 1));
  ntpServer.pstart = str;
  ntpServer.size = strlen(str);

  if ((result = openapiNtpServerTableMaxPollSet(clientHandle, &ntpServer, interval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set NTP server maximum poll interval. (result = %d)\n", result);
  }
  else
  {
    printf("NTP server maximum poll interval is set successfully. (result = %d)\n", result);
  }

  return;

}

/*********************************************************************
* @purpose  Set the server minimum poll interval in seconds as a power of two
*
* @param    clientHandle   @b{(input)}  Client handle from 
*                                        registration API
* @param    ntpServerAddr   @b{(input)} NTP server
* @param    interval        @b{(input)} power of two of poll interval  
*
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpServerTableMinPollSet(openapiClientHandle_t *clientHandle,
                                     char *ntpServerAddr, uint16_t interval)
{
  open_error_t result;
  open_buffdesc ntpServer;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, ntpServerAddr, (sizeof(str) - 1));
  ntpServer.pstart = str;
  ntpServer.size = strlen(str);

  if ((result = openapiNtpServerTableMinPollSet(clientHandle, &ntpServer, interval)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set NTP server minimum poll interval. (result = %d)\n", result);
  }
  else
  {
    printf("NTP server minimum poll interval is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose Set the NTP prefer status of the server table entry 
*
* @param    clientHandle   @b{(input)}  Client handle from 
*                                        registration API
* @param    ntpServerAddr   @b{(input)} NTP server
* @param    prefer          @b{(input)} server preference status 
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpServerTablePreferSet(openapiClientHandle_t *clientHandle,
                                char *ntpServerAddr, OPEN_BOOL_t prefer)
{ 
  open_error_t result;
  open_buffdesc ntpServer;
  char str[100];
  
  memset(str, 0, sizeof(str));
  strncpy(str, ntpServerAddr, (sizeof(str) - 1));
  ntpServer.pstart = str;
  ntpServer.size = strlen(str);

  if ((result = openapiNtpServerTablePreferSet(clientHandle, &ntpServer, prefer)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set NTP server prefer status. (result = %d)\n", result);
  }
  else
  {
    printf("NTP server prefer status is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose Set the NTP burst status of the server table entry 
*
* @param    clientHandle   @b{(input)}  Client handle from 
*                                        registration API
* @param    ntpServerAddr   @b{(input)} NTP server
* @param    burst           @b{(input)} server burst status 
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpServerTableBurstSet(openapiClientHandle_t *clientHandle,
                            char *ntpServerAddr, OPEN_BOOL_t burst)
{
  open_error_t result;
  open_buffdesc ntpServer;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, ntpServerAddr, (sizeof(str) - 1));
  ntpServer.pstart = str;
  ntpServer.size = strlen(str);

  if ((result = openapiNtpServerTableBurstSet(clientHandle, &ntpServer, burst)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set NTP server burst status. (result = %d)\n", result);
  }
  else
  {
    printf("NTP server burst status is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose Set the NTP iburst status of the server table entry 
*
* @param    clientHandle   @b{(input)}  Client handle from 
*                                        registration API
* @param    ntpServerAddr   @b{(input)} NTP server
* @param    iburst          @b{(input)} server iburst status 
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpServerTableIBurstSet(openapiClientHandle_t *clientHandle,
                            char *ntpServerAddr, OPEN_BOOL_t iburst)
{ 
  open_error_t result;
  open_buffdesc ntpServer;
  char str[100];
  
  memset(str, 0, sizeof(str));
  strncpy(str, ntpServerAddr, (sizeof(str) - 1));
  ntpServer.pstart = str;
  ntpServer.size = strlen(str);
  
  if ((result = openapiNtpServerTableIBurstSet(clientHandle, &ntpServer, iburst)) != OPEN_E_NONE)
  { 
    printf("Bad return code trying to set NTP server iburst status. (result = %d)\n", result);
  }
  else
  {
    printf("NTP server iburst status is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set the NTP version of the server table entry 
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    ntpServerAddr   @b{(input)} NTP server
* @param    version         @b{(input)} server version
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpServerTableVersionSet(openapiClientHandle_t *clientHandle,
                                    char *ntpServerAddr, uint16_t version)
{
  open_error_t result;
  open_buffdesc ntpServer;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, ntpServerAddr, (sizeof(str) - 1));
  ntpServer.pstart = str;
  ntpServer.size = strlen(str);

  if ((result = openapiNtpServerTableVersionSet(clientHandle, &ntpServer, version)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set NTP server version. (result = %d)\n", result);
  }
  else
  {
    printf("NTP server version is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Add a new authentication table entry specified by key number.                       
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)} NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                       OPEN_NTP_MAX_KEY_NUM                            
* @param    keyValue        @b{(input)} ASCII key string OPEN_NTP_MIN_KEY_LENGTH-
*                                       OPEN__NTP_MAX_KEY_LENGTH
* @param    isEncrypted     @b{(input)}  whether the key is encrypted or not
* @param    msgAuthAlg      @b{(input)}  Message authentication algorithm
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpAuthTableKeyAdd(openapiClientHandle_t *clientHandle,
                              uint32_t keyNumber, char *keyValue,
                              OPEN_BOOL_t isEncrypted, OPEN_NTP_MSG_AUTH_ALG_t msgAuthAlg)
{
  open_error_t result;
  open_buffdesc key;
  char str[OPEN_NTP_MAX_KEY_LENGTH];

  memset(str, 0, sizeof(str));
  strncpy(str, keyValue, (sizeof(str) - 1));
  key.pstart = str;
  key.size = strlen(str);

  if ((result = openapiNtpAuthTableKeyAdd(clientHandle, keyNumber, &key, isEncrypted, msgAuthAlg)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add NTP authetication table entry. (result = %d)\n", result);
  }
  else
  {
    printf("NTP authentication table entry is added successfully. (result = %d)\n", result);
  }

  return;

}

/*********************************************************************
* @purpose  Delete an authentication table entry specified by key number.                       
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)} NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                       OPEN_NTP_MAX_KEY_NUM
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpAuthTableKeyDelete(openapiClientHandle_t *clientHandle,
                                          uint32_t keyNumber)
{
  open_error_t result;

  if ((result = openapiNtpAuthTableKeyDelete(clientHandle, keyNumber)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete NTP authetication table entry. (result = %d)\n", result);
  }
  else
  {
    printf("NTP authentication table entry is deleted successfully. (result = %d)\n", result);
  }

  return;

}

/*********************************************************************
* @purpose  Set the key value of the authenticate table entry.  
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)} NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                       OPEN_NTP_MAX_KEY_NUM                            
* @param    keyValue        @b{(input)} ASCII key string OPEN_NTP_MIN_KEY_LENGTH-
*                                       OPEN__NTP_MAX_KEY_LENGTH
* @param    isEncrypted     @b{(input)}  whether the key is encrypted or not
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpAuthTableKeyValueSet(openapiClientHandle_t *clientHandle,
                              uint32_t keyNumber, char *keyValue,
                              OPEN_BOOL_t isEncrypted)
{
  open_error_t result;
  open_buffdesc key;
  char str[OPEN_NTP_MAX_KEY_LENGTH];

  memset(str, 0, sizeof(str));
  strncpy(str, keyValue, (sizeof(str) - 1));
  key.pstart = str;
  key.size = strlen(str);

  if ((result = openapiNtpAuthTableKeyValueSet(clientHandle, keyNumber, &key, isEncrypted)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set key value of NTP authetication table entry. (result = %d)\n", result);
  }
  else
  {
    printf("NTP authentication entry key value is set successfully. (result = %d)\n", result);
  }

  return;

}

/*********************************************************************
* @purpose  Set the message authentication algorithm for the specified
*           authentication key table entry index.
*
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)} NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                                    OPEN_NTP_MAX_KEY_NUM 
* @param    msgAuthAlg      @b{(input)} message authentication algorithm
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpAuthTableKeyMsgAuthAlgSet(openapiClientHandle_t *clientHandle,
                                  uint32_t keyNumber,
                                  OPEN_NTP_MSG_AUTH_ALG_t msgAlg)
{
  open_error_t result;

  if ((result = openapiNtpAuthTableKeyMsgAuthAlgSet(clientHandle, keyNumber, msgAlg)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set message algorithm of NTP authetication table entry. (result = %d)\n", result);
  }
  else
  {
    printf("NTP authentication entry algorithm is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Set a key as trusted or untrusted specified by the authenticate 
*            table entry index. 
* 
* @param    clientHandle   @b{(input)} Client handle from 
*                                       registration API
* @param    keyNumber       @b{(input)} NTP key id, range OPEN_NTP_MIN_KEY_NUM - 
*                                                    OPEN_NTP_MAX_KEY_NUM 
* @param    trustedStatus         @b{(input)} OPEN_TRUE or OPEN_FALSE 
*
*
* @returns  none 
*
* @notes Calling this API will change the running configuration of the switch
*
* @end
********************************************************************/
void ntpAuthTableKeyTrustedSet(openapiClientHandle_t *clientHandle,
                                  uint32_t keyNumber,
                                  OPEN_BOOL_t trustedStatus)
{
  open_error_t result;

  if ((result = openapiNtpAuthTableKeyTrustedSet(clientHandle, keyNumber, trustedStatus)) != OPEN_E_NONE)
  { 
    printf("Bad return code trying to set NTP authetication table entry trusted status. (result = %d)\n", result);
  }
  else
  {
    printf("NTP authentication entry trusted status is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the NTP details.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void ntpInfoGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t ntpMode;
  OPEN_NTP_SUPPORTED_MODE_t ntpSuppMode;
  OPEN_BOOL_t ntpAuthMode = OPEN_FALSE;
  uint32_t val;
  char *str;
  open_buffdesc vrfName;  
 
  if ((str = (char *)malloc(OPEN_VRF_MAX_NAME)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, OPEN_VRF_MAX_NAME+1);
  vrfName.pstart = str;
  vrfName.size = OPEN_VRF_MAX_NAME+1;

  printf("Getting NTP information. \n"); 
  
  if ((result = openapiNtpBroadcastClientModeGet(clientHandle, &ntpMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get NTP broadcast client mode. (result = %d)\n", result);
  }
  else
  {
    printf("NTP broadcast client mode: ");
    switch(ntpMode)
    {
      case OPEN_DISABLE:
        printf("Disabled \n"); 
        break;
      case OPEN_ENABLE:
        printf("Enabled \n");
        break;
      default:
        break;
    }
  }

  if ((result = openapiNtpSupportedModeGet(clientHandle, &ntpSuppMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get NTP supported mode. (result = %d)\n", result);
  }
  else
  {
    printf("NTP supported mode: ");
    switch(ntpSuppMode)
    {
      case OPEN_NTP_SUPPORTED_DISABLED:
        printf("Disabled \n");
        break;
      case OPEN_NTP_SUPPORTED_UNICAST:
        printf("Unicast \n");
        break;
      case OPEN_NTP_SUPPORTED_BROADCAST:
        printf("Broadcast \n");
        break;
      case OPEN_NTP_SUPPORTED_UNICAST_AND_BROADCAST:
        printf("Unicast Broadcast \n");
        break;
      default:
      break;
    }
  }
  
  if ((result = openapiNtpAuthenticationModeGet(clientHandle, &ntpAuthMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get NTP authentication mode. (result = %d)\n", result);
  }
  else
  {
    printf("NTP authentication mode: ");
    switch(ntpAuthMode)
    {
      case OPEN_FALSE:
        printf("Disabled \n");
        break;
      case OPEN_TRUE:
        printf("Enabled \n");
        break;
      default:
        break;
    }
  }

  if ((result = openapiNtpBroadcastDelayGet(clientHandle, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get NTP broadcast delay. (result = %d)\n", result);
  }
  else
  {
    printf("NTP broadcast delay: %d\n", val);
  }

  if ((result = openapiNtpSourceInterfaceGet(clientHandle, &val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get NTP source interface. (result = %d)\n", result);
  }
  else
  {
    printf("NTP source interface: %d\n", val);
  }

  if ((result = openapiNtpVrfNameGet(clientHandle, &vrfName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get NTP VRF name. (result = %d)\n", result);
  }
  else
  {
    printf("NTP VRF name: %s\n", str);
  }

  return;
}

/*********************************************************************
* @purpose  Get the NTP authentication table details.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void ntpAuthTableInfoGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc keyValue;
  char *str;
  uint32_t keyNumber = 0, val = 0;
  OPEN_BOOL_t status = OPEN_FALSE;

  if ((str = (char *)malloc(OPEN_NTP_MAX_KEY_LENGTH+1)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(str, 0, OPEN_NTP_MAX_KEY_LENGTH);
  keyValue.pstart = str;
  keyValue.size = OPEN_NTP_MAX_KEY_LENGTH;

  printf("\n");
  printf("NTP authentication details: \n");
  printf("---------------------------\n");

  if ((result = openapiNtpAuthTableEntryFirstGet(clientHandle, &keyNumber)) != OPEN_E_NONE)
  {
    printf("Bad return code while getting first NTP authentication table entry. (result = %d)\n", result);
    free(str);
    return;
  }
  do
  {
    printf("NTP authentication key number: %u \n", keyNumber);
    printf("Key Value: ");
    if ((result = openapiNtpAuthTableKeyValueGet(clientHandle, keyNumber, &keyValue)) != OPEN_E_NONE)
    {
      printf("Key value get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%s\n", str);
    }
    printf("Message Authentication Algorithm: ");
    if ((result = openapiNtpAuthTableKeyMsgAuthAlgGet(clientHandle, keyNumber, &val)) != OPEN_E_NONE)
    {
      printf("Message authentication algorithm get failed. (result = %d) \n", result);
    }
    else
    {
      switch(val)
      {
        case OPEN_NTP_MSG_AUTH_ALG_MD5:
          printf("MD5 \n"); 
          break;
        case OPEN_NTP_MSG_AUTH_ALG_SHA1:
          printf("SHA1 \n");
          break;
        case OPEN_NTP_MSG_AUTH_ALG_SHA256:
          printf("SHA256 \n"); 
          break;
        default:
          result = OPEN_E_PARAM;
          break;
      }
    }
    printf("Trusted Status: ");
    if ((result = openapiNtpAuthTableKeyTrustedGet(clientHandle, keyNumber, &status)) != OPEN_E_NONE)
    {
      printf("Unable to get NTP auth table trusted status. (result = %d) \n", result);
    }
    else
    {
      switch(status)
      {
        case OPEN_TRUE:
          printf("Yes \n");
          break;
        case OPEN_FALSE:
          printf("No \n");
          break;
        default:
          break;
      }
    }
    printf("\n");
  }while(openapiNtpAuthTableEntryGetNext(clientHandle, &keyNumber, &keyNumber) == OPEN_E_NONE);

  free(str);
  return;
}

/*********************************************************************
* @purpose  Get the NTP software information.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void ntpSwInfoGet(openapiClientHandle_t *clientHandle)
{
  uint32_t version;
  char str[OPEN_NTP_BUFFER_SIZE] = {0};
  open_buffdesc ntpBuf;
  open_error_t result = OPEN_E_NONE;

  memset(str, 0, OPEN_NTP_BUFFER_SIZE);
  ntpBuf.pstart = str;
  ntpBuf.size = OPEN_NTP_BUFFER_SIZE;

  printf("NTP protocol version: ");
  if ((result = openapiNtpVersionGet(clientHandle, &version)) != OPEN_E_NONE)
  {
    printf("NTP version get failed. (result = %d) \n", result);
  }
  else
  {
    printf("%d\n", version);
  }

  printf("NTP Software Name: ");
  if ((result = openapiNtpEntSoftwareNameGet(clientHandle, &ntpBuf)) != OPEN_E_NONE)
  {
    printf("NTP software name get failed. (result = %d) \n", result);
  }
  else
  {
    printf("%s\n", str);
  }

  memset(str, 0, OPEN_NTP_BUFFER_SIZE);
  ntpBuf.size = OPEN_NTP_BUFFER_SIZE;

  printf("NTP Software Version: ");
  if ((result = openapiNtpEntSoftwareVersionGet(clientHandle, &ntpBuf)) != OPEN_E_NONE)
  {
    printf("NTP software version get failed. (result = %d) \n", result);
  }
  else
  {
    printf("%s\n", str);
  }

  memset(str, 0, OPEN_NTP_BUFFER_SIZE);
  ntpBuf.size = OPEN_NTP_BUFFER_SIZE;

  printf("NTP System Type: ");
  if ((result = openapiNtpEntSystemTypeGet(clientHandle, &ntpBuf)) != OPEN_E_NONE)
  {
    printf("NTP system type get failed. (result = %d) \n", result);
  }
  else
  {
    printf("%s\n", str);
  }

  return;
}

/*********************************************************************
* @purpose  Get the NTP packets information.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void ntpPktInfoGet(openapiClientHandle_t *clientHandle)
{
  uint32_t val = 0;
  open_error_t result = OPEN_E_NONE;

  printf("NTP In Packets: ");
  if ((result = openapiNtpEntStatusInPktsGet(clientHandle, &val)) != OPEN_E_NONE)
  {
    printf("NTP in packets count get failed. (result = %d) \n", result);
  }
  else
  {
    printf("%u\n", val);
  }

  val = 0;
  printf("NTP Out Packets: ");
  if ((result = openapiNtpEntStatusOutPktsGet(clientHandle, &val)) != OPEN_E_NONE)
  {
    printf("NTP out packets count get failed. (result = %d) \n", result);
  }
  else
  {
    printf("%u\n", val);
  }

  val = 0;
  printf("NTP Old Version Packets: ");
  if ((result = openapiNtpEntStatusBadVersionGet(clientHandle, &val)) != OPEN_E_NONE)
  {
    printf("NTP old version packets count get failed. (result = %d) \n", result);
  }
  else
  {
    printf("%u\n", val);
  }

  val = 0;
  printf("NTP Protocol Error Packets: ");
  if ((result = openapiNtpEntStatusProtocolErrorGet(clientHandle, &val)) != OPEN_E_NONE)
  {
    printf("NTP protocol error packets count get failed. (result = %d) \n", result);
  }
  else
  {
    printf("%u\n", val);
  }
  return;
}

/*********************************************************************
* @purpose  Get the NTP associations details.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void ntpAssocInfoGet(openapiClientHandle_t *clientHandle)
{
  uint32_t assocId = 0, val = 0;
  char str[OPEN_NTP_BUFFER_SIZE] = {0};
  open_buffdesc ntpBuf;
  open_error_t result = OPEN_E_NONE;
  OPEN_NTP_ADDRESS_TYPE_t addressType = OPEN_NTP_ADDRESS_UNKNOWN;

  memset(str, 0, OPEN_NTP_BUFFER_SIZE);
  ntpBuf.pstart = str;
  ntpBuf.size = OPEN_NTP_BUFFER_SIZE;

  while (OPEN_E_NONE == openapiNtpAssocIdNextGet(clientHandle, &assocId))
  {
    printf("Association ID: %u\n", assocId);

    printf("Association Name: ");
    if ((result = openapiNtpAssocNameGet(clientHandle, assocId, &ntpBuf)) != OPEN_E_NONE)
    {
      printf("NTP association name get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%s\n", str);
    }

    memset(str, 0, OPEN_NTP_BUFFER_SIZE);
    ntpBuf.size = OPEN_NTP_BUFFER_SIZE;
    
    printf("Association Reference ID: ");
    if ((result = openapiNtpAssocRefIdGet(clientHandle, assocId, &ntpBuf)) != OPEN_E_NONE)
    {
      printf("NTP association reference ID get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%s\n", str);
    }

    printf("Association Address Type: ");
    if ((result = openapiNtpAssocAddressTypeGet(clientHandle, assocId, &addressType)) != OPEN_E_NONE)
    {
      printf("NTP association address type get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%u\n", addressType);
    }
   
    memset(str, 0, OPEN_NTP_BUFFER_SIZE);
    ntpBuf.size = OPEN_NTP_BUFFER_SIZE;

    printf("Association Address: ");
    if ((result = openapiNtpAssocAddressGet(clientHandle, assocId, &ntpBuf)) != OPEN_E_NONE)
    {
      printf("NTP association address get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%s\n", str);
    }

    memset(str, 0, OPEN_NTP_BUFFER_SIZE);
    ntpBuf.size = OPEN_NTP_BUFFER_SIZE;

    printf("Association Offset: ");
    if ((result = openapiNtpAssocOffsetGet(clientHandle, assocId, &ntpBuf)) != OPEN_E_NONE)
    {
      printf("NTP association offset get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%s\n", str);
    } 

    printf("Association Stratum: ");
    if ((result = openapiNtpAssocStratumGet(clientHandle, assocId, &val)) != OPEN_E_NONE)
    {
      printf("NTP association stratum get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%u\n", val);
    }

    memset(str, 0, OPEN_NTP_BUFFER_SIZE);
    ntpBuf.size = OPEN_NTP_BUFFER_SIZE;

    printf("Association Jitter: ");
    if ((result = openapiNtpAssocStatusJitterGet(clientHandle, assocId, &ntpBuf)) != OPEN_E_NONE)
    {
      printf("NTP association jitter get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%s\n", str);
    }

    memset(str, 0, OPEN_NTP_BUFFER_SIZE);
    ntpBuf.size = OPEN_NTP_BUFFER_SIZE;

    printf("Association Delay: ");
    if ((result = openapiNtpAssocStatusDelayGet(clientHandle, assocId, &ntpBuf)) != OPEN_E_NONE)
    {
      printf("NTP association delay get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%s\n", str);
    }

    memset(str, 0, OPEN_NTP_BUFFER_SIZE);
    ntpBuf.size = OPEN_NTP_BUFFER_SIZE;

    printf("Association Dispersion: ");
    if ((result = openapiNtpAssocStatusDispersionGet(clientHandle, assocId, &ntpBuf)) != OPEN_E_NONE)
    {
      printf("NTP association dispersion get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%s\n", str);
    }

    memset(str, 0, OPEN_NTP_BUFFER_SIZE);
    ntpBuf.size = OPEN_NTP_BUFFER_SIZE;

    printf("\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the NTP associations statictics.
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void ntpAssocStatsGet(openapiClientHandle_t *clientHandle)
{
  uint32_t assocId = 0, val = 0;
  open_error_t result = OPEN_E_NONE;

  while (OPEN_E_NONE == openapiNtpAssocIdNextGet(clientHandle, &assocId))
  {
    printf("\nAssociation ID: %u\n", assocId);

    val = 0;
    printf("NTP In Packets: ");
    if ((result = openapiNtpAssocStatInPktsGet(clientHandle, assocId, &val)) != OPEN_E_NONE)
    {
      printf("NTP association in packets count get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%u\n", val);
    }
      
    val = 0; 
    printf("NTP Out Packets: ");
    if ((result = openapiNtpAssocStatOutPktsGet(clientHandle, assocId, &val)) != OPEN_E_NONE)
    {
      printf("NTP association out packets count get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%u\n", val);
    } 

    val = 0;
    printf("NTP Protocol Error Packets: ");
    if ((result = openapiNtpAssocStatProtocolErrorGet(clientHandle, assocId, &val)) != OPEN_E_NONE)
    {
      printf("NTP association protocol error packets count get failed. (result = %d) \n", result);
    }
    else
    {
      printf("%u\n", val);
    }

    printf("\n");
  }
  return;
}


/*********************************************************************
* @purpose  Perform NTP OpEN APIs Sanity
*
* @param    clientHandle     @b{(input)}   client handle from registration API
*
* @returns  none
* 
* @notes
* 
* @end
*********************************************************************/
void ntpOpENAPISanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc ntpServer;
  uint32_t ntpServerMaxLen, maxNtpServers;  
  uint32_t stratum = 0;
  uint32_t ntpServerRefIdStrLen = 0;
  OPEN_NTP_SERVER_MODE_t ntpServerMode;
  uint32_t currNtpServers = 0;
  open_buffdesc referenceId;
  OPEN_NTP_ADDRESS_TYPE_t addressType;
  OPEN_NTP_SUPPORTED_MODE_t ntpSuppMode = OPEN_NTP_SUPPORTED_DISABLED;
  OPEN_CONTROL_t ntpMode = OPEN_DISABLE;
  char *str;
  OPEN_CONTROL_t mode = OPEN_DISABLE; 
 
  printf("Testing NTP OpEN APIs sanity:\n"); 

  if ((result = openapiNtpServerMaxAddrLenGet(clientHandle, &ntpServerMaxLen)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get maximum length of NTP server. (result = %d)\n", result);
    return;
  }

  if ((str = (char *)malloc(ntpServerMaxLen)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  memset(&referenceId,0,sizeof(referenceId));
  memset(str, 0, ntpServerMaxLen);
  ntpServer.pstart = str;
  ntpServer.size = ntpServerMaxLen;
  
  /* openapiNtpServerMaxAddrLenGet () */
  printf("\nTesting openapiNtpServerMaxAddrLenGet(): \n");
  result = openapiNtpServerMaxAddrLenGet(NULL, &ntpServerMaxLen);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpServerMaxAddrLenGet(clientHandle, NULL);
  printf("NULL parameter to NTP server length. (result = %d)\n", result);

  printf("openapiNtpServerMaxAddrLenGet() sanity successful. \n");

  /* openapiNtpServerAdd() */
  printf("\nTesting openapiNtpServerAdd(): \n");
  result = openapiNtpServerAdd(NULL, &ntpServer);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpServerAdd(clientHandle, NULL);
  printf("NULL buff descriptor to NTP server. (result = %d)\n", result);

  printf("openapiNtpServerAdd() sanity successful. \n");

  /* openapiNtpServersMaxSupportedGet() */
  printf("\nTesting openapiNtpServersMaxSupportedGet(): \n");
  result = openapiNtpServersMaxSupportedGet(NULL, &maxNtpServers);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpServersMaxSupportedGet(clientHandle, NULL);
  printf("NULL parameter to maximum no. of NTP servers. (result = %d)\n", result);

  printf("openapiNtpServersMaxSupportedGet() sanity successful. \n");

  /* openapiNtpServerDelete() */
  printf("\nTesting openapiNtpServerDelete(): \n");

  result = openapiNtpServerDelete(NULL, &ntpServer);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpServerDelete(clientHandle, NULL);
  printf("NULL buff descriptor to NTP server. (result = %d)\n", result);

  printf("openapiNtpServerDelete() sanity successful. \n");
  

  /* openapiNtpServerFirstGet() */
  printf("\nTesting openapiNtpServerFirstGet(): \n");

  result = openapiNtpServerFirstGet(NULL, &ntpServer);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpServerFirstGet(clientHandle, NULL);
  printf("NULL buff descriptor to NTP server. (result = %d)\n", result);

  printf("openapiNtpServerFirstGet() sanity successful. \n");

  /* openapiNtpServerNextGet() */
  printf("\nTesting openapiNtpServerNextGet(): \n");

  result = openapiNtpServerNextGet(NULL, &ntpServer, &ntpServer);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpServerNextGet(clientHandle, NULL, &ntpServer);
  printf("NULL buff descriptor to previous NTP server. (result = %d)\n", result);

  result = openapiNtpServerNextGet(clientHandle, &ntpServer, NULL);
  printf("NULL buff descriptor to next NTP server. (result = %d)\n", result);

  printf("openapiNtpServerNextGet() sanity successful. \n");

  /* openapiNtpServerAddressTypeGet() */
  printf("\nTesting openapiNtpServerAddressTypeGet(): \n");

  result = openapiNtpServerAddressTypeGet(NULL, &ntpServer, &addressType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpServerAddressTypeGet(clientHandle, NULL, &addressType);
  printf("NULL buff descriptor to NTP server. (result = %d)\n", result);

  result = openapiNtpServerAddressTypeGet(clientHandle, &ntpServer, NULL);
  printf("NULL address type. (result = %d)\n", result);

  printf("openapiNtpServerAddressTypeGet() sanity successful. \n");

  /* openapiNtpBroadcastClientModeSet() */
  printf("\nTesting openapiNtpBroadcastClientModeSet(): \n");

  result = openapiNtpBroadcastClientModeSet(NULL, mode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpBroadcastClientModeSet(clientHandle, 10);
  printf("Invalid NTP broadcast client mode. (result = %d)\n", result);

  printf("openapiNtpBroadcastClientModeSet() sanity successful. \n");

  /* openapiNtpBroadcastClientModeGet() */
  printf("\nTesting openapiNtpBroadcastClientModeGet(): \n");

  result = openapiNtpBroadcastClientModeGet(NULL, &ntpMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpBroadcastClientModeGet(clientHandle, NULL);
  printf("NULL parameter to NTP broadcast client mode. (result = %d)\n", result);

  printf("openapiNtpBroadcastClientModeGet() sanity successful. \n");

  /* openapiNtpSupportedModeGet() */
  printf("\nTesting openapiNtpSupportedModeGet(): \n");

  result = openapiNtpSupportedModeGet(NULL, &ntpSuppMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpSupportedModeGet(clientHandle, NULL);
  printf("NULL parameter to NTP supported mode. (result = %d)\n", result);

  printf("openapiNtpSupportedModeGet() sanity successful. \n");

  /* openapiActiveNtpServerAddressTypeGet() */
  printf("\nTesting openapiActiveNtpServerAddressTypeGet(): \n");
  result = openapiActiveNtpServerAddressTypeGet(NULL, &addressType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiActiveNtpServerAddressTypeGet(clientHandle, NULL);
  printf("NULL address type. (result = %d)\n", result);

  printf("openapiActiveNtpServerAddressTypeGet() sanity successful. \n");


  /* openapiActiveNtpServerIpAddressGet() */
  printf("\nTesting openapiActiveNtpServerIpAddressGet(): \n");

  result = openapiActiveNtpServerIpAddressGet(NULL, &ntpServer);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiActiveNtpServerIpAddressGet(clientHandle, NULL);
  printf("NULL buff descriptor to NTP server. (result = %d)\n", result); 

  printf("openapiActiveNtpServerIpAddressGet() sanity successful. \n");

  /* openapiActiveNtpServerStratumGet() */
  printf("\nTesting openapiActiveNtpServerStratumGet(): \n");

  result = openapiActiveNtpServerStratumGet(NULL, &stratum);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiActiveNtpServerStratumGet(clientHandle, NULL);
  printf("NULL NTP server stratum. (result = %d)\n", result);

  printf("openapiActiveNtpServerStratumGet() sanity successful. \n");

  /* openapiActiveNtpServerReferenceIdGet() */
  printf("\nTesting openapiActiveNtpServerReferenceIdGet(): \n");

  result = openapiActiveNtpServerReferenceIdGet(NULL, &referenceId);
  printf("NULL Client Handle. (result = %d)\n", result);
 
  result = openapiActiveNtpServerReferenceIdGet(clientHandle, NULL);
  printf("NULL NTP reference ID. (result = %d)\n", result);

  printf("openapiActiveNtpServerReferenceIdGet() sanity successful. \n");

  /* openapiNtpServerReferenceStringLengthGet() */
  printf("\nTesting openapiNtpServerReferenceStringLengthGet(): \n");

  result = openapiNtpServerReferenceStringLengthGet(NULL, &ntpServerRefIdStrLen);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpServerReferenceStringLengthGet(clientHandle, NULL);
  printf("NULL parameter to NTP server reference ID. (result = %d)\n", result);

  printf("openapiNtpServerReferenceStringLengthGet() sanity successful. \n");

  /* openapiActiveNtpServerModeGet() */
  printf("\nTesting openapiActiveNtpServerModeGet(): \n");

  result = openapiActiveNtpServerModeGet(NULL, &ntpServerMode);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiActiveNtpServerModeGet(clientHandle, NULL);
  printf("NULL NTP server mode. (result = %d)\n", result);

  printf("openapiActiveNtpServerModeGet() sanity successful. \n");

  /* openapiNtpServerCurrentEntriesGet() */
  printf("\nTesting openapiNtpServerCurrentEntriesGet(): \n");

  result = openapiNtpServerCurrentEntriesGet(NULL, &currNtpServers);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiNtpServerCurrentEntriesGet(clientHandle, NULL);
  printf("NULL parameter to current NTP servers. (result = %d)\n", result);
  
  printf("openapiNtpServerCurrentEntriesGet() sanity successful. \n");

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, arg1, arg2, arg3;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  
  if (argc < 2)
  {
    printNtpAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("ntp_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting NTP API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printNtpAppMenu();
        exit(1);
      }
      ntpServerAdd(&clientHandle, argv[2]);
      break;
    case 2:
      if (argc != 3)
      {
        printNtpAppMenu();
        exit(1);
      }
      ntpServerDelete(&clientHandle, argv[2]);
      break;
    case 3:
      if (argc != 2)
      {
        printNtpAppMenu();
        exit(1);
      }
      ntpServerInfoGet(&clientHandle);
      break;
    case 4:
      if (argc != 3)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ntpModeSet(&clientHandle, arg1);
      break;
    case 5:
      if (argc != 2)
      {
        printNtpAppMenu();
        exit(1);
      }
      ntpInfoGet(&clientHandle);
      break;
    case 6:
      if (argc != 3)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ntpAuthenticationModeSet(&clientHandle, arg1);
      break;
    case 7:
      if (argc != 3)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ntpBroadcastDelaySet(&clientHandle, arg1);
      break;
    case 8:
      if (argc != 3)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ntpSourceInterfaceSet(&clientHandle, arg1);
      break;
    case 9:
      if (argc != 3)
      {
        printNtpAppMenu();
        exit(1);
      }
      ntpVrfNameSet(&clientHandle, argv[2]);
      break;
    case 10:
      if (argc != 4)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      ntpServerTableAuthKeySet(&clientHandle, argv[2], arg1);
      break;
    case 11:
      if (argc != 4)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      ntpServerTableMaxPollSet(&clientHandle, argv[2], arg1);
      break;
    case 12:
      if (argc != 4)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      ntpServerTableMinPollSet(&clientHandle, argv[2], arg1);
      break;
    case 13:
      if (argc != 4)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      ntpServerTablePreferSet(&clientHandle, argv[2], arg1);
      break;
    case 14:
      if (argc != 4)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      ntpServerTableBurstSet(&clientHandle, argv[2], arg1);
      break;
    case 15:
      if (argc != 4)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      ntpServerTableIBurstSet(&clientHandle, argv[2], arg1);
      break;
    case 16:
      if (argc != 4)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[3]);
      ntpServerTableVersionSet(&clientHandle, argv[2], arg1);
      break;
    case 17:
      if (argc != 6)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = strtoul(argv[2], NULL, 0);
      arg2 = atoi(argv[4]);
      arg3 = atoi(argv[5]);
      ntpAuthTableKeyAdd(&clientHandle, arg1, argv[3], arg2, arg3);
      break;
    case 18:
      if (argc != 3)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ntpAuthTableKeyDelete(&clientHandle, arg1);
      break;
    case 19:
      if (argc != 5)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[4]);
      ntpAuthTableKeyValueSet(&clientHandle, arg1, argv[3], arg2);
      break;
    case 20:
      if (argc != 4)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      ntpAuthTableKeyMsgAuthAlgSet(&clientHandle, arg1, arg2);
      break;
    case 21:
      if (argc != 4)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      ntpAuthTableKeyTrustedSet(&clientHandle, arg1, arg2);
      break; 
    case 22:
      if (argc != 2)
      {
        printNtpAppMenu();
        exit(1);
      }
      ntpAuthTableInfoGet(&clientHandle);
      break;
    case 23:
      if (argc != 2)
      {
        printNtpAppMenu();
        exit(1);
      }
      ntpOpENAPISanity(&clientHandle);
      break;
    case 24:
      if (argc != 3)
      {
        printNtpAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      ntpBroadcastClientModeSet(&clientHandle, arg1);
      break;
    case 25:
      if (argc != 2)
      {
        printNtpAppMenu();
        exit(1);
      }
      ntpSwInfoGet(&clientHandle);
      break;
    case 26:
      if (argc != 2)
      {
        printNtpAppMenu();
        exit(1);
      }
      ntpPktInfoGet(&clientHandle);
      break; 
    case 27:
      if (argc != 2)
      {
        printNtpAppMenu();
        exit(1);
      }
      ntpAssocInfoGet(&clientHandle);
      break;   
    case 28:
      if (argc != 2)
      {
        printNtpAppMenu();
        exit(1);
      }
      ntpAssocStatsGet(&clientHandle);
      break;
 
    default:
      printNtpAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping NTP API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
