/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  diffserv_policy_example.c
*
* @purpose   OpEN diffServPolicy example.
*
* @component OpEN
*
* @create    05/25/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_diffserv_policy.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the diffServPolicy Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Determine next sequential row in the Policy Table: %s 1 <prevPolicyIndex>\n", name);
  printf("Test 2: Get the policy name: %s 2 <policyIndex>\n", name);
  printf("Test 3: Set the policy name: %s 3 <policyIndex><policy-name>\n", name);
  printf("Test 4: Translate a policy name into its associated Policy id in Table index: %s 4 <stringPtr>\n", name);
  printf("Test 5: Check if class definition is suitable for inclusion by specified policy: %s 5 <classIndex><policyIndex>\n", name);
  printf("Test 6: Get the current status of the specified Policy: %s 6 <policyIndex>\n", name);
  printf("Test 7: Create a policy in the Policy Table for the specified index: %s 7 <policyIndex>\n", name);
  printf("Test 8: Delete a Policy in Table for the specified index: %s 8 <policyIndex>\n", name);
  printf("Test 9: Get the Policy type/direction of policy definition: %s 9 <policyIndex>\n", name);
  printf("Test 10: Set the type of policy : %s 10 <policyIndex><policyType>\n", name);
  printf("Test 11: Obtain the next available index value to be used when creating a new row in the Policy-Class Instance Table: %s 11 <policyIndex>\n", name);
  printf("Test 12: Get the L3 protocol of class definition: %s 12 <classIndex><pClassL3Proto>\n", name);
  printf("Test 13: Determine next sequential row in the Policy-Class Instance Table: %s 13 <prevPolicyIndex><prevPolicyInstIndex>\n", name);
  printf("Test 14: Get the class index referenced as the policy class instance: %s 14 <policyIndex><policyInstIndex>\n", name);
  printf("Test 15: Set the class index referenced as the policy class instance: %s 15 <policyIndex><policyInstIndex><classIndex>\n", name);
  printf("Test 16: Find the policy instance index from the policy and class indexes: %s 16 <policyIndex><classIndex>\n", name);
  printf("Test 17: Create a new row in the Policy-Class Instance Table for the specified index: %s 17 <policyIndex><policyInstIndex><activateRow>\n", name);
  printf("Test 18: Delete a row from the Policy-Class Instance Table for the specified index: %s 18 <policyIndex><policyInstIndex>\n", name);
  printf("Test 19: Get the current status of the specified Policy-Class Instance Table row: %s 19 <policyIndex><policyInstIndex>\n", name);
  printf("Test 20: Obtain the next available index value to be used when creating a new row in the Policy Table: %s 20 <pPolicyIndex>\n", name);
  printf("Test 21: Get the storage type for the specified Policy Attribute Table row: %s 21 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 22: Get the current status of the specified Policy Attribute Table row: %s 22 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 23: Obtain the next available index value to be used when creating a new row in the Policy Attribute Table: %s 23 <policyIndex><policyInstIndex>\n", name);
  printf("Test 24: Retrieve the maximum policy attribute index value allowed for the Policy Attribute Table: %s 24 \n", name);
  printf("Test 25: Get the Policy Attribute statement entry type: %s 25 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 26: Get the Policy Attribute statement entry associated data: %s 26 <policyIndex><policyInstIndex><policyAttrIndex><entryType>\n", name);
  printf("Test 27: Create a new row in the Policy Attribute Table for the specified index: %s 27 <policyIndex><policyInstIndex><policyAttrIndex><activateRow>\n", name);
  printf("Test 28: Delete a row from the Policy Attribute Table for the specified index: %s 28 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 29: Set the Policy Attribute statement entry type and associated data: %s 29 <policyIndex><policyInstIndex><policyAttrIndex><entryType><data>\n", name);
  printf("Test 30: Find the Policy Attribute index for the specified attribute: %s 30 <policyIndex><policyInstIndex><inEntry>\n", name);
  printf("Test 31: Set the simple policing committed rate attribute statement value: %s 31 <policyIndex><policyInstIndex><policyAttrIndex><entryType><rate_unit><rate>\n", name);
  printf("Test 32: Get the policing committed rate unit attribute statement value: %s 32 <policyIndex><policyInstIndex><policyAttrIndex><entryType>\n", name);
  printf("Test 33: Set the policing committed burst size attribute statement value: %s 33 <policyIndex><policyInstIndex><policyAttrIndex><entryType><burstType><burst>\n", name);
  printf("Test 34: Get the policing committed burst size attribute statement value: %s 34 <policyIndex><policyInstIndex><policyAttrIndex><entryType><burstType>\n", name);
  printf("Test 35: Get the policing committed rate attribute statement value: %s 35 <policyIndex><policyInstIndex><policyAttrIndex><entryType>\n", name);
  printf("Test 36: Set the policing exceed value attribute statement value: %s 36 <policyIndex><policyInstIndex><policyAttrIndex><value>\n", name);
  printf("Test 37: Get the policing exceed value attribute statement value: %s 37 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 38: Set the policing exceed action attribute statement value: %s 38 <policyIndex><policyInstIndex><policyAttrIndex><action>\n", name);
  printf("Test 39: Get the policing exceed action attribute statement value: %s 39 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 40: Determine if a DiffServ class is valid for use as a policing color aware designation: %s 40 <policyIndex><policyInstIndex><classIndex>\n", name);
  printf("Test 41: Get the policing color conform value attribute statement value: %s 41 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 42: Set the policing color conform attribute class index value: %s 42 <policyIndex><policyInstIndex><policyAttrIndex><classIndex>\n", name);
  printf("Test 43: Get the policing color conform attribute class index value: %s 43 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 44: Set the policing nonconform action attribute statement value: %s 44 <policyIndex><policyInstIndex><policyAttrIndex><action>\n", name);
  printf("Test 45: Get the policing nonconform action attribute statement value: %s 45 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 46: Set the policing nonconform value attribute statement value: %s 46 <policyIndex><policyInstIndex><policyAttrIndex><value>\n", name);
  printf("Test 47: Get the policing nonconform value attribute statement value: %s 47 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 48: Set the policing conform value attribute statement value: %s 48 <policyIndex><policyInstIndex><policyAttrIndex><value>\n", name);
  printf("Test 49: Get the policing conform value attribute statement value: %s 49 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 50: Determine next sequential row in the Policy Attribute Table: %s 50 <prevPolicyIndex><prevPolicyInstIndex><prevPolicyAttrIndex>\n", name);
  printf("Test 51: Set the policing conform action attribute statement value: %s 51 <policyIndex><policyInstIndex><policyAttrIndex><action>\n", name);
  printf("Test 52: Get the policing conform action attribute statement value: %s 52 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 53: Set the two-rate policing peak rate attribute statement value: %s 53 <policyIndex><policyInstIndex><policyAttrIndex><rate>\n", name);
  printf("Test 54: Set the simple policing committed rate attribute statement value: %s 54 <policyIndex><policyInstIndex><policyAttrIndex><rate>\n", name);
  printf("Test 55: Get the two-rate policing peak rate attribute statement value: %s 55 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 56: Get the simple policing committed rate attribute statement value: %s 56 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 57: Set the two-rate policing peak rate attribute statement value as per unit: %s 57 <policyIndex><policyInstIndex><policyAttrIndex><rate_unit><rate>\n", name);
  printf("Test 58: Get the two-rate policing peak rate attribute statement value as per unit: %s 58 <policyIndex><policyInstIndex><policyAttrIndex><rate_unit>\n", name);
  printf("Test 59: Get the policing peak rate attribute statement value: %s 59 <policyIndex><policyInstIndex><policyAttrIndex><entryType>\n", name);
  printf("Test 60: Get the two-rate policing committed rate attribute statement value: %s 60 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 61: Set the two-rate policing committed rate attribute statement value: %s 61 <policyIndex><policyInstIndex><policyAttrIndex><rate>\n", name);
  printf("Test 62: Set the policing color exceed attribute class index value: %s 62 <policyIndex><policyInstIndex><policyAttrIndex><classIndex>\n", name);
  printf("Test 63: Get the policing color exceed attribute class index value: %s 63 <policyIndex><policyInstIndex><policyAttrIndex>\n", name);
  printf("Test 64: Run API sanity checks: %s 64 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *client_handle)
{
  open_error_t result = OPEN_E_NONE;

  uint32_t pClassIndex = 0;
  uint32_t prevPolicyInstIndex = 0;
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_t pEntryType = 0;
  uint32_t rate = 0;
  uint32_t pPolicyIndex = 0;
  uint32_t prevPolicyIndex = 0;
  uint32_t pRate = 0;
  uint32_t burst = 0;
  OPEN_DIFFSERV_ROW_STATUS_t rowVal = 0;
  OPEN_DIFFSERV_STORAGE_TYPE_t storageType = 0;
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_t inEntry = 0;
  uint32_t policyIndex = 0;
  uint32_t prevPolicyAttrIndex = 0;
  uint32_t pRate_percent = 0;
  uint32_t policyAttrIndex = 0;
  OPEN_RATE_UNIT_t pRate_unit = 0;
  OPEN_DIFFSERV_POLICY_TYPE_t pPolicyType = 0;
  uint32_t pIndexMax = 0;
  uint32_t classIndex = 0;
  uint32_t data = 0;
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType = 0;
  uint32_t pPolicyInstIndex = 0;
  uint32_t policyInstIndex = 0;
  uint32_t value = 0;
  uint32_t pVal = 0;
  uint32_t pPolicyAttrIndex = 0;
  OPEN_DIFFSERV_POLICE_ACT_t action = 0;
  OPEN_BURST_POLICE_TYPE_t burstType = 0;
  char buffer[128];
  open_buffdesc bufd;
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  badBufdescPointer.size = 0;
  bufd.pstart = buffer;
  bufd.size = sizeof(buffer);
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;

  printf("Testing diffServPolicy OpEN APIs sanity:\n");

  printf("Testing openapiDiffServPolicyGetNext():\n");
  result = openapiDiffServPolicyGetNext(NULL, prevPolicyIndex, &policyIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyGetNext(client_handle, prevPolicyIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyNameGet():\n");
  result = openapiDiffServPolicyNameGet(NULL, policyIndex, &bufd);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyNameGet(client_handle, policyIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDiffServPolicyNameGet(client_handle, policyIndex, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiDiffServPolicyNameGet(client_handle, policyIndex, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyNameSet():\n");
  result = openapiDiffServPolicyNameSet(NULL, policyIndex, &bufd);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyNameSet(client_handle, policyIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDiffServPolicyNameSet(client_handle, policyIndex, &badBufdescPointer);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiDiffServPolicyNameSet(client_handle, policyIndex, &zeroLenBuf);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyNameToIndex():\n");
  result = openapiDiffServPolicyNameToIndex(NULL, &bufd, &pPolicyIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyNameToIndex(client_handle, NULL, &pPolicyIndex);
  printf("NULL argument 3:(result = %d)\n", result);
  result = openapiDiffServPolicyNameToIndex(client_handle, &badBufdescPointer, &pPolicyIndex);
  printf("NULL buffdesc pstart argument 3:(result = %d)\n", result);
  result = openapiDiffServPolicyNameToIndex(client_handle, &zeroLenBuf, &pPolicyIndex);
  printf("buffdesc 0 length size argument 3:(result = %d)\n", result);
  result = openapiDiffServPolicyNameToIndex(client_handle, &bufd, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyRowStatusGet():\n");
  result = openapiDiffServPolicyRowStatusGet(NULL, policyIndex, &rowVal);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyRowStatusGet(client_handle, policyIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyTypeGet():\n");
  result = openapiDiffServPolicyTypeGet(NULL, policyIndex, &pPolicyType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyTypeGet(client_handle, policyIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyInstIndexNext():\n");
  result = openapiDiffServPolicyInstIndexNext(NULL, policyIndex, &pPolicyInstIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyInstIndexNext(client_handle, policyIndex, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyInstGetNext():\n");
  result = openapiDiffServPolicyInstGetNext(NULL, prevPolicyIndex, prevPolicyInstIndex, &pPolicyIndex, &pPolicyInstIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyInstGetNext(client_handle, prevPolicyIndex, prevPolicyInstIndex, NULL, &pPolicyInstIndex);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiDiffServPolicyInstGetNext(client_handle, prevPolicyIndex, prevPolicyInstIndex, &pPolicyIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyInstClassIndexGet():\n");
  result = openapiDiffServPolicyInstClassIndexGet(NULL, policyIndex, policyInstIndex, &pClassIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyInstClassIndexGet(client_handle, policyIndex, policyInstIndex, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyInstIndexFromClass():\n");
  result = openapiDiffServPolicyInstIndexFromClass(NULL, policyIndex, classIndex, &pPolicyInstIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyInstIndexFromClass(client_handle, policyIndex, classIndex, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyInstRowStatusGet():\n");
  result = openapiDiffServPolicyInstRowStatusGet(NULL, policyIndex, policyInstIndex, &rowVal);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyInstRowStatusGet(client_handle, policyIndex, policyInstIndex, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyIndexNext():\n");
  result = openapiDiffServPolicyIndexNext(NULL, &pPolicyIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyIndexNext(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStorageTypeGet():\n");
  result = openapiDiffServPolicyAttrStorageTypeGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &storageType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStorageTypeGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrRowStatusGet():\n");
  result = openapiDiffServPolicyAttrRowStatusGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &rowVal);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrRowStatusGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrIndexNext():\n");
  result = openapiDiffServPolicyAttrIndexNext(NULL, policyIndex, policyInstIndex, &pPolicyAttrIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrIndexNext(client_handle, policyIndex, policyInstIndex, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrIndexMaxGet():\n");
  result = openapiDiffServPolicyAttrIndexMaxGet(NULL, &pIndexMax);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrIndexMaxGet(client_handle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtEntryTypeGet():\n");
  result = openapiDiffServPolicyAttrStmtEntryTypeGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &pEntryType);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtEntryTypeGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtEntryDataGet():\n");
  result = openapiDiffServPolicyAttrStmtEntryDataGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, entryType, &data);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtEntryDataGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, NULL);
  printf("NULL argument 6:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrIndexFind():\n");
  result = openapiDiffServPolicyAttrIndexFind(NULL, policyIndex, policyInstIndex, inEntry, &pPolicyAttrIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrIndexFind(client_handle, policyIndex, policyInstIndex, inEntry, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceDataRateUnitGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceDataRateUnitGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, entryType, &pRate_unit);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceDataRateUnitGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, NULL);
  printf("NULL argument 6:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceBurstGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceBurstGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, entryType, burstType, &burst);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceBurstGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, burstType, NULL);
  printf("NULL argument 7:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceDataRatePercentGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceDataRatePercentGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, entryType, &pRate_percent);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceDataRatePercentGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, NULL);
  printf("NULL argument 6:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceExceedValGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceExceedValGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &data);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceExceedValGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceExceedActGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceExceedActGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &action);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceExceedActGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceColorConformValGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceColorConformValGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &pVal);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceColorConformValGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceColorConformIndexGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceColorConformIndexGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &pClassIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceColorConformIndexGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceNonconformActGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceNonconformActGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &action);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceNonconformActGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceNonconformValGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceNonconformValGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &value);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceNonconformValGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceConformValGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceConformValGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &value);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceConformValGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrGetNext():\n");
  result = openapiDiffServPolicyAttrGetNext(NULL, prevPolicyIndex, prevPolicyInstIndex, prevPolicyAttrIndex, &pPolicyIndex, &pPolicyInstIndex, &pPolicyAttrIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrGetNext(client_handle, prevPolicyIndex, prevPolicyInstIndex, prevPolicyAttrIndex, NULL, &pPolicyInstIndex, &pPolicyAttrIndex);
  printf("NULL argument 7:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrGetNext(client_handle, prevPolicyIndex, prevPolicyInstIndex, prevPolicyAttrIndex, &pPolicyIndex, NULL, &pPolicyAttrIndex);
  printf("NULL argument 7:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrGetNext(client_handle, prevPolicyIndex, prevPolicyInstIndex, prevPolicyAttrIndex, &pPolicyIndex, &pPolicyInstIndex, NULL);
  printf("NULL argument 7:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceConformActGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceConformActGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &action);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceConformActGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceTworatePrateGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceTworatePrateGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &rate);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceTworatePrateGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceSimpleCrateGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceSimpleCrateGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &rate);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceSimpleCrateGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPolicePdataRatePercentGet():\n");
  result = openapiDiffServPolicyAttrStmtPolicePdataRatePercentGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, entryType, &pRate_percent);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPolicePdataRatePercentGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, NULL);
  printf("NULL argument 6:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceTworateCrateGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceTworateCrateGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &pRate);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceTworateCrateGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyAttrStmtPoliceColorExceedIndexGet():\n");
  result = openapiDiffServPolicyAttrStmtPoliceColorExceedIndexGet(NULL, policyIndex, policyInstIndex, policyAttrIndex, &pClassIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyAttrStmtPoliceColorExceedIndexGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Determine next sequential row in the Policy Table.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    prevPolicyIndex       @b{(input)}  Policy index to begin search
* @param    *pPolicyIndex       @b{(output)}  Next sequential policy index value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyGetNext(openapiClientHandle_t *client_handle, uint32_t prevPolicyIndex, uint32_t *pPolicyIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyGetNext(client_handle, prevPolicyIndex, pPolicyIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to determine next sequential row in the Policy Table. (result = %d)\n", result);
  }
  else
  {
    printf("PolicyIndex Get Next is successful next index is %d\n", *pPolicyIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policy name
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    bufd       @b{(output)}  buffer descriptor for policy name

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyNameGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, open_buffdesc *bufd)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyNameGet(client_handle, policyIndex, bufd)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policy name (result = %d)\n", result);
  }
  else
  {
    printf("Policy Name received is successful. Name %s\n", (char *)bufd->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Set the policy name
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    bufd       @b{(input)}  buffer descriptor for policy name

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyNameSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, open_buffdesc *bufd)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyNameSet(client_handle, policyIndex, bufd)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the policy name (result = %d)\n", result);
  }
  else
  {
    printf("Policy Name set is successful. Name %s\n", (char *)bufd->pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Translate a policy name into its associated Policy Table index
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    stringPtr       @b{(input)}  Policy name string pointer
* @param    pPolicyIndex       @b{(output)}  Pointer to policy index value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyNameToIndex(openapiClientHandle_t *client_handle, open_buffdesc *stringPtr, uint32_t *pPolicyIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyNameToIndex(client_handle, stringPtr, pPolicyIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to translate a policy name into its associated Policy Table index (result = %d)\n", result);
  }
  else
  {
    printf("Policy Name to Index is successful name %s Index %d\n", (char *)stringPtr->pstart, *pPolicyIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Check if class definition is suitable for inclusion by specified policy
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    classIndex       @b{(input)}  Class index
* @param    policyIndex       @b{(input)}  Policy index

*
* @returns  none
*
* @end
*********************************************************************/
void diffServClassValidForPolicyCheck(openapiClientHandle_t *client_handle, uint32_t classIndex, uint32_t policyIndex)
{
  open_error_t result;

  if ((result = openapiDiffServClassValidForPolicyCheck(client_handle, classIndex, policyIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if class definition is suitable for inclusion by specified policy (result = %d)\n", result);
  }
  else
  {
    printf("Policy Check is valid for diffserv class\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the current status of the specified Policy Table row
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    pValue       @b{(output)}  Pointer to row status value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyRowStatusGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, OPEN_DIFFSERV_ROW_STATUS_t *pValue)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyRowStatusGet(client_handle, policyIndex, pValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the current status of the specified Policy Table row (result = %d)\n", result);
  }
  else
  {
    printf("Policy Row status value %d\n", *pValue);
  }
  return;
}


/*********************************************************************
* @purpose  Create a new row in the Policy Table for the specified index
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    actiateRow       @b{(input)}  Activate row when ready flag

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyCreate(openapiClientHandle_t *client_handle, uint32_t policyIndex, OPEN_BOOL_t actiateRow)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyCreate(client_handle, policyIndex, actiateRow)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create a new row in the Policy Table for the specified index (result = %d)\n", result);
  }
  else
  {
    printf("Policy Create is successful\n");
  }
  return;
}


/*********************************************************************
* @purpose  Delete a row from the Policy Table for the specified index
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyDelete(openapiClientHandle_t *client_handle, uint32_t policyIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyDelete(client_handle, policyIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete a row from the Policy Table for the specified index (result = %d)\n", result);
  }
  else
  {
    printf("Policy deletion is successful\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the type of policy definition
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    pPolicyType       @b{(output)}  Pointer to policy type value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyTypeGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, OPEN_DIFFSERV_POLICY_TYPE_t *pPolicyType)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyTypeGet(client_handle, policyIndex, pPolicyType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the type of policy definition (result = %d)\n", result);
  }
  else
  {
    printf("Policy type is %d\n", *pPolicyType);
  }
  return;
}


/*********************************************************************
* @purpose  Set the type of policy definition
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyType       @b{(input)}  policy type value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyTypeSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, OPEN_DIFFSERV_POLICY_TYPE_t policyType)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyTypeSet(client_handle, policyIndex, policyType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the type of policy definition (result = %d)\n", result);
  }
  else
  {
    printf("policy type is successfully set to %d\n", policyType);
  }
  return;
}


/*********************************************************************
* @purpose  Obtain the next available index value to be used when creating a new row in the Policy-Class Instance Table
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    pPolicyInstIndex       @b{(output)}  Pointer to instance index value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyInstIndexNext(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t *pPolicyInstIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyInstIndexNext(client_handle, policyIndex, pPolicyInstIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to obtain the next available index value to be used when creating a new row in the Policy-Class Instance Table (result = %d)\n", result);
  }
  else
  {
    printf("Policy Inst index received is %d\n", *pPolicyInstIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Determine next sequential row in the Policy-Class Instance Table
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    prevPolicyIndex       @b{(input)}   Policy index to begin search
* @param    prevPolicyInstIndex       @b{(input)}  Policy instance index to begin search
* @param    pPolicyIndex       @b{(output)}  next sequential policy index value
* @param    pPolicyInstIndex       @b{(output)}  next sequential policy instance index value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyInstGetNext(openapiClientHandle_t *client_handle, uint32_t prevPolicyIndex, uint32_t prevPolicyInstIndex, uint32_t *pPolicyIndex, uint32_t *pPolicyInstIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyInstGetNext(client_handle, prevPolicyIndex, prevPolicyInstIndex, pPolicyIndex, pPolicyInstIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to determine next sequential row in the Policy-Class Instance Table (result = %d)\n", result);
  }
  else
  {
    printf("Next policy Index %d and next policy inst index %d\n", *pPolicyIndex, *pPolicyInstIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Get the class index referenced as the policy class instance
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    pClassIndex       @b{(output)}  class index value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyInstClassIndexGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t *pClassIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyInstClassIndexGet(client_handle, policyIndex, policyInstIndex, pClassIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the class index referenced as the policy class instance (result = %d)\n", result);
  }
  else
  {
    printf("next Inst class Index %d\n", *pClassIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Set the class index referenced as the policy class instance
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    classIndex       @b{(input)}  class index value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyInstClassIndexSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t classIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyInstClassIndexSet(client_handle, policyIndex, policyInstIndex, classIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the class index referenced as the policy class instance (result = %d)\n", result);
  }
  else
  {
    printf("Diff serv policy Inst classIndex set is successful\n");
  }
  return;
}


/*********************************************************************
* @purpose  Find the policy instance index from the policy and class indexes
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    classIndex       @b{(input)}  Class index
* @param    pPolicyInstIndex       @b{(output)}  policy instance index

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyInstIndexFromClass(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t classIndex, uint32_t *pPolicyInstIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyInstIndexFromClass(client_handle, policyIndex, classIndex, pPolicyInstIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to find the policy instance index from the policy and class indexes (result = %d)\n", result);
  }
  else
  {
    printf("policy inst index for policy Index and class index is %d\n", *pPolicyInstIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Create a new row in the Policy-Class Instance Table for the specified index
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    activateRow       @b{(input)}  Activate row when ready flag

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyInstCreate(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, OPEN_BOOL_t activateRow)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyInstCreate(client_handle, policyIndex, policyInstIndex, activateRow)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create a new row in the Policy-Class Instance Table for the specified index (result = %d)\n", result);
  }
  else
  {
    printf("policy Inst Creation is successful\n");
  }
  return;
}


/*********************************************************************
* @purpose  Delete a row from the Policy-Class Instance Table for the specified index
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyInstDelete(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyInstDelete(client_handle, policyIndex, policyInstIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete a row from the Policy-Class Instance Table for the specified index (result = %d)\n", result);
  }
  else
  {
    printf("Policy Instance delete is successful\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the current status of the specified Policy-Class Instance Table row
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    pValue       @b{(output)}  row status value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyInstRowStatusGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, OPEN_DIFFSERV_ROW_STATUS_t *pValue)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyInstRowStatusGet(client_handle, policyIndex, policyInstIndex, pValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the current status of the specified Policy-Class Instance Table row (result = %d)\n", result);
  }
  else
  {
    printf("Policy Inst row status is %d\n", *pValue);
  }
  return;
}


/*********************************************************************
* @purpose  Obtain the next available index value to be used when creating a new row in the Policy Table
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pPolicyIndex       @b{(inout)}  policy index value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyIndexNext(openapiClientHandle_t *client_handle, uint32_t *pPolicyIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyIndexNext(client_handle, pPolicyIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to obtain the next available index value to be used when creating a new row in the Policy Table (result = %d)\n", result);
  }
  else
  {
    printf("Next policy Index is %d\n", *pPolicyIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Get the storage type for the specified Policy Attribute Table row
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    pValue       @b{(output)}  storage type value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStorageTypeGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_STORAGE_TYPE_t *pValue)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStorageTypeGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, pValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the storage type for the specified Policy Attribute Table row (result = %d)\n", result);
  }
  else
  {
    printf("Storage Type received is %d\n", *pValue);
  }
  return;
}


/*********************************************************************
* @purpose  Get the current status of the specified Policy Attribute Table row
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    pValue       @b{(output)}  row status value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrRowStatusGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_ROW_STATUS_t *pValue)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrRowStatusGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, pValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the current status of the specified Policy Attribute Table row (result = %d)\n", result);
  }
  else
  {
    printf("Diff serv poicy Attr Row Status value is %d\n", *pValue);
  }
  return;
}


/*********************************************************************
* @purpose  Obtain the next available index value to be used when creating a new row in the Policy Attribute Table
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    pPolicyAttrIndex       @b{(output)}  policy attribute index

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrIndexNext(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t *pPolicyAttrIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrIndexNext(client_handle, policyIndex, policyInstIndex, pPolicyAttrIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to obtain the next available index value to be used when creating a new row in the Policy Attribute Table (result = %d)\n", result);
  }
  else
  {
    printf("Next attr Index is %d\n", *pPolicyAttrIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Retrieve the maximum policy attribute index value allowed for the Policy Attribute Table
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    pIndexMax       @b{(output)}  index max

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrIndexMaxGet(openapiClientHandle_t *client_handle, uint32_t *pIndexMax)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrIndexMaxGet(client_handle, pIndexMax)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to retrieve the maximum policy attribute index value allowed for the Policy Attribute Table (result = %d)\n", result);
  }
  else
  {
    printf("Diffserv Policy Attr Index max is %d\n", *pIndexMax);
  }
  return;
}


/*********************************************************************
* @purpose  Get the Policy Attribute statement entry type
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    pEntryType       @b{(output)}  entry type value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtEntryTypeGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICY_ATTR_TYPE_t *pEntryType)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtEntryTypeGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, pEntryType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Policy Attribute statement entry type (result = %d)\n", result);
  }
  else
  {
    printf("Diff serv Policy Attr Statement Entry type is %d\n", *pEntryType);
  }
  return;
}


/*********************************************************************
* @purpose  Get the Policy Attribute statement entry associated data
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    entryType       @b{(input)}  Entry type
* @param    data       @b{(output)}  associated data

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtEntryDataGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType, uint32_t *data)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtEntryDataGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, data)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Policy Attribute statement entry associated data (result = %d)\n", result);
  }
  else
  {
    printf("data associated with Statement entry is %d\n", *data);
  }
  return;
}


/*********************************************************************
* @purpose  Create a new row in the Policy Attribute Table for the specified index
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    activateRow       @b{(input)}  Activate row when ready flag

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrCreate(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_BOOL_t activateRow)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrCreate(client_handle, policyIndex, policyInstIndex, policyAttrIndex, activateRow)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create a new row in the Policy Attribute Table for the specified index (result = %d)\n", result);
  }
  else
  {
    printf("Diff serv policy Attr Create is successful\n");
  }
  return;
}


/*********************************************************************
* @purpose  Delete a row from the Policy Attribute Table for the specified index
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrDelete(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrDelete(client_handle, policyIndex, policyInstIndex, policyAttrIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete a row from the Policy Attribute Table for the specified index (result = %d)\n", result);
  }
  else
  {
    printf("Diff serv policy attr delete is successful\n");
  }
  return;
}


/*********************************************************************
* @purpose  Set the Policy Attribute statement entry type and associated data
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    entryType       @b{(input)}  Entry type value
* @param    data       @b{(input)}  value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtEntryTypeDataSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType, uint32_t data)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtEntryTypeDataSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, data)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Policy Attribute statement entry type and associated data (result = %d)\n", result);
  }
  else
  {
    printf("Diffserv Policy Attr Statement data set is successful\n");
  }
  return;
}


/*********************************************************************
* @purpose  Find the Policy Attribute index for the specified attribute
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    inEntry       @b{(input)}  Policy attribute
* @param    pPolicyAttrIndex       @b{(output)}  attribute index

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrIndexFind(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, OPEN_DIFFSERV_POLICY_ATTR_TYPE_t inEntry, uint32_t *pPolicyAttrIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrIndexFind(client_handle, policyIndex, policyInstIndex, inEntry, pPolicyAttrIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to find the Policy Attribute index for the specified attribute (result = %d)\n", result);
  }
  else
  {
    printf("Found policy Attr Index %d for policy Index %d, policy Inst %d, entry Type %d\n", *pPolicyAttrIndex, policyIndex, policyInstIndex, inEntry);
  }
  return;
}


/*********************************************************************
* @purpose  Set the simple policing committed rate attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    entryType       @b{(input)}  Entry type value
* @param    rate_unit       @b{(input)}  Rate unit (in kbpsor percentage)
* @param    rate       @b{(input)}  Rate value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceDataRateSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType, OPEN_RATE_UNIT_t rate_unit, uint32_t rate)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceDataRateSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, rate_unit, rate)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the simple policing committed rate attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("simple policing committed rate attribute statement value set is successful\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing committed rate unit attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    entryType       @b{(input)}  Entry type value
* @param    pRate_unit       @b{(output)}  Value unit

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceDataRateUnitGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType, OPEN_RATE_UNIT_t *pRate_unit)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceDataRateUnitGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, pRate_unit)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing committed rate unit attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("Getting the policing committed rate unit attribute statement value is successful "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d entryType:%d rate_unit:%d.\n",
           policyIndex, policyInstIndex, policyAttrIndex, entryType, *pRate_unit);
  }
  return;
}


/*********************************************************************
* @purpose  Set the policing committed burst size attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    entryType       @b{(input)}  Entry type
* @param    burstType       @b{(input)}  burst type
* @param    burst       @b{(input)}  Burst size value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceBurstSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType, OPEN_BURST_POLICE_TYPE_t burstType, uint32_t burst)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceBurstSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, burstType, burst)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the policing committed burst size attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("Set the policing committed burst size attribute statement value is successful\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing committed burst size attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    entryType       @b{(input)}  entry type
* @param    burstType       @b{(input)}  burst type 
* @param    burst       @b{(output)}  Burst size value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceBurstGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType, OPEN_BURST_POLICE_TYPE_t burstType, uint32_t *burst)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceBurstGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, burstType, burst)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing committed burst size attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing committed burst size attribute statement value get is successful "
           "policyIndex:%d policyInstIndex:%d policyAttrIndex:%d entryType:%d burstType:%d burst:%d.\n",
           policyIndex, policyInstIndex, policyAttrIndex, entryType, burstType, *burst);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing committed rate attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    entryType       @b{(input)}  Entry Type
* @param    pRate_percent       @b{(output)}  rate percentage

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceDataRatePercentGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType, uint32_t *pRate_percent)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceDataRatePercentGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, pRate_percent)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing committed rate attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing committed rate attribute statement value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d entryType:%d] is %d.\n",
           policyIndex, policyInstIndex, policyAttrIndex, entryType, *pRate_percent);
  }
  return;
}


/*********************************************************************
* @purpose  Set the policing exceed value attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    value       @b{(input)}  Police exceed value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceExceedValSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t value)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceExceedValSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the policing exceed value attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("Set of the policing exceed value attribute statement value to %d is successful\n", value);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing exceed value attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    pValue       @b{(output)}  police exceed value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceExceedValGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t *pValue)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceExceedValGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, pValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to et the policing exceed value attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing exceed value attribute statement value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d] is %d\n",
           policyIndex, policyInstIndex, policyAttrIndex, *pValue);
  }
  return;
}


/*********************************************************************
* @purpose  Set the policing exceed action attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    action       @b{(input)}  Police action value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceExceedActSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICE_ACT_t action)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceExceedActSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, action)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the policing exceed action attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("Set the policing exceed action  %d successful\n", action);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing exceed action attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    action       @b{(output)}  Police action value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceExceedActGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICE_ACT_t *action)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceExceedActGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, action)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing exceed action attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("The policing exceed action for [policyIndex:%d policyInstIndex:%d policyAttrIndex:%d] is %d\n",
           policyIndex, policyInstIndex, policyAttrIndex, *action);
  }
  return;
}


/*********************************************************************
* @purpose  Determine if a DiffServ class is valid for use as a policing color aware designation
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    classIndex       @b{(input)}  Class index value to be set

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceColorAwareClassValidate(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t classIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceColorAwareClassValidate(client_handle, policyIndex, policyInstIndex, classIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to determine if a DiffServ class is valid for use as a policing color aware designation (result = %d)\n", result);
  }
  else
  {
    printf("DiffServ class is valid for use as a policing color aware designation for "
           "[policyIndex:%d policyInstIndex:%d classIndex:%d.\n",
            policyIndex, policyInstIndex, classIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing color conform value attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    pVal       @b{(output)}  police color value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceColorConformValGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t *pVal)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceColorConformValGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, pVal)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing color conform value attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing color conform value attribute statement value "
           " for [policyIndex:%d policyInstIndex:%d policyAttrIndex:%d] is %d\n", 
           policyIndex, policyInstIndex, policyAttrIndex, *pVal);
  }
  return;
}


/*********************************************************************
* @purpose  Set the policing color conform attribute class index value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    classIndex       @b{(input)}  Class index value to be set

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceColorConformIndexSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t classIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceColorConformIndexSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, classIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the policing color conform attribute class index value (result = %d)\n", result);
  }
  else
  {
    printf("policing color conform attribute class index value is set to %d\n", classIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing color conform attribute class index value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    pClassIndex       @b{(output)}  class index output value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceColorConformIndexGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t *pClassIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceColorConformIndexGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, pClassIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing color conform attribute class index value (result = %d)\n", result);
  }
  else
  {
    printf("olicing color conform attribute class index value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d] is %d\n",
           policyIndex, policyInstIndex, policyAttrIndex, *pClassIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Set the policing nonconform action attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    action       @b{(input)}  Police action value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceNonconformActSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICE_ACT_t action)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceNonconformActSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, action)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the policing nonconform action attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing nonconform action attribute statement value is set to %d\n", action);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing nonconform action attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    action       @b{(output)}  Police action value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceNonconformActGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICE_ACT_t *action)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceNonconformActGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, action)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing nonconform action attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("Policing nonconform action attribute statement value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d] is %d\n",
           policyIndex, policyInstIndex, policyAttrIndex, *action);
  }
  return;
}


/*********************************************************************
* @purpose  Set the policing nonconform value attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    value       @b{(input)}  Police noncofirm value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceNonconformValSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t value)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceNonconformValSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the policing nonconform value attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing nonconform value attribute statement value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex%d] is set to %d\n",
            policyIndex, policyInstIndex, policyAttrIndex, value);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing nonconform value attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    value       @b{(output)}  Police noncofrm value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceNonconformValGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t *value)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceNonconformValGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing nonconform value attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing nonconform value attribute statement value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d] is %d\n",
           policyIndex, policyInstIndex, policyAttrIndex, *value);
  }
  return;
}


/*********************************************************************
* @purpose  Set the policing conform value attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    value       @b{(output)}  Police conform value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceConformValSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t value)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceConformValSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the policing conform value attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing conform value attribute statement value is set to %d\n", value);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing conform value attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    value       @b{(output)}  Police conform value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceConformValGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t *value)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceConformValGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, value)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing conform value attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing conform value attribute statement value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d] is %d\n",
            policyIndex, policyInstIndex, policyAttrIndex, *value);
  }
  return;
}


/*********************************************************************
* @purpose  Determine next sequential row in the Policy Attribute Table
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    prevPolicyIndex       @b{(input)}  Policy index to begin search
* @param    prevPolicyInstIndex       @b{(input)}  Policy instance index to begin search
* @param    prevPolicyAttrIndex       @b{(input)}  Policy attribute index to begin search
* @param    pPolicyIndex       @b{(output)}  next sequential policy index
* @param    pPolicyInstIndex       @b{(output)}  next sequential policy instance index
* @param    pPolicyAttrIndex       @b{(output)}  next sequential policy attribute index

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrGetNext(openapiClientHandle_t *client_handle, uint32_t prevPolicyIndex, uint32_t prevPolicyInstIndex, uint32_t prevPolicyAttrIndex, uint32_t *pPolicyIndex, uint32_t *pPolicyInstIndex, uint32_t *pPolicyAttrIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrGetNext(client_handle, prevPolicyIndex, prevPolicyInstIndex, prevPolicyAttrIndex, pPolicyIndex, pPolicyInstIndex, pPolicyAttrIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to determine next sequential row in the Policy Attribute Table (result = %d)\n", result);
  }
  else
  {
    printf("next policy Id %d next Policy Inst Index %d next Attr Index %d\n", *pPolicyIndex, *pPolicyInstIndex, *pPolicyAttrIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Set the policing conform action attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    action       @b{(input)}  Police action value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceConformActSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICE_ACT_t action)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceConformActSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, action)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the policing conform action attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing conform action attribute statement value is set to %d\n", action);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing conform action attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    action       @b{(output)}  Police action value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceConformActGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICE_ACT_t *action)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceConformActGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, action)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing conform action attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing conform action attribute statement value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d] is %d\n",
           policyIndex, policyInstIndex, policyAttrIndex, *action);
  }
  return;
}


/*********************************************************************
* @purpose  Set the two-rate policing peak rate attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    rate       @b{(input)}  Rate value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceTworatePrateSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t rate)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceTworatePrateSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, rate)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the two-rate policing peak rate attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("two-rate policing peak rate attribute statement value is set to %d\n", rate);
  }
  return;
}


/*********************************************************************
* @purpose  Set the simple policing committed rate attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    rate       @b{(input)}  Rate value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceSimpleCrateSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t rate)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceSimpleCrateSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, rate)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the simple policing committed rate attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("Simple policing committed rate attribute statement value to %d\n", rate);
  }
  return;
}


/*********************************************************************
* @purpose  Get the two-rate policing peak rate attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    rate       @b{(output)}  Rate value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceTworatePrateGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t *rate)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceTworatePrateGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, rate)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the two-rate policing peak rate attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("two-rate policing peak rate attribute statement value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d] is %d\n",
            policyIndex, policyInstIndex, policyAttrIndex, *rate);
  }
  return;
}


/*********************************************************************
* @purpose  Get the simple policing committed rate attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    rate       @b{(output)}  Rate value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceSimpleCrateGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t *rate)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceSimpleCrateGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, rate)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the simple policing committed rate attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("simple policing committed rate attribute statement value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d] is %d\n",
           policyIndex, policyInstIndex, policyAttrIndex, *rate);
  }
  return;
}


/*********************************************************************
* @purpose  Set the two-rate policing peak rate attribute statement value as per unit
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    rate_unit       @b{(input)}   Rate unit (in kbps or percentage)
* @param    rate       @b{(input)}  Rate value in percentage

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceTworatePdataRateSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_RATE_UNIT_t rate_unit, uint32_t rate)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceTworatePdataRateSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, rate_unit, rate)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the two-rate policing peak rate attribute statement value as per unit (result = %d)\n", result);
  }
  else
  {
    printf("two-rate policing peak rate attribute statement value as per unit %d is %d\n", rate_unit, rate);
  }
  return;
}

#if 0
/*********************************************************************
* @purpose  Get the two-rate policing peak rate attribute statement value as per unit
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    rate_unit       @b{(input)}  Rate unit (in kbps or percentage)
* @param    rate       @b{(output)}  Rate value in percentage

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceTworatePdataRateGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_RATE_UNIT_t rate_unit, uint32_t *rate)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceTworatePdataRateGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, rate_unit, rate)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the two-rate policing peak rate attribute statement value as per unit (result = %d)\n", result);
  }
  else
  {
    printf("two-rate policing peak rate attribute statement value as per unit %d is %d\n", rate_unit, *rate);
  }
  return;
}
#endif

/*********************************************************************
* @purpose  Get the policing peak rate attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    entryType       @b{(input)}  Entry type
* @param    pRate_percent       @b{(output)}  rate percentage

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPolicePdataRatePercentGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType, uint32_t *pRate_percent)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPolicePdataRatePercentGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, pRate_percent)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing peak rate attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("policing peak rate attribute statement value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d entryType:%d] is %d\n", 
           policyIndex, policyInstIndex, policyAttrIndex, entryType, *pRate_percent);
  }
  return;
}


/*********************************************************************
* @purpose  Get the two-rate policing committed rate attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    pRate       @b{(output)}  rate value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceTworateCrateGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t *pRate)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceTworateCrateGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, pRate)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the two-rate policing committed rate attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("two-rate policing committed rate attribute statement value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d] is %d\n",
           policyIndex, policyInstIndex, policyAttrIndex, *pRate);
  }
  return;
}


/*********************************************************************
* @purpose  Set the two-rate policing committed rate attribute statement value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    rate       @b{(input)}  rate value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceTworateCrateSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t rate)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceTworateCrateSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, rate)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the two-rate policing committed rate attribute statement value (result = %d)\n", result);
  }
  else
  {
    printf("two-rate policing committed rate attribute statement value is %d\n", rate);
  }
  return;
}


/*********************************************************************
* @purpose  Set the policing color exceed attribute class index value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    classIndex       @b{(input)}  Class index

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceColorExceedIndexSet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t classIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceColorExceedIndexSet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, classIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the policing color exceed attribute class index value (result = %d)\n", result);
  }
  else
  {
    printf("policing color exceed attribute class index value is set to %d\n", classIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policing color exceed attribute class index value
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex       @b{(input)}  Policy instance index
* @param    policyAttrIndex       @b{(input)}  Policy attribute index
* @param    pClassIndex       @b{(output)}  Class index value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyAttrStmtPoliceColorExceedIndexGet(openapiClientHandle_t *client_handle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t policyAttrIndex, uint32_t *pClassIndex)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyAttrStmtPoliceColorExceedIndexGet(client_handle, policyIndex, policyInstIndex, policyAttrIndex, pClassIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policing color exceed attribute class index value (result = %d)\n", result);
  }
  else
  {
    printf("policing color exceed attribute class index value for "
           "[policyIndex:%d policyInstIndex:%d policyAttrIndex:%d is %d\n",
           policyIndex, policyInstIndex, policyAttrIndex, *pClassIndex);
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for diffServPolicy
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  char policyName[256] = {0};
  uint32_t policyIndex = 0, prevPolicyIndex = 0;
  uint32_t classIndex = 0;
  uint32_t policyInstIndex = 0, prevPolicyInstIndex = 0;
  uint32_t policyAttrIndex = 0, prevPolicyAttrIndex = 0;;
  uint32_t indexMax = 0, data = 0;
  uint32_t rate = 0;
  uint32_t ratePercent = 0;
  uint32_t burst = 0;
  OPEN_DIFFSERV_ROW_STATUS_t rowVal = 0;
  OPEN_BOOL_t activateRow = 0;
  OPEN_DIFFSERV_POLICY_TYPE_t policyType = 0;
  OPEN_DIFFSERV_STORAGE_TYPE_t storageVal = 0;
  OPEN_DIFFSERV_POLICY_ATTR_TYPE_t entryType = 0;
  OPEN_RATE_UNIT_t rateUnit = 0;
  OPEN_BURST_POLICE_TYPE_t burstType = 0xFF;
  OPEN_DIFFSERV_POLICE_ACT_t action = 0;
  open_buffdesc bufd = {.pstart = policyName, .size = sizeof(policyName)};

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("diffServPolicy example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting diffServPolicy API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        prevPolicyIndex = atoi(argv[2]);
        diffServPolicyGetNext(&client_handle, prevPolicyIndex, &policyIndex);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        policyIndex = atoi(argv[2]);
        memset(policyName, 0, sizeof(policyName));
        diffServPolicyNameGet(&client_handle, policyIndex, &bufd);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 4)
      {
        policyIndex = atoi(argv[2]);
        memset(policyName, 0, sizeof(policyName));
        strncpy(policyName, argv[3], sizeof(policyName)-1);
        bufd.size = strlen(policyName);
        diffServPolicyNameSet(&client_handle, policyIndex, &bufd);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        memset(policyName, 0, sizeof(policyName));
        strncpy(policyName, argv[2], sizeof(policyName)-1);
        bufd.size = strlen(policyName);
        policyIndex = 0;
        diffServPolicyNameToIndex(&client_handle, &bufd, &policyIndex);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 4)
      {
        classIndex = atoi(argv[2]);
        policyIndex = atoi(argv[3]);
        diffServClassValidForPolicyCheck(&client_handle, classIndex, policyIndex);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        policyIndex = atoi(argv[2]);
        diffServPolicyRowStatusGet(&client_handle, policyIndex, &rowVal);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        policyIndex = atoi(argv[2]);
        activateRow = 1;
        diffServPolicyCreate(&client_handle, policyIndex, activateRow);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
        policyIndex = atoi(argv[2]);
        diffServPolicyDelete(&client_handle, policyIndex);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        policyIndex = atoi(argv[2]);
        diffServPolicyTypeGet(&client_handle, policyIndex, &policyType);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 4)
      {
        policyIndex = atoi(argv[2]);
        policyType = atoi(argv[3]);
        diffServPolicyTypeSet(&client_handle, policyIndex, policyType);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 3)
      {
        policyIndex = atoi(argv[2]);
        diffServPolicyInstIndexNext(&client_handle, policyIndex, &policyInstIndex);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 4)
      {
        /*
        diffServClassL3ProtoGet(&client_handle, classIndex, &pClassL3Proto);*/
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 4)
      {
        prevPolicyIndex = atoi(argv[2]);
        prevPolicyInstIndex = atoi(argv[3]);
        diffServPolicyInstGetNext(&client_handle, prevPolicyIndex, prevPolicyInstIndex, &policyIndex, &policyInstIndex);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        diffServPolicyInstClassIndexGet(&client_handle, policyIndex, policyInstIndex, &classIndex);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        classIndex = atoi(argv[4]);
        diffServPolicyInstClassIndexSet(&client_handle, policyIndex, policyInstIndex, classIndex);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 4)
      {
        policyIndex = atoi(argv[2]);
        classIndex = atoi(argv[3]);
        diffServPolicyInstIndexFromClass(&client_handle, policyIndex, classIndex, &policyInstIndex);
        show_help = 0;
      }
      break;
    case 17:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        activateRow = atoi(argv[4]);
        diffServPolicyInstCreate(&client_handle, policyIndex, policyInstIndex, activateRow);
        show_help = 0;
      }
      break;
    case 18:
      if (argc == 4)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        diffServPolicyInstDelete(&client_handle, policyIndex, policyInstIndex);
        show_help = 0;
      }
      break;
    case 19:
      if (argc == 4)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        diffServPolicyInstRowStatusGet(&client_handle, policyIndex, policyInstIndex, &rowVal);
        show_help = 0;
      }
      break;
    case 20:
      if (argc == 3)
      {
        policyIndex = atoi(argv[2]);
        diffServPolicyIndexNext(&client_handle, &policyIndex);
        show_help = 0;
      }
      break;
    case 21:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStorageTypeGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &storageVal);
        show_help = 0;
      }
      break;
    case 22:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrRowStatusGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &rowVal);
        show_help = 0;
      }
      break;
    case 23:
      if (argc == 4)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        diffServPolicyAttrIndexNext(&client_handle, policyIndex, policyInstIndex, &policyAttrIndex);
        show_help = 0;
      }
      break;
    case 24:
      if (argc == 2)
      {
        diffServPolicyAttrIndexMaxGet(&client_handle, &indexMax);
        show_help = 0;
      }
      break;
    case 25:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtEntryTypeGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &entryType);
        show_help = 0;
      }
      break;
    case 26:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        entryType = atoi(argv[5]);
        diffServPolicyAttrStmtEntryDataGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, &data);
        show_help = 0;
      }
      break;
    case 27:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        activateRow = atoi(argv[5]);
        diffServPolicyAttrCreate(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, activateRow);
        show_help = 0;
      }
      break;
    case 28:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrDelete(&client_handle, policyIndex, policyInstIndex, policyAttrIndex);
        show_help = 0;
      }
      break;
    case 29:
      if (argc == 7)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        entryType = atoi(argv[5]);
        data = atoi(argv[6]);
        diffServPolicyAttrStmtEntryTypeDataSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, data);
        show_help = 0;
      }
      break;
    case 30:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        entryType = atoi(argv[4]);
        diffServPolicyAttrIndexFind(&client_handle, policyIndex, policyInstIndex, entryType, &policyAttrIndex);
        show_help = 0;
      }
      break;
    case 31:
      if (argc == 8)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        entryType = atoi(argv[5]);
        rateUnit = atoi(argv[6]);
        rate = atoi(argv[7]);
        diffServPolicyAttrStmtPoliceDataRateSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, rateUnit, rate);
        show_help = 0;
      }
      break;
    case 32:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        entryType = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceDataRateUnitGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, &rateUnit);
        show_help = 0;
      }
      break;
    case 33:
      if (argc == 8)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        entryType = atoi(argv[5]);
        burstType = atoi(argv[6]);
        burst = atoi(argv[7]);
        diffServPolicyAttrStmtPoliceBurstSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, burstType, burst);
        show_help = 0;
      }
      break;
    case 34:
      if (argc == 7)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        entryType = atoi(argv[5]);
        burstType = atoi(argv[6]);
        diffServPolicyAttrStmtPoliceBurstGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, burstType, &burst);
        show_help = 0;
      }
      break;
    case 35:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        entryType = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceDataRatePercentGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, &ratePercent);
        show_help = 0;
      }
      break;
    case 36:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        data = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceExceedValSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, data);
        show_help = 0;
      }
      break;
    case 37:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceExceedValGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &data);
        show_help = 0;
      }
      break;
    case 38:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        action = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceExceedActSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, action);
        show_help = 0;
      }
      break;
    case 39:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceExceedActGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &action);
        show_help = 0;
      }
      break;
    case 40:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        classIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceColorAwareClassValidate(&client_handle, policyIndex, policyInstIndex, classIndex);
        show_help = 0;
      }
      break;
    case 41:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceColorConformValGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &data);
        show_help = 0;
      }
      break;
    case 42:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        classIndex = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceColorConformIndexSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, classIndex);
        show_help = 0;
      }
      break;
    case 43:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceColorConformIndexGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &classIndex);
        show_help = 0;
      }
      break;
    case 44:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        action = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceNonconformActSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, action);
        show_help = 0;
      }
      break;
    case 45:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceNonconformActGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &action);
        show_help = 0;
      }
      break;
    case 46:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        data = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceNonconformValSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, data);
        show_help = 0;
      }
      break;
    case 47:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceNonconformValGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &data);
        show_help = 0;
      }
      break;
    case 48:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        data = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceConformValSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, data);
        show_help = 0;
      }
      break;
    case 49:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceConformValGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &data);
        show_help = 0;
      }
      break;
    case 50:
      if (argc == 5)
      {
        prevPolicyIndex = atoi(argv[2]);
        prevPolicyInstIndex = atoi(argv[3]);
        prevPolicyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrGetNext(&client_handle, prevPolicyIndex, prevPolicyInstIndex, prevPolicyAttrIndex, &policyIndex, &policyInstIndex, &policyAttrIndex);
        show_help = 0;
      }
      break;
    case 51:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        action = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceConformActSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, action);
        show_help = 0;
      }
      break;
    case 52:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceConformActGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &action);
        show_help = 0;
      }
      break;
    case 53:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        rate = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceTworatePrateSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, rate);
        show_help = 0;
      }
      break;
    case 54:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        rate = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceSimpleCrateSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, rate);
        show_help = 0;
      }
      break;
    case 55:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceTworatePrateGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &rate);
        show_help = 0;
      }
      break;
    case 56:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceSimpleCrateGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &rate);
        show_help = 0;
      }
      break;
    case 57:
      if (argc == 7)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        rateUnit = atoi(argv[5]);
        rate = atoi(argv[6]);
        diffServPolicyAttrStmtPoliceTworatePdataRateSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, rateUnit, rate);
        show_help = 0;
      }
      break;
    case 58:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        rateUnit = atoi(argv[5]);
        /* diffServPolicyAttrStmtPoliceTworatePdataRateGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, rateUnit, &rate); */
        show_help = 0;
      }
      break;
    case 59:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        entryType = atoi(argv[5]);
        diffServPolicyAttrStmtPolicePdataRatePercentGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, entryType, &ratePercent);
        show_help = 0;
      }
      break;
    case 60:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceTworateCrateGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &rate);
        show_help = 0;
      }
      break;
    case 61:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        rate = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceTworateCrateSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, rate);
        show_help = 0;
      }
      break;
    case 62:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        classIndex = atoi(argv[5]);
        diffServPolicyAttrStmtPoliceColorExceedIndexSet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, classIndex);
        show_help = 0;
      }
      break;
    case 63:
      if (argc == 6)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        policyAttrIndex = atoi(argv[4]);
        diffServPolicyAttrStmtPoliceColorExceedIndexGet(&client_handle, policyIndex, policyInstIndex, policyAttrIndex, &classIndex);
        show_help = 0;
      }
      break;
    case 64:
      if (argc == 2)
      {
        runSanity(&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping diffServPolicy API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}


