/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  diffserv_serv_ex.c
*
* @purpose   OpEN diffServ_serv example.
*
* @component OpEN
*
* @create    05/23/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_diffserv_service.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the diffServ_serv Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Create a new row in the Service Table for the specified interface and direction: %s 1 <intIfNum><ifDir><activateRow>\n", name);
  printf("Test 2: Delete a row from the Service Table for the specified interface and direction: %s 2 <intIfNum><ifDir>\n", name);
  printf("Test 3: Gets a row from the Service Table for the specified interface and direction: %s 3 <intIfNum><ifDir>\n", name);
  printf("Test 4: Determine next sequential row in the Service Table: %s 4 <intIfNum><ifDir>\n", name);
  printf("Test 5: Set the policy index attached to the specified interface and direction: %s 5 <intIfNum><ifDir><policyIndex>\n", name);
  printf("Test 6: Get the policy index attached to the specified interface and direction: %s 6 <intIfNum><ifDir>\n", name);
  printf("Test 7: Get the service interface operational status for the specified Service Table row: %s 7 <intIfNum><ifDir>\n", name);
  printf("Test 8: Check if the specified internal interface number is in use by the ACL component: %s 8 <intIfNum><ifDir>\n", name);
  printf("Test 9: Set the given status for the specified Service Table row: %s 9 <intIfNum><ifDir><status>\n", name);
  printf("Test 10: Get the current status for the specified Service Table row: %s 10 <intIfNum><ifDir>\n", name);
  printf("Test 11: Verifies the current status for the specified Service Table row: %s 11 <intIfNum><ifDir><status>\n", name);
  printf("Test 12: Get the inbound offered packets count for the specified policy class instance: %s 12 <policyIndex><policyInstIndex><intIfNum>\n", name);
  printf("Test 13: Get the outbound offered packets count for the specified policy class instance: %s 13 <policyIndex><policyInstIndex><intIfNum>\n", name);
  printf("Test 14: Get the inbound discarded packet count for the specified policy class instance: %s 14 <policyIndex><policyInstIndex><intIfNum>\n", name);
  printf("Test 15: Get the outbound discarded packet count for the specified policy class instance: %s 15 <policyIndex><policyInstIndex><intIfNum>\n", name);
  printf("Test 16: Run API sanity checks: %s 16 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    client_handle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *client_handle)
{
  open_error_t result;
  uint32_t policyIndex=1;
  OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir=1, nxtIfDir;
  uint32_t policyInstIndex=1;
  OPEN_MIB_DIFFSERV_IF_OPER_STATUS_t ifOperStatus;
  uint32_t intIfNum=1;
  uint32_t nxtIntIfNum;
  uint64_t packtCount;
  OPEN_BOOL_t isAclIntfInUse;
  OPEN_MIB_DIFFSERV_ROW_STATUS_t activateRow; 
  printf("Testing diffServ_serv OpEN APIs sanity:\n");
  printf("Testing openapiDiffServServiceGetNext():\n");
  result = openapiDiffServServiceGetNext(NULL, intIfNum, ifDir, &nxtIntIfNum, &nxtIfDir);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServServiceGetNext(client_handle, intIfNum, ifDir, NULL, &nxtIfDir);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiDiffServServiceGetNext(client_handle, intIfNum, ifDir, &nxtIntIfNum, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServServicePolicyIndexGet():\n");
  result = openapiDiffServServicePolicyIndexGet(NULL, intIfNum, ifDir, &policyIndex);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServServicePolicyIndexGet(client_handle, intIfNum, ifDir, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiDiffServServiceIfOperStatusGet():\n");
  result = openapiDiffServServiceIfOperStatusGet(NULL, intIfNum, ifDir, &ifOperStatus);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServServiceIfOperStatusGet(client_handle, intIfNum, ifDir, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiDiffServIsAclIntfInUse():\n");
  result = openapiDiffServIsAclIntfInUse(NULL, intIfNum, ifDir, &isAclIntfInUse);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServIsAclIntfInUse(client_handle, intIfNum, ifDir, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiDiffServServiceRowStatusGet():\n");
  result = openapiDiffServServiceRowStatusGet(NULL, intIfNum, ifDir, &activateRow);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServServiceRowStatusGet(client_handle, intIfNum, ifDir, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyPerfInOfferedPacketsGet():\n");
  result = openapiDiffServPolicyPerfInOfferedPacketsGet(NULL, policyIndex, policyInstIndex, intIfNum, &packtCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyPerfInOfferedPacketsGet(client_handle, policyIndex, policyInstIndex, intIfNum, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyPerfOutOfferedPacketsGet():\n");
  result = openapiDiffServPolicyPerfOutOfferedPacketsGet(NULL, policyIndex, policyInstIndex, intIfNum, &packtCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyPerfOutOfferedPacketsGet(client_handle, policyIndex, policyInstIndex, intIfNum, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyPerfInDiscardedPacketsGet():\n");
  result = openapiDiffServPolicyPerfInDiscardedPacketsGet(NULL, policyIndex, policyInstIndex, intIfNum, &packtCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyPerfInDiscardedPacketsGet(client_handle, policyIndex, policyInstIndex, intIfNum, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  printf("Testing openapiDiffServPolicyPerfOutDiscardedPacketsGet():\n");
  result = openapiDiffServPolicyPerfOutDiscardedPacketsGet(NULL, policyIndex, policyInstIndex, intIfNum, &packtCount);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiDiffServPolicyPerfOutDiscardedPacketsGet(client_handle, policyIndex, policyInstIndex, intIfNum, NULL);
  printf("NULL argument 5:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Create a new row in the Service Table for the specified interface and direction.
*
* @param    pClientHandle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    ifDir          @b{(input)}  Interface direction
* @param    activateRow    @b{(input)}  Activate row when value is TRUE

*
* @returns  none
*
* @end
*********************************************************************/
void diffServServiceCreate(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir, OPEN_BOOL_t activateRow)
{
  open_error_t result;

  if ((result = openapiDiffServServiceCreate(pClientHandle, intIfNum, ifDir, activateRow)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to create a new row in the Service Table for the specified interface and direction. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully created a new row in service table for interface %d of direction %d and row activation status %d. \n",
           intIfNum, ifDir, activateRow);
  }
  return;
}


/*********************************************************************
* @purpose  Delete a row from the Service Table for the specified interface and direction.
*
* @param    pClientHandle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    ifDir          @b{(input)}  Interface direction

*
* @returns  none
*
* @end
*********************************************************************/
void diffServServiceDelete(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir)
{
  open_error_t result;

  if ((result = openapiDiffServServiceDelete(pClientHandle, intIfNum, ifDir)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete a row from the Service Table for the specified interface and direction. (result = %d)\n", result);
  }
  else
  {
    printf("Deleted row in service table for interface %d of direction %d. \n",
           intIfNum, ifDir);
  }
  return;
}


/*********************************************************************
* @purpose  Fetch a row from the Service Table for the specified interface and direction.
*
* @param    pClientHandle     @b{(input)}  client handle from registration API
* @param    intIfNum          @b{(input)}  Internal interface number
* @param    ifDir             @b{(input)}  Interface direction

*
* @returns  none
*
* @end
*********************************************************************/
void diffServServiceGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir)
{
  open_error_t result;

  if ((result = openapiDiffServServiceGet(pClientHandle, intIfNum, ifDir)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to fetch row from the Service Table for the specified interface and direction. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully fetch row in service table for interface %d of direction %d. \n",
           intIfNum, ifDir);
  }
  return;
}


/*********************************************************************
* @purpose  Determine next sequential row in the Service Table.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b{(input)}  Internal interface number
* @param    ifDir               @b{(input)}  Interface direction
* @param    nxtIntIfNum         @b{(output)}  Next sequential internal interface number
* @param    nxtIfDir            @b{(output)}  Next sequential interface direction

*
* @returns  none
*
* @end
*********************************************************************/
void diffServServiceGetNext(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir, uint32_t *nxtIntIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t *nxtIfDir)
{
  open_error_t result;

  if ((result = openapiDiffServServiceGetNext(pClientHandle, intIfNum, ifDir, nxtIntIfNum, nxtIfDir)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to determine next sequential row in the Service Table. (result = %d)\n", result);
  }
  else
  {
    printf("Next sequential row in the Service Table for given[interface %d and direction %d] is"
            "interface %d and direction %d. \n", intIfNum, ifDir, *nxtIntIfNum, *nxtIfDir);
  }
  return;
}


/*********************************************************************
* @purpose  Set the policy index attached to the specified interface and direction.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b{(input)}  Internal interface number
* @param    ifDir               @b{(input)}  Interface direction
* @param    policyIndex         @b{(input)}  Policy index

*
* @returns  none
*
* @end
*********************************************************************/
void diffServServicePolicyIndexSet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir, uint32_t policyIndex)
{
  open_error_t result;

  if ((result = openapiDiffServServicePolicyIndexSet(pClientHandle, intIfNum, ifDir, policyIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the policy index attached to the specified interface and direction. (result = %d)\n", result);
  }
  else
  {
    printf("Successfuly set policy index %d attached to the interface %d and direction %d.\n",
            policyIndex, intIfNum, ifDir);
  }
  return;
}


/*********************************************************************
* @purpose  Get the policy index attached to the specified interface and direction.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b{(input)}  Internal interface number
* @param    ifDir               @b{(input)}  Interface direction
* @param    pPolicyIndex        @b{(output)}  Policy index

*
* @returns  none
*
* @end
*********************************************************************/
void diffServServicePolicyIndexGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir, uint32_t *pPolicyIndex)
{
  open_error_t result;

  if ((result = openapiDiffServServicePolicyIndexGet(pClientHandle, intIfNum, ifDir, pPolicyIndex)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the policy index attached to the specified interface and direction. (result = %d)\n", result);
  }
  else
  {
    printf("Fetched policy index information for interface %d and direction %d is %d.\n",
            intIfNum, ifDir, *pPolicyIndex);
  }
  return;
}


/*********************************************************************
* @purpose  Get the service interface operational status for the specified Service Table row.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b{(input)}  Internal interface number
* @param    ifDir               @b{(input)}  Interface direction
* @param    pIfOperStatus       @b{(output)}  Interface operational status

*
* @returns  none
*
* @end
*********************************************************************/
void diffServServiceIfOperStatusGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir, OPEN_MIB_DIFFSERV_IF_OPER_STATUS_t *pIfOperStatus)
{
  open_error_t result;

  if ((result = openapiDiffServServiceIfOperStatusGet(pClientHandle, intIfNum, ifDir, pIfOperStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the service interface operational status for the specified Service Table row. (result = %d)\n", result);
  }
  else
  {
    printf("Operational status of service table row for interface %d direction %d is %d.\n",
           intIfNum, ifDir, *pIfOperStatus);
  }
  return;
}


/*********************************************************************
* @purpose  Check if the specified internal interface number is in use by the ACL component.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b{(input)}  Internal interface number
* @param    ifDir               @b{(input)}  Interface direction
* @param    pIsAclIntfInUse     @b{(output)}  status of interface number in use by the ACL component

*
* @returns  none
*
* @end
*********************************************************************/
void diffServIsAclIntfInUse(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir, OPEN_BOOL_t *pIsAclIntfInUse)
{
  open_error_t result;

  if ((result = openapiDiffServIsAclIntfInUse(pClientHandle, intIfNum, ifDir, pIsAclIntfInUse)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if the specified internal interface number is in use by the ACL component. (result = %d)\n", result);
  }
  else
  {
    printf("Interface %d of direction %d is %s by the ACL component.\n", intIfNum, ifDir,
           (OPEN_TRUE == *pIsAclIntfInUse) ? "in use" : "unused");
  }
  return;
}


/*********************************************************************
* @purpose  Set the given status for the specified Service Table row.
*
* @param    pClientHandle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    ifDir          @b{(input)}  Interface direction
* @param    status         @b{(input)}  Row status value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServServiceRowStatusSet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir, OPEN_MIB_DIFFSERV_ROW_STATUS_t status)
{
  open_error_t result;

  if ((result = openapiDiffServServiceRowStatusSet(pClientHandle, intIfNum, ifDir, status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the given status for the specified Service Table row. (result = %d)\n", result);
  }
  else
  {
    printf("Row status is successfully set to %d for interface %d and direction %d.\n", status, intIfNum, ifDir);
  }
  return;
}


/*********************************************************************
* @purpose  Get the current status for the specified Service Table row.
*
* @param    pClientHandle  @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Internal interface number
* @param    ifDir          @b{(input)}  Interface direction
* @param    pStatus        @b{(output)}  Row status value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServServiceRowStatusGet(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir, OPEN_MIB_DIFFSERV_ROW_STATUS_t *pStatus)
{
  open_error_t result;

  if ((result = openapiDiffServServiceRowStatusGet(pClientHandle, intIfNum, ifDir, pStatus)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the current status for the specified Service Table row. (result = %d)\n", result);
  }
  else
  {
    printf("Status for the specified Service Table row with interface %d and direction %d is %d.\n",
           intIfNum, ifDir, *pStatus);
  }
  return;
}


/*********************************************************************
* @purpose  Verifies the current status for the specified Service Table row.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    intIfNum            @b{(input)}  Internal interface number
* @param    ifDir               @b{(input)}  Interface direction
* @param    status              @b{(input)}  Row status value

*
* @returns  none
*
* @end
*********************************************************************/
void diffServServiceRowStatusVerify(openapiClientHandle_t *pClientHandle, uint32_t intIfNum, OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir, OPEN_MIB_DIFFSERV_ROW_STATUS_t status)
{
  open_error_t result;

  if ((result = openapiDiffServServiceRowStatusVerify(pClientHandle, intIfNum, ifDir, status)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to verifies the current status for the specified Service Table row. (result = %d)\n", result);
  }
  else
  {
    printf("Successfully verified status for the specified Service Table row with interface %d , direction %d and row status  %d.\n",
           intIfNum, ifDir, status);
  }
  return;
}


/*********************************************************************
* @purpose  Get the inbound offered packets count for the specified policy class instance.
*
* @param    pClientHandle       @b{(input)}  client handle from registration API
* @param    policyIndex         @b{(input)}  Policy index
* @param    policyInstIndex     @b{(input)}  Policy instance index
* @param    intIfNum            @b{(input)}  Internal interface number
* @param    pPacktCount         @b{(output)}  Packet count

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyPerfInOfferedPacketsGet(openapiClientHandle_t *pClientHandle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t intIfNum, uint64_t *pPacktCount)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyPerfInOfferedPacketsGet(pClientHandle, policyIndex, policyInstIndex, intIfNum, pPacktCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the inbound offered packets count for the specified policy class instance. (result = %d)\n", result);
  }
  else
  {
    printf("Inbound offered packet count for policy class instance(policy index:%d policy instance index:%d) on interface %d is %lu. \n",
            policyIndex, policyInstIndex, intIfNum, *pPacktCount);
  }
  return;
}


/*********************************************************************
* @purpose  Get the outbound offered packets count for the specified policy class instance.
*
* @param    pClientHandle     @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex   @b{(input)}  Policy instance index
* @param    intIfNum          @b{(input)}  Internal interface number
* @param    pPacktCount       @b{(output)}  Packet count

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyPerfOutOfferedPacketsGet(openapiClientHandle_t *pClientHandle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t intIfNum, uint64_t *pPacktCount)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyPerfOutOfferedPacketsGet(pClientHandle, policyIndex, policyInstIndex, intIfNum, pPacktCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the outbound offered packets count for the specified policy class instance. (result = %d)\n", result);
  }
  else
  {
    printf("Outbound offered packet count for policy class instance(policy index:%d policy instance index:%d) on interface %d is %lu. \n",
            policyIndex, policyInstIndex, intIfNum, *pPacktCount);
  }
  return;
}


/*********************************************************************
* @purpose  Get the inbound discarded packet count for the specified policy class instance.
*
* @param    pClientHandle     @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex   @b{(input)}  Policy instance index
* @param    intIfNum          @b{(input)}  Internal interface number
* @param    pPacktCount       @b{(output)}  Packet count

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyPerfInDiscardedPacketsGet(openapiClientHandle_t *pClientHandle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t intIfNum, uint64_t *pPacktCount)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyPerfInDiscardedPacketsGet(pClientHandle, policyIndex, policyInstIndex, intIfNum, pPacktCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the inbound discarded packet count for the specified policy class instance. (result = %d)\n", result);
  }
  else
  {
    printf("Inbound discarded packet count for policy class instance(policy index:%d policy instance index:%d) on interface %d is %lu. \n",
            policyIndex, policyInstIndex, intIfNum, *pPacktCount);
  }
  return;
}


/*********************************************************************
* @purpose  Get the outbound discarded packet count for the specified policy class instance.
*
* @param    pClientHandle     @b{(input)}  client handle from registration API
* @param    policyIndex       @b{(input)}  Policy index
* @param    policyInstIndex   @b{(input)}  Policy instance index
* @param    intIfNum          @b{(input)}  Internal interface number
* @param    pPacktCount       @b{(output)}  Packet count

*
* @returns  none
*
* @end
*********************************************************************/
void diffServPolicyPerfOutDiscardedPacketsGet(openapiClientHandle_t *pClientHandle, uint32_t policyIndex, uint32_t policyInstIndex, uint32_t intIfNum, uint64_t *pPacktCount)
{
  open_error_t result;

  if ((result = openapiDiffServPolicyPerfOutDiscardedPacketsGet(pClientHandle, policyIndex, policyInstIndex, intIfNum, pPacktCount)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the outbound discarded packet count for the specified policy class instance. (result = %d)\n", result);
  }
  else
  {
    printf("Outbound discarded packet count for policy class instance(policy index:%d policy instance index:%d) on interface %d is %lu. \n",
            policyIndex, policyInstIndex, intIfNum, *pPacktCount);
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for diffServ_serv
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  uint32_t intIfNum, nxtIntIfNum;
  OPEN_MIB_DIFFSERV_IF_DIRECTION_t ifDir, nxtIfDir;
  OPEN_BOOL_t activateRow;
  uint32_t policyIndex;
  OPEN_MIB_DIFFSERV_IF_OPER_STATUS_t ifOperStatus;
  OPEN_BOOL_t isAclIntfInUse;
  OPEN_MIB_DIFFSERV_ROW_STATUS_t status;
  uint32_t policyInstIndex;
  uint64_t packtCount;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("diffServ_serv example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting diffServ_serv API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 5)
      {
        intIfNum = atoi(argv[2]);
        ifDir = atoi(argv[3]);
        activateRow = atoi(argv[4]);
        diffServServiceCreate(&client_handle, intIfNum, ifDir, activateRow);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        ifDir = atoi(argv[3]);
        diffServServiceDelete(&client_handle, intIfNum, ifDir);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        ifDir = atoi(argv[3]);
        diffServServiceGet(&client_handle, intIfNum, ifDir);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        ifDir = atoi(argv[3]);
        diffServServiceGetNext(&client_handle, intIfNum, ifDir, &nxtIntIfNum, &nxtIfDir);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 5)
      {
        intIfNum = atoi(argv[2]);
        ifDir = atoi(argv[3]);
        policyIndex = atoi(argv[4]);
        diffServServicePolicyIndexSet(&client_handle, intIfNum, ifDir, policyIndex);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        ifDir = atoi(argv[3]);
        diffServServicePolicyIndexGet(&client_handle, intIfNum, ifDir, &policyIndex);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        ifDir = atoi(argv[3]);
        diffServServiceIfOperStatusGet(&client_handle, intIfNum, ifDir, &ifOperStatus);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        ifDir = atoi(argv[3]);
        diffServIsAclIntfInUse(&client_handle, intIfNum, ifDir, &isAclIntfInUse);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 5)
      {
        intIfNum = atoi(argv[2]);
        ifDir = atoi(argv[3]);
        status = atoi(argv[4]);
        diffServServiceRowStatusSet(&client_handle, intIfNum, ifDir, status);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 4)
      {
        intIfNum = atoi(argv[2]);
        ifDir = atoi(argv[3]);
        diffServServiceRowStatusGet(&client_handle, intIfNum, ifDir, &status);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 5)
      {
        intIfNum = atoi(argv[2]);
        ifDir = atoi(argv[3]);
        status = atoi(argv[4]);
        diffServServiceRowStatusVerify(&client_handle, intIfNum, ifDir, status);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        intIfNum = atoi(argv[4]);
        diffServPolicyPerfInOfferedPacketsGet(&client_handle, policyIndex, policyInstIndex, intIfNum, &packtCount);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        intIfNum = atoi(argv[4]);
        diffServPolicyPerfOutOfferedPacketsGet(&client_handle, policyIndex, policyInstIndex, intIfNum, &packtCount);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        intIfNum = atoi(argv[4]);
        diffServPolicyPerfInDiscardedPacketsGet(&client_handle, policyIndex, policyInstIndex, intIfNum, &packtCount);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 5)
      {
        policyIndex = atoi(argv[2]);
        policyInstIndex = atoi(argv[3]);
        intIfNum = atoi(argv[4]);
        diffServPolicyPerfOutDiscardedPacketsGet(&client_handle, policyIndex, policyInstIndex, intIfNum, &packtCount);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 2)
      {
        runSanity(&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping diffServ_serv API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}

