#!/mnt/fastpath/usr/bin/python

"""suppfile_example.py: OpEN API support file generation example"""

import OpEN_py as OpEN
from OpENUtil import *
import sys

example_name = "suppfile_example"

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#


#
# Python 2.6.6
#

open_ = OpENUtil()

#
# Define a reasonable number of lines to display by default
# (a typical support file can contain upwards of 20K lines!)
#
display_default = 100

class SuppFileExample:
  """Support File OpEN API Example."""

  def __init__(self, client):
    self.m_client = client
    self.support_filename = None

  def support_file_create(self):
    """Execute 'show tech-support file' command internally to generate file."""

    # Get the maximum length of a file name string
    max_filename_len_p = OpEN.new_uint32_tp()
    OpEN.openapiSystemFileNameMaxLengthGet(self.m_client, max_filename_len_p)
    max_filename_len = OpEN.uint32_tp_value(max_filename_len_p)
    OpEN.delete_uint32_tp(max_filename_len_p)

    # Allocate a buffer and descriptor to receive output filename
    # (add extra byte for end-of-string termination character)
    bufsize = max_filename_len + 1
    try:
      outfile_buf = open_.getStringBuffer(bufsize)
    except OpENBufferSizeError:
      print("support_file_create: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("support_file_create: getStringBuffer raised TypeError")
      return
    outfile_desc = OpEN.open_buffdesc()
    outfile_desc.pstart = outfile_buf
    outfile_desc.size = bufsize

    filename = None
    ret = OpEN.openapiSupportFileGenerate(self.m_client, outfile_desc)
    if (ret == OpEN.OPEN_E_NONE) and (len(outfile_buf.cast()) > 0):
      filename = outfile_buf.cast()
      print("Tech support file created successfully.\n")
    else:
      if ret == OpEN.OPEN_E_UNAVAIL:
        print("Feature not supported on this platform.\n")
      elif ret == OpEN.OPEN_E_PARAM:
        print("Error: Invalid parameter specified.\n")
      else:
        print("Error: Failed to create tech support file.\n")

    self.support_filename = filename
    return filename

  def support_file_display(self, maxout=display_default):
    """"Display file contents line-by-line. Optional maxout arg controls output:
          0  : displays entire file
          >0 : displays specified number of file lines
    """
    if self.support_filename is not None:
      filename = self.support_filename
      if maxout >= 0:
        for line in open(filename):
          # Do not add any extra newlines
          print(line, end=" ")
          if maxout > 1:
            maxout -= 1
          if maxout == 1:
            break

def main(argv):
  """Demonstrate OpEN usage for generating Tech Support file."""

  # display_max parameter is optional
  numargs = len(argv)
  if (numargs > 1) or ((numargs == 1) and (argv[0] == '?')):
    print("%s.py [display_max]\n" % (example_name,))
    print("  display_max: 0=all, >0=number of lines (default %d)\n" % (display_default,))
    sys.exit()

  try:
    display_max = int(argv[0])
    if display_max < 0:
      raise ValueError()
  except ValueError:
    print("The specified input parameter value must be 0 or greater.\n")
    sys.exit()
  except:
    # Display default number of lines if optional parm not specified
    display_max = display_default

  ret = open_.connect(example_name)
  if ret == OpEN.OPEN_E_NONE:
    print("")
    open_.getNetworkOSVersion()
    open_.getAPIVersion()
    client = open_.get_client()
    xmp = SuppFileExample(client)
    print("Creating Tech Support file (this may take a while)...")
    xmp.support_file_create()
    if display_max >= 0:
      if display_max == 0:
        lines_msg = "all lines"
      else:
        lines_msg = "first line" if display_max == 1 else "first %d lines" % (display_max,)
      print("Displaying %s of file %s:\n" % (lines_msg, xmp.support_filename))
    xmp.support_file_display(display_max)
    open_.terminate()
  else :
    print("Unable to connect.\n")

if __name__ == '__main__':
  main(sys.argv[1:])

