#!/mnt/fastpath/usr/bin/python

"""snmp_example.py: OpEN API  Simple Network Management Protocol (SNMP) configuration example"""

import OpEN_py as OpEN
from OpENUtil import *
import socket
import struct
import sys
import getopt
from sys import version_info

open_ = OpENUtil()

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License")
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

#
# Python 2.6.6
#

# Define some constants for comparison convenience
SET                      = 'set'
MAP                      = 'map'

# Define SNMP users
SNMP_USER_LIST = [
  ("admin1", "network-admin1", OpEN.OPEN_USM_USER_AUTH_PROTOCOL_NONE, "", "", OpEN.OPEN_USM_USER_PRIV_PROTOCOL_NONE, "", "", ""),
  ("admin2", "network-admin",  OpEN.OPEN_USM_USER_AUTH_PROTOCOL_SHA, "ThisIsaShaPasswordThatIs32Chars.", "", OpEN.OPEN_USM_USER_PRIV_PROTOCOL_NONE, "", "", "0123456789abcdef"),
  ("admin3", "network-admin",  OpEN.OPEN_USM_USER_AUTH_PROTOCOL_SHA, "", "ThisIsaShaKeywordThatMustBe48CharactersInLength.", OpEN.OPEN_USM_USER_PRIV_PROTOCOL_NONE, "", "", "bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb"),
  ("admin4", "network-admin",  OpEN.OPEN_USM_USER_AUTH_PROTOCOL_SHA, "ThisIsaShaPasswordThatIs32Chars.", "", OpEN.OPEN_USM_USER_PRIV_PROTOCOL_DES, "ThisIsaDesPasswordThatIs32Chars.", "", "cccccccccccccccccccccccccccccccc"),
  ("admin5", "network-admin",  OpEN.OPEN_USM_USER_AUTH_PROTOCOL_SHA, "", "ThisIsaShaKeywordThatMustBe48CharactersInLength.", OpEN.OPEN_USM_USER_PRIV_PROTOCOL_DES, "", "ThisIsaDesKeywordThatMustBe32chr", "dddddddddddddddddddddddddddddddd"),
  ("delete-user", "network-admin", OpEN.OPEN_USM_USER_AUTH_PROTOCOL_SHA, "", "ThisIsaShaKeywordThatMustBe48CharactersInLength.", OpEN.OPEN_USM_USER_PRIV_PROTOCOL_DES, "", "ThisIsaDesKeywordThatMustBe32chr", "dddddddddddddddddddddddddddddddd"),
]

# Define SNMP groups
SNMP_GROUP_LIST = [
  ("network-admin", OpEN.OPEN_SNMP_SECURITY_MODEL_USM, OpEN.OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV, "context-1", "view-1", "view-1", "view-1"),
  ("network-guest", OpEN.OPEN_SNMP_SECURITY_MODEL_SNMPV1, OpEN.OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV, "context-2", "view-2", "view-2", "view-2"),
  ("delete-group",  OpEN.OPEN_SNMP_SECURITY_MODEL_SNMPV2C, OpEN.OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV, "context-3", "view-3", "view-3", "view-3"),
]

# Define SNMP views
SNMP_VIEW_LIST = [
  ("view-1", "1.3.6.1.4.1.4413", OpEN.OPEN_SNMP_VIEW_TYPE_INCLUDED),
  ("view-2", "ifEntry.*.1", OpEN.OPEN_SNMP_VIEW_TYPE_EXCLUDED),
]

# Define SNMP filters
SNMP_FILTER_LIST = [
  ("filter-1", "1.3.6.1.4.1.4413", OpEN.OPEN_SNMP_VIEW_TYPE_INCLUDED),
  ("filter-2", "ifEntry.*.1", OpEN.OPEN_SNMP_VIEW_TYPE_EXCLUDED),
]

# Define SNMP hosts
SNMP_HOST_LIST = [
  ("10.10.10.1", 0, "public", OpEN.OPEN_SNMP_SECURITY_MODEL_SNMPV1, OpEN.OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV, OpEN.OPEN_SNMP_NOTIFY_TYPE_TRAP, 0, 0, ""),
  ("10.10.10.2", 0, "public", OpEN.OPEN_SNMP_SECURITY_MODEL_SNMPV2C, OpEN.OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV, OpEN.OPEN_SNMP_NOTIFY_TYPE_TRAP, 0, 0, ""),
  ("2001::", 65535, "admin", OpEN.OPEN_SNMP_SECURITY_MODEL_USM, OpEN.OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV, OpEN.OPEN_SNMP_NOTIFY_TYPE_INFORM, 100, 200, "ipv6-filter"),
]

# Define SNMP communities and community maps
SNMP_COMMUNITY_LIST = [
  {SET:["public-test", OpEN.OPEN_SNMP_COMMUNITY_ACCESS_NONE, "view_test", "20.20.20.1"]},
  {SET:["private-test", OpEN.OPEN_SNMP_COMMUNITY_ACCESS_RW, "", ""]},
  {SET:["delete-community", OpEN.OPEN_SNMP_COMMUNITY_ACCESS_RO, "", ""]},
  {MAP:["delete-private-map", OpEN.OPENAPI_SNMP_DEFAULT_WRITE_STR, "30.30.30.1"]},
]

def ip_to_int(addr):
  """Convert ipv4 string to integer"""

  return struct.unpack("!I", socket.inet_aton(addr))[0]

def int_to_ip(addr):
  """Convert ipv4 integer to string"""

  return socket.inet_ntoa(struct.pack("!I", addr))

def ipv6_to_int(addr):
  """Convert ipv6 string to integer"""

  str_ = socket.inet_pton(socket.AF_INET6, addr)
  a, b = struct.unpack('!2Q', str_)
  return (a << 64) | b

def int_to_ipv6(addr):
  """Convert ipv6 integer to string"""

  a = addr >> 64
  b = addr & ((1 << 64) - 1)
  return socket.inet_ntop(socket.AF_INET6, struct.pack('!2Q', a, b))

def get_type(itype) :
  """Return text representation based on interface type"""

  if itype == OpEN.OPEN_INTF_TYPE_PHY:
    return 'Port'
  elif itype == OpEN.OPEN_INTF_TYPE_VLAN:
    return 'VLAN'
  elif itype == OpEN.OPEN_INTF_TYPE_LOOPBACK:
    return 'Loop'
  elif itype == OpEN.OPEN_INTF_TYPE_TUNNEL:
    return 'Tnnl'
  else:
   return ''

def get_link(state):
  if state == OpEN.OPEN_LINK_UP:
    return 'Up'
  elif state == OpEN.OPEN_LINK_DOWN:
    return 'Down'
  else:
    return 'Unknown'

def print_sanity_results(result, test, msg, feat):
  """Print overall comparison results"""

  if result == OpEN.OPEN_E_UNAVAIL:
    print("Sanity test skipped.")
  elif result == OpEN.OPEN_E_EXISTS:
    print("Sanity Skipped (already exists) - %s - %s." % (msg, feat))
  elif result == OpEN.OPEN_E_NONE and test == True:
    print("Sanity Success - %s - %s." % (msg, feat))
  else:
    print("Sanity Failure - %s - %s." % (msg, feat))

def print_bad_result(result, msg):
  """Print some general error messages if the result is bad"""
   
  if result == OpEN.OPEN_E_UNAVAIL:
    print("Feature not supported - %s (err %d)." % (msg, result))
  elif result == OpEN.OPEN_E_NOT_FOUND:
      print("Test Skipped (not found) - ", msg)
  elif result != OpEN.OPEN_E_NONE:
    print("Test Failure - %s (err %d)." % (msg, result))

class SnmpExample :
  """Simple SNMP class implementing basic CRUD examples """

  def __init__(self, client) :
    self.m_client = client
      
  def test_snmp_sys_name_set(self, name) :
    """Set the SNMP system name.

    An administratively-assigned name for this managed node.
    By convention, this is the node's fully-qualified domain name."""

    try:
      name_string = open_.getStringBuffer(len(name) + 1, name)
    except OpENBufferSizeError:
      print("test_snmp_sys_name_set: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_sys_name_set: getStringBuffer raised TypeError")
      return
    name_buff = OpEN.open_buffdesc()
    name_buff.pstart = name_string
    name_buff.size = len(name)+1

    result = OpEN.openapiSnmpSysNameSet(self.m_client, name_buff) 
    print_bad_result(result, 'openapiSnmpSysNameSet')
    print_sanity_results(result, (1 == 1), 'openapiSnmpSysNameSet', name_string.cast()) 

  def test_snmp_sys_location_set(self, location) :
    """Set the SNMP system location.
    The physical location of this node, for example;
    'telephone closet' or '3rd floor'."""

    try:
      name_string = open_.getStringBuffer(len(location) + 1, location)
    except OpENBufferSizeError:
      print("test_snmp_sys_location_set: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_sys_location_set: getStringBuffer raised TypeError")
      return
    name_buff = OpEN.open_buffdesc()
    name_buff.pstart = name_string
    name_buff.size = len(location)+1

    result = OpEN.openapiSnmpSysLocationSet(self.m_client, name_buff) 
    print_bad_result(result, 'openapiSnmpSysLocationSet')
    print_sanity_results(result, (1 == 1), 'openapiSnmpSysLocationSet', name_string.cast()) 

  def test_snmp_sys_contact_set(self, contact) :
    """Set the SNMP system contact.

    The textual identification of the contact person for this managed node,
    together with information on how to contact this person."""

    try:
      name_string = open_.getStringBuffer(len(contact) + 1, contact)
    except OpENBufferSizeError:
      print("test_snmp_sys_contact_set: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_sys_contact_set: getStringBuffer raised TypeError")
      return
    name_buff = OpEN.open_buffdesc()
    name_buff.pstart = name_string
    name_buff.size = len(contact)+1

    result = OpEN.openapiSnmpSysContactSet(self.m_client, name_buff) 
    print_bad_result(result, 'openapiSnmpSysContactSet')
    print_sanity_results(result, (1 == 1), 'openapiSnmpSysContactSet', name_string.cast()) 

  def test_snmp_local_engine_id_set(self, engine_id) :
    """Set the SNMP engine id on the local device."""

    if engine_id == 'default':
      engine_id = '1'

    try:
      name_string = open_.getStringBuffer(len(engine_id) + 1, engine_id)
    except OpENBufferSizeError:
      print("test_snmp_local_engine_id_set: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_local_engine_id_set: getStringBuffer raised TypeError")
      return
    name_buff = OpEN.open_buffdesc()
    name_buff.pstart = name_string
    name_buff.size = len(engine_id)+1

    result = OpEN.openapiSnmpLocalEngineIdSet(self.m_client, name_buff) 
    print_bad_result(result, 'openapiSnmpLocalEngineIdSet')
    print_sanity_results(result, (1 == 1), 'openapiSnmpLocalEngineIdSet', name_string.cast()) 

  def test_snmp_user_create(self, user_list) :
    """Create a new SNMP user"""

    for val in user_list:
      name       = val[0]
      group      = val[1]
      auth_proto = val[2]
      auth_pwd   = val[3]
      auth_key   = val[4]
      priv_proto = val[5]
      priv_pwd   = val[6]
      priv_key   = val[7]
      engine_id  = val[8]

      try:
        name_string = open_.getStringBuffer(len(name) + 1, name)
      except OpENBufferSizeError:
        print("test_snmp_user_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_create: getStringBuffer raised TypeError")
        return
      name_buff = OpEN.open_buffdesc()
      name_buff.pstart = name_string
      name_buff.size = len(name)+1

      try:
        group_string = open_.getStringBuffer(len(group) + 1, group)
      except OpENBufferSizeError:
        print("test_snmp_user_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_create: getStringBuffer raised TypeError")
        return
      group_buff = OpEN.open_buffdesc()
      group_buff.pstart = group_string
      group_buff.size = len(group)+1

      try:
        auth_pwd_string = open_.getStringBuffer(len(auth_pwd) + 1, auth_pwd)
      except OpENBufferSizeError:
        print("test_snmp_user_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_create: getStringBuffer raised TypeError")
        return
      auth_pwd_buff = OpEN.open_buffdesc()
      auth_pwd_buff.pstart = auth_pwd_string
      auth_pwd_buff.size = len(auth_pwd)+1

      try:
        auth_key_string = open_.getStringBuffer(len(auth_key) + 1, auth_key)
      except OpENBufferSizeError:
        print("test_snmp_user_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_create: getStringBuffer raised TypeError")
        return
      auth_key_buff = OpEN.open_buffdesc()
      auth_key_buff.pstart = auth_key_string
      auth_key_buff.size = len(auth_key)+1

      try:
        priv_pwd_string = open_.getStringBuffer(len(priv_pwd) + 1, priv_pwd)
      except OpENBufferSizeError:
        print("test_snmp_user_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_create: getStringBuffer raised TypeError")
        return
      priv_pwd_buff = OpEN.open_buffdesc()
      priv_pwd_buff.pstart = priv_pwd_string
      priv_pwd_buff.size = len(priv_pwd)+1

      try:
        priv_key_string = open_.getStringBuffer(len(priv_key) + 1, priv_key)
      except OpENBufferSizeError:
        print("test_snmp_user_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_create: getStringBuffer raised TypeError")
        return
      priv_key_buff = OpEN.open_buffdesc()
      priv_key_buff.pstart = priv_key_string
      priv_key_buff.size = len(priv_key)+1

      try:
        engine_id_string = open_.getStringBuffer(len(engine_id) + 1, engine_id)
      except OpENBufferSizeError:
        print("test_snmp_user_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_create: getStringBuffer raised TypeError")
        return
      engine_id_buff = OpEN.open_buffdesc()
      engine_id_buff.pstart = engine_id_string
      engine_id_buff.size = len(engine_id)+1

      result = OpEN.openapiSnmpUserCreate(self.m_client,
          name_buff, group_buff,
          auth_proto, auth_pwd_buff, auth_key_buff,
          priv_proto, priv_pwd_buff, priv_key_buff,
          engine_id_buff)
      print_sanity_results(result, (1 == 1), 'openapiSnmpUserCreate', name_string.cast()) 

  def test_snmp_group_create(self, group_list) :
    """Create a new SNMP group"""

    for val in group_list:
      group          = val[0]
      security_model = val[1]
      security_level = val[2]
      context_prefix = val[3]
      read_view      = val[4]
      write_view     = val[5]
      notify_view    = val[6]

      try:
        group_string = open_.getStringBuffer(len(group) + 1, group)
      except OpENBufferSizeError:
        print("test_snmp_group_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_group_create: getStringBuffer raised TypeError")
        return
      group_buff = OpEN.open_buffdesc()
      group_buff.pstart = group_string
      group_buff.size = len(group)+1

      try:
        context_prefix_string = open_.getStringBuffer(len(context_prefix) + 1, context_prefix)
      except OpENBufferSizeError:
        print("test_snmp_group_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_group_create: getStringBuffer raised TypeError")
        return
      context_prefix_buff = OpEN.open_buffdesc()
      context_prefix_buff.pstart = context_prefix_string
      context_prefix_buff.size = len(context_prefix)+1

      try:
        read_view_string = open_.getStringBuffer(len(read_view) + 1, read_view)
      except OpENBufferSizeError:
        print("test_snmp_group_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_group_create: getStringBuffer raised TypeError")
        return
      read_view_buff = OpEN.open_buffdesc()
      read_view_buff.pstart = read_view_string
      read_view_buff.size = len(read_view)+1

      try:
        write_view_string = open_.getStringBuffer(len(write_view) + 1, write_view)
      except OpENBufferSizeError:
        print("test_snmp_group_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_group_create: getStringBuffer raised TypeError")
        return
      write_view_buff = OpEN.open_buffdesc()
      write_view_buff.pstart = write_view_string
      write_view_buff.size = len(write_view)+1

      try:
        notify_view_string = open_.getStringBuffer(len(notify_view) + 1, notify_view)
      except OpENBufferSizeError:
        print("test_snmp_group_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_group_create: getStringBuffer raised TypeError")
        return
      notify_view_buff = OpEN.open_buffdesc()
      notify_view_buff.pstart = notify_view_string
      notify_view_buff.size = len(notify_view)+1

      result = OpEN.openapiSnmpGroupCreate(self.m_client,
          group_buff, security_model, security_level,
          context_prefix_buff, read_view_buff, write_view_buff, notify_view_buff)
      print_sanity_results(result, (1 == 1), 'openapiSnmpGroupCreate', group_string.cast()) 

  def test_snmp_view_create(self, view_list) :
    """Create a new SNMP view"""

    for val in view_list:
      view_name = val[0]
      oid_tree  = val[1]
      view_type = val[2]

      try:
        view_name_string = open_.getStringBuffer(len(view_name) + 1, view_name)
      except OpENBufferSizeError:
        print("test_snmp_view_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_view_create: getStringBuffer raised TypeError")
        return
      view_name_buff = OpEN.open_buffdesc()
      view_name_buff.pstart = view_name_string
      view_name_buff.size = len(view_name)+1

      try:
        oid_tree_string = open_.getStringBuffer(len(oid_tree) + 1, oid_tree)
      except OpENBufferSizeError:
        print("test_snmp_view_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_view_create: getStringBuffer raised TypeError")
        return
      oid_tree_buff = OpEN.open_buffdesc()
      oid_tree_buff.pstart = oid_tree_string
      oid_tree_buff.size = len(oid_tree)+1

      result = OpEN.openapiSnmpViewCreate(self.m_client, view_name_buff, oid_tree_buff, view_type)
      print_sanity_results(result, (1 == 1), 'openapiSnmpViewCreate', view_name_string.cast()) 

  def test_snmp_filter_create(self, filter_list) :
    """Create a new SNMP filter"""

    for val in filter_list:
      filter_name = val[0]
      oid_tree    = val[1]
      view_type   = val[2]

      try:
        filter_name_string = open_.getStringBuffer(len(filter_name) + 1, filter_name)
      except OpENBufferSizeError:
        print("test_snmp_filter_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_filter_create: getStringBuffer raised TypeError")
        return
      filter_name_buff = OpEN.open_buffdesc()
      filter_name_buff.pstart = filter_name_string
      filter_name_buff.size = len(filter_name)+1

      try:
        oid_tree_string = open_.getStringBuffer(len(oid_tree) + 1, oid_tree)
      except OpENBufferSizeError:
        print("test_snmp_filter_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_filter_create: getStringBuffer raised TypeError")
        return
      oid_tree_buff = OpEN.open_buffdesc()
      oid_tree_buff.pstart = oid_tree_string
      oid_tree_buff.size = len(oid_tree)+1

      result = OpEN.openapiSnmpFilterCreate(self.m_client, filter_name_buff, oid_tree_buff, view_type)
      print_sanity_results(result, (1 == 1), 'openapiSnmpFilterCreate', filter_name_string.cast()) 

  def test_snmp_host_create(self, host_list) :
    """Create a new SNMP host"""

    for val in host_list:
      host           = val[0]
      server_port    = val[1]
      security_name  = val[2]
      security_model = val[3]
      security_level = val[4]
      notify_type    = val[5]
      timeout        = val[6]
      retries        = val[7]
      filter_name    = val[8]

      try:
        host_string = open_.getStringBuffer(len(host) + 1, host)
      except OpENBufferSizeError:
        print("test_snmp_host_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_host_create: getStringBuffer raised TypeError")
        return
      host_buff = OpEN.open_buffdesc()
      host_buff.pstart = host_string
      host_buff.size = len(host)+1

      try:
        security_name_string = open_.getStringBuffer(len(security_name) + 1, security_name)
      except OpENBufferSizeError:
        print("test_snmp_host_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_host_create: getStringBuffer raised TypeError")
        return
      security_name_buff = OpEN.open_buffdesc()
      security_name_buff.pstart = security_name_string
      security_name_buff.size = len(security_name)+1

      try:
        filter_name_string = open_.getStringBuffer(len(filter_name) + 1, filter_name)
      except OpENBufferSizeError:
        print("test_snmp_host_create: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_host_create: getStringBuffer raised TypeError")
        return
      filter_name_buff = OpEN.open_buffdesc()
      filter_name_buff.pstart = filter_name_string
      filter_name_buff.size = len(filter_name)+1

      result = OpEN.openapiSnmpHostCreate(self.m_client,
          host_buff, server_port,
          security_name_buff, security_model, security_level,
          notify_type, timeout, retries, filter_name_buff)
      print_sanity_results(result, (1 == 1), 'openapiSnmpHostCreate', host_string.cast())

  def test_snmp_community_create(self, community_list) :
    """Create a new SNMP community

    SET key defines a basic community creation.
    MAP key indicates a map to a new internal security name for SNMP v1 and SNMP v2 security
    models to an existing group name."""

    for community in community_list:
      for key, val in community.items():

        if key == SET:
          name    = val[0]
          access  = val[1]
          view    = val[2]
          address = val[3]

          try:
            name_string = open_.getStringBuffer(len(name) + 1, name)
          except OpENBufferSizeError:
            print("test_snmp_community_create: getStringBuffer raised OpENBufferSizeError")
            return
          except TypeError:
            print("test_snmp_community_create: getStringBuffer raised TypeError")
            return
          name_buff = OpEN.open_buffdesc()
          name_buff.pstart = name_string
          name_buff.size = len(name)+1

          try:
            view_string = open_.getStringBuffer(len(view) + 1, view)
          except OpENBufferSizeError:
            print("test_snmp_community_create: getStringBuffer raised OpENBufferSizeError")
            return
          except TypeError:
            print("test_snmp_community_create: getStringBuffer raised TypeError")
            return
          view_buff = OpEN.open_buffdesc()
          view_buff.pstart = view_string
          view_buff.size = len(view)+1

          ip_addr = OpEN.open_inet_addr_t()
          if len(address) > 0:
            ip_addr.addr.ipv4 = ip_to_int(address)
          ip_addr.family = socket.AF_INET

          result = OpEN.openapiSnmpCommunityCreate(self.m_client, name_buff, access, view_buff, ip_addr)
          print_sanity_results(result, (1 == 1), 'openapiSnmpCommunityCreate', name_string.cast())

        elif key == MAP:
          name     = val[0]
          map_name = val[1]
          address  = val[2]

          try:
            name_string = open_.getStringBuffer(len(name) + 1, name)
          except OpENBufferSizeError:
            print("test_snmp_community_create: getStringBuffer raised OpENBufferSizeError")
            return
          except TypeError:
            print("test_snmp_community_create: getStringBuffer raised TypeError")
            return
          name_buff = OpEN.open_buffdesc()
          name_buff.pstart = name_string
          name_buff.size = len(name)+1

          try:
            map_name_string = open_.getStringBuffer(len(map_name) + 1, map_name)
          except OpENBufferSizeError:
            print("test_snmp_community_create: getStringBuffer raised OpENBufferSizeError")
            return
          except TypeError:
            print("test_snmp_community_create: getStringBuffer raised TypeError")
            return
          map_name_buff = OpEN.open_buffdesc()
          map_name_buff.pstart = map_name_string
          map_name_buff.size = len(map_name)+1

          ip_addr = OpEN.open_inet_addr_t()
          if len(address) > 0:
            ip_addr.addr.ipv4 = ip_to_int(address)
          ip_addr.family = socket.AF_INET

          result = OpEN.openapiSnmpCommunityGroupCreate(self.m_client, name_buff, map_name_buff, ip_addr)
          print_sanity_results(result, (1 == 1), 'openapiSnmpCommunityGroupCreate', name_string.cast())

  def testSnmpTrapFlagSet(self, flags, mode) :
    """Enable or disable the traps identified via the flags selection"""

    val = mode
    flag_bits = 0

    for flag in flags:
      flag_bits |= flag

    result = OpEN.openapiSnmpTrapFlagSet(self.m_client, flag_bits, val)
    print_sanity_results(result, (1 == 1), 'openapiSnmpTrapFlagSet', str(flag_bits))

  def test_snmp_sys_name_get(self) :
    """Get the SNMP system name.

    An administratively-assigned name for this managed node.
    By convention, this is the node's fully-qualified domain name."""

    max_len = 255+1

    try:
      name_string       = open_.getStringBuffer(max_len) 
    except OpENBufferSizeError:
      print("test_snmp_sys_name_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_sys_name_get: getStringBuffer raised TypeError")
      return
    name_buff         = OpEN.open_buffdesc()
    name_buff.pstart  = name_string
    name_buff.size    = max_len

    result = OpEN.openapiSnmpSysNameGet(self.m_client, name_buff)
    print_bad_result(result, 'openapiSnmpSysNameGet')
    print_sanity_results(result, (1 == 1), 'openapiSnmpSysNameGet', name_string.cast()) 

  def test_snmp_sys_location_get(self) :
    """Get the SNMP system location.

    The physical location of this node, for example;
    'telephone closet' or '3rd floor'."""

    max_len = 255+1

    try:
      name_string       = open_.getStringBuffer(max_len) 
    except OpENBufferSizeError:
      print("test_snmp_sys_location_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_sys_location_get: getStringBuffer raised TypeError")
      return
    name_buff         = OpEN.open_buffdesc()
    name_buff.pstart  = name_string
    name_buff.size    = max_len

    result = OpEN.openapiSnmpSysLocationGet(self.m_client, name_buff)
    print_bad_result(result, 'openapiSnmpSysLocationGet')
    print_sanity_results(result, (1 == 1), 'openapiSnmpSysLocationGet', name_string.cast()) 

  def test_snmp_sys_contact_get(self) :
    """Get the SNMP system contact.

    The textual identification of the contact person for this managed node,
    together with information on how to contact this person."""

    max_len = 255+1

    try:
      name_string       = open_.getStringBuffer(max_len) 
    except OpENBufferSizeError:
      print("test_snmp_sys_contact_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_sys_contact_get: getStringBuffer raised TypeError")
      return
    name_buff         = OpEN.open_buffdesc()
    name_buff.pstart  = name_string
    name_buff.size    = max_len

    result = OpEN.openapiSnmpSysContactGet(self.m_client, name_buff)
    print_bad_result(result, 'openapiSnmpSysContactGet')
    print_sanity_results(result, (1 == 1), 'openapiSnmpSysContactGet', name_string.cast()) 

  def test_snmp_local_engine_id_get(self) :
      """Get the SNMP engine id on the local device."""

      max_len = 32+1

      try:
        engine_id_string       = open_.getStringBuffer(max_len) 
      except OpENBufferSizeError:
        print("test_snmp_local_engine_id_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_local_engine_id_get: getStringBuffer raised TypeError")
        return
      engine_id_buff         = OpEN.open_buffdesc()
      engine_id_buff.pstart  = engine_id_string
      engine_id_buff.size    = max_len
      result = OpEN.openapiSnmpLocalEngineIdGet(self.m_client, engine_id_buff)
      print_bad_result(result, 'openapiSnmpLocalEngineIdGet')

      if result == OpEN.OPEN_E_NONE:
        engine_id = ''
        if version_info >= (3,0,0):
          for idx in range(0, engine_id_buff.size):
            engine_id += engine_id_string[idx].encode('hex')
        else:
          for idx in xrange(0, engine_id_buff.size):
            engine_id += engine_id_string[idx].encode('hex')

      print_sanity_results(result, (1 == 1), 'openapiSnmpLocalEngineIdGet', engine_id) 

  def test_snmp_user_get(self, user_name) :
      """Get the specified SNMP user"""

      found = False
      max_user_len = 30+1
      max_engine_id_len = 32+1

      auth_p = OpEN.new_OPEN_USM_USER_AUTH_PROTOCOL_tp()
      priv_p = OpEN.new_OPEN_USM_USER_PRIV_PROTOCOL_tp()

      try:
        user_string       = open_.getStringBuffer(max_user_len)
      except OpENBufferSizeError:
        print("test_snmp_user_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_get: getStringBuffer raised TypeError")
        return
      user_buff         = OpEN.open_buffdesc()
      user_buff.pstart  = user_string
      user_buff.size    = max_user_len

      try:
        group_string      = open_.getStringBuffer(max_user_len)
      except OpENBufferSizeError:
        print("test_snmp_user_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_get: getStringBuffer raised TypeError")
        return
      group_buff        = OpEN.open_buffdesc()
      group_buff.pstart = group_string
      group_buff.size   = max_user_len

      try:
        engine_id_string      = open_.getStringBuffer(max_engine_id_len) 
      except OpENBufferSizeError:
        print("test_snmp_user_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_get: getStringBuffer raised TypeError")
        return
      engine_id_buff        = OpEN.open_buffdesc()
      engine_id_buff.pstart = engine_id_string
      engine_id_buff.size   = max_engine_id_len

      try:
        next_user_string      = open_.getStringBuffer(max_user_len)
      except OpENBufferSizeError:
        print("test_snmp_user_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_get: getStringBuffer raised TypeError")
        return
      next_user_buff        = OpEN.open_buffdesc()
      next_user_buff.pstart = next_user_string
      next_user_buff.size   = max_user_len

      try:
        next_engine_id_string      = open_.getStringBuffer(max_engine_id_len)
      except OpENBufferSizeError:
        print("test_snmp_user_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_user_get: getStringBuffer raised TypeError")
        return
      next_engine_id_buff        = OpEN.open_buffdesc()
      next_engine_id_buff.pstart = next_engine_id_string
      next_engine_id_buff.size   = max_engine_id_len

      while OpEN.openapiSnmpUserGetNext(self.m_client, 
                                        user_buff, 
                                        engine_id_buff, 
                                        group_buff, 
                                        auth_p,
                                        priv_p, 
                                        next_user_buff, 
                                        next_engine_id_buff) == OpEN.OPEN_E_NONE:

        if user_name == next_user_string.cast():
          found = True
          break

        user_buff.pstart = next_user_buff.pstart
        user_buff.size   = next_user_buff.size
        engine_id_buff.pstart = next_engine_id_buff.pstart
        engine_id_buff.size   = next_engine_id_buff.size
        next_user_buff.size = max_user_len
        next_engine_id_buff.size = max_engine_id_len
    
      print_sanity_results(OpEN.OPEN_E_NONE, found, 'openapiSnmpUserGetNext', user_name) 

      if found and verbose:
        print("  User name      : ", next_user_string.cast())
        print("  Group name     : ", group_string.cast())
        print("  Authentication : ", OpEN.OPEN_USM_USER_AUTH_PROTOCOL_tp_value(auth_p))
        print("  Privilege      : ", OpEN.OPEN_USM_USER_PRIV_PROTOCOL_tp_value(priv_p))

        engine_id = ''
        if version_info >= (3,0,0):
          for idx in range(0, next_engine_id_buff.size):
            engine_id += next_engine_id_string[idx].encode('hex')
        else:
          for idx in xrange(0, next_engine_id_buff.size):
            engine_id += next_engine_id_string[idx].encode('hex')
        print("  Engine Id      : ", engine_id)

      OpEN.delete_OPEN_USM_USER_AUTH_PROTOCOL_tp(auth_p)
      OpEN.delete_OPEN_USM_USER_PRIV_PROTOCOL_tp(priv_p)

  def test_snmp_group_get(self, group_name, security_model, security_level, context_name) :
    """Get the specified SNMP group"""

    found = False
    max_user_len = 30+1
    max_context_len = 30+1
    max_view_len = 30+1

    model_p = OpEN.new_OPEN_SNMP_SECURITY_MODEL_tp()
    level_p = OpEN.new_OPEN_SNMP_SECURITY_LEVEL_tp()

    try:
      group_string      = open_.getStringBuffer(max_user_len)
    except OpENBufferSizeError:
      print("test_snmp_group_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_group_get: getStringBuffer raised TypeError")
      return
    group_buff        = OpEN.open_buffdesc()
    group_buff.pstart = group_string
    group_buff.size   = max_user_len

    try:
      read_string      = open_.getStringBuffer(max_view_len)
    except OpENBufferSizeError:
      print("test_snmp_group_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_group_get: getStringBuffer raised TypeError")
      return
    read_buff        = OpEN.open_buffdesc()
    read_buff.pstart = read_string
    read_buff.size   = max_view_len

    try:
      write_string      = open_.getStringBuffer(max_view_len)
    except OpENBufferSizeError:
      print("test_snmp_group_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_group_get: getStringBuffer raised TypeError")
      return
    write_buff        = OpEN.open_buffdesc()
    write_buff.pstart = write_string
    write_buff.size   = max_view_len

    try:
      notify_string      = open_.getStringBuffer(max_view_len)
    except OpENBufferSizeError:
      print("test_snmp_group_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_group_get: getStringBuffer raised TypeError")
      return
    notify_buff        = OpEN.open_buffdesc()
    notify_buff.pstart = notify_string
    notify_buff.size   = max_view_len

    try:
      context_string      = open_.getStringBuffer(max_user_len)
    except OpENBufferSizeError:
      print("test_snmp_group_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_group_get: getStringBuffer raised TypeError")
      return
    context_buff        = OpEN.open_buffdesc()
    context_buff.pstart = context_string
    context_buff.size   = max_context_len

    try:
      next_group_string      = open_.getStringBuffer(max_user_len)
    except OpENBufferSizeError:
      print("test_snmp_group_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_group_get: getStringBuffer raised TypeError")
      return
    next_group_buff        = OpEN.open_buffdesc()
    next_group_buff.pstart = next_group_string
    next_group_buff.size   = max_user_len

    try:
      next_context_string      = open_.getStringBuffer(max_user_len)
    except OpENBufferSizeError:
      print("test_snmp_group_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_group_get: getStringBuffer raised TypeError")
      return
    next_context_buff        = OpEN.open_buffdesc()
    next_context_buff.pstart = next_context_string
    next_context_buff.size   = max_user_len

    while OpEN.openapiSnmpGroupGetNext(self.m_client, 
                                       group_buff, 
                                       model_p,
                                       level_p, 
                                       context_buff,
                                       read_buff, 
                                       write_buff, 
                                       notify_buff, 
                                       next_group_buff, 
                                       next_context_buff) == OpEN.OPEN_E_NONE:

      if (group_name == next_group_string.cast() and
          context_name == next_context_string.cast()):
        found = True
        break

      try:
        read_string        = open_.getStringBuffer(max_view_len)
      except OpENBufferSizeError:
        print("test_snmp_group_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_group_get: getStringBuffer raised TypeError")
        return
      read_buff.pstart   = read_string
      read_buff.size     = max_view_len
      try:
        write_string       = open_.getStringBuffer(max_view_len)
      except OpENBufferSizeError:
        print("test_snmp_group_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_group_get: getStringBuffer raised TypeError")
        return
      write_buff.pstart  = write_string
      write_buff.size    = max_view_len
      try:
        notify_string      = open_.getStringBuffer(max_view_len)
      except OpENBufferSizeError:
        print("test_snmp_group_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_group_get: getStringBuffer raised TypeError")
        return
      notify_buff.pstart = notify_string
      notify_buff.size   = max_view_len

      group_buff.pstart = next_group_buff.pstart
      group_buff.size   = next_group_buff.size
      context_buff.pstart = next_context_buff.pstart
      context_buff.size   = next_context_buff.size
      next_group_buff.size = max_user_len
      next_context_buff.size = max_context_len
    
    print_sanity_results(OpEN.OPEN_E_NONE, found, 'openapiSnmpGroupGetNext', group_name) 

    if found and verbose:
      print("  Group name     : ", next_group_string.cast())
      print("  Security model : ", OpEN.OPEN_SNMP_SECURITY_MODEL_tp_value(model_p))
      print("  Security level : ", OpEN.OPEN_SNMP_SECURITY_LEVEL_tp_value(level_p))
      print("  Context name   : ", next_context_string.cast())
      print("  Read view      : ", read_string.cast())
      print("  Write view     : ", write_string.cast())
      print("  Notify view    : ", notify_string.cast())

    OpEN.delete_OPEN_SNMP_SECURITY_MODEL_tp(model_p)
    OpEN.delete_OPEN_SNMP_SECURITY_LEVEL_tp(level_p)

  def test_snmp_view_get(self, view_name, oid_tree) :
    """Get the specified SNMP view name

    Iterate through the existing SNMP views until a match for view and oid is found.
    If found, display the associated view parameters."""

    found = False
    max_oid_len = 128+1
    max_view_len = 30+1

    view_p = OpEN.new_OPEN_SNMP_VIEW_TYPE_tp()

    try: 
      view_string      = open_.getStringBuffer(len(view_name) + 1,view_name)
    except OpENBufferSizeError:
      print("test_snmp_view_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_view_get: getStringBuffer raised TypeError")
      return
    view_buff        = OpEN.open_buffdesc()
    view_buff.pstart = view_string
    view_buff.size   = len(view_name) + 1

    try:
      oid_string       = open_.getStringBuffer(len(oid_tree) + 1,oid_tree)
    except OpENBufferSizeError:
      print("test_snmp_view_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_view_get: getStringBuffer raised TypeError")
      return
    oid_buff         = OpEN.open_buffdesc()
    oid_buff.pstart  = oid_string
    oid_buff.size    = len(oid_tree) + 1

    try:
      next_view_string      = open_.getStringBuffer(max_view_len)
    except OpENBufferSizeError:
      print("test_snmp_view_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_view_get: getStringBuffer raised TypeError")
      return
    next_view_buff        = OpEN.open_buffdesc()
    next_view_buff.pstart = next_view_string
    next_view_buff.size   = max_view_len

    try:
      next_oid_string       = open_.getStringBuffer(max_oid_len)
    except OpENBufferSizeError:
      print("test_snmp_view_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_view_get: getStringBuffer raised TypeError")
      return
    next_oid_buff         = OpEN.open_buffdesc()
    next_oid_buff.pstart  = next_oid_string
    next_oid_buff.size    = max_oid_len

    try:
      oid_string_string      = open_.getStringBuffer(max_oid_len)
    except OpENBufferSizeError:
      print("test_snmp_view_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_view_get: getStringBuffer raised TypeError")
      return
    oid_string_buff        = OpEN.open_buffdesc()
    oid_string_buff.pstart = oid_string_string
    oid_string_buff.size   = max_oid_len

    try:
      oid_val_string       = open_.getStringBuffer(max_oid_len)
    except OpENBufferSizeError:
      print("test_snmp_view_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_view_get: getStringBuffer raised TypeError")
      return
    oid_val_buff         = OpEN.open_buffdesc()
    oid_val_buff.pstart  = oid_val_string
    oid_val_buff.size    = max_oid_len

    OpEN.openapiSnmpTreeFamilyOidStringGet(self.m_client, view_buff, oid_buff, OpEN.OPEN_SNMP_VIEW_TREE, oid_string_buff)
    OpEN.openapiSnmpTreeFamilyOidValGet(self.m_client, view_buff, oid_buff, OpEN.OPEN_SNMP_VIEW_TREE, oid_val_buff)

    try:
      view_string      = open_.getStringBuffer(max_view_len)
    except OpENBufferSizeError:
      print("test_snmp_view_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_view_get: getStringBuffer raised TypeError")
      return
    view_buff        = OpEN.open_buffdesc()
    view_buff.pstart = view_string
    view_buff.size   = max_view_len

    try:
      oid_string       = open_.getStringBuffer(max_oid_len)
    except OpENBufferSizeError:
      print("test_snmp_view_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_view_get: getStringBuffer raised TypeError")
      return
    oid_buff         = OpEN.open_buffdesc()
    oid_buff.pstart  = oid_string
    oid_buff.size    = max_oid_len

    while OpEN.openapiSnmpViewGetNext(self.m_client, 
                                       view_buff, 
                                       oid_buff, 
                                       view_p,
                                       next_view_buff, 
                                       next_oid_buff) == OpEN.OPEN_E_NONE:

      if (view_name == next_view_string.cast() and
          oid_val_string.cast() == next_oid_string.cast()):
        found = True
        break

      view_buff.pstart = next_view_buff.pstart
      view_buff.size   = next_view_buff.size
      oid_buff.pstart = next_oid_buff.pstart
      oid_buff.size   = next_oid_buff.size
      next_view_buff.size = max_view_len
      next_oid_buff.size = max_oid_len
    
    print_sanity_results(OpEN.OPEN_E_NONE, found, 'openapiSnmpViewGetNext', view_name) 

    if found and verbose:
      print("  View name  : ", next_view_string.cast())
      print("  OID Tree   : ", next_oid_string.cast())
      print("  OID String : ", oid_string_string.cast())
      print("  View Type  : ", OpEN.OPEN_SNMP_VIEW_TYPE_tp_value(view_p))

    OpEN.delete_OPEN_SNMP_VIEW_TYPE_tp(view_p)

  def test_snmp_filter_get(self, filter_name, oid_tree) :
    """Get the specified SNMP filter name

    Iterate through the existing SNMP views until a match for view and oid is found.
    If found, display the associated view parameters."""

    found = False
    max_oid_len = 128+1
    max_filter_len = 30+1

    filter_p = OpEN.new_OPEN_SNMP_NOTIFY_FILTER_TYPE_tp()

    try:
      filter_string      = open_.getStringBuffer(len(filter_name) + 1,filter_name)
    except OpENBufferSizeError:
      print("test_snmp_filter_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_filter_get: getStringBuffer raised TypeError")
      return
    filter_buff        = OpEN.open_buffdesc()
    filter_buff.pstart = filter_string
    filter_buff.size   = len(filter_name) + 1

    try:
      oid_string       = open_.getStringBuffer(len(oid_tree) + 1,oid_tree)
    except OpENBufferSizeError:
      print("test_snmp_filter_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_filter_get: getStringBuffer raised TypeError")
      return
    oid_buff         = OpEN.open_buffdesc()
    oid_buff.pstart  = oid_string
    oid_buff.size    = len(oid_tree)

    try:
      next_filter_string      = open_.getStringBuffer(max_filter_len)
    except OpENBufferSizeError:
      print("test_snmp_filter_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_filter_get: getStringBuffer raised TypeError")
      return
    next_filter_buff        = OpEN.open_buffdesc()
    next_filter_buff.pstart = next_filter_string
    next_filter_buff.size   = max_filter_len

    try:
      next_oid_string       = open_.getStringBuffer(max_oid_len)
    except OpENBufferSizeError:
      print("test_snmp_filter_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_filter_get: getStringBuffer raised TypeError")
      return
    next_oid_buff         = OpEN.open_buffdesc()
    next_oid_buff.pstart  = next_oid_string
    next_oid_buff.size    = max_oid_len

    try:
      oid_string_string      = open_.getStringBuffer(max_oid_len)
    except OpENBufferSizeError:
      print("test_snmp_filter_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_filter_get: getStringBuffer raised TypeError")
      return
    oid_string_buff        = OpEN.open_buffdesc()
    oid_string_buff.pstart = oid_string_string
    oid_string_buff.size   = max_oid_len

    try:
      oid_val_string       = open_.getStringBuffer(max_oid_len)
    except OpENBufferSizeError:
      print("test_snmp_filter_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_filter_get: getStringBuffer raised TypeError")
      return
    oid_val_buff         = OpEN.open_buffdesc()
    oid_val_buff.pstart  = oid_val_string
    oid_val_buff.size    = max_oid_len

    OpEN.openapiSnmpTreeFamilyOidStringGet(self.m_client, filter_buff, oid_buff, OpEN.OPEN_SNMP_FILTER_TREE, oid_string_buff)
    OpEN.openapiSnmpTreeFamilyOidValGet(self.m_client, filter_buff, oid_buff, OpEN.OPEN_SNMP_FILTER_TREE, oid_val_buff)

    try:
      filter_string      = open_.getStringBuffer(max_filter_len)
    except OpENBufferSizeError:
      print("test_snmp_filter_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_filter_get: getStringBuffer raised TypeError")
      return
    filter_buff        = OpEN.open_buffdesc()
    filter_buff.pstart = filter_string
    filter_buff.size   = max_filter_len

    try:
      oid_string       = open_.getStringBuffer(max_oid_len)
    except OpENBufferSizeError:
      print("test_snmp_filter_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_filter_get: getStringBuffer raised TypeError")
      return
    oid_buff         = OpEN.open_buffdesc()
    oid_buff.pstart  = oid_string
    oid_buff.size    = max_oid_len

    while OpEN.openapiSnmpFilterGetNext(self.m_client, 
                                        filter_buff, 
                                        oid_buff, 
                                        filter_p,
                                        next_filter_buff, 
                                        next_oid_buff) == OpEN.OPEN_E_NONE:

      if (filter_name == next_filter_string.cast() and
          oid_val_string.cast() == next_oid_string.cast()):
        found = True
        break

      filter_buff.pstart = next_filter_buff.pstart
      filter_buff.size   = next_filter_buff.size
      oid_buff.pstart = next_oid_buff.pstart
      oid_buff.size   = next_oid_buff.size
      next_filter_buff.size = max_filter_len
      next_oid_buff.size = max_oid_len
    
    print_sanity_results(OpEN.OPEN_E_NONE, found, 'openapiSnmpFilterGetNext', filter_name) 

    if found and verbose:
      print("  Filter name : ", next_filter_string.cast())
      print("  OID Tree    : ", next_oid_string.cast())
      print("  OID String  : ", oid_string_string.cast())
      print("  View Type   : ", OpEN.OPEN_SNMP_NOTIFY_FILTER_TYPE_tp_value(filter_p))

    OpEN.delete_OPEN_SNMP_NOTIFY_FILTER_TYPE_tp(filter_p)

  def test_snmp_host_get(self, host) :
    """Get the specified SNMP host name

    Iterate through the existing SNMP hosts until a match for
    host is found. If found, display the associated host parameters."""

    found = False
    max_host_len = 158+1
    max_filter_len = 30+1
    max_security_len = 30+1

    port_p = OpEN.new_uint32_tp()
    model_p = OpEN.new_OPEN_SNMP_SECURITY_MODEL_tp()
    level_p = OpEN.new_OPEN_SNMP_SECURITY_LEVEL_tp()
    type_p = OpEN.new_OPEN_SNMP_NOTIFY_TYPE_tp()
    timeout_p = OpEN.new_uint32_tp()
    retries_p = OpEN.new_uint32_tp()

    try:
      host_string           = open_.getStringBuffer(max_host_len)
    except OpENBufferSizeError:
      print("test_snmp_host_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_host_get: getStringBuffer raised TypeError")
      return
    host_buff             = OpEN.open_buffdesc()
    host_buff.pstart      = host_string
    host_buff.size        = max_host_len

    try:
      security_string       = open_.getStringBuffer(max_security_len)
    except OpENBufferSizeError:
      print("test_snmp_host_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_host_get: getStringBuffer raised TypeError")
      return
    security_buff         = OpEN.open_buffdesc()
    security_buff.pstart  = security_string
    security_buff.size    = max_security_len

    try:
      filter_string         = open_.getStringBuffer(max_filter_len)
    except OpENBufferSizeError:
      print("test_snmp_host_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_host_get: getStringBuffer raised TypeError")
      return
    filter_buff           = OpEN.open_buffdesc()
    filter_buff.pstart    = filter_string
    filter_buff.size      = max_filter_len

    try:
      next_host_string      = open_.getStringBuffer(max_host_len)
    except OpENBufferSizeError:
      print("test_snmp_host_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_host_get: getStringBuffer raised TypeError")
      return
    next_host_buff        = OpEN.open_buffdesc()
    next_host_buff.pstart = next_host_string
    next_host_buff.size   = max_host_len

    while OpEN.openapiSnmpHostGetNext(self.m_client, 
                                      host_buff, 
                                      port_p, 
                                      security_buff,
                                      model_p,
                                      level_p,
                                      type_p,
                                      timeout_p,
                                      retries_p,
                                      filter_buff, 
                                      next_host_buff) == OpEN.OPEN_E_NONE:

      if host == next_host_string.cast():
        found = True
        break

      try:
        security_string       = open_.getStringBuffer(max_security_len)
      except OpENBufferSizeError:
        print("test_snmp_host_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_host_get: getStringBuffer raised TypeError")
        return
      security_buff         = OpEN.open_buffdesc()
      security_buff.pstart  = security_string
      security_buff.size    = max_security_len

      try:
        filter_string         = open_.getStringBuffer(max_filter_len)
      except OpENBufferSizeError:
        print("test_snmp_host_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_host_get: getStringBuffer raised TypeError")
        return
      filter_buff           = OpEN.open_buffdesc()
      filter_buff.pstart    = filter_string
      filter_buff.size      = max_filter_len

      host_buff.pstart = next_host_buff.pstart
      host_buff.size   = next_host_buff.size
      next_host_buff.size = max_host_len

    print_sanity_results(OpEN.OPEN_E_NONE, found, 'openapiSnmpHostGetNext', host) 

    if found and verbose:
      print("  Address port   : %s %d" % (next_host_string.cast(), OpEN.uint32_tp_value(port_p)))
      print("  Security name  : ", security_string.cast())
      print("  Security model : ", OpEN.OPEN_SNMP_SECURITY_MODEL_tp_value(model_p))
      print("  Security level : ", OpEN.OPEN_SNMP_SECURITY_LEVEL_tp_value(level_p))
      print("  Notify type    : ", OpEN.OPEN_SNMP_NOTIFY_TYPE_tp_value(type_p))
      print("  Timeout        : ", OpEN.uint32_tp_value(timeout_p))
      print("  Retries        : ", OpEN.uint32_tp_value(retries_p))
      print("  Filter name    : ", filter_string.cast())

    OpEN.delete_uint32_tp(port_p)
    OpEN.delete_OPEN_SNMP_SECURITY_MODEL_tp(model_p)
    OpEN.delete_OPEN_SNMP_SECURITY_LEVEL_tp(level_p)
    OpEN.delete_OPEN_SNMP_NOTIFY_TYPE_tp(type_p)
    OpEN.delete_uint32_tp(timeout_p)
    OpEN.delete_uint32_tp(retries_p)

  def test_snmp_community_get(self, community_name) :
    """Get the specified SNMP community name

    Iterate through the existing SNMP hosts until a match for
    host is found. If found, display the associated host parameters."""

    max_community_len = 20+1
    max_view_len = 30+1
    access_p = OpEN.new_OPEN_SNMP_COMMUNITY_ACCESS_TYPE_tp()
    addr = OpEN.open_inet_addr_t()

    try:
      community_string       = open_.getStringBuffer(max_community_len)
    except OpENBufferSizeError:
      print("test_snmp_community_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_community_get: getStringBuffer raised TypeError")
      return
    community_buff         = OpEN.open_buffdesc()
    community_buff.pstart  = community_string
    community_buff.size    = max_community_len

    try:
      view_string            = open_.getStringBuffer(max_view_len)
    except OpENBufferSizeError:
      print("test_snmp_community_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_community_get: getStringBuffer raised TypeError")
      return
    view_buff              = OpEN.open_buffdesc()
    view_buff.pstart       = view_string
    view_buff.size         = max_view_len

    try:
      next_community_string      = open_.getStringBuffer(max_community_len)
    except OpENBufferSizeError:
      print("test_snmp_community_get: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_community_get: getStringBuffer raised TypeError")
      return
    next_community_buff        = OpEN.open_buffdesc()
    next_community_buff.pstart = next_community_string
    next_community_buff.size   = max_community_len

    while OpEN.openapiSnmpCommunityGetNext(self.m_client,
                                           community_buff,
                                           access_p,
                                           view_buff,
                                           addr,
                                           next_community_buff) == OpEN.OPEN_E_NONE:

      if community_name == next_community_string.cast():
        found = True
        break

      try:
        view_string      = open_.getStringBuffer(max_view_len)
      except OpENBufferSizeError:
        print("test_snmp_community_get: getStringBuffer raised OpENBufferSizeError")
        return
      except TypeError:
        print("test_snmp_community_get: getStringBuffer raised TypeError")
        return
      view_buff        = OpEN.open_buffdesc()
      view_buff.pstart = view_string
      view_buff.size   = max_view_len

      community_buff.pstart = next_community_buff.pstart
      community_buff.size   = next_community_buff.size
      next_community_buff.size = max_community_len

    print_sanity_results(OpEN.OPEN_E_NONE, found, 'openapiSnmpCommunityGetNext', community_name) 

    if found and verbose:
      print("  Community   :", next_community_string.cast())
      print("  Access Type :", OpEN.OPEN_SNMP_COMMUNITY_ACCESS_TYPE_tp_value(access_p))
      print("  View Name   :", view_string.cast())
      print("  IP Address  :", int_to_ip(addr.addr.ipv4))

    OpEN.delete_OPEN_SNMP_COMMUNITY_ACCESS_TYPE_tp(access_p)

  def test_snmp_trap_flag_get(self, flag_bits, mode) :
    """Retrieve the trap mode for a single trap."""

    mode_p = OpEN.new_bool_tp()

    result = OpEN.openapiSnmpTrapFlagGet(self.m_client, flag_bits, mode_p)
    print_bad_result(result, 'openapiSnmpTrapFlagGet')
    print_sanity_results(result, mode == OpEN.bool_tp_value(mode_p), 'openapiSnmpTrapFlagGet', str(flag_bits))
    OpEN.delete_bool_tp(mode_p)

  def test_snmp_local_engine_id_delete(self) :
    """Delete the SNMP engine id on the local device.

    The engine id is actually reset to the default id which is
    automatically generated based on the local MAC address."""

    result = OpEN.openapiSnmpLocalEngineIdDelete(self.m_client)
    print_bad_result(result, 'openapiSnmpLocalEngineIdDelete')
    print_sanity_results(OpEN.OPEN_E_NONE, True, 'openapiSnmpLocalEngineIdDelete', '') 

  def test_snmp_view_delete(self, view_name, oid_tree) :
    """Delete an existing SNMP view."""

    try:
      view_string      = open_.getStringBuffer(len(view_name) + 1,view_name)
    except OpENBufferSizeError:
      print("test_snmp_view_delete: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_view_delete: getStringBuffer raised TypeError")
      return
    view_buff        = OpEN.open_buffdesc()
    view_buff.pstart = view_string
    view_buff.size   = len(view_name) + 1

    try:
      oid_string       = open_.getStringBuffer(len(oid_tree) + 1,oid_tree)
    except OpENBufferSizeError:
      print("test_snmp_view_delete: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_view_delete: getStringBuffer raised TypeError")
      return
    oid_buff         = OpEN.open_buffdesc()
    oid_buff.pstart  = oid_string
    oid_buff.size    = len(oid_tree)

    result = OpEN.openapiSnmpViewDelete(self.m_client, view_buff, oid_buff)
    print_bad_result(result, 'openapiSnmpViewDelete')
    print_sanity_results(result, True, 'openapiSnmpViewDelete', view_name) 

  def test_snmp_filter_delete(self, filter_name, oid_tree) :
    """Delete an existing SNMP filter."""

    filter_string      = open_.getStringBuffer(len(filter_name) + 1,filter_name)
    filter_buff        = OpEN.open_buffdesc()
    filter_buff.pstart = filter_string
    filter_buff.size   = len(filter_name) + 1

    try:
      oid_string       = open_.getStringBuffer(len(oid_tree) + 1,oid_tree)
    except OpENBufferSizeError:
      print("test_snmp_filter_delete: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_filter_delete: getStringBuffer raised TypeError")
      return
    oid_buff         = OpEN.open_buffdesc()
    oid_buff.pstart  = oid_string
    oid_buff.size    = len(oid_tree) + 1

    result = OpEN.openapiSnmpFilterDelete(self.m_client, filter_buff, oid_buff)
    print_bad_result(result, 'openapiSnmpFilterDelete')
    print_sanity_results(result, True, 'openapiSnmpFilterDelete', filter_name) 

  def test_snmp_host_delete(self, host, notify_type) :
    """Delete an existing SNMP host."""

    try:
      host_string      = open_.getStringBuffer(len(host) + 1, host)
    except OpENBufferSizeError:
      print("test_snmp_host_delete: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_host_delete: getStringBuffer raised TypeError")
      return
    host_buff        = OpEN.open_buffdesc()
    host_buff.pstart = host_string
    host_buff.size   = len(host) + 1

    result = OpEN.openapiSnmpHostDelete(self.m_client, host_buff, notify_type)
    print_bad_result(result, 'openapiSnmpHostDelete')
    print_sanity_results(result, True, 'openapiSnmpHostDelete', host) 

  def test_snmp_user_delete(self, user_name, engine_id) :
    """Delete a SNMP user."""

    try:
      user_string           = open_.getStringBuffer(len(user_name) + 1, user_name)
    except OpENBufferSizeError:
      print("test_snmp_user_delete: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_user_delete: getStringBuffer raised TypeError")
      return
    user_buff             = OpEN.open_buffdesc()
    user_buff.pstart      = user_string
    user_buff.size        = len(user_name) + 1

    try:
      engine_id_string      = open_.getStringBuffer(len(engine_id) + 1, engine_id) 
    except OpENBufferSizeError:
      print("test_snmp_user_delete: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_user_delete: getStringBuffer raised TypeError")
      return
    engine_id_buff        = OpEN.open_buffdesc()
    engine_id_buff.pstart = engine_id_string
    engine_id_buff.size   = len(engine_id) + 1

    result = OpEN.openapiSnmpUserDelete(self.m_client, user_buff, engine_id_buff)
    print_bad_result(result, 'openapiSnmpUserDelete')
    print_sanity_results(result, True, 'openapiSnmpUserDelete', user_name) 

  def test_snmp_group_delete(self, group_name, model, level, context_name) :
    """Delete a SNMP group."""

    try:
      group_string           = open_.getStringBuffer(len(group_name) + 1, group_name)
    except OpENBufferSizeError:
      print("test_snmp_group_delete: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_group_delete: getStringBuffer raised TypeError")
      return
    group_buff             = OpEN.open_buffdesc()
    group_buff.pstart      = group_string
    group_buff.size        = len(group_name) + 1

    try:
      context_string           = open_.getStringBuffer(len(context_name) + 1, context_name)
    except OpENBufferSizeError:
      print("test_snmp_group_delete: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_group_delete: getStringBuffer raised TypeError")
      return
    context_buff             = OpEN.open_buffdesc()
    context_buff.pstart      = context_string
    context_buff.size        = len(context_name) + 1

    result = OpEN.openapiSnmpGroupDelete(self.m_client, group_buff, model, level, context_buff)
    print_bad_result(result, 'openapiSnmpGroupDelete')
    print_sanity_results(result, True, 'openapiSnmpGroupDelete', group_name) 

  def test_snmp_community_delete(self, community_name) :
    """Delete an SNMP community."""

    try:
      community_string           = open_.getStringBuffer(len(community_name) + 1, community_name)
    except OpENBufferSizeError:
      print("test_snmp_community_delete: getStringBuffer raised OpENBufferSizeError")
      return
    except TypeError:
      print("test_snmp_community_delete: getStringBuffer raised TypeError")
      return
    community_buff             = OpEN.open_buffdesc()
    community_buff.pstart      = community_string
    community_buff.size        = len(community_name) + 1

    result = OpEN.openapiSnmpCommunityDelete(self.m_client, community_buff)
    print_bad_result(result, 'openapiSnmpCommunityDelete')
    print_sanity_results(result, True, 'openapiSnmpCommunityDelete', community_name) 

  def test_snmp_trap_source_interface(self, interface_type) :
    """Set, then get the source interface which SNMP trap originates.

    For demonstration purposes, this function retrieves the first
    interface based on interface type and attempts to set it as the
    source interface. If successful, the interface attributes are
    retrieved and verified. """

    val_p   = OpEN.new_uint32_tp()
    intf_p  = OpEN.new_uint32_tp()
    type_p  = OpEN.new_OPEN_INTF_TYPE_tp()
    state_p = OpEN.new_OPEN_LINK_STATE_tp()

    msg = 'interface type : ' + get_type(interface_type)

    # retrieve first available interface for demonstration
    result = OpEN.openapiIfFirstGet(self.m_client, interface_type, val_p)
    if result == OpEN.OPEN_E_FAIL:
      result = OpEN.OPEN_E_NOT_FOUND
    print_bad_result(result, '%s - %s' % ('openapiIfFirstGet', msg))

    if result == OpEN.OPEN_E_NONE:
      intf = OpEN.uint32_tp_value(val_p)
      result = OpEN.openapiSnmpTrapSrcIntfSet(self.m_client, intf)
      print_bad_result(result, '%s - %s' % ('openapiSnmpTrapSrcIntfSet', msg))

    if result == OpEN.OPEN_E_NONE:
      tmp_addr = OpEN.open_inet_addr_t()
      result = OpEN.openapiSnmpTrapSrcIntfGet(self.m_client, intf_p, type_p, tmp_addr)
      print_bad_result(result, '%s - %s' % ('openapiSnmpTrapSrcIntfGet', msg))

    if result == OpEN.OPEN_E_NONE:
      if tmp_addr.family == OpEN.OPEN_AF_INET or tmp_addr.family == OpEN.OPEN_AF_UNIX:
        ip_addr = int_to_ip(tmp_addr.addr.ipv4)
      else:
        result = OpEN.OPEN_E_PARAM
      print_bad_result(result, '%s - %s' % ('inet_ntoa', msg))

    if result == OpEN.OPEN_E_NONE:
      result = OpEN.openapiIfLinkStateGet(self.m_client, intf, state_p)
      print_bad_result(result, '%s - %s' % ('openapiIfLinkStateGet', msg))

    if result == OpEN.OPEN_E_NONE:
      print_sanity_results(result, True, 'openapiSnmpTrapSrcIntfSet', msg)
      if verbose:
        print("  Interface : ", intf)
        print("  Type      : ", get_type(interface_type))
        print("  Address   : ", ip_addr)
        print("  State     : ", get_link(OpEN.OPEN_LINK_STATE_tp_value(state_p)))

    OpEN.delete_uint32_tp(val_p)
    OpEN.delete_uint32_tp(intf_p)
    OpEN.delete_OPEN_INTF_TYPE_tp(type_p)
    OpEN.delete_OPEN_LINK_STATE_tp(state_p)

def main(argv):
  """Demonstrate OpEN usage for SNMP APIs"""

  try:
    opts, args = getopt.getopt(argv,"hv")
  except getopt.GetoptError:
    print("snmp_example.py <-v>")
    sys.exit(2)
  for opt, arg in opts:
    if opt == '-h':
      print("snmp_example.py <-v>")
      sys.exit()
    elif opt == '-v':
      global verbose
      verbose = True

  ret = open_.connect("snmp_example")
  if ret == OpEN.OPEN_E_NONE :
    open_.getNetworkOSVersion()
    client = open_.get_client()
    example = SnmpExample(client)
    print("Begin Sanity tests...")
    example.test_snmp_sys_name_set('system-name')
    example.test_snmp_sys_location_set("system-location")
    example.test_snmp_sys_contact_set("system-contact")
    example.test_snmp_local_engine_id_set("default")
    print("\nCreate SNMP users...")
    example.test_snmp_user_create(SNMP_USER_LIST)
    print("\nCreate SNMP groups...")
    example.test_snmp_group_create(SNMP_GROUP_LIST)
    print("\nCreate SNMP views...")
    example.test_snmp_view_create(SNMP_VIEW_LIST)
    print("\nCreate SNMP filters...")
    example.test_snmp_filter_create(SNMP_FILTER_LIST)
    print("\nCreate SNMP hosts...")
    example.test_snmp_host_create(SNMP_HOST_LIST)
    print("\nCreate SNMP communities...")
    example.test_snmp_community_create(SNMP_COMMUNITY_LIST)
    print("\nEnable misc SNMP trap flags...")
    example.testSnmpTrapFlagSet([OpEN.OPEN_SNMP_MULTI_USERS_TRAP_FLAG, OpEN.OPEN_SNMP_USER_AUTH_TRAP_FLAG], True)
    print("\nRetrieve some SNMP records...")
    example.test_snmp_sys_name_get()
    example.test_snmp_sys_location_get()
    example.test_snmp_sys_contact_get()
    example.test_snmp_local_engine_id_get()
    example.test_snmp_user_get('admin1')
    example.test_snmp_group_get('network-admin', OpEN.OPEN_SNMP_SECURITY_MODEL_USM, OpEN.OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV, 'context-1')
    example.test_snmp_view_get('view-1', 'broadcom')
    example.test_snmp_view_get('view-2', 'ifEntry.*.1')
    example.test_snmp_filter_get('filter-1', 'broadcom')
    example.test_snmp_filter_get('filter-2', 'ifEntry.*.1')
    example.test_snmp_host_get('2001::')
    example.test_snmp_community_get('public-test')
    example.test_snmp_community_get('private-test')
    example.test_snmp_community_get('delete-private-map')
    example.test_snmp_trap_flag_get(OpEN.OPEN_SNMP_MULTI_USERS_TRAP_FLAG, True)
    example.test_snmp_trap_flag_get(OpEN.OPEN_SNMP_USER_AUTH_TRAP_FLAG, True)
    print("\nDelete some SNMP records...")
    example.test_snmp_local_engine_id_delete()
    example.test_snmp_view_delete("view-2", "ifEntry.*.1")
    example.test_snmp_filter_delete("filter-2", "ifEntry.*.1")
    example.test_snmp_host_delete("10.10.10.2", OpEN.OPEN_SNMP_NOTIFY_TYPE_TRAP)
    example.test_snmp_user_delete("delete-user", "dddddddddddddddddddddddddddddddd")
    example.test_snmp_group_delete("delete-group", OpEN.OPEN_SNMP_SECURITY_MODEL_SNMPV2C, OpEN.OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV, "context-3")
    example.test_snmp_community_delete("delete-community")
    example.test_snmp_community_delete("delete-private-map")
    print("\nAttempt to set the trap source-interface...")
    # Routing must be enabled on first interface
    example.test_snmp_trap_source_interface(OpEN.OPEN_INTF_TYPE_PHY)

    open_.terminate()
  else :
    print("Unable to connect")

if __name__ == '__main__':
  verbose = False
  main(sys.argv[1:])


