/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  telnet_example.c
*
* @purpose   OpEN mgmttelnet example.
*
* @component OpEN
*
* @create    02/06/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_mgmttelnet.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the mgmttelnet Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Sets admin mode of telnet Session: %s 1 <mode>\n", name);
  printf("Test 2: Gets admin mode of telnet session: %s 2\n", name);
  printf("Test 3: Sets the port-number for Telnet Access: %s 3 <port><flag>\n", name);
  printf("Test 4: Gets telnet access port number: %s 4\n", name);
  printf("Test 5: Sets maximum number of telnet sessions: %s 5 <maxSession>\n", name);
  printf("Test 6: Gets maximum number of telnet sessions: %s 6\n", name);
  printf("Test 7: Sets  telnet session timeout (in minutes): %s 7 <timeout>\n", name);
  printf("Test 8: Gets telnet session timeout information: %s 8\n", name);
  printf("Test 9: Sets new telnet sessions acceptance state: %s 9 <state>\n", name);
  printf("Test 10: Gets new telnet session acceptance state: %s 10\n", name);
  printf("Test 11: Gets current active number of telnet sessions: %s 11\n", name);
  printf("Test 12: Run API sanity checks: %s 12 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  OPEN_CONTROL_t state;
  uint32_t timeout;
  uint32_t session;
  uint32_t port;

  printf("Testing mgmttelnet OpEN APIs sanity:\n");

  printf("Testing openapiAgentTelnetAdminModeGet():\n");
  result = openapiAgentTelnetAdminModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAgentTelnetAdminModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSwDevCtrlTelnetMgmtPortNumGet():\n");
  result = openapiSwDevCtrlTelnetMgmtPortNumGet(NULL, &port);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSwDevCtrlTelnetMgmtPortNumGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiAgentTelnetNumSessionsGet():\n");
  result = openapiAgentTelnetNumSessionsGet(NULL, &session);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAgentTelnetNumSessionsGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiAgentTelnetTimeoutGet():\n");
  result = openapiAgentTelnetTimeoutGet(NULL, &timeout);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAgentTelnetTimeoutGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiAgentTelnetNewSessionsGet():\n");
  result = openapiAgentTelnetNewSessionsGet(NULL, &state);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAgentTelnetNewSessionsGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiAgentTelnetCurrentNumSessionsGet():\n");
  result = openapiAgentTelnetCurrentNumSessionsGet(NULL, &session);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiAgentTelnetCurrentNumSessionsGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Sets admin mode of telnet Session.
*
* @param    clientHandle @b{(input)}  client handle from registration API
* @param    mode          @b{(input)}  Telnet Access Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void agentTelnetAdminModeSet(openapiClientHandle_t *clientHandle,
                             OPEN_CONTROL_t mode)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiAgentTelnetAdminModeSet(clientHandle, mode)))
  {
    printf("Bad return code trying to sets admin mode of telnet Session."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Telnet access mode is successfully %s\n", 
          (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets admin mode of telnet session.
*
* @param    clientHandle @b{(input)}  client handle from registration API
* @param    pMode         @b{(output)}  Telnet access Mode [Enable/Disable]
*
*
* @returns  none
*
* @end
*********************************************************************/
void agentTelnetAdminModeGet(openapiClientHandle_t *clientHandle,
                             OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiAgentTelnetAdminModeGet(clientHandle, pMode)))
  {
    printf("Bad return code trying to gets admin mode of telnet session."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Admin mode of telnet session access is in %s state\n",
           (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets the port-number for Telnet Access.
*
* @param    clientHandle @b{(input)}  client handle from registration API
* @param    port          @b{(input)}  Telnet Access port-number
* @param    flag          @b{(input)}  Flag indicates if the port number needs
*                                      to be applied immediately
*
*
* @returns  none
*
* @end
*********************************************************************/
void swDevCtrlTelnetMgmtPortNumSet(openapiClientHandle_t *clientHandle,
                                   uint32_t port, OPEN_BOOL_t flag)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSwDevCtrlTelnetMgmtPortNumSet(clientHandle, port, flag)))
  {
    printf("Bad return code trying to sets the port-number for Telnet Access."
           " (result = %d)\n", result);
  }
  else
  {
    printf("port-number for telnet access is succesfully set\n");
  }
  return;
}

/*********************************************************************
* @purpose  Gets telnet access port number.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pPort       @b{(output)}  telnet access port number

*
* @returns  none
*
* @end
*********************************************************************/
void swDevCtrlTelnetMgmtPortNumGet(openapiClientHandle_t *clientHandle, uint32_t *pPort)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSwDevCtrlTelnetMgmtPortNumGet(clientHandle, pPort)))
  {
    printf("Bad return code trying to gets telnet access port number. (result = %d)\n", result);
  }
  else
  {
    printf("Telnet access port number is %d\n", *pPort);
  }
  return;
}


/*********************************************************************
* @purpose  Sets maximum number of telnet sessions.
*
* @param   clientHandle @b{(input)} client handle from registration API
* @param   maxSession    @b{(input)} maximum allowable number of telnet sessions
*
*
* @returns  none
*
* @end
*********************************************************************/
void agentTelnetNumSessionsSet(openapiClientHandle_t *clientHandle,
                               uint32_t maxSession)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiAgentTelnetNumSessionsSet(clientHandle, maxSession)))
  {
    printf("Bad return code trying to sets maximum number of telnet sessions."
           " (result = %d)\n", result);
  }
  else
  {
    printf("maximum number of telnet sessions is succesfully set to %d\n",
           maxSession);
  }
  return;
}


/*********************************************************************
* @purpose  Gets maximum number of telnet sessions.
*
* @param    clientHandle @b{(input)}   client handle from registration API
* @param    pSession      @b{(output)}  maximum allowable number of configured
*                                       telnet sessions
*
*
* @returns  none
*
* @end
*********************************************************************/
void agentTelnetNumSessionsGet(openapiClientHandle_t *clientHandle,
                               uint32_t *pSession)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiAgentTelnetNumSessionsGet(clientHandle, pSession)))
  {
    printf("Bad return code trying to gets maximum number of telnet sessions."
           " (result = %d)\n", result);
  }
  else
  {
    printf("maximum number of telnet sessions configured is %d\n", *pSession);
  }
  return;
}


/*********************************************************************
* @purpose  Sets  telnet session timeout (in minutes).
*
* @param    clientHandle @b{(input)}  client handle from registration API
* @param    timeout       @b{(input)}  telnet session timeout
*
*
* @returns  none
*
* @end
*********************************************************************/
void agentTelnetTimeoutSet(openapiClientHandle_t *clientHandle, uint32_t timeout)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiAgentTelnetTimeoutSet(clientHandle, timeout)))
  {
    printf("Bad return code trying to sets  telnet session timeout (in minutes)."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Telnet session timeout is succesfully set to %d\n", timeout);
  }
  return;
}


/*********************************************************************
* @purpose  Gets telnet session timeout information.
*
* @param    clientHandle  @b{(input)}  client handle from registration API
* @param    pTimeout       @b{(output)}  Telnet session timeout

*
* @returns  none
*
* @end
*********************************************************************/
void agentTelnetTimeoutGet(openapiClientHandle_t *clientHandle,
                           uint32_t *pTimeout)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiAgentTelnetTimeoutGet(clientHandle, pTimeout)))
  {
    printf("Bad return code trying to gets telnet session timeout information."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Telnet session timeout is %d\n", *pTimeout);
  }
  return;
}


/*********************************************************************
* @purpose  Sets new telnet sessions acceptance state.
*
* @param    clientHandle @b{(input)}  client handle from registration API
* @param    state         @b{(input)}  Acceptance state of new telnet 
*                                      sessions[Enable/Disable]
*
*
* @returns  none
*
* @end
*********************************************************************/
void agentTelnetNewSessionsSet(openapiClientHandle_t *clientHandle,
                               OPEN_CONTROL_t state)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiAgentTelnetNewSessionsSet(clientHandle, state)))
  {
    printf("Bad return code trying to sets new telnet sessions acceptance state."
           " (result = %d)\n", result);
  }
  else
  {
    printf("New telnet sessions acceptance state is successfully %s\n",
          (OPEN_ENABLE == state) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets new telnet session acceptance state.
*
* @param    clientHandle @b{(input)}  client handle from registration API
* @param    pState        @b{(output)}  Telnet access Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void agentTelnetNewSessionsGet(openapiClientHandle_t *clientHandle,
                               OPEN_CONTROL_t *pState)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiAgentTelnetNewSessionsGet(clientHandle, pState)))
  {
    printf("Bad return code trying to gets new telnet session acceptance state."
           " (result = %d)\n", result);
  }
  else
  {
    printf("New telnet sessions acceptance is in %s state,\n",
        (OPEN_ENABLE == *pState) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets current active number of telnet sessions.
*
* @param    clientHandle  @b{(input)}  client handle from registration API
* @param    pSession       @b{(output)}  current active telnet sessions count
*
*
* @returns  none
*
* @end
*********************************************************************/
void agentTelnetCurrentNumSessionsGet(openapiClientHandle_t *clientHandle,
                                      uint32_t *pSession)
{
  open_error_t result;

  if (OPEN_E_NONE != 
      (result = openapiAgentTelnetCurrentNumSessionsGet(clientHandle, pSession)))
  {
    printf("Bad return code trying to gets current active number of telnet sessions."
           " (result = %d)\n", result);
  }
  else
  {
    printf("maximum number of current active telnet sessions are %d\n", *pSession);
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for mgmttelnet
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_CONTROL_t mode;
  OPEN_CONTROL_t state;
  uint32_t session;
  uint32_t port;
  uint32_t timeout;
  OPEN_BOOL_t flag;
  uint32_t recvMode;
  uint32_t recvState;
  uint32_t recvFlag;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("mgmttelnet example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting mgmttelnet API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = (1 == recvMode) ? OPEN_ENABLE : OPEN_DISABLE;
          agentTelnetAdminModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 2)
      {
        agentTelnetAdminModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 4)
      {
        port = atoi(argv[2]);
        recvFlag = atoi(argv[3]);
        if ((0 == recvFlag) || (1 == recvFlag))
        {
          flag = (1 == recvFlag) ? OPEN_TRUE :  OPEN_FALSE;
          swDevCtrlTelnetMgmtPortNumSet(&clientHandle, port, flag);
        }
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 2)
      {
        swDevCtrlTelnetMgmtPortNumGet(&clientHandle, &port);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        session = atoi(argv[2]);
        agentTelnetNumSessionsSet(&clientHandle, session);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 2)
      {
        agentTelnetNumSessionsGet(&clientHandle, &session);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        timeout = atoi(argv[2]);
        agentTelnetTimeoutSet(&clientHandle, timeout);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 2)
      {
        agentTelnetTimeoutGet(&clientHandle, &timeout);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        recvState = atoi(argv[2]);
        if ((0 == recvState) || (1 == recvState))
        {
          state = (1 == recvState) ? OPEN_ENABLE : OPEN_DISABLE;
          agentTelnetNewSessionsSet(&clientHandle, state);
        }
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 2)
      {
        agentTelnetNewSessionsGet(&clientHandle, &state);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 2)
      {
        agentTelnetCurrentNumSessionsGet(&clientHandle, &session);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping mgmttelnet API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
