/*********************************************************************
*
*  Copyright 2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  webmgmt_example.c
*
* @purpose   OpEN WebMgmt example.
*
* @component OpEN
*
* @create    01/31/2023
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_webmgmt.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the WebMgmt Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Sets System Web Mode: %s 1 <mode>\n", name);
  printf("Test 2: Gets admin mode of System Web: %s 2\n", name);
  printf("Test 3: Sets the port-number for HTTP Access: %s 3 <port><flag>\n", name);
  printf("Test 4: Gets HTTP access port number: %s 4 \n", name);
  printf("Test 5: Sets maximum number of HTTP web sessions: %s 5 <maxSession>\n", name);
  printf("Test 6: Gets  maximum number of HTTP web sessions: %s 6\n", name);
  printf("Test 7: Sets  HTTP session soft timeout (in minutes): %s 7 <timeout>\n", name);
  printf("Test 8: Gets HTTP session soft timeout information: %s 8\n", name);
  printf("Test 9: Sets  HTTP session hard timeout (in hours): %s 9 <timeout>\n", name);
  printf("Test 10: Gets HTTP session hard timeout information (in hours): %s 10\n", name);
  printf("Test 11: Run API sanity checks: %s 11 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  OPEN_CONTROL_t pMode;
  uint32_t pSession;
  uint32_t pPort;
  uint32_t pTimeout;

  printf("Testing WebMgmt OpEN APIs sanity:\n");
  printf("Testing openapiSwDevCtrlWebMgmtModeGet():\n");
  result = openapiSwDevCtrlWebMgmtModeGet(NULL, &pMode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSwDevCtrlWebMgmtModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiSwDevCtrlWebMgmtPortNumGet():\n");
  result = openapiSwDevCtrlWebMgmtPortNumGet(NULL, &pPort);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiSwDevCtrlWebMgmtPortNumGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiCliWebHttpNumSessionsGet():\n");
  result = openapiCliWebHttpNumSessionsGet(NULL, &pSession);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCliWebHttpNumSessionsGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiCliWebHttpSessionSoftTimeOutGet():\n");
  result = openapiCliWebHttpSessionSoftTimeOutGet(NULL, &pTimeout);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCliWebHttpSessionSoftTimeOutGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiCliWebHttpSessionHardTimeOutGet():\n");
  result = openapiCliWebHttpSessionHardTimeOutGet(NULL, &pTimeout);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiCliWebHttpSessionHardTimeOutGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Sets System Web Mode.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Web Access Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void swDevCtrlWebMgmtModeSet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSwDevCtrlWebMgmtModeSet(clientHandle, mode)))
  {
    printf("Bad return code trying to sets System Web Mode. (result = %d)\n", result);
  }
  else
  {
    printf("System Web Mode is successfully %s\n", (OPEN_ENABLE == mode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Gets admin mode of System Web.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pMode       @b{(output)}  Web Access Mode [Enable/Disable]

*
* @returns  none
*
* @end
*********************************************************************/
void swDevCtrlWebMgmtModeGet(openapiClientHandle_t *clientHandle, OPEN_CONTROL_t *pMode)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSwDevCtrlWebMgmtModeGet(clientHandle, pMode)))
  {
    printf("Bad return code trying to gets admin mode of System Web. (result = %d)\n", result);
  }
  else
  {
    printf("Admin mode of System Web is in %s state\n", (OPEN_ENABLE == *pMode) ? "enabled" : "disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Sets the port-number for HTTP Access.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    port       @b{(input)}  HTTP Access port-number
* @param    flag       @b{(input)}  Flag indicates if the port number needs to be applied immediately

*
* @returns  none
*
* @end
*********************************************************************/
void swDevCtrlWebMgmtPortNumSet(openapiClientHandle_t *clientHandle, uint32_t port, OPEN_BOOL_t flag)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSwDevCtrlWebMgmtPortNumSet(clientHandle, port, flag)))
  {
    printf("Bad return code trying to sets the port-number for HTTP Access. (result = %d)\n", result);
  }
  else
  {
    printf("port-number for HTTP Access is succesfully set\n");
  }
  return;
}


/*********************************************************************
* @purpose  Gets HTTP access port number.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pPort       @b{(output)}   HTTP access port number

*
* @returns  none
*
* @end
*********************************************************************/
void swDevCtrlWebMgmtPortNumGet(openapiClientHandle_t *clientHandle, uint32_t *pPort)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiSwDevCtrlWebMgmtPortNumGet(clientHandle, pPort)))
  {
    printf("Bad return code trying to gets HTTP access port number. (result = %d)\n", result);
  }
  else
  {
    printf("HTTP access port number is %d\n", *pPort);
  }
  return;
}


/*********************************************************************
* @purpose  Sets maximum number of HTTP web sessions.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    maxSession       @b{(input)}  maximum allowable number of HTTP web sessions

*
* @returns  none
*
* @end
*********************************************************************/
void cliWebHttpNumSessionsSet(openapiClientHandle_t *clientHandle, uint32_t maxSession)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiCliWebHttpNumSessionsSet(clientHandle, maxSession)))
  {
    printf("Bad return code trying to sets maximum number of HTTP web sessions. (result = %d)\n", result);
  }
  else
  {
    printf("maximum number of HTTP web sessions is succesfully set to %d\n", maxSession);
  }
  return;
}


/*********************************************************************
* @purpose  Gets  maximum number of HTTP web sessions.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pSession       @b{(output)}  maximum allowable number of web sessions configured

*
* @returns  none
*
* @end
*********************************************************************/
void cliWebHttpNumSessionsGet(openapiClientHandle_t *clientHandle, uint32_t *pSession)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiCliWebHttpNumSessionsGet(clientHandle, pSession)))
  {
    printf("Bad return code trying to gets  maximum number of HTTP web sessions. (result = %d)\n", result);
  }
  else
  {
    printf("maximum number of HTTP web sessions is %d\n", *pSession);
  }
  return;
}


/*********************************************************************
* @purpose  Sets  HTTP session soft timeout (in minutes).
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    timeout       @b{(input)}  HTTP session soft timeout

*
* @returns  none
*
* @end
*********************************************************************/
void cliWebHttpSessionSoftTimeOutSet(openapiClientHandle_t *clientHandle, uint32_t timeout)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiCliWebHttpSessionSoftTimeOutSet(clientHandle, timeout)))
  {
    printf("Bad return code trying to sets  HTTP session soft timeout (in minutes). (result = %d)\n", result);
  }
  else
  {
    printf("HTTP session soft timeout is succesfully set to %d\n", timeout);
  }
  return;
}


/*********************************************************************
* @purpose  Gets HTTP session soft timeout information.
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pTimeout       @b{(output)}  HTTP session soft timeout

*
* @returns  none
*
* @end
*********************************************************************/
void cliWebHttpSessionSoftTimeOutGet(openapiClientHandle_t *clientHandle, uint32_t *pTimeout)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiCliWebHttpSessionSoftTimeOutGet(clientHandle, pTimeout)))
  {
    printf("Bad return code trying to gets HTTP session soft timeout information. (result = %d)\n", result);
  }
  else
  {
    printf("HTTP session soft timeout is %d\n", *pTimeout);
  }
  return;
}


/*********************************************************************
* @purpose  Sets  HTTP session hard timeout (in hours).
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    timeout       @b{(input)}  HTTP session hard timeout

*
* @returns  none
*
* @end
*********************************************************************/
void cliWebHttpSessionHardTimeOutSet(openapiClientHandle_t *clientHandle, uint32_t timeout)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiCliWebHttpSessionHardTimeOutSet(clientHandle, timeout)))
  {
    printf("Bad return code trying to sets  HTTP session hard timeout (in hours). (result = %d)\n", result);
  }
  else
  {
    printf("HTTP session hard timeout is succesfully set to %d\n", timeout);
  }
  return;
}


/*********************************************************************
* @purpose  Gets HTTP session hard timeout information (in hours).
*
* @param    clientHandle       @b{(input)}  client handle from registration API
* @param    pTimeout       @b{(output)}  HTTP session hard timeout

*
* @returns  none
*
* @end
*********************************************************************/
void cliWebHttpSessionHardTimeOutGet(openapiClientHandle_t *clientHandle, uint32_t *pTimeout)
{
  open_error_t result;

  if (OPEN_E_NONE != (result = openapiCliWebHttpSessionHardTimeOutGet(clientHandle, pTimeout)))
  {
    printf("Bad return code trying to gets HTTP session hard timeout information (in hours). (result = %d)\n", result);
  }
  else
  {
    printf("HTTP session hard timeout is %d\n", *pTimeout);
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for WebMgmt
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_CONTROL_t mode;
  uint32_t session;
  uint32_t port;
  uint32_t timeout;
  OPEN_BOOL_t flag;
  uint32_t recvMode;
  uint32_t recvFlag;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if (OPEN_E_NONE != (result = openapiClientRegister("WebMgmt example", &clientHandle)))
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (OPEN_E_NONE != openapiConnectivityCheck(&clientHandle))
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting WebMgmt API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (OPEN_E_NONE == openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision))
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        recvMode = atoi(argv[2]);
        if ((0 == recvMode) || (1 == recvMode))
        {
          mode = (1 == recvMode) ? OPEN_ENABLE : OPEN_DISABLE;
          swDevCtrlWebMgmtModeSet(&clientHandle, mode);
        }
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 2)
      {
        swDevCtrlWebMgmtModeGet(&clientHandle, &mode);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 4)
      {
        port = atoi(argv[2]);
        recvFlag = atoi(argv[3]);
        if ((0 == recvFlag) || (1 == recvFlag))
        {
          flag = (1 == recvFlag) ? OPEN_TRUE :  OPEN_FALSE;
          swDevCtrlWebMgmtPortNumSet(&clientHandle, port, flag);
        }
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 2)
      {
        swDevCtrlWebMgmtPortNumGet(&clientHandle, &port);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
        session = atoi(argv[2]);
        cliWebHttpNumSessionsSet(&clientHandle, session);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 2)
      {
        cliWebHttpNumSessionsGet(&clientHandle, &session);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
        timeout = atoi(argv[2]);
        cliWebHttpSessionSoftTimeOutSet(&clientHandle, timeout);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 2)
      {
        cliWebHttpSessionSoftTimeOutGet(&clientHandle, &timeout);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
        timeout = atoi(argv[2]);
        cliWebHttpSessionHardTimeOutSet(&clientHandle, timeout);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 2)
      {
        cliWebHttpSessionHardTimeOutGet(&clientHandle, &timeout);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping WebMgmt API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

