/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  accounting_example.c
*
* @purpose   Accounting OpEN APIs Example.
*
* @component OpEN
*
* @comments
*
* @create    15/03/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_user_manager.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the Accounting Example 
*         Application Menu.
*
* @param    none
*
* @returns  none
*
*********************************************************************/
void printAccountingAppMenu()
{
  printf("Usage: accounting_example <test#> <arg1> <arg2> ... \n");
  printf("Test 1: Create accounting list name: accounting_example 1 <accounting-type> <record-type> <list-name>\n");
  printf("Test 2: Add methods to accounting list: accounting_example 2 <accounting-type> <list-name> <method1> <method2> <method3> .... \n");
  printf("Test 3: Get accounting list names and configured methods of an access acctType: accounting_example 3 <accounting-type>\n");
  printf("Test 4: Set accounting list to an access line: accounting_example 4 <access-line> <accounting-type> <list-name> \n");
  printf("Test 5: Get accounting list name applied to access line: accounting_example 5 <access-line> <accounting-type> \n");
  printf("Test 6: Delete accounting list applied to the access-line: accounting_example 6 <access-line> <accounting-type> \n");
  printf("Test 7: Delete accounting list created: accounting_example 7 <accounting-type> <list-name> \n");
  printf("Test 8: Accounting OpEN APIs sanity: accounting_example 8 \n");

  return;
}

/*******************************************************************
* @purpose  Create accounting list for an accounting type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    acctType       @b{(input)} Accounting type 
* @param    recordType     @b{(input)} Record type
* @param    listName       @b{(input)} Accounting List name.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void accountingListCreate(openapiClientHandle_t *clientHandle, OPEN_USER_MGR_ACCT_TYPES_t acctType,
                          OPEN_USER_MGR_RECORD_TYPES_t recordType, char *listName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str)+1;

  result = openapiAccountingListNameValidate(clientHandle, &buffDesc);

  if (result == OPEN_E_NONE)
  {
    if ((result = openapiAccountingListCreate(clientHandle, acctType, recordType, &buffDesc)) != OPEN_E_NONE)
    {
      printf("Bad return code trying to create accounting list name for acctType %d. (result = %d) \n", acctType, result);
    }
    else
    {
      printf("Accounting list name %s created successfully. \n", str);
    }
  }
  else
  {
    printf("Accounting list name is too long or contains invalid characters. \n");
  }
  return;
}

/*******************************************************************
* @purpose  Add methods to accounting list for an accounting type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    acctType       @b{(input)} Accounting type 
* @param    listName       @b{(input)} Accounting List name.
* @param    acctMethods    @b{(input)} Accounting methods
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void accountingMethodsAdd(openapiClientHandle_t *clientHandle, OPEN_USER_MGR_ACCT_TYPES_t acctType, 
                          char *listName, open_buffdesc *acctMethods)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str)+1;


  if ((result = openapiAccountingMethodsAdd(clientHandle, acctType, &buffDesc, acctMethods)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to add methods to accounting list for acctType %d. (result = %d) \n", acctType, result);
  }
  else
  {
    printf("Methods added to accounting list successfully. \n");
  }
  return;    
}

/*******************************************************************
* @purpose  Get Accounting list information for an accounting type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    acctType       @b{(input)} Accounting type 
*
* @returns  none
* 
* @notes  
*
* @end
*********************************************************************/
void accountingInfoGet(openapiClientHandle_t *clientHandle, OPEN_USER_MGR_ACCT_TYPES_t acctType)
{
  open_error_t result;
  uint32_t acctListNameSize;
  uint32_t maxAcctMethods;
  uint32_t i = 0;
  char *str;
  char *methodStr;
  OPEN_USER_MGR_RECORD_TYPES_t recordType = OPEN_ACCT_RECORD_TYPE_UNDEFINED;
  open_buffdesc listName;
  open_buffdesc methodList;

  if ((result = openapiAccountingListNameSizeGet(clientHandle, &acctListNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get accounting list name size. (result = %d)", result);
    return;
  }

  if ((result = openapiAccountingMethodsMaxGet(clientHandle, &maxAcctMethods)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get maximum accounting methods supported. (result = %d) \n", result);
  }

  if ((str = (char *)malloc(acctListNameSize)) == NULL)
  {
    printf("Could not allocate memory.\n");
    return;
  }

  if ((methodStr = (char *)malloc(maxAcctMethods)) == NULL)
  {
    printf("Could not allocate memory.\n");
    free(str);
    return;
  }

  memset(str, 0, acctListNameSize);
  listName.pstart = str;
  listName.size = acctListNameSize;

  memset(methodStr, 0, maxAcctMethods);
  methodList.pstart = methodStr;
  methodList.size = maxAcctMethods;

  printf("Getting the first accounting list name. \n");
  if ((result = openapiAccountingListFirstGet(clientHandle, acctType, &listName)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get first accounting list name for acctType: %d (result = %d) \n", acctType, result);
    free(methodStr);
    free(str);
    return;
  }

  printf("Acct_list_name   Record Type   Methods\n");
  do
  {
    printf("%-15s  ", str);

    if (openapiAccountingListRecordTypeGet(clientHandle, acctType, &listName, &recordType) == OPEN_E_NONE)
    {
      switch (recordType)
      {
        case OPEN_ACCT_RECORD_TYPE_START_STOP:
          printf("start-stop   ");
          break;
        case OPEN_ACCT_RECORD_TYPE_STOP_ONLY:
          printf("stop-only    ");
          break;
        case OPEN_ACCT_RECORD_TYPE_NONE:
          printf("None         ");
          break;
        default:
          printf("             ");
          break;
      }
    }
    else
    {
      printf("Record Type get failed. (result = %d)", result);
    }

    if (openapiAccountingMethodListGet(clientHandle, acctType, &listName, &methodList) == OPEN_E_NONE)
    {
      for (i = 0; i < methodList.size; i++)
      {
        switch ((methodStr[i]-'0'))
        {
          case OPEN_ACCT_METHOD_TACACS:
            printf("Tacacs ");
            break;
          case OPEN_ACCT_METHOD_RADIUS:
            printf("Radius ");
            break;
          default:
            break;
        }
      }
    }
    else
    {
      printf("Method list get failed. (result = %d) \n", result);
    }
    printf("\n");
    listName.size = acctListNameSize;
    methodList.size = maxAcctMethods;
  }while(openapiAccountingListNextGet(clientHandle, acctType, &listName, &listName) == OPEN_E_NONE);

  free(str);
  free(methodStr);
  return;
}

/*******************************************************************
* @purpose  Apply accounting list to an access line for an accounting type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    accessLine     @b{(input)} Access line
* @param    acctType       @b{(input)} Accounting type 
* @param    listName       @b{(input)} Accounting List name.
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void accountingListLineApply(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LINE_t accessLine,
                             OPEN_USER_MGR_ACCT_TYPES_t acctType, char *listName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];
  
  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str; 
  buffDesc.size = strlen(str)+1;
  
  if ((result = openapiAccountingListLineSet(clientHandle, accessLine, acctType, &buffDesc)) != OPEN_E_NONE)
  { 
    printf("Bad return code trying to apply accounting list name to access line %d. (result = %d) \n", accessLine, result);
  }
  else
  { 
    printf("Accounting list name applied to access line successfully \n");
  }
  
  return; 
}

/*******************************************************************
* @purpose  Get accounting list applied to an access line for an accounting type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    accessLine     @b{(input)} Access line
* @param    acctType       @b{(input)} Accounting type 
*
* @returns  none
* 
* @notes  
* @end
*********************************************************************/
void accountingListLineGet(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LINE_t accessLine,
                           OPEN_USER_MGR_ACCT_TYPES_t acctType)
{
  open_error_t result;
  char *str;
  open_buffdesc listName;
  uint32_t acctListNameSize;
  
  if ((result = openapiAccountingListNameSizeGet(clientHandle, &acctListNameSize)) != OPEN_E_NONE)
  { 
    printf("Bad return code trying to get accounting list name size. (result = %d)", result);
    return;
  }
  if ((str = (char *)malloc(acctListNameSize)) == NULL)
  { 
    printf("Could not allocate memory.\n");
    return;
  }
  
  memset(str, 0, acctListNameSize);
  listName.pstart = str;
  listName.size = acctListNameSize;
  
  printf(" Getting the accounting list name applied to access line %d\n", accessLine);
  if ((result = openapiAccountingListLineGet(clientHandle, accessLine, acctType, &listName)) != OPEN_E_NONE)
  { 
    printf("Bad return code while getting accounting list name applied to access line. (result = %d) \n", result);
  }
  else
  {
    printf("Accounting list name: %s \n", str);
  }
  free(str);
  return;
}

/*******************************************************************
* @purpose  Delete accounting list applied to an access line for an accounting type.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    accessLine     @b{(input)} Access line
* @param    acctType       @b{(input)} Accounting type 
*
* @returns  none
* 
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void accountingListLineDelete(openapiClientHandle_t *clientHandle, OPEN_ACCESS_LINE_t accessLine,
                              OPEN_USER_MGR_ACCT_TYPES_t acctType)
{
  open_error_t result;

  printf("Deleting accounting list applied to access line %d. \n", accessLine);

  if ((result = openapiAccountingListLineDelete(clientHandle, accessLine, acctType)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete accounting list name applied to access line. (result = %d) \n", result);
  }
  else
  {
    printf("Deleted the accounting list name from the access line %d successfully. \n", accessLine);
  }
  return;
}

/*******************************************************************
* @purpose  Delete accounting list applied of an accounting type.
* 
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    acctType       @b{(input)} Accounting type 
* @param    listName       @b{(input)} Accounting list name
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
void accountingListDelete(openapiClientHandle_t *clientHandle, OPEN_USER_MGR_ACCT_TYPES_t acctType, char *listName)
{
  open_error_t result;
  open_buffdesc buffDesc;
  char str[100];

  printf("Deleting the accounting list %s \n", listName);
  memset(str, 0, sizeof(str));
  strncpy(str, listName, (sizeof(str) - 1));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str)+1;

  if ((result = openapiAccountingListDelete(clientHandle, acctType, &buffDesc)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to delete accounting list. (result = %d) \n", result);
  }
  else
  {
    printf("Accounting list %s deleted successfully. \n", (char *)buffDesc.pstart);
  }

  return;
}

/*******************************************************************
* @purpose  Accounting OpEN API sanity.
* 
* @param    clientHandle   @b{(input)} client handle from registration API
* 
* @returns  none
*   
* @notes  
*
* @end
*********************************************************************/
void accountingOpENAPIsSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  uint32_t acctListNameSize = 0, maxAcctMethods =0;
  char *str, *str1;
  open_buffdesc buffDesc1;
  open_buffdesc buffDesc2;
  OPEN_USER_MGR_ACCT_TYPES_t acctType = OPEN_ACCT_TYPE_UNDEFINED;
  OPEN_ACCESS_LINE_t  accessLine = OPEN_ACCESS_LINE_UNKNOWN;
  OPEN_USER_MGR_RECORD_TYPES_t recordType = OPEN_ACCT_RECORD_TYPE_UNDEFINED;

  printf("\nTesting accounting OpEN APIs sanity\n");

  if ((result = openapiAccountingListNameSizeGet(clientHandle, &acctListNameSize)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get accounting name size. (result = %d) \n", result);
    return;
  }

  if ((result = openapiAccountingMethodsMaxGet(clientHandle, &maxAcctMethods)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get maximum accounting methods supported. (result = %d) \n", result);
    return;
  }

  if ((str = (char *)malloc(acctListNameSize)) == NULL)
  {
    printf("Could not allocate memory\n");
    return;
  }
  memset(str, 0, acctListNameSize);
  buffDesc1.pstart = str;
  buffDesc1.size = acctListNameSize;

  if ((str1 = (char *)malloc(maxAcctMethods)) == NULL)
  {
    printf("Could not allocate memory\n");
    free(str);
    return;
  }

  memset(str1, 0, maxAcctMethods);
  buffDesc2.pstart = str1;
  buffDesc2.size = maxAcctMethods;

  /* openapiAccountingListNameSizeGet() */
  printf("\nTesting openapiAccountingListNameSizeGet(): \n");

  result = openapiAccountingListNameSizeGet(NULL, &acctListNameSize);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingListNameSizeGet(clientHandle, NULL);
  printf("NULL string length. (result = %d)\n", result);

  printf("openapiAccountingListNameSizeGet() sanity successful.\n");

  /* openapiAccountingMethodsMaxGet() */
  printf("\nTesting openapiAccountingMethodsMaxGet(): \n");

  result = openapiAccountingMethodsMaxGet(NULL, &maxAcctMethods);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingMethodsMaxGet(clientHandle, NULL);
  printf("NULL parameter that gets max accounting methods. (result = %d)\n", result);

  printf("openapiAccountingMethodsMaxGet() sanity successful.\n");

  /* openapiAccountingListCreate() */
  printf("\nTesting openapiAccountingListCreate(): \n");

  result = openapiAccountingListCreate(NULL, acctType, recordType, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingListCreate(clientHandle, 4, recordType, &buffDesc1);
  printf("Invalid accounting acctType. (result = %d)\n", result);

  result = openapiAccountingListCreate(clientHandle, acctType, 6, &buffDesc1);
  printf("Invalid accounting record acctType. (result = %d)\n", result);

  result = openapiAccountingListCreate(clientHandle, acctType, recordType, NULL);
  printf("NULL buff descriptor to accounting list name. (result = %d)\n", result);

  printf("openapiAccountingListCreate() sanity successful.\n");

  /* openapiAccountingMethodsAdd() */
  printf("\nTesting openapiAccountingMethodsAdd(): \n");

  result = openapiAccountingMethodsAdd(NULL, acctType, &buffDesc1, &buffDesc2);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingMethodsAdd(clientHandle, 5, &buffDesc1, &buffDesc2); 
  printf("Invalid accounting acctType. (result = %d)\n", result);

  result = openapiAccountingMethodsAdd(clientHandle, acctType, NULL, &buffDesc2); 
  printf("NULL buff descriptor to accounting list name. (result = %d)\n", result);

  result = openapiAccountingMethodsAdd(clientHandle, acctType, &buffDesc1, NULL); 
  printf("NULL buff descriptor to accounting methods. (result = %d)\n", result);

  printf("openapiAccountingMethodsAdd() sanity successful.\n");

  /* openapiAccountingMethodListGet() */
  printf("\nTesting openapiAccountingMethodListGet(): \n");

  result = openapiAccountingMethodListGet(NULL, acctType, &buffDesc1, &buffDesc2);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingMethodListGet(clientHandle, 6, &buffDesc1, &buffDesc2);
  printf("Invalid accounting acctType. (result = %d)\n", result);

  result = openapiAccountingMethodListGet(clientHandle, acctType, NULL, &buffDesc2);
  printf("NULL buff descriptor to accounting list name. (result = %d)\n", result);

  result = openapiAccountingMethodListGet(clientHandle, acctType, &buffDesc1, NULL);
  printf("NULL buff descriptor to accounting methods. (result = %d)\n", result);

  printf("openapiAccountingMethodListGet() sanity successful.\n");

  /* openapiAccountingListRecordTypeGet() */
  printf("\nTesting openapiAccountingListRecordTypeGet(): \n");

  result = openapiAccountingListRecordTypeGet(NULL, acctType, &buffDesc1, &recordType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingListRecordTypeGet(clientHandle,  5, &buffDesc1, &recordType);
  printf("Invalid accounting acctType. (result = %d)\n", result);

  result = openapiAccountingListRecordTypeGet(clientHandle, acctType, NULL, &recordType);
  printf("NULL buff descriptor to accounting list name. (result = %d)\n", result);

  result = openapiAccountingListRecordTypeGet(clientHandle, acctType, &buffDesc1, NULL );
  printf("NULL record acctType. (result = %d)\n", result);

  printf("openapiAccountingListRecordTypeGet() sanity successful.\n");

  /* openapiAccountingListFirstGet() */ 
  printf("\nTesting openapiAccountingListFirstGet(): \n");

  result = openapiAccountingListFirstGet(NULL, acctType, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingListFirstGet(clientHandle, 5, &buffDesc1);
  printf("Invalid accounting acctType. (result = %d)\n", result);

  result = openapiAccountingListFirstGet(clientHandle, acctType, NULL);
  printf("NULL buff descriptor to accounting list name. (result = %d)\n", result);

  printf("openapiAccountingListFirstGet() sanity successful.\n");

  /* openapiAccountingListNextGet()*/
  printf("\nTesting openapiAccountingListNextGet(): \n");

  result = openapiAccountingListNextGet(NULL, acctType, &buffDesc1, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingListNextGet(clientHandle, 6, &buffDesc1, &buffDesc1);
   printf("Invalid accounting acctType. (result = %d)\n", result);

  result = openapiAccountingListNextGet(clientHandle, acctType, NULL, &buffDesc1);
  printf("NULL buff descriptor to previous accounting list name. (result = %d)\n", result);

  result = openapiAccountingListNextGet(clientHandle, acctType, &buffDesc1, NULL);
  printf("NULL buff descriptor to next accounting list name. (result = %d)\n", result);

  printf("openapiAccountingListNextGet() sanity successful.\n");

  /* openapiAccountingListDelete() */
  printf("\nTesting openapiAccountingListDelete(): \n");

  result = openapiAccountingListDelete(NULL, acctType, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingListDelete(clientHandle, 5, &buffDesc1);
  printf("Invalid accounting acctType. (result = %d)\n", result);

  result = openapiAccountingListDelete(clientHandle, acctType, NULL);
  printf("NULL buff descriptor to accounting list name. (result = %d)\n", result);

  printf("openapiAccountingListDelete() sanity successful.\n");

  /* openapiAccountingListLineSet() */
  printf("\nTesting openapiAccountingListLineSet(): \n");

  result = openapiAccountingListLineSet(NULL, accessLine, acctType, &buffDesc1);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingListLineSet(clientHandle, 4, acctType, &buffDesc1);
  printf("Invalid access line. (result = %d)\n", result);

  result = openapiAccountingListLineSet(clientHandle, accessLine, 5, &buffDesc1);
  printf("Invalid accounting acctType. (result = %d)\n", result);

  result = openapiAccountingListLineSet(clientHandle, accessLine, acctType, NULL);
  printf("NULL buff descriptor to accounting list name. (result = %d)\n", result);

  printf("openapiAccountingListLineSet() sanity successful.\n");

  /* openapiAccountingListLineGet() */
  printf("\nTesting openapiAccountingListLineGet(): \n");

  result = openapiAccountingListLineGet(NULL, accessLine, acctType, &buffDesc1); 
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingListLineGet(clientHandle, 5, acctType, &buffDesc1); 
  printf("Invalid access line. (result = %d)\n", result);

  result = openapiAccountingListLineGet(clientHandle, accessLine, 4, &buffDesc1); 
  printf("Invalid accounting acctType. (result = %d)\n", result);

  result = openapiAccountingListLineGet(clientHandle, accessLine, acctType, NULL); 
  printf("NULL buff descriptor to accounting list name. (result = %d)\n", result);

  printf("openapiAccountingListLineGet() sanity successful.\n");

  /* openapiAccountingListLineDelete() */
  printf("\nTesting openapiAccountingListLineDelete(): \n");

  result = openapiAccountingListLineDelete(NULL, accessLine, acctType);
  printf("NULL Client Handle. (result = %d)\n", result);

  result = openapiAccountingListLineDelete(clientHandle, 6, acctType);
  printf("Invalid access line. (result = %d)\n", result);

  result = openapiAccountingListLineDelete(clientHandle, accessLine, 4);
  printf("Invalid accounting acctType. (result = %d)\n", result);

  printf("openapiAccountingListLineDelete() sanity successful.\n");

  free(str);
  free(str1);

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, arg1, arg2;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  open_buffdesc acctMethods;
  char str[100];
  uint32_t i = 0;

  if (argc < 2)
  {
    printAccountingAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("accounting_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Accounting API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

    printf("\n");

  /* exercise various OPEN API Accounting functions */

  switch (testNum)
  {
    case 1:
      if (argc != 5)
      {
        printAccountingAppMenu();
        exit(1);
      } 
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      accountingListCreate(&clientHandle, arg1, arg2, argv[4]);
      break;
    case 2:
      if (argc < 5)
      {
        printAccountingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      memset(str, 0, sizeof(str));
      for (i = 4; i < argc; i++)
      {
        strncat(str, argv[i], strlen(argv[i]));
      }
      acctMethods.pstart = str;
      acctMethods.size = strlen(str)+1;
      accountingMethodsAdd(&clientHandle, arg1, argv[3], &acctMethods);
      break;
    case 3:
      if (argc != 3)
      {
        printAccountingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      accountingInfoGet(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 5)
      {
        printAccountingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      accountingListLineApply(&clientHandle, arg1, arg2, argv[4]);
      break;
    case 5:
      if (argc != 4)
      {
        printAccountingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      accountingListLineGet(&clientHandle, arg1, arg2);
      break;
    case 6:
      if (argc != 4)
      {
        printAccountingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      arg2 = atoi(argv[3]);
      accountingListLineDelete(&clientHandle, arg1, arg2);
      break;    case 7:
      if (argc != 4)
      {
        printAccountingAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      accountingListDelete(&clientHandle, arg1, argv[3]);
      break;
    case 8:
      if (argc != 2)
      {
        printAccountingAppMenu();
        exit(1);
      }
      accountingOpENAPIsSanity(&clientHandle);
      break;
    default:
      printAccountingAppMenu();
      break;
  }

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Accounting API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
  
