/*********************************************************************
*
*  Copyright 2022-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename mfa_example.c
*
* @purpose   OpEN Mfa  example.
*
* @component OpEN
*
* @create    10/31/2022
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_user_manager.h"

#define OPEN_API_BUFFER_SIZE            512
#define OPENAPI_USER_MGR_LOGIN_SIZE     65 /* L7_LOGIN_SIZE */
#define OPENAPI_USER_MGR_MAX_MAIL_ADDRESS_SIZE  255 /*L7_SMTP_EMAIL_MAX_SIZE */
/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the  Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get user's multifactor authentication state: %s 1 <userName>\n", name);
  printf("Test 2: Set user's multifactor authentication state: %s 2 <userName><mfaPwdEnable>\n", name);
  printf("Test 3: Get user's Configured emailId information: %s 3 <userName>\n", name);
  printf("Test 4: Set user's emailId information: %s 4 <userName><emailInfo>\n", name);
  printf("Test 5: Get global multifactor authentication state: %s 5 \n", name);
  printf("Test 6: Set global multifactor authentication state: %s 6 <mfaCnfgEnable>\n", name);
  printf("Test 7: Run API sanity checks: %s 7 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  OPEN_BOOL_t mfaCnfgEnable;
  OPEN_BOOL_t mfaPwdEnable;
  char buffer[512];
  open_buffdesc zeroLenBuf;
  open_buffdesc badBufdescPointer;
  open_buffdesc badUserNameLen;
  open_buffdesc badEmailIdLen;
  char uname[OPEN_API_BUFFER_SIZE] = {0};
  char emailId[OPEN_API_BUFFER_SIZE] = {0};
  open_buffdesc userName, emailInfo;

  badBufdescPointer.pstart = (void *) NULL;
  badBufdescPointer.size = sizeof(buffer);
  zeroLenBuf.pstart = buffer;
  zeroLenBuf.size = 0;
  badUserNameLen.pstart = buffer;
  badUserNameLen.size = sizeof(uname);
  badEmailIdLen.pstart = buffer;
  badEmailIdLen.size = sizeof(emailId);
  userName.pstart = uname;
  userName.size = OPENAPI_USER_MGR_LOGIN_SIZE;
  emailInfo.pstart = emailId;
  emailInfo.size = OPENAPI_USER_MGR_MAX_MAIL_ADDRESS_SIZE+1; 
  

  printf("Testing  OpEN APIs sanity:\n");

  printf("Testing openapiUserMultiFactorAuthStateGet():\n");
  result = openapiUserMultiFactorAuthStateGet(NULL, &userName, &mfaPwdEnable);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiUserMultiFactorAuthStateGet(clientHandle, NULL, &mfaPwdEnable);
  printf("NULL username:(result = %d)\n", result);
  result = openapiUserMultiFactorAuthStateGet(clientHandle, &badBufdescPointer, &mfaPwdEnable);
  printf("NULL username buffdesc pstart :(result = %d)\n", result);
  result = openapiUserMultiFactorAuthStateGet(clientHandle, &zeroLenBuf, &mfaPwdEnable);
  printf("username buffdesc size with 0 length:(result = %d)\n", result);
  result = openapiUserMultiFactorAuthStateGet(clientHandle, &badUserNameLen, &mfaPwdEnable);
  printf("username buffdesc size greater than acceptable length:(result = %d)\n", result);
  result = openapiUserMultiFactorAuthStateGet(clientHandle, &userName, NULL);
  printf("NULL Mfa enable state arg :(result = %d)\n", result);

  printf("Testing openapiUserMultiFactorAuthStateSet():\n");
  result = openapiUserMultiFactorAuthStateSet(NULL, &userName, mfaPwdEnable);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiUserMultiFactorAuthStateSet(clientHandle, NULL, mfaPwdEnable);
  printf("NULL username:(result = %d)\n", result);
  result = openapiUserMultiFactorAuthStateSet(clientHandle, &badBufdescPointer, mfaPwdEnable);
  printf("NULL username buffdesc pstart :(result = %d)\n", result);
  result = openapiUserMultiFactorAuthStateSet(clientHandle, &zeroLenBuf, mfaPwdEnable);
  printf("username buffdesc size with 0 length:(result = %d)\n", result);
  result = openapiUserMultiFactorAuthStateSet(clientHandle, &badUserNameLen, mfaPwdEnable);
  printf("username buffdesc size greater than acceptable length:(result = %d)\n", result);

  printf("Testing openapiUserEmailIdGet():\n");
  result = openapiUserEmailIdGet(NULL, &userName, &emailInfo);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiUserEmailIdGet(clientHandle, NULL, &emailInfo);
  printf("NULL username:(result = %d)\n", result);
  result = openapiUserEmailIdGet(clientHandle, &badBufdescPointer, &emailInfo);
  printf("NULL username's buffdesc pstart:(result = %d)\n", result);
  result = openapiUserEmailIdGet(clientHandle, &zeroLenBuf, &emailInfo);
  printf("username's buffdesc 0 length size:(result = %d)\n", result);
  result = openapiUserEmailIdGet(clientHandle, &badUserNameLen, &emailInfo);
  printf("username's buffdesc size greater than acceptable length:(result = %d)\n", result);
  result = openapiUserEmailIdGet(clientHandle, &userName, NULL);
  printf("NULL emailid info:(result = %d)\n", result);
  result = openapiUserEmailIdGet(clientHandle, &userName, &badBufdescPointer);
  printf("NULL emailId buffdesc pstart:(result = %d)\n", result);
  result = openapiUserEmailIdGet(clientHandle, &userName, &zeroLenBuf);
  printf("emailId buffdesc 0 length size:(result = %d)\n", result);
  result = openapiUserEmailIdGet(clientHandle, &userName, &badEmailIdLen);
  printf("emailId buffdesc size greater than acceptable length:(result = %d)\n", result);

  printf("Testing openapiUserEmailIdSet():\n");
  result = openapiUserEmailIdSet(NULL, &userName, &emailInfo);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiUserEmailIdSet(clientHandle, NULL, &emailInfo);
  printf("NULL username:(result = %d)\n", result);
  result = openapiUserEmailIdSet(clientHandle, &badBufdescPointer, &emailInfo);
  printf("NULL username's buffdesc pstart:(result = %d)\n", result);
  result = openapiUserEmailIdSet(clientHandle, &zeroLenBuf, &emailInfo);
  printf("username's buffdesc 0 length size:(result = %d)\n", result);
  result = openapiUserEmailIdSet(clientHandle, &badUserNameLen, &emailInfo);
  printf("username's buffdesc size greater than acceptable length:(result = %d)\n", result);
  result = openapiUserEmailIdSet(clientHandle, &userName, NULL);
  printf("NULL emailid info:(result = %d)\n", result);
  result = openapiUserEmailIdSet(clientHandle, &userName, &badBufdescPointer);
  printf("NULL emailId buffdesc pstart:(result = %d)\n", result);
  result = openapiUserEmailIdSet(clientHandle, &userName, &zeroLenBuf);
  printf("emailId buffdesc 0 length size:(result = %d)\n", result);
  result = openapiUserEmailIdSet(clientHandle, &userName, &badEmailIdLen);
  printf("emailId buffdesc size greater than acceptable length:(result = %d)\n", result);

  printf("Testing openapiGlobalMultiFactorAuthGet():\n");
  result = openapiGlobalMultiFactorAuthGet(NULL, &mfaCnfgEnable);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiGlobalMultiFactorAuthGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiGlobalMultiFactorAuthSet():\n");
  result = openapiGlobalMultiFactorAuthSet(NULL, mfaCnfgEnable);
  printf("NULL client handle:(result = %d)\n", result);

  return;
}


/*********************************************************************
* @purpose  Get user's multifactor authentication state.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    pUname              @b{(input)}  user name.
* @param    pMfaPwdEnable       @b{(output)}  user level multifactor authentication enable state.

*
* @returns  none
*
* @end
*********************************************************************/
void userMultiFactorAuthStateGet(openapiClientHandle_t *client_handle, char *pUname, OPEN_BOOL_t *pMfaPwdEnable)
{
  open_error_t result;
  open_buffdesc userName = {.pstart = pUname,
                            .size = strlen(pUname)};
  
  if ((result = openapiUserMultiFactorAuthStateGet(client_handle, &userName, pMfaPwdEnable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get user's multifactor authentication state. (result = %d)\n", result);
  }
  else
  {
    if (OPEN_TRUE == *pMfaPwdEnable)
    {
      printf("Multifactor authentication is enabled for user:%s\n", (char*)userName.pstart);
    }
    else
    {
      printf("Multifactor authentication is disabled for user:%s\n", (char*)userName.pstart);
    }
  }
  return;
}


/*********************************************************************
* @purpose  Set user's multifactor authentication state
*
* @param    client_handle      @b{(input)}  client handle from registration API.
* @param    pUname             @b{(input)}  user name.
* @param    mfaPwdEnable       @b{(input)}  user level multifactor authentication state.

*
* @returns  none
*
* @end
*********************************************************************/
void userMultiFactorAuthStateSet(openapiClientHandle_t *client_handle, char *pUname, OPEN_BOOL_t mfaPwdEnable)
{
  open_error_t result;
  open_buffdesc userName = {.pstart = pUname,
                             .size = strlen(pUname)} ;

  if ((result = openapiUserMultiFactorAuthStateSet(client_handle, &userName, mfaPwdEnable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set user's multifactor authentication state (result = %d)\n", result);
  }
  else
  {
    printf("Multifactor authentication for user:%s is set to %s state\n",
            (char*)userName.pstart, ((mfaPwdEnable == OPEN_TRUE)? "enable":"disable"));
  }
  return;
}


/*********************************************************************
* @purpose  Get user's Configured emailId information.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    pUname              @b{(input)}  user name.
* @param    pEmail              @b{(output)}  email Id.

*
* @returns  none
*
* @end
*********************************************************************/
void userEmailIdGet(openapiClientHandle_t *client_handle, char *pUname, char *pEmail)
{
  open_error_t result;
  open_buffdesc userName = {.pstart = pUname,
                            .size = strlen(pUname)} ;
  open_buffdesc emailInfo = {.pstart = pEmail,
                             .size = OPENAPI_USER_MGR_MAX_MAIL_ADDRESS_SIZE+1} ;

  if ((result = openapiUserEmailIdGet(client_handle, &userName, &emailInfo)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get user's Configured emailId information. (result = %d)\n", result);
  }
  else
  {
    printf("user '%s' is configured with emailId:%s\n", (char*)userName.pstart, (char*)emailInfo.pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Set user's emailId information.
*
* @param    client_handle   @b{(input)}  client handle from registration API.
* @param    pUname          @b{(input)}  user name.
* @param    pEmail          @b{(input)}  email id.

*
* @returns  none
*
* @end
*********************************************************************/
void userEmailIdSet(openapiClientHandle_t *client_handle, char *pUname, char *pEmail)
{
  open_error_t result;
  open_buffdesc userName = {.pstart = pUname,
                             .size = strlen(pUname)} ;
  open_buffdesc emailInfo = {.pstart = pEmail,
                              .size = strlen(pEmail)} ;

  if ((result = openapiUserEmailIdSet(client_handle, &userName, &emailInfo)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set user's emailId information. (result = %d)\n", result);
  }
  else
  {
    printf("user '%s' is successfully configured with emailId:%s\n", (char*)userName.pstart, (char*)emailInfo.pstart);
  }
  return;
}


/*********************************************************************
* @purpose  Get global multifactor authentication state
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    pMfaCnfgEnable       @b{(output)}  Global multifactor authentication enable state.

*
* @returns  none
*
* @end
*********************************************************************/
void globalMultiFactorAuthGet(openapiClientHandle_t *client_handle, OPEN_BOOL_t *pMfaCnfgEnable)
{
  open_error_t result;

  if ((result = openapiGlobalMultiFactorAuthGet(client_handle, pMfaCnfgEnable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global multifactor authentication state (result = %d)\n", result);
  }
  else
  {
    if (OPEN_TRUE == *pMfaCnfgEnable)
    {
      printf("Multifactor authentication feature is enabled \n");
    }
    else
    {
      printf("Multifactor authentication feature is disabled \n");
    }
  }
  return;
}


/*********************************************************************
* @purpose  Set global multifactor authentication state.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    mfaCnfgEnable       @b{(input)}  Global multifactor authentication enable state.

*
* @returns  none
*
* @end
*********************************************************************/
void globalMultiFactorAuthSet(openapiClientHandle_t *client_handle, OPEN_BOOL_t mfaCnfgEnable)
{
  open_error_t result;

  if ((result = openapiGlobalMultiFactorAuthSet(client_handle, mfaCnfgEnable)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global multifactor authentication state. (result = %d)\n", result);
  }
  else
  {
    printf("Multifactor authentication feature is %s \n", (( mfaCnfgEnable == OPEN_TRUE)? "enable":"disable"));
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for
*
* @returns   0: Success
* @returns  -1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  char userName[OPEN_API_BUFFER_SIZE] = {0};
  char emailInfo[OPEN_API_BUFFER_SIZE] = {0};
  OPEN_BOOL_t mfaPwdEnable, mfaCnfgEnable;
  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("Mfa example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Mfa API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 3)
      {
        strncpy(userName, argv[2], OPEN_API_BUFFER_SIZE-1);
        userMultiFactorAuthStateGet(&clientHandle, userName, &mfaPwdEnable);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 4)
      {
        OPEN_BOOL_t mfaPwdEnable;
        strncpy(userName, argv[2], OPEN_API_BUFFER_SIZE-1);
        if (1 == atoi(argv[3]))
        {
          mfaPwdEnable = OPEN_TRUE; 
          userMultiFactorAuthStateSet(&clientHandle, userName, mfaPwdEnable);
        }
        else if (0 == atoi(argv[3]))
        {
          mfaPwdEnable = OPEN_FALSE; 
          userMultiFactorAuthStateSet(&clientHandle, userName, mfaPwdEnable);
        }
        else
        {
          printf ("Incorrect arguments passed\n");
        }
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        strncpy(userName, argv[2], OPEN_API_BUFFER_SIZE-1);
        userEmailIdGet(&clientHandle, userName, emailInfo);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 4)
      {
        strncpy(userName, argv[2], OPEN_API_BUFFER_SIZE-1);
        strncpy(emailInfo, argv[3], OPEN_API_BUFFER_SIZE-1);
        userEmailIdSet(&clientHandle, userName, emailInfo);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 2)
      {
        globalMultiFactorAuthGet(&clientHandle, &mfaCnfgEnable);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
        OPEN_BOOL_t mfaCnfgEnable;
        if (1 == atoi(argv[2]))
        {
          mfaCnfgEnable = OPEN_TRUE; 
          globalMultiFactorAuthSet(&clientHandle, mfaCnfgEnable);
        }
        else if (0 == atoi(argv[2]))
        {
          mfaCnfgEnable = OPEN_FALSE; 
          globalMultiFactorAuthSet(&clientHandle, mfaCnfgEnable);
        }
        else
        {
          printf ("Incorrect arguments passed\n");
        }
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 2)
      {
        runSanity(&clientHandle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping  API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

