/*********************************************************************
*
* Copyright 2016-2023 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  vlan_example.c
*
* @purpose   VLAN APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    09/04/2012
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_vlan.h"

#define VLAN_ID_MIN 1
#define VLAN_ID_MAX 1024
#define IF_PRIORITY 3

/* 
   Some OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/
#define RETRY_COUNT 10
#define RETRY_INTERVAL_MSEC 100000

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         create and verify a created VLAN. If a VLAN is created
*         successfully it also configures and then gets the 
*         name configured for the VLAN.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    vlanId        @b{(input)} VLAN Id to be created.
*
* @returns  OPEN_E_NONE        The VLAN was successfully created, name configured and verified.
* @returns  OPEN_E_FAIL        The VLAN was not successfully created or verified or 
*                              VLAN name could not be configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*********************************************************************/
open_error_t vlanCreateAndVerify(openapiClientHandle_t *clientHandle, uint32_t vlanId)
{
  open_error_t result;
  uint32_t i;
  uint32_t vlanNameLen;
  open_buffdesc vlanNameDesc;
  char *vlanName = NULL;

  if ((result = openapiVlanCreate(clientHandle, vlanId)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to create VLAN %u. (result = %d)\n", vlanId, result);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanCreatedCheck(clientHandle, vlanId)) == OPEN_E_NONE)
      {
        /* got what we expected, move on */
        break;
      }
      else
      {
        usleep(RETRY_INTERVAL_MSEC);
      }
    }

    if (result == OPEN_E_NONE)
    {
      if ((result = openapiVlanNameLenGet(clientHandle, &vlanNameLen)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to get VLAN Name Length. (result = %d)\n", result);
      }
      else
      {
        vlanName = (char*)malloc(vlanNameLen);
      
        if (vlanName == NULL)
        {
          printf("  Could not allocate memory for VLAN name.\n");
          result = OPEN_E_FAIL;
        }
        else
        {
          memset(vlanName, 0, vlanNameLen);
          sprintf(vlanName, "vlanName%u", vlanId);
          vlanNameDesc.pstart = vlanName;
          vlanNameDesc.size = strlen(vlanName) + 1;
        
          if ((result = openapiVlanNameSet(clientHandle, vlanId, &vlanNameDesc)) != OPEN_E_NONE)
          {
            printf("  Bad return code trying to set VLAN %u name. (result = %d)\n", vlanId, result);
          }
          else
          {
            memset(vlanName, 0, vlanNameLen);
            vlanNameDesc.size = vlanNameLen;
            if ((result = openapiVlanNameGet(clientHandle, vlanId, &vlanNameDesc)) != OPEN_E_NONE)
            {
              printf("  Bad return code trying to get VLAN %u name. (result = %d)\n", vlanId, result);
            }
            else
            {
              printf("  VLAN %u name = %s\n", vlanId, vlanName);
            }
          }
        }
      }
    }
    else
    {
      printf("  Could not verify that VLAN %u is created. (result = %d)\n", vlanId, result);
    }
  }

  if (vlanName)
  {
    free(vlanName);
  }
  
  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         delete and verify a created VLAN.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    vlanId         @b{(input)} VLAN Id to be deleted.
*
* @returns  OPEN_E_NONE        The VLAN was successfully deleted and verified.
* @returns  OPEN_E_FAIL        The VLAN was not successfully deleted or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*********************************************************************/
open_error_t vlanDeleteAndVerify(openapiClientHandle_t *clientHandle, uint32_t vlanId)
{
  open_error_t result;
  uint32_t i;

  if ((result = openapiVlanDelete(clientHandle, vlanId)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to create VLAN %u. (result = %d)\n", vlanId, result);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanCreatedCheck(clientHandle, vlanId)) != OPEN_E_NONE)
      {
        /* got what we expected, move on */
        break;
      }
      else
      {
        usleep(RETRY_INTERVAL_MSEC);
      }
    }
  }

  /* testing for a negative, adjust return code */
  if (result == OPEN_E_FAIL)
  {
    result = OPEN_E_NONE;
  }

  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         configure and verify the Port Acceptable Frame Type for an interface.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum         @b{(input)} switch interface.
* @param    type          @b{(input)} OPEN_DOT1Q_ACCEPT_FRAME_TYPE_t
*
* @returns  OPEN_E_NONE        The type was successfully configured and verified.
* @returns  OPEN_E_FAIL        The type was not successfully configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*********************************************************************/
open_error_t interfaceAcceptFrameTypeSetAndVerify(openapiClientHandle_t *clientHandle,
                                                  uint32_t ifNum,
                                                  OPEN_DOT1Q_ACCEPT_FRAME_TYPE_t type)
{
  open_error_t result;
  uint32_t newType;
  uint32_t i;

  if ((result = openapiVlanIfAcceptFrameTypeSet(clientHandle, ifNum, type)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to set Accept Frame Type. (result = %d)\n", result);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfAcceptFrameTypeGet(clientHandle, ifNum, &newType)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        printf("  Bad return code trying to get Accept Frame Type. (result = %d)\n", result);
        break;
      }
      else
      {
        if (newType != type)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (newType != type)
    {
      result = OPEN_E_FAIL;
    }
  }

  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         configure and verify the Port VLAN ID for an interface.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum          @b{(input)} switch interface.
* @param    vlanId         @b{(input)} PVID.
*
* @returns  OPEN_E_NONE        The PVID was successfully configured and verified.
* @returns  OPEN_E_FAIL        The PVID was not successfully configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Setting PVID not valid for the specified interface.
* @returns  OPEN_E_NOT_FOUND   The VLAN specified is not created.
*
*********************************************************************/
open_error_t interfacePvidSetAndVerify(openapiClientHandle_t *clientHandle, uint32_t ifNum, uint32_t vlanId)
{
  open_error_t result;
  uint32_t newVlanId;
  uint32_t i;

  if ((result = openapiVlanIfPvidSet(clientHandle, ifNum, vlanId)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to set PVID. (result = %d)\n", result);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfPvidGet(clientHandle, ifNum, &newVlanId)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        printf("  Bad return code trying to get PVID. (result = %d)\n", result);
        break;
      }
      else
      {
        if (newVlanId != vlanId)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (newVlanId != vlanId)
    {
      result = OPEN_E_FAIL;
    }
  }

  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         configure and verify the VLAN participation for an interface.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum          @b{(input)} switch interface.
* @param    vlanId         @b{(input)} PVID.
* @param    partStatus     @b{(input)} Port VLAN Participation Mode.
*
* @returns  OPEN_E_NONE        The participation mode was successfully configured and verified.
* @returns  OPEN_E_FAIL        The participation mode was not successfully configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Setting participation mode not valid for the specified interface.
* @returns  OPEN_E_NOT_FOUND   The VLAN specified is not created.
*
*********************************************************************/
open_error_t interfaceVlanParticipationSetAndVerify(openapiClientHandle_t *clientHandle, uint32_t ifNum, 
                                                    uint32_t vlanId, OPEN_VLAN_PARTICIPATION_MODE_t partStatus)
{
  open_error_t result;
  OPEN_VLAN_PARTICIPATION_MODE_t newPartStatus;
  uint32_t i;

  if ((result = openapiVlanIfParticipationSet(clientHandle, vlanId, ifNum, partStatus)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to set VLAN participation configuration. (result = %d, ifNum = %u, partStatus = %u)\n", result, ifNum, partStatus);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfParticipationGet(clientHandle, vlanId, ifNum, &newPartStatus)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        printf("  Bad return code trying to get VLAN participation configuration. (result = %d, ifNum = %u, partStatus = %u)\n", result, ifNum, partStatus);
        break;
      }
      else
      {
        if (newPartStatus != partStatus)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (newPartStatus != partStatus)
    {
      result = OPEN_E_FAIL;
    }
  }
  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         configure and verify the VLAN tagging for an interface.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum          @b{(input)} switch interface.
* @param    vlanId         @b{(input)} PVID.
* @param    taggingMode    @b{(input)} Port VLAN Tagging Mode.
*
* @returns  OPEN_E_NONE        The VLAN tagging was successfully configured and verified.
* @returns  OPEN_E_FAIL        The VLAN tagging was not successfully configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* @returns  OPEN_E_UNAVAIL     Setting VLAN tagging not valid for the specified interface.
* @returns  OPEN_E_NOT_FOUND   The VLAN specified is not created.
*
*********************************************************************/
open_error_t interfaceVlanTaggingSetAndVerify(openapiClientHandle_t *clientHandle, uint32_t ifNum, 
                                              uint32_t vlanId, OPEN_CONTROL_t taggingMode)
{
  open_error_t result;
  OPEN_CONTROL_t newTaggingMode;
  uint32_t i;

  if ((result = openapiVlanIfTaggingSet(clientHandle, ifNum, vlanId, taggingMode)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to set VLAN tagging configuration. (result = %d, ifNum = %u, taggingMode = %u)\n", result, ifNum, taggingMode);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfTaggingGet(clientHandle, ifNum, vlanId, &newTaggingMode)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        printf("  Bad return code trying to get VLAN tagging configuration. (result = %d, ifNum = %u, taggingMode = %u)\n", result, ifNum, taggingMode);
        break;
      }
      else
      {
        if (newTaggingMode != taggingMode)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (newTaggingMode != taggingMode)
    {
      result = OPEN_E_FAIL;
    }
  }
  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         configure and verify the user priority for an interface.
*
* @param    clientHandle  @b{(input)} client handle from registration API
* @param    ifNum         @b{(input)} switch interface.
* @param    priority      @b{(input)} priority
*
* @returns  OPEN_E_NONE        The default priority was successfully configured and verified.
* @returns  OPEN_E_FAIL        The default priority was not successfully configured or verified.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
*********************************************************************/
open_error_t interfacePrioritySetAndVerify(openapiClientHandle_t *clientHandle, uint32_t ifNum,
                                           uint32_t priority)
{
  open_error_t result;
  uint32_t i;
  uint32_t numTrafficClasses;
  uint32_t newPriority;

  if ((result = openapiVlanIfNumTrafficClassesGet(clientHandle, ifNum, &numTrafficClasses)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to the number of traffic classes supported. (result = %d, ifNum = %u)\n", result, ifNum);
    return OPEN_E_FAIL;
  }

  if (priority > numTrafficClasses)
  {
    printf("  Parameter value for priority out of accepted range. (priority = %u, numTrafficClasses = %u)\n", priority, numTrafficClasses);
    return OPEN_E_PARAM;
  }

  if ((result = openapiVlanIfDefaultUserPrioritySet(clientHandle, ifNum, priority)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to set user priority. (result = %d, ifNum = %u, priority = %u)\n", result, ifNum, priority);
  }
  else
  {
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfDefaultUserPriorityGet(clientHandle, ifNum, &newPriority)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        printf("  Bad return code trying to get user priority. (result = %d, ifNum = %u)\n", result, ifNum);
        break;
      }
      else
      {
        if (newPriority != priority)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (newPriority != priority)
    {
      result = OPEN_E_FAIL;
    }
  }
  return(result);
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         create and verify Private VLANs. If a VLAN is created
*         successfully then it configures that VLAN as private VLAN. 
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    primary_vlan   @b{(input)} VLAN Id to be created and configured as
*                                      Primary VLAN.
* @param    isolated_vlan  @b{(input)} VLAN Id to be created and configured as
*                                      Isolated VLAN.
* @param    community_vlan @b{(input)} VLAN Id to be created and configured as
*                                      Community VLAN.
*
* @returns  void 
*
*********************************************************************/
void privateVlanCreateAndVerify(openapiClientHandle_t *clientHandle, 
                                uint32_t primary_vlan,
                                uint32_t isolated_vlan,
                                uint32_t community_vlan)
{
  open_error_t             result = OPEN_E_NONE;
  OPEN_DOT1Q_PVLAN_TYPE_t  pvlanType, tmpVal;
  uint32_t                 i;
  OPEN_VLAN_LIST_t         vidList, tmpList; 

  /*Private VLANs creation and verification.*/

  result = openapiVlanCreate(clientHandle, primary_vlan);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to create VLAN %u. (result = %d)\n", primary_vlan, result);
  } 
  else
  {
    pvlanType = OPEN_PRIVATE_VLAN_PRIMARY; 
    result = openapiDot1qPrivateVlanEnable(clientHandle, primary_vlan, pvlanType);
    if (result != OPEN_E_NONE)
    {
      printf("Bad return code trying to configure a VLAN as Primary VLAN %u. (result = %d)\n", primary_vlan, result);
    } 
 
    if (result == OPEN_E_NONE)
    {
      for (i=0; i < RETRY_COUNT; i++)
      {  
        if (openapiDot1qPrivateVlanGet(clientHandle, primary_vlan, &tmpVal) != OPEN_E_NONE)
        {
          printf("Bad return code trying to get Type for VLAN %u. (result = %d)\n", primary_vlan, result);
        } 
        else 
        {
          if (tmpVal != pvlanType)
          {
            usleep(RETRY_INTERVAL_MSEC); 
          }  
          else 
          {
            break;
          }
        } 
      }  

      if (tmpVal != pvlanType)
      {
        printf("Failure configuring Primary VLAN %u. \n", primary_vlan);
      }
      else
      {
        printf("Successfully configured and verified Primary VLAN %u. \n", primary_vlan);
      }
    }  
  } 

  result = openapiVlanCreate(clientHandle, isolated_vlan);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to create VLAN %u. (result = %d)\n", isolated_vlan, result);
  } 
  else
  {
    pvlanType = OPEN_PRIVATE_VLAN_SECONDARY_ISOLATED; 
    result = openapiDot1qPrivateVlanEnable(clientHandle, isolated_vlan, pvlanType);
    if (result != OPEN_E_NONE)
    {
      printf("Bad return code trying to configure a VLAN as Isolated VLAN %u. (result = %d)\n", isolated_vlan, result);
    } 

    if (result == OPEN_E_NONE)
    {
      for (i=0; i < RETRY_COUNT; i++)
      {  
        if (openapiDot1qPrivateVlanGet(clientHandle, isolated_vlan, &tmpVal) != OPEN_E_NONE)
        {
          printf("Bad return code trying to get Type for VLAN %u. (result = %d)\n", isolated_vlan, result);
        } 
        else 
        {
          if (tmpVal != pvlanType)
          {
            usleep(RETRY_INTERVAL_MSEC); 
          }  
          else 
          {
            break;
          }
        } 
      }

      if (tmpVal != pvlanType)
      {
        printf("Failure configuring Isolated VLAN %u. \n", isolated_vlan);
      }
      else
      {
        printf("Successfully configured and verified Isolated VLAN %u. \n", isolated_vlan);
      }
    }
  } 

  result = openapiVlanCreate(clientHandle, community_vlan);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to create VLAN %u. (result = %d)\n", community_vlan, result);
  } 
  else
  {
    pvlanType = OPEN_PRIVATE_VLAN_SECONDARY_COMMUNITY; 
    result = openapiDot1qPrivateVlanEnable(clientHandle, community_vlan, pvlanType);
    if (result != OPEN_E_NONE)
    {
      printf("Bad return code trying to configure a VLAN as Community VLAN %u. (result = %d)\n", community_vlan, result);
    } 

    if (result == OPEN_E_NONE)
    {
      for (i=0; i < RETRY_COUNT; i++)
      {  
        if (openapiDot1qPrivateVlanGet(clientHandle, community_vlan, &tmpVal) != OPEN_E_NONE)
        {
          printf("Bad return code trying to get Type for VLAN %u. (result = %d)\n", community_vlan, result);
        } 
        else 
        {
          if (tmpVal != pvlanType)
          {
            usleep(RETRY_INTERVAL_MSEC); 
          }  
          else 
          {
            break;
          }
        } 
      }

      if (tmpVal != pvlanType)
      {
        printf("Failure configuring Community VLAN %u. \n", community_vlan);
      }
      else
      {
        printf("Successfully configured and verified Community VLAN %u. \n", community_vlan);
      }
    }
  }

  /* Associating VLANs with Primary VLAN */ 
 
  memset(&vidList, 0, sizeof(vidList));

  vidList.ids[0] = isolated_vlan;
  vidList.ids[1] = community_vlan;
  vidList.numEntries = 2;

  result = openapiDot1qApiPrivateVlanAssocAdd(clientHandle, primary_vlan, vidList);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to associate VLANs with Primary VLAN %u. (result = %d)\n", primary_vlan, result);
  }
  else
  {
    /* Association get */
    memset(&tmpList, 0, sizeof(tmpList));
    if (openapiDot1qPrivateVlanAssociationGet(clientHandle, primary_vlan, &tmpList) != OPEN_E_NONE)
    {
      printf("Bad return code trying to get associated VLANs with Primary VLAN %u. (result = %d)\n", primary_vlan, result);
    } 
    else
    {
      printf("VLAN list set: ");
      for (i=0; i < vidList.numEntries; i++)
      {
        printf("%d", vidList.ids[i]);
        if (i < vidList.numEntries - 1)
        {
          printf(", ");
        }
      }

      printf("\n");
      printf("VLAN list retrieved: ");
      for (i = 0; i < tmpList.numEntries; i++)  
      {
        printf("%d", tmpList.ids[i]);
        if (i < tmpList.numEntries - 1)
        {
          printf(", ");
        }
      }
      printf("\n");  
    }

    /* Association remove */
    if (openapiDot1qApiPrivateVlanAssocRemove(clientHandle, primary_vlan, vidList) != OPEN_E_NONE)
    {
      printf("Bad return code trying to remove VLANs associated with Primary VLAN %u. (result = %d)\n", primary_vlan, result);
    } 
    else
    {
      printf("Successfully removed VLANs associated with Primary VLAN %u. \n", primary_vlan);
    } 
  }

  printf("Successfully configured and verified VLAN association with Primary VLAN. \n");

  return;
}

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         create and verify VLAN bindings for Private VLAN interface.
*         configured as Host interface.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    ifNum          @b{(input)} switch interface.
* @param    primary_vlan   @b{(input)} VLAN Id to be created and configured as
*                                      Primary VLAN.
* @param    secondary_vlan @b{(input)} VLAN Id to be created and configured as
*                                      secondary VLAN.
*
* @returns  void 
*
*********************************************************************/
void interfaceHostPrivateVlanCreateAndVerify(openapiClientHandle_t *clientHandle, 
                                             uint32_t ifNum,
                                             uint32_t primary_vlan,
                                             uint32_t secondary_vlan)
{
  open_error_t  result = OPEN_E_NONE;
  uint32_t      tmpVal1, tmpVal2;

  result = openapiDot1qIntfPVlanHostAssocSet(clientHandle, ifNum, primary_vlan, secondary_vlan);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to bind VLANs for Private VLAN interface %u. (result = %d)\n", ifNum, result);
  }  
  else
  {
    printf("Successfully set VLAN bindings for Private VLAN interface %u. \n", ifNum);
  }

  result = openapiPvlanIntfHostAssocGet(clientHandle, ifNum, &tmpVal1, &tmpVal2);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get VLAN bindings for Private VLAN interface %u. (result = %d)\n", ifNum, result);
  }  
  else
  {
    if (tmpVal1 == primary_vlan)
    {
      printf("Successfully binded Primary VLAN to Private VLAN interface %u. \n", ifNum);  
    }   
    if (tmpVal2 == secondary_vlan)
    {
      printf("Successfully binded Secondary VLAN to Private VLAN interface %u. \n", ifNum);  
    }   
  }

  result = openapiDot1qPVlanIntfHostAssocRemove(clientHandle, ifNum);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to remove VLAN bindings for Private VLAN interface %u. (result = %d)\n", ifNum, result);
  } 
  else
  {
    printf("Successfully removed VLAN bindings on interface %u \n", ifNum);
  } 

  printf("Successfully configured and verified VLAN bindings for Private VLAN interface %u \n", ifNum);
  return;
}  

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         create and verify VLAN bindings for Private VLAN interface.
*         configured as Promiscous interface.
*
* @param    clientHandle   @b{(input)} client handle from registration API
* @param    ifNum          @b{(input)} switch interface.
* @param    primary_vlan   @b{(input)} VLAN Id to be created and configured as
*                                      Primary VLAN.
* @param    vidList        @b{(input)} VLAN Id to be created and configured as
*                                      secondary VLAN.
*
* @returns  void 
*
*********************************************************************/
void interfacePromiscPrivateVlanCreateAndVerify(openapiClientHandle_t *clientHandle, 
                                                uint32_t ifNum,
                                                uint32_t primary_vlan,
                                                OPEN_VLAN_LIST_t vidList)
{
  open_error_t      result = OPEN_E_NONE;
  uint32_t          tmpVal, i = 0;
  OPEN_VLAN_LIST_t  tmpList;

  result = openapiDot1qIntfPVlanPromiscAssocSet(clientHandle, ifNum, primary_vlan, vidList);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to bind VLANs for Private VLAN interface %u. (result = %d)\n", ifNum, result);
  }  
  else
  {
    printf("Successfully set VLAN bindings for Private VLAN interface %u. \n", ifNum);
  }

  result = openapiPvlanIntfPromiscAssocGet(clientHandle, ifNum, &tmpVal, &tmpList);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to get VLAN bindings for Private VLAN interface %u. (result = %d)\n", ifNum, result);
  }  
  else
  {
    if (tmpVal == primary_vlan)
    {
      printf("Successfully binded Primary VLAN to Private VLAN interface %u. \n", ifNum);  
    } 

    printf("VLAN list set: ");
    for (i=0; i < tmpList.numEntries; i++)
    {
      printf("%d", tmpList.ids[i]);
      if (i < tmpList.numEntries - 1)
      {
        printf(", ");
      }
    }
    printf("\n");
  }

  result = openapiDot1qPVlanIntfPromiscAssocRemove(clientHandle, ifNum);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to remove VLAN bindings for Private VLAN interface %u. (result = %d)\n", ifNum, result);
  } 
  else
  {
    printf("Successfully removed VLAN bindings on interface %u \n", ifNum);
  } 

  printf("Successfully configured and verified VLAN bindings for Private VLAN interface %u \n", ifNum);
  return;
}  

/*******************************************************************
*
* @brief  This function demonstrates the usage of OpEN APIs that
*         enable and disable Auto-Trunk in the system.
*
* @param    clientHandle   @b{(input)} client handle from registration API
*
* @returns  void 
*
*********************************************************************/
void autoTrunkConfigAndVerify(openapiClientHandle_t *clientHandle)
{
  OPEN_CONTROL_t mode;
  open_error_t  result = OPEN_E_NONE;
  OPEN_BOOL_t   operMode;
  uint32_t      ifNum;

  result = openapiVlanGlobalAutoTrunkSet(NULL, OPEN_ENABLE);
  if (result != OPEN_E_PARAM)
  {
    printf("Failed to verify parameter error while setting global Auto-Trunk mode. (result = %d)\n", result);
  }  
  else
  {
    printf("Successfully verified parameter error while setting global Auto-Trunk mode. \n");
  }

  result = openapiVlanGlobalAutoTrunkGet(NULL, &mode);
  if (result != OPEN_E_PARAM)
  {
    printf("Failed to verify parameter error while getting global Auto-Trunk mode. (result = %d)\n", result);
  }  
  else
  {
    printf("Successfully verified parameter error while getting global Auto-Trunk mode. \n");
  }

  result = openapiVlanGlobalAutoTrunkGet(clientHandle, NULL);
  if (result != OPEN_E_PARAM)
  {
    printf("Failed to verify parameter error while getting global Auto-Trunk mode. (result = %d)\n", result);
  }  
  else
  {
    printf("Successfully verified parameter error while getting global Auto-Trunk mode. \n");
  }

  result = openapiVlanGlobalAutoTrunkSet(clientHandle, OPEN_ENABLE);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to enable Auto-Trunk. (result = %d)\n", result);
  }  
  else
  {
    printf("Successfully set global Auto-Trunk mode to enable. \n");
  }

  result = openapiVlanGlobalAutoTrunkGet(clientHandle, &mode);
  if ((result == OPEN_E_NONE) && (OPEN_ENABLE == mode))
  {
    printf("Successfully verified that global Auto-Trunk mode is enabled. \n");
  }
  else
  {
    printf("Bad return code trying to get global Auto-Trunk mode. (result = %d)\n", result);
  }  

  result = openapiVlanGlobalAutoTrunkSet(clientHandle, OPEN_DISABLE);
  if (result != OPEN_E_NONE)
  {
    printf("Bad return code trying to disable Auto-Trunk. (result = %d)\n", result);
  }  
  else
  {
    printf("Successfully set global Auto-Trunk mode to disable. \n");
  }

  result = openapiVlanGlobalAutoTrunkGet(clientHandle, &mode);
  if ((result == OPEN_E_NONE) && (OPEN_DISABLE == mode))
  {
    printf("Successfully verified that global Auto-Trunk mode is disabled. \n");
  }
  else
  {
    printf("Bad return code trying to get global Auto-Trunk mode. (result = %d)\n", result);
  }

  printf("Successfully configured and verified global Auto-Trunk mode \n");

  if ((result = openapiIfFirstGet (clientHandle, OPEN_INTF_TYPE_PHY,
                                   &ifNum)) != OPEN_E_NONE)
  {
    printf("  Bad return code trying to get first physical interface. (result = %d)\n",result);
  }

  printf("Using physical interface %d\n", ifNum);

  result = openapiDot1qOperAutoTrunkModeGet(clientHandle, ifNum, &operMode);
  if ((result == OPEN_E_NONE) && (OPEN_FALSE == operMode))
  {
    printf("Successfully fetched the operational auto-trunk mode on interface %d. \n", ifNum);
  }
  else
  {
    printf("Bad return code trying to fetch the operational auto-trunk mode on interface %d. (result = %d)\n", ifNum, result);
  }  

  printf("Successfully fetched the operational auto-trunk mode \n");

  return;
}  

/*********************************************************************
* @purpose  Get dot1q Max Supported VLANs.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    val           @b{(output)} max supported VLAN count
*
*
* @returns  none
*
* @end
*********************************************************************/
void dot1qMaxSupportedVlansGet(openapiClientHandle_t *client_handle, 
                               uint32_t *val)
{
  open_error_t result;

  if ((result = openapiDot1qMaxSupportedVlansGet(client_handle, val)) != 
       OPEN_E_NONE)
  {
    printf("Bad return code trying to get dot1q Max Supported VLANs. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("\nMaximum supported VLANs: %d\n", *val);
  }
  return;
}


/*********************************************************************
* @purpose  Get the max VLAN entries allowed.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    val           @b{(output)} number of VLANs in use on device
*
* @returns  none
*
* @end
*********************************************************************/
void mostVlanEntriesGet(openapiClientHandle_t *client_handle, uint32_t *val)
{
  open_error_t result;

  if ((result = openapiMostVlanEntriesGet(client_handle, val)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to get the max VLAN entries allowed. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Number of VLANs in use: %d\n", *val);
  }
  return;
}

/*********************************************************************
* @purpose  Get the number of static VLANs.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    val           @b{(output)} number of static VLANs 
*
* @returns  none
*
* @end
*********************************************************************/
void staticVlanEntriesGet(openapiClientHandle_t *client_handle, uint32_t *val)
{
  open_error_t result;

  if ((result = openapiStaticVlanEntriesGet(client_handle, val)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of static VLANs. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Number of static VLANs: %d\n", *val);
  }
  return;
}


/*********************************************************************
* @purpose  Get the number of dynamic VLANs.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    val           @b{(output)} number of dynamic VLANs 
*
* @returns  none
*
* @end
*********************************************************************/
void dynamicVlanEntriesGet(openapiClientHandle_t *client_handle, uint32_t *val)
{
  open_error_t result;

  if ((result = openapiDynamicVlanEntriesGet(client_handle, val)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of dynamic VLANs. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Number of dynamic VLANs: %d\n", *val);
  }
  return;
}
/*********************************************************************
* @purpose  Get the number of times a VLAN entry has been deleted from 
*           DOT1Q table.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    val           @b{(output)} number of times VLANs are deleted

*
* @returns  none
*
* @end
*********************************************************************/
void dot1qVlanNumDeletesGet(openapiClientHandle_t *client_handle, uint32_t *val)
{
  open_error_t result;

  if ((result = openapiDot1qVlanNumDeletesGet(client_handle, val)) != 
       OPEN_E_NONE)
  {
    printf("Bad return code trying to get the number of times a VLAN entry "
           "has been deleted from DOT1Q table. (result = %d)\n", result);
  }
  else
  {
    printf("Number of times VLANs are deleted: %d\n", *val);
  }
  return;
}


/*********************************************************************
* @purpose  Get the tagging configuration of a port.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    vindex        @b{(input)}  VLAN Id
* @param    index         @b{(input)}  physical or logical interface
* @param    val           @b{(output)} tagged status of a port - tagged
*                                      or untagged
*
* @returns  none
*
* @end
*********************************************************************/
void vlanTaggedSetGet(openapiClientHandle_t *client_handle, uint32_t vindex, 
                      uint32_t index, uint32_t *val)
{
  open_error_t result;

  if ((result = openapiVlanTaggedSetGet(client_handle, vindex, index, val)) != 
      OPEN_E_NONE)
  {
    printf("Bad return code trying to get the tagging configuration for a "
           "member port. (result = %d)\n", result);
  }
  else
  {
    printf("Tagged status of interface %d in VLAN %d is %s\n", index, vindex, 
           (*val == OPEN_DOT1Q_UNTAGGED)? "untagged":"tagged");
  }
  return;
}
/*********************************************************************
* @purpose  Gets the next VLAN ID.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    vid           @b{(input)}  current VLAN Id
* @param    nextVid       @b{(output)} next VLAN id
*
* @returns  none
*
* @end
*********************************************************************/
void nextVlanGet(openapiClientHandle_t *client_handle, uint32_t vid, 
                 uint32_t *nextVid)
{
  open_error_t result;

  if ((result = openapiNextVlanGet(client_handle, vid, nextVid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the next VLAN ID. (result = %d)\n", 
           result);
  }
  else
  {
    printf("Given VLAN: %d Next VLAN: %d\n", vid, *nextVid);
  }
  return;
}


/*********************************************************************
* @purpose  Get the Ingress Filtering configuration for a port.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    index         @b{(input)}  physical or logical interface
* @param    val           @b{(output)} enabled or disabled 
*
* @returns  none
*
* @end
*********************************************************************/
void qportsEnableIngressFilteringGet(openapiClientHandle_t *client_handle, 
                                     uint32_t index, uint32_t *val)
{
  open_error_t result;

  if ((result = openapiQportsEnableIngressFilteringGet(client_handle, index, 
       val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Ingress Filtering configuration "
           "for a port. (result = %d)\n", result);
  }
  else
  {
    printf("Ingress filtering mode of interface %d is %s\n", index,
            (*val == OPEN_ENABLE)?"enabled":"disabled");
  }
  return;
}
/*********************************************************************
* @purpose  Set the Ingress Filtering configuration for a port.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    interface     @b{(input)}  physical/logical intf/all intfs
* @param    mode          @b{(input)}  enable (1) or disable (0)

*
* @returns  none
*
* @end
*********************************************************************/
void qportsEnableIngressFilteringSet(openapiClientHandle_t *client_handle, 
                                     uint32_t interface, uint32_t mode)
{
  open_error_t result;

  if ((result = openapiQportsEnableIngressFilteringSet(client_handle, 
       interface, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the Ingress Filtering "
           "configuration for a port. (result = %d)\n", result);
  }
  else
  {
    printf("Ingress filtering mode for interface %d is set successfully.\n",
           interface);
  }
  return;
}


/*********************************************************************
* @purpose  Get VLAN ingress filtering status per port.
*
* @param    client_handle  @b{(input)}  client handle from regn API
* @param    basePort       @b{(input)}  interface number
* @param    val            @b{(output)} ingress filter mode - enable 
*                                       or disable
*
*
* @returns  none
*
* @end
*********************************************************************/
void dot1qPortIngressFiltering(openapiClientHandle_t *client_handle, 
                               uint32_t basePort, uint32_t *val)
{
  open_error_t result;

  if ((result = openapiDot1qPortIngressFiltering(client_handle, basePort, val)) 
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get VLAN ingress filtering status per port."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Ingress filtering of interface %d is %s\n", basePort,
            (*val == OPEN_ENABLE)?"enabled":"disabled");
  }
  return;
}
/*********************************************************************
* @purpose  Get the Current Ingress Filtering configuration for a port.
*
* @param    client_handle  @b{(input)}  client handle from regn API
* @param    index          @b{(input)}  physical or logical interface
* @param    val            @b{(output)} ingress filter mode - enable 
*                                       or disable
*
* @returns  none
*
* @end
*********************************************************************/
void qportsCurrentIngressFilteringGet(openapiClientHandle_t *client_handle, 
                                      uint32_t index, uint32_t *val)
{
  open_error_t result;

  if ((result = openapiQportsCurrentIngressFilteringGet(client_handle, index, 
      val)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the Current Ingress Filtering "
           "configuration for a port. (result = %d)\n", result);
  }
  else
  {
    printf("Current ingress filtering mode of interface %d is %s\n", index,
            (*val == OPEN_ENABLE)?"enabled":"disabled");
  }
  return;
}


/*********************************************************************
* @purpose  Get a list of VLANs an interface is a member of.
*
* @param    client_handle @b{(input)}  client handle from regn API
* @param    intfNum       @b{(input)}  interface number
* @param    data          @b{(output)} pointer to data
* @param    numOfElements @b{(output)} length of the VLAN list

*
* @returns  none
*
* @end
*********************************************************************/
void vlanVIDListGet(openapiClientHandle_t *client_handle, uint32_t intfNum, 
                    OPEN_VLAN_LIST_t *data, uint32_t *numOfElements)
{
  uint8_t i;
  open_error_t result;

  if ((result = openapiVlanVIDListGet(client_handle, intfNum, data, 
      numOfElements)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get a list of VLANs an interface is a "
           "member of. (result = %d)\n", result);
  }
  else
  {
    printf("Interface %d is member of %d VLANs: ", intfNum, *numOfElements);
    for (i = 0; i < data->numEntries; i++)
    {
      printf("%d", data->ids[i]);
      if (i < data->numEntries - 1)
      {
        printf(", ");
      }
    }
    printf("\n");
  }
  return;

}

/*********************************************************************
 * @purpose  Get the packets that are not forwarded on the specified
 *           interface.
 *
 * @param    client_handle @b{(input)}  client handle from registration
 *                                      API
 * @param    ifNum         @b{(input)}  internal interface number
 *
 * @returns  none
 *
 * @end
 *********************************************************************/
void dot1qRxPacketsNotForwardedCountGet(openapiClientHandle_t *client_handle,
                                        uint32_t ifNum)
{
  open_error_t result;
  uint64_t ctrValue;
  char intfName[256];
  open_buffdesc intfNameBuff = {.pstart = intfName, .size = sizeof(intfName)};

  memset(intfName, 0, sizeof(intfName));
  intfNameBuff.pstart = intfName;
  intfNameBuff.size = sizeof(intfName);

  if ((result = openapiIfNameGet(client_handle, ifNum, &intfNameBuff)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the interface name for "
           "interface %d. (result = %d)\n", ifNum, result);
    return;
  }
  if ((result = openapiDot1qRxPacketsNotForwardedCountGet(client_handle, ifNum,
                                                          &ctrValue)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the packets that are not forwarded on "
           "interface %s(%d). (result = %d)\n", intfName, ifNum, result);
  }
  else
  {
    printf("Number of packets that are not forwarded on interface %s(%d) - 0x%016llx\n",
           intfName, ifNum, ctrValue);
  }
  return;
}

static char *vlanTypeStr(OPEN_DOT1Q_VLAN_TYPE_t type)
{
  char *ret = "Unknown";

  switch (type)
  {
    case OPEN_DOT1Q_VLAN_TYPE_NOT_PARTICIPATING:
      ret = "Not participating";
      break;

    case OPEN_DOT1Q_VLAN_TYPE_DEFAULT:             
      ret = "Default VLAN";
      break;

    case OPEN_DOT1Q_VLAN_TYPE_ADMIN:     
      ret = "Static VLAN";
      break;

    case OPEN_DOT1Q_VLAN_TYPE_GVRP_REGISTERED:
      ret = "GVRP VLAN";
      break;

    case OPEN_DOT1Q_VLAN_TYPE_WS_L2_TNNL_REGISTERED:
      ret = "L2 Tunnel VLAN";
      break;

    case OPEN_DOT1Q_VLAN_TYPE_IP_INTERNAL_VLAN:  
      ret = "IP VLAN";
      break;

    case OPEN_DOT1Q_VLAN_TYPE_AUTHMGR_REGISTERED:
      ret = "Auth Manager/DOT1X VLAN";
      break;

    case OPEN_DOT1Q_VLAN_TYPE_MVRP_REGISTERED: 
      ret = "MVRP VLAN";
      break;

    case OPEN_DOT1Q_VLAN_TYPE_OPENFLOW_REGISTERED:
      ret = "OPENFLOW VLAN";
      break;

    case OPEN_DOT1Q_VLAN_TYPE_AUTO_VIDEO_REGISTERED:
      ret = "Auto-Video VLAN";
      break;

    case OPEN_DOT1Q_VLAN_TYPE_EVC_REGISTERED:
      ret = "EVC VLAN";
      break;

    case OPEN_DOT1Q_VLAN_TYPE_PBB_REGISTERED:
      ret = "PBB VLAN";
      break;

    default:
      break;
  }

  return ret;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for VLAN.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  uint32_t ifNum, i;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t partStatus;
  uint32_t vlanId, nextVlanId;
  uint32_t vlanFlag = 0;
  uint32_t vlanId1 = 0, vlanId2 = 0, vlanId3 = 0;
  OPEN_VLAN_LIST_t vidList;
  uint32_t val, numOfElements;
  OPEN_DOT1Q_VLAN_TYPE_t vlanType = 0;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("vlan_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting VLAN API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");
  printf("\n");
  /* exercise various OPEN API VLAN functions */
  do
  {
    /* find a VLAN ID that is not being used by the user configuration */
    for (vlanId = VLAN_ID_MIN; vlanId < VLAN_ID_MAX; vlanId++)
    {
      if (openapiVlanCreatedCheck(&clientHandle, vlanId) != OPEN_E_NONE)
      {
        /* found a VLAN that is not already configured */
        break;
      }
    }
    if (vlanId >= VLAN_ID_MAX)
    {
      printf("  Could not find an unused VLAN to configure.\n");
      break;
    }

    printf("Using VLAN ID %u.\n", vlanId);

    printf("\n");
    /* create and delete VLAN */
    printf("Testing VLAN create/delete/check APIs.\n");
    printf("Attempting to create VLAN %u.\n", vlanId);
    if (vlanCreateAndVerify(&clientHandle, vlanId) != OPEN_E_NONE)
    {
      printf("  Failure creating VLAN %u.\n", vlanId);
      break;
    }
    printf("  Successfully created VLAN %u.\n", vlanId);

    printf("Attempting to delete VLAN %u.\n", vlanId);
    if (vlanDeleteAndVerify(&clientHandle, vlanId) != OPEN_E_NONE)
    {
      printf("  Failure deleting VLAN %u.\n", vlanId);
      break;
    }
    printf("  Successfully deleted VLAN %u.\n", vlanId);

    /* program the PVID on interfaces */
    printf("\n");
    printf("Testing PVID set/get APIs.\n");
    printf("Attempting to create VLAN %u.\n", vlanId);
    if (vlanCreateAndVerify(&clientHandle, vlanId) != OPEN_E_NONE)
    {
      printf("  Failure creating VLAN %u.\n", vlanId);
      break;
    }
    printf("  Successfully created VLAN %u.\n", vlanId);

    printf("Attempting to set the first physical interface to participate in VLAN %u.\n", vlanId);
    if ((result = openapiIfFirstGet(&clientHandle, OPEN_INTF_TYPE_PHY, &ifNum)) != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get first physical interface. (result = %d)\n", result);
      break;
    }
    printf("Using physical interface with ifNum = %u.\n", ifNum);

    /* program the Frame Type */
    printf("\n");
    printf("Attempting to set Acceptable Frame Type for ifNum %u to type %d.\n", ifNum, OPEN_DOT1Q_ADMIN_ONLY_VLAN_UNTAGGED);
    if (interfaceAcceptFrameTypeSetAndVerify(&clientHandle, ifNum, OPEN_DOT1Q_ADMIN_ONLY_VLAN_UNTAGGED) != OPEN_E_NONE)
    {
      printf("  Failure setting Acceptable Frame Type for ifNum %u.\n", ifNum);
      break;
    }
    printf("  Successfully set Acceptable Frame Type for ifNum %u.\n", ifNum);

    /* program the PV ID */
    printf("\n");
    printf("Attempting to set PVID for interface to %u.\n", vlanId);
    if (interfacePvidSetAndVerify(&clientHandle, ifNum, vlanId) != OPEN_E_NONE)
    {
      printf("  Failure setting PVID for ifNum %u to VLAN %u.\n", ifNum, vlanId);
      break;
    }
    printf("  Successfully set PVID for ifNum %u to VLAN %u.\n", ifNum, vlanId);

    /* program the participation mode on interfaces */
    printf("\n");
    printf("Attempting to set participation mode for interface %u on VLAN %u to INCLUDE.\n", ifNum, vlanId);
    if (interfaceVlanParticipationSetAndVerify(&clientHandle, ifNum, vlanId, OPEN_VLAN_PARTICIPATION_MODE_INCLUDE) != OPEN_E_NONE)
    {
      printf("  Failure setting VLAN participation mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);
      break;
    }
    printf("  Successfully set VLAN participation mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);

    printf("Attempting to get the VLAN participation status.\n");
    for (i=0; i < RETRY_COUNT; i++)
    {
      if ((result = openapiVlanIfParticipationStatusGet(&clientHandle, vlanId, ifNum, &partStatus)) != OPEN_E_NONE)
      {
        /* error from API, move on */
        break;
      }
      else
      {
        if (partStatus != OPEN_ENABLE)
        {
          usleep(RETRY_INTERVAL_MSEC);
        }
        else
        {
          /* got what we want, move on */
          break;
        }
      }
    }

    if (result != OPEN_E_NONE)
    {
      printf("  Bad return code trying to get participation status. (result = %d)\n", result);
      break;
    }
    else
    {
      printf("  Successfully retrieved participation status for ifNum = %u. - ", ifNum);
      printf("This is %sas expected.\n", (partStatus == OPEN_ENABLE) ? "":"NOT ");
    }

    /* program the tagging mode on interfaces */
    printf("\n");
    printf("Attempting to set tagging mode for interface %u on VLAN %u to ENABLE.\n", ifNum, vlanId);
    if (interfaceVlanTaggingSetAndVerify(&clientHandle, ifNum, vlanId, OPEN_ENABLE) != OPEN_E_NONE)
    {
      printf("  Failure setting VLAN tagging mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);
      break;
    }
    printf("  Successfully set VLAN tagging mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);

    printf("Attempting to set tagging mode for interface %u on VLAN %u to DISABLE.\n", ifNum, vlanId);
    if (interfaceVlanTaggingSetAndVerify(&clientHandle, ifNum, vlanId, OPEN_DISABLE) != OPEN_E_NONE)
    {
      printf("  Failure setting VLAN tagging mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);
      break;
    }
    printf("  Successfully set VLAN tagging mode for ifNum %u on VLAN %u.\n", ifNum, vlanId);

    /* clean up */
    printf("\n");
    printf("Attempting to delete VLAN %u.\n", vlanId);
    if (vlanDeleteAndVerify(&clientHandle, vlanId) != OPEN_E_NONE)
    {
      printf("  Failure deleting VLAN %u.\n", vlanId);
      break;
    }
    printf("  Successfully deleted VLAN %u.\n", vlanId);

    /* program the default user priority on interfaces */
    printf("\n");
    printf("Attempting to set default priority for interface %u to %u.\n", ifNum, IF_PRIORITY);
    if (interfacePrioritySetAndVerify(&clientHandle, ifNum, IF_PRIORITY) != OPEN_E_NONE)
    {
      printf("  Failure setting default priority for interface %u.\n", ifNum);
      break;
    }
    printf("  Successfully set default priority for interface %u.\n", ifNum);

  } while (0);

  /* Testing the next vlan get */
  /*Checking for the vlans ither than default vlan */
  printf("\n");
  printf("Testing vlan get next API.\n");
  printf("Attempting to check if there are any vlans.\n");
  for (vlanId = VLAN_ID_MIN+1; vlanId < VLAN_ID_MAX; vlanId++)
  {
    if (openapiVlanCreatedCheck(&clientHandle, vlanId) == OPEN_E_NONE)
    {
      /* found a VLAN that is already configured */
      vlanFlag = 1;
      break;
    }
  }

  if (!vlanFlag)
  {
    printf("  Creating vlans as there are no vlans other than default vlan.\n");
    for (vlanId = VLAN_ID_MIN+1; vlanId <= 10; vlanId++)
    {
      if ((result = vlanCreateAndVerify(&clientHandle, vlanId)) != OPEN_E_NONE)
      {
        printf("  Bad return code trying to create VLAN %u. (result = %d)\n", vlanId, result);
        break;
      }
    }
  }

  result = openapiVlanNextGet(&clientHandle, 0, &nextVlanId);

  if (result == OPEN_E_NONE)
  {
    printf("  Vlans present: ");
  }
  else
  {
    printf("  Failed to get next vlan.(result = %d)\n", result);
  }

  while (result == OPEN_E_NONE)
  {
    printf(" %u", nextVlanId);
    result = openapiVlanNextGet(&clientHandle, nextVlanId, &nextVlanId);
  }
  printf("\n\n");

  if (!vlanFlag)
  {
    printf("Attempting to delete created vlans.\n");
    nextVlanId = VLAN_ID_MIN;
    while (openapiVlanNextGet(&clientHandle, nextVlanId, &nextVlanId) == OPEN_E_NONE)
    {
      if ((result = openapiVlanDelete(&clientHandle, nextVlanId)) != OPEN_E_NONE)
      {
        printf(" Bad return code trying to delete VLAN %u. (result = %d)\n", nextVlanId, result);
        break;
      }
    }
    if (result == OPEN_E_NONE)
    {
      printf("  Successfully deleted the created vlans.\n");
    }
  }

  /* Private VLANs */
  vlanId1 = 10;
  vlanId2 = 20;
  vlanId3 = 30;

  privateVlanCreateAndVerify(&clientHandle, vlanId1, vlanId2, vlanId3); 
 
  ifNum = 5;

  interfaceHostPrivateVlanCreateAndVerify(&clientHandle, ifNum, vlanId1, vlanId2);

  ifNum = 6;
  vidList.ids[0] = 20;
  vidList.ids[1] = 30;
  vidList.numEntries = 2;
  interfacePromiscPrivateVlanCreateAndVerify(&clientHandle, ifNum, vlanId1, vidList);

  /* Auto-Trunk related OpEN API tests */
  autoTrunkConfigAndVerify(&clientHandle);

  dot1qMaxSupportedVlansGet(&clientHandle, &val);

  mostVlanEntriesGet(&clientHandle, &val);

  staticVlanEntriesGet(&clientHandle, &val);

  dynamicVlanEntriesGet(&clientHandle, &val);

  dot1qVlanNumDeletesGet(&clientHandle, &val);

  vlanTaggedSetGet(&clientHandle, 1, 1, &val);

  nextVlanGet(&clientHandle, 1, &val);

  qportsEnableIngressFilteringGet(&clientHandle, 1, &val);

  qportsEnableIngressFilteringSet(&clientHandle, 1, 1);

  qportsEnableIngressFilteringGet(&clientHandle, 1, &val);

  dot1qPortIngressFiltering(&clientHandle, 1, &val);

  qportsCurrentIngressFilteringGet(&clientHandle, 1, &val);

  memset(&vidList, 0, sizeof(vidList));
  vlanVIDListGet(&clientHandle, 1, &vidList, &numOfElements);

  if (OPEN_E_NONE == openapiIfFirstGet(&clientHandle, OPEN_INTF_TYPE_PHY, &ifNum))
  {
    dot1qRxPacketsNotForwardedCountGet(&clientHandle, ifNum);
  }

  vlanType = 0;
  nextVlanId = 0;
  result = openapiVlanNextGet(&clientHandle, 0, &nextVlanId);
  if (OPEN_E_NONE == result)
  {
   printf("\r\nVLAN and VLAN Type:\r\n");
  }
  while (OPEN_E_NONE == result)
  {
    vlanType = 0;
    printf("  VLAN: %u\r\n", nextVlanId);
    if (OPEN_E_NONE == openapiVlanTypeGet(&clientHandle, nextVlanId, &vlanType))
    {
      printf("  VLAN Type: %s(%u)\r\n\r\n", vlanTypeStr(vlanType), vlanType);
    }
    result = openapiVlanNextGet(&clientHandle, nextVlanId, &nextVlanId);
  }
  printf("\r\n");

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping VLAN API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
