/*********************************************************************
*
*  Copyright 2022 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  vlan_mac_example.c
*
* @purpose   OpEN MAC VLAN example.
*
* @component OpEN
*
* @create    11/14/2022
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_vlan.h"

#define MAC_STRING_SIZE 18
/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the VLAN MAC example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage:  %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Add MAC association to VLAN: %s 1 macAddress vlanId\n", name);
  printf("Test 2: Delete MAC association to VLAN: %s 2 macAddress\n", name);
  printf("Test 3: Gets the vlan association of a given MAC: %s 3 macAddress>\n", name);
  printf("Test 4: Gets the next vlan mac association of a given MAC: %s 4 macAddress>\n", name);
  return;
}

/*********************************************************************
* @purpose  To add a mac to a VLAN.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    mac           @b{(input)}  Mac address
* @param    vlanID        @b{(input)}  Vlan id
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch
*
* @end
**********************************************************************/
void macVlanAddAssociation (openapiClientHandle_t *clientHandle,
                                    char *macAddr, uint32_t vlanId)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (macAddr);

  if (OPEN_E_NONE != (result = openapiVlanMacAdd (clientHandle, &bufId, vlanId)))
  {
    printf("Bad return code trying to add mac vlan association. (result = %d)\n", result);
  }
  else
  {
    printf ("MAC VLAN association added successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  To delete a mac vlan entry.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    mac           @b{(input)}  Mac address
*
* @returns  none
*
* @notes    Calling this API will change the running configuration of the switch
*
* @end
**********************************************************************/
void macVlanDelAssociation (openapiClientHandle_t *clientHandle, char *macAddr)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;
  uint32_t vlanId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (macAddr);

  if (OPEN_E_NONE != (result = openapiVlanMacGet (clientHandle, &bufId, &vlanId)))
  {
    printf("Bad return code trying to get mac vlan association for MAC %s . (result = %d)\n",
           macAddr, result);
  }
  else
  {
    if (OPEN_E_NONE != (result = openapiVlanMacDelete (clientHandle, &bufId, vlanId)))
    {
      printf("Bad return code trying to del mac vlan association. (result = %d)\n", result);
    }
    else
    {
      printf ("MAC VLAN association deleted successfully.\n");
    }
  }
  return;
}

/*********************************************************************
* @purpose  To get a mac vlan entry.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    mac           @b{(input)}  Mac address
*
* @returns  none
*
* @notes none
*
* @end
**********************************************************************/
void macVlanGetAssociation (openapiClientHandle_t *clientHandle, char *macAddr)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;
  uint32_t vlanId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (macAddr);

  if (OPEN_E_NONE != (result = openapiVlanMacGet (clientHandle, &bufId, &vlanId)))
  {
    printf("Bad return code trying to get mac vlan association for MAC %s . (result = %d)\n",
           macAddr, result);
  }
  else
  {
    printf ("VLAN ID associated to MAC %s is %u.\n", macAddr, vlanId);
  }
  return;
}

/*********************************************************************
* @purpose  To get a next mac vlan entry of a given MAC.
*
* @param    client_handle @b{(input)}  Client handle from registration API
* @param    mac           @b{(input)}  Mac address
*
* @returns  none
*
* @notes none
*
* @end
**********************************************************************/
void macVlanGetNextAssociation (openapiClientHandle_t *clientHandle, char *macAddr)
{
  open_error_t result;
  char mac[MAC_STRING_SIZE] = "";
  open_buffdesc bufId;
  char nextMac[MAC_STRING_SIZE] = "";
  open_buffdesc nextBufId = {};
  uint32_t vlanId;

	strncpy(mac, macAddr, MAC_STRING_SIZE-1);
  bufId.pstart = mac;
  bufId.size   = strlen (macAddr);

  nextBufId.pstart = nextMac;
  nextBufId.size   = MAC_STRING_SIZE;

  if (OPEN_E_NONE != (result = openapiVlanMacGetNext (clientHandle, &bufId, &nextBufId, &vlanId)))
  {
    printf("Bad return code trying to get mac vlan association for MAC %s . (result = %d)\n",
           macAddr, result);
  }
  else
  {
    printf ("Next MAC VLAN association of MAC %s is %s-%u.\n", macAddr, nextMac, vlanId);
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for MAC VLAN
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;

  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("MAC VLAN example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting MAC VLAN API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 4)
      {
        macVlanAddAssociation (&client_handle, argv[2], atoi(argv[3]));
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
        macVlanDelAssociation (&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 3)
      {
        macVlanGetAssociation (&client_handle, argv[2]);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
        macVlanGetNextAssociation (&client_handle, argv[2]);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping MAC VLAN API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
