/*********************************************************************
*
*  Copyright 2016-2023 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  voice_vlan_example.c
*
* @purpose   Voice VLAN OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    11/03/2016
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_voice_vlan.h"

#define OPENAPI_TITLE "Voice VLAN"
#define EXAMPLE_APP_NAME "voice_vlan_example"

typedef void exampleFunction_t(openapiClientHandle_t *clientHandle, int argc, char **argv);
typedef struct
{
  exampleFunction_t *function;
  char              *description;
  char              *parmSyntax;
} functionTableEntry_t;

void adminModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  int modeValue;
  OPEN_CONTROL_t mode;

  if (argc != 3)
  {
    printf("Mode parameter required.\n");
    return;
  }

  errno = 0;
  modeValue = strtol(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[2]);
    return;
  }

  switch (modeValue)
  {
    case 0:
      mode = OPEN_DISABLE;
      break;
    default:
      mode = OPEN_ENABLE;
      break;
  }

  result = openapiVoiceVlanAdminModeSet(clientHandle, mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s administrative mode: (result = %d)\n", 
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s administrative mode set: %s. (result = %d)\n",
           OPENAPI_TITLE, (mode == OPEN_ENABLE)?"Enable":"Disable", result);
  }
  return;
}

void adminModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;

  result = openapiVoiceVlanAdminModeGet(clientHandle, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s administrative mode: (result = %d)\n", 
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s administrative mode get: %s. (result = %d)\n",
           OPENAPI_TITLE, (mode == OPEN_ENABLE)?"Enable":"Disable", result);
  }
  return;
}

void portAdminModeSet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  int ifNum;
  int modeValue;
  int parmValue;

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtol(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  if (argc < 4)
  {
    printf("Mode parameter required.\n");
    return;
  }

  errno = 0;
  modeValue = strtol(argv[3], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid mode parameter: (%s)\n", argv[2]);
    return;
  }

  switch (modeValue)
  {
    case OPEN_VOICE_VLAN_CONFIG_DISABLE:
    case OPEN_VOICE_VLAN_CONFIG_NONE:
    case OPEN_VOICE_VLAN_CONFIG_UNTAGGED:
      break;
    case OPEN_VOICE_VLAN_CONFIG_VLAN_ID:
    case OPEN_VOICE_VLAN_CONFIG_DOT1P:
      if (argc < 5)
      {
        printf("VLAN_ID/DOT1P parameter required.\n");
        return;
      }
      errno = 0;
      parmValue = strtol(argv[4], NULL, 0);
      if (errno != 0)
      {
        printf("Invalid VLAN_ID/DOT1P parameter: (%s)\n", argv[2]);
        return;
      }
      break;
    default:
      printf("Invalid mode parameter: (%s)\n", argv[2]);
      return;
      /* early return */
  }

  result = openapiVoiceVlanPortAdminModeSet(clientHandle, ifNum, modeValue, parmValue);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to set %s port administrative mode: (result = %d)\n", 
           OPENAPI_TITLE, result);
  }
  else
  {
    printf("%s administrative mode set: %d. (result = %d)\n",
           OPENAPI_TITLE, modeValue, result);
  }
  return;
}

void portAdminModeGet(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  int ifNum;
  OPEN_VOICE_VLAN_CONFIG_TYPE_t mode;
  char *modeString;
  uint32_t parmValue;
  char parmString[64];

  if (argc < 3)
  {
    printf("Interface parameter required.\n");
    return;
  }

  errno = 0;
  ifNum = strtol(argv[2], NULL, 0);
  if (errno != 0)
  {
    printf("Invalid interface parameter: (%s)\n", argv[2]);
    return;
  }

  parmString[0] = 0;

  result = openapiVoiceVlanPortAdminModeGet(clientHandle, ifNum, &mode);
  if (OPEN_E_NONE != result)
  {
    printf("Bad return code trying to get %s port administrative mode: ifNum = %d (result = %d)\n", 
           OPENAPI_TITLE, ifNum, result);
  }
  else
  {
    switch (mode)
    {
      case OPEN_VOICE_VLAN_CONFIG_DISABLE:
        modeString = "DISABLE";
        break;
      case OPEN_VOICE_VLAN_CONFIG_VLAN_ID:
        modeString = "VLAN_ID";
        result = openapiVoiceVlanPortVlanIdGet(clientHandle, ifNum, &parmValue);
        if (result == OPEN_E_NONE)
        {
          snprintf(parmString, sizeof(parmString), " VLAN ID = %d", parmValue);
        }
        else
        {
          printf("Bad return code trying to get %s port VLAN ID: ifNum = %d (result = %d)\n", 
                 OPENAPI_TITLE, ifNum, result);
          return;
        }
        break;
      case OPEN_VOICE_VLAN_CONFIG_DOT1P:
        modeString = "DOT_1P";
        result = openapiVoiceVlanPortDot1pPriorityGet(clientHandle, ifNum, &parmValue);
        if (result == OPEN_E_NONE)
        {
          snprintf(parmString, sizeof(parmString), " DOT_1P priority = %d", parmValue);
        }
        else
        {
          printf("Bad return code trying to get %s port DOT_1P priority: ifNum = %d (result = %d)\n", 
                 OPENAPI_TITLE, ifNum, result);
          return;
        }
        break;
      case OPEN_VOICE_VLAN_CONFIG_NONE:
        modeString = "NONE";
        break;
      case OPEN_VOICE_VLAN_CONFIG_UNTAGGED:
        modeString = "UNTAGGED";
        break;
      default:
        printf("Invalid mode parameter: (%s)\n", argv[2]);
        return;
        /* early return */
    }
    printf("%s port administrative mode get: %s%s. (result = %d)\n",
           OPENAPI_TITLE, modeString, parmString, result);
  }
  return;
}

/*********************************************************************
* @purpose  Get the value of the cos override mode.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    mode          @b{(output)} Cos override mode
*
* @returns  none
*
* @end
*********************************************************************/
void voiceVlanPortCosOverrideModeGet(openapiClientHandle_t *client_handle, 
                                     uint32_t ifNum, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiVoiceVlanPortCosOverrideModeGet(client_handle, ifNum, mode))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the value of the cos override mode."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Cos override mode is %u.\n", *mode);
  }
  return;
}

/*********************************************************************
* @purpose  Set the value of the cos override mode.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    mode           @b{(input)}  Cos override mode
*
* @returns  none
*
* @end
*********************************************************************/
void voiceVlanPortCosOverrideModeSet(openapiClientHandle_t *client_handle,
                                     uint32_t ifNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiVoiceVlanPortCosOverrideModeSet(client_handle, ifNum, mode))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the value of the cos override mode."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Cos override mode set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get voice vlan Auth State.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    state          @b{(output)} Auth state
*
* @returns  none
*
* @end
*********************************************************************/
void voiceVlanPortAuthStateGet(openapiClientHandle_t *client_handle, 
                               uint32_t ifNum, OPEN_CONTROL_t *state)
{
  open_error_t result;

  if ((result = openapiVoiceVlanPortAuthStateGet(client_handle, ifNum, state))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get voice vlan Auth State. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Auth state is %u.\n", *state);
  }
  return;
}

/*********************************************************************
* @purpose  Set voice vlan Auth State.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
* @param    state           @b{(input)}  Auth state
*
* @returns  none
*
* @end
*********************************************************************/
void voiceVlanPortAuthStateSet(openapiClientHandle_t *client_handle, 
                               uint32_t ifNum, OPEN_CONTROL_t state)
{
  open_error_t result;

  if ((result = openapiVoiceVlanPortAuthStateSet(client_handle, ifNum, state))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set voice vlan Auth State. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Auth state set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the value of the dscp.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    ifNum          @b{(input)}  interface
* @param    dscp           @b{(output)} Dscp value
*
* @returns  none
*
* @end
*********************************************************************/
void voiceVlanPortDscpGet(openapiClientHandle_t *client_handle, 
                          uint32_t ifNum, uint32_t *dscp)
{
  open_error_t result;

  if ((result = openapiVoiceVlanPortDscpGet(client_handle, ifNum, dscp))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the value of the dscp. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Voice vlan port dscp value is %u\n", *dscp);
  }
  return;
}

/*********************************************************************
* @purpose  Set the value of the dscp.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
* @param    dscp            @b{(input)}  Dscp value
*
* @returns  none
*
* @end
*********************************************************************/
void voiceVlanPortDscpSet(openapiClientHandle_t *client_handle, 
                          uint32_t ifNum, uint32_t dscp)
{
  open_error_t result;

  if ((result = openapiVoiceVlanPortDscpSet(client_handle, ifNum, dscp))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the value of the dscp."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Voice vlan port dscp value set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Set the value of the dot1p.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    ifNum               @b{(input)}  interface
* @param    dot1pPriority       @b{(input)}  Dot1p priority value
*
* @returns  none
*
* @end
*********************************************************************/
void voiceVlanPortDot1pPrioritySet(openapiClientHandle_t *client_handle, 
                                   uint32_t ifNum, uint32_t dot1pPriority)
{
  open_error_t result;

  if ((result = openapiVoiceVlanPortDot1pPrioritySet(client_handle, ifNum,
                                                     dot1pPriority)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the value of the dot1p. (result = %d)\n",
           result);
  }
  else
  {
    printf("Voice vlan port dot1p set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Set the value of the vlan ID.
*
* @param    client_handle @b{(input)}  client handle from registration API
* @param    ifNum         @b{(input)}  interface
* @param    vlanId        @b{(input)}  Vlan ID
*
* @returns  none
*
* @end
*********************************************************************/
void voiceVlanPortVlanIdSet(openapiClientHandle_t *client_handle, 
                            uint32_t ifNum, uint32_t vlanId)
{
  open_error_t result;

  if ((result = openapiVoiceVlanPortVlanIdSet(client_handle, ifNum, vlanId))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the value of the vlan ID. "
           "(result = %d)\n", result);
  }
  else
  {
    printf("Voice vlan port vlan set successfully.\n");
  }
  return;
}

/*********************************************************************
* @purpose  Get the value voice vlan status.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    ifNum           @b{(input)}  interface
* @param    mode            @b{(output)} Voice vlan status
*
* @returns  none
*
* @end
*********************************************************************/
void voiceVlanPortStatusGet(openapiClientHandle_t *client_handle, 
                            uint32_t ifNum, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiVoiceVlanPortStatusGet(client_handle, ifNum, mode))
      != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the value voice vlan status."
           " (result = %d)\n", result);
  }
  else
  {
    printf("Voice vlan port status is %u\n", *mode);
  }
  return;
}

void openapiSanity(openapiClientHandle_t *clientHandle, int argc, char **argv)
{
  open_error_t result;
  OPEN_CONTROL_t mode;
  uint32_t ifNum;
  OPEN_VOICE_VLAN_CONFIG_TYPE_t portMode;
  uint32_t parm;

  printf("\nTesting openapiVoiceVlanAdminModeGet(): \n");

  result = openapiVoiceVlanAdminModeGet(NULL, &mode);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiVoiceVlanAdminModeGet(clientHandle, &mode);
  printf(" %s -- Admin Mode Get: %s (%d). (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);

  printf("\nTesting openapiVoiceVlanAdminModeSet(): \n");

  result = openapiVoiceVlanAdminModeSet(NULL, OPEN_ENABLE);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  mode = OPEN_ENABLE;
  result = openapiVoiceVlanAdminModeSet(clientHandle, mode);
  printf(" %s -- Admin Mode Set: %s (%d). (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);
  mode = OPEN_DISABLE;
  result = openapiVoiceVlanAdminModeSet(clientHandle, mode);
  printf(" %s -- Admin Mode Set: %s (%d). (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         mode==OPEN_ENABLE?"Enable":"Disable", mode, result);

  printf("\nTesting openapiVoiceVlanPortAdminModeGet(): \n");

  ifNum = 1;
  result = openapiVoiceVlanPortAdminModeGet(NULL, ifNum, &portMode);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiVoiceVlanPortAdminModeGet(clientHandle, ifNum, &portMode);
  printf(" %s -- Port Admin Mode Get: %d. (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         portMode, result);

  printf("\nTesting openapiVoiceVlanPortVlanIdGet(): \n");
  result = openapiVoiceVlanPortVlanIdGet(NULL, ifNum, &parm);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiVoiceVlanPortVlanIdGet(clientHandle, ifNum, &parm);
  printf(" %s -- Port VLAN Id Get: %d. (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         parm, result);

  printf("\nTesting openapiVoiceVlanPortDot1pPriorityGet(): \n");
  result = openapiVoiceVlanPortDot1pPriorityGet(NULL, ifNum, &parm);
  printf(" %s -- NULL Client Handle. (result = %d)\n", (OPEN_E_PARAM==result)?"PASSED":"FAILED", result);

  result = openapiVoiceVlanPortDot1pPriorityGet(clientHandle, ifNum, &parm);
  printf(" %s -- Port DOT1P priority Get: %d. (result = %d)\n",
         (OPEN_E_NONE==result)?"PASSED":"FAILED",
         parm, result);

}

functionTableEntry_t functionTable[] =
{
  {adminModeSet,        "Set "OPENAPI_TITLE" global administrative mode", "<mode: 0-off/1-on>"},
  {adminModeGet,        "Get "OPENAPI_TITLE" global administrative mode", ""},
  {portAdminModeSet,    "Set "OPENAPI_TITLE" port mode", "<interface> <mode> <val> "
                                                         "\n\t\t(mode: 0/disable, 1/VLAN ID, 2/DOT1P, 3/NONE, 4/UNTAGGED)"
                                                         "\n\t\tModes 1 and 2 require parameter for VLAN ID or DOT1p priority"},
  {portAdminModeGet,    "Get "OPENAPI_TITLE" port mode", ""},
  {openapiSanity,       OPENAPI_TITLE" OpEN APIs Sanity", ""},
};
#define NUMBER_OF_FUNCTIONS sizeof(functionTable)/sizeof(functionTable[0])

void printAppMenu()
{
  int i;

  printf("\nUsage: %s <test#> <arg1> <arg2> ... \n\n", EXAMPLE_APP_NAME);
  for (i = 0; i < NUMBER_OF_FUNCTIONS; i++)
  {
    printf("\tTest %d: %s:  %s %d %s\n",
           i, functionTable[i].description, EXAMPLE_APP_NAME, i, functionTable[i].parmSyntax);
  }
  printf("\tTest 5: Get the value of the cos override mode: voice_vlan_example 5 <interface>\n");
  printf("\tTest 6: Set the value of the cos override mode: voice_vlan_example 6 <interface> <mode>\n");
  printf("\tTest 7: Get voice vlan Auth State: voice_vlan_example 7 <interface>\n");
  printf("\tTest 8: Set voice vlan Auth State: voice_vlan_example 8 <interface> <state>\n");
  printf("\tTest 9: Get the value of the dscp: voice_vlan_example 9 <interface> \n");
  printf("\tTest 10: Set the value of the dscp: voice_vlan_example 10 <interface> <dscp>\n");
  printf("\tTest 11: Set the value of the dot1p: voice_vlan_example 11 <interface> <dot1pPriority>\n");
  printf("\tTest 12: Set the value of the vlan ID: voice_vlan_example 12 <interface> <vlanId>\n");
  printf("\tTest 13: Get the value voice vlan status: voice_vlan_example 13 <interface>\n");
  printf("\n");

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum, ifNum, val;
  open_buffdesc buffDesc;
  char switch_os_revision_string[100];
  open_revision_data_t openApiVersion;
  OPEN_CONTROL_t mode;
  int  show_help = 1;

  if (argc < 2)
  {
    printAppMenu();
    exit(1);
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister(""OPENAPI_TITLE" OpEN API example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting %s OpEN API example application", OPENAPI_TITLE);

  printf("\n");
  buffDesc.pstart = switch_os_revision_string;
  buffDesc.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &buffDesc) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  if (openapiApiVersionGet(&clientHandle, &openApiVersion) == OPEN_E_NONE)
    printf("OpEN API Version = %d.%d.%d.%d\n",
           openApiVersion.release,
           openApiVersion.version,
           openApiVersion.maint_level,
           openApiVersion.build_num);
  else
    printf("OpEN API Version retrieve error\n");

  printf("\n");

  testNum = atoi(argv[1]);

  if (testNum < NUMBER_OF_FUNCTIONS)
  {
    functionTable[testNum].function(&clientHandle, argc, argv);
  }
  else if ((testNum > 4) && (testNum <= 13))
  {
    if (argc > 2)
    {
      ifNum = atoi(argv[2]);  
    }
    switch (testNum)
    {
      case 5:
        if (argc == 3)
        {
          voiceVlanPortCosOverrideModeGet(&clientHandle, ifNum, &mode);
          show_help = 0;
        }
        break;
      case 6:
        if (argc == 4)
        {
          voiceVlanPortCosOverrideModeSet(&clientHandle, ifNum, atoi(argv[3]));
          show_help = 0;
        }
        break;
      case 7:
        if (argc == 3)
        {
          voiceVlanPortAuthStateGet(&clientHandle, ifNum, &mode);
          show_help = 0;
        }
        break;
      case 8:
        if (argc == 4)
        {
          voiceVlanPortAuthStateSet(&clientHandle, ifNum, atoi(argv[3]));
          show_help = 0;
        }
        break;
      case 9:
        if (argc == 3)
        {
          voiceVlanPortDscpGet(&clientHandle, ifNum, &val);
          show_help = 0;
        }
        break;
      case 10:
        if (argc == 4)
        {
          voiceVlanPortDscpSet(&clientHandle, ifNum, atoi(argv[3]));
          show_help = 0;
        }
        break;
      case 11:
        if (argc == 4)
        {
          voiceVlanPortDot1pPrioritySet(&clientHandle, ifNum, atoi(argv[3]));
          show_help = 0;
        }
        break;
      case 12:
        if (argc == 4)
        {
          voiceVlanPortVlanIdSet(&clientHandle, ifNum, atoi(argv[3]));
          show_help = 0;
        }
        break;
      case 13:
        if (argc == 3)
        {
          voiceVlanPortStatusGet(&clientHandle, ifNum, &mode);
          show_help = 0;
        }
        break;

      default:
        break;
    }

    if (show_help == 1)
    {
      printAppMenu();
    }
  }
  else
  {
    printAppMenu();
    exit(1);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping %s OpEN API example application", OPENAPI_TITLE);
        
  return 0;
}
