/**************************************************************************************
*
*  Copyright Broadcom Corporation 2012-2018
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_bridge.h
*
* @purpose   BRIDGE support
*
* @component OPEN
*
* @create    06/02/2015
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_BRIDGE_H_INCLUDED
#define OPENAPI_BRIDGE_H_INCLUDED

#include "openapi_common.h"

#define OPEN_STP_PROTOCOL_SPEC_UNKNOWN      1
#define OPEN_STP_PROTOCOL_SPEC_IEEE8021D    3

/* string representation of MAC in "XX:XX:XX:XX:XX:XX" format including EOL symbol */
#define OPEN_STP_DOT1D_DESIGNATED_ROOT_LEN  18

#define OPEN_FDB_ADDR_LEN                   8
#define OPEN_FDB_ADDR_STR_LEN               ((OPEN_FDB_ADDR_LEN * 2) + (OPEN_FDB_ADDR_LEN - 1))

/* Spanning Tree Port States */
#define OPEN_DOT1D_BLOCKING           1
#define OPEN_DOT1D_LISTENING          2
#define OPEN_DOT1D_LEARNING           3
#define OPEN_DOT1D_FORWARDING         4
#define OPEN_DOT1D_DISABLED           5
#define OPEN_DOT1D_MANUAL_FWD         6
#define OPEN_DOT1D_NOT_PARTICIPATE    7
#define OPEN_NULLPTR ((void *)0)

/**************************************************************************************
*
* @purpose Get Internal Interface Number of next valid port.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    intIfNum            @b{(input)}  starting interface.
* @param    nextIntIfNum        @b{(output)} next interface.
* 
* @returns  OPEN_E_NONE        If the next interface was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dBasePortEntryNextGet(openapiClientHandle_t *client_handle, 
                                               uint32_t intIfNum,
                                               uint32_t *nextIntIfNum);

/***************************************************************************************************
* @purpose  Get the unique MAC address used to refer to the bridge.
*
* @param    client_handle  @b{(input)}   client handle from registration API.
* @param    bridgeAddress  @b{(output)}  the pointer to the MAC address.
*
* @returns  OPEN_E_NONE        If the unique MAC address was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @comments It is recommended that this be the numerically smallest MAC
*           address of all ports that belong to this bridge.  However
*           it is only required to be unique.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dBaseBridgeAddressGet(openapiClientHandle_t *client_handle, 
                                              open_buffdesc *bridgeAddress);

/**************************************************************************************
* @purpose Get number of ports controlled by the bridging entity.
*
* @param    client_handle  @b{(input)}   client handle from registration API.
* @param    numPorts       @b{(output)}  number of ports.
*
* @returns  OPEN_E_NONE   If the counter was successfully found and stored in the space provided.       
* @returns  OPEN_E_FAIL   Returned values are invalid.
* @returns  OPEN_E_PARAM  Error in parameter passed. 
* @returns  OPEN_E_ERROR  If the counter requested does not exist.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dBaseNumPortsGet(openapiClientHandle_t *client_handle, 
                                         uint32_t *numPorts);

/**************************************************************************************
*
* @purpose Get dot1d Base type of bridging.
*
* @param    client_handle  @b{(input)}   client handle from registration API.
* @param    type           @b{(output)}  type of bridging.
* 
* @returns  OPEN_E_NONE   If the counter was successfully found and stored in the space provided.       
* @returns  OPEN_E_FAIL   Returned values are invalid.
* @returns  OPEN_E_PARAM  Error in parameter passed. 
* @returns  OPEN_E_ERROR   If the counter requested does not exisit.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dBaseTypeGet(openapiClientHandle_t *client_handle, uint32_t *type);

/**************************************************************************************
*
* @purpose Get admin state of the port.
*
* @param    client_handle  @b{(input)}   client handle from registration API.
* @param    port           @b{(input)}   internal interface number.
* 
* @returns  OPEN_E_NONE   If the port of bridge was successfully found and stored in the space provided.       
* @returns  OPEN_E_FAIL   Returned values are invalid.
* @returns  OPEN_E_PARAM  Error in parameter passed. 
* @returns  OPEN_E_ERROR   If the counter requested does not exisit.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dBasePortEntryGet(openapiClientHandle_t *client_handle, 
                                          uint32_t port); 

/**************************************************************************************
*
* @purpose Get name of an object instance unique to this port.
* 
* @param    client_handle  @b{(input)}   client handle from registration API.
* @param    port           @b{(input)}   internal interface number.
* @param    buffer         @b{(output)}  data buffer.
* 
*
* @returns  OPEN_E_NONE        If the name of an object instance was successfully found.           
* @returns  OPEN_E_FAIL        Returned values are invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @comments This function is used for a port which (potentially) has
*           the same value of dot1dBasePortIfIndex as another port on
*           the same bridge.
* 
* @supported-in-version OpEN API Version: 1.7
* 
* @end
*********************************************************************/
open_error_t openapiDot1dBasePortCircuitGet(openapiClientHandle_t *client_handle, 
                                            uint32_t port, open_buffdesc *buffer);

/**************************************************************************************
*
* @purpose Get ifIndex associated with the internal interface number.
*
* @param    client_handle  @b{(input)}   client handle from registration API.
* @param    port           @b{(input)}   internal interface number.
* @param    ifIndex        @b{(output)}  pointer to ifIndex.
* 
* @returns  OPEN_E_NONE   If the counter was successfully found and stored in the space provided.       
* @returns  OPEN_E_FAIL   Returned values are invalid.
* @returns  OPEN_E_PARAM  Error in parameter passed. 
* @returns  OPEN_E_ERROR   If the counter requested does not exisit.
* 
* @comments This function is used for a port which (potentially) has
*           the same value of dot1dBasePortIfIndex as another port on
*           the same bridge.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dBasePortIfIndexGet(openapiClientHandle_t *client_handle, 
                                            uint32_t port, uint32_t *ifIndex); 

/**************************************************************************************
*
* @purpose Check if an interface is valid for DOT1D.
*
* @param    client_handle  @b{(input)}  client handle from registration API.
* @param    intIfNum       @b{(input)}  internal interface number.
* 
* @returns  OPEN_E_NONE        If the interface is valid.            
* @returns  OPEN_E_FAIL        Returned values are invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dIsValidIntf(openapiClientHandle_t *client_handle, uint32_t intIfNum); 

/**************************************************************************************
*
* @purpose Get number of frames discarded by this port due to delay through the bridge.
*
* @param    client_handle              @b{(input)}  client handle from registration API.
* @param    port                       @b{(input)}  port internal interface number.
* @param    delayExceededDiscardsValue @b{(output)} delay exceeded value.
* 
* @returns  OPEN_E_NONE        If the number of frames was successfully found.           
* @returns  OPEN_E_FAIL        Returned values are invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @comments It is incremented by both transparent and source route bridges.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t 
openapiDot1dBasePortDelayExceededDiscardsGet(openapiClientHandle_t *client_handle,
                                             uint32_t port, 
                                             uint32_t *delayExceededDiscardsValue);

/**************************************************************************************
*
* @purpose Get number of frames discarded by this port due to delay through the bridge.
*
* @param    client_handle              @b{(input)}  client handle from registration API.
* @param    port                       @b{(input)}  port internal interface number.
* @param    mtuExceededDiscardsValue   @b{(output)} mtu exceededDiscards value.
* 
* @returns  OPEN_E_NONE        If the number of frames was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @comments It is incremented by both transparent and source route bridges.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dBasePortMtuExceededDiscardsGet(openapiClientHandle_t *client_handle,
                                                        uint32_t port, 
                                                        uint32_t *mtuExceededDiscardsValue);

/**************************************************************************************
*
* @purpose Get indication of what version of the Spanning Tree Protocol is being run.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} spanning tree protocol version.
* 
* @returns  OPEN_E_NONE        If the indication version of STP was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpProtocolSpecificationGet(openapiClientHandle_t *client_handle,
                                                     uint32_t *value);

/**************************************************************************************
*
* @purpose Get the Spanning Tree Bridge priority.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} bridge priority.
* 
* @returns  OPEN_E_NONE        If the STP bridge priority was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpPriorityGet(openapiClientHandle_t *client_handle,
                                        uint32_t *value);

/**************************************************************************************
*
* @purpose Get the time since last topology change was detected by the bridge.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} time since last spanning tree topology change.
* 
* @returns  OPEN_E_NONE        If the time was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpTimeSinceTopologyChangeGet(openapiClientHandle_t *client_handle,
                                                       uint64_t *value);

/**************************************************************************************
*
* @purpose Get the STP change count.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} spanning tree topology change count.
* 
* @returns  OPEN_E_NONE        If the STP change count was successfully found.          
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpTopChangesGet(openapiClientHandle_t *client_handle,
                                          uint32_t *value);

/**************************************************************************************
*
* @purpose Get Spanning Tree Designated root for the switch.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    buffer              @b{(output)} STP designated root for the switch.
* 
* @returns  OPEN_E_NONE        If the STP designated root was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @comments This value is used as the Root Identifier parameter in all
*           Configuration Bridge PDUs originated by this node.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpDesignatedRootGet(openapiClientHandle_t *client_handle,
                                              open_buffdesc *buffer);

/**************************************************************************************
*
* @purpose Get the cost of the path to the root as seen from this bridge.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} root cost for switch.
* 
* @returns  OPEN_E_NONE        If the root cost for switch was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpRootCostGet(openapiClientHandle_t *client_handle, 
                                        uint32_t *value);

/**************************************************************************************
*
* @purpose Get port number with lowest cost path from bridge to the root bridge.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} root port for the switch.
* 
* @returns  OPEN_E_NONE        If the port number with lowest cost was successfully found.            
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpRootPortGet(openapiClientHandle_t *client_handle, 
                                        uint32_t *value);

/**************************************************************************************
*
* @purpose Get Spanning Tree max age.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} spanning tree max age.
* 
* @returns  OPEN_E_NONE        If the max age was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @comments Time in hundredths of a second.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpMaxAgeGet(openapiClientHandle_t *client_handle, 
                                      uint32_t *value);

/**************************************************************************************
*
* @purpose Get Spanning Tree hello time.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} spanning tree hello time.
* 
* @returns  OPEN_E_NONE        If the hello time was successfully found.            
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @comments Time in hundredths of a second.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpHelloTimeGet(openapiClientHandle_t *client_handle, 
                                         uint32_t *value);

/**************************************************************************************
*
* @purpose Get Spanning Tree hold time.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} spanning tree hold time.
* 
* @returns  OPEN_E_NONE        If the hold time was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @comments Units of hundredths of a second.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpHoldTimeGet(openapiClientHandle_t *client_handle, 
                                        uint32_t *value);

/**************************************************************************************
*
* @purpose Get Spanning Tree forward delay.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} spanning tree forward delay.
* 
* @returns  OPEN_E_NONE        If the forward delay was successfully found.            
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpForwardDelayGet(openapiClientHandle_t *client_handle, 
                                            uint32_t *value);

/**************************************************************************************
*
* @purpose Get Spanning Tree bridge max age.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} spanning tree bridge max age.
* 
* @returns  OPEN_E_NONE        If the bridge max age was successfully found.              
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @comments The granularity of this timer is specified by 802.1D-1990 to be 1 second.
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpBridgeMaxAgeGet(openapiClientHandle_t *client_handle,
                                            uint32_t *value);

/**************************************************************************************
*
* @purpose Get Spanning Tree Bridge Hello Time.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} spanning tree bridge hello time.
* 
* @returns  OPEN_E_NONE        If the bridge hello time was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpBridgeHelloTimeGet(openapiClientHandle_t *client_handle,
                                               uint32_t *value); 

/**************************************************************************************
*
* @purpose Get Spanning Tree Bridge Forward Delay.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} spanning tree bridge forward delay.
* 
* @returns  OPEN_E_NONE        If the bridge forward delay was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpBridgeForwardDelayGet(openapiClientHandle_t *client_handle,
                                                  uint32_t *value); 

/**************************************************************************************
*
* @purpose Get Spanning Tree Port Priority.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  internal interface number, physical or logical.
* @param    value               @b{(output)} port priority.
* 
* @returns  OPEN_E_NONE        If the port priority was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpPortPriorityGet(openapiClientHandle_t *client_handle, 
                                            uint32_t port, uint32_t *value);

/**************************************************************************************
*
* @purpose Determine if the interface is the currently configured STP instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    intIfNum            @b{(input)}  interface to remove.
* @param    isInInstance        @b{(output)} port priority.
* 
* @returns  OPEN_E_NONE        If the port priority was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dIsInterfaceInCurrentInstance(openapiClientHandle_t *client_handle,
                                                      uint32_t intIfNum, 
                                                      uint32_t *isInInstance);

/**************************************************************************************
*
* @purpose Get next interface for current instance.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    intIfNum            @b{(input)}  starting interface.
* @param    nextIntIfNum        @b{(output)} next interface.
* 
* @returns  OPEN_E_NONE        If the next interface was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dInstanceMemberGetNext(openapiClientHandle_t *client_handle, 
                                               uint32_t intIfNum,
                                               uint32_t *nextIntIfNum);

/**************************************************************************************
*
* @purpose Get Spanning Tree Port State.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  internal interface number, physical or logical.
* @param    value               @b{(output)} spanning tree port state.
* 
* @returns  OPEN_E_NONE        If the port state was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpPortStateGet(openapiClientHandle_t *client_handle, 
                                         uint32_t port, uint32_t *value);

/**************************************************************************************
*
* @purpose Get Spanning Tree Path Cost for this port.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  internal interface number, physical or logical.
* @param    value               @b{(output)} spanning tree port path cost.
* 
* @returns  OPEN_E_NONE        If the port path cost was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpPortPathCostGet(openapiClientHandle_t *client_handle, 
                                            uint32_t port, uint32_t *value);

/**************************************************************************************
*
* @purpose Get Spanning Tree Designated Root for the switch.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  internal interface number, physical or logical.
* @param    buffer              @b{(output)} spanning tree designated root for the switch.
* 
* @returns  OPEN_E_NONE        If the designated root was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpPortDesignatedRootGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t port, open_buffdesc *buffer);

/**************************************************************************************
*
* @purpose Get Spanning Tree Designated Cost for this port.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  internal interface number, physical or logical.
* @param    value               @b{(output)} spanning tree designated cost for this port.
* 
* @returns  OPEN_E_NONE        If the designated cost was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpPortDesignatedCostGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t port, uint32_t *value);

/**************************************************************************************
*
* @purpose Get Spanning Tree Designated Bridge for this port.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  internal interface number, physical or logical.
* @param    buffer              @b{(output)} spanning tree designated bridge for this port.
* 
* @returns  OPEN_E_NONE        If the designated bridge was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpPortDesignatedBridgeGet(openapiClientHandle_t *client_handle, 
                                                    uint32_t port, open_buffdesc *buffer);

/**************************************************************************************
*
* @purpose Get Spanning Tree Designated port ID.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  internal interface number, physical or logical.
* @param    value               @b{(output)} spanning tree port id for this port.
* 
* @returns  OPEN_E_NONE        If the port id was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpPortDesignatedPortGet(openapiClientHandle_t *client_handle, 
                                                  uint32_t port, uint16_t *value);

/**************************************************************************************
*
* @purpose Get number of times this port has transitioned from Learning state to Forwarding state.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  internal interface number, physical or logical.
* @param    value               @b{(output)} number of times port has transitioned from Learning to Forwarding.
* 
* @returns  OPEN_E_NONE        If the number of times port was successfully found.             
* @returns  OPEN_E_UNAVAIL     Feature is not supported.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dStpPortForwardTransitionsGet(openapiClientHandle_t *client_handle, 
                                                      uint32_t port, uint32_t *value);

/**************************************************************************************
*
* @purpose Get number of Forwarding Database entries that have been discarded
*          due to a lack of space to store them.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} number of discarded entries.
* 
* @returns  OPEN_E_NONE        If the number of discarded entries was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are not invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dTpLearnedEntryDiscardsGet(openapiClientHandle_t *client_handle,
                                                   uint32_t *value);

/**************************************************************************************
*
* @purpose Get timeout period for aging out dynamically learned forwarding information.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(output)} timeout period.
* 
* @returns  OPEN_E_NONE        If the timeout period was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are not invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @comments In seconds 
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDbDot1dTpAgingTimeGet(openapiClientHandle_t *client_handle, 
                                          uint32_t *value);

/**************************************************************************************
*
* @purpose Set timeout period for aging out dynamically learned forwarding information.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    value               @b{(input)}  timeout period.
* 
* @returns  OPEN_E_NONE        If the timeout period was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are not invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
* 
* @comments In seconds 
* 
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDbDot1dTpAgingTimeSet(openapiClientHandle_t *client_handle, 
                                          uint32_t value);

/**************************************************************************************
*
* @purpose Get info about a specific unicast MAC address which the bridge
*          has forwarding and/or filtering information.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    address             @b{(input)}  MAC address.
* 
* @returns  OPEN_E_NONE        If the info about MAC was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are not invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dTpFdbEntryNextGet(openapiClientHandle_t *client_handle, 
                                           open_buffdesc *address);

/**************************************************************************************
*
* @purpose Get info about a specific unicast MAC address which the bridge
*          has forwarding and/or filtering information.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    address             @b{(input)}  MAC address.
* 
* @returns  OPEN_E_NONE        If the info about MAC was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are not invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dTpFdbEntryGet(openapiClientHandle_t *client_handle, 
                                       open_buffdesc *address);

/**************************************************************************************
*
* @purpose Returns the port number.
*
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    address             @b{(output)}  MAC address.
* @param    value               @b{(output)}  port number.
* 
* @returns  OPEN_E_NONE        If the port number was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are not invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
* 
* @comments " Either the value of '0', or the port number of the port
*             on which a frame having a source address equal to the
*             value of the corresponding instance of dot1dTpFdbAddress
*             has been seen. A value of '0' indicates that the port
*             number has not been learned by that the bridge does
*             have some forwarding/filtering information about this
*             address (e.g. in the dot1dStaticTable). Implementers
*             are encouraged to assign the port value to this
*             object whenever it is learned even for addresses
*             for which the corresponding value of dot1dTpFdbStatus
*             is not learned(3)."
* 
* @end
***************************************************************************************/
open_error_t openapiDot1dTpFdbPortGet(openapiClientHandle_t *client_handle, 
                                      open_buffdesc *address, uint32_t *value);

/**************************************************************************************
*
* @purpose Get the status of the dot1dTpFdbEntry.
*
* @param    client_handle       @b{(input)}   client handle from registration API.
* @param    address             @b{(output)}  MAC address.
* @param    value               @b{(output)}  port status.
* 
* @returns  OPEN_E_NONE        If the status was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are not invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
* 
* @comments Status types:
*               other(1)
*               invalid(2)
*               learned(3)
*               self(4)
*               mgmt(5)
*
*           " The status of this entry.  The meaning of the values are:
* 
*             other(1): none of the following.  This would include the
*                       case where some other MIB object (not
*                       corresponding instance of dot1dTpFdbPort, nor
*                       an entry in the dot1dStaticTable) is being
*                       used to determine if and how frames addressed
*                       to the value of the corresponding instance of
*                       dot1dTpFdbAddress are being forwarded.
*
*            invalid(2): this entry is no longer valid (e.g. it was
*                        learned but has since aged-out), but has not
*                        yet been flushed from the table.
*
*            learned(3): the value of the corresponding instance
*                        of dot1dTpFdbPort was learned and is being
*                        used.
*
*            self(4):    the value of the corresponding instance of
*                        dot1dTpFdbAddress represents one of the
*                        bridge's addresses.  The corresponding
*                        instance of dot1dTpFdbPort indicates
*                        which of the bridge's ports has this address.
*
*            mgmt(5):    the value of this corresponding instance of
*                        dot1dTpFdbAddress is also the value of an
*                        existing instance of dot1dStaticAddress."
* 
* @end
***************************************************************************************/
open_error_t openapiDot1dTpFdbStatusGet(openapiClientHandle_t *client_handle, 
                                        open_buffdesc *address, uint32_t *value);

/**************************************************************************************
*
* @purpose Get maximum size of the INFO field this port will receive or transmit.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  internal interface number, physical or logical.
* @param    value               @b{(output)} maximum size.
* 
* @returns  OPEN_E_NONE        If the number of times port was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are not invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dTpPortMaxInfoGet(openapiClientHandle_t *client_handle, 
                                          uint32_t port, uint32_t *value);

/**************************************************************************************
*
* @purpose Get number of frames received by this port from its segment.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  interface number.
* @param    value               @b{(output)} number of frames received.
* 
* @returns  OPEN_E_NONE        If the number of times port was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are not invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dTpPortInFramesGet(openapiClientHandle_t *client_handle, 
                                           uint32_t port, uint32_t *value);

/**************************************************************************************
*
* @purpose Get number of frames transmitted by this port to its segment.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  interface number.
* @param    value               @b{(output)} number of frames received.
* 
* @returns  OPEN_E_NONE        If the number of times port was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are not invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dTpPortOutFramesGet(openapiClientHandle_t *client_handle, 
                                            uint32_t port, uint32_t *value);

/**************************************************************************************
*
* @purpose Get count of valid frames received which were
*          discarded by the Forwarding Process.
*
* @param    client_handle       @b{(input)}  client handle from registration API.
* @param    port                @b{(input)}  interface number.
* @param    value               @b{(output)} number of valid frames received.
* 
* @returns  OPEN_E_NONE        If the number of valid frames was successfully found.             
* @returns  OPEN_E_FAIL        Returned values are not invalid.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.7
*
* @end
***************************************************************************************/
open_error_t openapiDot1dTpPortInDiscardsGet(openapiClientHandle_t *client_handle, 
                                             uint32_t port, uint32_t *value);

#endif /* OPENAPI_BRIDGE_H_INCLUDED */
