/**********************************************************************
*
*  Copyright 2020-2021 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_capability_licensing.h
*
* @purpose   Capability Lisensing feature
*
* @component OPEN
*
* @create    03/31/2020
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_CAPABILITY_LICENSING_H_INCLUDED
#define OPENAPI_CAPABILITY_LICENSING_H_INCLUDED

#include "openapi_common.h" 

#define OPEN_CAPAB_LIC_MAX_LICENSE_FILES      16
#define OPEN_CAPAB_LIC_MAX_VER_LEN            6
#define OPEN_CAPAB_LIC_MAX_KEY_LEN            32
#define OPEN_CAPAB_LIC_MAX_MODEL_LEN          32
#define OPEN_CAPAB_LIC_MAX_SERIAL_LEN         64
#define OPEN_CAPAB_LIC_MAX_FILE_PARAM_VAL_LEN    64

/* License status code */
typedef enum
{
  OPEN_CAPAB_LICENSE_STATUS_FILE_NOT_PRESENT = 1,  /**< License file is not present */
  OPEN_CAPAB_LICENSE_STATUS_VALID,                 /**< Valid license file after installation */
  OPEN_CAPAB_LICENSE_STATUS_APPLIED,               /**< License applied successfully */
  OPEN_CAPAB_LICENSE_STATUS_REMOVED,               /**< License file was applied but got removed */
  OPEN_CAPAB_LICENSE_STATUS_INVALID,               /**< Invalid license */
  OPEN_CAPAB_LICENSE_STATUS_UNSUPPORTED,           /**< Unsupported capability/features */
  OPEN_CAPAB_LICENSE_STATUS_DUPLICATE,             /**< Duplicate license */
  OPEN_CAPAB_LICENSE_STATUS_ERROR,                 /**< Other Error */
} OPEN_CAPAB_LICENSE_STATUS_TYPE_t;

/* License type */
typedef enum
{
  OPEN_CAPAB_LICENSE_TYPE_UNKNOWN = 1,
  OPEN_CAPAB_LICENSE_TYPE_PORT,         /**< Port license */
  OPEN_CAPAB_LICENSE_TYPE_FEATURE,      /**< Feature license */
  OPEN_CAPAB_LICENSE_TYPE_PORT_FEATURE, /**< Port and Feature license */
} OPEN_CAPAB_LICENSE_TYPE_t;

typedef enum
{
  OPEN_CAPAB_LICENSE_FILE_PARAM_COPY = 1,  /**< License copy */
  OPEN_CAPAB_LICENSE_FILE_PARAM_DATE,      /**< License Date */
  OPEN_CAPAB_LICENSE_FILE_PARAM_DESC,      /**< License description */
} OPEN_CAPAB_LICENSE_FILE_PARAM_TYPE_t;


/*****************************************************************************
 * @purpose   Get the number of license files in the system
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     numOfLicenses      @b{(output)} Number of license(s) present in the system.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Some platforms/systems allow more than one license file. This API gives the number of license files present in the system.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseCountGet(openapiClientHandle_t *client_handle, 
                                         uint32_t *numOfLicenses);

/*****************************************************************************
 * @purpose   Get the first license index in the system.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     firstLicenseIndex  @b{(output)} First license index in the system.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Each License file in the system has specific index. This API gets the first valid index in the system.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseFirstIndexGet(openapiClientHandle_t *client_handle, 
                                              uint32_t *firstLicenseIndex);

/*****************************************************************************
 * @purpose   Get the next license index in the system.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     CurrentIndex       @b{(input)}  Current license index.
 * @param     nextLicenseIndex   @b{(output)} Next valid license index in the system.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     Each License file in the system has specific index. This API gets the next valid index in the system.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseNextIndexGet(openapiClientHandle_t *client_handle, 
                                             uint32_t currentIndex,
                                             uint32_t *nextLicenseIndex);

/*****************************************************************************
 * @purpose   Get the type of license.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     licenseIndex       @b{(input)}  License index.
 * @param     type               @b{(output)} Type of the license.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the type of the license with licenseIndex.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseTypeGet(openapiClientHandle_t *client_handle, 
                                        uint32_t  licenseIndex, 
                                        OPEN_CAPAB_LICENSE_TYPE_t *type);

/*****************************************************************************
 * @purpose   Get the status of license in the system.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     licenseIndex       @b{(input)}  License index.
 * @param     status             @b{(output)} Status of the license.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the present status of the license with licenseIndex.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseStatusGet(openapiClientHandle_t *client_handle, 
                                          uint32_t  licenseIndex, 
                                          OPEN_CAPAB_LICENSE_STATUS_TYPE_t *status);

/*****************************************************************************
 * @purpose   Get the parameter value of license file in the system.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     licenseIndex       @b{(input)}  License index.
 * @param     paramType          @b{(input)}  Parameter type.
 * @param     paramValue         @b{(output)} Parameter value associated with the parameter type.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the parameter value of license file, it takes, the type of the parameter and returns its value.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseParamGet(openapiClientHandle_t *client_handle, 
                                         uint32_t  licenseIndex, 
                                         OPEN_CAPAB_LICENSE_FILE_PARAM_TYPE_t paramType, 
                                         open_buffdesc *paramValue);

/*****************************************************************************
 * @purpose   Get the version number of license file in the system.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     licenseIndex       @b{(input)}  License index.
 * @param     version            @b{(output)} Version number of the license file.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the version number of license file.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseVersionGet(openapiClientHandle_t *client_handle, 
                                           uint32_t  licenseIndex, 
                                           open_buffdesc *version);

/*****************************************************************************
 * @purpose   Get the key of license file in the system.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     licenseIndex       @b{(input)}  License index.
 * @param     version            @b{(output)} Key of the license file.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the key of license file.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseKeyGet(openapiClientHandle_t *client_handle, 
                                       uint32_t  licenseIndex, 
                                       open_buffdesc *version);

/*****************************************************************************
 * @purpose   Get the model name of license file.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     licenseIndex       @b{(input)}  License index.
 * @param     model              @b{(output)} Model of the license file.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the model name of license file.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseModelGet(openapiClientHandle_t *client_handle, 
                                         uint32_t  licenseIndex, 
                                         open_buffdesc *model);

/*****************************************************************************
 * @purpose   Get the serial number of license file.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     licenseIndex       @b{(input)}  License index.
 * @param     serialNum          @b{(output)} Serial number of the license file.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API gets the serial number license file.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseSerialNumberGet(openapiClientHandle_t *client_handle, 
                                                uint32_t  licenseIndex, 
                                                open_buffdesc *serialNum);


/*****************************************************************************
 * @purpose   Remove a license file.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 * @param     licenseIndex       @b{(input)}  License index.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API removes a license file.
 *
 * @supported-in-version OpEN API Version: 1.21
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseLicenseRemove(openapiClientHandle_t *client_handle, 
                                              uint32_t  licenseIndex); 

/*****************************************************************************
 * @purpose   Remove all license files in the System.
 *
 * @param     client_handle      @b{(input)}  client handle from registration API
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     This API removes all the license files present in the System.
 *
 * @supported-in-version OpEN API Version: 1.21
 *
 * @end
 *****************************************************************************/
open_error_t openapiCapabLicenseLicenseRemoveAll(openapiClientHandle_t *client_handle);

#endif /* OPENAPI_CAPABILITY_LICENSING_H_INCLUDED */
