/**********************************************************************
*
*  Copyright 2019 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename  openapi_dot1cb.h
*
* @purpose   This code implements the OpEN Dot1cb API
*
* @component OPEN
*
* @create    05/29/2019
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_DOT1CB_H_INCLUDED
#define OPENAPI_DOT1CB_H_INCLUDED

#include "openapi_common.h" 

#define OPEN_MAX_DOT1CB_VLANS            16   /**< maximum VLANs enabled for 802.1CB concurrently */
#define OPEN_MAX_DOT1CB_IPV4             512  /**< maximum IPv4 classifiers for 802.1CB flows */
#define OPEN_MAX_DOT1CB_IPV6             512  /**< maximum IPv6 classifiers for 802.1CB flows */
#define OPEN_MIN_DOT1CB_INITIATOR_FLOWS  1    /**< minimum flow ID for 802.1CB initiator flows */
#define OPEN_MAX_DOT1CB_INITIATOR_FLOWS  1024 /**< maximum flow ID for 802.1CB initiator flows */
#define OPEN_MIN_DOT1CB_TRANSIT_FLOWS    1    /**< minimum flow ID for 802.1CB transit flows */
#define OPEN_MAX_DOT1CB_TRANSIT_FLOWS    1024 /**< maximum flow ID for 802.1CB transit flows */
#define OPEN_MFDB_MAX_MAC_ENTRIES        1024 /**< maximum MFDB classifiers for 802.1CB */

#define OPEN_DOT1CB_IPV4_PREFIX_LEN  32
#define OPEN_DOT1CB_IPV6_PREFIX_LEN  128

/* 802.1CB flow types */
typedef enum
{
  OPEN_DOT1CB_FLOW_TYPE_NONE      = 0,  /**< none */
  OPEN_DOT1CB_FLOW_TYPE_INITIATOR = 1,  /**< initiator */
  OPEN_DOT1CB_FLOW_TYPE_TRANSIT   = 2,  /**< transit */
} OPEN_DOT1CB_FLOW_TYPE_t;

/* 802.1CB global statistics. These counters are aggregated
   across all VLANs enabled in the 802.1CB mode.
*/
typedef struct 
{
  uint64_t arp_to_cpu;             /**< number of ARP requests and replies sent to CPU */
  uint64_t bcast_to_cpu;           /**< number of broadcast packets sent to CPU */
  uint64_t mcast_to_cpu;           /**< number of multicast packets sent to CPU */
  uint64_t ucast_dropped;          /**< number of unicast packets dropped */
} OPEN_DOT1CB_GLOBAL_STATS_t;

/* 802.1CB statistics on physical ports enabled for R-Tag mode */
typedef struct
{
  uint64_t rtag_packets_rx;        /**< number of received R-Tagged packets */ 
  uint64_t rtag_packets_accepted;  /**< number of received R-Tagged packets not dropped */ 
  uint64_t rtag_packets_dups;      /**< number of duplicate R-Tagged packets dropped */
  uint64_t rtag_packets_tx;        /**< number of transmitted R-Tagged packets */
} OPEN_DOT1CB_INTF_STATS_t;

/*****************************************************************************
 * @purpose   Get the 802.1CB mode for a given VLAN.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     vlanID            @b{(input)}  VLAN identifier
 * @param     -                 Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID
 * @param     -                 Maximum value @ref OPENAPI_DOT1Q_MAX_VLAN_ID
 * @param     mode              @b{(output)} configured 802.1cb mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbModeGet(openapiClientHandle_t *client_handle,
                                  uint32_t vlanID, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the 802.1CB mode for a given VLAN.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     vlanID            @b{(input)}  VLAN identifier
 * @param     -                 Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID
 * @param     -                 Maximum value @ref OPENAPI_DOT1Q_MAX_VLAN_ID
 * @param     mode              @b{(input)}  802.1cb mode to be set
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbModeSet(openapiClientHandle_t *client_handle,
                                  uint32_t vlanID, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the 802.1CB ARP/NDP forwarding mode for a given VLAN.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     vlanID            @b{(input)}  VLAN identifier
 * @param     -                 Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID
 * @param     -                 Maximum value @ref OPENAPI_DOT1Q_MAX_VLAN_ID
 * @param     mode              @b{(output)} configured 802.1cb ARP/NDP forwarding mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbForwardArpNdpModeGet(openapiClientHandle_t *client_handle,
                                               uint32_t vlanID, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the 802.1CB ARP/NDP forwarding mode for a given VLAN.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     vlanID            @b{(input)}  VLAN identifier
 * @param     -                 Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID
 * @param     -                 Maximum value @ref OPENAPI_DOT1Q_MAX_VLAN_ID
 * @param     mode              @b{(input)}  802.1cb ARP/NDP forwarding mode to be set
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbForwardArpNdpModeSet(openapiClientHandle_t *client_handle,
                                               uint32_t vlanID, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Get the configured flow ID.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     buf               @b{(input)}  static MAC address for the filter 
 *                              in one of the following formats:
 *                              "xx:xx:xx:xx:xx:xx" or 
 *                              "xx-xx-xx-xx-xx-xx" or 
 *                              "xxxx.xxxx.xxxx"
 *
 * @param     vlanID            @b{(input)}  VLAN ID for the filter
 * @param     -                 Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID
 * @param     -                 Maximum value @ref OPENAPI_DOT1Q_MAX_VLAN_ID
 * @param     flowId            @b{(output)} configured 802.1cb flow ID
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterFlowIdGet(openapiClientHandle_t *client_handle,
                                    open_buffdesc *buf, uint32_t vlanID,
                                    uint32_t *flowId);

/*****************************************************************************
 * @purpose   Get the configured flow type.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     buf               @b{(input)}  static MAC address for the filter 
 *                              in one of the following formats:
 *                              "xx:xx:xx:xx:xx:xx" or 
 *                              "xx-xx-xx-xx-xx-xx" or 
 *                              "xxxx.xxxx.xxxx"
 *
 * @param     vlanID            @b{(input)}  VLAN ID for the filter
 * @param     -                 Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID
 * @param     -                 Maximum value @ref OPENAPI_DOT1Q_MAX_VLAN_ID
 * @param     flowType          @b{(output)} configured 802.1cb flow type
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterFlowTypeGet(openapiClientHandle_t *client_handle,
                                      open_buffdesc *buf, uint32_t vlanID,
                                      OPEN_DOT1CB_FLOW_TYPE_t *flowType);

/*****************************************************************************
 * @purpose   Set the flow type and ID of an existing macfilter.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     buf               @b{(input)}  static MAC address for the filter 
 *                              in one of the following formats:
 *                              "xx:xx:xx:xx:xx:xx" or 
 *                              "xx-xx-xx-xx-xx-xx" or 
 *                              "xxxx.xxxx.xxxx"
 *
 * @param     vlanID            @b{(input)}  VLAN ID for the filter
 * @param     -                 Minimum value @ref OPENAPI_DOT1Q_MIN_VLAN_ID
 * @param     -                 Maximum value @ref OPENAPI_DOT1Q_MAX_VLAN_ID
 * @param     flowType          @b{(input)}  802.1cb flow type to be set
 * @param     flowId            @b{(input)}  802.1cb flow ID to be set
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiFilterFlowTypeIdSet(openapiClientHandle_t *client_handle,
                                        open_buffdesc *buf, uint32_t vlanID,
                                        OPEN_DOT1CB_FLOW_TYPE_t flowType, uint32_t flowId);

/*****************************************************************************
 * @purpose   Get the 802.1CB RTAG mode for the specified interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     mode              @b{(output)} configured RTAG mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbRtagModeGet(openapiClientHandle_t *client_handle,
                                      uint32_t ifNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the 802.1CB RTAG mode for the specified interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     mode              @b{(input)}  RTAG mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbRtagModeSet(openapiClientHandle_t *client_handle,
                                      uint32_t ifNum, OPEN_CONTROL_t mode);

/*****************************************************************************
 * @purpose   Clear all the 802.1CB statistics.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbStatsClear(openapiClientHandle_t *client_handle);

/*****************************************************************************
 * @purpose   Get the 802.1CB global statistics.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     stat              @b{(output)} 802.1CB global statistics
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes     Statistics are computed as follows: current = latest - initial
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbGlobalStatsGet(openapiClientHandle_t *client_handle,
                                         OPEN_DOT1CB_GLOBAL_STATS_t *stat);

/*****************************************************************************
 * @purpose   Get the 802.1CB statistics for the specified interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     -                 Maximum value @ref OPENAPI_DOT1CB_INTF_MAX_COUNT
 * @param     stat              @b{(output)} 802.1CB interface statistics
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes     Statistics are computed as follows: current = latest - initial
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIntfStatsGet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum, OPEN_DOT1CB_INTF_STATS_t *stat);

/*****************************************************************************
 * @purpose   Get the next valid 802.1CB interface.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     ifNum             @b{(input)}  interface
 * @param     -                 Maximum value @ref OPENAPI_DOT1CB_INTF_MAX_COUNT
 * @param     nextIfNum         @b{(output)} pointer to next interface
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIntfNextGet(openapiClientHandle_t *client_handle,
                                      uint32_t ifNum, uint32_t *nextIfNum);

/*****************************************************************************
 * @purpose   Add an IPv4 classifier.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     srcAddr           @b{(input)}  source IP address
 * @param     srcMaskLen        @b{(input)}  source IP mask length
 * @param     dstAddr           @b{(input)}  destination IP address
 * @param     dstMaskLen        @b{(input)}  destination IP mask length
 * @param     flowType          @b{(input)}  flow type
 * @param     flowId            @b{(input)}  flow identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 * @returns   OPEN_E_EXISTS           Already exists.
 * @returns   OPEN_E_FULL             Maximum count reached or table full.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv4ClassifierAdd(openapiClientHandle_t *client_handle,
                                            open_inet_addr_t *srcAddr, uint32_t srcMaskLen,
                                            open_inet_addr_t *dstAddr, uint32_t dstMaskLen,
                                            OPEN_DOT1CB_FLOW_TYPE_t flowType, uint32_t flowId);

/*****************************************************************************
 * @purpose   Delete an IPv4 classifier.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     srcAddr           @b{(input)}  source IP address
 * @param     srcMaskLen        @b{(input)}  source IP mask length
 * @param     dstAddr           @b{(input)}  destination IP address
 * @param     dstMaskLen        @b{(input)}  destination IP mask length
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 * @returns   OPEN_E_NOT_FOUND        Parameter not found.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv4ClassifierDelete(openapiClientHandle_t *client_handle,
                                               open_inet_addr_t *srcAddr, uint32_t srcMaskLen,
                                               open_inet_addr_t *dstAddr, uint32_t dstMaskLen);

/*****************************************************************************
 * @purpose   Get an IPv4 classifier.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     srcAddr           @b{(input)}  source IP address
 * @param     srcMaskLen        @b{(input)}  source IP mask length
 * @param     dstAddr           @b{(input)}  destination IP address
 * @param     dstMaskLen        @b{(input)}  destination IP mask length
 * @param     flowType          @b{(output)} flow type
 * @param     flowId            @b{(output)} flow identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 * @returns   OPEN_E_NOT_FOUND        Parameter not found.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv4ClassifierGet(openapiClientHandle_t *client_handle,
                                            open_inet_addr_t *srcAddr, uint32_t srcMaskLen,
                                            open_inet_addr_t *dstAddr, uint32_t dstMaskLen,
                                            OPEN_DOT1CB_FLOW_TYPE_t *flowType, uint32_t *flowId);

/*****************************************************************************
 * @purpose   Get the next IPv4 classifier.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     srcAddr           @b{(inout)}  source IP address
 * @param     srcMaskLen        @b{(inout)}  source IP mask length
 * @param     dstAddr           @b{(inout)}  destination IP address
 * @param     dstMaskLen        @b{(inout)}  destination IP mask length
 * @param     flowType          @b{(output)} flow type
 * @param     flowId            @b{(output)} flow identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 * @returns   OPEN_E_NOT_FOUND        Parameter not found.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv4ClassifierNextGet(openapiClientHandle_t *client_handle,
                                                open_inet_addr_t *srcAddr, uint32_t *srcMaskLen,
                                                open_inet_addr_t *dstAddr, uint32_t *dstMaskLen,
                                                OPEN_DOT1CB_FLOW_TYPE_t *flowType, uint32_t *flowId);

/*****************************************************************************
 * @purpose   Get the number of configured IPv4 classifiers.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     value             @b{(output)} number of IPv4 classifiers
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv4ClassifierCountGet(openapiClientHandle_t *client_handle,
                                                 uint32_t *value);

/*****************************************************************************
 * @purpose   Get the status flag indicating whether IPv4 classifiers are installed in the hardware.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     value             @b{(output)} 1-Installed, 0-Not Installed
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv4ClassifierInHardwareGet(openapiClientHandle_t *client_handle,
                                                      uint32_t *value);

/*****************************************************************************
 * @purpose   Add an IPv6 classifier.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     srcAddr           @b{(input)}  source IP address
 * @param     srcMaskLen        @b{(input)}  source IP mask length
 * @param     dstAddr           @b{(input)}  destination IP address
 * @param     dstMaskLen        @b{(input)}  destination IP mask length
 * @param     flowType          @b{(input)}  flow type
 * @param     flowId            @b{(input)}  flow identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 * @returns   OPEN_E_EXISTS           Already exists.
 * @returns   OPEN_E_FULL             Maximum count reached or table full.
 * @returns   OPEN_E_UNAVAIL          IPv6 feature not supported.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv6ClassifierAdd(openapiClientHandle_t *client_handle,
                                            open_in6_addr_t *srcAddr, uint32_t srcMaskLen,
                                            open_in6_addr_t *dstAddr, uint32_t dstMaskLen,
                                            OPEN_DOT1CB_FLOW_TYPE_t flowType, uint32_t flowId);

/*****************************************************************************
 * @purpose   Delete an IPv6 classifier.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     srcAddr           @b{(input)}  source IP address
 * @param     srcMaskLen        @b{(input)}  source IP mask length
 * @param     dstAddr           @b{(input)}  destination IP address
 * @param     dstMaskLen        @b{(input)}  destination IP mask length
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 * @returns   OPEN_E_NOT_FOUND        Parameter not found.
 * @returns   OPEN_E_UNAVAIL          IPv6 feature not supported.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv6ClassifierDelete(openapiClientHandle_t *client_handle,
                                               open_in6_addr_t *srcAddr, uint32_t srcMaskLen,
                                               open_in6_addr_t *dstAddr, uint32_t dstMaskLen);

/*****************************************************************************
 * @purpose   Get an IPv6 classifier.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     srcAddr           @b{(input)}  source IP address
 * @param     srcMaskLen        @b{(input)}  source IP mask length
 * @param     dstAddr           @b{(input)}  destination IP address
 * @param     dstMaskLen        @b{(input)}  destination IP mask length
 * @param     flowType          @b{(output)} flow type
 * @param     flowId            @b{(output)} flow identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 * @returns   OPEN_E_NOT_FOUND        Parameter not found.
 * @returns   OPEN_E_UNAVAIL          IPv6 feature not supported.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv6ClassifierGet(openapiClientHandle_t *client_handle,
                                            open_in6_addr_t *srcAddr, uint32_t srcMaskLen,
                                            open_in6_addr_t *dstAddr, uint32_t dstMaskLen,
                                            OPEN_DOT1CB_FLOW_TYPE_t *flowType, uint32_t *flowId);

/*****************************************************************************
 * @purpose   Get the next IPv6 classifier.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     srcAddr           @b{(inout)}  source IP address
 * @param     srcMaskLen        @b{(inout)}  source IP mask length
 * @param     dstAddr           @b{(inout)}  destination IP address
 * @param     dstMaskLen        @b{(inout)}  destination IP mask length
 * @param     flowType          @b{(output)} flow type
 * @param     flowId            @b{(output)} flow identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 * @returns   OPEN_E_NOT_FOUND        Parameter not found.
 * @returns   OPEN_E_UNAVAIL          IPv6 feature not supported.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv6ClassifierNextGet(openapiClientHandle_t *client_handle,
                                                open_in6_addr_t *srcAddr, uint32_t *srcMaskLen,
                                                open_in6_addr_t *dstAddr, uint32_t *dstMaskLen,
                                                OPEN_DOT1CB_FLOW_TYPE_t *flowType, uint32_t *flowId);

/*****************************************************************************
 * @purpose   Get the number of configured IPv6 classifiers.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     value             @b{(output)} number of IPv6 classifiers
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 * @returns   OPEN_E_UNAVAIL          IPv6 feature not supported.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv6ClassifierCountGet(openapiClientHandle_t *client_handle,
                                                 uint32_t *value);

/*****************************************************************************
 * @purpose   Get the status flag indicating whether IPv6 classifiers are installed in the hardware.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     value             @b{(output)} 1-Installed, 0-Not Installed
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 * @returns   OPEN_E_UNAVAIL          IPv6 feature not supported.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbIpv6ClassifierInHardwareGet(openapiClientHandle_t *client_handle,
                                                      uint32_t *value);

/*****************************************************************************
 * @purpose   Get the number of transit flows installed in the hardware.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     value             @b{(output)} number of transit flows
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbNumActiveTransitFlowsGet(openapiClientHandle_t *client_handle,
                                                   uint32_t *value);

/*****************************************************************************
 * @purpose   Get the number of initiator flows installed in the hardware.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     value             @b{(output)} number of initiator flows
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbNumActiveInititatorFlowsGet(openapiClientHandle_t *client_handle,
                                                      uint32_t *value);

/*****************************************************************************
 * @purpose   Get the number of MFDB entries associated with flows.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     value             @b{(output)} number of MFDB entries
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbNumMfdbClassifiersGet(openapiClientHandle_t *client_handle,
                                                uint32_t *value);

/*****************************************************************************
 * @purpose   Get the number of 802.1CB enabled VLANs.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     value             @b{(output)} number of VLANs enabled for 802.1CB
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbApiNumDot1cbVlansGet(openapiClientHandle_t *client_handle,
                                               uint32_t *value);

/*****************************************************************************
 * @purpose   Get the next 802.1CB enabled VLAN.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     vid               @b{(inout)}  VLAN identifier
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes     Input VLAN value 0 to get the first 802.1CB enabled VLAN.
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbApiDot1cbEnabledVlanNextGet(openapiClientHandle_t *client_handle,
                                                      uint32_t *vid);

/*****************************************************************************
 * @purpose   Get the 802.1CB initiator flow and its status/statistics.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     flowId            @b{(input)}  initiator flow identifier
 * @param     classifiers       @b{(output)} number of classifiers using this flow
 * @param     nextTxSequence    @b{(output)} next sequence number in the RTAG
 * @param     txPackets         @b{(output)} number of packets sent by the switch that match this flow
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbApiInitFlowGet(openapiClientHandle_t *client_handle,
                                         uint32_t flowId, uint32_t *classifiers,
                                         uint32_t *nextTxSequence, uint64_t *txPackets);

/*****************************************************************************
 * @purpose   Get the next 802.1CB initiator flow and its status/statistics.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     flowId            @b{(inout)}  initiator flow identifier
 * @param     classifiers       @b{(output)} number of classifiers using this flow
 * @param     nextTxSequence    @b{(output)} next sequence number in the RTAG
 * @param     txPackets         @b{(output)} number of packets sent by the switch that match this flow
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbApiInitFlowNextGet(openapiClientHandle_t *client_handle,
                                             uint32_t *flowId, uint32_t *classifiers,
                                             uint32_t *nextTxSequence, uint64_t *txPackets);

/*****************************************************************************
 * @purpose   Get the 802.1CB transit flow and its status/statistics.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     flowId            @b{(input)}  transit flow identifier
 * @param     classifiers       @b{(output)} number of classifiers using this flow
 * @param     lastRxSequence    @b{(output)} last accepted sequence number on this flow
 * @param     matchingPackets   @b{(output)} number of packets received by the switch that match this flow
 * @param     acceptedPackets   @b{(output)} number of packets received by the switch that match this flow and are not discarded
 * @param     duplicatePackets  @b{(output)} number of duplicate packets received by the switch that match this flow and are discarded
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbApiTransitFlowGet(openapiClientHandle_t *client_handle,
                                            uint32_t flowId, uint32_t *classifiers,
                                            uint32_t *lastRxSequence, uint64_t *matchingPackets,
                                            uint64_t *acceptedPackets, uint64_t *duplicatePackets);

/*****************************************************************************
 * @purpose   Get the next 802.1CB transit flow and its status/statistics.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     flowId            @b{(inout)}  transit flow identifier
 * @param     classifiers       @b{(output)} number of classifiers using this flow
 * @param     lastRxSequence    @b{(output)} last accepted sequence number on this flow
 * @param     matchingPackets   @b{(output)} number of packets received by the switch that match this flow
 * @param     acceptedPackets   @b{(output)} number of packets received by the switch that match this flow and are not discarded
 * @param     duplicatePackets  @b{(output)} number of duplicate packets received by the switch that match this flow and are discarded
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbApiTransitFlowNextGet(openapiClientHandle_t *client_handle,
                                                uint32_t *flowId, uint32_t *classifiers,
                                                uint32_t *lastRxSequence, uint64_t *matchingPackets,
                                                uint64_t *acceptedPackets, uint64_t *duplicatePackets);

/*****************************************************************************
 * @purpose   Get the 802.1CB transit flow configuration.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     flowId            @b{(input)}  transit flow identifier
 * @param     historyWindow     @b{(output)} history window size
 * @param     acceptanceWindow  @b{(output)} acceptance window size
 * @param     ageTime           @b{(output)} age time in 10ms ticks
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbApiTransitFlowConfigGet(openapiClientHandle_t *client_handle,
                                                  uint32_t flowId, uint32_t *historyWindow,
                                                  uint32_t *acceptanceWindow, uint32_t *ageTime);

/*****************************************************************************
 * @purpose   Set the history window size for an 802.1CB transit flow.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     flowId            @b{(input)}  transit flow identifier
 * @param     historyWindow     @b{(input)}  history window size in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbApiTransitFlowHistoryWindowSizeSet(openapiClientHandle_t *client_handle,
                                                             uint32_t flowId, uint32_t historyWindow);

/*****************************************************************************
 * @purpose   Set the acceptance window size for an 802.1CB transit flow.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     flowId            @b{(input)}  transit flow identifier
 * @param     acceptanceWindow  @b{(input)}  acceptance window size in bytes
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbApiTransitFlowAcceptanceWindowSizeSet(openapiClientHandle_t *client_handle,
                                                                uint32_t flowId, uint32_t acceptanceWindow);

/*****************************************************************************
 * @purpose   Set the age time for an 802.1CB transit flow.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     flowId            @b{(input)}  transit flow identifier
 * @param     ageTime           @b{(input)}  age time in 10ms ticks
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbApiTransitFlowAgeTimeSet(openapiClientHandle_t *client_handle,
                                                   uint32_t flowId, uint32_t ageTime);

/*****************************************************************************
 * @purpose   Get the SA-MAC/VLAN flow association mode.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     mode              @b{(output)} SA-MAC/VLAN flow association mode
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbSaModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Set the SA-MAC/VLAN flow association mode.
 *
 * @param     client_handle     @b{(input)}  client handle from registration API
 * @param     mode              @b{(input)}  802.1cb mode to be set
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 * @returns   OPEN_E_INTERNAL         Internal error.
 *
 * @notes      
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiDot1cbSaModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode);

#endif /* OPENAPI_DOT1CB_H_INCLUDED */
