/*********************************************************************
*
*  Copyright 2016-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_image.h
*
* @purpose   Image Management and Configuration
*
* @component OpEN
*
* @create    11/13/2015
*
* @end
*                                    
**********************************************************************/
#ifndef OPENAPI_IMAGE_H_INCLUDED
#define OPENAPI_IMAGE_H_INCLUDED

#include "openapi_common.h"

#define OPEN_UBOOT_VERSION_LENGTH  17 /* UBOOT_VERSION_LENGTH + 1 */ 

#define OPEN_IMAGE_NAME_NONE "none"

/*********************************************************************
* @purpose  Updates the bootcode from the currently activated image.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    unit             @b{(input)}  Unit number
* @param    -                             Minimum value 1  
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* 
* @returns  OPEN_E_NONE      if validation is successful.
* @returns  OPEN_E_PARAM     if parameters is not defined correctly.
* @returns  OPEN_E_UNAVAIL   Not supported for this platform.
*
* @notes    This API is valid for the management unit in a stacking build. 
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiBootCodeUpdate(openapiClientHandle_t *client_handle,
                                   uint32_t              unit);

/*********************************************************************
* @purpose  Sets the specified image as the active image for the subsequent re-boots.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    unit             @b{(input)}  Unit number
* @param    -                             Minimum value 1  
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    imageLabel       @b{(input)}  Image label name
* @param    -                             Minimum length 1  
* @param    -                             Maximum value openapiImageFileNameMaxLengthGet()
* @param    updateBootCode   @b{(input)}  @ref OPEN_TRUE to activate, else @ref OPEN_FALSE
* 
* @returns  OPEN_E_NONE      if text set successful.
* @returns  OPEN_E_FAIL      if text set failed.
* @returns  OPEN_E_PARAM     if parameters is not defined correctly.
* @returns  OPEN_E_NOT_FOUND if image not found.
* @returns  OPEN_E_UNAVAIL   if the specified image is not the backup image
* @returns  OPEN_E_EXISTS    if the image is already activated
*
* @notes 
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiImageActivate(openapiClientHandle_t *client_handle,
                                  uint32_t              unit,
                                  open_buffdesc         *imageLabel,
                                  OPEN_BOOL_t           updateBootCode);

/*********************************************************************
* @purpose  Deletes the specified image.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    unit             @b{(input)}  Unit number
* @param    -                             Minimum value 1  
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    imageLabel       @b{(input)}  Image label name
* @param    -                             Minimum length 1  
* @param    -                             Maximum value openapiImageFileNameMaxLengthGet()
* 
* @returns  OPEN_E_NONE      if text set successful.
* @returns  OPEN_E_FAIL      if text set failed.
* @returns  OPEN_E_PARAM     if parameters is not defined correctly.
* @returns  OPEN_E_NOT_FOUND if image not found.
* @returns  OPEN_E_UNAVAIL   if the specified image is active/activated.
* 
* @notes    The current active image is never deleted.
* @notes    The activated backup image is never deleted.
* @Notes    Some products with dual image capability permit the user to delete the backup image only.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiImageDelete(openapiClientHandle_t *client_handle,
                                uint32_t              unit,
                                open_buffdesc         *imageLabel);

/*********************************************************************
* @purpose  Get the text description for a given image.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    imageLabel       @b{(input)}  Image label name
* @param    -                             Buffer size should be openapiImageFileNameMaxLengthGet()
* @param    imageDescr       @b{(output)} Associated text
* @param    -                             Buffer size should be @ref OPENAPI_DEFAULT_IMAGE_DESCR_MAX
*
* @returns  OPEN_E_NONE      if text get successful.
* @returns  OPEN_E_FAIL      if text get failed.
* @returns  OPEN_E_PARAM     if parameters is not defined correctly.
* @returns  OPEN_E_NOT_FOUND if image not found.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiImageDescrGet(openapiClientHandle_t *client_handle,
                                  open_buffdesc         *imageLabel,
                                  open_buffdesc         *imageDescr); 

/*********************************************************************
* @purpose  Set the text description for a given image.
*
* @param    client_handle    @b{(input)} Client handle from registration API
* @param    imageLabel       @b{(input)} Image label name
* @param    -                            Minimum length 1  
* @param    -                            Maximum value openapiImageFileNameMaxLengthGet()
* @param    imageDescr       @b{(input)} Associated text
* @param    -                            Minimum length 1  
* @param    -                            Maximum length @ref OPENAPI_DEFAULT_IMAGE_DESCR_MAX 
*
* @returns  OPEN_E_NONE      if text set successful.
* @returns  OPEN_E_FAIL      if text set failed.
* @returns  OPEN_E_PARAM     if parameters is not defined correctly.
* @returns  OPEN_E_NOT_FOUND if image not found.
* @returns  OPEN_E_UNAVAIL   The image is currently in-use.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiImageDescrSet(openapiClientHandle_t *client_handle,
                                  open_buffdesc         *imageLabel,
                                  open_buffdesc         *imageDescr); 

/*********************************************************************
* @purpose  Determines if the image label refers to a valid image name.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    imageLabel       @b{(input)}  Image label name
* @param    -                             Minimum length 1  
* @param    -                             Maximum value openapiImageFileNameMaxLengthGet()
* @param    isValid          @b{(output)} Returns @ref OPEN_TRUE if image label
*                                         is valid, else @ref OPEN_FALSE if not
* 
* @returns  OPEN_E_NONE      validation was successful. @ref isValid contains
*                            a legitimate value only in this case.
* @returns  OPEN_E_FAIL      validation failed. 
* @returns  OPEN_E_PARAM     parameters were not defined correctly.
*
* @notes    This API expects the identifying string (label) used to refer to 
*           each code image file residing in non-volatile memory on the switch,
*           such as "active" and "backup". It is not the actual name of a
*           specific image (e.g. "8.2.0.1"). 
*
* @notes    This function doesn't test validity of the label itself, but the 
*           validity of what the label refers to. In some cases, what this 
*           function expects to be passed in is either "active" or "backup",
*           while on some platforms, a valid label might be "image1" or 
*           "image2". @ref openapiImageLabelsGet can be used to obtain these
*           label names. 
*
* @notes    A deleted image (e.g., via @ref openapiImageDelete) will be 
*           reported as invalid by this function, because what the 
*           corresponding label (e.g., "backup") refers to is not present or 
*           available.
*
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiImageLabelIsValid(openapiClientHandle_t *client_handle,
                                      open_buffdesc         *imageLabel,
                                      OPEN_BOOL_t           *isValid);

/*********************************************************************
* @purpose  Get the label names for both active and backup images.
*
* @param    client_handle    @b{(input)}  Client handle from registration API
* @param    activeLabel      @b{(output)} Active label name
* @param    backupLabel      @b{(output)} Backup label name
*
* @returns  OPEN_E_NONE      if label get successful.
* @returns  OPEN_E_FAIL      if label get failed.
* @returns  OPEN_E_PARAM     if parameters is not defined correctly.
*
* @notes    This API provides the identifying string (label) used to refer to 
*           each code image file residing in non-volatile memory on the switch,
*           such as "active" and "backup". It is not the actual name of a
*           specific image (e.g. "8.2.0.1"). 
* @notes    Buffer size for image labels should be openapiImageFileNameMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiImageLabelsGet(openapiClientHandle_t *client_handle,
                                   open_buffdesc         *activeLabel,
                                   open_buffdesc         *backupLabel);

/*********************************************************************
* @purpose  Get the image names for both active, backup, and activated.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    unit             @b{(input)}  Unit number
* @param    -                             Minimum value 1  
* @param    -                             Maximum value openapiUnitMgrUnitMaxGet()
* @param    activeImage      @b{(output)} Active image name
* @param    backupImage      @b{(output)} Backup image name
* @param    activatedImage   @b{(output)} Activated image name
* 
* @returns  OPEN_E_NONE      if name get successful.
* @returns  OPEN_E_FAIL      if name get failed.
* @returns  OPEN_E_PARAM     if parameters is not defined correctly.
*
* @notes    This API provides the names of the code image files residing in
*           non-volatile memory on the switch (e.g. "8.2.1.1"). 
* @notes    Buffer size for image names should be openapiImageFileNameMaxLengthGet().
* @notes    An image name of @ref OPEN_IMAGE_NAME_NONE indicates the image is
*           not present on the device.
*
* @supported-in-version OpEN API Version: 1.8
*
* @end
*********************************************************************/
open_error_t openapiImageNamesGet(openapiClientHandle_t *client_handle,
                                  uint32_t              unit,
                                  open_buffdesc         *activeImage,
                                  open_buffdesc         *backupImage,
                                  open_buffdesc         *activatedImage);

/*********************************************************************
* @purpose  Get a locally copied image path, perform validation on it 
*           and on success, upgrade the switch with this image 
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    imagePath        @b{(input)}  Complete path for the image in Linux  
* @param    imageLabel       @b{(input)}  Active or Backup image label
* 
* @returns  OPEN_E_NONE      if image validation and upgrade are successful.
* @returns  OPEN_E_FAIL      if image validation or upgrade fails.
* @returns  OPEN_E_PARAM     if parameters is not defined correctly.
* @returns  OPEN_E_UNAVAIL   if platform does not support function.
*
* @notes    This API validates a locally downloaded image. If successful,
*           the switch image identified by the imageLabel is updated.   
*
* @notes    This API expects the imageLabel parameter to contain the 
*           identifying string (label) used to refer to each code image
*           file residing in non-volatile memory on the switch, such as 
*           "active" and "backup". It is not the actual name of a
*           specific image (e.g. "8.2.0.1").  
*
* @notes    Buffer size for imagePath and imageLabel should be 
*           openapiImageFileNameMaxLengthGet().
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiImageUpgrade(openapiClientHandle_t *client_handle,
                                 open_buffdesc   *imagePath,
                                 open_buffdesc   *imageLabel);

/*********************************************************************
* @purpose  Gets the boot code version from the current active image.
*
* @param    client_handle    @b{(input)}  Client handle from registration API.
* @param    version          @b{(output)} Boot code version
* 
* @returns  OPEN_E_NONE      Boot code version get successful.
* @returns  OPEN_E_FAIL      Boot code version get failed.
* @returns  OPEN_E_PARAM     Error in parameter passed.
*
* @notes     
*
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiBootCodeVersion(openapiClientHandle_t *client_handle, 
                                    open_buffdesc *version); 

#endif /* OPENAPI_IMAGE_H_INCLUDED */
