/*********************************************************************
*
*  Copyright 2019 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_ptp_tc.h
*
* @purpose   PTP TC Configuration and Status
*
* @component OpEN
*
* @comments  This code must not include any ICOS header files.
*
* @create    03/13/2019
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_PTP_TC_H_INCLUDED
#define OPENAPI_PTP_TC_H_INCLUDED

#include "openapi_common.h"

/*********************************************************************
* @purpose  Sets PTP TC global administrative mode.
*
* @param    client_handle   @b{(input)}   client handle from registration API
* @param    mode            @b{(input)}   administrative mode
*
* @returns  OPEN_E_NONE        Administrative mode set successful.
* @returns  OPEN_E_FAIL        Administrative mode set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @notes  Calling this API will change the running configuration of the switch
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
*********************************************************************/
open_error_t openapiPtpTcAdminModeSet(openapiClientHandle_t *client_handle,
                                      OPEN_CONTROL_t mode);

/*********************************************************************
* @purpose  Gets PTP TC global administrative mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    mode            @b{(output)} administrative mode
*
* @returns  OPEN_E_NONE        Administrative mode get successful.
* @returns  OPEN_E_FAIL        Administrative mode get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.
*
* @supported-in-version OpEN API Version: 1.20
*
* @end
*********************************************************************/
open_error_t openapiPtpTcAdminModeGet(openapiClientHandle_t *client_handle,
                                      OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Check if the PTP TC is supported.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiPtpTcIsSupportedGet(openapiClientHandle_t *client_handle,
                                        OPEN_BOOL_t  *isSupported);

/*****************************************************************************
 * @purpose   Check if the unit is valid for PTP TC.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     unit           @b{(input)}  Unit number.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiPtpTcIsValidUnitGet(openapiClientHandle_t *client_handle, 
                                        uint32_t unit, OPEN_BOOL_t *isValid);

/*****************************************************************************
 * @purpose   Determine if interface configuration can be applied in hardware.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intfNum        @b{(input)}  Interface number.
 * @param     isAllowed      @b{(output)} Flag to tell if Hardware 
 *                                        configuration allowed.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiPtpTcIntfHwConfigAllowedGet(openapiClientHandle_t *client_handle,                                                 uint32_t  intfNum, 
                                                OPEN_BOOL_t *isAllowed);

/*****************************************************************************
 * @purpose   Get PTP TC mode on the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intfNum        @b{(input)}  Interface number.
 * @param     adminMode      @b{(output)} Interface PTP TC admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiPtpTcIntfAdminModeGet(openapiClientHandle_t *client_handle, 
                                          uint32_t intfNum, OPEN_CONTROL_t *adminMode);

/*****************************************************************************
 * @purpose   Set PTP TC mode on the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intfNum        @b{(input)}  Interface number.
 * @param     adminMode      @b{(input)} Interface PTP TC admin mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiPtpTcIntfAdminModeSet(openapiClientHandle_t *client_handle, 
                                          uint32_t intfNum, OPEN_CONTROL_t adminMode);

/*****************************************************************************
 * @purpose   Gets PTP TC operational mode on the specified interface.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intfNum        @b{(input)}  Interface number.
 * @param     mode           @b{(output)} PTP TC operational mode.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiPtpTcIntfOperAdminModeGet(openapiClientHandle_t *client_handle, 
                                              uint32_t intfNum, OPEN_CONTROL_t *mode);

/*****************************************************************************
 * @purpose   Check if the interface is valid for PTP TC.
 *
 * @param     client_handle  @b{(input)}  client handle from registration API
 * @param     intfNum        @b{(input)}  Interface number.
 * @param     isValid        @b{(output)} Flag to tell if the interface 
 *                                        is valid for PTP TC.
 *
 * @returns   OPEN_E_NONE             On success.
 * @returns   OPEN_E_FAIL             On failure.
 * @returns   OPEN_E_PARAM            Invalid argument.
 *
 * @notes     none
 *
 * @supported-in-version OpEN API Version: 1.20
 *
 * @end
 *****************************************************************************/
open_error_t openapiPtpTcIsValidIntfGet(openapiClientHandle_t *client_handle, 
                                        uint32_t  intfNum, OPEN_BOOL_t *isValid);


#endif /* OPENAPI_PTPTC_H_INCLUDED */

