/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_sflow.h
*
* @purpose   sFlow Configuration and Status
*
* @component OpEN
*
* @create    09/14/2016
*
* @end
*
*********************************************************************/
#ifndef OPENAPI_SFLOW_H_INCLUDED
#define OPENAPI_SFLOW_H_INCLUDED

#include "openapi_common.h"

#define OPEN_SFLOW_RCVR_NO_TIMEOUT (-1)

typedef enum
{
  OPEN_SFLOW_SAMPLING_TYPE_INGRESS     = 0,
  OPEN_SFLOW_SAMPLING_TYPE_FLOW_BASED  = 1,
  OPEN_SFLOW_SAMPLING_TYPE_EGRESS      = 2,
} OPEN_SFLOW_SAMPLING_TYPE_t;

/*********************************************************************
* @purpose  Get the source interface number of the sFlow client.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(output)} interface number
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowSourceInterfaceGet(openapiClientHandle_t *client_handle,
                                            uint32_t *ifNum);

/*********************************************************************
* @purpose  Set the source interface number of the sFlow client.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowSourceInterfaceSet(openapiClientHandle_t *client_handle,
                                            uint32_t ifNum);

/*********************************************************************
* @purpose  Get the sFlow remote agent client source interface number.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(output)} interface number
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_UNAVAIL   Feature not supported 
* @returns  OPEN_E_INTERNAL  Failed to get source interface number 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentSourceInterfaceGet(openapiClientHandle_t *client_handle,
                                                       uint32_t *ifNum);

/*********************************************************************
* @purpose  Set the sFlow remote agent client source interface number. 
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_UNAVAIL   Feature not supported 
* @returns  OPEN_E_INTERNAL  Failed to set source interface number 
*
* @notes
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentSourceInterfaceSet(openapiClientHandle_t *client_handle,
                                                       uint32_t ifNum);

/*********************************************************************
* @purpose  Get the source IPv4 address of the sFlow Remote agents client.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ipAddr           @b{(output)} IPv4 address
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentSourceIpv4AddrGet(openapiClientHandle_t *client_handle,
                                                      uint32_t *ipAddr);

/*********************************************************************
* @purpose  Get the source IPv6 address of the sFlow Remote agents client.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ipv6Addr         @b{(output)} IPv6 address
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentSourceIpv6AddrGet(openapiClientHandle_t *client_handle,
                                                      open_in6_addr_t *ipv6Addr);

/*********************************************************************
* @purpose  Get the source IPv4 address of the sFlow client.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ipAddr           @b{(output)} IPv4 address
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowSourceIpv4AddrGet(openapiClientHandle_t *client_handle,
                                           uint32_t *ipAddr);

/*********************************************************************
* @purpose  Get the source IPv6 address of the sFlow client.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ipv6Addr         @b{(output)} IPv6 address
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowSourceIpv6AddrGet(openapiClientHandle_t *client_handle,
                                           open_in6_addr_t *ipv6Addr);

/*********************************************************************
* @purpose  Get the number of sFlow receivers the switch supports.
*
* @param    client_handle    @b{(input)}  client handle from registration API
*
* @returns  number of sFlow receivers
*
* @notes The value returned by this function is used to determine the valid
*        range of receiver indices. The valid indices are in the range from 1
*        to the value returned by this function.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiSflowRcvrCountGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the maximum length string for the sFlow receiver owner.
*
* @param    client_handle    @b{(input)}  client handle from registration API
*
* @returns  The maximum length the owner string including the terminating NULL.
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiSflowRcvrOwnerStringLenGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the sFlow receiver owner string.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow receiver index
* @param    owner            @b{(output)} sFlow receiver owner string, buffer must have
*                                         space to store a string the length of
*                                         the value returned by
*                                         @ref openapiSflowOwnerStringLenGet()
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRcvrOwnerGet(openapiClientHandle_t *client_handle,
                                      uint32_t index,
                                      open_buffdesc *owner);

/*********************************************************************
* @purpose  Set the sFlow receiver owner string and timeout.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow receiver index
* @param    owner            @b{(input)}  sFlow receiver owner, call
*                                         @ref openapiSflowOwnerStringLenGet()
*                                         to get maximum owner string length
* @param    timeout          @b{(input)}  sFlow receiver timeout value in seconds,
*                                         see note for information about special
*                                         values
*
* @notes Calling this API will change the running configuration of the switch.
* @notes Setting timeout to OPEN_SFLOW_RCVR_NO_TIMEOUT disables expiration of the
*        receiver entry. Set to a value greater than 0 to configure the number
*        of seconds until the entry times out. Set to 0 to delete the receiver
*        entry immediately.
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_EXISTS    Receiver owner is already configured
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRcvrOwnerSet(openapiClientHandle_t *client_handle,
                                      uint32_t index,
                                      open_buffdesc *owner,
                                      int32_t timeout);

/*********************************************************************
* @purpose  Get the time remaining before the sFlow receiver entry expires.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow receiver index
* @param    timeRemaining    @b{(output)} remaining time before receiver expires
*                                         in seconds
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes A returned value of OPEN_SFLOW_RCVR_NO_TIMEOUT indicates expiration of the
*        receiver entry is disabled.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRcvrTimeRemainingGet(openapiClientHandle_t *client_handle,
                                              uint32_t index,
                                              int32_t *timeRemaining);

/*********************************************************************
* @purpose  Get the sFlow receiver IP address
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow receiver index
* @param    ipAddr           @b{(output)} sFlow receiver IP address
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRcvrAddressGet(openapiClientHandle_t *client_handle,
                                        uint32_t index,
                                        open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Set the sFlow receiver IP address
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow receiver index
* @param    ipAddr           @b{(input)}  sFlow receiver IP address
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRcvrAddressSet(openapiClientHandle_t *client_handle,
                                        uint32_t index,
                                        open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Get the sFlow receiver port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow receiver index
* @param    port             @b{(output)} sFlow receiver port
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRcvrPortGet(openapiClientHandle_t *client_handle,
                                     uint32_t index,
                                     uint32_t *port);

/*********************************************************************
* @purpose  Set the sFlow receiver port
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow receiver index
* @param    port             @b{(input)}  sFlow receiver port, accepted port
*                                         number values are 1 to 65535
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRcvrPortSet(openapiClientHandle_t *client_handle,
                                     uint32_t index,
                                     uint32_t port);

/*********************************************************************
* @purpose  Get the sFlow receiver maximum datagram size parameter
*           minimum and maximum values supported by the switch.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    minSize          @b{(output)} smallest sFlow receiver max datagram size
*                                         parameter supported
* @param    maxSize          @b{(output)} largest sFlow receiver max datagram size
*                                         parameter supported
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRcvrMaxDatagramSizeRangeGet(openapiClientHandle_t *client_handle,
                                                     uint32_t *minSize, uint32_t *maxSize);

/*********************************************************************
* @purpose  Get the sFlow receiver maximum datagram size
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow receiver index
* @param    size             @b{(output)} sFlow receiver max datagram size
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRcvrMaxDatagramSizeGet(openapiClientHandle_t *client_handle,
                                                uint32_t index,
                                                uint32_t *size);

/*********************************************************************
* @purpose  Set the sFlow receiver maximum datagram size
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow receiver index
* @param    size             @b{(input)}  sFlow receiver max datagram size, 
*                                         accepted values are returned by
*                                         @ref openapiSflowRcvrMaxDatagramSizeRangeGet()
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRcvrMaxDatagramSizeSet(openapiClientHandle_t *client_handle,
                                                uint32_t index,
                                                uint32_t size);

/*********************************************************************
* @purpose  Get the supported range of sFlow instance ID values.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    minInstance      @b{(output)} instance ID minimum value
* @param    maxInstance      @b{(output)} instance ID maximum value
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowInstanceRangeGet(openapiClientHandle_t *client_handle,
                                          uint32_t *minInstance,
                                          uint32_t *maxInstance);

/*********************************************************************
* @purpose  Get the sFlow sampler receiver index
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  sampler instance Id
* @param    rcvrIndex        @b{(output)} sampler receiver index
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowFsRcvrIndexGet(openapiClientHandle_t *client_handle,
                                        uint32_t ifNum,
                                        uint32_t instance,
                                        uint32_t *rcvrIndex);

/*********************************************************************
* @purpose  Set the sFlow sampler receiver index
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  sampler instance Id
* @param    rcvrIndex        @b{(input)}  sampler receiver index
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_EXISTS    Receiver index is already configured
*
* @notes Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowFsRcvrIndexSet(openapiClientHandle_t *client_handle,
                                        uint32_t ifNum,
                                        uint32_t instance,
                                        uint32_t rcvrIndex);

/*********************************************************************
* @purpose  Get the supported range of sFlow flow sampler packet sampling
*           rate parameter values.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    minRate          @b{(output)} minimum sampling rate
* @param    maxRate          @b{(output)} maximum sampling rate
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowFsPacketSamplingRateRangeGet(openapiClientHandle_t *client_handle,
                                                      uint32_t *minRate,
                                                      uint32_t *maxRate);

/*********************************************************************
* @purpose  Get the sFlow sampler packet sampling rate
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  sampler instance Id
* @param    type             @b{(input)}  sampling type
* @param    rate             @b{(output)} sampling rate
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowFsPacketSamplingRateGet(openapiClientHandle_t *client_handle,
                                                 uint32_t ifNum,
                                                 uint32_t instance,
                                                 OPEN_SFLOW_SAMPLING_TYPE_t type,
                                                 uint32_t *rate);

/*********************************************************************
* @purpose  Set the sFlow sampler packet sampling rate
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  sampler instance Id
* @param    type             @b{(input)}  sampling type
* @param    rate             @b{(input)}  sampling rate
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowFsPacketSamplingRateSet(openapiClientHandle_t *client_handle,
                                                 uint32_t ifNum,
                                                 uint32_t instance,
                                                 OPEN_SFLOW_SAMPLING_TYPE_t type,
                                                 uint32_t rate);

/*********************************************************************
* @purpose  Get the supported range of sFlow maximum header size ID values.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    minHeaderSize    @b{(output)} header size minimum value
* @param    maxHeaderSize    @b{(output)} header size maximum value
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowFsMaximumHeaderSizeRangeGet(openapiClientHandle_t *client_handle,
                                                     uint32_t *minHeaderSize,
                                                     uint32_t *maxHeaderSize);

/*********************************************************************
* @purpose  Get the sFlow sampler maximum header size
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  sampler instance Id
* @param    size             @b{(output)} maximum header size
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowFsMaximumHeaderSizeGet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum,
                                                uint32_t instance,
                                                uint32_t *size);

/*********************************************************************
* @purpose  Set the sFlow sampler maximum header size
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  sampler instance Id
* @param    size             @b{(input)}  maximum header size
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowFsMaximumHeaderSizeSet(openapiClientHandle_t *client_handle,
                                                uint32_t ifNum,
                                                uint32_t instance,
                                                uint32_t size);

/*********************************************************************
* @purpose  Get the flow-based sFlow sampler attached IP ACL
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    aclNum           @b{(output)} IP ACL number
*
* @returns  OPEN_E_UNAVAIL   API/feature not supported. 
*
* @notes    This API is deprecated. To enable flow based sFlow sampling
*           configure IP ACL rules(permit/deny) sflow-remote-agent action. 
*
* @open-status-deprecated
* @supported-in-version OpEN API Version:  1.14
* @deprecated-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiSflowSamplerIpAclGet(openapiClientHandle_t *client_handle,
                                         uint32_t ifNum,
                                         uint32_t *aclNum);

/*********************************************************************
* @purpose  Set the flow-based sFlow sampler IP ACL action
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sampler receiver index
* @param    ifNum            @b{(input)}  interface number
* @param    aclNum           @b{(input)}  IP ACL number
* @param    action           @b{(input)}  OPEN_ENABLE to add ACL, OPEN_DISABLE to remove ACL
*
* @returns  OPEN_E_UNAVAIL   API/feature not supported. 
*
* @notes    This API is deprecated. To enable flow based sFlow sampling
*           configure IP ACL rules(permit/deny) sflow-remote-agent action. 
*
* @open-status-deprecated
* @supported-in-version OpEN API Version:  1.14
* @deprecated-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiSflowSamplerIpAclSet(openapiClientHandle_t *client_handle,
                                         uint32_t index,
                                         uint32_t ifNum,
                                         uint32_t aclNum,
                                         OPEN_CONTROL_t action);

/*********************************************************************
* @purpose  Get the flow-based sFlow sampler attached MAC ACL
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    aclNum           @b{(output)} MAC ACL number
*
* @returns  OPEN_E_UNAVAIL   API/feature not supported. 
*
* @notes    This API is deprecated. To enable flow based sFlow sampling
*           configure MAC ACL rules(permit/deny) sflow-remote-agent action. 
*
* @open-status-deprecated
* @supported-in-version OpEN API Version:  1.14
* @deprecated-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiSflowSamplerMacAclGet(openapiClientHandle_t *client_handle,
                                          uint32_t ifNum,
                                          uint32_t *aclNum);

/*********************************************************************
* @purpose  Set the flow-based sFlow sampler MAC ACL action
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sampler receiver index
* @param    ifNum            @b{(input)}  interface number
* @param    aclNum           @b{(input)}  MAC ACL number
* @param    action           @b{(input)}  OPEN_ENABLE to add ACL, OPEN_DISABLE to remove ACL
*
* @returns  OPEN_E_UNAVAIL   API/feature not supported. 
*
* @notes    This API is deprecated. To enable flow based sFlow sampling
*           configure MAC ACL rules(permit/deny) sflow-remote-agent action. 
*
* @open-status-deprecated
* @supported-in-version OpEN API Version:  1.14
* @deprecated-in-version OpEN API Version: 1.15
*
* @end
*********************************************************************/
open_error_t openapiSflowSamplerMacAclSet(openapiClientHandle_t *client_handle,
                                          uint32_t index,
                                          uint32_t ifNum,
                                          uint32_t aclNum,
                                          OPEN_CONTROL_t action);

/*********************************************************************
* @purpose  Get the receiver index associated with a counter poller.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  poller instance Id
* @param    rcvrIndex        @b{(output)} poller receiver index
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_ERROR     Internal error
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowCpRcvrIndexGet(openapiClientHandle_t *client_handle,
                                        uint32_t ifNum,
                                        uint32_t instance,
                                        uint32_t *rcvrIndex);

/*********************************************************************
* @purpose  Set the receiver index associated with a counter poller.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  poller instance Id
* @param    rcvrIndex        @b{(input)}  poller receiver index
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_EXISTS    Already associated with another receiver
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowCpRcvrIndexSet(openapiClientHandle_t *client_handle,
                                        uint32_t ifNum,
                                        uint32_t instance,
                                        uint32_t rcvrIndex);

/*********************************************************************
* @purpose  Get the supported range of sFlow counter polling interval values.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    minInterval      @b{(output)} polling interval minimum value
* @param    maxInterval      @b{(output)} polling interval maximum value
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowCpIntervalRangeGet(openapiClientHandle_t *client_handle,
                                            uint32_t *minInterval, uint32_t *maxInterval);

/*********************************************************************
* @purpose  Get the counter polling interval.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  poller instance Id
* @param    interval         @b{(output)} interval
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowCpIntervalGet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum,
                                       uint32_t instance,
                                       uint32_t *interval);

/*********************************************************************
* @purpose  Set the counter polling interval.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  poller instance Id
* @param    interval         @b{(input)}  interval
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_EXISTS    Already configured with this interval
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowCpIntervalSet(openapiClientHandle_t *client_handle,
                                       uint32_t ifNum,
                                       uint32_t instance,
                                       uint32_t interval);

/*********************************************************************
* @purpose  Get the sFlow sampler remote agent index maximum value supported.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* 
* @returns  maximum remote agent index value
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiSflowFsRemoteAgentMaxIndexGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the sFlow sampler remote agent index.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  sampler instance ID
* @param    index            @b{(output)} remote agent index
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_UNAVAIL   sFlow remote agent feature is not supported on the platform
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowFsRemoteAgentIndexGet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum,
                                               uint32_t instance,
                                               uint32_t *index);

/*********************************************************************
* @purpose  Set the sFlow sampler remote agent index.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    ifNum            @b{(input)}  interface number
* @param    instance         @b{(input)}  sampler instance ID
* @param    index            @b{(input)}  remote agent index
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_UNAVAIL   sFlow remote agent feature is not supported on the platform
*
* @notes Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowFsRemoteAgentIndexSet(openapiClientHandle_t *client_handle,
                                               uint32_t ifNum,
                                               uint32_t instance,
                                               uint32_t index);

/*********************************************************************
* @purpose  Get the sFlow sampler remote agent session ID maximum value supported.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* 
* @returns  maximum remote agent session ID value
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
uint32_t openapiSflowFsRemoteAgentMaxMonitorSessionIdGet(openapiClientHandle_t *client_handle);

/*********************************************************************
* @purpose  Get the sFlow sampler remote agent index.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  remote agent index
* @param    monitorSessionId @b{(output)} monitor session ID
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_UNAVAIL   sFlow remote agent feature is not supported on the platform
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentMonitorSessionGet(openapiClientHandle_t *client_handle,
                                                      uint32_t index,
                                                      uint32_t *monitorSessionId);

/*********************************************************************
* @purpose  Set the sFlow sampler remote agent index.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  remote agent index
* @param    monitorSessionId @b{(input)}  monitor session ID
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
* @returns  OPEN_E_UNAVAIL   sFlow remote agent feature is not supported on the platform
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentMonitorSessionSet(openapiClientHandle_t *client_handle,
                                                      uint32_t index,
                                                      uint32_t monitorSessionId);

/*********************************************************************
* @purpose  Get the sFlow remote agent IP address.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow remote agent index
* @param    ipAddr           @b{(output)} sFlow remote agent IP address
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentAddressGet(openapiClientHandle_t *client_handle,
                                               uint32_t index,
                                               open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Set the sFlow remote agent IP address
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow remote agent index
* @param    ipAddr           @b{(input)}  sFlow remote agent IP address
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentAddressSet(openapiClientHandle_t *client_handle,
                                               uint32_t index,
                                               open_inet_addr_t *ipAddr);

/*********************************************************************
* @purpose  Get the sFlow remote agent destination UDP port.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow remote agent index
* @param    port             @b{(output)} sFlow remote agent destination UDP port
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentUdpDestPortGet(openapiClientHandle_t *client_handle,
                                                   uint32_t index,
                                                   uint32_t *port);

/*********************************************************************
* @purpose  Set the sFlow remote agent destination UDP port.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow remote agent index
* @param    port             @b{(input)}  sFlow remote agent destination UDP port
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentUdpDestPortSet(openapiClientHandle_t *client_handle,
                                                   uint32_t index,
                                                   uint32_t port);

/*********************************************************************
* @purpose  Get the sFlow remote agent monitor session destination interface.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow remote agent index
* @param    ifNum            @b{(output)} monitor session destination interface
*
* @returns  OPEN_E_NONE      Get successful
* @returns  OPEN_E_FAIL      Get failed
* @returns  OPEN_E_NOT_FOUND Monitor session does not exist
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentMonitorSessionDestIfaceGet(openapiClientHandle_t *client_handle,
                                                               uint32_t index,
                                                               uint32_t *ifNum);

/*********************************************************************
* @purpose  Set the sFlow remote agent monitor session destination interface.
*
* @param    client_handle    @b{(input)}  client handle from registration API
* @param    index            @b{(input)}  sFlow remote agent index
* @param    ifNum            @b{(input)}  monitor session destination interface
*
* @returns  OPEN_E_NONE      Set successful
* @returns  OPEN_E_FAIL      Set failed
* @returns  OPEN_E_NOT_FOUND Monitor session does not exist
* @returns  OPEN_E_PARAM     Error in parameter passed
*
* @notes Calling this API will change the running configuration of the switch.
*
* @supported-in-version OpEN API Version: 1.14
*
* @end
*********************************************************************/
open_error_t openapiSflowRemoteAgentMonitorSessionDestIfaceSet(openapiClientHandle_t *client_handle,
                                                               uint32_t index,
                                                               uint32_t ifNum);

#endif /* OPENAPI_SFLOW_H_INCLUDED */
