/*********************************************************************
*
*  Copyright 2017-2018 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at        
*
*      http://www.apache.org/licenses/LICENSE-2.0 
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  openapi_timeranges.h
*
* @purpose   Time Range Configuration
*
* @component OpEN
*
* @comments  This code must not include any ICOS header files.
*
* @create    09/20/2017
*
* @end
*
**********************************************************************/
#ifndef OPENAPI_TIMERANGES_H_INCLUDED
#define OPENAPI_TIMERANGES_H_INCLUDED

#include "openapi_common.h"

#define OPEN_TIMERANGE_BUF_LEN 32
#define OPEN_TIMERANGE_NAME_LEN_MAX  32
#define OPEN_TIMERANGE_DATE_TIME_BUF_LEN  20
#define OPEN_TIMERANGE_DATE_TIME_PARAM_COUNT  5
#define OPEN_TIMERANGE_DATE_PARAM_COUNT 3

#define OPEN_TIMERANGE_PERIODIC_FREQUENCY_MIN  0
#define OPEN_TIMERANGE_PERIODIC_FREQUENCY_MAX  255

typedef enum
{
  OPEN_TIMERANGE_STATUS_ACTIVE = 1,
  OPEN_TIMERANGE_STATUS_INACTIVE = 2
}OPEN_TIMERANGESTATUS_t;

/* Time range periodic entry recurrence pattern */
typedef enum 
{
  OPEN_TIMERANGE_PATTERN_WEEKLY = 0,                   
  OPEN_TIMERANGE_PATTERN_DAILY = 1,
  OPEN_TIMERANGE_PATTERN_MONTHLY = 2
}OPEN_TIMERANGE_PERIODIC_ENTRY_PATTERN_t;

typedef enum 
{
  OPEN_TIMERANGE_EVERY_WEEKDAY = 0,
  OPEN_TIMERANGE_EVERY_WEEKEND = 1,
  OPEN_TIMERANGE_EVERY_DAY = 2
}OPEN_TIMERANGE_DAILY_t;

typedef enum
{
  OPEN_TIMERANGE_DAY_SUN = (1 << 1),
  OPEN_TIMERANGE_DAY_MON = (1 << 2),
  OPEN_TIMERANGE_DAY_TUE = (1 << 3),
  OPEN_TIMERANGE_DAY_WED = (1 << 4),
  OPEN_TIMERANGE_DAY_THU = (1 << 5),
  OPEN_TIMERANGE_DAY_FRI = (1 << 6),
  OPEN_TIMERANGE_DAY_SAT = (1 << 7)  
}OPEN_TIMERANGE_DAY_t;

typedef struct OPEN_TIMERANGE_DATE_AND_TIME_s
{
  uint16_t year;
  uint8_t  month;
  uint8_t  day;
  uint8_t  hour;
  uint8_t  minute; 
}OPEN_TIMERANGE_DATE_AND_TIME_t;

typedef struct OPEN_TIMERANGE_PERIODIC_ENTRY_s
{
  OPEN_TIMERANGE_DATE_AND_TIME_t openDateAndTime;   /* year-month-day hh:mm */
  OPEN_TIMERANGE_PERIODIC_ENTRY_PATTERN_t pattern;  
  uint8_t dayMask;    /* mask of OPEN_TIMERANGE_DAY_t values or day in a month */
  uint16_t frequency;
}OPEN_TIMERANGE_PERIODIC_ENTRY_t;

/*********************************************************************
* @purpose  Set the admin mode of time range.          
*
* @param    client_handle      @b{(input)} client handle from registration API
* @param    val                @b{(input)} time range admin mode
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeAdminModeSet(openapiClientHandle_t *client_handle, 
                                          OPEN_CONTROL_t val);   

/*********************************************************************
* @purpose  Get the admin mode of time range.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    val                @b{(output)} time range admin mode
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeAdminModeGet(openapiClientHandle_t *client_handle, 
                                          OPEN_CONTROL_t *val);   

/*********************************************************************
* @purpose  Get the current number of time ranges configured.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    val                @b{(output)} number of time ranges  
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeCurrNumGet(openapiClientHandle_t *client_handle, 
                                        uint32_t *val);   

/*********************************************************************
* @purpose  Get the maximum number of time ranges that can be configured.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    val                @b{(output)} number of time ranges  
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeMaxNumGet(openapiClientHandle_t *client_handle, 
                                       uint32_t *val);   

/*********************************************************************
* @purpose  Get the first time range created in the system.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    name               @b{(output)} time range name
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeNameFirstGet(openapiClientHandle_t *client_handle,
                                          open_buffdesc *name);

/*********************************************************************
* @purpose  Get the next time range for the given time range.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    prevName           @b{(input)}  previous time range name
* @param    name               @b{(output)} time range name
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeNameNextGet(openapiClientHandle_t *client_handle,
                                         open_buffdesc *prevName, 
                                         open_buffdesc *name);

/*********************************************************************
* @purpose  Get the status for the given time range.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    name               @b{(input)}  time range name
* @param    val                @b{(output)} status
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeStatusGet(openapiClientHandle_t *client_handle,
                                       open_buffdesc *name,
                                       OPEN_TIMERANGESTATUS_t *val);

/*********************************************************************
* @purpose  Get the current number of entries configures for the specified time range.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    name               @b{(input)}  time range name
* @param    val                @b{(output)} entry count 
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeNumEntriesGet(openapiClientHandle_t *client_handle,
                                           open_buffdesc *name,
                                           uint32_t *val);

/*********************************************************************
* @purpose  Get the absolute time range entry number for the given time range.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    name               @b{(input)}  time range name
* @param    val                @b{(output)} entry number 
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeAbsoluteEntryNumGet(openapiClientHandle_t *client_handle,
                                                 open_buffdesc *name,
                                                 uint32_t *val);

/*********************************************************************
* @purpose  Creates a time range.          
*
* @param    client_handle      @b{(input)} client handle from registration API
* @param    name               @b{(input)} time range name
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeAdd(openapiClientHandle_t *client_handle,
                                 open_buffdesc *name);

/*********************************************************************
* @purpose  Deletes a time range.          
*
* @param    client_handle      @b{(input)} client handle from registration API
* @param    name               @b{(input)} time range name
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeDelete(openapiClientHandle_t *client_handle,
                                    open_buffdesc *name);

/*********************************************************************
* @purpose  Get start date and time for the given absolute time range entry.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    name               @b{(input)}  time range name
* @param    val                @b{(output)} date and time (year-month-day hours:minutes) 
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeAbsoluteEntryStartDateAndTimeGet(openapiClientHandle_t *client_handle,
                                                              open_buffdesc *name,
                                                              open_buffdesc *val);

/*********************************************************************
* @purpose  Get end date and time for the given absolute time range entry.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    name               @b{(input)}  time range name
* @param    val                @b{(output)} date and time (year-month-day hours:minutes) 
*
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeAbsoluteEntryEndDateAndTimeGet(openapiClientHandle_t *client_handle,
                                                            open_buffdesc *name,
                                                            open_buffdesc *val);

/*********************************************************************
* @purpose  Set start date and time for the given absolute time range entry.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    name               @b{(input)}  time range name
* @param    val                @b{(input)}  date and time (year-month-day hours:minutes) 
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeAbsoluteEntryStartDateAndTimeAdd(openapiClientHandle_t *client_handle,
                                                              open_buffdesc *name,
                                                              open_buffdesc *val);

/*********************************************************************
* @purpose  Set end date and time for the given absolute time range entry.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    name               @b{(input)}  time range name
* @param    val                @b{(input)}  date and time (year-month-day hours:minutes) 
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeAbsoluteEntryEndDateAndTimeAdd(openapiClientHandle_t *client_handle,
                                                            open_buffdesc *name,
                                                            open_buffdesc *val);

/*********************************************************************
* @purpose  Remove an entry from an existing time range.          
*
* @param    client_handle      @b{(input)}  client handle from registration API
* @param    name               @b{(input)}  time range name
* @param    val                @b{(input)}  time range entry ID created on 
*                                           adding absolute or periodic time range entry
*
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangeEntryDelete(openapiClientHandle_t *client_handle,
                                         open_buffdesc *name,
                                         uint32_t val);

/*********************************************************************
* @purpose  Set periodic start date and time at which the time range 
*           becomes active.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    name           @b{(input)}  time range name
* @param    val            @b{(input)}  periodic start date and time 
*     
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangePeriodicEntryStartDateAndTimeSet(openapiClientHandle_t *client_handle,
                                                              open_buffdesc *name, 
                                                              OPEN_TIMERANGE_PERIODIC_ENTRY_t val);

/*********************************************************************
* @purpose  Gets periodic start date and time at which the time range 
*           becomes active.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    name           @b{(input)}  time range name
* @param    entry          @b{(input)}  time range entry
* @param    val            @b{(output)} periodic start date and time 
*     
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangePeriodicEntryStartDateAndTimeGet(openapiClientHandle_t *client_handle,
                                                              open_buffdesc *name,
                                                              uint32_t entry, 
                                                              OPEN_TIMERANGE_PERIODIC_ENTRY_t *val);

/*********************************************************************
* @purpose  Set periodic end date and time at which the time range 
*           becomes inactive.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    name           @b{(input)}  time range name
* @param    entry          @b{(input)}  time range entry
* @param    val            @b{(input)}  periodic end date and time 
*     
* @returns  OPEN_E_NONE        Set successful.    
* @returns  OPEN_E_FAIL        Set failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangePeriodicEntryEndDateAndTimeSet(openapiClientHandle_t *client_handle,
                                                            open_buffdesc *name, 
                                                            uint32_t entry, 
                                                            OPEN_TIMERANGE_PERIODIC_ENTRY_t val);

/*********************************************************************
* @purpose  Gets periodic end date and time at which the time range 
*           becomes inactive.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    name           @b{(input)}  time range name
* @param    entry          @b{(input)}  time range entry
* @param    val            @b{(output)} periodic end date and time 
*     
* @returns  OPEN_E_NONE        Get successful.    
* @returns  OPEN_E_FAIL        Get failed.
* @returns  OPEN_E_PARAM       Error in parameter passed.  
* @returns  OPEN_E_INTERNAL    Other Internal Failure.
* 
* @notes  
* 
* @supported-in-version OpEN API Version: 1.19
*
* @end
*********************************************************************/
open_error_t openapiTimeRangePeriodicEntryEndDateAndTimeGet(openapiClientHandle_t *client_handle,
                                                            open_buffdesc *name,
                                                            uint32_t entry, 
                                                            OPEN_TIMERANGE_PERIODIC_ENTRY_t *val);

#endif
