#
# Chef Cookbook   : netdev_eos
# File            : provider/lag.rb
#
# Copyright 2016 Broadcom.
#
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#
def whyrun_supported?
  true
end

action :create do
  lag_exists, intf_num = get_intf_num('netdev-icos-lag', new_resource.name, OpEN::OPEN_INTF_TYPE_LAG)
  if !lag_exists && intf_num != -1
    converge_by("create lag #{@new_resource.name}") do
      msg = "lag can not be created. Only its properties can be set"
      log_msg("info", msg)
      modify_lag(intf_num)
    end
  elsif lag_exists && intf_num != -1
    converge_by("edit lag #{@new_resource.name}") do
      modify_lag(intf_num)
    end
  else
    msg = "Failed to create lag #{@new_resource.name}, not a valid lag"
    log_msg("info", msg)
  end
end

action :delete do
  if @current_resource.exists
    converge_by("reset lag #{@current_resource.name}") do
      reset_lag
    end
  else
    msg = "lag doesn't exist or not a valid lag, nothing to delete/reset"
    log_msg("info", msg)
  end
end

def reset_lag
  lag_exists, lag_intf_num = get_intf_num('netdev-icos-lag', new_resource.name, OpEN::OPEN_INTF_TYPE_LAG)

  if lag_exists && lag_intf_num != -1
    msg = "lag '#{@new_resource.name}' can not be deleted. Only its properties can be reset"
    log_msg("info", msg)

    open = OpENUtil.new()
    conn_ret = open.connect("netdev-icos-lag")
    if conn_ret == OpEN::OPEN_E_NONE
      client = open.client
      # Reset the LAG interface lacp mode disabled i.e., static LAG
      static_mode = OpEN::OPEN_ENABLE
      ret = OpEN::openapiLagStaticModeSet(client, lag_intf_num, static_mode)
      if ret != OpEN::OPEN_E_NONE
        msg = "Failed to reset LAG interface '#{@new_resource.name}' lacp mode, returned '#{ret}'"
        log_msg("info", msg)
      end

      # Set the minimum uplinks (default value to: 1) for the LAG interface
      ret = OpEN::openapiLagMinUplinksSet(client, lag_intf_num, 1)
      if ret != OpEN::OPEN_E_NONE
        msg = "Failed to set LAG interface '#{new_resource.name}' minimum uplinks '#{new_resource.minimum_links}', returned value = '#{ret}'"
        log_msg("info", msg)
      end

      intf_num_p      = OpEN.new_uint32_tp()
      lacp_mode       = OpEN::OPEN_ENABLE
      buffdesc        = OpEN::Open_buffdesc.new
      # Remove the physical interfaces from the LAG
      @current_resource.links.each do |interface|
        # Get the internal interface number
        buffdesc.size   = interface.to_s.length + 1
        buffdesc.pstart = open.getCharBuffer(buffdesc.size, interface.to_s)
        ret = OpEN::openapiIfNumGet(client, buffdesc, intf_num_p)
        if ret == OpEN::OPEN_E_NONE
          intf_num = OpEN.uint32_tp_value(intf_num_p)
          ret1 = OpEN::openapiLagPortDelete(client, lag_intf_num, intf_num)
          ret2 = OpEN::openapiDot3adAggPortLacpModeSet(client, intf_num, lacp_mode)
          if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
            msg = "Failed to remove interface '#{interface}' from the LAG '#{new_resource.name}', returned '#{ret}'"
            log_msg("info", msg)
          end
        end
      end

      OpEN.delete_uint32_tp(intf_num_p)
    else
      msg = "Failed to connect to 'netdev-icos-lag'"
      log_msg("info", msg)
    end
    open.terminate()

  else
    msg = "Failed to reset lag '#{@new_resource.name}', not a valid lag"
    log_msg("info", msg)
  end
end

def load_current_resource
  msg = "Loading current resource '#{@new_resource.name}'"
  log_msg("info", msg)
  lag = {}
  @current_resource = Chef::Resource::NetdevLag.new(@new_resource.name)
  @current_resource.exists = false
  
  lag_exists, intf_num = get_intf_num('netdev-icos-lag', new_resource.name, OpEN::OPEN_INTF_TYPE_LAG)
  if lag_exists && intf_num != -1

    open = OpENUtil.new()
    conn_ret = open.connect("netdev-icos-lag")
    if conn_ret == OpEN::OPEN_E_NONE
      client = open.client
      # Get the maximum supported members in any LAG interface
      lag_max_member_count_p = OpEN.new_uint32_tp()
      OpEN::openapiLagMaxMemberCountGet(client, lag_max_member_count_p)
      lag_max_member_count = OpEN.uint32_tp_value(lag_max_member_count_p)

      buffdesc           = OpEN::Open_buffdesc.new
      static_mode_p      = OpEN.new_OPEN_CONTROL_tp()
      lacp_mode_p        = OpEN.new_OPEN_CONTROL_tp()
      lag_min_up_links_p = OpEN.new_uint32_tp()
      lag_member_count_p = OpEN.new_uint32_tp()
      lag_buffdesc       = OpEN::Open_buffdesc.new
      lag_mem_buffdesc   = OpEN::Open_buffdesc.new

      # Get the LAG interface mode
      OpEN::openapiLagStaticModeGet(client, intf_num, static_mode_p)
      static_mode = OpEN.OPEN_CONTROL_tp_value(static_mode_p)
      if static_mode == OpEN::OPEN_ENABLE
        lacp = 'disable'
      else
        lacp = 'active'
      end

      # Get the LAG minimum up links
      OpEN::openapiLagMinUplinksGet(client, intf_num, lag_min_up_links_p)
      lag_min_up_links = OpEN.uint32_tp_value(lag_min_up_links_p)

      lag_buffdesc.size   = lag_max_member_count * 4
      tmpIntBuf           = open.getIntBuffer(lag_max_member_count)
      lag_buffdesc.pstart = tmpIntBuf

      OpEN.uint32_tp_assign(lag_member_count_p, lag_max_member_count)
      lag_member_count = OpEN.uint32_tp_value(lag_member_count_p)
      ret = OpEN::openapiLagMembersGet(client, intf_num, lag_member_count_p, lag_buffdesc)
      lag_member_count = OpEN.uint32_tp_value(lag_member_count_p)

      lag_members = []
      for x in 0..(lag_member_count - 1)
        this_if = tmpIntBuf[x]

        if static_mode == OpEN::OPEN_DISABLE
          ret = OpEN::openapiDot3adAggPortLacpModeGet(client, this_if, lacp_mode_p)
          if ret == OpEN::OPEN_E_NONE
            lacp_mode = OpEN.OPEN_CONTROL_tp_value(lacp_mode_p)
            if lacp_mode == OpEN::OPEN_ENABLE
              lacp = 'active'
            else
              lacp = 'passive'
            end
          end
        end

        # Get the LAG member interface name
        lag_mem_buffdesc.size   = 256
        tmpCharBuf              = open.getCharBuffer(lag_mem_buffdesc.size, nil)
        lag_mem_buffdesc.pstart = tmpCharBuf
        OpEN::openapiIfNameGet(client, this_if, lag_mem_buffdesc)
        intf_name = tmpCharBuf.cast()

        lag_members << intf_name
      end

      lag['links'] = lag_members
      lag['minimum_links'] = lag_min_up_links
      lag['lacp'] = lacp

      @current_resource.name(new_resource.name)
      @current_resource.links(lag['links'])
      @current_resource.minimum_links(lag['minimum_links'])
      @current_resource.lacp(lag['lacp'])
      @current_resource.exists = lag_exists

      OpEN.delete_uint32_tp(lag_max_member_count_p)
      OpEN.delete_OPEN_CONTROL_tp(static_mode_p)
      OpEN.delete_OPEN_CONTROL_tp(lacp_mode_p)
      OpEN.delete_uint32_tp(lag_min_up_links_p)
      OpEN.delete_uint32_tp(lag_member_count_p)

    else
      msg = "Failed to connect to 'netdev-icos-lag'"
      log_msg("info", msg)
    end
    open.terminate()

  else
    msg = "lag #{@new_resource.name} doesn't exist, or not a valid lag"
    log_msg("info", msg)
  end

end

def modify_lag(lag_intf_num)
  open = OpENUtil.new()
  conn_ret = open.connect("netdev-icos-lag")
  if conn_ret == OpEN::OPEN_E_NONE
    client = open.client
    # Set the LAG interface lacp mode
    if new_resource.lacp == 'active'|| new_resource.lacp == 'passive'
      static_mode = OpEN::OPEN_DISABLE
    else
      static_mode = OpEN::OPEN_ENABLE
    end

    if new_resource.lacp
      if has_changed?(current_resource.lacp, new_resource.lacp)
        ret = OpEN::openapiLagStaticModeSet(client, lag_intf_num, static_mode)
        if ret != OpEN::OPEN_E_NONE
          msg = "Failed to set LAG interface '#{new_resource.name}' lacp mode, returned '#{ret}'"
          log_msg("info", msg)
        end
      end
    end

    if new_resource.minimum_links
      if has_changed?(current_resource.minimum_links, new_resource.minimum_links)
        # Set the minimum uplinks for the LAG interface
        ret = OpEN::openapiLagMinUplinksSet(client, lag_intf_num, new_resource.minimum_links.to_i)
        if ret != OpEN::OPEN_E_NONE
          msg = "Failed to set LAG interface '#{new_resource.name}' minimum uplinks '#{new_resource.minimum_links}', returned value = '#{ret}'"
          log_msg("info", msg)
        end
      end
    end

    # Set the dynamic LAG lacp mode
    if new_resource.lacp == 'active'
      lacp_mode = OpEN::OPEN_ENABLE
    else
      lacp_mode = OpEN::OPEN_DISABLE
    end

    intf_num_p      = OpEN.new_uint32_tp()
    buffdesc        = OpEN::Open_buffdesc.new
    # Add the physical interfaces to the LAG

    if new_resource.links
      if has_changed?(current_resource.links, new_resource.links)
        new_resource.links.each do |interface|
          # Get the internal interface number
          buffdesc.size   = interface.to_s.length + 1
          buffdesc.pstart = open.getCharBuffer(buffdesc.size, interface.to_s)
          ret = OpEN::openapiIfNumGet(client, buffdesc, intf_num_p)
          if ret == OpEN::OPEN_E_NONE
            intf_num = OpEN.uint32_tp_value(intf_num_p)
            ret1 = OpEN::openapiLagPortAdd(client, lag_intf_num, intf_num)
            if static_mode != OpEN::OPEN_ENABLE
              ret2 = OpEN::openapiDot3adAggPortLacpModeSet(client, intf_num, lacp_mode)
            else
              ret2 = OpEN::OPEN_E_NONE
            end
            if ret1 != OpEN::OPEN_E_NONE || ret2 != OpEN::OPEN_E_NONE
              msg = "Failed to add interface '#{interface}' to the LAG '#{new_resource.name}', returned '#{ret1}', '#{ret2}'"
              log_msg("info", msg)
            end
          end
        end
      end
    end

    OpEN.delete_uint32_tp(intf_num_p)
  else
    msg = "Failed to connect to 'netdev-icos-lag'"
    log_msg("info", msg)
  end
  open.terminate()

end

