/*********************************************************************
*
* Copyright 2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  dai_example.c
*
* @purpose   Dynamic ARP Inspection (DAI) OpEN APIs Example 
*
* @component OpEN
*
* @comments
*
* @create    07/09/2018
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_dynamic_arp_inspection.h"

/* 
   OpEN API set functions are processed asynchronously.  There may be some
   delay between when the set function call returns and when the DAI 
   state is updated to reflect the change.  These parameters control how
   long the test code retries the get functions to retrieve a change.
*/
/*******************************************************************
*
* @brief  This function prints the DAI Example Application Menu.
*
* @param  none
*
* @returns  none
*
* @end
*********************************************************************/
void printDaiAppMenu()
{
  printf("Usage:  dai_example <test#> <arg1> ... \n");
  printf("Test 1: Set global source MAC address validation mode: dai_example 1 <0|1> \n");
  printf("Test 2: Get global source MAC address validation mode: dai_example 2 \n");
  printf("Test 3: Set global destination MAC address validation mode: dai_example 3 <0|1> \n");
  printf("Test 4: Get global destination MAC address validation mode: dai_example 4 \n");
  printf("Test 5: Set global IP address validation mode: dai_example 5 <0|1> \n");
  printf("Test 6: Get global IP address validation mode: dai_example 6 \n");
  printf("Test 7: DAI OpEN APIs Sanity: dai_example 7 \n");
  return;
}

/*********************************************************************
* @purpose  Set the global DAI source MAC validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* @param    mode             @b{(input)}   Source MAC validate mode.
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
void daiSourceMacValidateModeSet(openapiClientHandle_t *clientHandle,
                                 OPEN_CONTROL_t        mode)
{
  open_error_t result;

  if ((result = openapiDaiSourceMacValidateModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global DAI source MAC validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI source MAC validate mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the global DAI source MAC validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void daiSourceMacValidateModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t   result;
  OPEN_CONTROL_t mode;

  if ((result = openapiDaiSourceMacValidateModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global source MAC validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI global source MAC validate mode: %d (result = %d)\n", mode, result);
  }

  return; 
}

/*********************************************************************
* @purpose  Set the global DAI destination MAC validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* @param    mode             @b{(input)}   Destination MAC validate mode.
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
void daiDestMacValidateModeSet(openapiClientHandle_t *clientHandle,
                               OPEN_CONTROL_t        mode)
{
  open_error_t result;

  if ((result = openapiDaiDestMacValidateModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global DAI destination MAC validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI destination MAC validate mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the global DAI destination MAC validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void daiDestMacValidateModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t   result;
  OPEN_CONTROL_t mode;

  if ((result = openapiDaiDestMacValidateModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global destination MAC validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI global destination MAC validate mode: %d (result = %d)\n", mode, result);
  }

  return; 
}

/*********************************************************************
* @purpose  Set the global DAI IP address validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* @param    mode             @b{(input)}   IP address validate mode.
* 
* @returns  none
*   
* @notes  Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
void daiIpAddrValidateModeSet(openapiClientHandle_t *clientHandle,
                              OPEN_CONTROL_t        mode)
{
  open_error_t result;

  if ((result = openapiDaiIpAddrValidateModeSet(clientHandle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set global DAI IP address validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI IP address validate mode is set successfully. (result = %d)\n", result);
  }

  return;
}

/*********************************************************************
* @purpose  Get the global DAI IP address validate mode.
* 
* @param    clientHandle     @b{(input)}   Client handle from registration API.
* 
* @returns  none
*   
* @notes  
* 
* @end
*********************************************************************/
void daiIpAddrValidateModeGet(openapiClientHandle_t *clientHandle)
{
  open_error_t   result;
  OPEN_CONTROL_t mode;

  if ((result = openapiDaiIpAddrValidateModeGet(clientHandle, &mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get global IP address validate mode. (result = %d)\n", result);
  }
  else
  {
    printf("DAI global IP address validate mode: %d (result = %d)\n", mode, result);
  }

  return; 
}

/*********************************************************************
* @purpose  DAI OpEN APIs Sanity.
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes 
* 
* @end
*********************************************************************/
void daiOpENAPIsSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t              result;
  OPEN_CONTROL_t            mode = OPEN_DISABLE;

  /* openapiDaiSourceMacValidateModeSet() */ 
  printf("\nTesting openapiDaiSourceMacValidateModeSet(): \n");
  result = openapiDaiSourceMacValidateModeSet(NULL, mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiSourceMacValidateModeSet(clientHandle, 9);
  printf("Invalid source MAC validate mode. (result = %d)\n", result);
  printf("openapiDaiSourceMacValidateModeSet() sanity successful.\n");

  /* openapiDaiSourceMacValidateModeGet() */
  printf("\nTesting openapiDaiSourceMacValidateModeGet(): \n");
  result = openapiDaiSourceMacValidateModeGet(NULL, &mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiSourceMacValidateModeGet(clientHandle, NULL);
  printf("NULL DAI source MAC validate mode parameter. (result = %d)\n", result);
  printf("openapiDaiSourceMacValidateGet() sanity successful.\n");

  /* openapiDaiDestMacValidateModeSet() */ 
  printf("\nTesting openapiDaiDestMacValidateModeSet(): \n");
  result = openapiDaiDestMacValidateModeSet(NULL, mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiDestMacValidateModeSet(clientHandle, 9);
  printf("Invalid destination MAC validate mode. (result = %d)\n", result);
  printf("openapiDaiDestMacValidateModeSet() sanity successful.\n");

  /* openapiDaiDestMacValidateModeGet() */
  printf("\nTesting openapiDaiDestMacValidateModeGet(): \n");
  result = openapiDaiDestMacValidateModeGet(NULL, &mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiDestMacValidateModeGet(clientHandle, NULL);
  printf("NULL DAI destination MAC validate mode parameter. (result = %d)\n", result);
  printf("openapiDaiDestMacValidateGet() sanity successful.\n");

  /* openapiDaiIpAddrValidateModeSet() */ 
  printf("\nTesting openapiDaiIpAddrValidateModeSet(): \n");
  result = openapiDaiIpAddrValidateModeSet(NULL, mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiIpAddrValidateModeSet(clientHandle, 9);
  printf("Invalid IP address validate mode. (result = %d)\n", result);
  printf("openapiDaiIpAddrValidateModeSet() sanity successful.\n");

  /* openapiDaiIpAddrValidateModeGet() */
  printf("\nTesting openapiDaiIpAddrValidateModeGet(): \n");
  result = openapiDaiIpAddrValidateModeGet(NULL, &mode);
  printf("NULL client handle. (result = %d)\n", result);
  result = openapiDaiIpAddrValidateModeGet(clientHandle, NULL);
  printf("NULL DAI IP address validate mode parameter. (result = %d)\n", result);
  printf("openapiDaiIpAddrValidateGet() sanity successful.\n");

  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  uint32_t arg1;

  if (argc < 2)
  {
    printDaiAppMenu();
    exit(1);
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("dai_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Dynamic ARP Inspection (DAI) API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      daiSourceMacValidateModeSet(&clientHandle, arg1);
      break;
    case 2:
      if (argc != 2)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiSourceMacValidateModeGet(&clientHandle);
      break;
    case 3:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      daiDestMacValidateModeSet(&clientHandle, arg1);
      break;
    case 4:
      if (argc != 2)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiDestMacValidateModeGet(&clientHandle);
      break;
    case 5:
      if (argc != 3)
      {
        printDaiAppMenu();
        exit(1);
      }
      arg1 = atoi(argv[2]);
      daiIpAddrValidateModeSet(&clientHandle, arg1);
      break;
    case 6:
      if (argc != 2)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiIpAddrValidateModeGet(&clientHandle);
      break;
    case 7:
      if (argc != 2)
      {
        printDaiAppMenu();
        exit(1);
      }
      daiOpENAPIsSanity(&clientHandle);
      break;
    default:
      printDaiAppMenu();
      break;
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Dynamic ARP Inspection (DAI) API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}
