#!/mnt/fastpath/usr/bin/python

"""instru_example.py: OpEN API Instrumentation BST example"""

import OpEN_py as OpEN
from OpENUtil import *

open_ = OpENUtil()

#
# Copyright 2016 Broadcom.
#
#  Licensed under the Apache License, Version 2.0 (the "License")
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#

#
# Python 2.6.6
#

# Define some constants for comparison convenience

def print_sanity_results(result, test, msg, feat):
  """Print overall comparision results"""

  if result == OpEN.OPEN_E_UNAVAIL:
    print "Sanity test skipped."
  elif result == OpEN.OPEN_E_NONE and test == True:
    print "Sanity Success - %s - %s." % (msg, feat)
  else:
    print "Sanity Failure - %s - %s." % (msg, feat)

def print_bad_result(result, msg):
  """Print some general error messages if the result is bad"""
   
  if result == OpEN.OPEN_E_UNAVAIL:
    print "Feature not supported - %s (err %d)." % (msg, result)
  elif result != OpEN.OPEN_E_NONE:
    print "Test Failure - %s (err %d)." % (msg, result)  

class InstruExample:
  """Simple Instru Examples """

  def __init__(self, client, asic) :
    self.m_client = client
    self.m_asic = asic
    self.m_ports = 0
    self.m_ucq = 0                  
    self.m_ucqg = 0                  
    self.m_mcq = 0                  
    self.m_spool = 0
    self.m_cpuq = 0                  
    self.m_rqe = 0                  
    self.m_pg = 0


  def show_asic_capability(self):
    asic_cap_p = OpEN.new_OPEN_ASIC_CAPABILITIES_tp()
    result=OpEN.openapiBstAsicCapabilityGet(self.m_client, self.m_asic, asic_cap_p)
    if (result != OpEN.OPEN_E_NONE):
      print "Error ! failed to get asic capability"
      OpEN.delete_OPEN_ASIC_CAPABILITIES_tp(asic_cap_p)
      return                             

    asic_cap = OpEN.OPEN_ASIC_CAPABILITIES_tp_value(asic_cap_p)
    print "Asic Capability"
    print "==============================================="
    print "Asic numPorts - %d" % (asic_cap.numPorts)  
    print "Asic numUnicastQueues - %d" % (asic_cap.numUnicastQueues)  
    print "Asic numUnicastQueueGroups - %d" % (asic_cap.numUnicastQueueGroups)  
    print "Asic numMulticastQueues - %d" % (asic_cap.numMulticastQueues)  
    print "Asic numServicePools - %d" % (asic_cap.numServicePools)  
    print "Asic numCommonPools - %d" % (asic_cap.numCommonPools)  
    print "Asic numCpuQueues - %d" % (asic_cap.numCpuQueues)  
    print "Asic numRqeQueues - %d" % (asic_cap.numRqeQueues)  
    print "Asic numRqeQueuePools - %d" % (asic_cap.numRqeQueuePools)  
    print "Asic numPriorityGroups - %d" % (asic_cap.numPriorityGroups)  

    self.m_ports = asic_cap.numPorts
    self.m_ucq = asic_cap.numUnicastQueues               
    self.m_ucqg = asic_cap.numUnicastQueueGroups        
    self.m_mcq = asic_cap.numMulticastQueues                  
    self.m_cpuq = asic_cap.numCpuQueues                  
    self.m_rqe = asic_cap.numRqeQueues             
    self.m_spool = asic_cap.numServicePools                  
    self.m_pg = asic_cap.numPriorityGroups 

    OpEN.delete_OPEN_ASIC_CAPABILITIES_tp(asic_cap_p)

  def config_ipsp_threshold(self, spool, port, thre):
    print ""
    print "Setting IPSP Threshold"
    print "========================================="
    ipspThre_p = OpEN.new_OPEN_BST_INGRESS_PORT_SP_THRESHOLD_tp()
    ipspThre = OpEN.OPEN_BST_INGRESS_PORT_SP_THRESHOLD_tp_value(ipspThre_p)

    ipspThre.umShareThreshold = thre;
                                  
    print "Setting threshold %d for service pool %d , port %d" % (thre, spool, port)

    OpEN.OPEN_BST_INGRESS_PORT_SP_THRESHOLD_tp_assign(ipspThre_p, ipspThre)
    result=OpEN.openapiBstIpspThresholdSet(self.m_client, self.m_asic, port, spool, ipspThre_p)

    if (result != OpEN.OPEN_E_NONE) :
      print "Error ! failed to set IPSP threshold"

    OpEN.delete_OPEN_BST_INGRESS_PORT_SP_THRESHOLD_tp(ipspThre_p)

  def show_ippg_stat(self):
    """ Build all arguments to be passed """
    ippg_p = OpEN.new_OPEN_BST_INGRESS_PORT_PG_DATA_tp()
    time_p = OpEN.new_time_tp()
    cnt_p = OpEN.new_uint32_tArray(2)                           

    result = OpEN.openapiBstIppgDataGet(self.m_client, self.m_asic, ippg_p, time_p)
    if (result != OpEN.OPEN_E_NONE) :
      print "Error ! failed to get IPPG stat"
    else:
      print "Showing IPPG Statistics"
      print "==============================================="
      for port in range(0, self.m_ports):

        for pg in range(0, self.m_pg):
          print "Port %d, Pririty Group %d - " % (port + 1, pg),
          OpEN.GetIPPGData(ippg_p, port, pg, cnt_p)
          print "[%d , %d]" % (OpEN.uint32_tArray_getitem(cnt_p, 0),
                           OpEN.uint32_tArray_getitem(cnt_p, 1))

    OpEN.delete_time_tp(time_p)
    OpEN.delete_uint32_tp(cnt_p)
    OpEN.delete_OPEN_BST_INGRESS_PORT_PG_DATA_tp(ippg_p)

  def config_bst_feature(self, enable, mode):
    print ""
    print "Setting and Getting Bst feature"
    print "========================================="
    bstCfg_p = OpEN.new_OPEN_BST_CONFIG_tp()
    bstCfg = OpEN.OPEN_BST_CONFIG_tp_value(bstCfg_p)
    bstCfg.enableStatsMonitoring = enable
    bstCfg.enableDeviceStatsMonitoring = 1                                
    bstCfg.enableIngressStatsMonitoring = 1                                
    bstCfg.enableEgressStatsMonitoring = 0
    bstCfg.mode = mode                                       
                                  
    print "Setting values and getting - enableStatsMonitoring - %d, mode - %d" % (bstCfg.enableStatsMonitoring, bstCfg.mode)
    OpEN.OPEN_BST_CONFIG_tp_assign(bstCfg_p, bstCfg)
    result=OpEN.openapiBstConfigSet(self.m_client, self.m_asic, bstCfg_p)

    if result == OpEN.OPEN_E_NONE :
      #reset values to 0 
      bstCfg.enableStatsMonitoring = 0
      bstCfg.mode  = 0
      OpEN.OPEN_BST_CONFIG_tp_assign(bstCfg_p, bstCfg)
      result=OpEN.openapiBstConfigGet(self.m_client, self.m_asic, bstCfg_p)
      bstCfg = OpEN.OPEN_BST_CONFIG_tp_value(bstCfg_p)
    else:
      print "Error ! failed to set bst feature"
      

    print_sanity_results(result, ((bstCfg.enableStatsMonitoring == enable) and (bstCfg.mode == mode)) , "bst feature config ", "set and get")

    OpEN.delete_OPEN_BST_CONFIG_tp(bstCfg_p)

  def show_esp_stat(self):
    """ Build all arguments to be passed """
    esp_p = OpEN.new_OPEN_BST_EGRESS_SP_DATA_tp()
    time_p = OpEN.new_time_tp()
    cnt_p = OpEN.new_uint32_tArray(3)                           

    result = OpEN.openapiBstEspDataGet(self.m_client, self.m_asic, esp_p, time_p)
    if (result != OpEN.OPEN_E_NONE) :
      print "Error ! failed to get ESP stat"
    else:
      print ""
      print "Showing ESP Statistics"
      print "==============================================="
      for sp in range(0, self.m_spool):
        print "Service Pool %d - " % (sp),
        OpEN.GetESPData(esp_p, sp, cnt_p)
        print "[%d , %d, %d]" % (OpEN.uint32_tArray_getitem(cnt_p, 0),
                         OpEN.uint32_tArray_getitem(cnt_p, 1), OpEN.uint32_tArray_getitem(cnt_p, 2))

    OpEN.delete_time_tp(time_p)
    OpEN.delete_uint32_tp(cnt_p)
    OpEN.delete_OPEN_BST_EGRESS_SP_DATA_tp(esp_p)

  def show_ipsp_threshold(self):
    """ Build all arguments to be passed """
    snap_p = OpEN.new_OPEN_BST_ASIC_SNAPSHOT_DATA_tp()
    ucq_p = OpEN.new_OPEN_BST_EGRESS_UC_QUEUE_DATA_tp()
    mcq_p = OpEN.new_OPEN_BST_EGRESS_MC_QUEUE_DATA_tp()
    time_p = OpEN.new_time_tp()
    thre_p = OpEN.new_uint32_tp()                           

    """ Call OpEN API to get threshold values """
    result = OpEN.openapiBstThresholdGet(self.m_client, self.m_asic, snap_p, ucq_p, mcq_p, time_p)
    snap = OpEN.OPEN_BST_ASIC_SNAPSHOT_DATA_tp_value(snap_p)
    ipsp_p = OpEN.copy_OPEN_BST_INGRESS_PORT_SP_DATA_tp(snap.iPortSp)

    if result != OpEN.OPEN_E_NONE :
      print "Error ! failed to get IPSP threshold"
    else:
      print ""
      print "Showing IPSP Threshold"
      print "==============================================="
      for port in range(0, self.m_ports):
        print "Port %d - [" % (port + 1),
        for sp in range(0, self.m_spool):
          OpEN.GetIPSPData(ipsp_p, port, sp, thre_p)
          thre = OpEN.uint32_tp_value(thre_p)
          if sp > 0 :                                
            print ", %d" % (thre),
          else :                                  
            print "%d " % (thre),
        print " ]"

    OpEN.delete_time_tp(time_p)
    OpEN.delete_uint32_tp(thre_p)
    OpEN.delete_OPEN_BST_ASIC_SNAPSHOT_DATA_tp(snap_p)
    OpEN.delete_OPEN_BST_EGRESS_UC_QUEUE_DATA_tp(ucq_p)
    OpEN.delete_OPEN_BST_EGRESS_MC_QUEUE_DATA_tp(mcq_p)
    OpEN.delete_OPEN_BST_INGRESS_PORT_SP_DATA_tp(ipsp_p)

def main():
  """Demonstrate OpEN usage for BST APIs"""

  ret = open_.connect("instru_example")
  if ret == OpEN.OPEN_E_NONE :
    open_.getNetworkOSVersion()
    client = open_.get_client()

    example = InstruExample(client, 0)
    example.show_asic_capability()
    example.config_bst_feature(1, OpEN.OPEN_BST_MODE_PEAK)
    example.show_ippg_stat()
    example.show_esp_stat()
    example.show_ipsp_threshold()
    example.config_ipsp_threshold(0, 5, 4160)
    example.config_ipsp_threshold(0, 7, 2288)
    example.config_ipsp_threshold(0, 8, 7696)
    example.show_ipsp_threshold()

    open_.terminate()
  else :
    print "Unable to connect"

if __name__ == '__main__': main()
