/*********************************************************************
*
*  Copyright 2020 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  mvrp_example.c
*
* @purpose   OpEN mvrp example.
*
* @component OpEN
*
* @create    06/18/2020
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_mvrp.h"
#define L7_DOT1Q_MAX_VLAN_ID
/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the mvrp Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Get the global MVRP mode: %s 1 n", name);
  printf("Test 2: set the global MVRP mode: %s 2 <mode>(0-Disable,1-Enable)\n", name);
  printf("Test 3: Gets the MVRP periodic state machine mode: %s 3 n", name);
  printf("Test 4: Sets the MVRP periodic state machine mode: %s 4 <mode>\n", name);
  printf("Test 5: Get the MVRP mode for the specified interface: %s 5 <intIfNum> <mode>(0-Disable,1-Enable)\n", name);
  printf("Test 6: Set the MVRP mode for the given interface: %s 6 <intIfNum><mode>(0-Disable,1-Enable)\n", name);
  printf("Test 7: Check if the current interface is valid for MVRP: %s 7 <intIfNum>\n", name);
  printf("Test 8: Given the current interface, get the next valid MVRP interface: %s 8 <intIfNum>\n", name);
  printf("Test 9: Get the value of specified MVRP counter: %s 9 <type> \n", name);
  printf("Test 10: Get the value of specified MVRP counter for an interface: %s 10 <intIfNum> <type> \n", name);
  printf("Test 11: Get MVRP VLAN array in the system: %s 11 <vlanType>>\n", name);
  printf("Test 12: Get MVRP VLAN array on an interface: %s 12 <intIfNum> <vlanType>\n", name);
  printf("Test 13: Clear MVRP traffic counters on specified interface: %s 13 <intIfNum>\n", name);
  printf("Test 14: Clear global MVRP traffic counters: %s 14 \n", name);
  printf("Test 15: Clear all MVRP traffic counters: %s 15 \n", name);
  printf("Test 16: Run API sanity checks: %s 16 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  uint32_t count;
  OPEN_VLAN_LIST_t vlanList;
  OPEN_BOOL_t isValid;
  uint32_t counter;
  OPEN_CONTROL_t mode;
  OPEN_MVRP_VLAN_TYPE_t vlanType;
  OPEN_MVRP_STATS_t type;
  uint32_t intIfNum, nextIfNum;

  uint32_t      max_interfaces;

  if (OPEN_E_NONE != openapiMaxInterfaceCountGet(clientHandle, &max_interfaces))
  {
    printf ("Can't get Interface Count.\n");
  }

  printf("Testing mvrp OpEN APIs sanity:\n");

  printf("Testing openapiMvrpModeGet():\n");
  result = openapiMvrpModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMvrpModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiMvrpPeriodicStateMachineModeGet():\n");
  result = openapiMvrpPeriodicStateMachineModeGet(NULL, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMvrpPeriodicStateMachineModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiMvrpIntfModeGet():\n");
  intIfNum = 1;
  result = openapiMvrpIntfModeGet(NULL, intIfNum, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMvrpIntfModeGet(clientHandle, max_interfaces + 1, &mode);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiMvrpIntfModeGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiMvrpIsValidIntfGet():\n");
  result = openapiMvrpIsValidIntfGet(NULL, intIfNum, &isValid);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMvrpIsValidIntfGet(clientHandle, max_interfaces + 1, &isValid);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiMvrpIsValidIntfGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiMvrpValidIntfNextGet():\n");
  result = openapiMvrpValidIntfNextGet(NULL, intIfNum, &nextIfNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMvrpValidIntfNextGet(clientHandle, max_interfaces + 1, &nextIfNum);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiMvrpValidIntfNextGet(clientHandle, intIfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiMvrpTrafficPduCounterGet():\n");
  type = 0;
  result = openapiMvrpTrafficPduCounterGet(NULL, type, &counter);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMvrpTrafficPduCounterGet(clientHandle, type, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiMvrpIntfTrafficPduCounterGet():\n");
  result = openapiMvrpIntfTrafficPduCounterGet(NULL, intIfNum, type, &counter);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMvrpIntfTrafficPduCounterGet(clientHandle, max_interfaces + 1, type, &counter);
  printf("greater than maximum value argument 4:(result = %d)\n", result);
  result = openapiMvrpIntfTrafficPduCounterGet(clientHandle, intIfNum, type, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiMvrpVlanArrayGet():\n");
  vlanType = 0;
  result = openapiMvrpVlanArrayGet(NULL, vlanType, &vlanList, &count);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMvrpVlanArrayGet(clientHandle, vlanType, NULL, &count);
  printf("NULL argument 4:(result = %d)\n", result);
  result = openapiMvrpVlanArrayGet(clientHandle, vlanType, &vlanList, NULL);
  printf("NULL argument 4:(result = %d)\n", result);

  printf("Testing openapiMvrpIntfVlanArrayGet():\n");
  result = openapiMvrpIntfVlanArrayGet(NULL, intIfNum, vlanType, &vlanList, &count);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMvrpIntfVlanArrayGet(clientHandle, max_interfaces + 1, vlanType, &vlanList, &count);
  printf("greater than maximum value argument 5:(result = %d)\n", result);
  result = openapiMvrpIntfVlanArrayGet(clientHandle, intIfNum, vlanType, NULL, &count);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiMvrpIntfVlanArrayGet(clientHandle, intIfNum, vlanType, &vlanList, NULL);
  printf("NULL argument 5:(result = %d)\n", result);
  result = openapiMvrpIntfVlanArrayGet(clientHandle, intIfNum, vlanType, &vlanList, &count);
  printf("greater than maximum value argument 5:(result = %d)\n", result);

  printf("Testing openapiMvrpTrafficCountersPerIfClear():\n");
  result = openapiMvrpTrafficCountersPerIfClear(NULL, intIfNum);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiMvrpTrafficCountersPerIfClear(clientHandle, max_interfaces + 1);
  printf("greater than maximum value argument 2:(result = %d)\n", result);

  printf("Testing openapiMvrpTrafficGlobalCountersClear():\n");
  result = openapiMvrpTrafficGlobalCountersClear(NULL);
  printf("NULL client handle:(result = %d)\n", result);

  printf("Testing openapiMvrpTrafficCountersClear():\n");
  result = openapiMvrpTrafficCountersClear(NULL);
  printf("NULL client handle:(result = %d)\n", result);

  return;
}

/*********************************************************************
* @purpose  Get the global MVRP mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(output)}  Global admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiMvrpModeGet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the global MVRP mode. (result = %d)\n", result);
  }
  else
  {
    printf("Global MVRP mode is %d\n", *mode);
  }
  return;
}


/*********************************************************************
* @purpose  set the global MVRP mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Global admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiMvrpModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the global MVRP mode. (result = %d)\n", result);
  }
  else
  {
    printf("MVRP mode is set\n");
  }
  return;
}


/*********************************************************************
* @purpose  Gets the MVRP periodic state machine mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(output)}  Periodic state machine mode.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpPeriodicStateMachineModeGet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiMvrpPeriodicStateMachineModeGet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets the MVRP periodic state machine mode. (result = %d)\n", result);
  }
  else
  {
    printf("MVRP periodic state machine mode is %d\n", *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Sets the MVRP periodic state machine mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    mode       @b{(input)}  Periodic state machine mode.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpPeriodicStateMachineModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t  mode)
{
  open_error_t result;

  if ((result = openapiMvrpPeriodicStateMachineModeSet(client_handle, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to sets the MVRP periodic state machine mode. (result = %d)\n", result);
  }
  else
  {
    printf("MVRP Periodic state machine mode set\n");
  }
  return;
}


/*********************************************************************
* @purpose  Get the MVRP mode for the specified interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    mode       @b{(output)}  Interface MVRP admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpIntfModeGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiMvrpIntfModeGet(client_handle, intIfNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the MVRP mode for the specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("MVRP mode on interface %d is %d\n", intIfNum, *mode);
  }
  return;
}


/*********************************************************************
* @purpose  Set the MVRP mode for the given interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    mode       @b{(input)}  Interface MVRP admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpIntfModeSet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_CONTROL_t mode)
{
  open_error_t result;

  if ((result = openapiMvrpIntfModeSet(client_handle, intIfNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set the MVRP mode for the given interface. (result = %d)\n", result);
  }
  else
  {
    printf("MVRP mode on interface %d is set\n", intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Check if the current interface is valid for MVRP.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    isValid       @b{(output)}  Flag to tell if Interface is valid MVRP interface.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpIsValidIntfGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_BOOL_t *isValid)
{
  open_error_t result;

  if ((result = openapiMvrpIsValidIntfGet(client_handle, intIfNum, isValid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if the current interface is valid for MVRP. (result = %d)\n", result);
  }
  else
  {
    if (*isValid == OPEN_TRUE)
    {
      printf("\n This is valid MVRP interface \n");
    }
    else
    {
      printf("\n This is not valid MVRP interface \n");
    }
  }
  return;
}


/*********************************************************************
* @purpose  Given the current interface, get the next valid MVRP interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    nextIfNum       @b{(output)}  Next valid MVRP Interface Number.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpValidIntfNextGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, uint32_t *nextIfNum)
{
  open_error_t result;

  if ((result = openapiMvrpValidIntfNextGet(client_handle, intIfNum, nextIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to given the current interface, get the next valid MVRP interface. (result = %d)\n", result);
  }
  else
  {
    printf("Next valid MVRP interface is %d\n", *nextIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Get the value of specified MVRP counter.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    type       @b{(input)}  Type of counter.
* @param    counter       @b{(output)}  Counter value.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpTrafficPduCounterGet(openapiClientHandle_t *client_handle, OPEN_MVRP_STATS_t type, uint32_t *counter)
{
  open_error_t result;

  if ((result = openapiMvrpTrafficPduCounterGet(client_handle, type, counter)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the value of specified MVRP counter. (result = %d)\n", result);
  }
  else
  {
    printf("MSRP Pdu counter of type %d is %d\n", type, *counter);
  }
  return;
}


/*********************************************************************
* @purpose  Get the value of specified MVRP counter for an interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    type       @b{(input)}  Type of counter.
* @param    counter       @b{(output)}  Counter value.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpIntfTrafficPduCounterGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MVRP_STATS_t type, uint32_t *counter)
{
  open_error_t result;

  if ((result = openapiMvrpIntfTrafficPduCounterGet(client_handle, intIfNum, type, counter)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get the value of specified MVRP counter for an interface. (result = %d)\n", result);
  }
  else
  {
    printf("MVRP Pdu counter of type %d on interface %d is %dn",
	   type, intIfNum, *counter);
  }
  return;
}


/*********************************************************************
* @purpose  Get MVRP VLAN array in the system.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    vlanType       @b{(input)}  Flag to tell if vlan is Declared/Registered.
* @param    vlanList       @b{(output)}  MVRP VLAN List in the system.
* @param    count       @b{(output)}  Number of MVRP Vlans.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpVlanArrayGet(openapiClientHandle_t *client_handle, OPEN_MVRP_VLAN_TYPE_t vlanType, OPEN_VLAN_LIST_t *vlanList, uint32_t *count)
{
  open_error_t result;
  int i = 0, max = 0;

  if ((result = openapiMvrpVlanArrayGet(client_handle, vlanType, vlanList, count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get MVRP VLAN array in the system. (result = %d)\n", result);
  }
  else
  {
    max = *count;
    printf("\n MVRP Vlan list is\n");
    for(i=0;i<max;i++)
    {
      printf("%d,", vlanList->ids[i]);
    }
  }
  return;
}


/*********************************************************************
* @purpose  Get MVRP VLAN array on an interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.
* @param    vlanType       @b{(input)}  Flag to tell if vlan is Declared/Registered.
* @param    vlanList       @b{(output)}  MVRP VLAN List on the interface given.
* @param    count       @b{(output)}  Number of MVRP Vlans.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpIntfVlanArrayGet(openapiClientHandle_t *client_handle, uint32_t intIfNum, OPEN_MVRP_VLAN_TYPE_t vlanType, OPEN_VLAN_LIST_t *vlanList, uint32_t *count)
{
  open_error_t result;
  int i = 0, max = 0;

  if ((result = openapiMvrpIntfVlanArrayGet(client_handle, intIfNum, vlanType, vlanList, count)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get MVRP VLAN array on an interface. (result = %d)\n", result);
  }
  else
  {
    max = *count;
    printf("\n MVRP Vlan list is\n");
    for(i=0;i<max;i++)
    {
      printf("%d,", vlanList->ids[i]);
    }
  }
  return;
}


/*********************************************************************
* @purpose  Clear MVRP traffic counters on specified interface.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  Interface Number.

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpTrafficCountersPerIfClear(openapiClientHandle_t *client_handle, uint32_t intIfNum)
{
  open_error_t result;

  if ((result = openapiMvrpTrafficCountersPerIfClear(client_handle, intIfNum)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear MVRP traffic counters on specified interface. (result = %d)\n", result);
  }
  else
  {
    printf("MVRP Counters on interface %d cleared\n", intIfNum);
  }
  return;
}


/*********************************************************************
* @purpose  Clear global MVRP traffic counters.
*
* @param    client_handle       @b{(input)}  client handle from registration API

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpTrafficGlobalCountersClear(openapiClientHandle_t *client_handle)
{
  open_error_t result;

  if ((result = openapiMvrpTrafficGlobalCountersClear(client_handle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear global MVRP traffic counters. (result = %d)\n", result);
  }
  else
  {
    printf("MVRP Global counters are cleared.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Clear all MVRP traffic counters.
*
* @param    client_handle       @b{(input)}  client handle from registration API

*
* @returns  none
*
* @end
*********************************************************************/
void mvrpTrafficCountersClear(openapiClientHandle_t *client_handle)
{
  open_error_t result;

  if ((result = openapiMvrpTrafficCountersClear(client_handle)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to clear all MVRP traffic counters. (result = %d)\n", result);
  }
  else
  {
    printf("All MVRP Traffic ounters are cleared.\n");
  }
  return;
}



/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for mvrp
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_VLAN_LIST_t vlanList;
  OPEN_BOOL_t isValid;
  uint32_t counter, count;
  OPEN_CONTROL_t mode;
  OPEN_MVRP_VLAN_TYPE_t vlanType;
  OPEN_MVRP_STATS_t type;
  uint32_t intIfNum;
  uint32_t nextIfNum;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("mvrp example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting mvrp API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 2)
      {
        mvrpModeGet(&client_handle, &mode);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
	mode = atoi(argv[2]);
        mvrpModeSet(&client_handle, mode);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 2)
      {
        mvrpPeriodicStateMachineModeGet(&client_handle, &mode);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
	mode = atoi(argv[2]);
        mvrpPeriodicStateMachineModeSet(&client_handle, mode);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
	intIfNum = atoi(argv[2]);
        mvrpIntfModeGet(&client_handle, intIfNum, &mode);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 4)
      {
	intIfNum = atoi(argv[2]);
	mode = atoi(argv[3]);
        mvrpIntfModeSet(&client_handle, intIfNum, mode);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 3)
      {
	intIfNum = atoi(argv[2]);
        mvrpIsValidIntfGet(&client_handle, intIfNum, &isValid);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
	intIfNum = atoi(argv[2]);
        mvrpValidIntfNextGet(&client_handle, intIfNum, &nextIfNum);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
	type = atoi(argv[2]);
        mvrpTrafficPduCounterGet(&client_handle, type, &counter);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 4)
      {
	intIfNum = atoi(argv[2]);
	type = atoi(argv[3]);
        mvrpIntfTrafficPduCounterGet(&client_handle, intIfNum, type, &counter);
        show_help = 0;
      }
      break;
    case 11:
      if (argc == 3)
      {
	vlanType = atoi(argv[2]);
        mvrpVlanArrayGet(&client_handle, vlanType, &vlanList, &count);
        show_help = 0;
      }
      break;
    case 12:
      if (argc == 4)
      {
	intIfNum = atoi(argv[2]);
	vlanType = atoi(argv[3]);
        mvrpIntfVlanArrayGet(&client_handle, intIfNum, vlanType, &vlanList, &count);
        show_help = 0;
      }
      break;
    case 13:
      if (argc == 3)
      {
	intIfNum = atoi(argv[2]);
        mvrpTrafficCountersPerIfClear(&client_handle, intIfNum);
        show_help = 0;
      }
      break;
    case 14:
      if (argc == 2)
      {
        mvrpTrafficGlobalCountersClear(&client_handle);
        show_help = 0;
      }
      break;
    case 15:
      if (argc == 2)
      {
        mvrpTrafficCountersClear(&client_handle);
        show_help = 0;
      }
      break;
    case 16:
      if (argc == 2)
      {
        runSanity(&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping msrp API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
