/*********************************************************************
*
* (C) Copyright Broadcom 2016-2018
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  pbvlan_example.c
*
* @purpose   Protocol-Based VLAN OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    04/25/2016
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_pbvlan.h"
#include "openapi_vlan.h"

#define MAX_STRING_LEN  256

/*******************************************************************
*
* @brief  This is the main function that will demonstrates
*         Protocol-Based VLAN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result = OPEN_E_NONE;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100] = {0};

  uint32_t groupId = 0;
  uint32_t newGroupId = 0;
  uint32_t vlanId = 0;
  uint32_t newVlanId = 0;
  uint32_t prot = 0;
  uint32_t ifNum = 0;
  uint32_t nextIfNum = 0;
  OPENAPI_PBVLAN_PROT_TYPE_t protType = OPENAPI_PBVLAN_STR_PROT_TYPE;
  char bufGroupName[17] = {0};
  open_buffdesc groupName = {.pstart = bufGroupName, .size = 0};
  char bufProtList[512] = {0};
  open_buffdesc protList = {.pstart = bufProtList, .size = sizeof(bufProtList)};

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("pbvlan_example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (Result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fails until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting Protocol-Based VLAN API example application");

  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&client_handle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("\nNetwork OS version = %s\n\n", switch_os_revision_string);
  }
  else
  {
    printf ("\nNetwork OS version retrieve error\n\n");
  }

  /* exercise various Protocol-Based VLAN OpEN API functions */
  do
  {
    /* Example of creating new Protocol-Based VLAN group */
    groupId = 10;
    result = openapiPbVlanGroupCreate(&client_handle, groupId);
    switch (result)
    {
      case OPEN_E_NONE:
        printf("The new Protocol-Based VLAN group with specified ID (%d) has been successfully created. (result = %d)\n",
               groupId, result);
        break;
      case OPEN_E_EXISTS:
        printf("Protocol-Based VLAN group with specified ID (%d) already exists. (result = %d)\n",
               groupId, result);
        break;
      case OPEN_E_PARAM:
        printf("Invalid parameter was passed. (result = %d)\n",
               result);
        break;
      default:
        printf("Failed to created new Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
               groupId, result);
        break;
    }

    if ((OPEN_E_NONE != result) && 
        (OPEN_E_NONE != OPEN_E_EXISTS))
    {
      break;
    }

    /* Example of getting the ID of the first created Protocol-Based VLAN group */
    result = openapiPbVlanGroupGetFirst(&client_handle, &newGroupId);
    switch (result)
    {
      case OPEN_E_NONE:
        printf("ID of the first created Protocol-Based VLAN group is %d. (result = %d)\n",
               newGroupId, result);
        break;
      case OPEN_E_PARAM:
        printf("Invalid parameter was passed. (result = %d)\n",
               result);
        break;
      default:
        printf("Failed to get ID of the first Protocol-Based VLAN group. (result = %d)\n",
               result);
        break;
    }

    /* Example of getting the ID of the next created Protocol-Based VLAN group */
    result = openapiPbVlanGroupGetNext(&client_handle, groupId, &newGroupId);
    switch (result)
    {
      case OPEN_E_NONE:
        printf("ID of the next created Protocol-Based VLAN group is %d. (result = %d)\n",
               newGroupId, result);
        break;
      case OPEN_E_NOT_FOUND:
        printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
               groupId, result);
        break;
      case OPEN_E_PARAM:
        printf("Invalid parameter was passed. (result = %d)\n",
               result);
        break;
      default:
        printf("Failed to get ID of the next Protocol-Based VLAN group. (result = %d)\n",
               result);
        break;
    }

    /* Example of setting the name of the Protocol-Based VLAN group with specified ID */
    strcpy(groupName.pstart, "new_name");
    groupName.size = strlen(groupName.pstart) + 1;
    result = openapiPbVlanGroupNameSet(&client_handle, groupId, &groupName);
    switch (result)
    {
      case OPEN_E_NONE:
        printf("The name (%s) of the Protocol-Based VLAN group with specified ID (%d) has been successfully set. (result = %d)\n",
               bufGroupName, groupId, result);
        break;
      case OPEN_E_NOT_FOUND:
        printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
               groupId, result);
        break;
      case OPEN_E_PARAM:
        printf("Invalid parameter was passed. (result = %d)\n",
               result);
        break;
      default:
        printf("Failed to set the name (%s) of the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
               bufGroupName, groupId, result);
        break;
    }

    if (OPEN_E_NONE == result)
    {
      /* Example of getting the name of the Protocol-Based VLAN group with specified ID */
      memset(bufGroupName, 0, sizeof(bufGroupName));
      groupName.size = 17;
      result = openapiPbVlanGroupNameGet(&client_handle, groupId, &groupName);
      switch (result)
      {
        case OPEN_E_NONE:
          printf("The name of the Protocol-Based VLAN group with specified ID (%d) is %s. (result = %d)\n",
                 groupId, bufGroupName, result);
          break;
        case OPEN_E_NOT_FOUND:
          printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
                 groupId, result);
          break;
        case OPEN_E_PARAM:
          printf("Invalid parameter was passed. (result = %d)\n",
                 result);
          break;
        default:
          printf("Failed to get the name of the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 groupId, result);
          break;
      }

      /* Example of getting the ID of the Protocol-Based VLAN group with specified name */
      result = openapiPbVlanGroupIdGet(&client_handle, &groupName, &newGroupId);
      switch (result)
      {
        case OPEN_E_NONE:
          printf("The ID of the Protocol-Based VLAN group with specified name (%s) is %d. (result = %d)\n",
                 bufGroupName, newGroupId, result);
          break;
        case OPEN_E_PARAM:
          printf("Invalid parameter was passed. (result = %d)\n",
                 result);
          break;
        default:
          printf("Failed to get the ID of the Protocol-Based VLAN group with specified name (%s). (result = %d)\n",
                 bufGroupName, result);
          break;
      }
    }

    vlanId = 10;
    if (OPEN_E_NONE == openapiVlanCreate(&client_handle, vlanId))
    {
      /* Example of adding the VLAN ID to the Protocol-Based VLAN group with specified ID */
      result = openapiPbVlanGroupVlanIDAdd(&client_handle, groupId, vlanId);
      switch (result)
      {
        case OPEN_E_NONE:
          printf("The VLAN (%d) has been successfully added to the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 vlanId, groupId, result);
          break;
        case OPEN_E_NOT_FOUND:
          printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
                 groupId, result);
          break;
        case OPEN_E_PARAM:
          printf("Invalid parameter was passed. (result = %d)\n",
                 result);
          break;
        default:
          printf("Failed to add the VLAN (%d) to the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 vlanId, groupId, result);
          break;
      }

      if ((OPEN_E_NONE == result) ||
          (OPEN_E_EXISTS == result))
      {
        /* Example of getting the VLAN ID of the Protocol-Based VLAN group with specified ID */
        result = openapiPbVlanGroupVlanIDGet(&client_handle, groupId, &newVlanId);
        switch (result)
        {
          case OPEN_E_NONE:
            printf("The VLAN ID of the Protocol-Based VLAN group with specified ID (%d) is %d. (result = %d)\n",
                   groupId, newVlanId, result);
            break;
          case OPEN_E_NOT_FOUND:
            printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
                   groupId, result);
            break;
          case OPEN_E_PARAM:
            printf("Invalid parameter was passed. (result = %d)\n",
                   result);
            break;
          default:
            printf("Failed to get the VLAN ID of the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                   groupId, result);
            break;
        }

        /* Example of deleting the VLAN ID from the Protocol-Based VLAN group with specified ID */
        result = openapiPbVlanGroupVlanIDDelete(&client_handle, groupId, vlanId);
        switch (result)
        {
          case OPEN_E_NONE:
            printf("The VLAN (%d) has been successfully deleted from the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                   vlanId, groupId, result);
            break;
          case OPEN_E_NOT_FOUND:
            printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
                   groupId, result);
            break;
          case OPEN_E_ERROR:
            printf("VLAN (%d) was not found in the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                   vlanId, groupId, result);
            break;
          case OPEN_E_PARAM:
            printf("Invalid parameter was passed. (result = %d)\n",
                   result);
            break;
          default:
            printf("Failed to delete the VLAN (%d) from the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                   vlanId, groupId, result);
            break;
        }
      }

      (void) openapiVlanDelete(&client_handle, vlanId);
    }

    /* Example of adding the protocol to the Protocol-Based VLAN group with specified ID */
    prot = 0x0800;
    protType = OPENAPI_PBVLAN_NUM_PROT_TYPE;
    result = openapiPbVlanGroupProtocolAdd(&client_handle, groupId, prot, protType);
    switch (result)
    {
      case OPEN_E_NONE:
        printf("The protocol (%d) has been successfully added to the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
               prot, groupId, result);
        break;
      case OPEN_E_NOT_FOUND:
        printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
               groupId, result);
        break;
      case OPEN_E_EXISTS:
        printf("Protocol (%d) is already added to the group with specified ID (%d). (result = %d)\n",
               prot, groupId, result);
        break;
      case OPEN_E_PARAM:
        printf("Invalid parameter was passed. (result = %d)\n",
               result);
        break;
      default:
        printf("Failed to add the protocol (%d) to the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
               prot, groupId, result);
        break;
    }

    if ((OPEN_E_NONE == result) ||
        (OPEN_E_EXISTS == result))
    {
      /* Example of getting the list of protocols of the Protocol-Based VLAN group with specified ID */
      result = openapiPbVlanGroupProtocolGet(&client_handle, groupId, &protList);
      switch (result)
      {
        case OPEN_E_NONE:
          printf("The list of protocols of the Protocol-Based VLAN group with specified ID (%d) is %s. (result = %d)\n",
                 groupId, bufProtList, result);
          break;
        case OPEN_E_NOT_FOUND:
          printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
                 groupId, result);
          break;
        case OPEN_E_PARAM:
          printf("Invalid parameter was passed. (result = %d)\n",
                 result);
          break;
        default:
          printf("Failed to get the list of protocols of the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 groupId, result);
          break;
      }

      /* Example of deleting the protocol from the Protocol-Based VLAN group with specified ID */
      result = openapiPbVlanGroupProtocolDelete(&client_handle, groupId, prot);
      switch (result)
      {
        case OPEN_E_NONE:
          printf("The protocol (%d) has been successfully deleted from the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 prot, groupId, result);
          break;
        case OPEN_E_NOT_FOUND:
          printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
                 groupId, result);
          break;
        case OPEN_E_ERROR:
          printf("Protocol (%d) was not found in the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 prot, groupId, result);
          break;
        case OPEN_E_PARAM:
          printf("Invalid parameter was passed. (result = %d)\n",
                 result);
          break;
        default:
          printf("Failed to delete the protocol (%d) from the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 prot, groupId, result);
          break;
      }
    }

    /* Example of checking an interface to be valid for the Protocol-Based VLAN configuration */
    ifNum = 1; /* port "0/1" */
    result = openapiPbVlanIntfValidate(&client_handle, ifNum);
    switch (result)
    {
      case OPEN_E_NONE:
        printf("The interface (%d) could be used for the Protocol-Based VLAN configuration. (result = %d)\n",
               ifNum, result);
        break;
      case OPEN_E_PARAM:
        printf("Invalid parameter was passed. (result = %d)\n",
               result);
        break;
      default:
        printf("The interface (%d) could not be used for the Protocol-Based VLAN configuration. (result = %d)\n",
               ifNum, result);
        break;
    }

    /* Example of adding an interface to the Protocol-Based VLAN group with specified ID */
    result = openapiPbVlanGroupPortAdd(&client_handle, groupId, ifNum);
    switch (result)
    {
      case OPEN_E_NONE:
        printf("The interface (%d) has been successfully added to the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
               ifNum, groupId, result);
        break;
      case OPEN_E_NOT_FOUND:
        printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
               groupId, result);
        break;
      case OPEN_E_PARAM:
        printf("Invalid parameter was passed. (result = %d)\n",
               result);
        break;
      default:
        printf("Failed to add the protocol (%d) to the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
               ifNum, groupId, result);
        break;
    }

    if (OPEN_E_NONE == result)
    {
      /* Example of getting the first interface number of the Protocol-Based VLAN group with specified ID */
      result = openapiPbVlanGroupPortGetNext(&client_handle, groupId, nextIfNum, &nextIfNum);
      switch (result)
      {
        case OPEN_E_NONE:
          printf("The first interface number of the Protocol-Based VLAN group with specified ID (%d) is %d. (result = %d)\n",
                 groupId, nextIfNum, result);
          break;
        case OPEN_E_NOT_FOUND:
          printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
                 groupId, result);
          break;
        case OPEN_E_ERROR:
          printf("An interface (%d) was not found in the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 nextIfNum, groupId, result);
          break;
        case OPEN_E_PARAM:
          printf("Invalid parameter was passed. (result = %d)\n",
                 result);
          break;
        default:
          printf("Failed to get the first interface number of the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 groupId, result);
          break;
      }

      /* Example of deleting the interface number from the Protocol-Based VLAN group with specified ID */
      result = openapiPbVlanGroupPortDelete(&client_handle, groupId, ifNum);
      switch (result)
      {
        case OPEN_E_NONE:
          printf("The interface (%d) has been successfully deleted from the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 ifNum, groupId, result);
          break;
        case OPEN_E_NOT_FOUND:
          printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
                 groupId, result);
          break;
        case OPEN_E_ERROR:
          printf("Interface (%d) was not found in the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 ifNum, groupId, result);
          break;
        case OPEN_E_PARAM:
          printf("Invalid parameter was passed. (result = %d)\n",
                 result);
          break;
        default:
          printf("Failed to delete the interface (%d) from the Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
                 ifNum, groupId, result);
          break;
      }
    }

    /* Example of deleting the Protocol-Based VLAN group */
    result = openapiPbVlanGroupDelete(&client_handle, groupId);
    switch (result)
    {
      case OPEN_E_NONE:
        printf("Protocol-Based VLAN group with specified ID (%d) has been successfully deleted. (result = %d)\n",
               groupId, result);
        break;
      case OPEN_E_NOT_FOUND:
        printf("Protocol-Based VLAN group with specified ID (%d) was not found. (result = %d)\n",
               groupId, result);
        break;
      case OPEN_E_PARAM:
        printf("Invalid parameter was passed. (result = %d)\n",
               result);
        break;
      default:
        printf("Failed to delete Protocol-Based VLAN group with specified ID (%d). (result = %d)\n",
               groupId, result);
        break;
    }

    printf("Press enter to continue...\n");
    getchar();
  } while(0);

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Protocol-Based VLAN API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
