/*********************************************************************
*
*  Copyright 2019 Broadcom
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  ptp_tc_example.c
*
* @purpose   PTP TC OpEN APIs Example
*
* @component OpEN
*
* @comments
*
* @create    03/13/2019
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_ptp_tc.h"

/*
   OpEN API set functions are processed asynchronously. There may be some
   delay between when the set function call returns and when the system
   state is updated to reflect the change. These parameters control how
   long the test code retries the get functions to retrieve a change.
*/

/*******************************************************************
*
* @brief  This function prints the ptptc Example Application Menu.
*
* @param  name   @b{(input)} program name
*
* @returns  none
*
* @end
*********************************************************************/
static void printAppMenu(char *name)
{
  printf("Usage: %s <test#> <arg1> <arg2> ... \n", name);
  printf("Test 1: Check if the PTP TC is supported: %s 1 \n", name);
  printf("Test 2: Check if the unit is valid for PTP TC: %s 2 <unit>\n", name);
  printf("Test 3: Get PTP TC admin  mode: %s 3 \n", name);
  printf("Test 4: Set PTP TC admin  mode: %s 4 <adminMode 1-Enable,0-disable>\n", name);
  printf("Test 5: Determine if interface configuration can be applied in hardware: %s 5 <intfNum>\n", name);
  printf("Test 6: Get PTP TC mode on the specified interface: %s 6 <intfNum>\n", name);
  printf("Test 7: Set PTP TC mode on the specified interface: %s 6 <intfNum> <adminMode 1-Enable,0-disable>\n", name);
  printf("Test 8: Gets PTP TC operational mode on the specified interface: %s 7 <intfNum>\n", name);
  printf("Test 9: Check if the interface is valid for PTP TC: %s 8 <intfNum>\n", name);
  printf("Test 10: Run API sanity checks: %s 9 \n", name);

  return;
}

/***************************************************************************
* @purpose  Display results when incorrect inputs are passed to API.
*
* @param    clientHandle    @b{(input)}   client handle from registration API
*
* @returns  none
*
* @end
****************************************************************************/
static void runSanity(openapiClientHandle_t *clientHandle)
{
  open_error_t result;

  OPEN_BOOL_t isSupported;
  uint32_t intfNum;
  OPEN_CONTROL_t adminMode, mode;
  OPEN_BOOL_t isAllowed;
  OPEN_BOOL_t isValid;
  uint32_t unit;

  printf("Testing ptptc OpEN APIs sanity:\n");

  printf("Testing openapiPtpTcIsSupportedGet():\n");
  result = openapiPtpTcIsSupportedGet(NULL, &isSupported);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiPtpTcIsSupportedGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiPtpTcIsValidUnitGet():\n");
  unit = 1;
  result = openapiPtpTcIsValidUnitGet(NULL, unit, &isValid);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiPtpTcIsValidUnitGet(clientHandle, 1 - 1, &isValid);
  printf("less than minimum value argument 3:(result = %d)\n", result);
  result = openapiPtpTcIsValidUnitGet(clientHandle, 9, &isValid);
  printf("greater than maximum value argument 3:(result = %d)\n", result);
  result = openapiPtpTcIsValidUnitGet(clientHandle, unit, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiPtpTcAdminModeGet():\n");
  result = openapiPtpTcAdminModeGet(NULL, &adminMode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiPtpTcAdminModeGet(clientHandle, NULL);
  printf("NULL argument 2:(result = %d)\n", result);

  printf("Testing openapiPtpTcIntfHwConfigAllowedGet():\n");
  intfNum = 1;
  result = openapiPtpTcIntfHwConfigAllowedGet(NULL, intfNum, &isAllowed);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiPtpTcIntfHwConfigAllowedGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiPtpTcIntfAdminModeGet():\n");
  result = openapiPtpTcIntfAdminModeGet(NULL, intfNum, &adminMode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiPtpTcIntfAdminModeGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiPtpTcIntfAdminModeSet():\n");
  result = openapiPtpTcIntfAdminModeSet(NULL, intfNum, adminMode);
  printf("NULL client handle:(result = %d)\n", result);


  printf("Testing openapiPtpTcIntfOperAdminModeGet():\n");
  result = openapiPtpTcIntfOperAdminModeGet(NULL, intfNum, &mode);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiPtpTcIntfOperAdminModeGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  printf("Testing openapiPtpTcIsValidIntfGet():\n");
  result = openapiPtpTcIsValidIntfGet(NULL, intfNum, &isValid);
  printf("NULL client handle:(result = %d)\n", result);
  result = openapiPtpTcIsValidIntfGet(clientHandle, intfNum, NULL);
  printf("NULL argument 3:(result = %d)\n", result);

  return;
}

/*********************************************************************
* @purpose  Check if the PTP TC is supported.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    isSupported       @b{(output)}  Flag to tell if PTP TC is supported.

*
* @returns  none
*
* @end
*********************************************************************/
void ptpTcIsSupportedGet(openapiClientHandle_t *client_handle, 
                         OPEN_BOOL_t *isSupported)
{
  open_error_t result;

  if ((result = openapiPtpTcIsSupportedGet(client_handle, isSupported)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if PTP TC is supported. (result = %d)\n", result);
  }
  else
  {
    if (*isSupported == OPEN_TRUE)
    {
      printf("\n PTP TC is supported\n");
    }
    else
    {
      printf("\n PTP TC is not supported\n");
    }
  }
  return;
}


/*********************************************************************
* @purpose  Check if the unit is valid for PTP TC.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    unit       @b{(input)}  Unit number.
* @param    isValid       @b{(output)}  Flag to tell if this is Valid unit for PTP TC.

*
* @returns  none
*
* @end
*********************************************************************/
void ptpTcIsValidUnitGet(openapiClientHandle_t *client_handle, 
                         uint32_t unit, OPEN_BOOL_t *isValid)
{
  open_error_t result;

  if ((result = openapiPtpTcIsValidUnitGet(client_handle, unit, isValid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if the unit is valid for PTP TC. (result = %d)\n", result);
  }
  else
  {
    if (*isValid == OPEN_TRUE)
    {
      printf("\n This unit is valid for PTP TC\n");
    }
    else
    {
      printf("\n This unit is not valid for PTP TC\n");
    }
  }
  return;
}


/*********************************************************************
* @purpose  Get PTP TC admin  mode.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    adminMode       @b{(output)}  PTP TC admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void ptpTcAdminModeGet(openapiClientHandle_t *client_handle, 
                       OPEN_CONTROL_t *adminMode)
{
  open_error_t result;

  if ((result = openapiPtpTcAdminModeGet(client_handle, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PTP TC admin  mode. (result = %d)\n", result);
  }
  else
  {
    if (*adminMode == OPEN_ENABLE)
    {
      printf("\n PTP TC is enabled.\n");
    }
    else
    {
      printf("\n PTP TC is disabled.\n");
    }
  }
  return;
}


/*********************************************************************
* @purpose  Set PTP TC admin  mode.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    adminMode       @b{(input)}  PTP TC admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void ptpTcAdminModeSet(openapiClientHandle_t *client_handle, OPEN_CONTROL_t adminMode)
{
  open_error_t result;

  if ((result = openapiPtpTcAdminModeSet(client_handle, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set PTP TC admin  mode. (result = %d)\n", result);
  }
  else
  {
    printf("\n PTP TC admin mode is set.\n");
  }
  return;
}


/*********************************************************************
* @purpose  Determine if interface configuration can be applied in hardware.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    intfNum       @b{(input)}  Interface number.
* @param    isAllowed       @b{(output)}  Flag to tell if Hardware configuration allowed.

*
* @returns  none
*
* @end
*********************************************************************/
void ptpTcIntfHwConfigAllowedGet(openapiClientHandle_t *client_handle, 
                                 uint32_t  intfNum, OPEN_BOOL_t *isAllowed)
{
  open_error_t result;

  if ((result = openapiPtpTcIntfHwConfigAllowedGet(client_handle, intfNum, isAllowed)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to determine if interface configuration can be applied in hardware. (result = %d)\n", result);
  }
  else
  {
    if (*isAllowed == OPEN_TRUE)
    {
      printf("\nPTP TC Hardware configuration is allowed on interface %u.\n",intfNum);
    }
    else
    {
      printf("\nPTP TC Hardware configuration is not allowed on interface %u.\n", intfNum);
    }
  }
  return;
}


/*********************************************************************
* @purpose  Get PTP TC mode on the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    intfNum         @b{(input)}  Interface number.
* @param    adminMode       @b{(output)}  Interface PTP TC admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void ptpTcIntfAdminModeGet(openapiClientHandle_t *client_handle, 
                           uint32_t  intfNum, OPEN_CONTROL_t *adminMode)
{
  open_error_t result;

  if ((result = openapiPtpTcIntfAdminModeGet(client_handle, intfNum, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to get PTP TC mode on the specified interface. (result = %d)\n", result);
  }
  else
  {
    if (*adminMode == OPEN_ENABLE)
    {
      printf("\n PTP TC is admin enabled on interface %u.\n", intfNum);
    }
    else
    {
      printf("\n PTP TC is admin disabled on interface %u.\n", intfNum);
    }
  }
  return;
}


/*********************************************************************
* @purpose  Set PTP TC mode on the specified interface.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    intfNum         @b{(input)}  Interface number.
* @param    adminMode       @b{(input)}  Interface PTP TC admin mode.

*
* @returns  none
*
* @end
*********************************************************************/
void ptpTcIntfAdminModeSet(openapiClientHandle_t *client_handle, 
                           uint32_t  intfNum, OPEN_CONTROL_t adminMode)
{
  open_error_t result;

  if ((result = openapiPtpTcIntfAdminModeSet(client_handle, intfNum, adminMode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to set PTP TC mode on the specified interface. (result = %d)\n", result);
  }
  printf("\n PTP TC is admin mode is set on interface %u.\n", intfNum);
  return;
}

/*********************************************************************
* @purpose  Gets PTP TC operational mode on the specified interface.
*
* @param    client_handle  @b{(input)}  client handle from registration API
* @param    intfNum        @b{(input)}  Interface number.
* @param    mode           @b{(output)}  PTP TC operational mode.

*
* @returns  none
*
* @end
*********************************************************************/
void ptpTcIntfOperAdminModeGet(openapiClientHandle_t *client_handle, 
                               uint32_t  intfNum, OPEN_CONTROL_t *mode)
{
  open_error_t result;

  if ((result = openapiPtpTcIntfOperAdminModeGet(client_handle, intfNum, mode)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to gets PTP TC operational mode on the specified interface. (result = %d)\n", result);
  }
  else
  {
    if (*mode == OPEN_ENABLE)
    {
      printf("\n PTP TC is operational enabled on interface %u.\n", intfNum);
    }
    else
    {
      printf("\n PTP TC is operational disabled on interface %u.\n", intfNum);
    }
  }
  return;
}


/*********************************************************************
* @purpose  Check if the interface is valid for PTP TC.
*
* @param    client_handle   @b{(input)}  client handle from registration API
* @param    intfNum         @b{(input)}  Interface number.
* @param    isValid         @b{(output)}  Flag to tell if the interface is valid for PTP TC.

*
* @returns  none
*
* @end
*********************************************************************/
void ptpTcIsValidIntfGet(openapiClientHandle_t *client_handle, 
                         uint32_t  intfNum, OPEN_BOOL_t *isValid)
{
  open_error_t result;

  if ((result = openapiPtpTcIsValidIntfGet(client_handle, intfNum, isValid)) != OPEN_E_NONE)
  {
    printf("Bad return code trying to check if the interface is valid for PTP TC. (result = %d)\n", result);
  }
  else
  {
    if (*isValid == OPEN_TRUE)
    {
      printf("\n Interface %u is valid for PTP TC.\n", intfNum);
    }
    else
    {
      printf("\n Interface %u is not valid for PTP TC.\n",intfNum);
    }
  }
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for ptptc
*
* @returns   0: Success
* @returns  -1: Failure 
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t client_handle;
  open_error_t result;
  uint32_t testNum;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  int  show_help = 1;
  OPEN_BOOL_t isSupported;
  uint32_t intfNum;
  OPEN_CONTROL_t adminMode, mode;
  OPEN_BOOL_t isAllowed;
  OPEN_BOOL_t isValid;
  uint32_t unit;

  if (argc < 2)
  {
    printAppMenu(argv[0]);
    return -1;
  }

  testNum = atoi(argv[1]);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("ptptc example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    return -1;
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting ptptc API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&client_handle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  switch (testNum)
  {
    case 1:
      if (argc == 2)
      {
        ptpTcIsSupportedGet(&client_handle, &isSupported);
        show_help = 0;
      }
      break;
    case 2:
      if (argc == 3)
      {
	unit = atoi(argv[2]);
        ptpTcIsValidUnitGet(&client_handle, unit, &isValid);
        show_help = 0;
      }
      break;
    case 3:
      if (argc == 2)
      {
        ptpTcAdminModeGet(&client_handle, &adminMode);
        show_help = 0;
      }
      break;
    case 4:
      if (argc == 3)
      {
	adminMode = atoi(argv[2]);
        ptpTcAdminModeSet(&client_handle, adminMode);
        show_help = 0;
      }
      break;
    case 5:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        ptpTcIntfHwConfigAllowedGet(&client_handle, intfNum, &isAllowed);
        show_help = 0;
      }
      break;
    case 6:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        ptpTcIntfAdminModeGet(&client_handle, intfNum, &adminMode);
        show_help = 0;
      }
      break;
    case 7:
      if (argc == 4)
      {
	intfNum = atoi(argv[2]);
	adminMode = atoi(argv[3]);
        ptpTcIntfAdminModeSet(&client_handle, intfNum, adminMode);
        show_help = 0;
      }
      break;
    case 8:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        ptpTcIntfOperAdminModeGet(&client_handle, intfNum, &mode);
        show_help = 0;
      }
      break;
    case 9:
      if (argc == 3)
      {
	intfNum = atoi(argv[2]);
        ptpTcIsValidIntfGet(&client_handle, intfNum, &isValid);
        show_help = 0;
      }
      break;
    case 10:
      if (argc == 2)
      {
        runSanity(&client_handle);
        show_help = 0;
      }
      break;

    default:
      break;
  }

  if (show_help == 1)
  {
    printAppMenu(argv[0]);
  }

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping ptptc API example application");

  (void) openapiClientTearDown(&client_handle);
  return 0;
}
