
/*********************************************************************
*
* Copyright 2016-2020 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  snmp_example.c
*
* @purpose   Simple Network Management Protocol (SNMP) Example
*
* @component OPEN
*
* @comments
*
* @create    09/23/2013
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_snmp.h"

#define PRINTSANITYRESULTS(result, test, msg, feat) \
if (result==OPEN_E_UNAVAIL) { printf("Sanity test skipped (not available) - %s - %s\n", msg, feat); } \
else if (result==OPEN_E_EXISTS) { printf("Sanity Skipped (already exists) - %s - %s.\n", msg, feat); } \
else if ((result==OPEN_E_NONE) && (test)) { printf("Sanity Success - %s - %s.\n", msg, feat); } \
else { printf("Sanity Failure - %s - %s.\n", msg, feat); }

#define PRINTBADRESULT(result, msg, feat) \
if (result==OPEN_E_UNAVAIL) { printf("Feature not supported - %s.\n", msg); } \
else if (result==OPEN_E_NOT_FOUND) { printf("Test Skipped (not found) - %s %s.\n", msg, feat); } \
else if (result!=OPEN_E_NONE) { printf("Test Failure - %s (err %d).\n", msg, result); }

static bool fVerbose = false;

/*********************************************************************
* @purpose  Get the interface description for the particular interface type
*          
* @param    type  @b{(input)}  Interface Type
* 
* @returns The Interface Type Description
*
* @notes
*
* @end
*********************************************************************/
char* getInterfaceType(uint32_t type)
{
  switch (type)
  {
  case OPEN_INTF_TYPE_PHY:
    return "Port";
  case OPEN_INTF_TYPE_VLAN:
    return "VLAN";
  case OPEN_INTF_TYPE_LOOPBACK:
    return "Loop";
  case OPEN_INTF_TYPE_TUNNEL:
    return "Tnnl";
  }
  return "";
}

/*****************************************************************************
* @purpose  Delete the SNMP engine id on the local device. The engine id is
*           actually reset to the default id which is automatically generated
*           based on the local MAC address.
*
* @param    client_handle       @b{(input)}  client handle from registration API
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
****************************************************************************/
open_error_t testSnmpLocalEngineIdDelete(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;

  result = openapiSnmpLocalEngineIdDelete(clientHandle);

  PRINTBADRESULT(result, "openapiSnmpLocalEngineIdDelete", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, "");

  return (result);
}

/*****************************************************************************
* @purpose  Get the SNMP engine id on the local device.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    engineId            @b{(output)} local engine id for comparison
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes
* 
* @end
****************************************************************************/
open_error_t testSnmpLocalEngineIdGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[32+1];

  memset(str, 0, sizeof(str));
  desc.pstart = str;
  desc.size = sizeof(str);

  result = openapiSnmpLocalEngineIdGet(clientHandle, &desc);

  PRINTBADRESULT(result, "openapiSnmpLocalEngineIdGet", "");

  if (result == OPEN_E_NONE)
  {
    unsigned char *ptr = desc.pstart;
    uint32_t idx;

    PRINTSANITYRESULTS(result, (1==1), __func__, "");

    if (fVerbose)
    {
      printf("  Engine Id : ");
      for (idx=0; idx<desc.size; idx++)
      {
        printf("%02x", (unsigned char) *ptr++);
      }
      printf("\n");
    }
  }

  return (result);
}

/*****************************************************************************
* @purpose  Set the SNMP engine id on the local device.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    engineId            @b{(input)}  local engine id
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
****************************************************************************/
open_error_t testSnmpLocalEngineIdSet(openapiClientHandle_t *clientHandle,
                                      char *engineId)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[32+1];

  if (strcmp(engineId, "default") == 0) { strcpy(str, "1"); }
  else { strcpy(str, engineId); }

  desc.pstart = str;
  desc.size = strlen(str) + 1;

  result = openapiSnmpLocalEngineIdSet(clientHandle, &desc);

  PRINTBADRESULT(result, "openapiSnmpLocalEngineIdSet", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, engineId);

  return (result);
}

/*****************************************************************************
* @purpose  Get the system name.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysName             @b{(output)} system name
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes
* 
* @end
****************************************************************************/
open_error_t testSnmpSysNameGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  memset(str, 0, sizeof(str));
  desc.pstart = str;
  desc.size = sizeof(str);

  result = openapiSnmpSysNameGet(clientHandle, &desc);

  PRINTBADRESULT(result, "openapiSnmpSysNameGet", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, (char *)desc.pstart);

  return (result);
}

/*****************************************************************************
* @purpose  Set the SNMP system name. An administratively-assigned name for
*           this managed node. By convention, this is the node's fully-qualified
*           domain name.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysName             @b{(input)}  system name
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
****************************************************************************/
open_error_t testSnmpSysNameSet(openapiClientHandle_t *clientHandle, 
                                char *sysName)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  strcpy(str, sysName);
  desc.pstart = str;
  desc.size = strlen(str) + 1;

  result = openapiSnmpSysNameSet(clientHandle, &desc);

  PRINTBADRESULT(result, "openapiSnmpSysNameSet", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, sysName);

  return (result);
}

/*****************************************************************************
* @purpose  Get the system contact information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysContact          @b{(output)} system contact
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes
* 
* @end
****************************************************************************/
open_error_t testSnmpSysContactGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  memset(str, 0, sizeof(str));
  desc.pstart = str;
  desc.size = sizeof(str);

  result = openapiSnmpSysContactGet(clientHandle, &desc);

  PRINTBADRESULT(result, "openapiSnmpSysContactGet", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, (char *)desc.pstart);

  return (result);
}

/*****************************************************************************
* @purpose  Set the SNMP system contact. The textual identification of the
*           contact person for this managed node, together with information
*           on how to contact this person.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysContact          @b{(input)}  system contact
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
****************************************************************************/
open_error_t testSnmpSysContactSet(openapiClientHandle_t *clientHandle, 
                                   char *sysContact)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  strcpy(str, sysContact);
  desc.pstart = str;
  desc.size = strlen(str) + 1;

  result = openapiSnmpSysContactSet(clientHandle, &desc);

  PRINTBADRESULT(result, "openapiSnmpSysContactSet", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, sysContact);

  return (result);
}

/*****************************************************************************
* @purpose  Get the system location information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysLocation         @b{(output)} system location
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
*
* @notes
* 
* @end
****************************************************************************/
open_error_t testSnmpSysLocationGet(openapiClientHandle_t *clientHandle)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  memset(str, 0, sizeof(str));
  desc.pstart = str;
  desc.size = sizeof(str);

  result = openapiSnmpSysLocationGet(clientHandle, &desc);

  PRINTBADRESULT(result, "openapiSnmpSysLocationGet", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, (char *)desc.pstart);

  return (result);
}

/*****************************************************************************
* @purpose  Set the SNMP system location. The physical location of this node,
*           for example; 'telephone closet' or '3rd floor'.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    sysLocation         @b{(input)}  system location
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
****************************************************************************/
open_error_t testSnmpSysLocationSet(openapiClientHandle_t *clientHandle, 
                                    char *sysLocation)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc desc;
  char str[255+1];

  strcpy(str, sysLocation);
  desc.pstart = str;
  desc.size = strlen(str) + 1;

  result = openapiSnmpSysLocationSet(clientHandle, &desc);

  PRINTBADRESULT(result, "openapiSnmpSysLocationSet", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, sysLocation);

  return (result);
}

/*********************************************************************
* @purpose  Create a new SNMP user
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    userName            @b{(input)}  user name
* @param    groupName           @b{(input)}  group name
* @param    authProto           @b{(input)}  authentication protocol type
* @param    authPwd             @b{(input)}  HMAC-MD5-96 or HMAC-SHA-96 authentication password
* @param    authKey             @b{(input)}  pre-generated key
* @param    privProto           @b{(input)}  privilege protocol type
* @param    privPwd             @b{(input)}  CBC-DES symmetric encryption password
* @param    privKey             @b{(input)}  pre-generated key
* @param    engineId            @b{(input)}  remote engine id
* 
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testSnmpUserCreate(openapiClientHandle_t *clientHandle, 
                                   char *userName,
                                   char *groupName,
                                   OPEN_USM_USER_AUTH_PROTOCOL_t authProto,
                                   char *authPwd,
                                   char *authKey,
                                   OPEN_USM_USER_PRIV_PROTOCOL_t privProto,
                                   char *privPwd,
                                   char *privKey,
                                   char *engineId)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc userDesc;
  open_buffdesc groupDesc;
  open_buffdesc aPwdDesc;
  open_buffdesc aKeyDesc;
  open_buffdesc pPwdDesc;
  open_buffdesc pKeyDesc;
  open_buffdesc engineDesc;
  char userNameStr[30+1];
  char gStr[30+1];
  char aPwdStr[32+1];
  char aKeyStr[48+1];
  char pPwdStr[32+1];
  char pKeyStr[32+1];
  char eStr[32+1];

  strcpy(userNameStr, userName);
  userDesc.pstart = userNameStr;
  userDesc.size = strlen(userNameStr) + 1;

  strcpy(gStr, groupName);
  groupDesc.pstart = gStr;
  groupDesc.size = strlen(gStr) + 1;

  strcpy(aPwdStr, authPwd);
  aPwdDesc.pstart = aPwdStr;
  aPwdDesc.size = strlen(aPwdStr) + 1;

  strcpy(aKeyStr, authKey);
  aKeyDesc.pstart = aKeyStr;
  aKeyDesc.size = strlen(aKeyStr) + 1;

  strcpy(pPwdStr, privPwd);
  pPwdDesc.pstart = pPwdStr;
  pPwdDesc.size = strlen(pPwdStr) + 1;

  strcpy(pKeyStr, privKey);
  pKeyDesc.pstart = pKeyStr;
  pKeyDesc.size = strlen(pKeyStr) + 1;

  strcpy(eStr, engineId);
  engineDesc.pstart = eStr;
  engineDesc.size = strlen(eStr) + 1;

  result = openapiSnmpUserCreate(clientHandle,
                                 &userDesc, &groupDesc,
                                 authProto, &aPwdDesc, &aKeyDesc,
                                 privProto, &pPwdDesc, &pKeyDesc,
                                 &engineDesc);

  PRINTSANITYRESULTS(result, 1==1, __func__, userName);

  return (result);
}

/*********************************************************************
* @purpose  Delete a new SNMP user
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    userName            @b{(input)}  user name
* @param    engineId            @b{(input)}  remote engine id
* 
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testSnmpUserDelete(openapiClientHandle_t *clientHandle, 
                                char *userName,
                                char *engineId)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc userDesc;
  open_buffdesc engineDesc;
  char userNameStr[30+1];
  char eStr[32+1];

  strcpy(userNameStr, userName);
  userDesc.pstart = userNameStr;
  userDesc.size = strlen(userNameStr) + 1;

  strcpy(eStr, engineId);
  engineDesc.pstart = eStr;
  engineDesc.size = strlen(eStr) + 1;

  result = openapiSnmpUserDelete(clientHandle,
                                 &userDesc, 
                                 &engineDesc);

  PRINTBADRESULT(result, "openapiSnmpUserDelete", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, userName);

  return (result);
}

/*********************************************************************
* @purpose  Demonstrate how to loop through and retrieve the existing
*           SNMP users. This function simply stops at the first userName
*           match and displays its contents.
*
* @param    clientHandle        @b{(input)}  client handle from registration API
* @param    userName            @b{(input)}  user name
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
open_error_t testSnmpUserGet(openapiClientHandle_t *clientHandle, 
                             char *userName)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_USM_USER_AUTH_PROTOCOL_t authProto;
  OPEN_USM_USER_PRIV_PROTOCOL_t privProto;
  open_buffdesc userDesc;
  open_buffdesc engineDesc;
  open_buffdesc groupDesc;
  open_buffdesc nextUserDesc;
  open_buffdesc nextEngineDesc;
  char userNameStr[30+1];
  char groupNameStr[30+1];
  char engineStr[32];
  char nextUserNameStr[30+1];
  char nextEngineStr[32];
  bool fFound = false;

  memset(userNameStr, 0, sizeof(userNameStr));
  userDesc.pstart = userNameStr;
  userDesc.size = sizeof(userNameStr);

  memset(groupNameStr, 0, sizeof(groupNameStr));
  groupDesc.pstart = groupNameStr;
  groupDesc.size = sizeof(groupNameStr);

  memset(engineStr, 0, sizeof(engineStr));
  engineDesc.pstart = engineStr;
  engineDesc.size = sizeof(engineStr);

  memset(nextUserNameStr, 0, sizeof(nextUserNameStr));
  nextUserDesc.pstart = nextUserNameStr;
  nextUserDesc.size = sizeof(nextUserNameStr);

  memset(nextEngineStr, 0, sizeof(nextEngineStr));
  nextEngineDesc.pstart = nextEngineStr;
  nextEngineDesc.size = sizeof(nextEngineStr);

  while (openapiSnmpUserGetNext(clientHandle,
                                &userDesc, 
                                &engineDesc, 
                                &groupDesc,
                                &authProto, 
                                &privProto,
                                &nextUserDesc, 
                                &nextEngineDesc) == OPEN_E_NONE)
  {
    if (strcmp(userName, nextUserDesc.pstart)==0)
    {
      fFound = true;
      break;
    }

    /* Setup for next iteration */
    strncpy(userDesc.pstart, nextUserDesc.pstart, (userDesc.size - 1));
    userDesc.size = strlen(userDesc.pstart)+1;
    memcpy(engineDesc.pstart, nextEngineDesc.pstart, nextEngineDesc.size);
    engineDesc.size = nextEngineDesc.size;

    memset(groupNameStr, 0, sizeof(groupNameStr));
    groupDesc.size = sizeof(groupNameStr);

    /* Necessary to reset size for next user record */
    nextUserDesc.size = sizeof(nextUserNameStr);
    nextEngineDesc.size = sizeof(nextEngineStr);
  }

  if (fFound)
  {
    unsigned char *ptr = nextEngineDesc.pstart;
    uint32_t idx;

    PRINTSANITYRESULTS(OPEN_E_NONE, 1==1, __func__, userName);

    if (fVerbose)
    {
      printf("  User name      : %s\n"
             "  Group name     : %s\n"
             "  Authentication : %d\n"
             "  Privilege      : %d\n"
             "  Engine Id      : ",
             (char *) nextUserDesc.pstart,
             (char *) groupDesc.pstart,
             authProto,
             privProto);
      for (idx=0; idx<nextEngineDesc.size; idx++)
      {
        printf("%02x", (unsigned char) *ptr++);
      }
      printf("\n");
    }

  }

  return (result);
}

/*********************************************************************
* @purpose  Create a new SNMP group.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    groupName           @b{(input)}  group name
* @param    securityModel       @b{(input)}  version number
* @param    securityLevel       @b{(input)}  security level (valid only for securityModel @ref OPEN_SNMP_SECURITY_MODEL_USM)
* @param    contextPrefix       @b{(input)}  optional prefix name to associate the group with
* @param    readView            @b{(input)}  optional, enables user to view the contents of the agent
* @param    writeView           @b{(input)}  optional, enables user to enter data and configure the contents of the agent
* @param    notifyView          @b{(input)}  optional, enables user to specify an inform or a trap
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_EXITS        if SNMP group already exists
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
open_error_t testSnmpGroupCreate(openapiClientHandle_t *clientHandle, 
                                 char *groupName,
                                 OPEN_SNMP_SECURITY_MODEL_t securityModel,
                                 OPEN_SNMP_SECURITY_LEVEL_t securityLevel,
                                 char *contextPrefix,
                                 char *readView,
                                 char *writeView,
                                 char *notifyView)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc groupDesc;
  open_buffdesc contextDesc;
  open_buffdesc readDesc;
  open_buffdesc writeDesc;
  open_buffdesc notifyDesc;
  char groupStr[30+1];
  char contextStr[30+1];
  char readStr[30+1];
  char writeStr[30+1];
  char notifyStr[30+1];

  strcpy(groupStr, groupName);
  groupDesc.pstart = groupStr;
  groupDesc.size = strlen(groupStr) + 1;

  strcpy(contextStr, contextPrefix);
  contextDesc.pstart = contextStr;
  contextDesc.size = strlen(contextStr) + 1;

  strcpy(readStr, readView);
  readDesc.pstart = readStr;
  readDesc.size = strlen(readStr) + 1;

  strcpy(writeStr, writeView);
  writeDesc.pstart = writeStr;
  writeDesc.size = strlen(writeStr) + 1;

  strcpy(notifyStr, notifyView);
  notifyDesc.pstart = notifyStr;
  notifyDesc.size = strlen(notifyStr) + 1;

  result = openapiSnmpGroupCreate(clientHandle,
                                  &groupDesc, securityModel, securityLevel,
                                  &contextDesc, &readDesc, &writeDesc, &notifyDesc);

  PRINTSANITYRESULTS(result, 1==1, __func__, groupName);

  return (result);
}

/*********************************************************************
* @purpose  Delete an existing SNMP group.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    groupName           @b{(input)}  group name
* @param    securityModel       @b{(input)}  version number
* @param    securityLevel       @b{(input)}  security level (valid only for securityModel @ref OPEN_SNMP_SECURITY_MODEL_USM)
* @param    contextPrefix       @b{(input)}  optional prefix name to associate the group with
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch.
* 
* @end
*********************************************************************/
open_error_t testSnmpGroupDelete(openapiClientHandle_t *clientHandle, 
                                 char *groupName,
                                 OPEN_SNMP_SECURITY_MODEL_t securityModel,
                                 OPEN_SNMP_SECURITY_LEVEL_t securityLevel,
                                 char *contextPrefix)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc groupDesc;
  open_buffdesc contextDesc;
  char groupStr[30+1];
  char contextStr[30+1];

  strcpy(groupStr, groupName);
  groupDesc.pstart = groupStr;
  groupDesc.size = strlen(groupStr) + 1;

  strcpy(contextStr, contextPrefix);
  contextDesc.pstart = contextStr;
  contextDesc.size = strlen(contextStr) + 1;

  result = openapiSnmpGroupDelete(clientHandle,
                                  &groupDesc,
                                  securityModel,
                                  securityLevel,
                                  &contextDesc);

  PRINTSANITYRESULTS(result, 1==1, __func__, groupName);

  return (result);
}

/********************************************************************* 
* @purpose  Interate through the existing SNMP groups until a match for
*           group is found. If found, display the associated group parameters.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    host                @b{(input)}   IPv4 or IPv6 address
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
open_error_t testSnmpGroupGet(openapiClientHandle_t *clientHandle, 
                              char *group,     
                              OPEN_SNMP_SECURITY_MODEL_t model,
                              OPEN_SNMP_SECURITY_LEVEL_t level,
                              char *context)
{
  OPEN_SNMP_SECURITY_MODEL_t securityModel;
  OPEN_SNMP_SECURITY_LEVEL_t securityLevel;
  open_error_t result = OPEN_E_NONE;
  open_buffdesc groupDesc;
  open_buffdesc contextDesc;
  open_buffdesc readDesc;
  open_buffdesc writeDesc;
  open_buffdesc notifyDesc;
  open_buffdesc nextGroupDesc;
  open_buffdesc nextContextDesc;
  char groupStr[30+1];
  char contextStr[30+1];
  char readStr[30+1];
  char writeStr[30+1];
  char notifyStr[30+1];
  char nextGroupStr[30+1];
  char nextContextStr[30+1];
  bool fFound = false;

  memset(groupStr, 0, sizeof(groupStr));
  groupDesc.pstart = groupStr;
  groupDesc.size = sizeof(groupStr);

  memset(nextGroupStr, 0, sizeof(nextGroupStr));
  nextGroupDesc.pstart = nextGroupStr;
  nextGroupDesc.size = sizeof(nextGroupStr);

  memset(contextStr, 0, sizeof(contextStr));
  contextDesc.pstart = contextStr;
  contextDesc.size = sizeof(contextStr);

  memset(nextContextStr, 0, sizeof(nextContextStr));
  nextContextDesc.pstart = nextContextStr;
  nextContextDesc.size = sizeof(nextContextStr);

  memset(readStr, 0, sizeof(readStr));
  readDesc.pstart = readStr;
  readDesc.size = sizeof(readStr);

  memset(writeStr, 0, sizeof(writeStr));
  writeDesc.pstart = writeStr;
  writeDesc.size = sizeof(writeStr);

  memset(notifyStr, 0, sizeof(notifyStr));
  notifyDesc.pstart = notifyStr;
  notifyDesc.size = sizeof(notifyStr);

  while (openapiSnmpGroupGetNext(clientHandle,
                                 &groupDesc,
                                 &securityModel,
                                 &securityLevel,
                                 &contextDesc,
                                 &readDesc,
                                 &writeDesc,
                                 &notifyDesc,
                                 &nextGroupDesc,
                                 &nextContextDesc) == OPEN_E_NONE)
  {

    if ((strcmp(group, nextGroupDesc.pstart)==0) &&
        (strcmp(context, nextContextDesc.pstart)==0) &&
        (model == securityModel) &&
        (level == securityLevel))
    {
      fFound = true;
      break;
    }

    /* Setup for next iteration */
    memset(readStr,    0, sizeof(readStr));
    memset(writeStr,   0, sizeof(writeStr));
    memset(notifyStr,  0, sizeof(notifyStr));
    strncpy(groupDesc.pstart, nextGroupDesc.pstart, (groupDesc.size - 1));
    groupDesc.size = sizeof(nextGroupStr);
    strncpy(contextDesc.pstart, nextContextDesc.pstart, (contextDesc.size - 1));
    contextDesc.size = sizeof(nextContextStr);

    /* Necessary to reset size for next group record */
    nextGroupDesc.size = sizeof(nextGroupStr);
    nextContextDesc.size = sizeof(nextContextStr);
  }


  if (fFound)
  {
    PRINTSANITYRESULTS(OPEN_E_NONE, (1==1), __func__, group);

    if (fVerbose)
    {
      printf("  Group name     : %s\n"
             "  Security model : %d\n"
             "  Security level : %d\n"
             "  Context name   : %s\n"
             "  Read view      : %s\n"
             "  Write view     : %s\n"
             "  Notify view    : %s\n",
             (char *) nextGroupDesc.pstart,
             securityModel,
             securityLevel,
             (char *) nextContextDesc.pstart,
             (char *) readDesc.pstart,
             (char *) writeDesc.pstart,
             (char *) notifyDesc.pstart);
    }
  }

  return (result);
}

/********************************************************************* 
* @purpose  Create a new, or edit an existing SNMP view.
*           If the view name already exists, the associated MIBS
*           (vacmViewTreeFamilySubtree and vacmViewTreeFamilyMask) are updated
*           according to the excluded flag.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    viewName            @b{(input)}  view name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the view
* @param    viewType            @b{(input)}  included or excluded from the MIB view
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testSnmpViewCreate(openapiClientHandle_t *clientHandle, 
                                char *view,
                                char *oid,
                                OPEN_SNMP_VIEW_TYPE_t type)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc viewDesc;
  open_buffdesc oidDesc;
  char viewStr[30+1];
  char oidStr[128+1];

  strcpy(viewStr, view);
  viewDesc.pstart = viewStr;
  viewDesc.size = strlen(viewStr)+1;

  strcpy(oidStr, oid);
  oidDesc.pstart = oidStr;
  oidDesc.size = strlen(oidStr)+1;

  result = openapiSnmpViewCreate(clientHandle, &viewDesc, &oidDesc, type);

  PRINTSANITYRESULTS(result, 1==1, __func__, view);

  return (result);
}

/********************************************************************* 
* @purpose  Interate through the existing SNMP views until a match for
*           view and oid is found. If found, display the associated view parameters.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    viewName            @b{(input)}  view name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the view
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
open_error_t testSnmpViewGet(openapiClientHandle_t *clientHandle, 
                             char *view,
                             char *oid)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_SNMP_VIEW_TYPE_t type;
  open_buffdesc viewDesc;
  open_buffdesc oidDesc;
  open_buffdesc oidValDesc;
  open_buffdesc oidStringDesc;
  open_buffdesc nextViewDesc;
  open_buffdesc nextOidDesc;
  char viewStr[30+1];
  char oidStr[128+1];
  char oidValStr[128+1];
  char oidStringStr[128+1];
  char nextViewStr[30+1];
  char nextOidStr[128+1];
  bool fFound = false;

  memset(viewStr, 0, sizeof(viewStr));
  viewDesc.pstart = viewStr;
  viewDesc.size = sizeof(viewStr);

  memset(oidStr, 0, sizeof(oidStr));
  oidDesc.pstart = oidStr;
  oidDesc.size = sizeof(oidStr);

  memset(nextViewStr, 0, sizeof(nextViewStr));
  nextViewDesc.pstart = nextViewStr;
  nextViewDesc.size = sizeof(nextViewStr);

  memset(nextOidStr, 0, sizeof(nextOidStr));
  nextOidDesc.pstart = nextOidStr;
  nextOidDesc.size = sizeof(nextOidStr);

  memset(oidValStr, 0, sizeof(oidValStr));
  oidValDesc.pstart = oidValStr;
  oidValDesc.size = sizeof(oidValStr);

  memset(oidStringStr, 0, sizeof(oidStringStr));
  oidStringDesc.pstart = oidStringStr;
  oidStringDesc.size = sizeof(oidStringStr);

  strcpy(viewStr, view);
  strcpy(oidStr, oid);

  openapiSnmpTreeFamilyOidStringGet(clientHandle, &viewDesc, &oidDesc, OPEN_SNMP_VIEW_TREE, &oidStringDesc);
  openapiSnmpTreeFamilyOidValGet(clientHandle, &viewDesc, &oidDesc, OPEN_SNMP_VIEW_TREE, &oidValDesc);

  memset(viewStr, 0, sizeof(viewStr));
  memset(oidStr, 0, sizeof(oidStr));

  while (openapiSnmpViewGetNext(clientHandle,
                                &viewDesc,
                                &oidDesc,
                                &type,
                                &nextViewDesc,
                                &nextOidDesc) == OPEN_E_NONE)
  {
    if ((strcmp(view, nextViewDesc.pstart)==0) &&
        (strcmp(oidValDesc.pstart, nextOidDesc.pstart)==0))
    {
      fFound = true;
      break;
    }

    /* Setup for next iteration */
    strncpy(viewDesc.pstart, nextViewDesc.pstart, (viewDesc.size - 1));
    viewDesc.size = sizeof(nextViewStr);
    strncpy(oidDesc.pstart, nextOidDesc.pstart, (oidDesc.size - 1));
    oidDesc.size = sizeof(nextOidStr);

    /* Necessary to reset size for next group record */
    nextViewDesc.size = sizeof(nextViewStr);
    nextOidDesc.size = sizeof(nextOidStr);
  }

  if (fFound)
  {
    PRINTSANITYRESULTS(OPEN_E_NONE, (1==1), __func__, view);

    if (fVerbose)
    {
      printf("  View name  : %s\n"
             "  OID Tree   : %s\n"
             "  OID String : %s\n"
             "  View Type  : %d\n",
             (char *) nextViewDesc.pstart,
             (char *) nextOidDesc.pstart,
             (char *) oidStringDesc.pstart,
             type);
    }
  }

  return (result);
}

/********************************************************************* 
* @purpose  Delete an existing SNMP view.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    viewName            @b{(input)}  view name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the view
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testSnmpViewDelete(openapiClientHandle_t *clientHandle, 
                                char *view,
                                char *oid)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc viewDesc;
  open_buffdesc oidDesc;
  char viewStr[30+1];
  char oidStr[128+1];

  strcpy(viewStr, view);
  viewDesc.pstart = viewStr;
  viewDesc.size = strlen(viewStr)+1;

  strcpy(oidStr, oid);
  oidDesc.pstart = oidStr;
  oidDesc.size = strlen(oidStr)+1;

  result = openapiSnmpViewDelete(clientHandle, &viewDesc, &oidDesc);

  PRINTSANITYRESULTS(result, 1==1, __func__, view);

  return (result);
}

/********************************************************************* 
* @purpose  Create a new, or edit an existing SNMP filter.
*           If the filter name already exists, the associated MIBS
*           (vacmFilterTreeFamilySubtree and vacmFilterTreeFamilyMask) are updated
*           according to the excluded flag.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    filterName          @b{(input)}  filter name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the view
* @param    viewType            @b{(input)}  included or excluded from the MIB view
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testSnmpFilterCreate(openapiClientHandle_t *clientHandle, 
                                  char *filter,
                                  char *oid,
                                  OPEN_SNMP_VIEW_TYPE_t type)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc filterDesc;
  open_buffdesc oidDesc;
  char filterStr[30+1];
  char oidStr[128+1];

  strcpy(filterStr, filter);
  filterDesc.pstart = filterStr;
  filterDesc.size = strlen(filterStr)+1;

  strcpy(oidStr, oid);
  oidDesc.pstart = oidStr;
  oidDesc.size = strlen(oidStr)+1;

  result = openapiSnmpFilterCreate(clientHandle, &filterDesc, &oidDesc, type);

  PRINTSANITYRESULTS(result, 1==1, __func__, filter);

  return (result);
}

/********************************************************************* 
* @purpose  Delete an existing SNMP filter.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    filterName          @b{(input)}  filter name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the filter
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testSnmpFilterDelete(openapiClientHandle_t *clientHandle, 
                                  char *filter,
                                  char *oid)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc filterDesc;
  open_buffdesc oidDesc;
  char filterStr[30+1];
  char oidStr[128+1];

  strcpy(filterStr, filter);
  filterDesc.pstart = filterStr;
  filterDesc.size = strlen(filterStr)+1;

  strcpy(oidStr, oid);
  oidDesc.pstart = oidStr;
  oidDesc.size = strlen(oidStr)+1;

  result = openapiSnmpFilterDelete(clientHandle, &filterDesc, &oidDesc);

  PRINTSANITYRESULTS(result, 1==1, __func__, filter);

  return (result);
}

/********************************************************************* 
* @purpose  Interate through the existing SNMP filters until a match for
*           filter and oid is found. If found, display the associated filter parameters.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    filterName          @b{(input)}  filter name
* @param    oidTree             @b{(input)}  ASN.1 subtree OID to be included/excluded from the filter
* 
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
open_error_t testSnmpFilterGet(openapiClientHandle_t *clientHandle, 
                               char *filter,
                               char *oid)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_SNMP_NOTIFY_FILTER_TYPE_t type;
  open_buffdesc filterDesc;
  open_buffdesc oidDesc;
  open_buffdesc oidValDesc;
  open_buffdesc oidStringDesc;
  open_buffdesc nextFilterDesc;
  open_buffdesc nextOidDesc;
  char filterStr[30+1];
  char oidStr[128+1];
  char oidValStr[128+1];
  char oidStringStr[128+1];
  char nextFilterStr[30+1];
  char nextOidStr[128+1];
  bool fFound = false;

  memset(filterStr, 0, sizeof(filterStr));
  filterDesc.pstart = filterStr;
  filterDesc.size = sizeof(filterStr);

  memset(oidStr, 0, sizeof(oidStr));
  oidDesc.pstart = oidStr;
  oidDesc.size = sizeof(oidStr);

  memset(nextFilterStr, 0, sizeof(nextFilterStr));
  nextFilterDesc.pstart = nextFilterStr;
  nextFilterDesc.size = sizeof(nextFilterStr);

  memset(nextOidStr, 0, sizeof(nextOidStr));
  nextOidDesc.pstart = nextOidStr;
  nextOidDesc.size = sizeof(nextOidStr);

  memset(oidValStr, 0, sizeof(oidValStr));
  oidValDesc.pstart = oidValStr;
  oidValDesc.size = sizeof(oidValStr);

  memset(oidStringStr, 0, sizeof(oidStringStr));
  oidStringDesc.pstart = oidStringStr;
  oidStringDesc.size = sizeof(oidStringStr);

  strcpy(filterStr, filter);
  strcpy(oidStr, oid);

  openapiSnmpTreeFamilyOidStringGet(clientHandle, &filterDesc, &oidDesc, OPEN_SNMP_FILTER_TREE, &oidStringDesc);
  openapiSnmpTreeFamilyOidValGet(clientHandle, &filterDesc, &oidDesc, OPEN_SNMP_FILTER_TREE, &oidValDesc);

  memset(filterStr, 0, sizeof(filterStr));
  memset(oidStr, 0, sizeof(oidStr));

  while (openapiSnmpFilterGetNext(clientHandle,
                                  &filterDesc,
                                  &oidDesc,
                                  &type,
                                  &nextFilterDesc,
                                  &nextOidDesc) == OPEN_E_NONE)
  {
    if ((strcmp(filter, nextFilterDesc.pstart)==0) &&
        (strcmp(oidValDesc.pstart, nextOidDesc.pstart)==0))
    {
      fFound = true;
      break;
    }

    /* Setup for next iteration */
    strncpy(filterDesc.pstart, nextFilterDesc.pstart, (filterDesc.size - 1));
    filterDesc.size = strlen(filterDesc.pstart)+1;
    strncpy(oidDesc.pstart, nextOidDesc.pstart, (oidDesc.size - 1));
    oidDesc.size = strlen(oidDesc.pstart)+1;

    /* Necessary to reset size for next group record */
    nextFilterDesc.size = sizeof(nextFilterStr);
    nextOidDesc.size = sizeof(nextOidStr);
  }

  if (fFound)
  {
    PRINTSANITYRESULTS(OPEN_E_NONE, (1==1), __func__, filter);

    if (fVerbose)
    {
      printf("  Filter name  : %s\n"
             "  OID Tree     : %s\n"
             "  OID String   : %s\n"
             "  View Type    : %d\n",
             (char *) nextFilterDesc.pstart,
             (char *) nextOidDesc.pstart,
             (char *) oidStringDesc.pstart,
             type);
    }
  }

  return (result);
}

/********************************************************************* 
* @purpose  Create an SNMPv1/2 or v3 host. This includes the target address 
*           object and its various parameters, including an optional notification 
*           filter.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    host                @b{(input)}  IPv4 or IPv6 address
* @param    serverPort          @b{(input)}  optional host receiver port number.
* @param    securityName        @b{(input)}  context name (community or user name)
* @param    securityModel       @b{(input)}  version number
* @param    securityLevel       @b{(input)}  security level 
* @param    notifyType          @b{(input)}  notification type
* @param    timeout             @b{(input)}  optional timeout in milliseconds for @ref OPEN_SNMP_NOTIFY_TYPE_INFORM trap type
* @param    retries             @b{(input)}  optional retries for @ref OPEN_SNMP_NOTIFY_TYPE_INFORM trap type
* @param    filter              @b{(input)}  optional notification filter name
* 
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch 
* 
* @end
*********************************************************************/
open_error_t testSnmpHostCreate(openapiClientHandle_t *clientHandle, 
                                char *host,
                                uint32_t serverPort,
                                char *securityName,
                                OPEN_SNMP_SECURITY_MODEL_t securityModel,
                                OPEN_SNMP_SECURITY_LEVEL_t securityLevel,
                                OPEN_SNMP_NOTIFY_TYPE_t notifyType,
                                uint32_t timeout,
                                uint32_t retries,
                                char *filter)

{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc hostDesc;
  open_buffdesc securityDesc;
  open_buffdesc filterDesc;
  char hostStr[158+1];
  char securityStr[30+1];
  char filterStr[30+1];

  strcpy(hostStr, host);
  hostDesc.pstart = hostStr;
  hostDesc.size = strlen(hostStr)+1;

  strcpy(securityStr, securityName);
  securityDesc.pstart = securityStr;
  securityDesc.size = strlen(securityStr)+1;

  strcpy(filterStr, filter);
  filterDesc.pstart = filterStr;
  filterDesc.size = strlen(filterStr)+1;

  result = openapiSnmpHostCreate(clientHandle,
                                 &hostDesc, serverPort,
                                 &securityDesc, securityModel,
                                 securityLevel, notifyType,
                                 timeout, retries, 
                                 &filterDesc);

  PRINTSANITYRESULTS(result, 1==1, __func__, hostStr);

  return (result);
}

/********************************************************************* 
* @purpose  Delete an existing SNMPv1/2 or v3 host entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    host                @b{(input)}  IPv4 or IPv6 address
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testSnmpHostDelete(openapiClientHandle_t *clientHandle, 
                                char *host, OPEN_SNMP_NOTIFY_TYPE_t notifyType)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc hostDesc;
  char hostStr[158+1];

  strcpy(hostStr, host);
  hostDesc.pstart = hostStr;
  hostDesc.size = strlen(hostStr)+1;

  result = openapiSnmpHostDelete(clientHandle, &hostDesc, notifyType);

  PRINTSANITYRESULTS(result, 1==1, __func__, host);

  return (result);
}

/********************************************************************* 
* @purpose  Create, or update an existing community, then retrieve it
*           for verification.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    community           @b{(input)}  community name
* @param    accessType          @b{(input)}  optional community access
* @param    viewName            @b{(input)}  optional view name
* @param    ipStr               @b{(input)}  optional IPv4 address, for allowed access
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testSnmpCommunity(openapiClientHandle_t *clientHandle, 
                               char *community,
                               OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t accessType,
                               char *viewName,
                               char *ipStr)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc communityDesc;
  open_buffdesc viewDesc;
  open_inet_addr_t ipAddr;
  char communityStr[20+1];
  char viewStr[30+1];

  strcpy(communityStr, community);
  communityDesc.pstart = communityStr;
  communityDesc.size = strlen(communityStr)+1;

  strcpy(viewStr, viewName);
  viewDesc.pstart = viewStr;
  viewDesc.size = strlen(viewStr)+1;

  memset(&ipAddr, 0, sizeof(ipAddr));
  inet_pton(AF_INET, ipStr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.family = OPEN_AF_INET;

  result = openapiSnmpCommunityCreate(clientHandle, &communityDesc, accessType, &viewDesc, ipAddr);

  PRINTSANITYRESULTS(result, 1==1, __func__, communityStr);

  return (result);
}

/********************************************************************* 
* @purpose  Delete an existing community entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    community           @b{(input)}  community name
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testSnmpCommunityDelete(openapiClientHandle_t *clientHandle, 
                                     char *community)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc communityDesc;
  char communityStr[20+1];

  strcpy(communityStr, community);
  communityDesc.pstart = communityStr;
  communityDesc.size = strlen(communityStr)+1;

  result = openapiSnmpCommunityDelete(clientHandle, &communityDesc);

  PRINTSANITYRESULTS(result, 1==1, __func__, community);

  return (result);
}

/********************************************************************* 
* @purpose  Interate through the existing SNMP hosts until a match for
*           host is found. If found, display the associated host parameters.
*
* @param    client_handle       @b{(input)} client handle from registration API
* @param    community           @b{(input)} community name
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
open_error_t testSnmpCommunityGet(openapiClientHandle_t *clientHandle, 
                                  char *community)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t accessType;
  open_inet_addr_t addr;
  char ipAddrStr[32];
  open_buffdesc communityDesc;
  open_buffdesc viewDesc;
  open_buffdesc nextCommunityDesc;
  char communityStr[20+1];
  char viewStr[30+1];
  char nextCommunityStr[20+1];
  bool fFound = false;

  memset(communityStr, 0, sizeof(communityStr));
  communityDesc.pstart = communityStr;
  communityDesc.size = sizeof(communityStr);

  memset(viewStr, 0, sizeof(viewStr));
  viewDesc.pstart = viewStr;
  viewDesc.size = sizeof(viewStr);

  memset(nextCommunityStr, 0, sizeof(nextCommunityStr));
  nextCommunityDesc.pstart = nextCommunityStr;
  nextCommunityDesc.size = sizeof(nextCommunityStr);

  while (openapiSnmpCommunityGetNext(clientHandle,
                                     &communityDesc,
                                     &accessType,
                                     &viewDesc,
                                     &addr,
                                     &nextCommunityDesc) == OPEN_E_NONE)
  {
    if (strcmp(community, nextCommunityDesc.pstart)==0)
    {
      fFound = true;
      break;
    }

    /* Setup for next iteration */
    memset(&ipAddrStr, 0, sizeof(ipAddrStr));

    memset(viewStr, 0, sizeof(viewStr));
    viewDesc.size = sizeof(viewStr);

    strncpy(communityDesc.pstart, nextCommunityDesc.pstart, (communityDesc.size - 1));
    communityDesc.size = sizeof(nextCommunityStr);

    /* Necessary to reset size for next host record */
    nextCommunityDesc.size = sizeof(nextCommunityStr);
    memset(nextCommunityStr, 0, sizeof(nextCommunityStr));
  }

  if (fFound)
  {
    if (addr.family == OPEN_AF_INET)
    {
      memset(&ipAddrStr, 0, sizeof(ipAddrStr));
      if (inet_ntop(AF_INET, (void*)&(addr.addr.ipv4), ipAddrStr, sizeof(ipAddrStr)) == NULL)
      {
        result = OPEN_E_PARAM;
      }
    }
    else
    {
      strcpy(ipAddrStr, "All");
    }
    PRINTBADRESULT(result, "inet_ntop", "");
      
    PRINTSANITYRESULTS(OPEN_E_NONE, (1==1), __func__, community);

    if (fVerbose)
    {
      printf("  Community   : %s\n"
             "  Access Type : %d\n"
             "  View Name   : %s\n"
             "  IP Address  : %s\n",
             (char *) nextCommunityDesc.pstart,
             accessType,
             (char *) viewDesc.pstart,
             ipAddrStr);
    }

    result = OPEN_E_NONE;
  }


  return (result);
}

/********************************************************************* 
* @purpose  Create, or update an existing community, then retrieve it
*           for verification.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    community           @b{(input)}  community name
* @param    group               @b{(input)}  group name
* @param    ipStr               @b{(input)}  optional IPv4 address, for allowed access
*
* @returns  OPEN_E_NONE         Create is successful
* @returns  OPEN_E_FAIL         Create failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testSnmpCommunityGroup(openapiClientHandle_t *clientHandle, 
                                    char *community,
                                    char *group,
                                    char *ipStr)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc communityDesc;
  open_buffdesc groupDesc;
  open_inet_addr_t ipAddr;
  char communityStr[20+1];
  char groupStr[30+1];

  strcpy(communityStr, community);
  communityDesc.pstart = communityStr;
  communityDesc.size = strlen(communityStr)+1;

  strcpy(groupStr, group);
  groupDesc.pstart = groupStr;
  groupDesc.size = strlen(groupStr) + 1;

  memset(&ipAddr, 0, sizeof(ipAddr));
  inet_pton(AF_INET, ipStr, (void*)&(ipAddr.addr.ipv4));
  ipAddr.family = OPEN_AF_INET;

  result = openapiSnmpCommunityGroupCreate(clientHandle, &communityDesc, &groupDesc, ipAddr);
  PRINTBADRESULT(result, "openapiSnmpCommunityGroupCreate", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, communityStr);

  return (result);
}

/********************************************************************* 
* @purpose  Interate through the existing SNMP hosts until a match for
*           host is found. If found, display the associated host parameters.
*
* @param    client_handle       @b{(input)}   client handle from registration API
* @param    host                @b{(input)}   IPv4 or IPv6 address
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
* 
* @notes
* 
* @end
*********************************************************************/
open_error_t testSnmpHostGet(openapiClientHandle_t *clientHandle, 
                             char *host)
{
  OPEN_SNMP_SECURITY_MODEL_t securityModel;
  OPEN_SNMP_SECURITY_LEVEL_t securityLevel;
  OPEN_SNMP_NOTIFY_TYPE_t notifyType;
  open_error_t result = OPEN_E_NONE;
  open_buffdesc hostDesc;
  open_buffdesc securityDesc;
  open_buffdesc filterDesc;
  open_buffdesc nextHostDesc;
  char hostStr[30+1];
  char securityStr[30+1];
  char filterStr[30+1];
  char nextHostStr[30+1];
  uint32_t serverPort = 0;
  uint32_t timeout = 0;
  uint32_t retries = 0;
  bool fFound = false;

  memset(hostStr, 0, sizeof(hostStr));
  hostDesc.pstart = hostStr;
  hostDesc.size = sizeof(hostStr);

  memset(securityStr, 0, sizeof(securityStr));
  securityDesc.pstart = securityStr;
  securityDesc.size = sizeof(securityStr);

  memset(filterStr, 0, sizeof(filterStr));
  filterDesc.pstart = filterStr;
  filterDesc.size = sizeof(filterStr);

  memset(nextHostStr, 0, sizeof(nextHostStr));
  nextHostDesc.pstart = nextHostStr;
  nextHostDesc.size = sizeof(nextHostStr);

  while (openapiSnmpHostGetNext(clientHandle,
                                &hostDesc,
                                &serverPort,
                                &securityDesc,
                                &securityModel,
                                &securityLevel,
                                &notifyType,
                                &timeout,
                                &retries,
                                &filterDesc,
                                &nextHostDesc) == OPEN_E_NONE)
  {
    if (strcmp(host, nextHostDesc.pstart)==0)
    {
      fFound = true;
      break;
    }

    /* Setup for next iteration */
    memset(securityStr, 0, sizeof(securityStr));
    strncpy(hostDesc.pstart, nextHostDesc.pstart, (hostDesc.size - 1));
    hostDesc.size = strlen(hostDesc.pstart)+1;

    /* Necessary to reset size for next host record */
    nextHostDesc.size = sizeof(nextHostStr);
  }

  if (fFound)
  {
    PRINTSANITYRESULTS(OPEN_E_NONE, (1==1), __func__, host);

    if (fVerbose)
    {
      printf("  Address:port   : %s:%d\n"
             "  Security name  : %s\n"
             "  Security model : %d\n"
             "  Security level : %d\n"
             "  Notify type    : %d\n"
             "  Timeout        : %d\n"
             "  Retries        : %d\n"
             "  Filter         : %s\n",
             (char *) nextHostDesc.pstart, serverPort,
             (char *) securityDesc.pstart,
             securityModel,
             securityLevel,
             notifyType,
             timeout,
             retries,
             (char *) filterDesc.pstart);
    }
  }

  return (result);
}

/*****************************************************************************
* @purpose  Enable or disable the traps identified via the trapFlags selection.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flagBits            @b{(input)}  selected traps, a bitmask of @ref OPEN_SNMP_TRAP_FLAGS_t
* @param    mode                @b{(input)}  Set on or off
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        If the set request was invalid
* @returns  OPEN_E_UNAVAIL      If one or more trap sets failed due to the feature availability.
* 
* @notes
* 
* @end
*****************************************************************************/
open_error_t testSnmpTrapFlagSet(openapiClientHandle_t *clientHandle,
                                 uint32_t flagBits,
                                 bool mode)
{
  open_error_t result = OPEN_E_NONE;
  char flagBitsStr[30];

  result = openapiSnmpTrapFlagSet(clientHandle, flagBits, mode);

  sprintf(flagBitsStr,"flag bits %d", flagBits);
  PRINTSANITYRESULTS(result, 1==1, __func__, flagBitsStr);

  return (result);
}

/*****************************************************************************
* @purpose  Retrieve the trap mode for a single trap.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    flagBits            @b{(input)}  selected trap from @ref OPEN_SNMP_TRAP_FLAGS_t
* @param    mode                @b{(output)} Set on or off
*
* @returns  OPEN_E_NONE         Get successful.
* @returns  OPEN_E_FAIL         Get failed.
* @returns  OPEN_E_PARAM        Error in parameter passed.  
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes
* 
* @end
****************************************************************************/
open_error_t testSnmpTrapFlagGet(openapiClientHandle_t *clientHandle,
                                 uint32_t flagBits,
                                 bool *mode)
{
  open_error_t result = OPEN_E_NONE;
  bool tmpMode = 0;

  result = openapiSnmpTrapFlagGet(clientHandle, flagBits, &tmpMode);
  if (result == OPEN_E_NONE)
  {
    *mode = tmpMode;
  }

  PRINTBADRESULT(result, "openapiSnmpTrapFlagGet", "");

  return (result);
}

/***************************************************************************** 
* @purpose  Set, then get the source interface which SNMP trap originates.
*           For demonstration purposes, this function retrieves the first
*           interface based on interface type and attempts to set it as the
*           source interface. If successful, the interface attributes are
*           retrieved and verified.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    type                @b{(input)}  type of interface to retrieve
*
* @returns  OPEN_E_NONE         Set is successful
* @returns  OPEN_E_FAIL         Set failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* @notes    The source interface must be a valid routing, loopback, tunnel, or VLAN interface.
* 
* @end
****************************************************************************/
open_error_t testSnmpTrapSourceInterface(openapiClientHandle_t *clientHandle,
                                         OPEN_INTF_TYPE_t type)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_LINK_STATE_t state;
  OPEN_INTF_TYPE_t tmpType;
  uint32_t tmpIntf;
  uint32_t intf;
  open_inet_addr_t addr;
  char msg[256];
  char ipAddr[32];

  memset(&ipAddr, 0, sizeof(ipAddr));

  sprintf(msg, "interface type : %s", getInterfaceType(type));

  result = openapiIfFirstGet(clientHandle, type, &intf);
  if (result == OPEN_E_FAIL)
  {
    result = OPEN_E_NOT_FOUND;
  }
  PRINTBADRESULT(result, "openapiIfFirstGet", msg);
  if (result == OPEN_E_NONE)
  {
    result = openapiSnmpTrapSrcIntfSet(clientHandle, intf);
    PRINTBADRESULT(result, "openapiSnmpTrapSrcIntfSet", msg);
    if (result == OPEN_E_NONE)
    {
      result = openapiSnmpTrapSrcIntfGet(clientHandle, &tmpIntf, &tmpType, &addr);
      PRINTBADRESULT(result, "openapiSnmpTrapSrcIntfGet", msg);

      if (result == OPEN_E_NONE)
      {
        if (addr.family == OPEN_AF_INET)
        {
          if (inet_ntop(AF_INET, (void*)&(addr.addr.ipv4), ipAddr, sizeof(ipAddr)) == NULL)
          {
            result = OPEN_E_PARAM;
          }
        }
        else
        {
          if (inet_ntop(AF_INET6, (void*)&(addr.addr.ipv6), ipAddr, sizeof(ipAddr)) == NULL)
          {
            result = OPEN_E_PARAM;
          }
        }
        PRINTBADRESULT(result, "inet_ntop", msg);
      }

      if (result == OPEN_E_NONE)
      {
        result = openapiIfLinkStateGet(clientHandle, intf, &state);
        PRINTBADRESULT(result, "openapiIfLinkStateGet", msg);
      }

      PRINTSANITYRESULTS(result, ((tmpIntf==intf)&&(tmpType==type)), __func__, msg);

      if (fVerbose)
      {
        if (result == OPEN_E_NONE)
        {
          printf("  Interface : %d\n"
                 "  Type      : %d\n"
                 "  Address   : %s\n"
                 "  State     : %d\n",
                 intf, type, ipAddr, state);
        }
      }
    }
  }


  return (result);
}

/********************************************************************* 
* @purpose  Create, or update an existing community.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    accessType          @b{(input)}  optional community access
* @param    ipAddr              @b{(input)}  optional IPv4 address, for allowed access
* @param    ipMask              @b{(input)}  optional IPv4 mask, for allowed access
* @param    status              @b{(input)}  community status
*
* @returns  OPEN_E_NONE         Set is successful 
* @returns  OPEN_E_FAIL         Set failed        
* @returns  OPEN_E_ERROR        The set request is invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
open_error_t testSnmpCommunityWithIPAddrCreate(openapiClientHandle_t *clientHandle,
                                               char *communityName,                               
                                               OPEN_SNMP_COMMUNITY_ACCESS_TYPE_t accessType,      
                                               char *ipAddr,                                      
                                               char *ipMask,
                                               OPEN_SNMP_COMMUNITY_STATUS_t status)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t temp = 0;
  open_buffdesc buffDesc;                         
  OPEN_SNMP_COMMUNITY_t community;
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];      

  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&community, 0, sizeof(community));
  community.type = accessType;

  inet_pton(AF_INET, ipAddr, (void*)&temp);
  community.ipAddr.family = OPEN_AF_INET;
  community.ipAddr.addr.ipv4 = ntohl(temp);

  temp = 0;
  inet_pton(AF_INET, ipMask, (void*)&temp);
  community.ipMask.family = OPEN_AF_INET;
  community.ipMask.addr.ipv4 = ntohl(temp);

  community.status = status;

  result = openapiSnmpCommunityAndIPAddrCreate(clientHandle,
                                               &buffDesc,
                                               community);

  PRINTBADRESULT(result, "testSnmpCommunityWithIPAddrCreate", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, communityName);
  return result;
}

/********************************************************************* 
* @purpose  Delete an existing community.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed
* @returns  OPEN_E_ERROR        The delete request is invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end
*********************************************************************/
open_error_t testSnmpCommunityAndIPAddrDelete(openapiClientHandle_t *clientHandle,
                                              char *communityName)
{
  open_error_t result = OPEN_E_NONE;
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_buffdesc buffDesc;  

  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  result = openapiSnmpCommunityAndIPAddrDelete(clientHandle, &buffDesc);
  
  PRINTBADRESULT(result, "testSnmpCommunityAndIPAddrDelete", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, communityName);
  return result;
}

/********************************************************************* 
* @purpose  Get an existing community.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
*
* @returns  OPEN_E_NONE         Get is successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        The get request is invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    
* 
* @end
*********************************************************************/
open_error_t testSnmpCommunityAndIPAddrGet(openapiClientHandle_t *clientHandle,
                                           char *communityName)
{
  open_error_t result = OPEN_E_NONE;
  OPEN_SNMP_COMMUNITY_t community;
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_buffdesc buffDesc;  

  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&community, 0, sizeof(community));

  result = openapiSnmpCommunityAndIPAddrGet(clientHandle, &buffDesc, &community);

  if (result == OPEN_E_NONE)
  {
    printf("SNMP community get is successful :%s \n", communityName);

    printf("Community Name: %s \n", str);
    printf("Access Type: %d \n", community.type);
    printf("IP address: %x \n", community.ipAddr.addr.ipv4);
    printf("IP address mask: %x \n", community.ipMask.addr.ipv4);
    printf("Status: %d \n", community.status);
  }  

  PRINTBADRESULT(result, "testSnmpCommunityAndIPAddrGet", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, communityName);
  return result;
}

/********************************************************************* 
* @purpose  Get next community.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
*
* @returns  OPEN_E_NONE         Get is successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        The get request is invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
* @notes    
* 
* @end
*********************************************************************/
open_error_t testSnmpCommunityAndIPAddrGetNext(openapiClientHandle_t *clientHandle,
                                               char *communityName)
{
  open_error_t result = OPEN_E_NONE;
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_buffdesc buffDesc;  

  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  result = openapiSnmpCommunityAndIPAddrGetNext(clientHandle, &buffDesc, &buffDesc);

  if (result == OPEN_E_NONE)
  {
    printf("SNMP community get next is successful :%s next community :%s \n", communityName, str);
  }  

  PRINTBADRESULT(result, "testSnmpCommunityAndIPAddrGet", "");
  PRINTSANITYRESULTS(result, 1==1, __func__, communityName);
  return result;
}

/********************************************************************* 
* @purpose  Create, or update an existing trap entry.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  IPv4 address, for allowed access
* @param    version             @b{(input)}  community version
* @param    status              @b{(input)}  community status
*
* @returns  OPEN_E_NONE         Set is successful 
* @returns  OPEN_E_FAIL         Set failed        
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_UNAVAIL      SNMP feature is not supported on this platform.
*
*
* @notes    Calling this API will change the running configuration of the switch
* 
* @end 
*********************************************************************/
open_error_t testSnmpTrapManagerConfigCreate(openapiClientHandle_t *clientHandle,
                                             char *communityName,                               
                                             char *ipAddr,                                      
                                             OPEN_SNMP_TRAP_VERSION_t version,
                                             OPEN_SNMP_COMMUNITY_STATUS_t status)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t temp = 0;
  open_buffdesc buffDesc;                         
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  OPEN_SNMP_TRAP_CONFIG_t trapConfig; 
  open_inet_addr_t addr;

  memset(str, 0, sizeof(str));
  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&trapConfig, 0, sizeof(trapConfig));
 
  memset(&addr, 0, sizeof(addr));  
  inet_pton(AF_INET, ipAddr, (void*)&temp);
  addr.family = OPEN_AF_INET;
  addr.addr.ipv4 = ntohl(temp);

  trapConfig.version = version;
  trapConfig.status = status;

  result = openapiTrapManagerConfigCreate(clientHandle,
                                          &buffDesc,
                                          addr,
                                          trapConfig);

  PRINTBADRESULT(result, __FUNCTION__, "");
  PRINTSANITYRESULTS(result, 1==1, __FUNCTION__, communityName);
  return result;
}

/*****************************************************************************
* @purpose  Deletes an existing SNMP Trap entry and related entries if 
*           SNMP config API is not supported.     
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  community IP address
*
* @returns  OPEN_E_NONE         Delete is successful
* @returns  OPEN_E_FAIL         Delete failed     
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform. 
*
* @notes    Calling this API will change the running configuration of the switch
*
* @end 
****************************************************************************/
open_error_t testSnmpTrapManagerConfigDelete(openapiClientHandle_t *clientHandle,
                                             char *communityName,                      
                                             char *ipAddr)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t temp = 0;
  open_buffdesc buffDesc;                         
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_inet_addr_t addr;

  memset(str, 0, sizeof(str));
  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&addr, 0, sizeof(addr));  
  inet_pton(AF_INET, ipAddr, (void*)&temp);
  addr.family = OPEN_AF_INET;
  addr.addr.ipv4 = ntohl(temp);

  result = openapiTrapManagerConfigDelete(clientHandle,
                                          &buffDesc,
                                          addr);

  PRINTBADRESULT(result, __FUNCTION__, "");
  PRINTSANITYRESULTS(result, 1==1, __FUNCTION__, communityName);
  return result;
}

/*****************************************************************************
* @purpose  Get SNMP Trap information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  community IP address
*
* @returns  OPEN_E_NONE         Get successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes
*
* @end
****************************************************************************/
open_error_t testSnmpTrapManagerConfigGet(openapiClientHandle_t *clientHandle,
                                          char *communityName,
                                          char *ipAddr)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t temp = 0;
  open_buffdesc buffDesc;                         
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_inet_addr_t addr;
  OPEN_SNMP_TRAP_CONFIG_t trapConfig;

  memset(str, 0, sizeof(str));
  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(&addr, 0, sizeof(addr));  
  inet_pton(AF_INET, ipAddr, (void*)&temp);
  addr.family = OPEN_AF_INET;
  addr.addr.ipv4 = ntohl(temp);

  memset(&trapConfig, 0, sizeof(trapConfig));

  result = openapiTrapManagerConfigGet(clientHandle,
                                       &buffDesc,
                                       addr,
                                       &trapConfig);

  if (result == OPEN_E_NONE)
  {
    printf("Get successful for SNMP trap entry. \n");
    printf("Community :%s \n", str);
    printf("IP address :%s [%x] \n", ipAddr, addr.addr.ipv4);
    printf("Version :%d \n", trapConfig.version);
    printf("Status :%d \n", trapConfig.status);
  }  
     
  PRINTBADRESULT(result, __FUNCTION__, "");
  PRINTSANITYRESULTS(result, 1==1, __FUNCTION__, communityName);
  return result;
}

/*****************************************************************************
* @purpose  Get next SNMP Trap information.
*
* @param    client_handle       @b{(input)}  client handle from registration API
* @param    communityName       @b{(input)}  community name
* @param    ipAddr              @b{(input)}  community IP address
*
* @returns  OPEN_E_NONE         Get successful
* @returns  OPEN_E_FAIL         Get failed
* @returns  OPEN_E_ERROR        if the set request was invalid
* @returns  OPEN_E_PARAM        Error in parameter passed
* @returns  OPEN_E_UNAVAIL      SNMP feature or SNMP config API is not supported 
*                               on this platform.
*
* @notes
*
* @end
****************************************************************************/
open_error_t testSnmpTrapManagerConfigGetNext(openapiClientHandle_t *clientHandle,
                                              char *communityName,
                                              char *ipAddr)
{
  open_error_t result = OPEN_E_NONE;
  uint32_t temp = 0;
  open_buffdesc buffDesc;                         
  open_buffdesc tmpBuf;                         
  char str[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  char tmp[OPENAPI_SNMP_COMMUNITY_NAME_MAX];
  open_inet_addr_t addr;
  char ipAddrStr[OPEN_IP_ADDR_LENGTH];

  memset(str, 0, sizeof(str));
  strncpy(str, communityName, sizeof(str));
  buffDesc.pstart = str;
  buffDesc.size = strlen(str) + 1;

  memset(tmp, 0, sizeof(tmp));
  tmpBuf.pstart = tmp;
  tmpBuf.size = sizeof(tmp);

  memset(&addr, 0, sizeof(addr));  
  inet_pton(AF_INET, ipAddr, (void*)&temp);
  addr.family = OPEN_AF_INET;
  addr.addr.ipv4 = ntohl(temp);

  result = openapiTrapManagerConfigCommunityGetNext(clientHandle,
                                                    &buffDesc,
                                                    addr,
                                                    &tmpBuf);

  if (result == OPEN_E_NONE)
  {
    printf("Get next successful for SNMP trap entry. \n");
    printf("Community :%s \n", communityName);
    printf("IP address :%s [%x] \n", ipAddr, addr.addr.ipv4);
    printf("Next Community :%s \n", tmp);
  }

  result = openapiTrapManagerConfigIPGetNext(clientHandle,
                                             &buffDesc,
                                             addr,
                                             &addr);

  if (result == OPEN_E_NONE)
  {
    if (addr.family == OPEN_AF_INET)
    {
      memset(&ipAddrStr, 0, sizeof(ipAddrStr));
      inet_ntop(AF_INET, (void*)&(addr.addr.ipv4), ipAddrStr, sizeof(ipAddrStr)); 
    }
    printf("Next IP address :%s [%x] \n", ipAddrStr, addr.addr.ipv4);
  }  

  PRINTBADRESULT(result, __FUNCTION__, "");
  PRINTSANITYRESULTS(result, 1==1, __FUNCTION__, communityName);
  return result;
}

/*******************************************************************
*
* @brief  This is the main function that will demonstrate 
*         SNMP OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_NONE;
  OPEN_SNMP_TRAP_FLAGS_t trapFlags = 0;
  bool trapMode;

  if (argc==2)
  {
    if (strcmp(argv[1],"-?")==0)
    {
      printf("\nUsage: snmp_example <-v> ! verbose mode\n\n");
      return 0;
    }
    else if (strcmp(argv[1],"-v")==0)
    {
      fVerbose = true;
    }
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("snmp_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting SNMP API example application");

  printf ("\n");

  /* Execute sanity tests */
  printf ("Begin Sanity tests...\n");

  testSnmpSysNameSet(&clientHandle, "system-name");
  testSnmpSysLocationSet(&clientHandle, "system-location");
  testSnmpSysContactSet(&clientHandle, "system-contact");
  testSnmpLocalEngineIdSet(&clientHandle, "default");

  printf ("\nCreate SNMP users...\n");
  testSnmpUserCreate(&clientHandle,
                     "admin1",
                     "network-admin1",
                     OPEN_USM_USER_AUTH_PROTOCOL_NONE,
                     "",
                     "",
                     OPEN_USM_USER_PRIV_PROTOCOL_NONE,
                     "",
                     "",
                     "");

  testSnmpUserCreate(&clientHandle,
                     "admin2",
                     "network-admin2",
                     OPEN_USM_USER_AUTH_PROTOCOL_SHA,
                     "",
                     "ThisIsaShaKeywordThatMustBe48CharactersInLength.",
                     OPEN_USM_USER_PRIV_PROTOCOL_NONE,
                     "",
                     "",
                     "bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb");

  testSnmpUserCreate(&clientHandle,
                     "admin3",
                     "network-admin",
                     OPEN_USM_USER_AUTH_PROTOCOL_SHA,
                     "ThisIsaShaPasswordThatIs32Chars.",
                     "",
                     OPEN_USM_USER_PRIV_PROTOCOL_DES,
                     "ThisIsaDesPasswordThatIs32Chars.",
                     "",
                     "cccccccccccccccccccccccccccccccc");

  testSnmpUserCreate(&clientHandle,
                     "admin4",
                     "network-admin2",
                     OPEN_USM_USER_AUTH_PROTOCOL_SHA,
                     "",
                     "ThisIsaShaKeywordThatMustBe48CharactersInLength.",
                     OPEN_USM_USER_PRIV_PROTOCOL_DES,
                     "",
                     "ThisIsaDesKeywordThatMustBe32chr",
                     "dddddddddddddddddddddddddddddddd");

  testSnmpUserCreate(&clientHandle, 
                     "admin5", 
                     "network-admin", 
                     OPEN_USM_USER_AUTH_PROTOCOL_MD5, 
                     "", 
                     "ABCDEFABCDEFABCDEFABCDEFABCDEFAB", 
                     OPEN_USM_USER_PRIV_PROTOCOL_DES, 
                     "", 
                     "ABCDEFABCDEFABCDEFABCDEFABCDEFAC", 
                     "999999999999999999999999999999");

  testSnmpUserCreate(&clientHandle,
                     "delete-user",
                     "network-admin",
                     OPEN_USM_USER_AUTH_PROTOCOL_SHA,
                     "ThisIsaShaPasswordThatIs32Chars.",
                     "",
                     OPEN_USM_USER_PRIV_PROTOCOL_NONE,
                     "",
                     "",
                     "0123456789abcdef");

  testSnmpUserCreate(&clientHandle,
                     "admin3-user",
                     "network-admin",
                     OPEN_USM_USER_AUTH_PROTOCOL_SHA,
                     "ThisIsaShaPasswordThatIs32Chars.",
                     "",
                     OPEN_USM_USER_PRIV_PROTOCOL_AES128,
                     "ThisIsaDesPasswordThatIs32Chars.",
                     "",
                     "cccccccccccccccccccccccccccccccc");

  testSnmpUserCreate(&clientHandle,
                     "admin4-user",
                     "network-admin",
                     OPEN_USM_USER_AUTH_PROTOCOL_SHA,
                     "",
                     "ThisIsaShaKeywordThatMustBe48CharactersInLength.",
                     OPEN_USM_USER_PRIV_PROTOCOL_AES128,
                     "",
                     "ThisIsaDesKeywordThatMustBe32chr",
                     "dddddddddddddddddddddddddddddddd");

  testSnmpUserCreate(&clientHandle, 
                     "admin5-user", 
                     "network-admin", 
                     OPEN_USM_USER_AUTH_PROTOCOL_MD5, 
                     "", 
                     "ABCDEFABCDEFABCDEFABCDEFABCDEFAB", 
                     OPEN_USM_USER_PRIV_PROTOCOL_AES128, 
                     "", 
                     "ABCDEFABCDEFABCDEFABCDEFABCDEFAC", 
                     "999999999999999999999999999999");

  printf ("\nCreate SNMP groups...\n");
  testSnmpGroupCreate(&clientHandle,
                      "network-admin",
                      OPEN_SNMP_SECURITY_MODEL_USM,
                      OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV,
                      "context-1",
                      "view-1",
                      "view-1",
                      "view-1");
  testSnmpGroupCreate(&clientHandle,
                      "network-guest",
                      OPEN_SNMP_SECURITY_MODEL_SNMPV1,
                      OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV,
                      "context-2",
                      "view-2",
                      "view-2",
                      "view-2");
  testSnmpGroupCreate(&clientHandle,
                      "delete-group",
                      OPEN_SNMP_SECURITY_MODEL_SNMPV2C,
                      OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV,
                      "context-3",
                      "view-3",
                      "view-3",
                      "view-3");

  printf ("\nCreate SNMP views...\n");
  testSnmpViewCreate(&clientHandle,
                     "view-1",
                     "1.3.6.1.4.1.4413",
                     OPEN_SNMP_VIEW_TYPE_INCLUDED);
  testSnmpViewCreate(&clientHandle,
                     "view-2",
                     "ifEntry.*.1",
                     OPEN_SNMP_VIEW_TYPE_EXCLUDED);

  printf ("\nCreate SNMP filters...\n");
  testSnmpFilterCreate(&clientHandle,
                     "filter-1",
                     "1.3.6.1.4.1.4413",
                     OPEN_SNMP_VIEW_TYPE_INCLUDED);
  testSnmpFilterCreate(&clientHandle,
                     "filter-2",
                     "ifEntry.*.1",
                     OPEN_SNMP_VIEW_TYPE_EXCLUDED);

  printf ("\nCreate SNMP hosts...\n");
  testSnmpHostCreate(&clientHandle,
                     "10.10.10.1", 0,
                     "public",
                     OPEN_SNMP_SECURITY_MODEL_SNMPV1,
                     OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV,
                     OPEN_SNMP_NOTIFY_TYPE_TRAP,
                     0, 0, "");
  testSnmpHostCreate(&clientHandle, 
                     "10.10.10.2", 0,
                     "public", 
                     OPEN_SNMP_SECURITY_MODEL_SNMPV2C, 
                     OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV, 
                     OPEN_SNMP_NOTIFY_TYPE_INFORM, 
                     0, 0, "");

  testSnmpHostCreate(&clientHandle,
                     "10.10.10.3", 0,
                     "public",
                     OPEN_SNMP_SECURITY_MODEL_SNMPV2C,
                     OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV,
                     OPEN_SNMP_NOTIFY_TYPE_TRAP,
                     0, 0, "");

  testSnmpHostCreate(&clientHandle,
                     "2001::", 65535,
                     "admin",
                     OPEN_SNMP_SECURITY_MODEL_USM,
                     OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV,
                     OPEN_SNMP_NOTIFY_TYPE_INFORM,
                     100, 200, "ipv6-filter");

  printf ("\nCreate SNMP Communities...\n");
  testSnmpCommunity(&clientHandle, "public-test", OPEN_SNMP_COMMUNITY_ACCESS_NONE, "view_test", "20.20.20.1");
  testSnmpCommunity(&clientHandle, "private-test", OPEN_SNMP_COMMUNITY_ACCESS_RW, "", "");
  testSnmpCommunity(&clientHandle, "delete-community", OPEN_SNMP_COMMUNITY_ACCESS_RO, "", "");
  testSnmpCommunityGroup(&clientHandle, "delete-private-map", OPENAPI_SNMP_DEFAULT_WRITE_STR, "30.30.30.1");

  printf ("\nEnable misc SNMP trap flags...\n");
  trapFlags |= OPEN_SNMP_MULTI_USERS_TRAP_FLAG;
  trapFlags |= OPEN_SNMP_USER_AUTH_TRAP_FLAG;
  trapFlags |= OPEN_SNMP_VIOLATION_TRAP_FLAG;
  testSnmpTrapFlagSet(&clientHandle, trapFlags, true);

  printf ("\nRetrieve some SNMP records...\n");
  testSnmpSysNameGet(&clientHandle);
  testSnmpSysLocationGet(&clientHandle);
  testSnmpSysContactGet(&clientHandle);
  testSnmpLocalEngineIdGet(&clientHandle);

  testSnmpUserGet(&clientHandle, "admin1");
  testSnmpGroupGet(&clientHandle, "network-admin",
                      OPEN_SNMP_SECURITY_MODEL_USM,
                      OPEN_SNMP_SECURITY_LEVEL_AUTHPRIV,
                      "context-1");
  testSnmpViewGet(&clientHandle, "view-1", "broadcom");
  testSnmpViewGet(&clientHandle, "view-2", "ifEntry.*.1");
  testSnmpFilterGet(&clientHandle, "filter-1", "broadcom");
  testSnmpFilterGet(&clientHandle, "filter-2", "ifEntry.*.1");
  testSnmpHostGet(&clientHandle, "2001::");
  testSnmpCommunityGet(&clientHandle, "public-test");
  testSnmpCommunityGet(&clientHandle, "private-test");
  testSnmpCommunityGet(&clientHandle, "delete-private-map");
  trapFlags |= OPEN_SNMP_MULTI_USERS_TRAP_FLAG;
  testSnmpTrapFlagGet(&clientHandle, trapFlags, &trapMode);
  PRINTSANITYRESULTS(result, (trapMode == true), "testSnmpTrapFlagGet", "OPEN_SNMP_MULTI_USERS_TRAP_FLAG");
  trapFlags |= OPEN_SNMP_USER_AUTH_TRAP_FLAG;
  testSnmpTrapFlagGet(&clientHandle, trapFlags, &trapMode);
  PRINTSANITYRESULTS(result, (trapMode == true), "testSnmpTrapFlagGet", "OPEN_SNMP_USER_AUTH_TRAP_FLAG");
  trapFlags |= OPEN_SNMP_VIOLATION_TRAP_FLAG;
  testSnmpTrapFlagGet(&clientHandle, trapFlags, &trapMode);
  PRINTSANITYRESULTS(result, (trapMode == true), "testSnmpTrapFlagGet", "OPEN_SNMP_VIOLATION_TRAP_FLAG");

  printf ("\nDelete some SNMP records...\n");
  testSnmpLocalEngineIdDelete(&clientHandle);
  testSnmpViewDelete(&clientHandle, "view-2", "ifEntry.*.1");
  testSnmpFilterDelete(&clientHandle, "filter-2", "ifEntry.*.1");
  testSnmpHostDelete(&clientHandle, "10.10.10.3", OPEN_SNMP_NOTIFY_TYPE_TRAP);
  testSnmpUserDelete(&clientHandle,
                     "delete-user",
                     "0123456789abcdef");
  testSnmpGroupDelete(&clientHandle,
                      "delete-group",
                      OPEN_SNMP_SECURITY_MODEL_SNMPV2C,
                      OPEN_SNMP_SECURITY_LEVEL_NOAUTHNOPRIV,
                      "context-3");
  testSnmpCommunityDelete(&clientHandle, "delete-community");
  testSnmpCommunityDelete(&clientHandle, "delete-private-map");

  printf ("\nAttempt to set the trap source-interface...\n");
  /* Routing must be enabled on first interface */
  testSnmpTrapSourceInterface(&clientHandle, OPEN_INTF_TYPE_VLAN);

  printf("\nCreating SNMP communities with IP address and mask\n");
  testSnmpCommunityWithIPAddrCreate(&clientHandle,
                                    "test1",
                                    OPEN_SNMP_COMMUNITY_ACCESS_RO,
                                    "10.11.12.1",
                                    "255.255.255.255",
                                    OPEN_SNMP_COMMUNITY_STATUS_VALID);

  testSnmpCommunityWithIPAddrCreate(&clientHandle,
                                    "test2",
                                    OPEN_SNMP_COMMUNITY_ACCESS_RO,
                                    "10.11.12.2",
                                    "255.255.255.255",
                                    OPEN_SNMP_COMMUNITY_STATUS_VALID);

  testSnmpCommunityAndIPAddrGet(&clientHandle,
                                "test1");

  testSnmpCommunityAndIPAddrGetNext(&clientHandle,
                                    "test1");

  testSnmpCommunityAndIPAddrDelete(&clientHandle,
                                   "test2");

  printf("\nCreating SNMP Trap configuration. \n");
  testSnmpTrapManagerConfigCreate(&clientHandle,
                                  "community1",
                                  "22.23.24.1",
                                  OPEN_SNMP_TRAP_VER_SNMPV2C,
                                  OPEN_SNMP_COMMUNITY_STATUS_VALID);
  
  testSnmpTrapManagerConfigCreate(&clientHandle,
                                  "community2",
                                  "22.23.24.2",
                                  OPEN_SNMP_TRAP_VER_SNMPV1,
                                  OPEN_SNMP_COMMUNITY_STATUS_INVALID);
  
  testSnmpTrapManagerConfigCreate(&clientHandle,
                                  "community3",
                                  "22.23.24.3",
                                  OPEN_SNMP_TRAP_VER_SNMPV1,
                                  OPEN_SNMP_COMMUNITY_STATUS_VALID);
  
  testSnmpTrapManagerConfigGet(&clientHandle,
                               "community1",
                               "22.23.24.1");

  testSnmpTrapManagerConfigGet(&clientHandle,
                               "community2",
                               "22.23.24.2");

  testSnmpTrapManagerConfigGet(&clientHandle,
                               "community3",
                               "22.23.24.3");

  testSnmpTrapManagerConfigGetNext(&clientHandle,
                                   "community1",
                                   "22.23.24.1");

  testSnmpTrapManagerConfigDelete(&clientHandle,
                                  "community2",
                                  "22.23.24.2");

  printf ("\nComplete.\n");

  /* Log goodbye message with OPEN */
  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Stopping SNMP API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

