/*********************************************************************
*
* Copyright 2016-2020 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  snooping_example.c
*
* @purpose   L2 Multicast Snooping APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    06/19/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_snooping.h"

/*******************************************************************
*
* @brief  This is the main function that will demonstrates
*         L2 Multicast Snooping APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_NONE;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100] = {0};

  open_buffdesc maskIn;
  char bufMaskIn[3000] = {0};
  open_buffdesc maskOut;
  char bufMaskOut[3000] = {0};
  OPEN_BOOL_t status = OPEN_FALSE;
  OPEN_BOOL_t floodAll = OPEN_FALSE;
  OPEN_CONTROL_t mode = OPEN_DISABLE;
  uint8_t family = OPEN_AF_INET;
  uint32_t processedFrames = 0;
  uint32_t forwardedFrames = 0;
  uint32_t intIfNum = 1;
  uint32_t queryTime = 0;
  uint32_t responseTime = 0;
  uint16_t vlanResponseTime = 0;
  OPEN_CONTROL_t vlanMode = OPEN_DISABLE;
  uint32_t vlanID = 1; 
  uint32_t auto_vlan = 0;

  maskIn.pstart = bufMaskIn;
  maskIn.size = sizeof(bufMaskIn);
  maskOut.pstart = bufMaskOut;
  maskOut.size = sizeof(bufMaskOut);

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("snooping_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (Result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fails until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error\n");
  }
  printf ("\n");

  do
  {
    /* Testing of checking if the passed family type snooping is supported or not */
    result = openapiSnoopProtocolGet(&clientHandle, family, &status);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to check if the passed family type snooping is supported or not. (Result = %d)\n", result);
      break;
    }
    else if (OPEN_FALSE == status)
    {
      printf("Family type isn't supported\n");
      break;
    }
    else
    {
      printf("Family type is supported\n");
    }

    /* Testing of retrieving the IGMP Snooping Admin mode */
    result = openapiSnoopAdminModeGet(&clientHandle, &mode, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to retrieve the IGMP Snooping Admin mode. (Result = %d)\n", result);
    }
    else
    {
      printf("The IGMP Snooping Admin mode is: %d\n", mode);
    }

    /* Testing of setting the IGMP Snooping Admin mode */
    result = openapiSnoopAdminModeSet(&clientHandle, mode, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to set the IGMP Snooping Admin mode. (Result = %d)\n", result);
    }
    else
    {
      printf("The IGMP Snooping Admin mode set with value: %d\n", mode);
    }

    /* Testing of retrieving the number of multicast control frames processed */
    result = openapiSnoopControlFramesProcessedGet(&clientHandle, &processedFrames, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to retrieve the number of multicast control frames processed. (Result = %d)\n", result);
    }
    else
    {
      printf("The number of multicast control frames processed is: %d\n", processedFrames);
    }

    /* Testing of retrieving the number of multicast data frames forwarded */
    result = openapiSnoopDataFramesForwardedGet(&clientHandle, &forwardedFrames, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to retrieve the number of multicast data frames forwarded. (Result = %d)\n", result);
    }
    else
    {
      printf("The number of multicast data frames forwarded is: %d\n", forwardedFrames);
    }

    /* Testing of retrieving the IGMP/MLD Proxy Querier Admin mode */
    result = openapiSnoopProxyQuerierAdminModeGet(&clientHandle, &mode, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to retrieve the IGMP/MLD Proxy Querier Admin mode. (Result = %d)\n", result);
    }
    else
    {
      printf("The IGMP/MLD Proxy Querier Admin mode is: %d\n", mode);
    }

    /* Testing of setting the IGMP/MLD Proxy Querier Admin mode */
    result = openapiSnoopAdminModeSet(&clientHandle, mode, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to set the IGMP/MLD Proxy Querier Admin mode. (Result = %d)\n", result);
    }
    else
    {
      printf("The IGMP/MLD Proxy Querier Admin mode set with value: %d\n", mode);
    }

    /* Testing of retrieving the configured mode for flooding unknown multicast traffic to all ports */
    result = openapiSnoopFloodUnknownAllPortsGet(&clientHandle, &floodAll);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to retrieve the configured mode for flooding unknown multicast traffic. (Result = %d)\n", result);
    }
    else
    {
      printf("The configured mode for flooding unknown multicast traffic is: %d\n", floodAll);
    }

    /* Testing of setting the configured mode for flooding unknown multicast traffic to all ports */
    result = openapiSnoopFloodUnknownAllPortsSet(&clientHandle, floodAll);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to set the configured mode for flooding unknown multicast traffic. (Result = %d)\n", result);
    }
    else
    {
      printf("The configured mode for flooding unknown multicast traffic set with value: %d\n", floodAll);
    }

    /* Testing of setting the IGMP interface mask of interfaces enabled for IGMP Snooping */
    strcpy((char *)maskIn.pstart, "1,3,5,10");
    maskIn.size = strlen((char *)maskIn.pstart) + 1;
    result = openapiSnoopIntfEnabledMaskSet(&clientHandle, &maskIn, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to set the interface mask of interfaces enabled for IGMP Snooping. (Result = %d)\n", result);
    }
    else
    {
      printf("The the interface mask of interfaces enabled for IGMP Snooping set with mask: %s\n", (char *)maskIn.pstart);
    }

    /* Testing of retrivieng the IGMP Snooping Admin mode */
    result = openapiSnoopIntfEnabledMaskGet(&clientHandle, &maskOut, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to retrive the interface mask of interfaces enabled for IGMP Snooping. (Result = %d)\n", result);
    }
    else
    {
      printf("The interface mask of interfaces enabled for IGMP Snooping is: %s\n", (char *)maskOut.pstart);
    }

    /* Testing of checking if interface is a valid IGMP snooping Interface */
    result = openapiSnoopIsValidIntf(&clientHandle, intIfNum, &status);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to check if interface is a valid IGMP snooping Interface. (Result = %d)\n", result);
      break;
    }
    else if (OPEN_FALSE == status)
    {
      printf("Interface %d isn't valid\n", intIfNum);
      break;
    }
    else
    {
      printf("Interface %d is valid\n", intIfNum);
    }

    /* Testing of retrieving the IGMP Snooping mode for the specified interface */
    result = openapiSnoopIntfModeGet(&clientHandle, intIfNum, &mode, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to retrieve IGMP Snooping mode for the interface %d. (Result = %d)\n", intIfNum, result);
    }
    else
    {
      printf("The IGMP Snooping mode for the interface %d is: %d\n", intIfNum, mode);
    }

    /* Testing of setting the IGMP Snooping mode for the specified interface */
    result = openapiSnoopIntfModeSet(&clientHandle, intIfNum, mode, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to set the IGMP Snooping mode for the interface %d. (Result = %d)\n", intIfNum, result);
    }
    else
    {
      printf("The IGMP Snooping mode for the interface %d set with value: %d\n", intIfNum, mode);
    }

    /* Testing of retrieving the IGMP group membership interval for the specified interface */
    result = openapiSnoopIntfGroupMembershipIntervalGet(&clientHandle, intIfNum, &queryTime, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to retrieve IGMP group membership interval for the interface %d. (Result = %d)\n", intIfNum, result);
    }
    else
    {
      printf("The IGMP group membership interval for the interface %d is: %d\n", intIfNum, queryTime);
    }

    /* Testing of setting the IGMP group membership interval for the specified interface */
    result = openapiSnoopIntfGroupMembershipIntervalSet(&clientHandle, intIfNum, queryTime, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to set the IGMP group membership interval for the interface %d. (Result = %d)\n", intIfNum, result);
    }
    else
    {
      printf("The IGMP group membership interval for the interface %d set with value: %d\n", intIfNum, queryTime);
    }

    /* Testing of retrieving the IGMP query response time for the specified interface */
    result = openapiSnoopIntfResponseTimeGet(&clientHandle, intIfNum, &responseTime, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to retrieve IGMP query response time for the interface %d. (Result = %d)\n", intIfNum, result);
    }
    else
    {
      printf("The IGMP query response time for the interface %d is: %d\n", intIfNum, responseTime);
    }

    /* Testing of setting the IGMP query response time for the specified interface */
    result = openapiSnoopIntfResponseTimeSet(&clientHandle, intIfNum, responseTime, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to set the IGMP query response time for the interface %d. (Result = %d)\n", intIfNum, result);
    }
    else
    {
      printf("The IGMP query response time for the interface %d set with value: %d\n", intIfNum, responseTime);
    }

    /* Testing of retrieving the Fast-Leave Admin mode for the specified interface */
    result = openapiSnoopIntfFastLeaveAdminModeGet(&clientHandle, intIfNum, &mode, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to retrieve Fast-Leave Admin mode for the interface %d. (Result = %d)\n", intIfNum, result);
    }
    else
    {
      printf("The Fast-Leave Admin mode for the interface %d is: %d\n", intIfNum, mode);
    }

    /* Testing of setting the Fast-Leave Admin mode for the specified interface */
    result = openapiSnoopIntfFastLeaveAdminModeSet(&clientHandle, intIfNum, mode, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to set the Fast-Leave Admin mode for the interface %d. (Result = %d)\n", intIfNum, result);
    }
    else
    {
      printf("The Fast-Leave Admin mode for the interface %d set with value: %d\n", intIfNum, mode);
    }

    /* Testing of retrieving the Proxy Querier Admin mode for the specified interface */
    result = openapiSnoopIntfProxyQuerierModeGet(&clientHandle, intIfNum, &mode, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to retrieve Proxy Querier Admin mode for the interface %d. (Result = %d)\n", intIfNum, result);
    }
    else
    {
      printf("The Proxy Querier Admin mode for the interface %d is: %d\n", intIfNum, mode);
    }

    /* Testing of setting the Proxy Querier Admin mode for the specified interface */
    result = openapiSnoopIntfProxyQuerierModeSet(&clientHandle, intIfNum, mode, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to set the Proxy Querier Admin mode for the interface %d. (Result = %d)\n", intIfNum, result);
    }
    else
    {
      printf("The Proxy Querier Admin mode for the interface %d set with value: %d\n", intIfNum, mode);
    }

	   /* Testing of flushing all snooping entries */
    result = openapiSnoopEntriesFlush(&clientHandle, family);
    if (OPEN_E_NONE != result)
    {
      printf("Failed to flush all entries. (Result = %d)\n", result);
    }
    else
    {
      printf("All entries flushed successful.\n");
    }

    printf("---Snooping VLAN---\n");
    do
    {
      if (OPEN_E_NONE == openapiSnoopVlanCheckValid(&clientHandle, vlanID)) 
      { 
        printf("Vlan ID is: %d.\n", vlanID);
      }
      else
      {
        if (OPEN_E_NONE == openapiSnoopVlanNextGet(&clientHandle, vlanID, &vlanID)) 
        {
          printf("Vlan ID is: %d.\n", vlanID);
        }
        else
        {
          printf("Failed to get vlan ID.\n");
          break;
        }
      }

      if (OPEN_E_NONE == openapiSnoopVlanFastLeaveModeGet(&clientHandle, vlanID, &status, family))
      {
        printf("Snooping fast leave mode is: ");
        if (status == OPEN_TRUE) 
        {
          printf("enabled.\n");
        }
        else if (status == OPEN_FALSE) 
        {
          printf("disabled.\n");
        }
        else
        {
          printf("undefined.\n");
        }
      }
      else
      {
        printf("Failed to get fast leave mode.\n");
      }

      if (OPEN_E_NONE == openapiSnoopVlanMaximumResponseTimeGet(&clientHandle, vlanID, &vlanResponseTime, family))
      {
        printf("IGMP snooping maximum response time for the specified VLAN[%d] is: %d ", vlanID, vlanResponseTime);
      }
      else
      {
        printf("Failed to get IGMP snooping maximum response time for the specified VLAN.\n");
      }

      if (OPEN_E_NONE == openapiSnoopVlanMaximumResponseTimeSet(&clientHandle, vlanID, vlanResponseTime, family))
      {
        printf("IGMP snooping maximum response time for the specified VLAN was set successfully\n");
      }
      else
      {
        printf("Failed to set IGMP snooping maximum response time for the specified VLAN.\n");
      }

      /* Testing of retrieving the IGMP Snooping VLAN mode */
      result = openapiSnoopQuerierVlanModeGet(&clientHandle, vlanID, &vlanMode, family);
      if (OPEN_E_NONE != result)
      {
        printf("Failed to retrieve the IGMP Snooping VLAN mode. (Result = %d)\n", result);
      }
      else
      {
        printf("IGMP Snooping VLAN mode for the specified VLAN[%d] is: %d\n", vlanID, vlanMode);
      }

      /* Testing of setting the IGMP Snooping VLAN mode */
      result = openapiSnoopQuerierVlanModeSet(&clientHandle, vlanID, vlanMode, family);
      if (OPEN_E_NONE != result)
      {
        printf("Failed to set the IGMP Snooping VLAN mode. (Result = %d)\n", result);
      }
      else
      {
        printf("IGMP Snooping VLAN mode for the specified VLAN[%d] is set with value: %d\n", vlanID, vlanMode);
      }

      /* Testing of retrieving the IGMP Snooping Auto-Video mode */
      result = openapiSnoopAutoVideoModeGet(&clientHandle, &vlanMode);
      if (OPEN_E_NONE != result)
      {
        printf("Failed to retrieve the IGMP Snooping Auto-Video mode. (Result = %d)\n", result);
      }
      else
      {
        printf("IGMP Snooping Auto-Video mode is: %s\n", 
               (vlanMode == OPEN_ENABLE) ? "ENABLE" : "DISABLE");
      }

      mode = (vlanMode == OPEN_ENABLE) ? OPEN_DISABLE : OPEN_ENABLE;
      /* Testing of setting the IGMP Snooping Auto-Video mode */
      result = openapiSnoopAutoVideoModeSet(&clientHandle, mode);
      if (OPEN_E_NONE != result)
      {
        printf("Failed to set the IGMP Snooping Auto-Video mode. (Result = %d)\n", result);
      }
      else
      {
        printf("IGMP Snooping Auto-Video mode is set with value: %s\n", 
               (mode == OPEN_ENABLE) ? "ENABLE" : "DISABLE");
      }

      result = openapiSnoopAutoVideoVlanGet(&clientHandle, &auto_vlan);
      if (OPEN_E_NONE != result)
      {
        printf("Failed to retrieve the IGMP Snooping Auto-Video VLAN ID. (Result = %d)\n", result);
      }
      else
      {
        printf("IGMP Snooping Auto-Video VLAN is: %d\n", auto_vlan);
      }

      break;
    }while (true);

    printf("Press enter to continue...\n");
    getchar();
  } while (0);

  /* Log goodbye message with OPEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping Snooping API example application");

  (void) openapiClientTearDown(&clientHandle);
  return 0;
}

