
/*********************************************************************
*
* (C) Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
*************************************************************************
*
* @filename  cosq_stats_example.c
*
* @purpose   Get COS Statistics Example.
*
* @component OPEN
*
* @comments
*
* @create    9/28/2016
*
* @end
*
***************************************************************************/
#include <stdlib.h>
#include <stdbool.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_cos_stats.h"

/*****************************************************************************
* @purpose  Read MC Queue transmits
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  interface number
* @param    queue          @b{(input)}  queue number (OPEN_COS_ALL_QUEUES all queues)
* @param    numQueues      @b{(input)}  number of CoS queues
* 
* @returns  none
* 
* @notes
* 
* @end
******************************************************************************/
open_error_t readMCQueueTx(openapiClientHandle_t *clientHandle,
                           uint32_t intIfNum,
                           uint32_t queue,
                           uint32_t numQueues)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc counts;
  uint64_t *p = (uint64_t *) 0;
  uint32_t countsRead;
  int i;

  p = (uint64_t *) calloc(numQueues, sizeof(*p));
  if (!p) 
  {
    printf("Unable to allocate buffer of %u bytes\n", numQueues * sizeof(*p));
    exit(1);
  }
  counts.pstart = p;
  counts.size = numQueues * sizeof(*p); 

  result = openapiCosQueueMcTxPacketsGet(clientHandle,
                                         intIfNum,
                                         queue,
                                         &counts,
                                         &countsRead);
 
  if (OPEN_E_NONE == result)
  {
    for (i = 0; i < countsRead; i++)
    {
      if (queue != OPEN_COS_ALL_QUEUES)
      {
        printf("MC Queue TX interface %u queue %u: %llu\n", intIfNum, queue, p[i]);
      }
      else
      {
        printf("MC Queue TX interface %u queue %d: %llu\n", intIfNum, i, p[i]);
      }
    }
  }
  else
  {
    printf("FAILED (%d) MC Queue TX interface %u queue %d\n", result, intIfNum, queue);
  }
 
  if (p) 
  {
    free(p);
  }

  return result;
}

/*****************************************************************************
* @purpose  Read UC Queue transmits
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    intIfNum       @b{(input)}  interface number
* @param    queue          @b{(input)}  queue number
* @param    numQueues      @b{(input)}  number of CoS queues
* 
* @returns  none
* 
* @notes
* 
* @end
******************************************************************************/
open_error_t readUCQueueTx(openapiClientHandle_t *clientHandle,
                           uint32_t intIfNum,
                           uint32_t queue,
                           uint32_t numQueues)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc counts;
  uint64_t *p = (uint64_t *) 0;
  uint32_t countsRead;
  int i;

  p = (uint64_t *) calloc(numQueues, sizeof(*p));
  if (!p)
  {
    printf("Unable to allocate buffer of %u bytes\n", numQueues * sizeof(*p));
    exit(1);
  }

  counts.pstart = p;
  counts.size = numQueues * sizeof(*p);

  result = openapiCosQueueUcTxPacketsGet(clientHandle,
                                         intIfNum,
                                         queue,
                                         &counts,
                                         &countsRead);

  if (OPEN_E_NONE == result)
  {
    for (i = 0; i < countsRead; i++)
    {
      if (queue != OPEN_COS_ALL_QUEUES)
      {
        printf("UC Queue TX interface %u queue %u: %llu\n", intIfNum, queue, p[i]);
      }
      else
      {
        printf("UC Queue TX interface %u queue %d: %llu\n", intIfNum, i, p[i]);
      }
    }
  }
  else
  {
    printf("FAILED (%d) UC Queue TX interface %u queue %d\n", result, intIfNum, queue);
  }

  if (p)
  {
    free(p);
  }

  return result;
}

/*****************************************************************************
* @purpose  Read CPU Queue transmits
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    queue          @b{(input)}  queue number
* @param    numQueues      @b{(input)}  number of CoS queues
* 
* @returns  none
* 
* @notes
* 
* @end
******************************************************************************/
open_error_t readCPUQueueTx(openapiClientHandle_t *clientHandle,
                           uint32_t queue,
                           uint32_t numQueues)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc counts;
  uint64_t *p = (uint64_t *) 0;
  uint32_t countsRead;
  int i;

  p = (uint64_t *) calloc(numQueues, sizeof(*p));
  if (!p)
  {
    printf("Unable to allocate buffer of %u bytes\n", numQueues * sizeof(*p));
    exit(1);
  }

  counts.pstart = p;
  counts.size = numQueues * sizeof(*p);

  result = openapiCosQueueCpuTxPacketsGet(clientHandle,
                                          queue,
                                          &counts,
                                          &countsRead);

  if (OPEN_E_NONE == result)
  {
    for (i = 0; i < countsRead; i++)
    {
      if (queue != OPEN_COS_ALL_QUEUES)
      {
        printf("CPU Queue TX queue %u: %llu\n", queue, p[i]);
      }
      else
      {
        printf("CPU Queue TX queue %d: %llu\n", i, p[i]);
      }
    }
  }
  else
  {
    printf("FAILED (%d) CPU Queue TX queue %d\n", result, queue);
  }

  if (p)
  {
    free(p);
  }

  return result;
}

/*****************************************************************************
* @purpose  Read CPU Queue drops
*
* @param    clientHandle   @b{(input)}  client handle from registration API
* @param    queue          @b{(input)}  queue number
* @param    numQueues      @b{(input)}  number of CoS queues
* 
* @returns  none
* 
* @notes
* 
* @end
******************************************************************************/
open_error_t readCPUQueueDrops(openapiClientHandle_t *clientHandle,
                               uint32_t queue,
                               uint32_t numQueues)
{
  open_error_t result = OPEN_E_NONE;
  open_buffdesc counts;
  uint64_t *p = (uint64_t *) 0;
  uint32_t countsRead;
  int i;

  p = (uint64_t *) calloc(numQueues, sizeof(*p));
  if (!p)
  {
    printf("Unable to allocate buffer of %u bytes\n", numQueues * sizeof(*p));
    exit(1);
  }

  counts.pstart = p;
  counts.size = numQueues * sizeof(*p);

  result = openapiCosQueueCpuDropPacketsGet(clientHandle,
                                            queue,
                                            &counts,
                                            &countsRead);

  if (OPEN_E_NONE == result)
  {
    for (i = 0; i < countsRead; i++)
    {
      if (queue != OPEN_COS_ALL_QUEUES)
      {
        printf("CPU Queue drops queue %u: %llu\n", queue, p[i]);
      }
      else
      {
        printf("CPU Queue drops queue %d: %llu\n", i, p[i]);
      }
    }
  }
  else
  {
    printf("FAILED (%d) CPU Queue drops queue %d\n", result, queue);
  }

  if (p)
  {
    free(p);
  }

  return result;
}

/********************************************************************
*
* @brief  This is the main function that will demonstrate 
*         CoS queue OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result = OPEN_E_NONE;
  uint32_t numQueues;
  int i;
  uint32_t cpuIntf;
  uint32_t ifNum;

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("cos_stats_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf ("Failed to initialize RPC to OpEN. result = %d exit test\n", result);
    exit (2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
  {
    sleep (1);
  }

  result = openapiCosQueueNumQueuesGet(&clientHandle, 1, &numQueues);

  if (result != OPEN_E_NONE) 
  {
    printf ("openapiCosQueueNumQueuesGet failed %d\n", result);
    exit (2);
  }

  printf("numQueues is %d\n", numQueues);

  /* following code reads all the queues for an interface and displays */
  ifNum = 0;
  if (openapiIfFirstGet(&clientHandle, OPEN_INTF_TYPE_PHY, &ifNum) == OPEN_E_NONE)
  {
      do
      {
        printf("UC Queue Transmits for interface %d\n", ifNum);
        printf("*********************************\n");
        readUCQueueTx(&clientHandle, ifNum, OPEN_COS_ALL_QUEUES, numQueues);
        printf("MC Queue Transmits for interface %d\n", ifNum);
        printf("*********************************\n");
        readMCQueueTx(&clientHandle, ifNum, OPEN_COS_ALL_QUEUES, numQueues);
      } while (openapiIfNextGet(&clientHandle, OPEN_INTF_TYPE_PHY, ifNum, &ifNum) == OPEN_E_NONE);
  }

  /* now read them one-by-one */

  ifNum = 0;
  if (openapiIfFirstGet(&clientHandle, OPEN_INTF_TYPE_PHY, &ifNum) == OPEN_E_NONE)
  {
      do
      {
        for (i = 0; i < numQueues; i++) 
        {
          readUCQueueTx(&clientHandle, ifNum, i, 1);
        }
        for (i = 0; i < numQueues; i++) 
        {
          readMCQueueTx(&clientHandle, ifNum, i, 1);
        }
      } while (openapiIfNextGet(&clientHandle, OPEN_INTF_TYPE_PHY, ifNum, &ifNum) == OPEN_E_NONE);
  }

  /* Now do the same for CPU queues */

  readCPUQueueTx(&clientHandle, OPEN_COS_ALL_QUEUES, numQueues);
  readCPUQueueDrops(&clientHandle, OPEN_COS_ALL_QUEUES, numQueues);

  result = openapiCpuIntfGet(&clientHandle, &cpuIntf);
  if (result != OPEN_E_NONE)
  {
    printf ("openapiCosQueueNumQueuesGet failed %d\n", result);
    exit (2);
  }  

  result = openapiCosQueueNumQueuesGet(&clientHandle, cpuIntf, &numQueues);

  if (result != OPEN_E_NONE) 
  {
    printf ("openapiCosQueueNumQueuesGet (for CPU) failed %d\n", result);
    exit (2);
  }

  for (i = 0; i < numQueues; i++) 
  {
    readCPUQueueTx(&clientHandle, i, 1);
  }

  /* Finally CPU drops. For front panel port drop counts per queue, see BST 
     example in the instrumentation OpEN APIs */

  for (i = 0; i < numQueues; i++) 
  {
    readCPUQueueDrops(&clientHandle, i, 1);
  }

  return 0;
}

