/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  registry_example.c
*
* @purpose   Example application to retrieve System Registry information using OpEN APIs
*
* @component OpEN
*
* @comments
*
* @create    01/11/2015
*
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_registry.h"

/*********************************************************************
* @purpose  Sanity test system registry parameters  
* 
* @param    clientHandle     @b{(input)}   client handle from registration API
* 
* @returns  none
*   
* @notes
* 
* @end
*********************************************************************/
void systemRegistrySanityTest (openapiClientHandle_t *clientHandle)
{
  open_error_t result;
  open_buffdesc buffdesc;
  char *str = NULL;

  if ((str = (char *)malloc(OPEN_REGISTRY_PARAM_SIZE)) == NULL)
  {
    printf("Could not allocate memory.\n"); 
    return;
  }

  buffdesc.pstart = str;
  buffdesc.size = OPEN_REGISTRY_PARAM_SIZE;

  if ((result = openapiSystemRegSerialNumGet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get serial number. (result = %d)\n", result);
  }
  else
  {
    printf("Serial number retrieved successfully. serial_num=%s size=%d (result = %d)\n",
            (char *)buffdesc.pstart, buffdesc.size, result);
  }

  buffdesc.size = OPEN_REGISTRY_PARAM_SIZE;
  if ((result = openapiSystemRegMachineModelGet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get model number. (result = %d)\n", result);
  }
  else
  {
    printf("Model number retrieved successfully. model_num=%s size=%d (result = %d)\n",
            (char *)buffdesc.pstart, buffdesc.size, result);
  }

  buffdesc.size = OPEN_REGISTRY_PARAM_SIZE;
  if ((result = openapiSystemRegMachineTypeGet (clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get model type. (result = %d)\n", result);
  }
  else
  {
    printf("Model type retrieved successfully. machine_type=\"%s\" size=%d (result = %d)\n",
            (char *)buffdesc.pstart, buffdesc.size, result);
  }

  buffdesc.size = OPEN_REGISTRY_PARAM_SIZE;
  if ((result = openapiSystemRegCodeVersionGet(clientHandle, &buffdesc) != OPEN_E_NONE))
  {
    printf("Bad return code trying to get code version. (result = %d)\n", result);
  }
  else
  {
    printf("Code version retrieved successfully. code_version=\"%s\" size=%d (result = %d)\n",
            (char *)buffdesc.pstart, buffdesc.size, result);
  }

  free(str);
  return;
}

/*******************************************************************
*
* @brief  This is the main() function of the example application that
*         demonstrates OpEN APIs for user configuration.
*
* @returns  0: Success
* @returns  1: Failure if the number of arguments are incorrect
* @returns  2: Other internal failure
*
*********************************************************************/
int main(int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((result = openapiClientRegister("system_example", &clientHandle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&clientHandle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Starting System API example application");

  printf("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof(switch_os_revision_string);
  if (openapiNetworkOSVersionGet(&clientHandle, &switch_os_revision) == OPEN_E_NONE)
    printf("Network OS version = %s\n", switch_os_revision_string);
  else
    printf("Network OS version retrieve error\n");

  printf("\n");

  systemRegistrySanityTest (&clientHandle);

  /* Log goodbye message with OpEN */
  L7PROC_LOGF(L7PROC_LOG_SEVERITY_INFO, 0, "Stopping System API example application");
        
  (void) openapiClientTearDown(&clientHandle);        
  return 0;
}
