/*********************************************************************
*
* Copyright 2017-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
**********************************************************************
*
* @filename  sdm_example.c
*
* @purpose   SDM Prefer APIs Example.
*
* @component OPEN
*
* @comments
*
* @create    1/03/2017
*
* @end
*
**********************************************************************/
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <getopt.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_sdm.h"

static openapiClientHandle_t client_handle;

static char *sdmTemplateEnumsAsText[] = {
  "OPEN_SDM_TEMPLATE_NONE",
  "OPEN_SDM_TEMPLATE_DUAL_DEFAULT",
  "OPEN_SDM_TEMPLATE_V4_DEFAULT",
  "OPEN_SDM_TEMPLATE_V4_DATA_CENTER",
  "OPEN_SDM_TEMPLATE_DATA_CENTER_PLUS",
  "OPEN_SDM_TEMPLATE_DUAL_DATA_CENTER",
  "OPEN_SDM_TEMPLATE_MPLS_DUAL_DATA_CENTER",
  "OPEN_SDM_TEMPLATE_DCVPN_DUAL_DATA_CENTER",
  "OPEN_SDM_TEMPLATE_DCVPN_V4_DATA_CENTER",
  "OPEN_SDM_TEMPLATE_ALPM",
  "OPEN_SDM_TEMPLATE_ALPM_MPLS_DUAL_DATA_CENTER"
};

/*********************************************************************
* @purpose  Dump list of all SDM template IDs
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void doListAll(void)
{
  OPEN_SDM_TEMPLATE_ID_t i;
  char tname[OPEN_SDM_TEMPLATE_NAME_LEN];
  open_buffdesc nameBuf;
  open_error_t rc;

  printf("All SDM template IDs\n");

  nameBuf.pstart = tname;

  for (i = OPEN_SDM_TEMPLATE_DUAL_DEFAULT; i <= OPEN_SDM_TEMPLATE_MAX; i++)
  {
    nameBuf.size = sizeof(tname);
    rc = openapiSdmTemplateNameGet(&client_handle, i, &nameBuf);
    if (rc != OPEN_E_NONE)
    {
			if (rc != OPEN_E_NOT_FOUND)
			{
        printf("Error getting name for template id %d\n", i);
			}
      tname[0] = '\0';
    }
    printf("%d %s (%s)\n", i, sdmTemplateEnumsAsText[i], tname);
  }
}

/*********************************************************************
* @purpose  Dump list of all supported SDM template IDs
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void doListAllSupported(void)
{
  OPEN_BOOL_t supported;
  open_error_t rc;
  OPEN_SDM_TEMPLATE_ID_t i;
  char tname[OPEN_SDM_TEMPLATE_NAME_LEN];
  open_buffdesc nameBuf;

  printf("All Supported SDM template IDs\n");

  nameBuf.pstart = tname;

  for (i = OPEN_SDM_TEMPLATE_DUAL_DEFAULT; i <= OPEN_SDM_TEMPLATE_MAX; i++)
  {
    rc = openapiSdmTemplateSupported(&client_handle, i, &supported);
    if (rc != OPEN_E_NONE)
    {
      printf("openapiSdmTemplateSupported failed (%d)\n", rc);    
    }
    else if (supported == OPEN_TRUE)
    {
      nameBuf.size = sizeof(tname);
      rc = openapiSdmTemplateNameGet(&client_handle, i, &nameBuf);
      if (rc != OPEN_E_NONE)
      {
        printf("Error getting name for template id %d\n", i);
        tname[0] = '\0';
      }
      printf("%i %s (%s)\n", i, sdmTemplateEnumsAsText[i], tname);
    }
  }
}

/*********************************************************************
* @purpose  Dump SDM template 
*
* @param    id    @b{(input)}   template ID
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void dumpSdmTemplate(const OPEN_SDM_TEMPLATE_ID_t id)
{
  char name[OPEN_SDM_TEMPLATE_NAME_LEN];
  open_buffdesc nameBuf;
  open_error_t rc;
  uint32_t val;

  nameBuf.pstart = name;
  nameBuf.size = sizeof(name);

  rc = openapiSdmTemplateNameGet(&client_handle, id, &nameBuf);
  if (rc == OPEN_E_NONE)
  {
    printf("Template %s (%s)\n", sdmTemplateEnumsAsText[id], name);
    rc = openapiSdmTemplateMaxArpEntriesGet(&client_handle, id, &val);
    if (rc != OPEN_E_NONE)
    {
      printf("openapiSdmTemplateMaxArpEntriesGet failed (%d)\n", rc);    
    }
    else
    {
      printf("ARP Entries.................................... %d\n", val);
    }
    rc = openapiSdmTemplateMaxIpv4RoutesGet(&client_handle, id, &val);
    if (rc != OPEN_E_NONE)
    {
      printf("openapiSdmTemplateMaxIpv4RoutesGet failed (%d)\n", rc);    
    }
    else
    {
      printf("IPv4 Unicast Routes............................ %d\n", val);
    }
    rc = openapiSdmTemplateMaxNdpEntriesGet(&client_handle, id, &val);
    if (rc != OPEN_E_NONE)
    {
      printf("openapiSdmTemplateMaxNdpEntriesGet failed (%d)\n", rc);    
    }
    else
    {
      printf("IPv6 NDP Entries............................... %d\n", val);
    }
    rc = openapiSdmTemplateMaxIpv6RoutesGet(&client_handle, id, &val);
    if (rc != OPEN_E_NONE)
    {
      printf("openapiSdmTemplateMaxIpv6RoutesGet failed (%d)\n", rc);    
    }
    else
    {
      printf("IPv6 Unicast Routes............................ %d\n", val);
    }
    rc = openapiSdmTemplateMaxEcmpNextHopsGet(&client_handle, id, &val);
    if (rc != OPEN_E_NONE)
    {
      printf("openapiSdmTemplateMaxEcmpNextHopsGet failed (%d)\n", rc);    
    }
    else
    {
      printf("ECMP Next Hops................................. %d\n", val);
    }
    rc = openapiSdmTemplateMaxIpv4McastRoutesGet(&client_handle, id, &val);
    if (rc != OPEN_E_NONE)
    {
      printf("openapiSdmTemplateMaxIpv4McastRoutesGet failed (%d)\n", rc);    
    }
    else
    {
      printf("IPv4 Multicast Routes.......................... %d\n", val);
    }
    rc = openapiSdmTemplateMaxIpv6McastRoutesGet(&client_handle, id, &val);
    if (rc != OPEN_E_NONE)
    {
      printf("openapiSdmTemplateMaxIpv6McastRoutesGet failed (%d)\n", rc);    
    }
    else
    {
      printf("IPv6 Multicast Routes.......................... %d\n", val);
    }
  }
  else 
  {
    printf("openapiSdmTemplateNameGet failed (%d)\n", rc);    
  }
}

/*********************************************************************
* @purpose  Show active SDM template
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void doShowActive(void)
{
  OPEN_SDM_TEMPLATE_ID_t active;
  open_error_t rc;

  printf("Active SDM template\n");
  rc = openapiSdmActiveTemplateGet(&client_handle, &active);
  if (rc != OPEN_E_NONE)
  {
    printf("openapiSdmActiveTemplateGet failed (%d)\n", rc);    
  }
  else
  {
    dumpSdmTemplate(active);
  }
}

/*********************************************************************
* @purpose  Show next active SDM template 
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void doShowNextActive(void)
{
  OPEN_SDM_TEMPLATE_ID_t nextActive;
  open_error_t rc;

  printf("Next Active SDM template\n");
  rc = openapiSdmNextActiveTemplateGet(&client_handle, &nextActive);
  if (rc != OPEN_E_NONE)
  {
    printf("openapiSdmNextActiveTemplateGet failed (%d)\n", rc);
  }
  else
  {
    dumpSdmTemplate(nextActive);
  }
}

/*********************************************************************
* @purpose  Show SDM template with given ID
*
* @param    id             @b{(input)}   template ID
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void doShowId(const OPEN_SDM_TEMPLATE_ID_t id)
{
  dumpSdmTemplate(id);
}

/*********************************************************************
* @purpose  Get the SDM template ID for the specified name if it exists.
*
* @param    name             @b{(input)}   Template name search key
* @param    id               @b{(output)}  If function returns true, matching
*                                          template ID. Otherwise, undefined.
*
* @returns  true  if name exists as a template.
* @returns  false if name does not exist or some error occurs. 
*
* @notes
*
* @end
*********************************************************************/
bool findTemplateByName(const char *name, OPEN_SDM_TEMPLATE_ID_t *id)
{
  bool found = false;
  char tname[OPEN_SDM_TEMPLATE_NAME_LEN];
  open_buffdesc nameBuf;
  open_error_t rc;
  OPEN_SDM_TEMPLATE_ID_t i;
 
  nameBuf.pstart = tname;

  for (i = OPEN_SDM_TEMPLATE_DUAL_DEFAULT; i <= OPEN_SDM_TEMPLATE_MAX; i++)
  {
    nameBuf.size = sizeof(tname);
    rc = openapiSdmTemplateNameGet(&client_handle, i, &nameBuf);
    if (rc == OPEN_E_NONE)
    {
      if (!strncmp(name, tname, strlen(name)))
      {
        *id = i;
        found = true;
        break;
      }
    }
  }
  return found;
}

/*********************************************************************
* @purpose  Show SDM template for the specified template name.
*
* @param    name             @b{(input)}   template name
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void doShowName(const char *name)
{
  OPEN_SDM_TEMPLATE_ID_t id;
  
  if (findTemplateByName(name, &id) == true)
  {
    dumpSdmTemplate(id);
  }
}

/*********************************************************************
* @purpose  Set the next SDM template based on ID.
*
* @param    nextActiveId    @b{(input)}   Id of template to make next
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void doSetNextActiveId(const int nextActiveId)
{
  open_error_t rc;

  rc = openapiSdmNextActiveTemplateSet(&client_handle, nextActiveId);
  if (rc != OPEN_E_NONE)
  {
    printf("openapiSdmNextActiveTemplateSet failed (%d)\n", rc); 
  }
}

/*********************************************************************
* @purpose  Set the next SDM template based on template name.
*
* @param    nextActiveName   @b{(input)}    Name of template to make next 
*
* @returns  none
*
* @notes
*
* @end
*********************************************************************/
void doSetNextActiveName(const char *nextActiveName)
{
  OPEN_SDM_TEMPLATE_ID_t id;
  
  if (findTemplateByName(nextActiveName, &id) == true)
  {
    doSetNextActiveId(id);
  }
}

void print_usage(const char *name)
{
  printf("%s <[-l][-s][-a][-n][-i id][-t name][-p id][-q name]>\n\n", name);
  printf("-l -- list all template IDs and names\n");
  printf("-s -- list all supported template IDs and names\n");
  printf("-a -- show active template ID and name\n");
  printf("-n -- show next active template ID and name\n");
  printf("-i id -- display data for template with given id\n");
  printf("-t name -- display data for template with given name\n");
  printf("-p id -- make specified template id next active.\n");
  printf("-q name -- make specified template id next active.\n");
  exit(0);
}

int main(int argc, char *argv[])
{
  int option = 0;
  int rc;
  bool listAll = false;
  bool listAllSupported = false;
  bool showActive = false;
  bool showNextActive = false;
  bool showId = false;
  int id;
  bool showName = false;
  char name[OPEN_SDM_TEMPLATE_NAME_LEN]; // size includes terminating '\0'
  bool setNextActiveId = false;
  bool setNextActiveName = false;
  int nextActiveId;
  char nextActiveName[OPEN_SDM_TEMPLATE_NAME_LEN]; // size includes terminating '\0'
	bool optionSpecified = false;

  while ((option = getopt(argc, argv,"lsani:t:p:q:")) != -1) {
    switch (option) {
      case 'l' : 
        listAll = true;
				optionSpecified = true;
        break;
      case 's' : 
        listAllSupported = true;
				optionSpecified = true;
        break;
      case 'a' : 
        showActive = true;
				optionSpecified = true;
        break;
      case 'n' : 
        showNextActive = true;
				optionSpecified = true;
        break;
      case 'i' : 
        showId = true;
        id = atoi(optarg); 
				optionSpecified = true;
        break;
      case 't' : 
        showName = true;
        strncpy(name, optarg, sizeof(name));
				optionSpecified = true;
        break;
      case 'p' : 
        setNextActiveId = true;
        nextActiveId = atoi(optarg); 
				optionSpecified = true;
        break;
      case 'q' : 
        setNextActiveName = true;
        strncpy(nextActiveName, optarg, sizeof(nextActiveName));
				optionSpecified = true;
        break;
      default: 
        print_usage(argv[0]); 
        break;
    }
  }

  l7proc_crashlog_register();

  /* Register with OpEN */
  if ((rc = openapiClientRegister("sdm_example", &client_handle)) != OPEN_E_NONE)
  {
    printf("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", rc);
    exit(2);
  }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck(&client_handle) != OPEN_E_NONE)
  {
    sleep(1);
  }

  if (listAll == true)
  {
    doListAll();
  } 

  if (listAllSupported == true)
  {
    doListAllSupported();
  } 

  if (showActive == true)
  {
    doShowActive();
  } 

  if (showNextActive == true)
  {
    doShowNextActive();
  } 

  if (showId == true)
  {
    doShowId(id);
  } 

  if (showName == true)
  {
    doShowName(name);
  } 

  if (setNextActiveId == true)
  {
    doSetNextActiveId(nextActiveId);
  } 

  if (setNextActiveName == true)
  {
    doSetNextActiveName(nextActiveName);
  } 

	if (optionSpecified == false)
	{
		printf("usage: ");
		print_usage(argv[0]); 
	}

  return 0;
}


