/*********************************************************************
*
* Copyright 2016-2018 Broadcom.
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*
***********************************************************************
*
* @filename tcam_example.c
*
* @purpose TCAM API example
*
* @detail  Example application that uses the OpEN API to insert
*          various policies in the TCAM.
*
* @component TCAM API Example
*
* @comments none
* @end
*
**********************************************************************/
#include <stdlib.h>
#include <unistd.h>

#include "rpcclt_openapi.h"
#include "proc_util.h"
#include "openapi_common.h"
#include "openapi_tcam.h"

/** Used for maximum string length on various variables */
#define TCAM_MAX_STR_LENGTH 256


void printMenu()
{
  printf("\nEnter 1 Initialize/reset Policy \n"
           "      2 to Add classifiers to Policy \n"
           "      3 to Add actions to Policy\n"
           "      4 to create the Policy\n"
           "      5 to add intfs     \n"
           "      6 to delete intfs   \n"
           "      7 to destroy the Policy \n"
           "      8 to Get            \n"
           "      9 to Delete all existing TCAM policies\n"
           "     10 to Add policy on all interfaces\n"
           "     11 to Get Policy Stats\n"
           "     12 to Clear Policy Stats\n"
           "     13 Initialize/reset version 2 Policy \n"
           "     14 to Add version 2 classifiers to Policy \n"
           "     15 to Add actions to Policy\n"
           "     16 to create version 2 Policy\n"
           "     17 to quit \n");
}

static open_error_t getIpAddrFromStr(int af, char *strIpAddr, open_inet_addr_t *addr)
{
  struct sockaddr_in sa;

/* read the ip address */
  if (af == AF_INET)
  {
    if (inet_pton(af,strIpAddr,&(sa.sin_addr)) > 0)
    {
      addr->addr.ipv4 = ntohl(sa.sin_addr.s_addr);
      return OPEN_E_NONE;
    }
  } 
  else if (af == AF_INET6)
  {
    if (inet_pton(AF_INET6, strIpAddr, (void*)&(addr->addr.ipv6)) > 0)
    {
      return OPEN_E_NONE;
    }


  }
  return OPEN_E_PARAM;
}

static open_error_t getMacAddrFromInput (char * input_mac, unsigned char * mac)
{
  int i;
  unsigned char c;
  unsigned char temp_mac [12];

  for ( i = 0; i < 12; i++ )
  {
    c = input_mac [i];
    if ( (c >= '0') && (c <= '9') )
      temp_mac[i] = c - '0';
    else if ( (c >= 'a') && (c <= 'f') )
      temp_mac[i] = c - 'a' + 10;
    else if ( (c >= 'A') && (c <= 'F') )
      temp_mac[i] = c - 'A' + 10;
    else return(1);   /* Invalid character */
  };

  for ( i = 0; i < 12; i+= 2 )
  {
    mac[i/2] = (temp_mac[i] * 16) + temp_mac[i+1];
  };
  return(0);
};

/** Global used for holding an interface string */
char intfString[100];
/** Global used for holding a redirect interface string */
char redirIntfString[100];

/*******************************************************************
*
* @brief  Example TCAM API: Build the header for the TCAM API policy.
*
*********************************************************************/
static void openBuildHeader(open_tcamApiPolicy_t *policyInfo)
{
  uint32_t prio;
  uint32_t policyType = 0;

  memset(policyInfo,0, sizeof(*policyInfo));

  policyInfo->policyHeader.versionInfo.versionMajor =OPEN_TCAM_API_CURR_VER_MAJOR;
  policyInfo->policyHeader.versionInfo.versionMinor = OPEN_TCAM_API_CURR_VER_MINOR;
  policyInfo->policyHeader.headerLen = sizeof(policyInfo->policyHeader);
  printf("Enter policy priority \n");
  scanf("%d", &prio);
  policyInfo->policyPrio = prio;

  printf("Select policy type 1 for openflow 2 for Gen (IPv6) 3 for Egress \n");
  scanf("%d", &policyType);

  switch (policyType)
  {
    case 1:
      policyInfo->policyType =  OPEN_TCAM_POLICY_TYPE_OPENFLOW ;
      break;
    case 2:
      policyInfo->policyType =  OPEN_TCAM_POLICY_TYPE_GEN ;
      break;
    case 3:
      policyInfo->policyType =  OPEN_TCAM_POLICY_TYPE_EGRESS ;
      break;
  }


}

static void openBuildHeaderV2(open_tcamApiPolicy_v2_t *policyInfo)
{
  uint32_t prio;
  uint32_t policyType = 0;

  memset(policyInfo,0, sizeof(*policyInfo));

  policyInfo->policyHeader.versionInfo.versionMajor =OPEN_TCAM_API_CURR_VER_MAJOR;
  policyInfo->policyHeader.versionInfo.versionMinor = OPEN_TCAM_API_CURR_VER_MINOR;
  policyInfo->policyHeader.headerLen = sizeof(policyInfo->policyHeader);
  printf("Enter policy priority \n");
  scanf("%d", &prio);
  policyInfo->policyPrio = prio;

  printf("Select policy type 1 for openflow, 2 for Gen (IPv6), 3 for Egress, 4 for Port \n");
  scanf("%d", &policyType);

  switch (policyType)
  {
    case 1:
      policyInfo->policyType =  OPEN_TCAM_POLICY_TYPE_OPENFLOW ;
      break;
    case 2:
      policyInfo->policyType =  OPEN_TCAM_POLICY_TYPE_GEN ;
      break;
    case 3:
      policyInfo->policyType =  OPEN_TCAM_POLICY_TYPE_EGRESS ;
      break;
    case 4:
      policyInfo->policyType =  OPEN_TCAM_POLICY_TYPE_PORT ;
      break;
  }
}

/*******************************************************************
*
* @brief  Example TCAM API: Add the classifiers.
*
*********************************************************************/
static void openAddClassifiers(open_tcamApiPolicy_t *policyInfo)
{
  char mac[TCAM_MAX_STR_LENGTH];
  uint32_t response;
  uint32_t value;
  char strSubnetMask[TCAM_MAX_STR_LENGTH];
  char strIpaddr[TCAM_MAX_STR_LENGTH];
  open_inet_addr_t addr;

  printf("This selection will add multiple classifiers to the policy\n");

  do
  {
    response = 0;
    printf("Enter classification type \n"
           "1 to match dest MAC\n"
           "2 to match Src MAC\n"
           "3 to match VLAN\n"
           "4 to match VLAN Priority\n"
           "5 to match ether type\n"
           "6 to match dest IP address\n"
           "7 to match src IP address \n"
           "8 to match dest IPv6 address\n"
           "9 to match src IPv6 address\n"
           "10 quit \n");
    scanf("%d", &response);
  /* Classification*/
    switch (response)
    {
      case 1:
        memset(mac, 0, sizeof(mac));
        printf("Enter Mac address\n");
        scanf("%s", mac);
        
        if (getMacAddrFromInput(mac, policyInfo->matchDstMac))
          printf("Error: Invalid input\n");
        break;
      case 2:
        memset(mac, 0, sizeof(mac));
        printf("Enter Mac address\n");
        scanf("%s", mac);
        if (getMacAddrFromInput(mac, policyInfo->matchSrcMac))
          printf("Error: Invalid input\n");
        break;
      case 3:
        printf("Enter VLAN id \n");
        scanf("%d",&value);
        policyInfo->matchVlanVal = value;
        break;
      case 4:
        printf("Enter VLAN priority \n");
        scanf("%d",&value);
        policyInfo->matchVlanPrio = value;
        break;
      case 5:
        printf("Enter ether type in hex\n");
        scanf("%x",&value);
        policyInfo->matchEthType = value;
        break;
      case 6:
        printf("Enter Dest IP address \n");
        scanf("%s",strIpaddr);
        if (getIpAddrFromStr(AF_INET, strIpaddr, &addr) == OPEN_E_NONE)
        {
          policyInfo->matchDstIp = addr.addr.ipv4;
          printf("Enter Mask \n");
          scanf("%s",strSubnetMask);
          if (getIpAddrFromStr(AF_INET, strSubnetMask, &addr) == OPEN_E_NONE)
          {
            policyInfo->matchDstIpMask = addr.addr.ipv4;
            break;
          }

        }

        printf("Error: Incorrect Input\n");
        break;
      case 7:
        printf("Enter Src IP address \n");
        scanf("%s",strIpaddr);
        if (getIpAddrFromStr(AF_INET, strIpaddr, &addr) == OPEN_E_NONE)
        {
          policyInfo->matchSrcIp = addr.addr.ipv4;
          printf("Enter Mask \n");
          scanf("%s",strSubnetMask);
          if (getIpAddrFromStr(AF_INET, strSubnetMask, &addr) == OPEN_E_NONE)
          {
            policyInfo->matchSrcIpMask = addr.addr.ipv4;
            break;
          }

        }
        printf("Error: Incorrect Input\n");
        break;
      case 8:
        printf("Enter Dest IPv6 address \n");
        scanf("%s",strIpaddr);
        if (getIpAddrFromStr(AF_INET6, strIpaddr, &addr) == OPEN_E_NONE)
        {
          memcpy(&policyInfo->matchDstIpv6Addr, &addr.addr.ipv6, 
                   sizeof(policyInfo->matchDstIpv6Addr));
          printf("Enter Mask \n");
          scanf("%s",strSubnetMask);
          if (getIpAddrFromStr(AF_INET6, strSubnetMask, &addr) == OPEN_E_NONE)
          {
            memcpy(&policyInfo->matchDstIpv6Mask, &addr.addr.ipv6, 
                   sizeof(policyInfo->matchDstIpv6Mask));
            break;
          }

        }
        printf("Error: Incorrect Input\n");
        break;
      case 9:
        printf("Enter Src IPv6 address \n");
        scanf("%s",strIpaddr);
        if (getIpAddrFromStr(AF_INET6, strIpaddr, &addr) == OPEN_E_NONE)
        {
          memcpy(&policyInfo->matchSrcIpv6Addr, &addr.addr.ipv6, 
                   sizeof(policyInfo->matchSrcIpv6Addr));
          printf("Enter Mask \n");
          scanf("%s",strSubnetMask);
          if (getIpAddrFromStr(AF_INET6, strSubnetMask, &addr) == OPEN_E_NONE)
          {
            memcpy(&policyInfo->matchSrcIpv6Mask, &addr.addr.ipv6, 
                   sizeof(policyInfo->matchSrcIpv6Mask));
            break;
          }

        }
        printf("Error: Incorrect Input\n");
        break;
      case 10:
      default:
        return;

    }
  } while (response != 10);

}

static void openAddClassifiersV2(open_tcamApiPolicy_v2_t *policyInfo)
{
  char mac[TCAM_MAX_STR_LENGTH];
  uint32_t response;
  uint32_t value;
  uint32_t mask;
  char strSubnetMask[TCAM_MAX_STR_LENGTH];
  char strIpaddr[TCAM_MAX_STR_LENGTH];
  open_inet_addr_t addr;

  printf("This selection will add multiple classifiers to the policy\n");

  do
  {
    response = 0;
    printf("Enter classification type \n"
           "1 to match dest MAC\n"
           "2 to match Src MAC\n"
           "3 to match VLAN\n"
           "4 to match VLAN Priority\n"
           "5 to match ether type\n"
           "6 to match dest IP address\n"
           "7 to match src IP address \n"
           "8 to match dest IPv6 address\n"
           "9 to match src IPv6 address\n"
           "10 to match CPU visibility \n"
           "11 quit \n");
    scanf("%d", &response);
  /* Classification*/
    switch (response)
    {
      case 1:
        memset(mac, 0, sizeof(mac));
        printf("Enter Mac address\n");
        scanf("%s", mac);
        
        if (getMacAddrFromInput(mac, policyInfo->matchDstMac))
          printf("Error: Invalid input\n");
        break;

      case 2:
        memset(mac, 0, sizeof(mac));
        printf("Enter Mac address\n");
        scanf("%s", mac);
        if (getMacAddrFromInput(mac, policyInfo->matchSrcMac))
          printf("Error: Invalid input\n");
        break;

      case 3:
        printf("Enter VLAN id \n");
        scanf("%d",&value);
        policyInfo->matchVlanVal = value;
        break;

      case 4:
        printf("Enter VLAN priority \n");
        scanf("%d",&value);
        policyInfo->matchVlanPrio = value;
        break;

      case 5:
        printf("Enter ether type in hex\n");
        scanf("%x",&value);
        policyInfo->matchEthType = value;
        break;

      case 6:
        printf("Enter Dest IP address \n");
        scanf("%s",strIpaddr);
        if (getIpAddrFromStr(AF_INET, strIpaddr, &addr) == OPEN_E_NONE)
        {
          policyInfo->matchDstIp = addr.addr.ipv4;
          printf("Enter Mask \n");
          scanf("%s",strSubnetMask);
          if (getIpAddrFromStr(AF_INET, strSubnetMask, &addr) == OPEN_E_NONE)
          {
            policyInfo->matchDstIpMask = addr.addr.ipv4;
            break;
          }
        }
        printf("Error: Incorrect Input\n");
        break;

      case 7:
        printf("Enter Src IP address \n");
        scanf("%s",strIpaddr);
        if (getIpAddrFromStr(AF_INET, strIpaddr, &addr) == OPEN_E_NONE)
        {
          policyInfo->matchSrcIp = addr.addr.ipv4;
          printf("Enter Mask \n");
          scanf("%s",strSubnetMask);
          if (getIpAddrFromStr(AF_INET, strSubnetMask, &addr) == OPEN_E_NONE)
          {
            policyInfo->matchSrcIpMask = addr.addr.ipv4;
            break;
          }
        }
        printf("Error: Incorrect Input\n");
        break;

      case 8:
        printf("Enter Dest IPv6 address \n");
        scanf("%s",strIpaddr);
        if (getIpAddrFromStr(AF_INET6, strIpaddr, &addr) == OPEN_E_NONE)
        {
          memcpy(&policyInfo->matchDstIpv6Addr, &addr.addr.ipv6, 
                   sizeof(policyInfo->matchDstIpv6Addr));
          printf("Enter Mask \n");
          scanf("%s",strSubnetMask);
          if (getIpAddrFromStr(AF_INET6, strSubnetMask, &addr) == OPEN_E_NONE)
          {
            memcpy(&policyInfo->matchDstIpv6Mask, &addr.addr.ipv6, 
                   sizeof(policyInfo->matchDstIpv6Mask));
            break;
          }
        }
        printf("Error: Incorrect Input\n");
        break;

      case 9:
        printf("Enter Src IPv6 address \n");
        scanf("%s",strIpaddr);
        if (getIpAddrFromStr(AF_INET6, strIpaddr, &addr) == OPEN_E_NONE)
        {
          memcpy(&policyInfo->matchSrcIpv6Addr, &addr.addr.ipv6, 
                   sizeof(policyInfo->matchSrcIpv6Addr));
          printf("Enter Mask \n");
          scanf("%s",strSubnetMask);
          if (getIpAddrFromStr(AF_INET6, strSubnetMask, &addr) == OPEN_E_NONE)
          {
            memcpy(&policyInfo->matchSrcIpv6Mask, &addr.addr.ipv6, 
                   sizeof(policyInfo->matchSrcIpv6Mask));
            break;
          }
        }
        printf("Error: Incorrect Input\n");
        break;

      case 10:
        printf("Enter CPU visibility qualifier value \n");
        scanf("%d",&value);
        if ((value != 0) &&
            (value != 1))
        {
          printf("Error: Incorrect Input, please enter either 0 or 1\n");
          break;
        }
        policyInfo->matchCpuVisibility = value;
        printf("Enter CPU visibility qualifier mask \n");
        scanf("%d",&mask);
        if ((mask != 0) &&
            (mask != 0xf))
        {
          printf("Error: Incorrect Input, please enter either 0 or 15\n");
          break;
        }
        policyInfo->matchCpuVisibilityMask = mask;
        break;

      case 11:
      default:
        return;

    }
  } while (response != 11);

}


/*******************************************************************
*
* @brief  Example TCAM API: Configure the policy.
*
*********************************************************************/

static void openAddPolicyAction(openapiClientHandle_t *clientHandle, open_tcamApiPolicy_t *policyInfo)
{
  char mac[TCAM_MAX_STR_LENGTH];
  uint32_t response;
  uint32_t value;
  uint32_t len;
  open_buffdesc buffDesc;

  printf("This selection will add multiple actions to the policy\n");
  do
  {
    response = 0;
    printf("Enter Action \n"
             "1 to add/change outer VLAN\n"
             "2 to set VLAN priority\n"
             "3 to set new src mac\n"
             "4 to set new dst mac\n"
             "5 to redirect the traffic to another port\n"
             "6 to change ether type\n"
             "7 to send packets to CPU\n"
             "8 to drop the packets\n"
             "10 quit \n");
      scanf("%d", &response);
   /* Action*/
    switch (response)
    {
      case 1:
        policyInfo->actionType |= OPEN_TCAM_ACTION_OUTERVLAN;
        printf("Enter VLAN ID \n");
        scanf("%d",&value);
        policyInfo->actionVal.vlanId = value;
        break;
      case 2:
        policyInfo->actionType |= OPEN_TCAM_ACTION_USERPRIO;
        printf("Enter VLAN priority\n");
        scanf("%d",&value);
        policyInfo->actionVal.vlanPrio = value;
        break;    
      case 3:
        memset(mac, 0, sizeof(mac));
        policyInfo->actionType |= OPEN_TCAM_ACTION_SRCMAC_NEW;
        printf("Enter Mac address\n");
        scanf("%s", mac);
        if (getMacAddrFromInput(mac, policyInfo->actionVal.srcAddr))
          printf("Invalid input\n");
        /*memcpy(policyInfo->actionVal.srcAddr, mac, TCAM_MAC_ADDR_LEN);*/
        break;           
      case 4:
        memset(mac, 0, sizeof(mac));
        policyInfo->actionType |= OPEN_TCAM_ACTION_DSTMAC_NEW;
        printf("Enter Mac address\n");
        scanf("%s", mac);
        if (getMacAddrFromInput(mac, policyInfo->actionVal.dstMac))
          printf("Error: Invalid input\n");
        /*memcpy(policyInfo->actionVal.dstMac, mac, TCAM_MAC_ADDR_LEN);*/
        break;           
      case 5:
        policyInfo->actionType |= OPEN_TCAM_ACTION_REDIRECT;
        printf("Enter intf string\n");
        scanf("%s", redirIntfString);
        len = strlen(redirIntfString);
       
        buffDesc.size = len < (sizeof(redirIntfString) -1) ? len : (sizeof(redirIntfString) -1);
        buffDesc.pstart =  redirIntfString;
        if (openapiIfNumGet(clientHandle,&buffDesc,
                            &policyInfo->actionVal.redirIntfNum) != OPEN_E_NONE)
        {
          printf("Error: Incorrect interface name %s \n", intfString);
        }
        
        break;
      case 6:
        policyInfo->actionType |= OPEN_TCAM_ACTION_OUTERTPID_NEW;
        printf("Enter ether type in hex\n");
        scanf("%x",&value);
        policyInfo->actionVal.etherType = value;
        break;
      case 7:
        policyInfo->actionType |= OPEN_TCAM_ACTION_REDIRECT_CPU;
        printf("Enter rule number\n");
        scanf("%d",&value);
        policyInfo->ruleNum = value;
        break;           
      case 8:
        policyInfo->actionType |= OPEN_TCAM_ACTION_DROP;
      case 10:
      default:
        return;
                  
    }
  } while (response != 10);
}

static void openAddPolicyActionV2(openapiClientHandle_t *clientHandle, open_tcamApiPolicy_v2_t *policyInfo)
{
  char mac[TCAM_MAX_STR_LENGTH];
  uint32_t response;
  uint32_t value;
  uint32_t len;
  open_buffdesc buffDesc;

  printf("This selection will add multiple actions to the policy\n");
  do
  {
    response = 0;
    printf("Enter Action \n"
             "1 to add/change outer VLAN\n"
             "2 to set VLAN priority\n"
             "3 to set new src mac\n"
             "4 to set new dst mac\n"
             "5 to redirect the traffic to another port\n"
             "6 to change ether type\n"
             "7 to send packets to CPU\n"
             "8 to drop the packets\n"
             "10 quit \n");
      scanf("%d", &response);
   /* Action*/
    switch (response)
    {
      case 1:
        policyInfo->actionType |= OPEN_TCAM_ACTION_OUTERVLAN;
        printf("Enter VLAN ID \n");
        scanf("%d",&value);
        policyInfo->actionVal.vlanId = value;
        break;
      case 2:
        policyInfo->actionType |= OPEN_TCAM_ACTION_USERPRIO;
        printf("Enter VLAN priority\n");
        scanf("%d",&value);
        policyInfo->actionVal.vlanPrio = value;
        break;    
      case 3:
        memset(mac, 0, sizeof(mac));
        policyInfo->actionType |= OPEN_TCAM_ACTION_SRCMAC_NEW;
        printf("Enter Mac address\n");
        scanf("%s", mac);
        if (getMacAddrFromInput(mac, policyInfo->actionVal.srcAddr))
          printf("Invalid input\n");
        /*memcpy(policyInfo->actionVal.srcAddr, mac, TCAM_MAC_ADDR_LEN);*/
        break;           
      case 4:
        memset(mac, 0, sizeof(mac));
        policyInfo->actionType |= OPEN_TCAM_ACTION_DSTMAC_NEW;
        printf("Enter Mac address\n");
        scanf("%s", mac);
        if (getMacAddrFromInput(mac, policyInfo->actionVal.dstMac))
          printf("Error: Invalid input\n");
        /*memcpy(policyInfo->actionVal.dstMac, mac, TCAM_MAC_ADDR_LEN);*/
        break;           
      case 5:
        policyInfo->actionType |= OPEN_TCAM_ACTION_REDIRECT;
        printf("Enter intf string\n");
        scanf("%s", redirIntfString);
        len = strlen(redirIntfString);
       
        buffDesc.size = len < (sizeof(redirIntfString) -1) ? len : (sizeof(redirIntfString) -1);
        buffDesc.pstart =  redirIntfString;
        if (openapiIfNumGet(clientHandle,&buffDesc,
                            &policyInfo->actionVal.redirIntfNum) != OPEN_E_NONE)
        {
          printf("Error: Incorrect interface name %s \n", intfString);
        }
        
        break;
      case 6:
        policyInfo->actionType |= OPEN_TCAM_ACTION_OUTERTPID_NEW;
        printf("Enter ether type in hex\n");
        scanf("%x",&value);
        policyInfo->actionVal.etherType = value;
        break;
      case 7:
        policyInfo->actionType |= OPEN_TCAM_ACTION_REDIRECT_CPU;
        printf("Enter rule number\n");
        scanf("%d",&value);
        policyInfo->ruleNum = value;
        break;           
      case 8:
        policyInfo->actionType |= OPEN_TCAM_ACTION_DROP;
      case 10:
      default:
        return;
                  
    }
  } while (response != 10);
}


/*******************************************************************
*
* @brief  Example TCAM API: Main menu.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int openTcamMainMenu(openapiClientHandle_t *clientHandle)
{
  open_tcamApiPolicy_t policyInfo;
  open_tcamApiPolicy_t getPolicyInfo;
  OPEN_TCAM_API_POLICYID_t policyHandle = OPEN_TCAM_API_POLICYID_INVALID;
  open_error_t errorCode;
  uint32_t action = 0, intfNum;
  uint32_t len;
  open_buffdesc buffDesc;
  uint64_t inProf = 0, outProf = 0;
  open_tcamApiPolicy_v2_t policyInfoV2;

  openapiTcamApiInit(clientHandle);

  do
  {
    printMenu();
    action = 0;
    scanf("%d",&action);
    switch (action)
    {
      /* initialize*/
      case 1:
        openBuildHeader(&policyInfo);
        /* Add classifiers*/
      case 2:
        openAddClassifiers(&policyInfo);
        /* add actions */
      case 3:
        openAddPolicyAction(clientHandle, &policyInfo);
        break;

        /* create*/
      case 4:
        {
          errorCode = openapiTcamPolicyCreate(clientHandle, policyInfo, &policyHandle);
          if (errorCode != OPEN_E_NONE)
          {
            printf("Error: Policy could not be created Reason %d", errorCode);
          }
          else
          {
            printf("Policy succesfully added,handle  is %d \n", policyHandle);
          }
        }
        break;
        /* Add/Remove interfaces*/
      case 5:
      case 6:
        if (policyHandle == OPEN_TCAM_API_POLICYID_INVALID)
        {
          printf("Enter policy Handle\n");
          scanf("%d", &policyHandle);
        }

          printf("Enter intf string\n");
          scanf("%s", intfString);
          len = strlen(intfString);
          buffDesc.size = len < (sizeof(intfString) -1) ? len : (sizeof(intfString) -1);
          buffDesc.pstart =  intfString;
          if (openapiIfNumGet(clientHandle,&buffDesc,&intfNum) != OPEN_E_NONE)
          {
            printf("Error: Incorrect interface name %s \n", intfString);
          }
          else
          {

            if (action == 22)
            {
              errorCode = openapiTcamPolicyIntfAdd(clientHandle, &policyHandle, intfNum);
            }
            else
            {
              errorCode = openapiTcamPolicyIntfRem(clientHandle, &policyHandle, intfNum);
            }

            if (errorCode != OPEN_E_NONE)
            {
              printf("Error: Policy could not be applied to intf(%d) %d", intfNum, errorCode);
            }
          }

        break;

         /* Destroy*/
      case 7:

        if (policyHandle == OPEN_TCAM_API_POLICYID_INVALID)
        {
          printf("Enter policy Handle\n");
          scanf("%d", &policyHandle);
        }

        if (policyHandle != OPEN_TCAM_API_POLICYID_INVALID)
        {
          errorCode = openapiTcamPolicyDestroy(clientHandle, &policyHandle);
        }
        policyHandle = OPEN_TCAM_API_POLICYID_INVALID;
        break;
        /* Get policy*/
      case 8:
        if (policyHandle == OPEN_TCAM_API_POLICYID_INVALID)
        {
          printf("Enter policy Handle\n");
          scanf("%d", &policyHandle);
        }

        if (policyHandle == OPEN_TCAM_API_POLICYID_INVALID)
        {
          printf("Error: Create policy first \n");
        }
        else
        {
          if (openapiTcamPolicyGet(clientHandle, (void *)&policyHandle, &getPolicyInfo) != OPEN_E_NONE)
            printf("Error: Policy does not exist\n");
        }
        break;

      case 9:
        openapiTcamPolicyDeleteAll(clientHandle);
        break;

      case 10:
        {
          /* Add policy on all interfaces*/
          errorCode = openapiTcamPolicyIntfAllAdd(clientHandle, &policyHandle);
          if (errorCode != OPEN_E_NONE)
          {
            printf("Error: Policy could not be applied on all interfaces, error %d\n", errorCode);
            return errorCode;
          }
          else
          {
            printf("Policy is applied on all interfaces\n");
          }
        }
        break;

      case 11:
        {
          /* Add policy on all interfaces */
          errorCode = openapiTcamPolicyIntfAllAdd(clientHandle, &policyHandle);
          if (errorCode != OPEN_E_NONE)
          {
            printf("Error: Policy could not be applied on all interfaces, error %d\n", errorCode);
            return errorCode;
          }
          else
          {
            printf("Policy is applied on all interfaces\n");
          }

          /* Get policy stats */
          errorCode = openapiTcamPolicyMeteringStatsGet(clientHandle, &policyHandle, &inProf, &outProf);
          if (errorCode != OPEN_E_NONE)
          {
            printf("Error: Unable to get Policy stats, error %d\n", errorCode);
            return errorCode;
          }
          else
          {
            printf("Policy stats retrieved successfully\n");
          }
        }
        break;

      case 12:
        {
          /* Add policy on all interfaces */
          errorCode = openapiTcamPolicyIntfAllAdd(clientHandle, &policyHandle);
          if (errorCode != OPEN_E_NONE)
          {
            printf("Error: Policy could not be applied on all interfaces, error %d\n", errorCode);
            return errorCode;
          }
          else
          {
            printf("Policy is applied on all interfaces\n");
          }

          printf("Start transmitting traffic and wait for 10 seconds ...");
          sleep(10);

          /* Clear policy stats */
          errorCode = openapiTcamPolicyStatsClear(clientHandle, &policyHandle);
          if (errorCode != OPEN_E_NONE)
          {
            printf("Error: Unable to clear Policy stats, error %d\n", errorCode);
            return errorCode;
          }
          else
          {
            printf("Policy stats cleared successfully\n");
          }
        }
        break;

      case 13:
        openBuildHeaderV2(&policyInfoV2);
        /* Add version 2 classifiers*/

      case 14:
        openAddClassifiersV2(&policyInfoV2);
        /* add actions */

      case 15:
        openAddPolicyActionV2(clientHandle, &policyInfoV2);
        break;

        /* create*/
      case 16:
        {
          errorCode = openapiTcamPolicyCreateV2(clientHandle, policyInfoV2, &policyHandle);
          if (errorCode != OPEN_E_NONE)
          {
            printf("Error: Policy could not be created Reason %d", errorCode);
          }
          else
          {
            printf("Policy succesfully added,handle  is %d \n", policyHandle);
          }
        }
        break;

      case 17:
        break;

      default:
        return 0;
    }
  }
  while (action != 17);

  openapiTcamApiFini(clientHandle);

  (void) openapiClientTearDown(clientHandle);
  return 0;
}

/*******************************************************************
*
* @brief  Example TCAM API: This is the main function that will demonstrate 
*         TCAM OpEN APIs.
*
* @returns  0: Success
* @returns  1: Failure
*
*********************************************************************/
int
main (int argc, char **argv)
{
  openapiClientHandle_t clientHandle;
  open_error_t result;
  open_buffdesc switch_os_revision;
  char switch_os_revision_string[100];
  open_tcam_version_t ver;
  
  l7proc_crashlog_register ();

  /* Register with OpEN */
  if ((result =
       openapiClientRegister ("tcam_example", &clientHandle)) != OPEN_E_NONE)
    {
      printf ("\nFailed to initialize RPC to OpEN. Exiting (result = %d)\n", result);
      exit (2);
    }

  /* RPC call can fail until server starts. Keep trying */
  while (openapiConnectivityCheck (&clientHandle) != OPEN_E_NONE)
    {
      sleep (1);
    }

  L7PROC_LOGF (L7PROC_LOG_SEVERITY_INFO, 0, "Starting TCAM API example application");

  printf ("\n");
  switch_os_revision.pstart = switch_os_revision_string;
  switch_os_revision.size = sizeof (switch_os_revision_string);
  if (openapiNetworkOSVersionGet (&clientHandle, &switch_os_revision) == OPEN_E_NONE)
  {
    printf ("Network OS version = %s\n", switch_os_revision_string);
  }
  else
  {
    printf ("Network OS version retrieve error\n");
  }


  if (openapiTcamApiVersionGet (&clientHandle, &ver) == OPEN_E_NONE)
  {
    printf ("TCAM API version = %d.%d\n", ver.versionMajor, ver.versionMinor);
  }
  else
  {
    printf ("TCAM API version retrieve error\n");
  }

  printf ("\n");

  return openTcamMainMenu(&clientHandle);


}
