/****************************************************************************/
/*  Single-Step      A debugging aid for E2      by Bryan Lewis  05/29/86   */
/*                                                                          */
/* SCENARIO:  Imagine you've written a complicated E2 procedure.  It's      */
/* screwing up somewhere but it's hard to pin down.  You could go back into */
/* the code and insert a bunch of prints and pauses (that is, sayerrors     */
/* and getkeys) to see what's happening, but that's tedious.                */
/*                                                                          */
/* Ta-da!  This automates the tedious part.  You position the cursor on the */
/* first statement you want to trace.  Press Single-Step (e.g., Ctrl-S)     */
/* repeatedly for as many lines as you want.  Recompile and run.            */
/*                                                                          */
/* WHAT IT DOES.  New lines will be inserted in your code to:               */
/*    1. Display the statement before it's executed:  "Before foo=foo+1".   */
/*    2. Pause for a keypress (so you can see the display).                 */
/*    3. Execute the statement.                                             */
/*    4. If it was an assignment, display the new value and pause, like:    */
/*       "Assigned foo=2".                                                  */
/* At any of those pauses you can press the Esc key to stop your procedure, */
/* which is pretty useful in itself.                                        */
/*                                                                          */
/* ------------------------------------------------------------------------ */
/*                                                                          */
/* Revised by Francesca Brunner Kennedy on 07/19/88 to add more function-   */
/* ality to this "debugger". Previously, it recognized only assignment      */
/* statements and performed a special function to give the programmer more  */
/* information about what results after the line is executed. Now it        */
/* performs similar functions for the following statements:                 */
/*      1. locate and change commands                                       */
/*      2. getline, insertline and replaceline statements, and              */
/*      3. parse statement                                                  */
/*      4. procedure call                                                   */
/*                                                                          */
/* I also added a capacity to deal with lines that contained a combination  */
/* of single and double quotes. Previously, the line was simply truncated,  */
/* and not printed in its entirety. Now STEP manages the multiple quotes,   */
/* prints the 'Before X' statement, and analyzes the line to see if any     */
/* special functions need to be performed to show the result of the line.   */
/*                                                                          */
/* My final contribution is that I strip all comments out of each line. I   */
/* find that these simply take up room, and often interfere with this       */
/* process.                                                                 */
/*                                                                          */
/* Slightly changed by JBL 8/10/88 to save and restore the user's previous  */
/* search string in EOS2.                                                   */
/* ------------------------------------------------------------------------ */
/*                                                                          */
/* When you're through debugging, reload your procedure and issue the       */
/* command "unstep" to remove all the ugly debugging lines.                 */
/*                                                                          */
/* Note that this is only a macro, a typing aid.  You still have to         */
/* recompile before the stepping works.                                     */
/*                                                                          */
/* INSTALLATION:  I've put this on Ctrl-S but you might prefer it on a      */
/* more obscure key; Shift-F8 would be nice.  To install:                   */
/*  - Enter the name of the desired key after the initial "def".            */
/*  - Delete all these wonderful comments at the top.                       */
/*  - Save with a memorable filename like "step.e".                         */
/*  - Include by adding "include step.e" to your main.e module.             */
/*                                                                          */
/****************************************************************************/
const
    SGL_QTE = "'"
    DBL_QTE = '"'


def c_f12 =                                   /* Single-step macro.  */

    getline line1

    /* additions by FBK */
    /* strip out any comments --> they may interfere later */
    open = pos('/*', line1)
    close = pos('*/', line1)
    hyphens = pos('--', line1)
    while open or hyphens do
	if open and close and (open < close) then
	    line1 = substr(line1, 1, open-1) || substr(line1, close+2)
	elseif hyphens then
	    line1 = substr(line1, 1, hyphens-1)
	endif
	open = pos('/*', line1)
	close = pos('*/', line1)
	hyphens = pos('--', line1)
    endwhile

    /* strip off leading and trailing blanks */
    line1 = strip(line1)

    /* modifications made by FBK */
    /* choose correct quote marks: single or double depending upon */
    /*   what type is incorporated into the line */
    posnS = pos(SGL_QTE, line1)
    posnD = pos(DBL_QTE, line1)

    /* special case for handling lines with both single and double  */
    /*  quote marks within them */
    if posnS and posnD then
	if posnS < posnD then
	    quote = DBL_QTE
	else
	    quote = SGL_QTE
	endif
	curpos = 1
	newline1 = ''
	/* divide the line into a series of smaller strings that */
	/*   will be concatenated together */
	loop
	    divider = pos(quote, line1, curpos)
	    if not divider then
	        newline1 = newline1||quote||substr(line1, curpos)||quote
	        leave
	    endif
	    tmpline = quote||substr(line1, curpos, divider-curpos)||quote||'||'
	    newline1 = newline1||tmpline
	    if quote = DBL_QTE then
	        quote = SGL_QTE
	    else
	        quote = DBL_QTE
	    endif
	    curpos = divider
	endloop
	myline1 = "/* STEP */ sayerror 'Before'" newline1

    else                                    /* both types of quotes do not exist in this line */
	newline1 = line1
	if posnS then
	    quote = DBL_QTE
	else
	    quote = SGL_QTE
	endif
	myline1 = "/* STEP */ sayerror" quote||'Before' newline1||quote
    endif

    insertline myline1
    myline1='/* STEP */ if getkey()=esc then stop endif;sayerror 0'
    insertline myline1

    /* Bug fix.  If we were working on the LAST line of the file, then the */
    /* upcoming insertline will not go where we want it.                   */
    if .line=.last then insert_line else down endif

    /* Now see if it was an assignment.  If so, show new value.   */
    /* How do we know it was an assignment?  Look for an = sign.  */
    posnD=pos('=',line1)                    /* Reuse variables although names     */
    myline1='no'                            /* aren't very appropriate now.       */
    if posnD>0 then                         /* PosnD=posn of = sign.  PosnS=space.*/
	posnS=pos(' ',line1)
	if posnD<posnS or posnS=0 then      /* If = before space, or no space,    */
	    myline1='yes'                   /* then yes it was an assignment.     */
	else
	    loop
	        posnS=posnS+1               /* Hmmm.  There's a space before the =. */
	        c=substr(line1,posnS,1)     /* If it's only spaces up to the =,     */
	        if c=' ' then               /* then it's an assignment.             */
	            iterate
	        elseif c='=' then
	            myline1='yes'
	        endif                       /* Else some other word intervenes.     */
	        leave                       /* For example "if foo=1".  It's not    */
	    endloop                         /* an assignment, leave 'no'.           */
	endif
    endif
    if myline1='yes' then
	myline1=substr(line1,1,posnD-1)
	myline1="/* STEP */ sayerror 'Assigned "myline1"='"myline1
	insertline myline1
	myline1='/* STEP */ if getkey()=esc then stop endif;sayerror 0'
	insertline myline1
    endif

    /* -------------------------------------------------------------------- */
    /* The following section is composed of improvements by Francesca       */
    /* Brunner Kennedy.                                                     */
    /* -------------------------------------------------------------------- */

    /* :::::::::::::::::::::::::::: */
    /* SEARCH and LOCATE commands : */
    /* :::::::::::::::::::::::::::: */

    /* if a search command is done, we want to see what is being searched for */
    /*   and what the cursor has found after the search */
    if pos("'/", line1) = 1 or pos('"/', line1) = 1 or
       pos("'L", line1) = 1 or pos('"L', line1) = 1 or
       pos("'l", line1) = 1 or pos('"l', line1) = 1 or
       pos("'C", line1) = 1 or pos('"C', line1) = 1 or
       pos("'c", line1) = 1 or pos('"c', line1) = 1 then /* if this is a search or replace */

	if pos(SGL_QTE, line1) = 1 then
	    open_quote = SGL_QTE
	    quote = SGL_QTE
	    parse value line1 with "'"command '/'string'/'rest"'"
	else
	    open_quote = DBL_QTE
	    quote = DBL_QTE
	    parse value line1 with '"'command '/'string'/'rest'"'
	endif

	myline2 = '/* STEP */ sayerror' quote'After search for: '
	quote_num = 1
	q = pos(open_quote, string)
	tmpline = string
	while q do
	    part = substr(tmpline, 1, q-1)
	    tmpline = substr(tmpline, q+1)
	    myline2 = myline2 || part || quote
	    quote_num = quote_num + 1

	    s = pos(SGL_QTE, tmpline)
	    d = pos(DBL_QTE, tmpline)
	    if s and (s < d or not d) then
	        q = s
	        open_quote = SGL_QTE
	        quote = SGL_QTE
	    elseif d then
	        q = d
	        open_quote = DBL_QTE
	        quote = DBL_QTE
	    else
	        q = 0
	    endif
	endwhile

	if tmpline and quote_num > 1 then
	    myline2 = myline2 || ' 'tmpline
	endif
	if quote_num = 1 then
	    myline2 = myline2 || string || quote
	elseif (quote_num/2 /== 0) then
	    /* odd number of quotes */
	    myline2 = strip(myline2, 'T', quote)
	endif

	insertline myline2
	myline2='/* STEP */ if getkey()=esc then stop endif;sayerror 0'
	insertline myline2
    endif

    /* :::::::::::::::::::::::::::: */
    /* GETLINE statement :          */
    /* :::::::::::::::::::::::::::: */

    /* if a getline is done, we want to see what the */
    /*   line is after it was gotten */
    if pos('getline', line1) then
	parse value line1 with 'getline' variable ',' line ',' fileid
	variable = strip(variable)          /* strip all of the variables */
	line = strip(line)                  /*   for better printability  */
	fileid = strip(fileid)
	myline3 ="/* STEP */ sayerror '"variable"='" variable
	if line then
	    if line > 0 and line < 10000 then           /* if it's a number ...   */
	        myline3 = myline3 || "' at line" line"'" /* ... leave it in quotes */
	    else
	        myline3 = myline3 || "' at line'" line   /* otherwise it's a variable */
	    endif
	endif
	if fileid then
	    myline3 = myline3 || "' in fileid'" fileid
	endif
	insertline myline3
	myline3='/* STEP */ if getkey()=esc then stop endif;sayerror 0'
	insertline myline3
    endif

   /* :::::::::::::::::::::::::::: */
   /* REPLACELINE statement :      */
   /* :::::::::::::::::::::::::::: */

   /* if a replaceline is done, we want to see what the */
   /*   new line will be */
   if pos('replaceline', line1) then
	parse value line1 with 'replaceline' variable ',' line ',' fileid
	variable = strip(variable)          /* strip all of the variables */
	line = strip(line)                  /*   for better printability  */
	fileid = strip(fileid)

	/* delete everything inside matching quote marks -- */
	/*   want to print only variables and their values */
	single = pos("'", variable)
	double = pos('"', variable)
	concat = pos('|', variable)
	while concat do
	    variable = substr(variable, 1, concat-1) || substr(variable, concat+1)
	    concat = pos('|', variable)
	endwhile
	while single or double do
	    if single < double and single or not double then
	        open_quote = single
	        quote = "'"
	    else
	        open_quote = double
	        quote = '"'
	    endif
	    close_quote = pos(quote, variable, open_quote+1)
	    for i = 1 to (close_quote - open_quote + 1)
	        variable = substr(variable, 1, open_quote-1) ||
	                      substr(variable, open_quote+1)
	    endfor
	    single = pos("'", variable)
	    double = pos('"', variable)
	endwhile
	variable = strip(variable)

	myline4 ="/* STEP */ sayerror '"variable"='" variable
	if line then
	    if line > 0 and line < 10000 then          /* if it's a number ...   */
	        myline4 = myline4 || "' at line" line"'" /* ... leave it in quotes */
	    else
	        myline4 = myline4 || "' at line'" line   /* otherwise it's a variable */
	    endif
	endif
	if fileid then
	    myline4 = myline4 || "' in fileid'" fileid
	endif
	insertline myline4
	myline4='/* STEP */ if getkey()=esc then stop endif;sayerror 0'
	insertline myline4
    endif

    /* :::::::::::::::::::::::::::: */
    /* INSERTLINE statement :       */
    /* :::::::::::::::::::::::::::: */

    /* if a insertline is done, we want to see what the */
    /*   new line will be */
    if pos('insertline', line1) then
	parse value line1 with 'insertline' variable ',' line ',' fileid
	variable = strip(variable)          /* strip all of the variables */
	line = strip(line)                  /*   for better printability  */
	fileid = strip(fileid)

	/* delete everything inside matching quote marks -- */
	/*   want to print only variables and their values */
	sgl = pos("'", variable)
	dbl = pos('"', variable)
	con = pos('|', variable)
	while con do
	    variable = substr(variable, 1, con-1) || substr(variable, con+1)
	    con = pos('|', variable)
	endwhile

	while sgl or dbl do
	    if sgl < dbl  and sgl or not dbl then
	        Lquote_pos = sgl
	        quote = "'"
	    else
	        Lquote_pos = dbl
	        quote = '"'
	    endif
	    close_quote = pos(quote, variable, Lquote_pos+1)
	    for i = 1 to (close_quote - Lquote_pos + 1)
	        variable = substr(variable, 1, Lquote_pos-1) ||
	                      substr(variable, Lquote_pos+1)
	    endfor
	    sgl = pos("'", variable)
	    dbl = pos('"', variable)
	endwhile
	variable = strip(variable)

	myline5 ="/* STEP */ sayerror '"variable"='" variable
	if line then
	    if line > 0 and line < 10000 then          /* if it's a number ...   */
	        myline5 = myline5 || "' at line" line"'" /* ... leave it in quotes */
	    else
	        myline5 = myline5 || "' at line'" line   /* otherwise it's a variable */
	    endif
	endif
	if fileid then
	    myline5 = myline5 || "' in fileid'" fileid
	endif
	insertline myline5
	myline5='/* STEP */ if getkey()=esc then stop endif;sayerror 0'
	insertline myline5
    endif


    /* :::::::::::::::::::::::::::: */
    /* PARSE statement :            */
    /* :::::::::::::::::::::::::::: */

    /* if a parse is done, we want to see what the values of */
    /*   the variables in the template are after the parse   */
    if pos('parse', line1) then
	parse value line1 with 'parse value' input 'with' template

	/* delete everything between matching quote marks -- */
	/*   we're only interested in parsing variables */
	s = pos("'", template)
	d = pos('"', template)
	while s or d do
	    if s < d  and s or not d then
	        quote_pos = s
	        quote = "'"
	    else
	        quote_pos = d
	        quote = '"'
	    endif
	    mate_pos = pos(quote, template, quote_pos+1)
	    for i = 1 to (mate_pos - quote_pos + 1)
	        template = substr(template, 1, quote_pos-1) ||
	                      substr(template, quote_pos+1)
	    endfor
	    template = substr(template, 1, quote_pos-1) || ' ' ||
	                    substr(template, quote_pos)
	    s = pos("'", template)
	    d = pos('"', template)
	endwhile

	/* eliminate any trigger variables, distinguished by */
	/*   their enclosing parentheses */
	parse value template with before '('trigger')' after
	while trigger do
	    template = before || after
	    parse value template with before '('trigger')' after
	endwhile

	/* parse out all the parsing variables -- up to 6 */
	/*   and print values */
	parse value template with var1 var2 var3 var4 var5 var6
	myline6 = ''
	-- jbl 10/25/88:  don't accept the dummy variable '.' .
	if var1 and var1<>'.' then
	    myline6 = "/* STEP */ sayerror "quote var1"="quote var1
	endif
	if var2 and var2<>'.' then
	   myline6 = myline6 || quote", "var2"="quote var2
	endif
	if var3 and var3<>'.' then
	   myline6 = myline6 || quote", "var3"="quote var3
	endif
	if var4 and var4<>'.' then
	   myline6 = myline6 || quote", "var4"="quote var4
	endif
	if var5 and var5<>'.' then
	   myline6 = myline6 || quote", "var5"="quote var5
	endif
	if var6 and var6<>'.' then
	   myline6 = myline6 || quote", "var6"="quote var6
	endif
	if myline6 then
	   insertline myline6
	   myline6='/* STEP */ if getkey()=esc then stop endif;sayerror 0'
	   insertline myline6
	endif
    endif

    /* :::::::::::::::::::::::::::: */
    /* Procedure CALL               */
    /* :::::::::::::::::::::::::::: */

    /* if a procedure is called, add a sayerror afterwards that tells */
    /*   user that control has returned from the called proc */
    if pos('call', line1) then
	parse value line1 with 'call' proc_name'(' rest
	proc_name = strip(proc_name)
	myline7 = "/* STEP */ sayerror 'Returned from call to" proc_name"()'"
	insertline myline7
	myline7='/* STEP */ if getkey()=esc then stop endif;sayerror 0'
	insertline myline7
    endif

    /* -------------------- */
    /* end of FBK additions */
    /* -------------------- */



/* A command (to be issued from the command line) to remove all the lines */
/* with STEP comments.  (Now you can the reason for the clever avoidance  */
/* of the comment string within the step macro itself.)                   */

defc unstep =

    cursor_data
    OldCol=.col; OldLine=.line
    top
    myline= 'L /*' 'STEP */'
    myline
    while rc==0 do
	delete_line; up
	repeat_find
    endwhile
    if OldCol>0  then .col    =OldCol  endif
    if OldLine<=0 then OldLine=1 endif
    if OldLine>.last then OldLine=.last endif
    .line=OldLine
    cursor_command
    delete_line; begin_line
    sayerror 0
    cursor_command

