#!/usr/bin/perl -w
##
## (c)  Copyright  IBM Corp.  2006  All Rights Reserved
##
## US Government Users Restricted Rights - Use, duplication or
## disclosure restricted by GSA ADP Schedule Contract with IBM Corp
##
## Module History:
##  -00 05/23/2006 John Thornton   D63J I0640 - Patch unmounted file systems
##  -01 09/21/2006 John Thornton   D63U /mnt/dos now permanently mounted
##
##

#
# Includes
#
use Getopt::Long;

##############################################################################
#####                          GLOBAL VARIABLES                          #####
##############################################################################

# Trace Constants
$TRC_T = "XMCSPUMT:";
$TRC_D = "XMCSPUMD:";
$TRC_F = "XMCSPUMF:";

# Error Constants
$ERROR_NO_ERROR               =  0;
$ERROR_NOTHING_TO_DO          =  1;     # So caller won't need to reboot
                                        #   on our account
$ERROR_MOUNTING               =  2;
$ERROR_UNMOUNTING             =  3;
$ERROR_CMP_FAILED             =  4;
$ERROR_CP_FAILED              =  5;
$ERROR_HELP                   = 99;

# Boolean Constants
$TRUE  = 1;
$FALSE = 0;

# Program Data
$me = $0;
$me =~ s/^(.*)\///;
$me =~ s/.pl$//;
$traceCmd = "actzTrace";

$consoleDir = "/console";
$rebootFlag = $FALSE;

# List the file systems to be mounted and unmounted. The ones to be unmounted
# will be determined by mount processing. It will only contain file systems
# that we were required to mount. If one was already mounted, we will leave it
# mounted.

@fsToMount   = ("/mnt/test");                                                   #-01
@fsToUnmount = ();


##############################################################################
#####                          SHARED METHODS                            #####
##############################################################################

#
#
#
sub determineConsoleDirectory
{
   my $this = $me . ".determineConsoleDirectory()";

   # Many possibilities in the non-production world
   if ($ENV{'CONSOLE_PATH'})
   {
      $consoleDir = $ENV{'CONSOLE_PATH'}.'/';
   }
   elsif ($ENV{'HWMCAHOME'})
   {
      $consoleDir = $ENV{'HWMCAHOME'}.'/';
   }
   else
   {
      $consoleDir = "/console/";
   }

   # Change any "//" to "/"
   $consoleDir =~ s/\/+/\//g;

   return();
}

#
#
#
sub trace
{
    my $text = shift;
    chomp($text);

    if ("actzTrace" eq "$traceCmd")
    {
       system(("$traceCmd", "$text"));
    }
    else
    {
       print(STDOUT "$text\n");
    }
}

#
# usage
#
sub usage
{
    print(<<EOM);
Usage: iqzmPatchUnmounted is a script that runs at startup to enable
       patching of files in unmounted file systems. It should never
       be run outside of that context.
       iqzmPatchUnmounted.pl [-help] [-tracecommand=cmd]
EOM
}

################################################################################
#####                        File System Mounting                          #####
################################################################################

#
#
#
sub doMounts
{
   my $this = $me . ".doMounts()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my $fileSystem = "";
   foreach $fileSystem (@fsToMount)
   {
      trace("$TRC_F    $this - processing the $fileSystem file system");

      # First, see if there is any indication of changes to the file
      # system. If there are, there will be a directory of the same name
      # under the $consoleDir/mcf/ directory. If there is not, we are done
      # with this file system. Nothing needs to be done.

      $updatesDir = $consoleDir . "mcf" . $fileSystem;
      if ( -d "$updatesDir" )
      {
         trace("$TRC_F    $this - there is work to do for $updatesDir");
      }
      else
      {
         trace("$TRC_F    $this - $updatesDir does not exist, there is nothing to do for this file system");
         next;
      }

      # We only want to mount the file system if it is not already mounted.
      # To check this, we will do an ls command for all files in the file
      # system. If it is not mounted, we will get a non-zero return code.
      # If it is mounted, we don't need to do anything else.

      trace("$TRC_F    $this - testing to see if the file system is already mounted");
      `ls $fileSystem/* >/dev/null 2>&1`;
      $rc = $? >> 8;
      if ($rc == 0)
      {
         trace("$TRC_F    $this - the $fileSystem file system is already mounted");
         next;
      }

      # OK, so we need to mount the file system. We won't know what to do with
      # any errors, so an error is fatal to the remainder of the process.

      trace("$TRC_F    $this - mounting $fileSystem");
      my $error = `mount '$fileSystem' 2>&1`;
      unless (defined($error)) { $error = ""; }
      chop $error;
      $rc = $? >> 8;
      if ($rc != 0)
      {
         trace("$TRC_F    $this() - mount error is \"$error\"");
         my $hex_crc = sprintf("%04X", $rc);
         $rc = $ERROR_MOUNTING;
         my $hex_rc  = sprintf("%04X", $rc);
         `logerror 0xAA 0x02 0x0000 0xE569 0x$hex_rc$hex_crc 0x0B20 XMCL`;
      }
      else
      {
         # We need to record that we mounted this file system so we can
         # unmount it when we are through.

         @fsToUnmount = (@fsToUnmount, $fileSystem);
      }
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

################################################################################
#####                       File System Unmounting                         #####
################################################################################

#
#
#
sub doUnmounts
{
   my $this = $me . ".doUnmounts()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   # We know that anything in this list was mounted earlier by this
   # process. Just go ahead and attempt the operation. We won't know what
   # to do with any errors, so it is a hard error and we will take a log.
   # We will continue to try to unmount everything else.

   my $fileSystem = "";
   foreach $fileSystem (@fsToUnmount)
   {
      trace("$TRC_F    $this - unmounting $fileSystem");
      my $error = `umount '$fileSystem' 2>&1`;
      unless (defined($error)) { $error = ""; }
      chop $error;
      $rc = $? >> 8;
      if ($rc != 0)
      {
         trace("$TRC_F    $this() - umount error is \"$error\"");
         my $hex_crc = sprintf("%04X", $rc);
         $rc = $ERROR_UNMOUNTING;
         my $hex_rc  = sprintf("%04X", $rc);
         `logerror 0xAA 0x02 0x0000 0xE569 0x$hex_rc$hex_crc 0x0B20 XMCL`;
      }
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

################################################################################
#####                           Update the Files                           #####
################################################################################

#
# In this outer routine we will run through all the mount points and call a
# subroutine to do the work associated with a single mount point.
#
sub updateFiles
{
   my $this = $me . ".updateFiles()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my $fileSystem = "";
   foreach $fileSystem (@fsToMount)
   {
      trace("$TRC_F    $this - processing the $fileSystem file system");

      # First, see if there is any indication of changes to the file
      # system. If there are, there will be a directory of the same name
      # under the $consoleDir/mcf/ directory. If there is not, we are done
      # with this file system. Nothing needs to be done.

      $updatesDir = $consoleDir . "mcf" . $fileSystem;

      $rc = processDirectory($updatesDir, $fileSystem);
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

#
# This subroutine processes all the entries in a directory. For a directory,
# it recursively calls itself. For a file, it calls the routine that will
# move the file to the file system we are updating if the file does not exist
# or is different.
#
sub processDirectory
{
   my $this = $me . ".processDirectory()";
   trace("$TRC_T -> $this");

   my $theDir        = shift;
   my $theFileSystem = shift;

   trace("$TRC_F    $this - processing the $theDir directory");

   my $rc = $ERROR_NO_ERROR;
   my $fullName = "";

   opendir(DIR, $theDir);
   my @files = readdir(DIR);
   closedir(DIR);

   foreach $file (@files)
   {
      trace("$TRC_F    $this - processing $file");

      # We must skip the "." and ".." directories.

      if ($file eq ".")  { next };
      if ($file eq "..") { next };

      # Well, we do one thing for a directory and a different thing for a file.

      $fullName = "$theDir/$file";

      if ( -d "$fullName" )
      {
         trace("$TRC_F    $this - $fullName is a directory");
         $rc = processDirectory($fullName, $theFileSystem);
      }
      else
      {
         trace("$TRC_F    $this - $fullName is a file");
         $rc = processFile($fullName, $theFileSystem);
      }
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

#
# This subroutine processes a single file. It will move the file to the
# file system we are updating if the file does not exist or is different.
#
sub processFile
{
   my $this = $me . ".processFile()";
   trace("$TRC_T -> $this");

   my $theFile       = shift;
   my $theFileSystem = shift;

   my $rc = $ERROR_NO_ERROR;
   $updateFile = $FALSE;

   trace("$TRC_F    $this - processing the $theFile file");

   # Generate the target file name.

   my $pos = index($theFile, $theFileSystem);
   my $target = substr($theFile, $pos);

   trace("$TRC_F    $this - target file is $target");

   # If the target file exists, do a compare to see if the new version is
   # different.
   # If the target file is not present, then we must update it.

   if (-f $target)
   {
      my $cmd = "cmp '$target' '$theFile' 2>&1";
      my $result = `$cmd`;
      unless (defined($result)) { $result = ""; }
      chomp $result;
      my $trc = $? >> 8;
      if ($trc == 0)
      {
         # The files match. This file is up-to-date.
         trace("$TRC_F    $this - file $target is up-to-date");
      }
      elsif ($trc == 1)
      {
         # The files do not match. We need to update the file.
         trace("$TRC_F    $this - file $target requires updating");
         $updateFile = $TRUE;
      }
      else
      {
         # Had a problem. Shouldn't happen. This is bad.
         trace("$TRC_F    $this - processing for file $theFile failed, rc = $rc");
         trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
         my $hex_crc = sprintf("%04X", $trc);
         $rc = $ERROR_CMP_FAILED;
         my $hex_rc  = sprintf("%04X", $rc);
         `logerror 0xAA 0x02 0x0000 0xE569 0x$hex_rc$hex_crc 0x0B20 XMCL`;
      }
   }
   else
   {
      trace("$TRC_F    $this - file $target is new");
      $updateFile = $TRUE;
   }

   # Now, if we need to, we need to move the new file into the target area.
   # We need to keep the permissions associated with the new file.

   if ($rc == $ERROR_NO_ERROR)
   {
      if ($updateFile == $TRUE)
      {
         trace("$TRC_F    $this - updating file $target");

         my $cmd = "cp --reply=yes '$theFile' '$target' 2>&1";
         my $result = `$cmd`;
         unless (defined($result)) { $result = ""; }
         chomp $result;
         my $trc = $? >> 8;
         trace("$TRC_D    $this - command is [$cmd], rc = $trc, result is [$result]");
         if ($trc != 0)
         {
            # Had a problem. Shouldn't happen. This is bad.
            trace("$TRC_F    $this - copying file $theFile to $target failed");
            trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
            my $hex_crc = sprintf("%04X", $trc);
            $rc = $ERROR_CP_FAILED;
            my $hex_rc  = sprintf("%04X", $rc);
            `logerror 0xAA 0x02 0x0000 0xE569 0x$hex_rc$hex_crc 0x0B20 XMCL`;
         }
         else
         {
            # If we moved a file, then we want to let our caller know that he
            # should reboot.

            $rebootFlag = $TRUE;
         }
      }
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

################################################################################
#####                           MAINLINE CODE                              #####
################################################################################

# Process the arguments
GetOptions(
      'help'            =>\$help,
      'tracecommand:s'  =>\$traceCmd,
);

unless (defined($traceCmd )) { $traceCmd = ""; }

determineConsoleDirectory();

trace("$TRC_T -> $me(@ARGV)");
trace("$TRC_F    $me() - console directory is $consoleDir");

my $rc = $ERROR_NO_ERROR;

if ($help)
{
   usage();

   $rc = $ERROR_HELP;
   trace("$TRC_T <- $me() - rc = $rc");
   exit($rc);
}

# Run each of the functions, unless an error interferes.

$rc = doMounts();

if ($rc == $ERROR_NO_ERROR)
{
   $rc = updateFiles();
}

$rc = doUnmounts();

# Now, if there was no error, but we didn't change any files, set the return
# code to indicate we did not update any files. The caller can use this
# information to decide if he wants to reboot or not.

if ($rebootFlag == $FALSE)
{
   $rc = $ERROR_NOTHING_TO_DO;
}

trace("$TRC_T <- $me() - rc = $rc");

exit($rc);

