#!/usr/bin/perl -w
##
## (c)  Copyright  IBM Corp.  2004, 2006  All Rights Reserved
##
## US Government Users Restricted Rights - Use, duplication or
## disclosure restricted by GSA ADP Schedule Contract with IBM Corp
##
## Module History:
##  -00 10/29/2004 John Thornton   D61C initial release
##  -01 03/31/2005 John Thornton   D61S link processing
##  -02 09/10/2005 John Thornton   D63J H0737 - ignore permissions if no file
##  -03 12/02/2005 John Thornton   D63J H2150 - wrong key for link processing
##  -04 04/19/2006 John Thornton   D64X H9373 - Permissions problems with '$' files
##  -05 04/23/2006 John Thornton   D66D G8721 - Locked file processing redesign
##  -06 05/23/2006 John Thornton   D63J I0640 - Patch unmounted files
##  -07 06/09/2006 John Thornton   D66G I1098 Change file timestamp when
##                                        changing file permissions
##  -08 08/04/2006 John Thornton   D66M Perl parsing indicates unused vars
##  -09 08/28/2006 John Thornton   D66Q New query file location facility
##

#
# Includes
#
use Getopt::Long;

##############################################################################
#####                          GLOBAL VARIABLES                          #####
##############################################################################

# Trace Constants
$TRC_T = "XMCFSPST:";
$TRC_D = "XMCFSPSD:";
$TRC_F = "XMCFSPSF:";

# Error Constants
$ERROR_NO_ERROR               =  0;
$ERROR_NOT_ROOT               =  1;

$ERROR_PERMS_FILE_ACCESS      =  4;
$ERROR_PERMS_CONTENT_ERROR    =  5;
$ERROR_PERMS_NO_ACTION        =  6;
$ERROR_PERMS_OWNER_FAILURE    =  7;
$ERROR_PERMS_GROUP_FAILURE    =  8;
$ERROR_PERMS_FLAGS_FAILURE    =  9;
# start -01
$ERROR_LINKS_FILE_ACCESS      = 10;
$ERROR_LINKS_CONTENT_ERROR    = 11;
$ERROR_LINKS_BAD_COMMAND      = 12;
$ERROR_LINKS_NO_ALIAS         = 13;
$ERROR_LINKS_REMOVE_FAILURE   = 14;
$ERROR_LINKS_BAD_TYPE         = 15;
$ERROR_LINKS_NO_TARGET        = 16;
$ERROR_LINKS_ALIAS_EXISTS     = 17;
$ERROR_LINKS_CREATE_FAILURE   = 18;
# end -01
$ERROR_UPDATE_FAILED          = 19;                                             #-06-08
# start -04
$ERROR_SWAP_FILE_ACCESS       = 20;
$ERROR_SWAP_FILE_CONTENT      = 21;
$ERROR_SWAP_BAD_COMMAND       = 22;
$ERROR_SWAP_NO_SOURCE         = 23;
$ERROR_SWAP_MV_FAILURE        = 24;
$ERROR_SWAP_PERMS_FAILURE     = 25;
$ERROR_SWAP_TARGET_IS_DIR     = 26;
$ERROR_SWAP_MKDIR             = 27;
$ERROR_SWAP_NEWDIR_OWNER      = 28;
$ERROR_SWAP_NEWDIR_PERMS      = 29;
$ERROR_SWAP_NEWSUBDIR_OWNER   = 30;
$ERROR_SWAP_NEWSUBDIR_FLAGS   = 31;
# end -04
$ERROR_PERMS_TOUCH            = 32;                                             #-07
$ERROR_DFC_QUERY              = 33;                                             #-09
$ERROR_HELP                   = 99;

# Boolean Constants
$TRUE  = 1;
$FALSE = 0;

# Program Data
$me = $0;
$me =~ s/^(.*)\///;
$me =~ s/.pl$//;
$traceCmd = "actzTrace";
$rebootRequired = $FALSE;                                                       #-05

##############################################################################
#####                          SHARED METHODS                            #####
##############################################################################

#
#
#
sub queryFileLocation
{
   my $this = $me . ".queryFileLocation()";
   my $file = shift;
   my $path = "/console/data/";
   trace("$TRC_T -> $this - $file");

   # start -09
   trace("$TRC_F    $this - calling actzQueryFileLocation() to get path");
   my $error = `actzQueryFileLocation '$file' 2>&1`;
   unless (defined($error)) { $error = ""; }
   chop $error;
   $rc = $? >> 8;
   if ($rc)
   {
      trace("$TRC_F    $this() - actzQueryFileLocation() error is \"$error\"");
      $rc = $ERROR_DFC_QUERY;
   }
   else
   {
      $path = $error;
   }
   # end -09

   trace("$TRC_T <- $this - $path");
   return($path);
}

# start -04
#
# getPerms
#
# To ensure the data returned is as expected, the caller should verify that the
# target of the operation is a file and that it exists.
#
sub getPerms
{
   my $this = $me . ".getPerms()";

   my $fn = shift;

   # Now get the info about the file
   my @filestats = stat($fn);
   my $owner     = $filestats[4];
   my $group     = $filestats[5];
   my $perms     = $filestats[2];

   trace("$TRC_F    $this - owner is $owner" .
                         ", group is $group" .
                         ", perms is $perms");

   # Convert to more useful form

   my $junk;
   ($owner, $junk) = getpwuid($owner);
   ($group, $junk) = getgrgid($group);

   $perms = $perms % (8*8*8);
   $perms = sprintf("%o", $perms);

   trace("$TRC_F    $this - owner is $owner" .
                         ", group is $group" .
                         ", perms is $perms");

   return ($owner, $group, $perms);
}
# end -04

# start -04
#
#
#
sub updateHistory
{
   # Save up to 9 old versions. It is a best effort thing.
   # We don't worry about errors.

   my $this = $me . ".updateHistory()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;
   my $file = shift;

   trace("$TRC_T    $this - base file is $file");

   `mv -f $file.8 $file.9 2> /dev/null`;
   `mv -f $file.7 $file.8 2> /dev/null`;
   `mv -f $file.6 $file.7 2> /dev/null`;
   `mv -f $file.5 $file.6 2> /dev/null`;
   `mv -f $file.4 $file.5 2> /dev/null`;
   `mv -f $file.3 $file.4 2> /dev/null`;
   `mv -f $file.2 $file.3 2> /dev/null`;
   `mv -f $file.1 $file.2 2> /dev/null`;
   `mv -f $file   $file.1 2> /dev/null`;

   trace("$TRC_T <- $this");

   return($rc);
}
# end -04

#
#
#
sub setHozd
{
   my $this = $me . ".setHosed()";
   my $msg = shift;
   trace("$TRC_T -> $this - message is $msg");

   my $rc = $ERROR_NO_ERROR;
   my $filename = "iqzmhozd.dat";

   my $path = queryFileLocation($filename);
   my $hozdfile = $path . $filename;
   # Change any "//" to "/"
   $hozdfile =~ s/\/+/\//g;
   trace("$TRC_F    $this - hosed file is $hozdfile");

   open(HOZDFILE, ">> " . $hozdfile);
   print HOZDFILE ($msg . "\n");
   close(HOZDFILE);

   trace("$TRC_F    $this - issue the HOZD refcode");
   `iqzmlogerror 0xE563`;                                                       #-05

   trace("$TRC_T <- $this");
   return;
}

#
#
#
sub trace
{
    my $text = shift;
    chomp($text);

    if ("actzTrace" eq "$traceCmd")
    {
       system(("$traceCmd", "$text"));
    }
    else
    {
       print(STDOUT "$text\n");
    }
}

#
# usage
#
sub usage
{
    print(<<EOM);
Usage: iqzmShutdownProcessing [-help] [-tracecommand=cmd]
       this command is intended to be run from the manager script as the
       manager is shutting down
EOM
}

##############################################################################
#####                           FILE SWAPPING                            #####
##############################################################################

# start -04
#
#
#
sub doFileSwapping
{
   my $this = $me . ".doFileSwapping()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;
   my $filename = "iqzmlock.dat";

   my $path = queryFileLocation($filename);
   my $datafile = $path . $filename;
   # Change any "//" to "/"
   $datafile =~ s/\/+/\//g;
   trace("$TRC_F    $this - swap data file is $datafile");

   # Does the file exist?
   if ( -f "$datafile" )
   {
      unless (open(DATAFILE, $datafile))
      {
         trace("$TRC_F    $this - error opening swap data file");
         $rc = $ERROR_SWAP_FILE_ACCESS;
      }

      if ($rc == $ERROR_NO_ERROR)
      {
         trace("$TRC_D    $this - reading swap data file");
         my @mydata = <DATAFILE>;

         trace("$TRC_D    $this - closing swap data file");
         close (DATAFILE);

         $rc = processSwaps(@mydata);
      }

      # Copy file to .err version if error
      if ($rc != $ERROR_NO_ERROR)
      {
         $target = $datafile . ".err";
         `cp -f $datafile $target`;
      }

      # Add file to file history
      updateHistory($datafile);

      if ($rc != $ERROR_NO_ERROR)
      {
         # Our file swapping failed. This is of major concern. We need to
         # become permanently disabled.
         trace("$TRC_F    $this - Failure swapping files at reboot time");
         `iqzmlogerror 0xE567 $rc`;
         setHozd("$this - Failure swapping files at reboot time");
      }
   }
   else
   {
      # There is nothing to do
      trace("$TRC_F    $this - No file swapping data was found");
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}
# end -04

# start -04
#
#
#
sub processSwaps
{
   my $this = $me . ".processSwaps()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my @lines = @_;

   my $command = "";
   my $source  = "";
   my $target  = "";

   my $line    = "";

   my $contentError = $FALSE;

   # An error here is severe. We will stop on the first error encountered.

   foreach $line (@lines)
   {
      # Remove the eol
      chomp($line);
      trace("$TRC_D    $this - line is [$line]");

      # if the line is all blanks or empty, just ignore it.

      my @parts = split(/ +/, $line);
      my $size = @parts;
      if ($size == 0)
      {
        next;
      }

      # Break the line into the 3 key and value pairs.

      ($command, $source, $target) = $line =~ /^<command>(.*)<source>(.*)<target>(.*)/;

      # These elements can end up undefined, so handle that to
      # avoid runtime error messages.

      unless (defined($command)) { $command = ""; }
      unless (defined($source))  { $source  = ""; }
      unless (defined($target))  { $target  = ""; }

      trace("$TRC_D    $this - command is [$command], source is [$source], target is [$target]");

      # If any of the values are missing, the line was not correctly formatted.

      if ($command eq "") { $contentError = $TRUE; }
      if ($source  eq "") { $contentError = $TRUE; }
      if ($target  eq "") { $contentError = $TRUE; }

      if ($contentError == $TRUE)
      {
         trace("$TRC_D    $this - improperly formatted line encountered in swap data file");
         $rc = $ERROR_SWAP_FILE_CONTENT;
         last;
      }
      else
      {
         my $trc = processOneSwap($command, $source, $target);
         # Save the rc if it represents an error
         if ($trc != $ERROR_NO_ERROR)
         {
            $rc = $trc;
            last;
         }
      }
   }

   # Any logging is done at the top level of swap file processing

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}
# end -04

# start -04
#
#
#
sub processOneSwap
{
   my $this = $me . ".processOneSwap()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my ($command, $source, $target) = @_;

   trace("$TRC_F    $this - parms are [$command], [$source], [$target]");

   # Do we have a valid command?

   my $validCommand = $FALSE;
   if ($command eq "move")
   {
      $validCommand = $TRUE;
   }

   if ($validCommand == $FALSE)
   {
      trace("$TRC_F    $this - swap processing command is not recognized");
      $rc = $ERROR_SWAP_BAD_COMMAND;
   }

   # Does the source file must exist (the target file might not).

   if ($rc == $ERROR_NO_ERROR)
   {
      if (!( -f "$source" ))
      {
         trace("$TRC_F    $this - source file does not exist");
         $rc = $ERROR_SWAP_NO_SOURCE;
      }
   }

   # Ensure the target is not a directory.

   if ($rc == $ERROR_NO_ERROR)
   {
      if ( -d "$target" )
      {
         trace("$TRC_F    $this - target is a directory, which is not valid");
         $rc = $ERROR_SWAP_TARGET_IS_DIR;
      }
   }

   # If we are doing erase processing, we may have to create the directory,
   # because there was no file in the MCF.

   if ($rc == $ERROR_NO_ERROR)
   {
      # Get just the path - remove the file name.

      my $i   = rindex($target, "/");
      my $dir = substr($target, 0, $i);

      if ( !( -d "$dir" ) )
      {
         # Must be an "erase" - we have to create the directory.
         trace("$TRC_F    $this - making directory $dir for target");
         my $error = `mkdir -p '$dir' 2>&1`;
         unless (defined($error)) { $error = ""; }
         chop $error;
         $rc = $? >> 8;
         if ($rc)
         {
            trace("$TRC_F    $this() - mkdir -p error is \"$error\"");
            $rc = $ERROR_SWAP_MKDIR;
         }

         # Now we must ensure the access authority allows the java
         # machine to see the directory (we created it running as root).

         my $mcfDir = "";

         if ($rc == $ERROR_NO_ERROR)
         {
            # Find the directory after /console/mcf/.
            $i = index($dir, "/");              # before console
            $i = index($dir, "/", $i+1);        # after console
            $i = index($dir, "/", $i+1);        # after mcf
            $i = index($dir, "/", $i+1);        # after the specific MCF dir
            $mcfDir = substr($dir, 0, $i);
            trace("$TRC_D    $this - MCF directory is $mcfDir");
         }

         # Change owner, group, and flags on the directory and all
         # contained directories.

         if ($rc == $ERROR_NO_ERROR)
         {
            my $cmd = "chown 'hmcmanager:nobody' '$mcfDir' 2>&1";
            my $result = `$cmd`;
            unless (defined($result)) { $result = ""; }
            chomp $result;
            my $trc = $? >> 8;
            if ($trc != 0)
            {
               # Had a problem. Shouldn't happen. This is bad.
               trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
               $rc = $ERROR_SWAP_NEWDIR_OWNER;
            }
         }

         if ($rc == $ERROR_NO_ERROR)
         {
            my $cmd = "chmod 775 '$mcfDir' 2>&1";
            my $result = `$cmd`;
            unless (defined($result)) { $result = ""; }
            chomp $result;
            my $trc = $? >> 8;
            if ($trc != 0)
            {
               # Had a problem. Shouldn't happen. This is bad.
               trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
               $rc = $ERROR_SWAP_NEWDIR_PERMS;
            }
         }

         if ($rc == $ERROR_NO_ERROR)
         {
            my $cmd = "chown -R 'hmcmanager:nobody' '$mcfDir/' 2>&1";
            my $result = `$cmd`;
            unless (defined($result)) { $result = ""; }
            chomp $result;
            my $trc = $? >> 8;
            if ($trc != 0)
            {
               # Had a problem. Shouldn't happen. This is bad.
               trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
               $rc = $ERROR_SWAP_NEWSUBDIR_OWNER;
            }
         }

         if ($rc == $ERROR_NO_ERROR)
         {
            my $cmd = "chmod -R 775 '$mcfDir/' 2>&1";
            my $result = `$cmd`;
            unless (defined($result)) { $result = ""; }
            chomp $result;
            my $trc = $? >> 8;
            if ($trc != 0)
            {
               # Had a problem. Shouldn't happen. This is bad.
               trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
               $rc = $ERROR_SWAP_NEWSUBDIR_FLAGS;
            }
         }
      }
   }

   # Note if the target file exists and if so, save its permission information.

   my $targetExists = $FALSE;

   if ($rc == $ERROR_NO_ERROR)
   {
      if ( -f "$target" )
      {
         trace("$TRC_F    $this - target file does exist");
         $targetExists = $TRUE;

         ($owner, $group, $perms) = getPerms($target);
      }
   }

   # Move the file with    mv -f    command.

   if ($rc == $ERROR_NO_ERROR)
   {
      my $error = `mv -f '$source' '$target' 2>&1`;
      unless (defined($error)) { $error = ""; }
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $this() - mv -f error is \"$error\"");
         $rc = $ERROR_SWAP_MV_FAILURE;
      }
      else
      {
         # If we successfully performed any file movement operation, we need
         # to reboot when we are done with all the rest of the work.

         $rebootRequired = $TRUE;
      }
   }

   # Put the old permissions on the replacement file.
   # The timestamp should not have changed.

   if ( ($rc == $ERROR_NO_ERROR) && ($targetExists) )
   {
      $rc = processPermissionsForOneFile($target, $owner, $group, $perms);
      if ($rc != $ERROR_NO_ERROR)
      {
         trace("$TRC_F    $this - unable to reassign permission information");
         $rc = $ERROR_SWAP_PERMS_FAILURE;
      }
   }

   # Any logging is done at the top level of swap file processing

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}
# end -04

##############################################################################
#####                          LINK PROCESSING                           #####
##############################################################################

#
#
# start -01
#
sub doLinkProcessing
{
   my $this = $me . ".doLinkProcessing()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;
   my $filename = "iqzmLinks.dat";

   my $path = queryFileLocation($filename);
   my $datafile = $path . $filename;
   # Change any "//" to "/"
   $datafile =~ s/\/+/\//g;
   trace("$TRC_F    $this - links data file is $datafile");

   # Does the file exist?
   if ( -f "$datafile" )
   {
      unless (open(DATAFILE, $datafile))
      {
         trace("$TRC_F    $this - error opening links data file");
         $rc = $ERROR_LINKS_FILE_ACCESS;
      }

      if ($rc == $ERROR_NO_ERROR)
      {
         trace("$TRC_D    $this - reading links data file");
         my @mydata = <DATAFILE>;

         trace("$TRC_D    $this - closing links data file");
         close (DATAFILE);

         $rc = processLinks(@mydata);
      }

      # Copy file to .err version if error
      if ($rc != $ERROR_NO_ERROR)
      {
         $target = $datafile . ".err";
         `cp -f $datafile $target`;
      }

      # Add file to file history                                                #-05
      updateHistory($datafile);                                                 #-05

      if ($rc != $ERROR_NO_ERROR)
      {
         # Our processing failed, so we may not be able to access the files
         # where links were to be changed.
         # This is fatal.
         trace("$TRC_F    $this - Failure updating file links");
         `iqzmlogerror 0xE566 $rc`;                                             #-05
         setHozd("$this - Unable to update file links");
      }
   }
   else
   {
      # There is nothing to do
      trace("$TRC_F    $this - No links data file was found");
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

#
#
#
sub processLinks
{
   my $this = $me . ".processLinks()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my @lines = @_;

   my $command = "";
   my $alias   = "";
   my $target  = "";
   my $type    = "";

   my $key     = "";
   my $value   = "";
   my $line    = "";

   my $contentError = $FALSE;

   foreach $line (@lines)
   {
      # Remove the eol
      chomp($line);
      trace("$TRC_D    $this - line is [$line]");

      # Break the line into a key and its value
      ($key, $value) = $line =~ /^<(.*?)>(.*)/;
      # These elements can end up undefined, so handle that to
      # avoid runtime error messages
      unless (defined($key))   { $key = "";   }
      unless (defined($value)) { $value = ""; }

      trace("$TRC_D    $this - key is [$key], value is [$value]");
      if ($key eq "cmd")
      {
         trace("$TRC_D    $this - command is [$value]");
         my $newcommand = $value;

         # process the data for the previous file
         if ($command ne "")
         {
            my $trc = processOneLink($command, $alias, $target, $type);
            # Save the rc if it represents an error
            if ($trc != $ERROR_NO_ERROR)
            {
               $rc = $trc
            }
         }
         $command = $newcommand;
         $alias   = "";
         $target  = "";
         $type    = "";
      }
      elsif ($key eq "link")                                                    #-03
      {
         trace("$TRC_D    $this - alias is [$value]");
         my $newalias = $value;
         if ($command eq "")
         {
            trace("$TRC_F    $this - no command defined - alias value ignored [$newalias]");
            $contentError = $TRUE;
         }
         else
         {
            if ($alias ne "")
            {
               trace("$TRC_F    $this - multiple alias records for $command - using last value [$newalias]");
               $contentError = $TRUE;
            }
            $alias = $newalias;
         }
      }
      elsif ($key eq "target")
      {
         trace("$TRC_D    $this - target is [$value]");
         my $newtarget = $value;
         if ($command eq "")
         {
            trace("$TRC_F    $this - no command defined - target value ignored [$newtarget]");
            $contentError = $TRUE;
         }
         else
         {
            if ($target ne "")
            {
               trace("$TRC_F    $this - multiple target records for $command - using last value [$newtarget]");
               $contentError = $TRUE;
            }
            $target = $newtarget;
         }
      }
      elsif ($key eq "type")
      {
         trace("$TRC_D    $this - type are [$value]");
         my $newtype = $value;
         if ($command eq "")
         {
            trace("$TRC_F    $this - no command defined - type value ignored [$newtype]");
            $contentError = $TRUE;
         }
         else
         {
            if ($type ne "")
            {
               trace("$TRC_F    $this - multiple type records for $command - using last value [$newtype]");
               $contentError = $TRUE;
            }
            $type = $newtype;
         }
      }
      else
      {
         trace("$TRC_F    $this - unrecognizable line [$line]");                #-05
         $contentError = $TRUE;
      }
   }

   # process the data for the previous (last) command
   if ($command ne "")
   {
      my $trc = processOneLink($command, $alias, $target, $type);
      # Save the rc if it represents an error
      if ($trc != $ERROR_NO_ERROR)
      {
         $rc = $trc
      }
   }

   # If there has been no other error so far, make a note of any content error
   if ($rc == $ERROR_NO_ERROR)
   {
      if ($contentError == $TRUE)
      {
         $rc = $ERROR_LINKS_CONTENT_ERROR;
      }
   }

   # Any logging is done at the top level of link processing

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

#
#
#
sub processOneLink
{
   my $this = $me . ".processOneLink()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my ($command, $alias, $target, $type) = @_;

   trace("$TRC_F    $this - parms are [$command], [$alias], [$target], [$type]");

   # Check the command

   if ($command eq "create")
   {
      # Call the subroutine to do create processing
      my $rc = createOneLink($alias, $target, $type);
   }
   elsif ($command eq "remove")
   {
      # Call the subroutine to do remove processing
      my $rc = removeOneLink($alias);
   }
   else
   {
      trace("$TRC_F    $this - command ($command) is not recognized");
      $rc = $ERROR_LINKS_BAD_COMMAND;
   }

   # Any logging is done at the top level of link processing

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

#
#
#
sub removeOneLink
{
   my $this = $me . ".removeOneLink()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my ($alias) = @_;

   trace("$TRC_F    $this - parms are [$alias]");

   # Ensure the target exists
   if ( -f "$alias" )
   {
      my $error = `rm -f '$alias' 2>&1`;                                        #-04
      unless (defined($error)) { $error = ""; }
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $this - rm error deleting link - \"$error\"");
         $rc = $ERROR_LINKS_REMOVE_FAILURE;
      }
   }
   else
   {
      # There is nothing to do
      trace("$TRC_F    $this - No alias was found to remove");
      $rc = $ERROR_LINKS_NO_ALIAS;
   }

   # Any logging is done at the top level of link processing

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

#
#
#
sub createOneLink
{
   my $this = $me . ".createOneLink()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my ($alias, $target, $type) = @_;
   my $opts = "";

   trace("$TRC_F    $this - parms are [$alias], [$target], [$type]");

   # Check for symbolic or hard link
   if ( $type eq "911" )
   {
      $opts = "-s";
   }
   elsif ( $type eq "912" )
   {
      $opts = "";
   }
   else
   {
      trace("$TRC_F    $this - unrecognizable type value: $type");              #04
      $rc = $ERROR_LINKS_BAD_TYPE;
   }

   # Ensure the target exists
   if ( -f "$target" )
   {
      if ( -e "$alias" )
      {
         # There is already some file present where the alias is
         # supposed to go
         trace("$TRC_F    $this - File already exists where alias should go");
         $rc = $ERROR_LINKS_ALIAS_EXISTS;
      }
      else
      {
         my $error = `ln $opts '$target' '$alias' 2>&1`;                        #-04
         unless (defined($error)) { $error = ""; }
         chop $error;
         $rc = $? >> 8;
         if ($rc)
         {
            trace("$TRC_F    $this - ln error creating link - \"$error\"");
            $rc = $ERROR_LINKS_CREATE_FAILURE;
         }
      }
   }
   else
   {
      # The target doesn't exist
      trace("$TRC_F    $this - The link target does not exist");
      $rc = $ERROR_LINKS_NO_TARGET;
   }

   # Any logging is done at the top level of link processing

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}
# end -01

##############################################################################
#####                        PERMISSION CHANGING                         #####
##############################################################################

#
#
#
sub doPermissionChanging
{
   my $this = $me . ".doPermissionChanging()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;
   my $filename = "iqzmPermissions.dat";

   my $path = queryFileLocation($filename);
   my $datafile = $path . $filename;                                            #-01
   # Change any "#" to "/"
   $datafile =~ s/\/+/\//g;                                                     #-01
   trace("$TRC_F    $this - permissions data file is $datafile");               #-01

   # Does the file exist?
   if ( -f "$datafile" )                                                        #-01
   {
      unless (open(DATAFILE, $datafile))                                        #-01
      {
         trace("$TRC_F    $this - error opening permissions data file");
         $rc = $ERROR_PERMS_FILE_ACCESS;
      }

      if ($rc == $ERROR_NO_ERROR)
      {
         trace("$TRC_D    $this - reading permissions data file");
         my @mydata = <DATAFILE>;                                               #-01

         trace("$TRC_D    $this - closing permissions data file");
         close (DATAFILE);                                                      #-01

         $rc = processPermissions(@mydata);                                     #-01
      }

      # Copy file to .err version if error
      if ($rc != $ERROR_NO_ERROR)
      {
         $target = $datafile . ".err";                                          #-01
         `cp -f $datafile $target`;                                             #-01
      }

      # Add file to file history                                                #-05
      updateHistory($datafile);                                                 #-05

      if ($rc != $ERROR_NO_ERROR)
      {
         # Our processing failed, so we may not be able to access the files
         # where permissions were to be changed.
         # This is fatal.
         trace("$TRC_F    $this - Failure updating file permissions");
         `iqzmlogerror 0xE564 $rc`;                                             #-05
         setHozd("$this - Unable to update file permissions");
      }
   }
   else
   {
      # There is nothing to do
      trace("$TRC_F    $this - No permissions data file was found");
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

#
#
#
sub processPermissions
{
   my $this = $me . ".processPermissions()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my @lines = @_;

   my $file    = "";
   my $owner   = "";
   my $group   = "";
   my $flags   = "";
   my $key     = "";
   my $value   = "";

   my $contentError = $FALSE;

   foreach $line (@lines)
   {
      # Remove the eol
      chomp($line);
      trace("$TRC_D    $this - line is [$line]");

      # Break the line into a key and its value
      ($key, $value) = $line =~ /^<(.*?)>(.*)/;
      # These elements can end up undefined, so handle that to
      # avoid runtime error messages
      unless (defined($key))   { $key = "";   }
      unless (defined($value)) { $value = ""; }

      trace("$TRC_D    $this - key is [$key], value is [$value]");
      if ($key eq "file")
      {
         trace("$TRC_D    $this - file is [$value]");
         my $newfile = $value;

         # process the data for the previous file
         if ($file ne "")
         {
            my $trc = processPermissionsForOneFile($file, $owner, $group, $flags);
            # Save the rc if it represents an error
            if ($trc != $ERROR_NO_ERROR)
            {
               $rc = $trc
            }
         }
         $owner   = "";
         $group   = "";
         $flags   = "";
         $file = $newfile;

      }
      elsif ($key eq "owner")
      {
         trace("$TRC_D    $this - owner is [$value]");
         my $newowner = $value;
         if ($file eq "")
         {
            trace("$TRC_F    $this - no file defined - owner value ignored [$newowner]");
            $contentError = $TRUE;
         }
         else
         {
            if ($owner ne "")
            {
               trace("$TRC_F    $this - multiple owner records for $file - using last value [$newowner]");
               $contentError = $TRUE;
            }
            $owner = $newowner;
         }
      }
      elsif ($key eq "group")
      {
         trace("$TRC_D    $this - group is [$value]");
         my $newgroup = $value;
         if ($file eq "")
         {
            trace("$TRC_F    $this - no file defined - group value ignored [$newgroup]");
            $contentError = $TRUE;
         }
         else
         {
            if ($group ne "")
            {
               trace("$TRC_F    $this - multiple group records for $file - using last value [$newgroup]");
               $contentError = $TRUE;
            }
            $group = $newgroup;
         }
      }
      elsif ($key eq "flags")
      {
         trace("$TRC_D    $this - flags are [$value]");
         my $newflags = $value;
         if ($file eq "")
         {
            trace("$TRC_F    $this - no file defined - flags value ignored [$newflags]");
            $contentError = $TRUE;
         }
         else
         {
            if ($flags ne "")
            {
               trace("$TRC_F    $this - multiple flags records for $file - using last value [$newflags]");
               $contentError = $TRUE;
            }
            $flags = $newflags;
         }
      }
      else
      {
         trace("$TRC_F    $this - unrecognizable line [$line]");                #-05
         $contentError = $TRUE;
      }
   }

   # process the data for the previous (last) file
   if ($file ne "")
   {
      my $trc = processPermissionsForOneFile($file, $owner, $group, $flags);
      # Save the rc if it represents an error
      if ($trc != $ERROR_NO_ERROR)
      {
         $rc = $trc
      }
   }

   # If there has been no other error so far, make a note of any content error
   if ($rc == $ERROR_NO_ERROR)
   {
      if ($contentError == $TRUE)
      {
         $rc = $ERROR_PERMS_CONTENT_ERROR;
      }
   }

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

#
#
#
sub processPermissionsForOneFile
{
   my $this = $me . ".processPermissionsForOneFile()";
   trace("$TRC_T -> $this");

   my $rc = $ERROR_NO_ERROR;

   my ($file, $owner, $group, $flags) = @_;

   trace("$TRC_F    $this - parms are [$file], [$owner], [$group], [$flags]");

   my $actionRequired = $FALSE;

   # Does the file exist? It could happen that a file is both added and         #-02
   # deleted in the same file swapping session. If so, then there is no         #-02
   # file left to process with the permissions. This is not an error, just      #-02
   # unusual, so we need to account for it.                                     #-02
   if ( -f "$file" )                                                            #-02
   {                                                                            #-02
      # Change the owner
      if ($owner ne "")
      {
         $actionRequired = $TRUE;
         my $cmd = "chown '$owner' '$file' 2>&1";                               #-04
         my $result = `$cmd`;
         unless (defined($result)) { $result = ""; }                            #-05
         chomp $result;                                                         #-05
         my $trc = $? >> 8;
         if ($trc != 0)
         {
            # Had a problem. Shouldn't happen. This is bad.
            trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
            $rc = $ERROR_PERMS_OWNER_FAILURE;
         }
         else
         {
            trace("$TRC_D    $this - command is [$cmd], result is [$result]");
         }
      }

      # Change the group
      if ($rc == $ERROR_NO_ERROR)
      {
         if ($group ne "")
         {
            $actionRequired = $TRUE;
            my $cmd = "chown ':$group' '$file' 2>&1";                           #-04
            my $result = `$cmd`;
            unless (defined($result)) { $result = ""; }                         #-05
            chomp $result;                                                      #-05
            my $trc = $? >> 8;
            if ($trc != 0)
            {
               # Had a problem. Shouldn't happen. This is bad.
               trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
               $rc = $ERROR_PERMS_GROUP_FAILURE;
            }
            else
            {
               trace("$TRC_D    $this - command is [$cmd], result is [$result]");
            }
         }
      }

      # Change the permission bits
      if ($rc == $ERROR_NO_ERROR)
      {
         if ($flags ne "")
         {
            $actionRequired = $TRUE;
            my $cmd = "chmod $flags '$file' 2>&1";                              #-04
            my $result = `$cmd`;
            unless (defined($result)) { $result = ""; }                         #-05
            chomp $result;                                                      #-05
            my $trc = $? >> 8;
            if ($trc != 0)
            {
               # Had a problem. Shouldn't happen. This is bad.
               trace("$TRC_F    $this - command is [$cmd], rc = $trc, result is [$result]");
               $rc = $ERROR_PERMS_FLAGS_FAILURE;
            }
            else
            {
               trace("$TRC_D    $this - command is [$cmd], result is [$result]");
            }
         }
      }

      # Was there something to do?
      if ($actionRequired == $FALSE)
      {
         trace("$TRC_F    $this - no action was specified for file $file");
         `iqzmlogerror 0xE565 $ERROR_PERMS_NO_ACTION`;                          #-05
         # This is not a fatal error.
      }
      # start -07
      else
      {
         # Update the file's timestamp so mirroring and backup will pick up the
         # change.
         if ($ERROR_NO_ERROR == $rc)                                            #-08
         {
            my $error = `touch -m '$file' 2>&1`;
            unless (defined($error)) { $error = ""; }
            chop $error;
            $rc = $? >> 8;
            if ($rc)
            {
               trace("$TRC_F    $this - touch error is \"$error\"");
               $rc = $ERROR_PERMS_TOUCH;
            }
         }
      }
      # end -07
   }                                                                            #-02
   else                                                                         #-02
   {                                                                            #-02
      trace("$TRC_F    $this - the file $file was not present");                #-02
   }                                                                            #-02
                                                                                #-02

   trace("$TRC_T <- $this - rc = $rc");
   return($rc);
}

##############################################################################
#####                          MAINLINE CODE                             #####
##############################################################################

# Process the arguments
GetOptions(
      'help'            =>\$help,
      'tracecommand:s'  =>\$traceCmd,
);

unless (defined($traceCmd )) { $traceCmd = ""; }                                #-01
                                                                                #-01
trace("$TRC_T -> $me(@ARGV)");

my $rc = $ERROR_NO_ERROR;

unless($> == 0)
{
   trace("$TRC_F    $me() - this must be run with root authority");
   $rc = $ERROR_NOT_ROOT;
   `iqzmlogerror 0xE560 $rc`;                                                   #-05
}

if ($help)
{
   usage();
   $rc = $ERROR_HELP;
}

# Run each of the functions, unless an error interferes.

if ($rc == $ERROR_NO_ERROR)
{
   $rc = doFileSwapping();
}

if ($rc == $ERROR_NO_ERROR)
{
   $rc = doLinkProcessing();
}

if ($rc == $ERROR_NO_ERROR)
{
   $rc = doPermissionChanging();
}

# start -06
if ($rc == $ERROR_NO_ERROR)
{
   my $cmd = "iqzmPatchUnmounted.pl 2>&1";
   my $result = `$cmd`;
   my $trc = $? >> 8;
   # start -07
   trace("$TRC_F    $me() - unmounted partition updating completed, rc = $trc");
   if ($trc == 0)
   {
      trace("$TRC_F    $me() - reboot required");

      # We made changes; we need to reboot.
      $rebootRequired = $TRUE;
   }
   elsif ($trc == 1)
   {
      trace("$TRC_F    $me() - there were no changes to any unmounted partitions");
   }
   else
   {
      # A return code of 0 means success and 1 means nothing to do.
      # Had a problem. Shouldn't happen. This is bad.
      # Any error is already logged.

      trace("$TRC_F    $me() - command is [$cmd], rc = $trc, result is [$result]");
      $rc = $ERROR_UPDATE_FAILED;
   }
   # end -07
}
# end -06

# start -05
if ($rebootRequired)
{
   trace("$TRC_F    $me() - rebooting ....");
   trace("$TRC_T <- $me() - rc = $rc");

   # Give time for tracing to complete
   sleep(5);
   `sync`;

   `reboot`;

   trace("$TRC_F    $me() - issued reboot command - going to sleep ....");

   # Wait 10 minutes for the reboot, then fall through.

   `sleep 600`;

   trace("$TRC_F    $me() - uh-oh - reboot did not occur (waited 10 minutes)");
   `iqzmlogerror 0xE568`;
}


trace("$TRC_T <- $me() - rc = $rc");

# Give time for tracing to complete in case reboot follows exit
sleep(5);
`sync`;

# end -05

exit($rc);

