#!/bin/bash

# Check parameters
if [ $# -ne 7 -a $# -ne 8 ]; then
    echo "Usage: firewallCtl.sh <hmc directory> <interface name> <source address> <source mask> <protocol> <port|icmp-type> [<appid>] on|off|query"
    exit 1
fi

# Grab parameters
hmcdir=$1
ifname=$2
sourceAddr=$3
sourceAddrMask=$4
protocol=$5
port=$6
if [ $# -eq 8 ]; then
    appid=$7
    case $8 in
        on) op="--append" ;;
        off) op="--delete" ;;
        query) line="$appid|$sourceAddr|$sourceAddrMask"
               file="$hmcdir/data/FirewallSettings.$ifname-NETAPP-INPUT"
               if grep $line $file > /dev/null 2>&1; then
                   echo "$appid is ON"
                   exit 0
               else
                   echo "$appid is OFF"
                   exit 1
               fi
               ;;
        *) echo "invalid operator $8: must be on|off|query"
           exit 4 ;;
    esac
else
    case $7 in
        on) op="--append" ;;
        off) op="--delete" ;;
        query) echo "query requires an appid."
               exit 5 ;;
        *) echo "invalid operator $7: must be on|off|query"
           exit 4 ;;
    esac
fi

# Search for iptables executable
if [ -x /sbin/iptables ]; then
    iptables=/sbin/iptables
elif [ -x /usr/sbin/iptables ]; then
    iptables=/usr/sbin/iptables
else
    echo "No iptables executable found"
    actzTrace "XSETNETF: enableService.sh: No iptables executable found"
    exit 2
fi

# Build the chain name
if [ "$appid" = "" ]; then
    chain="$ifname-HMC-INPUT"
else
    chain="$ifname-NETAPP-INPUT"
fi

# Create the chain if it doesn't already exist
if $iptables -L -n | grep $chain >/dev/null 2>&1; then
    actzTrace "XSETNETF: found the chain.. no need to create it"
else
    actzTrace "XSETNETF: creating $chain chain"
    $iptables --new $chain && \
    $iptables --insert INPUT --in-interface $ifname --jump $chain
    rc=$?
    actzTrace "XSETNETF: return code from create chain command: $rc"
fi

# Build the command to execute
if [ "$protocol" = "icmp" ]; then
    cmd="$iptables $op $chain --in-interface $ifname --protocol $protocol --source $sourceAddr/$sourceAddrMask --icmp-type $port --jump ACCEPT"
elif echo $port | grep ","; then
    # there are multiple ports specified.. use the 'multiport' module
    cmd="$iptables $op $chain --match multiport --in-interface $ifname --protocol $protocol --source $sourceAddr/$sourceAddrMask --destination-ports $port --jump ACCEPT"
else
    cmd="$iptables $op $chain --in-interface $ifname --protocol $protocol --source $sourceAddr/$sourceAddrMask --destination-port $port --jump ACCEPT"
fi

# Execute the command
actzTrace "XSETNETF: executing cmd: '$cmd'"
$cmd
rc=$?

# Check the return code
if [ $rc -ne 0 ]; then
    echo "Error executing iptables command: $rc"
    exit 3
fi

# if we are deleting, make sure we delete all matching rules
if [ $op = "--delete" ]; then
    while [ $rc -eq 0 ]; do
        actzTrace "XSETNETF: executing cmd: '$cmd'"
        $cmd > /dev/null 2>&1
        rc=$?
    done
fi

# if this is a NetApp, modify the FirewallSettings file
if [ "$appid" != "" ]; then
    file="$hmcdir/data/FirewallSettings.$chain"
    if [ $op = "--delete" ]; then
        sed $file -e "/^$appid|$sourceAddr|$sourceAddrMask/d" > "$file.new"
        mv -f "$file.new" $file
    else
        echo "$appid|$sourceAddr|$sourceAddrMask" >> $file
    fi
    chmod 664 $file
fi

exit 0
