#!/usr/bin/perl -w
##
## rcsControl.pl
##
##
## (c)  Copyright  IBM Corp. 2003, 2004, 2005  All Rights Reserved
##
## US Government Users Restricted Rights - Use, duplication or
## disclosure restricted by GSA ADP Schedule Contract with IBM Corp
##
##  Notes:
##
##  Module History:
##    10/15/2003  Stapels    Initial release.
##    ??/??/2003  Stapels    Numerous fixes, changes, etc.
##    01/14/2004  Stapels    Added an internal retry for initial ipsec initiate
##                           since whack gets release before ipsec has really
##                           given up. (-01)
##    07/28/2005  J.Stapels  Plethora of changes over the past month, all were
##                           minor variations to timing and small fixes for
##                           L2TP related connections.
##    10/27/2005  J.Stapels  Added command-line support to setup port forwarding
##                           so Megamouth can connect to PHYP, etc.
##    06/15/2006  Stapels    Added support for D-NAT (-02)
##    06/30/2006  S. Lyons   Monitor Modem Line item -03
##    09/15/2006  S. Lyons   Allow VPN selectable interface and route for ipsec communication -04
##    10/30/2006  S. Lyons   (aix cmvc 573751) Defined ip route disappears during pluto shutdown and restart.
##    11/02/2006  S. Lyons   (aix cmvc 577511) Call Home using VPN changes the default gateway
##    01/06/2007  S. Lyons  Changes to ATT PAP authentication for China. ODT I7562,
##                          zhmc defect i7562, aix defect 580648, and phmc defect A580648,
##                          there may be clearquest numbers associated with this change,
##                          but at present they are beyond my mental capability.
##    01/24/2007  S.Lyons   Failover to modem does not work with inbound connectivity. -05
##    07/20/2007  J.Stapels  Add in redirect rule so connections to port 443 go to 8443. (-06)
##
## This script is used by the RCS package on the HMC to setup and activate
## various call-home functions. This script is ugly, and eventually all the
## functionality in it should move up into the JVM for the HMC.
##

use warnings;
use strict;


#
# Includes
#
use POSIX "setsid";
use POSIX ":sys_wait_h";
use Fcntl ":flock";
use Getopt::Long;
use Socket;

##############################################################################
#####                          GLOBAL VARIABLES                          #####
##############################################################################

# Runtime Script Location.
my ($dir, $me) = $0 =~ /^(?:(.*)\/)?(.*)$/;

# Config Data
my $CONFIG_FILE    = "rcsTempConfig.props";

# Program Data
my $PATH           = "$ENV{PATH}:/sbin:/usr/sbin:/usr/local/sbin:/opt/hsc/bin:/opt/ccfw/bin/framework";
my $PS             = "/bin/ps";
my $ROUTE          = "/sbin/route";
my $IFCONFIG       = "/sbin/ifconfig";
my $KILL           = "/bin/kill";
my $IP             = "/sbin/ip";

my $RUN            = "/var/run";
my $PID_FILE       = "$RUN/rcs.pid";
my $FIFO_DAEMON    = "$RUN/rcs-daemon.fifo";
my $FIFO_CLIENT    = "$RUN/rcs-client.fifo";

sub PPP_PIDFILE         { return( "/var/run/ppp-" . shift() . ".pid" ); }
my $DIAL_LINKNAME       = "rcs-dial";
my $REM_DIAL_LINKNAME   = "rcs-rem-dial";
my $VPN_LINKNAME        = "rcs-vpn";
my $TUN_LINKNAME        = "rcs-vpn-tun";
my $REM_VPN_LINKNAME    = "rcs-rem-vpn";
my $REM_VPN_BKP_LINKNAME    = "rcs-rem-bkp";
my $RS_LINKNAME         = "rcs-rs";

my $RC_GENERAL_ERR = 100;
my $RC_DIAL_ERR    = 101;
my $RC_VPN_ERR     = 102;
my $RC_AUTH_ERR    = 119;

my $ALARM          = "alarm\n";

my $FIFO_TIMEOUT   = 600;

# Sets the maximum segment size for tcp connections over the routes this
# script creates. This relativily low value is used to circumvent a bug that
# was found in the Cisco VPN Gateway where is does not properly handle
# fragmenting packets.
my $MAX_MSS = 1200;

# This is the MSS value above minus 40 bytes (for TCP/IP header).
my $SYN_MSS = 1160;

# Daemon Globals
my $is_daemon = 0;

# Global Declarations;
my ($AUTH_FILE, $TEMPLATE_DIR, $LOG_FILE);
my ($PPPD,  $PPPD_OPT,    $PPPD_PARMS, %PPPD_CONFIG);
my ($IPSEC, $IPSEC_RETRY, $IPSEC_CONF, $IPSEC_PARMS, $IPSEC_LOCK, $IPSEC_LOCK2, %IPSEC_CONFIG);
my ($L2TPD, $L2TPD_RETRY, $L2TPD_CONF, $L2TPD_PARMS, $L2TPD_CONTROL, %L2TPD_CONFIG);
my ($VSLIP, $IPTABLES);
my ($MODEM_ANSWER, $MODEM_ANSWER_PARMS);
my ($MONITOR_MODEM, $MONITOR_MODEM_PARMS);
my ($STOP_MODEM);     # (-03)

my $JAVA;
my $FS_PATH = "/usr/websm/codebase/pluginjars";
my $FS_ARGS = "-cp $FS_PATH/hsc.jar:$FS_PATH/sniacimom.jar:$FS_PATH/ccfw.jar com.ibm.hsc.filesvr.FsServer";

my %dial;
my %vpn;
my %l2tp;
my %service;
my %config;
my %tpt_data;
my $trace_buffer;
my $error_data;
my $dial_return_code;

my $SOCK_TIMEOUT = 60;

# Descriptions of the different types of records to expect from the remote
# support server. Used for debug purposes.
my %RECORD_TYPE = (
    0x0001 => "Startup record from the HMC.",
    0x8001 => "Startup response from RS3.",
    0x0002 => "HMC query remote connection active status.",
    0x8002 => "Response to query remote connection active status.",
    0x0003 => "HMC disconnecting."
    );

# Descriptions of the different types of fields to expect from the remote
# support server. Used for dumping the contents of a record in a readable
# format.
my %FIELD_TYPE = (
    0x0101 => "Machine Type",
    0x0102 => "Model",
    0x0103 => "Serial Number",
    0x0104 => "Assigned IP Address",
    0x0105 => "Access Type",
    0x0106 => "CEC Machine Type : Serial Number",
    0x0107 => "Dial Connection",
    0x0108 => "Dial Account Name",
    0x0109 => "Dial Account User ID",
    0x010A => "Region",
    0x010B => "ISO Country Code",
    0x010C => "Timeout Value",
    0x010D => "IP Address for RCHASRS3",
    0x010E => "Port for RCHASR3",
    0x010F => "Return Code",
    0x0110 => "Partition ID + IP Address",
    0x0111 => "Conection Status",
    0x0112 => "Allow HMC Access",
    );

my $INITIAL_GW_TO_VPN_SERVER ;
my $INITIAL_DEV_TO_VPN_SERVER ;
my $INITIAL_VPN_SERVER ;

# UPDATE LOCALE !!! This script depends on output being in a certain format.
# If the locale changes, certain apects of this daemon will fail.
$ENV{'LANG'}="POSIX";


##############################################################################
#####                          UTILITY METHODS                           #####
##############################################################################

#
# pause
#
# A simple pause function.
#
sub pause
{
    #sleep( shift or 1 );
    select( undef, undef, undef, (shift or 1) );
}

#
# trace
#
# Used to trace out messages to a log file and the HMC trace utility.
#
sub trace
{
    my $text = shift;
    chomp( $text );

    if( $ENV{'CONSOLE_PATH'} )
    {
        for( split( /\n/, $text ) )
        {
            $_ = ($is_daemon ? 'D' : 'C') . " $_";
            system( "actzTrace", "XRCSZPLT: $_" );
        }
    }

    my ($sec, $min, $hour, $day, $mon, $year) = localtime();
    my $stamp =
        sprintf(
            "%04d-%02d-%02d %02d:%02d:%02d $me %s",
            $year+1900, $mon+1, $day, $hour, $min, $sec, $is_daemon ? "D" : "C"
            );

    $text =~ s/\n/\n$stamp: /g;

    unless( defined( $LOG_FILE ) )
    {
        $trace_buffer .= "$stamp: $text\n";
        return;
    }

    if( -f $LOG_FILE )
    {
        my $filesize = (stat( $LOG_FILE ))[7];

        if( $filesize > (10*1048576) )
        {
            for( my $i = 9; $i > 1; $i-- ) {
                rename( "$LOG_FILE." . $i, "$LOG_FILE." . ($i + 1) );
            }
            rename( $LOG_FILE, "$LOG_FILE.1" );
        }

    }

    unless( open( FILE, ">> $LOG_FILE" ) ) {
        print( STDERR "Unable to open log $LOG_FILE: $!\n" );
        return;
    }

    if( defined( $trace_buffer ) )
    {
        print( FILE $trace_buffer );
        undef( $trace_buffer );
    }

    print( FILE "$stamp: $text\n" );
    close( FILE );

}

#
# warning
#
# Traces out a warning and captures it to be returned to the client.
#
sub warning
{
    my $warn_text = shift;

    chomp( $warn_text );
    trace( "WARNING! $warn_text\n" );
    $error_data .= "$warn_text\n";
}

#
# error
#
# Trace out an error and captures it to be return to the client.
#
sub error
{
    my $error_text = shift;

    chomp( $error_text );
    trace( "ERROR! $error_text\n" );
    $error_data .= "$error_text\n";
}

#
# error_dump
#
# Grabs all stored warning and errors.
#
sub error_dump
{
    my $dump = $error_data;
    undef( $error_data );
    return( $dump );
}

#
# usage
#
sub usage
{
    print( <<EOM );
Usage:
    $me

    TODO!

EOM
}

#
# daemon_exists
#
# Checks to see if the daemon is already running.
#
sub daemon_exists
{
    trace( "<-> daemon_exists(@_)" );
    unless( -f $PID_FILE and -p $FIFO_DAEMON ) {
        return( 0 );
    }

    return( is_running( get_pid( $PID_FILE ) ) );
}

#
# client_exists
#
# Checks to see if a client is already running.
#
sub client_exists
{
    unless( -p $FIFO_CLIENT ) {
        return( 0 );
    }

    return( 1 );
}


#
# get_default_gw
#
# Gets the default gateway for this machine.
#
sub get_default_gw
{
    trace( "--> get_default_gw(@_)" );

    my ($gw, $dev);
    for( `$ROUTE -n` )
    {
        if( /^0.0.0.0\s+(\S+)\s+(?:\S+\s+){5}(\S+)/ ) {
            $gw = $1;
            $dev = $2;
            last;
        }
    }
    unless( $gw ) {
        trace( "    Error: No default gateway found" );
    }

    trace( "<-- get_default_gw() [$gw, $dev]" );
    return( wantarray() ? ( $gw, $dev ) : $gw );
}


# -04
# get_route_to_sever
#
# Gets the route to the VPN server from this machine.
#
sub get_route_to_server
{
    trace( "--> get_route_to_server(@_)" );

    my ($route, $dev);

    #There are two formats of the response to the 'ip route get' command,
    #when the queried address is on the same subnet as the source address
    #and one when they are on different subnets.
    #different subnet   207.25.252.196 via 9.60.73.1 dev eth0  src 9.60.73.194
    #                   cache  mtu 1500 advmss 1460 metric10 64
    #same subnet        9.60.73.200 dev eth0  src 9.60.73.194
    #                   cache  mtu 1500 advmss 1460 metric10 64

    trace("VPN server address is $config{vpn_server}");

    for (`$IP route get $config{vpn_server}`) {
       trace("ip command output = $_");
       # if via is found, the next token is the route address and if dev is found,
       # the next token is the device
       if ( m/via\s+(\S+)\s+dev\s+(\S+)\s+/ ) {
          #The server is not on the same subnet as the source.
          $route = $1;
          $dev   = $2;
       } else {
         # This is the case where the VPN server and source are on the
         # same subnet. The default gateway will be returned.
         ($route, $dev) = get_default_gw();
         trace( "The VPN server is on the same subnet as the source." );
       }
        last;
    }
    unless( $route ) {
        trace( "    Error: No route to server found." );
    }

    trace( "<-- get_route_to_server() [ $config{vpn_server}, $route, $dev]" );
    return( wantarray() ? ( $route, $dev ) : $route );
}

#
# ifconfig
#
# Gets information for a specific device and stored it into a hash.
#
sub ifconfig ($\%)
{
    trace( "<-> ifconfig(@_)" );

    my $dev = shift;
    my $ref = shift;

    my @ret = `$IFCONFIG $dev`;
    if( $? ) { return( 0 ); }

    trace( "device info: @ret" );

    $ref->{'dev'} = $dev;
    for( @ret )
    {
        if( /UP/ )                   { $ref->{'up'}     = 1; }
        if( /Link encap:(.*?)\s+HW/ ){ $ref->{'link'}   = $1; }
        if( /HWaddr (\S+)/ )         { $ref->{'hwaddr'} = $1; }
        if( /inet addr:(\S+)/ )      { $ref->{'ip'}     = $1; }
        if( /P-t-P:(\S+)/ )          { $ref->{'ptp'}    = $1; }
        if( /Bcast:(\S+)/ )          { $ref->{'bcast'}  = $1; }
        if( /Mask:(\S+)/ )           { $ref->{'mask'}   = $1; }
        if( /RX bytes:(\S+)/ )       { $ref->{'rx'}     = $1; }
        if( /TX bytes:(\S+)/ )       { $ref->{'tx'}     = $1; }
    }

    return( 1 );
}

#
# get_inet_address
#
# Gets the ip address of a particular device.
#
sub get_inet_address
{
    trace( "--> get_inet_address(@_)" );
    my $dev = shift;

    my ($ip, $gw);
    for( `$IFCONFIG $dev` )
    {
        trace( "Debug -- $_\n" );

        if( m/inet addr:(\S+)\s+(?:P-t-P:(\S+))?/ ) {
            $ip = $1;
            $gw = $2;
            last;
        }
    }

    trace( "<-- get_inet_address() [$ip]" );
    return( wantarray() ? ( $ip, $gw ) : $ip );
}

#
# parse_xml
#
# Used to parse a simple XML document (DEPRECATED)
#
sub parse_xml($\$)
{
    # Should be a stable method, no need to flood log.
    #trace( "--> parse_xml(@_)" );

    my $tag = shift;
    my $xml_ref = shift;

    if( $$xml_ref =~ /<$tag>(.*?)<\/$tag>\s*(.*?)\s*$/ )
    {
        $$xml_ref = $2;

        #trace( "<-- parse_xml [HIDDEN]" );
        return( $1 );
    }

    #trace( "<-- parse_xml []" );
    return( "" );
}

#
# get_auth
#
# Gets authentication information from an auth file (DEPRECATED)
#
sub get_auth
{
    trace( "--> get_auth(@_)" );
    my $auth = shift;

    unless( open( FILE, "< $AUTH_FILE" ) ) {
        trace( "Unable to load auth file $AUTH_FILE: $! " );
        return;
    }
    my $xml_data = join( "", <FILE> );
    close( FILE );
    $xml_data =~ s/\n//g;

    my $auth_data = parse_xml( "rcsAuth", $xml_data );

    my ($secret_data, $id, $login, $secret);
    while( $auth_data )
    {
        $secret_data = parse_xml( "auth", $auth_data );

        $id = parse_xml( "id", $secret_data );
        $login = parse_xml( "login", $secret_data );
        $secret = parse_xml( "secret", $secret_data );

        if( $id eq $auth ) {
            trace( "<-- get_auth [HIDDEN]" );
            return( wantarray() ? ( $login, $secret ) : $secret );
        }
    }

    trace( "Error parsing for auth info - $auth" );
    return;
}

#
# data_template
#
# Stored a data/config file based on an existing template by replacing
# keywords with actual values from a hash.
#
sub data_template
{
    my $template = shift;
    my $data_file = shift;
    my %table = @_;

    trace( "--> data_template($template $data_file HIDDEN)" );

    open( FILE, "$template" )
        or ( trace( "Unable to open $template: $!" ), return( 0 ) );
    my @data = <FILE>;
    close( FILE );

    for( @data )
    {
        for my $key (keys %table)
        {
            s/<$key>/$table{$key}/;
        }
    }

    open( FILE, ">$data_file" )
        or ( trace( "Unable to open $data_file for writing: $!" ), return( 0 ) );
    print( FILE @data );
    close( FILE );

    return( 1 );
    trace( "<-- data_template() [1]" );
}

#
# create_pipe
#
# Creates a named pipe file.
#
sub create_pipe
{
    trace( "--> create_pipe(@_)" );
    my $pipe = shift;

    unless( -p $pipe ) {
        unlink( $pipe );
        system( "mknod $pipe p" )
            and system( "mkfifo $pipe" )
                and return( 0 );
    }

    trace( "<-- create_pipe(@_)" );
    return( 1 );
}

#
# write_pipe
#
# Writes to the names pipe file.
#
sub write_pipe
{
    trace( "--> write_pipe(@_)" );
    my $pipe = shift;
    my $data = shift;
    my $timeout = (shift or 0);

    return( 0 ) unless( -p $pipe );

    eval
    {
        local $SIG{ALRM} = sub { die $ALARM };
        alarm( $timeout );
        open( PIPE, ">$pipe" );
        print( PIPE $data );
        close( PIPE );
        alarm( 0 );
    };
    if( $@ )
    {
        unless( $@ eq $ALARM ) {
            error( "Error writing to pipe: $@" );
        } else {
            error( "Timed out writing to pipe." );
        }

        trace( "<-- write_pipe [0]" );
        return( 0 );
    }

    trace( "<-- write_pipe [1]" );
    return( 1 );
}

#
# read_pipe
#
# Reads from a named pipe.
#
sub read_pipe($\$;$)
{
    trace( "<-> read_pipe(@_)" );
    my $pipe = shift;
    my $data_ref = shift;
    my $timeout = (shift or 0);

    return( 0 ) unless( -p $pipe );

    eval
    {
        local $SIG{ALRM} = sub { die $ALARM };
        alarm( $timeout );
        open( PIPE, "$pipe" );
        $$data_ref = join( "", <PIPE> );
        close( PIPE );
        alarm( 0 );
    };
    if( $@ )
    {
        unless( $@ eq $ALARM ) {
            error( "Error reading from pipe: $@" );
        } else {
            error( "Timed out reading from pipe." );
        }

        return( 0 );
    }

    return( 1 );
}

#
# spawn
#
# Spawns off a process (don't forget to reap the child).
#
sub spawn
{
    trace( "<-> spawn(@_)" );
    my $cmd = shift;
    my $pid;

    defined( $pid = fork )
        or return( 0 );


    if( $pid )
    {
        trace( "Just spawned with PID $pid\n" );
        return( $pid );
    }

    exec( $cmd );
}

#
# is_actively_running
#
# Checks to see if a process is actively running (not a zombie).
#
sub is_actively_running
{
    no warnings;
    trace( "<-> is_active_running(@_)" );
    my $pid = shift or return( 0 );
    my $rc = (system( "$PS -p $pid | grep defunct" ) >> 8);

    return( $rc );
}

#
# is_running
#
# Checks to see if a process is valid.
#
sub is_running
{
    no warnings;
    trace( "<-> is_running(@_)" );
    my $pid = shift or return( 0 );
    my $rc = (system( "$PS -p $pid > /dev/null" ) >> 8);

    return( ! $rc );
}

#
# get_pid
#
# Get the process ID from a PPP PID file.
#
sub get_pid
{
    trace( "<-> get_pid(@_)" );
    my $file = shift;
    unless( -f $file ) { return( 0 ); }

    my ($pid, $dev);

    open( FILE, $file );
    chomp( $pid = (readline( FILE ) or 0) );
    chomp( $dev = (readline( FILE ) or "") );
    close( FILE );

    return( wantarray() ? ( $pid, $dev ) : $pid );
}

#
# routes_create
#
# Creates routing entries for a specific device over a specific gateway.
#
sub routes_create
{
    my $rdev = shift;
    my $rgw = shift;
    my @route_keys = grep( /^host_/, keys( %config ) );

    for( @route_keys )
    {
        my $ip = $config{$_};
        trace( "Creating route for $ip (via $rgw on $rdev)" );
        my $rc = (system( "$ROUTE add $ip gw $rgw dev $rdev mss $MAX_MSS >> $LOG_FILE 2>&1" ) >> 8);
        if( $rc ) {
            warning( "Unable to create route for $ip. (rc: $rc)" );
        }
    }
}

#
# lookup_route
#
# Looks up the route for a particular destination. If successful, will return
# an array with the device as the first element and the ip address of that
# device as the second.
#
sub lookup_route
{
    trace( "--> lookup_route(@_)" );

    my $ip = shift;

    my @ret = `ip route get $ip`;

    my( $dev, $dev_ip );
    for( @ret )
    {
        if( /dev (\S+)/ ) { $dev = $1; }
        if( /src (\S+)/ ) { $dev_ip = $1; }
    }

    if( $dev ) {
        return( $dev, $dev_ip );
    }
    else {
        return( "", "" );
    }
}

my @iptables_rules;

#
# iptables_run
#
# Run the desired iptables command.
#
sub iptables_run
{
    my $rule = shift;

    my $cleanup = $rule;
    $cleanup =~ s/-A|-I/-D/;
    unshift( @iptables_rules, $cleanup );

    system( "$IPTABLES $rule\n" );
}

#
# iptables_cleanup
#
# Cleans up all iptables run so far by running their counterpart -D command.
#
sub iptables_cleanup
{
    for( @iptables_rules )
    {
        system( "$IPTABLES $_\n" );
    }

    undef @iptables_rules;
}

##############################################################################
#####                           DIAL METHODS                             #####
##############################################################################

#
# dial_start
#
# Starts a dial into ATT.
#
sub dial_start
{
    trace( "--> dial_start(@_)" );

    # Check to see if dial is already active.
    if( dial_active() ) { dial_stop(); }

    # Create config files.
    for( keys %PPPD_CONFIG ) {
        unless( data_template( $_, $PPPD_CONFIG{$_}, %tpt_data ) )
        {
            error( "Unable to create template $_." );
            return( 0 );
        }
    }

    # Make dial into ATT.
    my $rc = (system( "$PPPD $PPPD_PARMS >> $LOG_FILE 2>&1" ) >> 8);
    if( $rc ) {
        $dial_return_code = $rc;
        error( "Failed to dial into ATT. (rc: $rc)." );
        return( 0 );
    }

    # Give pppd a moment to the get the device properly created.
    pause( 1 );

    # Get the new device
    ( $dial{'pid'}, $dial{'dev'} ) = get_pid( PPP_PIDFILE( $DIAL_LINKNAME ) );

    unless( $dial{'dev'} ) {
        error( "Dial failed for unknown reason." );
        dial_stop();
        return( 0 );
    }

    my %dev_info;
    my $retry = 5;
    while( $retry )
    {
        if( ! ifconfig( $dial{'dev'}, %dev_info ) )
        {
            error( "Unable to determine device information." );
            dial_stop();
            return( 0 );
        }

        last if( $dev_info{'up'} );

        pause( 3 );
        $retry--;
    }

    unless( $retry )
    {
        error( "PPP failed to properly create device $dial{'dev'}." );
        dial_stop();
        return( 0 );
    }

    $dial{'ip'} = $dev_info{'ip'};
    $dial{'gw'} = $dev_info{'ptp'};

    unless( $dial{'ip'} )
    {
        error( "Unable to determine assigned IP for $dial{dev}." );
        dial_stop();
        return( 0 );
    }
    unless( $dial{'gw'} )
    {
        error( "Unable to determine gateway IP for $dial{dev}." );
        dial_stop();
        return( 0 );
    }

    # Remove these password files
    unlink( "/etc/ppp/pap-secrets" );
    unlink( "/etc/ppp/chap-secrets" );

    # Increase the transfer queue
    system( "ip link set $dial{dev} txqueuelen 32" );


    $dial{'active'} = 1;

    trace( "<-- dial_start() [1]" );
    return( 1 );
}

#
# dial_stop
#
# Stops an already active dial.
#
sub dial_stop
{
    trace( "<-> dial_stop(@_)" );

    # Remove these password files
    unlink( "/etc/ppp/pap-secrets" );
    unlink( "/etc/ppp/chap-secrets" );

    $dial{'active'} = 0;

    if( defined( $dial{'pid'} ) and is_running( $dial{'pid'} ) )
    {
        system( "$KILL $dial{'pid'}" );
        delete( $dial{'pid'} );
        return( 1 );
    }

    return( 0 );
}

#
# dial_active
#
# Checks to see if the dial is REALLY active.
#
sub dial_active
{
    return( is_running( get_pid( PPP_PIDFILE( $DIAL_LINKNAME ) ) ) );
}

##############################################################################
#####                            VPN METHODS                             #####
##############################################################################

#
# vpn_start
#
# Start a VPN session.
#
sub vpn_start
{
    trace( "--> vpn_start(@_)" );

    my $rc;

    # See if VPN is already active.
    if( vpn_active() ) { return( 1 ); }

    # Save initial route to vpn_server so it can be restored later.  This is needed
    # if the route is set to something other than the default by the customer.

    ($INITIAL_GW_TO_VPN_SERVER, $INITIAL_DEV_TO_VPN_SERVER) = get_route_to_server();
    $INITIAL_VPN_SERVER = $config{vpn_server};
    unless( $INITIAL_GW_TO_VPN_SERVER  ) {
            error( "No default route to connect to VPN server.");
            pop_initial_route();
            return( 0 );
    }
    trace("Initial path to vpn server $INITIAL_VPN_SERVER is gw $INITIAL_GW_TO_VPN_SERVER dev $INITIAL_DEV_TO_VPN_SERVER");


    # If we (remotely) dialed in using ATT, then we must manually add our
    # route to the vpn gateway so we don't accidentally use the default route.
    if( $dial{'active'} and (dial_active() or rem_dial_active()) )
    {
        $rc = (system( "$IP route rep $config{vpn_server} via $dial{gw} dev $dial{dev} >> $LOG_FILE 2>&1" ) >> 8);
        if( $rc ) {
            error( "Unable to create route to VPN server $config{vpn_server} via $dial{gw} on $dial{dev}." );
            pop_initial_route();
            return( 0 );
        }

        $vpn{'ip'} = $dial{'ip'};
        $vpn{'gw'} = $dial{'gw'};
        $vpn{'dev'} = $dial{'dev'};
    }

    # Dialing wasn't done which means we'll use the defined route to the VPN server  -04
    # from the routing table. (/sbin/ip route get vpn_server_address)                -04
    else {
        ($vpn{'gw'}, $vpn{'dev'}) = get_route_to_server();                         # -04

        unless( $vpn{'gw'} ) {
            error( "No default route to connect to VPN server.");                  # -04
            pop_initial_route();
            return( 0 );
        }

        $vpn{'ip'} = get_inet_address( $vpn{'dev'} );
    }

    trace("VPN server address is $config{vpn_server}, with route $vpn{'gw'} on device $vpn{'dev'}");  # -04

    ##################################################
    # IPSEC
    #####

    # Create the proper ipsec config files.
    $tpt_data{'vpn_ip'} = $config{'vpn_server'};
    $tpt_data{'vpn_nat_ip'} = $tpt_data{'vpn_ip'};

    # (-02)
    # Check to see if we have a d-nat setup.
    my $mmDNATFile = "$config{data}/mmDNAT.props";
    my %dnat;
    if (-f $mmDNATFile) {
        open( DNAT, $mmDNATFile );
        for( <DNAT> ) {
            next if( /^#/ );
            chomp;
            if( /^\s*(.*?)\s*=\s*(.*?)\s*$/ ) {
                $dnat{$1} = $2;
                $dnat{$1} =~ s/\\=/=/g;
            }
        }
    }

    if( $dnat{$tpt_data{'vpn_ip'}} ) {
        $tpt_data{'vpn_nat_ip'} = $dnat{$tpt_data{'vpn_ip'}};
    }

    $tpt_data{'local_ip'} = $vpn{'ip'};
    $tpt_data{'gateway'} = $vpn{'gw'};
    $tpt_data{'ipsec_dev'} = $vpn{'dev'};

    for( keys %IPSEC_CONFIG ) {
        if( ! data_template( $_, $IPSEC_CONFIG{$_}, %tpt_data ) ) {
            warning( "Unable to create template $_" );
        }
    }

    system( "echo 0 > /proc/sys/net/ipv4/conf/$vpn{dev}/rp_filter" );

    $rc = (system( "$IPSEC setup --start >> $LOG_FILE 2>&1" ) >> 8);
    if( $rc ) {
        warning( "$IPSEC failed to run. (rc: $rc)" );
        warning( "Will try to restart..." );
        $rc = (system( "$IPSEC setup --restart >> $LOG_FILE 2>&1" ) >> 8);
        if( $rc ) {
            error( "$IPSEC failed to run twice. (rc: $rc)" );
            pop_initial_route();
            return( 0 );
        }
    }

    # Sometimes it seems to take pluto an extra second or two to get up and
    # running.
    pause( 2 );

    # When ipsec setup --restart is issued the route to the VPN server is,
    # in some cases, deleted from the routing table.  The route replace commad
    # below puts it back. (aix cmvx 573751)
    if( $dial{'active'} and (dial_active() or rem_dial_active()) )
    {
        trace("Replacing modem route $config{vpn_server} gw $dial{gw} dev $dial{dev}" );
        $rc = (system( "$IP route rep  $config{vpn_server} via $dial{gw} dev $dial{dev} >> $LOG_FILE 2>&1" ) >> 8);
        trace( "Return code from the modem route add command is $rc .");

    } else {
        trace( "Replacing route  $config{vpn_server} via $vpn{'gw'}" );
        $rc = (system("$IP route rep $config{vpn_server} via $vpn{'gw'} >> $LOG_FILE 2>&1") >> 8);
        trace( "Return code from the ip route replace command is $rc .");
    }

    $rc = (system( "$IPSEC auto $IPSEC_PARMS --replace primary >> $LOG_FILE 2>&1" ) >> 8);
    if( $rc ) {
        error( "Failed to load primary profile. (rc: $rc)" );
        vpn_stop();
        return( 0 );
    }

	
    # Hack #342. Freeswan wants to release our whack which we use for status, so
    # it looks like we can rerun this script to get another status attempt. Unfortunately
    # this means that if we are failing, it's going to take an EVEN longer to do
    # so :(. (-01)
    for( 1 .. 3 )
    {
        trace( "Connecting IPSEC Attempt #$_ of 3" );
        $rc = (system( "$IPSEC auto $IPSEC_PARMS --up primary >> $LOG_FILE 2>&1" ) >> 8);
        last unless( $rc );
        pause( 3 );
    }

    if( $rc ) {
        error( "Failed to start primary VPN profile. (rc: $rc)" );
        vpn_stop();
        return( 0 );
    }

    # Give IPSec a second to initialize everything.
    pause( 1 );

    $vpn{'ipsec_active'} = 1;

    ##################################################
    # L2TPD
    #####
    l2tp_restart() unless (rem_dial_active());

    unless( l2tp_tunnel_start( $VPN_LINKNAME ) )
    {
        error( "Failed to initialize L2TP tunnel for VPN." );
        vpn_stop();
        return( 0 );
    }

    routes_create(
        $l2tp{'tunnel'}{$VPN_LINKNAME}{'dev'},
        $l2tp{'tunnel'}{$VPN_LINKNAME}{'gw'}
        );


    # Remove these password files
    unlink( "/etc/ipsec.secrets" );
    unlink( "/etc/ppp/pap-secrets" );
    unlink( "/etc/ppp/chap-secrets" );


    trace( "<-- vpn_start" );
    return( 1 );
}

#
# vpn_stop
#
# Stops an already active VPN session.
#
sub vpn_stop
{
    trace( "<-> vpn_stop(@_)" );

    # Remove these password files
    unlink( "/etc/ipsec.secrets" );
    unlink( "/etc/ppp/pap-secrets" );
    unlink( "/etc/ppp/chap-secrets" );

    # HACK!
    if( service_vpn_active() ) {
        service_vpn_stop();
    }

    # Bring down the l2tp tunnels
    for( keys %{ $l2tp{'tunnel'} } ) {
        l2tp_tunnel_stop( $_ );
    }

    if( defined( $vpn{'dev'} ) ) {
        system( "echo 1 > /proc/sys/net/ipv4/conf/$vpn{dev}/rp_filter" );
    }

    # Stop ipsec.
    if( -f $IPSEC_LOCK  or -f $IPSEC_LOCK2 )
    {
        trace( "vpn_stop() ipsec lock file present, stoping ipsec " );
        if( $vpn{'ipsec_active'} )
        {
            system( "$IPSEC auto --down primary" );
            $vpn{'ipsec_active'} = 0;
        }
        system( "$IPSEC setup --stop" );

        pop_initial_route();
        return( 1 );
    }

    pop_initial_route();
    return( 0 );
}

sub  pop_initial_route()
{
     trace( "<-> pop_initial_route(@_)" );

     trace ("vpn server $INITIAL_VPN_SERVER gw $INITIAL_GW_TO_VPN_SERVER dev $INITIAL_DEV_TO_VPN_SERVER");

     if (defined($INITIAL_VPN_SERVER) and defined($INITIAL_GW_TO_VPN_SERVER)) {
        my $rc = (system( "$IP route rep $INITIAL_VPN_SERVER via $INITIAL_GW_TO_VPN_SERVER >> $LOG_FILE 2>&1" )  >> 8);
        if( $rc ) {
            warning( "Unable to pop route for $INITIAL_VPN_SERVER. (rc: $rc)" );
        }
     } else {
         warning( "Unable to pop route for $INITIAL_VPN_SERVER , data for server of gateway missing. " );
     }
     undef($INITIAL_VPN_SERVER);
     undef($INITIAL_GW_TO_VPN_SERVER);
     undef($INITIAL_DEV_TO_VPN_SERVER);

     return(0)

}

#
# vpn_active
#
# Check to see if a VPN session is REALLY active.
#
sub vpn_active
{
    return( (-f $IPSEC_LOCK or -f $IPSEC_LOCK2) and is_running( get_pid( PPP_PIDFILE( $VPN_LINKNAME ) ) ) );
}


##############################################################################
#####                         REMOTE VPN METHODS                         #####
##############################################################################

#
# rem_vpn_start
#
# Starts a remote vpn sessions (via L2TP).
#
sub rem_vpn_start
{
    trace( "--> rem_vpn_start(@_)" );

    if( rem_vpn_active() ) { return( 1 ); }

    # Update the vpn ip address.
    $tpt_data{'vpn_ip'} = $config{'l2tp_remote_host'};

    unless( l2tp_restart() )
    {
        error( "Failed to start l2tp daemon." );
        return( 0 );
    }

    unless( l2tp_tunnel_start( $REM_VPN_LINKNAME ) )
    {
        error( "Failed to initialize remote VPN tunnel." );
        return( 0 );
    }

    routes_create(
        $l2tp{'tunnel'}{$REM_VPN_LINKNAME}{'dev'},
        $l2tp{'tunnel'}{$REM_VPN_LINKNAME}{'gw'}
        );

    trace( "<-- rem_vpn_start() [1]" );
    return( 1 );
}

#
# rem_vpn_stop
#
# Stops an active remote vpn session.
#
sub rem_vpn_stop
{
    trace( "<-> rem_vpn_stop(@_)" );

    if( rem_vpn_active() ) {
        l2tp_tunnel_stop( $REM_VPN_LINKNAME );
    }

    return( 0 );
}

#
# rem_vpn_bkp_start
#
# Starts a remote vpn sessions (via L2TP).
#
sub rem_vpn_bkp_start
{
    trace( "--> rem_vpn_bkp_start(@_)" );

    if( rem_vpn_bkp_active() ) { return( 1 ); }

    # Update the vpn ip address.
    $tpt_data{'vpn_ip'} = $config{'l2tp_remote_host'};

    unless( l2tp_restart() )
    {
        error( "Failed to start l2tp daemon." );
        return( 0 );
    }

    unless( l2tp_tunnel_start( $REM_VPN_BKP_LINKNAME ) )
    {
        error( "Failed to initialize remote VPN tunnel." );
        return( 0 );
    }

    routes_create(
        $l2tp{'tunnel'}{$REM_VPN_BKP_LINKNAME}{'dev'},
        $l2tp{'tunnel'}{$REM_VPN_BKP_LINKNAME}{'gw'}
        );

    trace( "<-- rem_vpn_bkp_start() [1]" );
    return( 1 );
}

#
# rem_vpn_stop
#
# Stops an active remote vpn session.
#
sub rem_vpn_bkp_stop
{
    trace( "<-> rem_vpn_stop(@_)" );

    if( rem_vpn_bkp_active() ) {
        l2tp_tunnel_stop( $REM_VPN_BKP_LINKNAME );
    }

    return( 0 );
}

#
# rem_vpn_active
#
# Checks to see if VPN really active.
#
sub rem_vpn_active
{
    return( is_running( get_pid( PPP_PIDFILE( $REM_VPN_LINKNAME ) ) ) );
}

#
# rem_vpn_bkp_active
#
# Checks to see if VPN really active.
#
sub rem_vpn_bkp_active
{
    return( is_running( get_pid( PPP_PIDFILE( $REM_VPN_BKP_LINKNAME ) ) ) );
}


##############################################################################
#####                        REMOTE DIAL METHODS                         #####
##############################################################################

#
# rem_dial_start
#
# Starts a remote dial session (via L2TP)
#
sub rem_dial_start
{
    trace( "--> rem_dial_start(@_)" );

    if( rem_dial_active() ) { rem_dial_stop(); }

    unless( l2tp_restart() )
    {
        error( "Failed to start l2tp daemon." );
        return( 0 );
    }

    RETRY: for( "*PRIMARY", "*BACKUP" )
    {
        $dial{'number'} = $_;

        # Initiate a call on the specified tunnel name.
        unless( l2tp_command( "o $REM_DIAL_LINKNAME $dial{number}" ) ) {
            error( "Unable to send command to l2tp daemon" );
            return( 0 );
        }

        # Give each attempt 120 seconds.
        my $timer = 120;
        while( $timer )
        {
            pause( 5 );
            if( -f PPP_PIDFILE( $REM_DIAL_LINKNAME ) ) { last RETRY; }
            $timer -= 5;
        }

        # Close the previous tunnel attempt (although it probably won't be
        # open).
        unless( l2tp_command( "d $REM_DIAL_LINKNAME" ) ) {
            error( "Unable to send command to l2tp daemon" );
            return( 0 );
        }

        # Give it time to close the tunnel.
        pause( 2 );
    }

    # Get the new device
    ( $dial{'pid'}, $dial{'dev'} ) = get_pid( PPP_PIDFILE( $REM_DIAL_LINKNAME ) );

    unless( $dial{'dev'} ) {
        error( "Remote dial failed because device could not be determined." );
        rem_dial_stop();
        return( 0 );
    }

    my %dev_info;
    my $retry = 5;
    while( $retry )
    {
        unless( ifconfig( $dial{'dev'}, %dev_info ) )
        {
            error( "Unable to determine device information." );
            rem_dial_stop();
            return( 0 );
        }

        if( $dev_info{'up'} ) { last };

        pause( 3 );
        $retry--;
    }

    unless( $retry ) {
        error( "PPP failed to properly create device $dial{dev}." );
        rem_dial_stop();
        return( 0 );
    }

    $dial{'ip'} = $dev_info{'ip'};
    $dial{'gw'} = $dev_info{'ptp'};

    unless( $dial{'ip'} ) {
        error( "Unable to determine assigned IP for $dial{dev}." );
        rem_dial_stop();
        return( 0 );
    }
    unless( $dial{'gw'} ) {
        error( "Unable to determine gateway IP for $dial{dev}." );
        rem_dial_stop();
        return( 0 );
    }

    # Remove these password files
    unlink( "/etc/ppp/pap-secrets" );
    unlink( "/etc/ppp/chap-secrets" );

    $dial{'active'} = 1;

    trace( "<-- rem_dial_start() [1]" );
    return( 1 );
}

#
# rem_dial_stop
#
# Stops a remote dial sessions.
#
sub rem_dial_stop
{
    trace( "<-> rem_dial_stop(@_)" );

    # Remove these password files
    unlink( "/etc/ppp/pap-secrets" );
    unlink( "/etc/ppp/chap-secrets" );

    $dial{'active'} = 0;

    if( rem_dial_active() or (defined( $dial{'pid'} ) and is_running( $dial{'pid'} )) )
    {
        l2tp_command( "d $REM_DIAL_LINKNAME" );
        pause( 3 );

        if( is_running( $dial{'pid'} ) ) {
            system( "$KILL $dial{'pid'}" );
        }

        $dial{'pid'} = 0;
        return( 1 );
    }

    return( 0 );
}

#
# rem_dial_active
#
# Checks to see if a remote dial session is REALLY active.
#
sub rem_dial_active
{
    return( is_running( get_pid( PPP_PIDFILE( $REM_DIAL_LINKNAME ) ) ) );
}


##############################################################################
#####                           L2TP METHODS                             #####
##############################################################################

#
# l2tp_restart
#
# Restarts an l2tp session.
#
sub l2tp_restart
{
    trace( "<-> l2tp_restart(@_)" );

    if( ! l2tp_stop() ) {
        warning( "Unable to stop l2tp daemon, may not be running." );
    }
    return( l2tp_start() );
}

#
# l2tp_start
#
# Starts an L2TP session if hasn't already been started.
#
sub l2tp_start
{
    trace( "--> l2tp_start(@_)" );

    # Create the proper l2tpd config files.
    $tpt_data{'linkname'} = $VPN_LINKNAME;
    for( keys %L2TPD_CONFIG ) {
        unless( data_template( $_, $L2TPD_CONFIG{$_}, %tpt_data ) ) {
            error( "Unable to create template $_." );
            return( 0 );
        }
    }

    if( l2tp_active() )
    {
        warning( "l2tp daemon is already running." );
        return( 1 );
    }

    $l2tp{'pid'} = spawn( "$L2TPD $L2TPD_PARMS" );
    unless( $l2tp{'pid'} ) {
        error( "Unable to spawn off l2tpd." );
        return( 0 );
    }

    # Give l2tp a second or two to startup.
    pause( 3 );

    # Put the l2tp daemon into "Always LNS" mode so it can support the special
    # STR_RMT_SPT OS/400 function.
    l2tp_command( "a lns" );

    trace( "<-- l2tp_start" );
    return( 1 );
}

#
# l2tp_stop
#
# Stops an already active l2tp session.
#
sub l2tp_stop
{
    trace( "<-> l2tp_stop(@_)" );

    if( l2tp_active() )
    {
        system( "$KILL $l2tp{pid}" );

        my $ret;
        my $timeout = 10;
        while( $timeout )
        {
            $ret = waitpid( $l2tp{'pid'}, WNOHANG );
            if( $ret ) { last; }
            pause( 1 );
            $timeout--;
        }

        unless( $timeout ) {
            error( "Timeout on waiting for l2tp daemon to die." );
            return( 0 );
        }

        $l2tp{'pid'} = 0;
        return( 1 );
    }

    warning( "L2tp daemon is not running." );
    return( 0 );
}

#
# l2tp_active
#
# Checks to see if l2tp is active.
#
sub l2tp_active
{
    return( defined( $l2tp{'pid'} ) and is_running( $l2tp{'pid'} ) and (-p "/var/run/l2tp-control") );
}

#
# l2tp_command
#
# Sends a command to the l2tp program.
#
sub l2tp_command
{
    trace( "<-> l2tp_command(@_)" );
    my $cmd = shift;

    # NOTE: I am hacking on a \0 character. This is because of some crazy bug
    # with l2tpd that I will have to figure out at some point. By adding on
    # the \0, I prevent the string command in L2TP from adding on extra junk.
    $cmd .= "\0";

    # Write the command to the l2tp control pipe.
    unless( write_pipe( $L2TPD_CONTROL, $cmd, 10 ) ) {
        return( 0 );
    }

    pause( 1 );
    return( 1 );
}

#
# l2tp_tunnel_start
#
# Starts a new l2tp tunnel based on the specified linkname.
#
sub l2tp_tunnel_start
{
    trace( "--> l2tp_tunnel_start(@_)" );
    my $linkname = shift;

    ##################################################
    # L2TPD
    #####

    # Update the l2tpd ppp options so that we can track the ppp device.
    $tpt_data{'linkname'} = $linkname;
    for( keys %L2TPD_CONFIG ) {
        unless( data_template( $_, $L2TPD_CONFIG{$_}, %tpt_data ) ) {
            error( "Unable to create template $_" );
        }
    }

    unless( l2tp_active() ) { l2tp_start(); }

    # Put the l2tp daemon into "Always LAC" mode while we attempt an outgoing
    # tunnel call.
    unless( l2tp_command( "a lac" ) ) {
        error( "Unable to send command to l2tp daemon" );
        return( 0 );
    }

    my $timer;
    RETRY: for( 1 .. $L2TPD_RETRY )
    {
        # Initiate a call on the specified tunnel name.
        unless( l2tp_command( "c $linkname" ) ) {
            error( "Unable to send command to l2tp daemon" );
            return( 0 );
        }

        # Give each attempt 120 seconds.
        $timer = 120;
        while( $timer )
        {
            pause( 1 );
            if( -f PPP_PIDFILE( $linkname ) ) { last RETRY; }
            $timer--;
        }

        # Close the previous tunnel attempt (although it probably won't be
        # open).
        unless( l2tp_command( "d $linkname" ) ) {
            error( "Unable to send command to l2tp daemon" );
            return( 0 );
        }

        # Give it time to close the tunnel.
        pause( 2 );
    }

    # Put the l2tp daemon into "Always LNS" mode so it can support the special
    # STR_RMT_SPT OS/400 function.
    unless( l2tp_command( "a lns" ) ) {
        error( "Unable to send command to l2tp daemon" );
        return( 0 );
    }

    unless( $timer ) {
        error( "Unable to start $linkname profile." );
        return( 0 );
    }

    my ($pid, $dev) = get_pid( PPP_PIDFILE( $linkname ) );

    my %dev_info;
    my $retry = 20;
    while( $retry )
    {
        if( ! ifconfig( $dev, %dev_info ) )
        {
            error( "Unable to determine device information." );
            return( 0 );
        }

        last if( $dev_info{'up'} );

        pause( 5 );
        $retry--;
    }

    unless( $retry ) {
        error( "PPP failed to properly create device $dev." );
        return( 0 );
    }

    my $ip = $dev_info{'ip'};
    my $gw = $dev_info{'ptp'};

    unless( $ip ) {
        error( "Unable to determine assigned IP for $dev." );
        return( 0 );
    }
    unless( $gw ) {
        error( "Unable to determine gateway IP for $dev." );
        return( 0 );
    }

    # Set a safe MTU since the VPN GW has a horrible bug they won't fix.
    system( "ip link set $dev mtu $MAX_MSS" );

    # Disable rp_filter (return path filter) for this device.
    system( "echo 0 > /proc/sys/net/ipv4/conf/$dev/rp_filter" );

    $l2tp{'tunnel'}{$linkname}{'pid'} = $pid;
    $l2tp{'tunnel'}{$linkname}{'dev'} = $dev;
    $l2tp{'tunnel'}{$linkname}{'ip'} = $ip;
    $l2tp{'tunnel'}{$linkname}{'gw'} = $gw;

    trace( "<-- l2tp_tunnel_start()" );
    return( 1 );
}

#
# l2tp_tunnel_stop
#
# Stop an active l2tp tunnel.
#
sub l2tp_tunnel_stop
{
    trace( "--> l2tp_tunnel_stop(@_)" );
    my $linkname = shift;

    unless( defined( $l2tp{'tunnel'}{$linkname}{'pid'} ) and is_running( $l2tp{'tunnel'}{$linkname}{'pid'} ) ) {
        return( 0 );
    }

    # Tell the l2tp daemon to stop the tunnel.
    l2tp_command( "d $linkname" );

    # Give it a few seconds to clean up.
    my $timeout = 3;
    while( $timeout )
    {
        unless( is_running( $l2tp{'tunnel'}{$linkname}{'pid'} ) ) { last; }
        pause( 1 );
        $timeout--;
    }

    # The tunnel is still up, so it's time to stop it our way.
    unless( $timeout ) {
        system( "$KILL $l2tp{'tunnel'}{$linkname}{'pid'}" );
    }

    delete( $l2tp{'tunnel'}{$linkname} );

    return( 1 );
    trace( "<-- l2tp_tunnel_stop()" );
}

##############################################################################
#####                  REMOTE SERVICE UTILITY METHODS                    #####
##############################################################################

#
# record_create
#
# Creates a new record of the specified type.
#
sub record_create
{
    my $rec_type = shift;

    return( pack( "nn", 4, $rec_type ) );
}

#
# record_add_field
#
# Adds a field to the passed in record (byref).
#
sub record_add_field (\$$$)
{
    my $rec_ref = shift;
    my $fld_type = shift;
    my $fld_data = shift;

    # Check for invalid record.
    unless( $$rec_ref ) { return( 0 ); }

    my ($rec_length, $rec_data) = unpack( "na*", $$rec_ref );

    # Check for invalid record.
    unless( $rec_length and $rec_data ) { return( 0 ); }

    # Calculate the length of the field plus the length attribute.
    my $fld_length = 4 + length( $fld_data );
    $rec_length += $fld_length;

    # Now we pack on the new data.
    $$rec_ref =
        pack( "na*nna*", $rec_length, $rec_data,
                         $fld_length, $fld_type, $fld_data );

    return( 1 );
}

#
# record_get_type
#
# Returns the type of record that was passed in.
#
sub record_get_type
{
    my $record = shift;

    return( unpack( "x[n]n", $record ) );
}

#
# record_get_field
#
# Returns a specific field from a data record.
#
sub record_get_field
{
    my $record = shift;
    my $field = shift;

    my ($rec_len, $rec_type, $data) = unpack( "nna*", $record );

    my ($fld_len, $fld_type, $fld_desc, $fld_data);
    while( $data )
    {
        ( $fld_len, $fld_type, $data ) = unpack( "nna*", $data );
        ( $fld_data, $data ) = unpack( "a" . ($fld_len - 4) . "a*", $data );
        if( $field == $fld_type ) { return( $fld_data ); }
    }

    return( undef );
}

#
# dump
#
# Dumps a record out to trace.
#
sub record_dump
{
    my $record = shift;

    my $hexdump = unpack( "H*", $record );
    trace( "hexdump: $hexdump\n" );

    my ($rec_len, $rec_type, $data) = unpack( "nna*", $record );

    my $rec_desc = $RECORD_TYPE{$rec_type};
    unless( $rec_desc ) { $rec_desc = "Unknown Record Type"; }

    trace( "="x60 . "\n" .
           "record length: $rec_len\n" .
           "record type:   " . sprintf( "%.4x", $rec_type ) .
                             " ($rec_desc)\n" );

    my ($fld_len, $fld_type, $fld_desc, $fld_data);
    while( $data )
    {
        ( $fld_len, $fld_type, $data ) = unpack( "nna*", $data );
        ( $fld_data, $data ) = unpack( "a" . ($fld_len - 4) . "a*", $data );
        $fld_desc = $FIELD_TYPE{$fld_type};
        unless( $fld_desc ) { $fld_desc = "Unknown Field Type"; }

        trace( "-"x60 . "\n" .
               "field length:  $fld_len\n" .
               "field type:    " . sprintf( "%.4x", $fld_type ) .
                                 " ($fld_desc)\n" .
               "field data:    $fld_data\n" );
    }
}

#
# sock_write
#
# Writes to a socket.
#
sub sock_write
{
    my $data = shift;

    my ($win, $wout) = ("", "");
    vec( $win, fileno( SOCK ), 1 ) = 1;
    unless( select( undef, $wout = $win, undef, $SOCK_TIMEOUT ) )
    {
        error( "Timed out waiting to write to socket.\n" );
        return( 0 );
    }
    my $bytes = syswrite( SOCK, $data, length( $data ) );

    unless( defined( $bytes ) and $bytes = length( $data ) )
    {
        error( "Couldn't write to the socket. $!\n" );
        return( 0 );
    }

    return( 1 );
}

#
# sock_read
#
# Reads from the socket.
#
sub sock_read(\$$)
{
    my $buffer = shift;
    my $count = shift;

    my ($rin, $rout) = ("", "");
    vec( $rin, fileno( SOCK ), 1 ) = 1;

    unless( select( $rout = $rin, undef, undef, $SOCK_TIMEOUT ) )
    {
        error( "Timed out waiting to read from the socket." );
        return( 0 );
    }

    my $bytes = sysread( SOCK, $$buffer, $count );

    unless( defined( $bytes ) )
    {
        error( "Could not read from the socket: $!" );
        return( 0 );
    }

    return( 1 );
}

#
# record_send
#
# Sends a record to the RS3 server and then returns the response.
#
sub record_send
{
    my $record = shift;
    my $no_response = shift;

    # Socket stuff.
    unless( socket( SOCK, PF_INET, SOCK_STREAM, getprotobyname( 'tcp' ) ) )
    {
        error( "Unable to create socket: $!" );
        return( 0 );
    }

    my $sockaddr = sockaddr_in( $config{'port_rs3'}, inet_aton( $config{'host_rs3'} ) );

    unless( connect( SOCK, $sockaddr ) )
    {
        error( "Unable to connect to host $config{host_rs3}:$config{port_rs3}: $!" );
        return( undef );
    }

    # Set autoflush.
    select( (select( SOCK ), $| = 1)[0] );

    unless( sock_write( $record ) )
    {
        error( "Unabe to send record:" );
        record_dump( $record );
        return( undef );
    }

    unless( $no_response )
    {
        my $buffer;
        unless( sock_read( $buffer, 2 ) )
        {
            error( "Unable to retrieve response record length." );
            return( undef );
        }

        my $total = unpack( "n", $buffer );

        unless( sock_read( $buffer, $total - 2 ) )
        {
            error( "Unable to retrieve response record." );
            return( undef );
        }

        my $newrec = pack( "na*", $total, $buffer );

        close( SOCK );

        return( $newrec );
    }

    return( 1 );
}

#
# vslip_spawn
#
# Spawns the vslip program.
#
sub vslip_spawn
{
    trace( "--> vslip_spawn(@_)" );

    # Kill an already running vslip
    vslip_stop();

    my $vslip_args;
    my $vslip_total = 0;
    for( sort( keys %{$service{'cec'}} ) )
    {
        unless( $service{'cec'}{$_}{'ext_ip'} ) { next; }
        unless( $service{'cec'}{$_}{'type'} ) { next; }

        $vslip_args .= " $_*"
                     . $service{'cec'}{$_}{'type'} . "-"
                     . $service{'cec'}{$_}{'model'} . "*"
                     . $service{'cec'}{$_}{'serial'};
        $vslip_total++;
    }

    unless( $vslip_total )
    {
        warning( "Nothing to do." );
        return( 1 );
    }

    trace( "Launching vslip with '$vslip_args'." );
    $service{'vslip_pid'} = open( VSLIP, "/opt/hsc/bin/actvslip $vslip_args |" );

    unless( $service{'vslip_pid'} ) {
        return( 0 );
    }

    my $vslip_left = $vslip_total;
    eval
    {
        local $SIG{ALRM} = sub { die $ALARM };
        alarm( 120 );

        while( $vslip_left and is_running( $service{'vslip_pid'} ) )
        {
            my $buffer = readline( VSLIP );

            trace( "vslip: $buffer" );
            # Look for a successful connect string.
            if( $buffer =~ /(\S+) \S+ 0 (\S+)/ )
            {
                trace( "Partition $1 successfully connected with IP $2." );
                $service{'cec'}{$1}{'int_ip'} = $2;
                $vslip_left--;
            }
        }
        alarm( 0 );
    };
    if( $@ )
    {
        unless( $@ eq $ALARM )
        {
            error( "Error reading from vslip: $@" );
            return( 0 );
        } else {
            error( "Timed out waiting for vslip to connect." );
        }

        if( $vslip_left < $vslip_total )
        {
            warning( "Not all partitions were connected, but at least one was." );
            return( 2 );
        }

        error( "No partitions were able to connect." );
        return( 0 );
    }

    trace( "<-- vslip_spawn()" );
    return( 1 );
}

#
# vslip_stop
#
# Stops a spawned vslip program
#
sub vslip_stop
{
    unless( defined( $service{'vslip_pid'} ) and is_running( $service{'vslip_pid'} ) ) {
        return( 0 );
    }

    system( "$KILL $service{vslip_pid}" );
    pause( 3 );
    close( VSLIP );

    return( 1 );
}

#
# file_server_spawn
#
# Spawns the file server program.
#
sub file_server_spawn
{
    $service{'fs_pid'} = spawn( "$JAVA $FS_ARGS" );
    unless( $service{'fs_pid'} ) {
        error( "Unable to spawn off fileserver for data dump." );
        return( 0 );
    }
}

#
# file_server_stop
#
# Spawns the file server program.
#
sub file_server_stop
{
    unless( defined( $service{'fs_pid'} ) and is_running( $service{'fs_pid'} ) ) {
        return( 0 );
    }

    system( "$KILL $service{fs_pid}" );

    return( 1 );
}


##############################################################################
#####                      REMOTE SERVICE METHODS                        #####
##############################################################################

sub service_dial_start
{
    trace( "--> service_dial_start(@_)" );

    if( service_dial_active() ) { return( 1 ); }

    # Create config files.
    for( keys %PPPD_CONFIG ) {
        unless( data_template( $_, $PPPD_CONFIG{$_}, %tpt_data ) )
        {
            error( "Unable to create template $_." );
            return( 0 );
        }
    }

    # Spawn off the agetty program.
    my $tmp_parms = $MODEM_ANSWER_PARMS;
    $tmp_parms =~ s/&/\\&/g;
    $dial{'rs_pid'} = spawn( "$MODEM_ANSWER $tmp_parms" );

    unless( $dial{'rs_pid'} ) {
        error( "Unable to spawn off $MODEM_ANSWER $tmp_parms" );
        return( 0 );
    }

    $dial{'rs_init'} = 1;

    return( 1 );
    trace( "<-- service_dial_stop()" );
}

sub service_dial_setup
{
    trace( "--> service_dial_setup(@_)" );

    $dial{'rs_init'} = 0;

    # Give PPP time to setup the connection each attempt 60 seconds.
    my $timer = 120;
    my ($pid, $dev);
    while( $timer )
    {
        pause( 2 );
        if( -f PPP_PIDFILE( $RS_LINKNAME ) ) {
            ($pid, $dev) = get_pid( PPP_PIDFILE( $RS_LINKNAME ) );
            if( is_running( $pid ) and $dev ) { last; }
        }

        $timer -= 2;
        trace( "$timer seconds left..." );
    }

    unless( $timer )
    {
        error( "Unable to detect service dial connection." );
        service_dial_stop();
        return( 0 );
    }

    my %dev_info;
    my $retry = 5;
    while( $retry )
    {
        if( ! ifconfig( $dev, %dev_info ) )
        {
            error( "Unable to determine device information." );
            service_dial_stop();
            return( 0 );
        }

        last if( $dev_info{'up'} );

        pause( 3 );
        $retry--;
    }

    unless( $retry )
    {
        error( "PPP failed to properly create device $dev." );
        service_dial_stop();
        return( 0 );
    }

    $service{'hmc_dev'} = $dev;
    my $hmc_ip = $dev_info{'ip'};
    my $remote_ip = $dev_info{'ptp'};

    trace( "hmc is $hmc_ip, remote is $remote_ip" );

    $config{'host_rs3'} = $remote_ip;
    $config{'remote_rs3'} = $remote_ip;

    #my $rrc = (system( "$ROUTE add $config{host_rs3} gw $hmc_ip dev $dev >> $LOG_FILE 2>&1" ) >> 8);
    #if( $rrc ) {
    #    warning( "Unables to create route for ? (rc: $rrc)" );
    #}

    # Grab information about the hmc from the config file.
    if( $config{'rs_hmc_mtms'} =~ /(\w+)-(\w+)\*(\w+)/ )
    {
        $service{'hmc'}{'type'} = $1;
        $service{'hmc'}{'model'} = $2;
        $service{'hmc'}{'serial'} = $3;
        $service{'hmc'}{'allow'} = $config{'rs_hmc_allow'};
    }
    else
    {
        error( "Unable to determine machine info for HMC.\n" );
        return( 0 );
    }

    # Grab information about each of the extra partitions that will be used
    # for remote servicing.
    my ($cec_mtms, $cec_serial);
    delete( $service{'cec'} );
    for( 0 .. 5 )
    {
        my $key = "rs_cec_" . $_;
        if( $config{$key} )
        {
            if( $config{$key} =~ /(\d+)\*(\w+)-(\w+)\*(\w+)/ )
            {
                $service{'cec'}{$1}{'type'} = $2;
                $service{'cec'}{$1}{'model'} = $3;
                $service{'cec'}{$1}{'serial'} = $4;

                unless( $cec_mtms )   { $cec_mtms   = "$2-$3"; }
                unless( $cec_serial ) { $cec_serial = $4; }
            }

            else {
                warning( "Unable to determine cec info for partition $_ ($config{$key}))" );
            }
        }

        # Remove the key so it won't iterate in the following loops.
        else {
            trace( "DEBUG: No key found for $key." );
        }
    }

    my $service_partitions = keys( %{$service{'cec'}} );

    # Now extra virtual ip addresses for each of the partitions must be
    # created. This will get us external ip addresses for each of the
    # partitions.
    my $octet = 1;
    for( sort( keys( %{$service{'cec'}} ) ) )
    {
trace( "DEBUG: Creating virtual ip address ($_: $service{cec}{$_}{serial})" );

        my $ip_address = "169.254." . hex( $config{'rs_hmc_station'} ) . ".$octet";
        $octet ++;
        my $rc = (system( "ip address add local $ip_address dev $dev" ) >> 8);
        unless( $rc )
        {
            $service{'cec'}{$_}{'ext_ip'} = $ip_address;
            $service{'cec'}{$_}{'ext_dev'} = $dev;
        }

        else
        {
            warning( "Failed to create a tunnel for partition: $_*"
                     . $service{'cec'}{$_}{'type'} . "-"
                     . $service{'cec'}{$_}{'model'} . "*"
                     . $service{'cec'}{$_}{'serial'} );
        }
    }

    # Next the vslip program must be used to create virtual slip tunnels to
    # get an internal ip address for each of the partitions.
    unless( vslip_spawn() )
    {
        error( "Failed to activate virtual slip connection." );
        service_vpn_stop();
        return( 0 );
    }

    # Go through each "partition" that really wants to connect an ip address.
    for( sort( keys %{$service{'cec'}} ) )
    {
        unless( $service{'cec'}{$_}{'ip'} ) { next; }

        my $dst_ip = $service{'cec'}{$_}{'ip'};
        my $ext_ip  = $service{'cec'}{$_}{'ext_ip'};
        my $ext_dev = $service{'cec'}{$_}{'ext_dev'};

        # Lookup the proper routing information for the ip address.
        my( $int_dev, $int_ip );
        ( $int_dev, $int_ip ) = lookup_route( $dst_ip );

        iptables_run( "-I FORWARD -p tcp -i $ext_dev -o $int_dev -d $dst_ip -j ACCEPT" );
        iptables_run( "-I FORWARD -p tcp -i $int_dev -o $ext_dev -d $ext_ip -j ACCEPT" );
        iptables_run( "-I FORWARD -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $SYN_MSS" );
        iptables_run( "-t nat -A PREROUTING -i $ext_dev -d $ext_ip -j DNAT --to-destination $dst_ip" );
        iptables_run( "-t nat -A POSTROUTING -o $int_dev -d $dst_ip -j SNAT --to-source $int_ip" );

    }

    # The next step is to go through each partition and do the proper ip
    # forwarding so that the external and internal addresses are connected.
    my $int_dev = "sl0";
    for( sort( keys %{$service{'cec'}} ) )
    {
        my $int_ip  = $service{'cec'}{$_}{'int_ip'};
        my $ext_ip  = $service{'cec'}{$_}{'ext_ip'};
        my $ext_dev = $service{'cec'}{$_}{'ext_dev'};

        # First check for an external ip address
        if( $ext_ip )
        {
            # Next see if there is an internal ip address.
            if( $int_ip )
            {
                # Setup the proper iptable entries.
                system( "$ROUTE add $ext_ip dev $ext_dev" );
                iptables_run( "-I FORWARD -i $ext_dev -o $int_dev -d $int_ip -j ACCEPT" );
                iptables_run( "-I FORWARD -i $int_dev -o $ext_dev -d $ext_ip -j ACCEPT" );
                iptables_run( "-I FORWARD -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $SYN_MSS" );
                iptables_run( "-t nat -A PREROUTING -i $ext_dev -d $ext_ip -j DNAT --to-destination $int_ip" );
            }

            # There is no internal ip address, so the virtual address should be
            # removed.
            else
            {
            }
        }
    }

    iptables_run( "-I FORWARD -o $dev -j ACCEPT" );
    iptables_run( "-t nat -A POSTROUTING -o $int_dev -j MASQUERADE" );

    # Ports allowed to hmc
    my @port_keys = grep( /^rs_hmc_ports_/, keys( %config ) );

    # Need to setup the proper iptable rules.
    if( $config{'rs_hmc_allow'} )
    {
        file_server_spawn();
        $service{'fs_active'} = 1;
        iptables_run( "-I INPUT -i $dev -j DROP" );
        iptables_run( "-I INPUT -i $dev -j LOG --log-prefix DROPPED_DIAL_VPN");
        iptables_run( "-I INPUT -i $dev -m state --state ESTABLISHED,RELATED -j ACCEPT" );
        for( @port_keys )
        {
            my $ports = $config{$_};
            iptables_run( "-I INPUT -i $dev --protocol tcp --match multiport --destination-port $ports -j ACCEPT" );
        }

        # Redirect 443 to 8443 (-06).
        iptables_run( "-t nat -I PREROUTING -i $dev --protocol tcp --dport 443 -j REDIRECT --to 8443" );
    }

    else
    {
        iptables_run( "-I INPUT -i $dev -j DROP" );
        iptables_run( "-I INPUT -i $dev -m state --state ESTABLISHED,RELATED -j ACCEPT" );
    }

    # Make sure that ip forwarding is enables.
    system( "echo 1 > /proc/sys/net/ipv4/ip_forward" );

    # Create a new startup record.
    my $record = record_create( 0x0001 );

    # Add the HMC information
    record_add_field( $record, 0x0101, $service{'hmc'}{'type'} );
    record_add_field( $record, 0x0102, $service{'hmc'}{'model'} );
    record_add_field( $record, 0x0103, $service{'hmc'}{'serial'} );
    record_add_field( $record, 0x0104, $hmc_ip );
    record_add_field( $record, 0x0105, "A" ); # attended support
    record_add_field( $record, 0x0107, "Y" ); # is a dial connection
    record_add_field( $record, 0x010A, $config{'loc_region'} );
    record_add_field( $record, 0x010B, $config{'loc_country'} );
    record_add_field( $record, 0x0112, $config{'rs_hmc_allow'} ? 'Y' : 'N' );

    if( $service_partitions ) {
        record_add_field( $record, 0x0106, "$cec_mtms:$cec_serial" );
    }

    for( sort keys %{$service{'cec'}} ) {
        record_add_field( $record, 0x0110, pack( "na*", $_, $service{'cec'}{$_}{'ext_ip'} ) );
    }

    record_dump( $record );
    my $response = record_send( $record );

    unless( defined( $response ) )
    {
        warning( "Unable to obtain a response from the remote service machine.");
    }

    else
    {
        record_dump( $response );

        my $return_type = record_get_type( $response );

        if( $return_type == 0x8001 )
        {
            my $return_code = unpack( "n", record_get_field( $response, 0x010F ) );

            if( $return_code == 0x0000 )
            {
                my $service_server = record_get_field( $response, 0x010D );
                my $service_port   = record_get_field( $response, 0x010E );
                trace( "DEBUG: RAS $service_server:$service_port" );
            }

            else
            {
                error( "Received a bad return code ($return_code)." );
                #service_vpn_stop();
                #return( 0 );
            }
        }

        else
        {
            error( "Received an incorrect record type ($return_type)." );
            #service_vpn_stop();
            #return( 0 );
        }
    }

    $service{'active'} = 1;
    $service{'status'} = "I";

    # Remove these password files
    unlink( "/etc/ppp/pap-secrets" );
    unlink( "/etc/ppp/chap-secrets" );

    trace( "<-- service_dial_setup()" );
    return( 1 );
}

sub service_dial_stop
{
    trace( "<-> service_dial_stop(@_)" );

    $dial{'rs_init'} = 0;
    $config{'remote_rs3'} = "";

    # Remove these password files
    unlink( "/etc/ppp/pap-secrets" );
    unlink( "/etc/ppp/chap-secrets" );

    if( is_running( get_pid( PPP_PIDFILE( $RS_LINKNAME ) ) ) )
    {
        my $record = record_create( 0x0003 );
        record_add_field( $record, 0x0101, $service{'hmc'}{'type'} );
        record_add_field( $record, 0x0102, $service{'hmc'}{'model'} );
        record_add_field( $record, 0x0103, $service{'hmc'}{'serial'} );
        record_dump( $record );
        record_send( $record, 1 );

        system( "$KILL " . get_pid( PPP_PIDFILE( $RS_LINKNAME ) ) );
    }

    elsif( defined( $dial{'rs_pid'} ) and is_running( $dial{'rs_pid'} ) )
    {
        system( "$KILL $dial{'rs_pid'}" );

        my $ret;
        my $timeout = 10;
        while( $timeout )
        {
            $ret = waitpid( $dial{'rs_pid'}, WNOHANG );
            if( $ret ) { last; }
            pause( 1 );
            $timeout--;
        }

        unless( $timeout ) {
            error( "Timeout on waiting for modem answer program to die." );
            return( 0 );
        }

        delete( $dial{'rs_pid'} );
        return( 1 );
    }

    # Take down the file server
    if( $service{'fs_active'} ) {
        file_server_stop();
        $service{'fs_active'} = 0;
    }


    # Stop the virtual slip connections.
    vslip_stop();

    # Cleanup the iptables rules
    iptables_cleanup();

    $service{'active'} = 0;

    trace( "<-- service_dial_stop()" );
    return( 1 );
}

sub service_dial_active
{
    return( is_running( $dial{'rs_pid'} ) or is_running( get_pid( PPP_PIDFILE( $RS_LINKNAME ) ) ) );
}

##############################################################################
#####                      REMOTE SERVICE METHODS                        #####
##############################################################################

sub service_vpn_start
{
    trace( "--> service_vpn_start(@_)" );

    if( service_vpn_active() ) { return( 1 ); }

    unless( vpn_active() or rem_vpn_active() or rem_vpn_bkp_active() )
    {
        error( "VPN is not active." );
        return( 0 );
    }


    # Grab information about the hmc from the config file.
    if( $config{'rs_hmc_mtms'} =~ /(\w+)-(\w+)\*(\w+)/ )
    {
        $service{'hmc'}{'type'} = $1;
        $service{'hmc'}{'model'} = $2;
        $service{'hmc'}{'serial'} = $3;
        $service{'hmc'}{'allow'} = $config{'rs_hmc_allow'};
    }
    else
    {
        error( "Unable to determine machine info for HMC.\n" );
        return( 0 );
    }

    # Grab information about each of the extra partitions that will be used
    # for remote servicing.
    my ($cec_mtms, $cec_serial);
    delete( $service{'cec'} );
    for( 0 .. 5 )
    {
        my $key = "rs_cec_" . $_;
        if( $config{$key} )
        {
            if( $config{$key} =~ /(\d+)\*(\w+)-(\w+)\*(\w+)/ )
            {
                $service{'cec'}{$1}{'linkname'} = $TUN_LINKNAME . $_;
                $service{'cec'}{$1}{'type'} = $2;
                $service{'cec'}{$1}{'model'} = $3;
                $service{'cec'}{$1}{'serial'} = $4;

                unless( $cec_mtms )   { $cec_mtms   = "$2-$3"; }
                unless( $cec_serial ) { $cec_serial = $4; }
            }

            elsif( $config{$key} =~ /(\d+)\*(\S+)/ )
            {
                $service{'cec'}{$1}{'linkname'} = $TUN_LINKNAME . $_;
                $service{'cec'}{$1}{'ip'} = $2;
            }

            else {
                warning( "Unable to determine cec info for partition $_ ($config{$key}))" );
            }
        }

        # Remove the key so it won't iterate in the following loops.
        else {
            trace( "DEBUG: No key found for $key." );
        }
    }

    unless( $cec_mtms )   { $cec_mtms   = "0000-00"; }
    unless( $cec_serial ) { $cec_serial = "000000"; }

    my $service_partitions = keys( %{$service{'cec'}} );

    # Now extra tunnels for each of the partitions must be created. This will
    # get us external ip addresses for each of the partitions.
    for( sort( keys( %{$service{'cec'}} ) ) )
    {
        my $linkname = $service{'cec'}{$_}{'linkname'};
trace( "DEBUG: Starting extra tunnel for $linkname ($_: $service{cec}{$_}{serial})" );
        if( l2tp_tunnel_start( $linkname ) )
        {
            $service{'cec'}{$_}{'ext_ip'} = $l2tp{'tunnel'}{$linkname}{'ip'};
            $service{'cec'}{$_}{'ext_dev'} = $l2tp{'tunnel'}{$linkname}{'dev'};
        }

        else
        {
            warning( "Failed to create a tunnel for partition: $_*"
                     . $service{'cec'}{$_}{'type'} . "-"
                     . $service{'cec'}{$_}{'model'} . "*"
                     . $service{'cec'}{$_}{'serial'} );
        }
    }

    # Next the vslip program must be used to create virtual slip tunnels to
    # get an internal ip address for each of the partitions.
    unless( vslip_spawn() )
    {
        error( "Failed to activate virtual slip connection." );
        service_vpn_stop();
        return( 0 );
    }

    # Go through each "partition" that really wants to connect an ip address.
    for( sort( keys %{$service{'cec'}} ) )
    {
        unless( $service{'cec'}{$_}{'ip'} ) { next; }

        my $dst_ip = $service{'cec'}{$_}{'ip'};
        my $ext_ip  = $service{'cec'}{$_}{'ext_ip'};
        my $ext_dev = $service{'cec'}{$_}{'ext_dev'};

        # Lookup the proper routing information for the ip address.
        my( $int_dev, $int_ip );
        ( $int_dev, $int_ip ) = lookup_route( $dst_ip );

        iptables_run( "-I FORWARD -p tcp -i $ext_dev -o $int_dev -d $dst_ip -j ACCEPT" );
        iptables_run( "-I FORWARD -p tcp -i $int_dev -o $ext_dev -d $ext_ip -j ACCEPT" );
        iptables_run( "-I FORWARD -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $SYN_MSS" );
        iptables_run( "-t nat -A PREROUTING -i $ext_dev -d $ext_ip -j DNAT --to-destination $dst_ip" );
        iptables_run( "-t nat -A POSTROUTING -o $int_dev -d $dst_ip -j SNAT --to-source $int_ip" );

    }

    # The next step is to go through each partition and do the proper ip
    # forwarding so that the external and internal addresses are connected.
    my $int_dev = "sl0";
    for( sort( keys %{$service{'cec'}} ) )
    {
        my $int_ip  = $service{'cec'}{$_}{'int_ip'};
        my $ext_ip  = $service{'cec'}{$_}{'ext_ip'};
        my $ext_dev = $service{'cec'}{$_}{'ext_dev'};

        # First check for an external ip address
        if( $ext_ip )
        {
            # Next see if there is an internal ip address.
            if( $int_ip )
            {
                # Setup the proper iptable entries.
                system( "$ROUTE add $ext_ip dev $ext_dev" );
                iptables_run( "-I FORWARD -i $ext_dev -o $int_dev -d $int_ip -j ACCEPT" );
                iptables_run( "-I FORWARD -i $int_dev -o $ext_dev -d $ext_ip -j ACCEPT" );
                iptables_run( "-I FORWARD -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $SYN_MSS" );
                iptables_run( "-t nat -A PREROUTING -i $ext_dev -d $ext_ip -j DNAT --to-destination $int_ip" );
            }

            # There is no internal ip address, so the external tunnel
            # should be stopped.
            else
            {
                l2tp_tunnel_stop( $TUN_LINKNAME . $_ );
            }
        }
    }

    my $ext_link;
    if( vpn_active() )     { $ext_link = $VPN_LINKNAME; }
    elsif( rem_vpn_active() ) { $ext_link = $REM_VPN_LINKNAME; }
    elsif( rem_vpn_bkp_active() ) { $ext_link = $REM_VPN_BKP_LINKNAME; }

    iptables_run( "-I FORWARD -o $l2tp{tunnel}{$ext_link}{dev} -j ACCEPT" );
    iptables_run( "-t nat -A POSTROUTING -o $int_dev -j MASQUERADE" );

    # Ports allowed to hmc
    my @port_keys = grep( /^rs_hmc_ports_/, keys( %config ) );

    # Need to setup the proper iptable rules.
    if( $config{'rs_hmc_allow'} )
    {
        file_server_spawn();
        $service{'fs_active'} = 1;
        iptables_run( "-I INPUT -i $l2tp{tunnel}{$ext_link}{dev} -j DROP" );
        iptables_run( "-I INPUT -i $l2tp{tunnel}{$ext_link}{dev} -j LOG --log-prefix DROPPED_INT_VPN");
        iptables_run( "-I INPUT -i $l2tp{tunnel}{$ext_link}{dev} -m state --state ESTABLISHED,RELATED -j ACCEPT" );
        for( @port_keys )
        {
            my $ports = $config{$_};
            iptables_run( "-I INPUT -i $l2tp{tunnel}{$ext_link}{dev} --protocol tcp --match multiport --destination-port $ports -j ACCEPT" );
        }

        # Redirect 443 to 8443 (-06).
        iptables_run( "-t nat -I PREROUTING -i $l2tp{tunnel}{$ext_link}{dev} --protocol tcp --dport 443 -j REDIRECT --to 8443" );
    }

    else
    {
        iptables_run( "-I INPUT -i $l2tp{tunnel}{$ext_link}{dev} -j DROP" );
        iptables_run( "-I INPUT -i $l2tp{tunnel}{$ext_link}{dev} -m state --state ESTABLISHED,RELATED -j ACCEPT" );
    }

    # Make sure that ip forwarding is enables.
    system( "echo 1 > /proc/sys/net/ipv4/ip_forward" );

    # Create a new startup record.
    my $record = record_create( 0x0001 );

    # Grab account and username.
    my $att_account = "";
    my $att_user = "";
    if( $config{'att_account_name'} )
    {
        if( $config{'att_account_name'} =~ /internet\.(.*?)\.(.*?)/ )
        {
            $att_account = $1;
            $att_user = $2;
        }
    }

    # Add the HMC information
    record_add_field( $record, 0x0101, $service{'hmc'}{'type'} );
    record_add_field( $record, 0x0102, $service{'hmc'}{'model'} );
    record_add_field( $record, 0x0103, $service{'hmc'}{'serial'} );
    record_add_field( $record, 0x0104, $l2tp{'tunnel'}{$ext_link}{'ip'} );
    record_add_field( $record, 0x0105, "A" ); # attended support
    record_add_field( $record, 0x0107, "N" ); # not a dial connection
    record_add_field( $record, 0x0108, $att_account );
    record_add_field( $record, 0x0109, $att_user );
    record_add_field( $record, 0x010A, $config{'loc_region'} );
    record_add_field( $record, 0x010B, $config{'loc_country'} );
    record_add_field( $record, 0x0112, $config{'rs_hmc_allow'} ? 'Y' : 'N' );

    if( $service_partitions ) {
        record_add_field( $record, 0x0106, "$cec_mtms:$cec_serial" );
    }

    for( sort keys %{$service{'cec'}} ) {
        record_add_field( $record, 0x0110, pack( "na*", $_, $service{'cec'}{$_}{'ext_ip'} ) );
    }

    record_dump( $record );
    my $response = record_send( $record );

    unless( defined( $response ) )
    {
        error( "Unable to obtain a response from the remote service machine.");
        service_vpn_stop();
        return( 0 );
    }

    record_dump( $response );

    my $return_type = record_get_type( $response );

    if( $return_type == 0x8001 )
    {
        my $return_code = unpack( "n", record_get_field( $response, 0x010F ) );

        if( $return_code == 0x0000 )
        {
            my $service_server = record_get_field( $response, 0x010D );
            my $service_port   = record_get_field( $response, 0x010E );
            trace( "DEBUG: RAS $service_server:$service_port" );
        }

        else
        {
            error( "Received a bad return code ($return_code)." );
            #service_vpn_stop();
            #return( 0 );
        }
    }

    else
    {
        error( "Received an incorrect record type ($return_type)." );
        #service_vpn_stop();
        #return( 0 );
    }

    $service{'active'} = 1;
    $service{'status'} = "I";

    # MegaMouth hack for Thomas (we do not care if it fails)
    system( "/opt/esshmc/bin/rsHmcPrepareConn -u " . $l2tp{'tunnel'}{$ext_link}{'ip'} . " &" );


    trace( "<-- service_vpn_start()" );
    return( 1 );
}

sub service_vpn_stop
{
    trace( "--> service_vpn_stop()" );

    # MegaMouth hack for Thomas (we do not care if it fails)
    system( "/opt/esshmc/bin/rsHmcPrepareConn -d &" );

    if( service_vpn_active() and (vpn_active() or rem_vpn_active() or rem_vpn_bkp_active()) )
    {
        my $record = record_create( 0x0003 );
        record_add_field( $record, 0x0101, $service{'hmc'}{'type'} );
        record_add_field( $record, 0x0102, $service{'hmc'}{'model'} );
        record_add_field( $record, 0x0103, $service{'hmc'}{'serial'} );
        record_dump( $record );
        record_send( $record, 1 );
    }

    # Take down the file server
    if( $service{'fs_active'} ) {
        file_server_stop();
        $service{'fs_active'} = 0;
    }

    my $ext_link;
    if( vpn_active() )     { $ext_link = $VPN_LINKNAME; }
    elsif( rem_vpn_active() ) { $ext_link = $REM_VPN_LINKNAME; }
    elsif( rem_vpn_bkp_active() ) { $ext_link = $REM_VPN_BKP_LINKNAME; }

    my @port_keys = grep( /^rs_hmc_ports_/, keys( %config ) );


    # Stop the virtual slip connections.
    vslip_stop();

    # Stop the l2tp service tunnels.
    for( sort( keys( %{$service{'cec'}} ) ) ) {
        l2tp_tunnel_stop( $service{'cec'}{$_}{'linkname'} );
    }

    # Cleanup iptables commands.
    iptables_cleanup();

    $service{'active'} = 0;

    trace( "<-- service_vpn_stop()" );
    return( 1 );
}

sub service_status
{
    trace( "--> service_status()" );

    # return true if service is and should be active.
    # return false if srevice is not or should not be active.

    if( $config{'remote_rs3'} ) {
        $config{'host_rs3'} = $config{'remote_rs3'};
    }

    # Check to see if we are in dial mode...
    if( $dial{'rs_init'} ) { return( 1 ); }

    # Check to see if it's a dial in, if not, assume we're doing VPN.
    if( ! service_dial_active() )
    {
        # Make sure service is still active.
        unless( service_vpn_active() ) { return( 0 ); }

        # Make sure VPN connection is still active
        unless( vpn_active() or rem_vpn_active() or rem_vpn_bkp_active() ) { return( 0 ); }
    }

    # Attempt to query the remote support server.
    my $record = record_create( 0x0002 );

    # Add the HMC information
    record_add_field( $record, 0x0101, $service{'hmc'}{'type'} );
    record_add_field( $record, 0x0102, $service{'hmc'}{'model'} );
    record_add_field( $record, 0x0103, $service{'hmc'}{'serial'} );

    record_dump( $record );
    my $response = record_send( $record );

    unless( defined( $response ) )
    {
        error( "Unable to obtain a response from the remote service machine.");
        return( 0 );
    }

    record_dump( $response );

    my $return_type = record_get_type( $response );

    # Check for valid response record.
    if( $return_type != 0x8002 )
    {
        error( "Invalid response record." );
        return( 0 );
    }

    my $return_code = unpack( "a", record_get_field( $response, 0x0111 ) );

    if( ! $service{'status'} ) {
        $service{'status'} = "I";
    }

    if( $return_code eq "I" and $service{'status'} eq "I" )
    {
        trace( "Remote support inactive." );
        return( 1 );
    }

    if( $return_code eq "A"  )
    {
        $service{'status'} = "A";
        trace( "Remote support active." );
        return( 1 );
    }

    else
    {
        trace( "Remote support is finished." );
        return( 0 );
    }

    trace( "<-- service_status()" );
}

sub service_vpn_active
{
    return( $service{'active'} );
}

##############################################################################
#####                      MONITOR MODEM METHODS                         #####
##############################################################################
sub modem_answer_run
{
    trace( "--> modem_answer_run()" );

    if( defined( $dial{'mm_pid'} ) and is_running( $dial{'mm_pid'} ) )
    {
        if( is_actively_running( $dial{'mm_pid'} ) ) {
            return( 1 );
        } else {
            modem_answer_stop();
        }
    }

    # Spawn off the monitor modem program.
    my $tmp_parms = $MONITOR_MODEM_PARMS;
    $tmp_parms =~ s/&/\\&/g;
    $dial{'mm_pid'} = spawn( "$MONITOR_MODEM $tmp_parms" );

    unless( $dial{'mm_pid'} ) {
        error( "Unable to spawn off $MONITOR_MODEM $tmp_parms" );
        return( 0 );
    }

    return( 1 );
    trace( "<-- modem_answer_run()" );
}

sub modem_answer_stop
{
    trace( "--> modem_answer_stop()" );

    if( defined( $dial{'mm_pid'} ) and is_running( $dial{'mm_pid'} ) )
    {
        system( "$KILL $dial{'mm_pid'}" );

        my $ret;
        my $timeout = 10;
        while( $timeout )
        {
            $ret = waitpid( $dial{'mm_pid'}, WNOHANG );
            if( $ret ) { last; }
            pause( 1 );
            $timeout--;
        }

        unless( $timeout ) {
            error( "Timeout on waiting for modem answer program to die." );
            return( 0 );
        }

        delete( $dial{'mm_pid'} );
    }
    else
    {
        warning( "We were told to stop $MONITOR_MODEM but it doesn't appear to be running." );
        warning( "Attemping to stop it the good old fashion way (-9 style)" );
        system( "killall -9 monitorModem" );

        if( defined( $dial{'mm_pid'} ) )
        {
            my $ret;
            my $timeout = 10;
            while( $timeout )
            {
                $ret = waitpid( $dial{'mm_pid'}, WNOHANG );
                if( $ret ) { last; }
                pause( 1 );
                $timeout--;
            }
            delete( $dial{'mm_pid'} );
        }
    }

    return( 1 );
    trace( "<-- modem_answer_stop()" );
}
##############################################################################
#####                   PORT / IP FORWARDING METHODS                     #####
##############################################################################

##
# mmPortForwardToLpar
#
# This will initiate a vslip connection to the proper partition and then setup
# the needed rules in iptables to forward the specified port.
#
sub mmPortForwardToLpar
{
    trace( "--> mmPortForwardToLpar(@_)" );

    my $mtms = shift;
    my $ext_port = shift;
    my $dst_port = shift;

    unless( $mtms and $ext_port and $dst_port )
    {
        error( "Not enough paremeters." );
        return( 0 );
    }

    # Make sure we're connect to vpn for service reasons.
    if( ! service_vpn_active() )
    {
        error( "This HMC is not currently connected for remote service." );
        return( 0 );
    }

    # Sanity check to verify VPN is really up
    unless( vpn_active() or rem_vpn_active() or rem_vpn_bkp_active() )
    {
        error( "VPN is not active." );
        return( 0 );
    }

    my $ext_link;
    if( vpn_active() )            { $ext_link = $VPN_LINKNAME; }
    elsif( rem_vpn_active() )     { $ext_link = $REM_VPN_LINKNAME; }
    elsif( rem_vpn_bkp_active() ) { $ext_link = $REM_VPN_BKP_LINKNAME; }

    my $ext_dev = $l2tp{'tunnel'}{$ext_link}{'dev'};
    my $ext_ip = $l2tp{'tunnel'}{$ext_link}{'ip'};

    my $part;
    if( $mtms =~ /(\d+)\*(\w+)-(\w+)\*(\w+)/ )
    {
        $part = $1;
        $service{'cec'}{$1}{'type'} = $2;
        $service{'cec'}{$1}{'model'} = $3;
        $service{'cec'}{$1}{'serial'} = $4;
        $service{'cec'}{$1}{'ext_dev'} = $ext_dev;
        $service{'cec'}{$1}{'ext_ip'} = $ext_ip;
    }
    else
    {
        error( "MTMS was given in an unexpected format. Given $mtms, expected 0*XXXXX-XXXX*XXXXXXX." );
        return( 0 );
    }

    # Next the vslip program must be used to create virtual slip tunnels to
    # get an internal ip address for each of the partitions.
    unless( vslip_spawn() )
    {
        error( "Failed to activate virtual slip connecion to $mtms." );
        return( 0 );
    }

    my $dst_ip = $service{'cec'}{$part}{'int_ip'};

    # Lookup the proper routing information for the ip address.
    my( $int_dev, $int_ip );
    ( $int_dev, $int_ip ) = lookup_route( $dst_ip );

    iptables_run( "-I FORWARD -p tcp -i $ext_dev -d $dst_ip --dport $dst_port -j ACCEPT" );
    iptables_run( "-I FORWARD -p tcp -i $int_dev -s $dst_ip -j ACCEPT" );
    iptables_run( "-I FORWARD -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $SYN_MSS" );
    iptables_run( "-t nat -A PREROUTING -i $ext_dev -d $ext_ip -p tcp --dport $ext_port -j DNAT --to-destination $dst_ip:$dst_port" );
    iptables_run( "-t nat -A POSTROUTING -o $int_dev -d $dst_ip -j SNAT --to-source $int_ip" );

    # Make sure that ip forwarding is enabled.
    system( "echo 1 > /proc/sys/net/ipv4/ip_forward" );

    trace( "<-- mmPortForwardToLpar()" );
    return( 1 );
}

##
# mmPortForwardToIp
#
# This will setup the proper rules need in iptables to forward a port on to a
# specified ip address.
#
sub mmPortForwardToIp
{
    trace( "--> mmPortForwardToIp(@_)" );

    my $dst_ip = shift;
    my $ext_port = shift;
    my $dst_port = shift;

    # Lookup the proper routing information for the ip address.
    my( $int_dev, $int_ip );
    ( $int_dev, $int_ip ) = lookup_route( $dst_ip );


    unless( $dst_ip and $ext_port and $dst_port )
    {
        error( "Not enough paremeters." );
        return( 0 );
    }

    # Make sure we're connect to vpn for service reasons.
    if( ! service_vpn_active() )
    {
        error( "This HMC is not currently connected for remote service." );
        return( 0 );
    }

    # Sanity check to verify VPN is really up
    unless( vpn_active() or rem_vpn_active() or rem_vpn_bkp_active() )
    {
        error( "VPN is not active." );
        return( 0 );
    }

    my $ext_link;
    if( vpn_active() )            { $ext_link = $VPN_LINKNAME; }
    elsif( rem_vpn_active() )     { $ext_link = $REM_VPN_LINKNAME; }
    elsif( rem_vpn_bkp_active() ) { $ext_link = $REM_VPN_BKP_LINKNAME; }

    my $ext_dev = $l2tp{'tunnel'}{$ext_link}{'dev'};
    my $ext_ip = $l2tp{'tunnel'}{$ext_link}{'ip'};

    iptables_run( "-I FORWARD -p tcp -i $ext_dev -d $dst_ip --dport $dst_port -j ACCEPT" );
    iptables_run( "-I FORWARD -p tcp -i $int_dev -s $dst_ip -j ACCEPT" );
    iptables_run( "-I FORWARD -p tcp --tcp-flags SYN,RST SYN -j TCPMSS --set-mss $SYN_MSS" );
    iptables_run( "-t nat -A PREROUTING -i $ext_dev -d $ext_ip -p tcp --dport $ext_port -j DNAT --to-destination $dst_ip:$dst_port" );
    iptables_run( "-t nat -A POSTROUTING -o $int_dev -d $dst_ip -j SNAT --to-source $int_ip" );

    # Make sure that ip forwarding is enabled.
    system( "echo 1 > /proc/sys/net/ipv4/ip_forward" );

    trace( "<-- mmPortForwardToIp()" );
    return( 1 );
}


##############################################################################
#####                          DAEMON METHODS                            #####
##############################################################################

#
# daemonize
#
sub daemonize
{
    trace( "--> daemonize(@_)" );

    unless( chdir( "/" ) )
    {
        print( STDERR "Can't chdir to /: $!" );
        return( 0 );
    }

    my $pid;
    unless( defined( $pid = fork ) )
    {
        print( STDERR "Can't fork: $!" );
        return( 0 )
    }

    if( $pid ) {
        trace( "    spawned child pid=$pid" );
        trace( "<-- daemonize() [0]" );
        return( 1 )
    }

    $is_daemon = 1;

    open( STDIN, "/dev/null" )
        or warning( "Can't read /dev/null: $!" );
    open( STDOUT, ">> $LOG_FILE" )
        or warning( "Can't write to $LOG_FILE: $!" );
    open( STDERR, ">> $LOG_FILE" )
        or warning( "Can't write to $LOG_FILE: $!" );

    setsid()
        or warning( "Can't start a new session: $!" );

    umask( 0077 );

    daemon_run();
}

#
# init
#
sub daemon_init
{
    trace( "--> daemon_init(@_)" );

    # Runtime Globals
    $AUTH_FILE      = "$config{data}/auth.xml";
    $TEMPLATE_DIR   = "$config{data}/templates";
    #$PROP_FILE      = "$config{data}/rcsInfo.props";
    $LOG_FILE       = "$config{data}/rcsControl.log";

    # Runtime Globals
    $PPPD_OPT       = "/etc/ppp/rcs-dial-options";
    $PPPD_PARMS     = "file $PPPD_OPT";
    %PPPD_CONFIG    = (
        "$TEMPLATE_DIR/rcs-dial-options.template" => $PPPD_OPT,
        "$TEMPLATE_DIR/rcs-rs-options.template" => "/etc/ppp/rcs-rs-options",
        "$TEMPLATE_DIR/pap-secrets.template" => "/etc/ppp/pap-secrets",
        "$TEMPLATE_DIR/chap-secrets.template" => "/etc/ppp/chap-secrets",
        "$TEMPLATE_DIR/chat-script.template" => "/etc/ppp/chat-script"
        );

    $IPSEC_CONF     = "/etc/ipsec.conf";
    $IPSEC_PARMS    = "--config $IPSEC_CONF";
    $IPSEC_LOCK     = "/var/run/ipsec_setup.pid";
    $IPSEC_LOCK2    = "/var/run/pluto/ipsec_setup.pid";
    %IPSEC_CONFIG   = (
        "$TEMPLATE_DIR/ipsec.conf.template" => $IPSEC_CONF,
        "$TEMPLATE_DIR/ipsec.secrets.template" => "/etc/ipsec.secrets"
        );

    $L2TPD_RETRY    = 1;
    $L2TPD_CONF     = "/etc/l2tpd.conf";
    $L2TPD_PARMS    = "-c $L2TPD_CONF -D";
    $L2TPD_CONTROL  = "/var/run/l2tp-control";
    %L2TPD_CONFIG   = (
        "$TEMPLATE_DIR/l2tpd.conf.template" => $L2TPD_CONF,
        "$TEMPLATE_DIR/rcs-vpn-options.template" => "/etc/ppp/rcs-vpn-options",
        "$TEMPLATE_DIR/rcs-rem-vpn-options.template" => "/etc/ppp/rcs-rem-vpn-options",
        "$TEMPLATE_DIR/rcs-rem-bkp-options.template" => "/etc/ppp/rcs-rem-bkp-options",
        "$TEMPLATE_DIR/rcs-vpn-tun-options.template" => "/etc/ppp/rcs-vpn-tun-options",
        "$TEMPLATE_DIR/rcs-rem-dial-options.template" => "/etc/ppp/rcs-rem-dial-options",
        "$TEMPLATE_DIR/pap-secrets.template" => "/etc/ppp/pap-secrets",
        "$TEMPLATE_DIR/chap-secrets.template" => "/etc/ppp/chap-secrets"
        );

    # Find paths to all programs
    sub get_path
    {
        my @path = split( ":", $PATH );
        my $prog = shift;
        for( @path )
        {
            next unless( $_ );
            s{/+$}{};
            if( -x "$_/$prog" ) { return( "$_/$prog" ); }
        }

        return( 0 );
    }

    $PPPD     = get_path( "pppd" )     or warning( "The pppd program could not be found." );
    $IPSEC    = get_path( "ipsec" )    or warning( "The ipsec program could not be found." );
    $L2TPD    = get_path( "l2tpd" )    or warning( "The l2tpd program could not be found." );
    $VSLIP    = get_path( "actvslip" ) or warning( "The actvslip program could not be fonud." );
    $IPTABLES = get_path( "iptables" ) or warning( "The iptables program could not be found." );
    $JAVA     = get_path( "java" )     or warning( "The java program could not be found." );
    $MODEM_ANSWER = get_path( "modemAnswer" ) or warning( "The modemAnswer program could not be found." );
    $MONITOR_MODEM = get_path( "monitorModem" ) or warning( "The monitorModem program could not be found." );
    $STOP_MODEM    = "/opt/ccfw/native/runAsRoot/rcsKillTree.sh";   # (-03)

    # Last, load the config file.
    load_config();

    trace( "<-- daemon_init(@_)" );
}

#
# reset
#
sub daemon_reset
{
    trace( "--> daemon_reset(@_)" );

    # Kill all instances of l2tpd.
    unless( system( "$PS -C l2tpd > /dev/null" ) ) {
        system( "killall l2tpd" );
    }
    pause( 1 );

    unlink( $L2TPD_CONTROL );

    # Checking to see if ipsec is already running.
    if( -f $IPSEC_LOCK or -f $IPSEC_LOCK2 ) {
        trace( "daemon_reset() ipsec lock file present, stoping ipsec " );
        system( "$IPSEC setup --stop" );
    }

    unlink( $FIFO_DAEMON );
    unlink( $PID_FILE );

    #unlink( $PROP_FILE );

    trace( "<-- daemon_reset()" );
}

#
# shutdown
#
sub daemon_shutdown
{
    trace( "--> daemonShutdown(@_)" );

    service_vpn_stop();
    service_dial_stop();

    vpn_stop();
    rem_vpn_stop();

    dial_stop();
    rem_dial_stop();

    l2tp_stop();

    unlink( $FIFO_DAEMON );
    unlink( $PID_FILE );

    client_response( 0, "Daemon successfully shutdown." );
    exit( 0 );
}

#
# daemon
#
sub daemon_run
{
    trace( "--> daemon_run(@_)" );

    # Create the PID file.
    open( FILE, "> $PID_FILE" );
    print( FILE "$$\n" );
    close( FILE );

    # Fire up the l2tp daemon.
    l2tp_start();

    # Add the signals
    $SIG{INT} = $SIG{TERM} = $SIG{QUIT} = 'daemon_shutdown';

    # This file has IBM confidential stuff, so remove it.
    unlink( "$config{data}/$CONFIG_FILE" );

    # Inifinite loop.
    my $cmd;
    for(;;)
    {
        unless( -p $FIFO_DAEMON )
        {
            unless( create_pipe( $FIFO_DAEMON ) ) {
                error( "Couldn't create fifo $FIFO_DAEMON: $!" );
            }
        }

        trace( "WAITING for command" );
        read_pipe( $FIFO_DAEMON, $cmd );

        # Clear out any residual messages that may have collected in the error_data variable.
        undef( $error_data );

        chomp( $cmd );

        # Don't load the config on a service dial setup because it shouldn't
        # exist.
        unless( $cmd eq "service_dial_setup" ) { load_config(); }

        if( $cmd eq "dial_start" )
        {
            unless( dial_start() ) {
                if ($dial_return_code eq 19) {
                   trace("dial_start() failed with PAP authentication error.");
                   $dial_return_code = 0;
                   client_response( $RC_AUTH_ERR, error_dump() );
                } else {
                   trace("dial_start() failed with non-authentication error.");
                   client_response( $RC_DIAL_ERR, error_dump() );
                }
            } else {
                client_response( 0, "Dial initiated." );
            }
        }

        elsif( $cmd eq "dial_stop" )
        {
            unless( dial_stop() ) {
                client_response( 0, "Dial not started." );
            } else {
                client_response( 0, "Dial stopped." );
            }
        }

        elsif( $cmd eq "rem_dial_start" )
        {
            unless( rem_dial_start() ) {
                client_response( $RC_DIAL_ERR, error_dump() );
            } else {
                client_response( 0, "Remote dial initiated." );
            }
        }

        elsif( $cmd eq "rem_dial_stop" )
        {
            unless( rem_dial_stop() ) {
                client_response( 0, "Remote dial not started." );
            } else {
                client_response( 0, "Remote dial stopped." );
            }
        }

        elsif( $cmd eq "vpn_start" )
        {
            unless( vpn_start() ) {
                client_response( $RC_VPN_ERR, error_dump() );
            } else {
                client_response( 0, "VPN initiated." );
            }
        }

        elsif( $cmd eq "vpn_stop" )
        {
            unless( vpn_stop() ) {
                client_response( 0, "VPN not started." );
            }
            else {
                client_response( 0, "VPN stopped." );
            }
        }

        elsif( $cmd eq "rem_vpn_start" )
        {
            unless( rem_vpn_start() ) {
                client_response( $RC_VPN_ERR, error_dump() );
            } else {
                client_response( 0, "Remote vpn initiated." );
            }
        }

        elsif( $cmd eq "rem_vpn_stop" )
        {
            unless( rem_vpn_stop() ) {
                client_response( 0, "Remote vpn not started." );
            } else {
                client_response( 0, "Remote vpn stopped." );
            }
        }

        elsif( $cmd eq "rem_vpn_bkp_start" )
        {
            unless( rem_vpn_bkp_start() ) {
                client_response( $RC_VPN_ERR, error_dump() );
            } else {
                client_response( 0, "Remote vpn bkp initiated." );
            }
        }

        elsif( $cmd eq "rem_vpn_bkp_stop" )
        {
            unless( rem_vpn_bkp_stop() ) {
                client_response( 0, "Remote vpn bkp not started." );
            } else {
                client_response( 0, "Remote vpn bkp stopped." );
            }
        }

        elsif( $cmd eq "service_vpn_start" )
        {
            unless( service_vpn_start() ) {
                client_response( $RC_VPN_ERR, error_dump() );
            } else {
                client_response( 0, "Remote service over vpn initiated." );
            }
        }

        elsif( $cmd eq "service_vpn_stop" )
        {
            unless( service_vpn_stop() ) {
                client_response( 0, "Remote service over vpn not started." );
            } else {
                client_response( 0, "Remote service over vpn stopped." );
            }
        }

        elsif( $cmd eq "service_status" )
        {
            if( service_status() )
            {
                client_response( 0, "Service is still running." );
            } else {
                client_response( 1, "Service should no longer be active." );
            }
        }

        elsif( $cmd eq "service_dial_start" )
        {
            unless( service_dial_start() ) {
                client_response( $RC_DIAL_ERR, error_dump() );
            } else {
                client_response( 0, "Remote service over dial initiated." );
            }
        }

        elsif( $cmd eq "service_dial_setup" )
        {
            unless( service_dial_setup() ) {
                client_response( $RC_DIAL_ERR, error_dump() );
            } else {
                client_response( 0, "Remote service over dial setup." );
            }
        }

        elsif( $cmd eq "service_dial_stop" )
        {
            unless( service_dial_stop() ) {
                client_response( 0, "Remote service over dial not started." );
            } else {
                client_response( 0, "Remote service over dial stopped." );
            }
        }

        elsif( $cmd eq "modem_answer_run" )
        {
            unless( modem_answer_run() ) {
                client_response( $RC_DIAL_ERR, error_dump() );
            } else {
                client_response( 0, "Modem monitor program has started." );
            }
        }

        elsif( $cmd eq "modem_answer_stop" )
        {
            unless( modem_answer_stop() ) {
                client_response( 0, "Modem monitor program not running." );
            } else {
                client_response( 0, "Modem monitor program stopped." );
            }
        }

        elsif( $cmd eq "update_config" )
        {
            unless( l2tp_restart() ) {
                client_response( $RC_GENERAL_ERR, error_dump() );
            } else {
                client_response( 0, "New config loaded and l2tpd restarted." );
            }
        }

        # Megamouth additions for port forwarding
        elsif( $cmd =~ /^mmPortForwardToLpar (\S+) (\S+) (\S+)/i )
        {
            unless( mmPortForwardToLpar( $1, $2, $3 ) ) {
                client_response( $RC_GENERAL_ERR, error_dump() );
            } else {
                client_response( 0, "Port forwarding has been configured." );
            }
        }

        # Megamouth additions for port forwarding
        elsif( $cmd =~ /^mmPortForwardToIp (\S+) (\S+) (\S+)/i )
        {
            unless( mmPortForwardToIp( $1, $2, $3 ) ) {
                client_response( $RC_GENERAL_ERR, error_dump() );
            } else {
                client_response( 0, "Port forwarding has been configured." );
            }
        }


        elsif( $cmd eq "daemon_status" ) {
            client_response( 0, daemon_status() );
        }

        elsif( $cmd eq "daemon_shutdown" ) {
            daemon_shutdown();
        }

        else {
            client_response( $RC_GENERAL_ERR, "Invalid command ($cmd)." );
        };
    }

    trace( "<-- daemon_run()" );
}

#
# daemon_status
#
sub daemon_status
{
    my $status;

    $status = "$me is running under pid - $$\n";
    $status .= "data - $config{data}\n";
    $status .= "log - $LOG_FILE\n";

    if( dial_active() )
    {
        $status .= "\n"
                .  "Dial is active\n"
                .  "ATT Account: $config{att_auth_user}\n"
                .  "Init String: $config{modem_init}\n"
                .  "Init String: $config{modem_post_init}\n"
                .  "Dial String: $config{modem_dial_mode}$config{modem_dial_number}\n"
                .  "PPPD Device: $dial{dev}\n";
    }

    if( rem_dial_active() )
    {
        $status .= "\n"
                .  "Remote Dial is active\n"
                .  "ATT Account: $config{att_auth_user}\n"
                .  "Phone Profile: $dial{number}\n"
                .  "PPPD Device: $dial{dev}\n";
    }

    if( vpn_active() )
    {
        $status .= "\n"
                .  "VPN is active\n"
                .  "Server  : $config{vpn_server}\n"
                .  "Local IP: $vpn{ip}\n"
                .  "Gateway : $vpn{gw}\n"
                .  "Tunnel  : $l2tp{tunnel}{$VPN_LINKNAME}{dev} -> "
                .            "$l2tp{tunnel}{$VPN_LINKNAME}{ip} (GW: $l2tp{tunnel}{$VPN_LINKNAME}{gw})\n";
    }

    if( service_vpn_active() )
    {
        $status .= "\n"
                .  "VPN Remote Service is active\n";

    }

    return( $status );
}

##############################################################################
#####                      COMMUNICATION METHODS                         #####
##############################################################################

#
# daemon_command
#
# TODO
#
sub daemon_command
{
    trace( "<-> daemon_command(@_)" );
    my $cmd = join( " ", @_ );

    unless( daemon_exists() ) {
        print( STDERR "$me not running.\n" );
        exit( $RC_GENERAL_ERR );
    }

    unless( create_pipe( $FIFO_CLIENT ) ) {
        print( STDERR "Couldn't create fifo $FIFO_CLIENT: $!" );
        exit( $RC_GENERAL_ERR );
    }

    my $data;

    trace( "Attempting to open $PID_FILE" );
    unless( open( LOCK, $PID_FILE ) )
    {
        print( STDERR "Unable to open $PID_FILE for locking\n" );
        exit( $RC_GENERAL_ERR );
    }

    my $timeout = $FIFO_TIMEOUT;

    trace( "Attempting to lock $PID_FILE" );
    while( ! flock( LOCK, LOCK_EX | LOCK_NB ) )
    {
        trace( "Lock failed, sleeping for 5 seconds." );
        pause( 5 );
        $timeout -= 5;
        unless( $timeout > 0 )
        {
            trace( "Failed to obtain a lock after timeout." );
            print( STDERR "Failed to obtain a lock after timeout.\n" );
            exit( $RC_GENERAL_ERR );
        }
    }

    trace( "Lock obtained." );

    write_pipe( $FIFO_DAEMON, $cmd, $FIFO_TIMEOUT );
    read_pipe( $FIFO_CLIENT, $data, $FIFO_TIMEOUT );
    unlink( $FIFO_CLIENT );

    trace( "Releasing lock on $FIFO_DAEMON" );
    flock( LOCK, LOCK_UN );
    close( LOCK );

    my $ret = 0;
    if( $data =~ s/^(\d+)(?:\n|\s)?// ) {
        $ret = $1;
    }
    chomp( $data );

    return( wantarray() ? ( $ret, $data ) : $ret );
}

#
# client_response
#
sub client_response
{
    trace( "<-> client_response(@_)" );
    my $cmd = join( " ", @_ );

    unless( client_exists() ) {
        trace( "Client doesn't exist." );
        return;
    }

    write_pipe( $FIFO_CLIENT, $cmd, 10 );
}


##############################################################################
#####                          CONFIG METHODS                            #####
##############################################################################

#
# load_config
#
# TODO
#
sub load_config
{
    trace( "<-> load_config" );

    # Hack! In case this file exists, it must be detroyed
    if( -f "$config{data}/auth.xml" ) { unlink( "$config{data}/auth.xml" ); }

    my $config_file = "$config{data}/$CONFIG_FILE";

    # Read in configuration.
    my $timeout = 5;
    while( $timeout )
    {
        #trace( "attempting to load $config_file..." );
        if( open( FILE, $config_file ) ) {
            last;
        }
        pause( 1 );
        $timeout--;
    }

    unless( $timeout )
    {
        warning( "Unable to open config file $config_file" );
        return( 1 );
    }

    for( <FILE> )
    {
        next if( /^#/ );
        chomp;
        if( /^\s*(.*?)\s*=\s*(.*?)\s*$/ ) {
            $config{$1} = $2;
            $config{$1} =~ s/\\=/=/g;
        }
    }

    close( FILE );

    # Remove file now that we have loaded data from it.
    unlink( $config_file );

    no warnings;

    # Get dial auth data.
    #my ($dial_secret, $vpn_secret, $l2tp_user, $l2tp_secret);
    #
    #$dial_secret = get_auth( $config{'dial_auth'} );
    #unless( $dial_secret ) {
    #    error( "Couldn't load dial auth $config{dial_auth}" );
    #    return( 0 );
    #}
    #$vpn_secret = get_auth( $config{'vpn_auth'} );
    #unless( $vpn_secret ) {
    #    error( "Couldn't load vpn auth $config{vpn_auth}" );
    #    return( 0 );
    #}
    #($l2tp_user, $l2tp_secret) = get_auth( $config{'l2tp_auth'} );
    #unless( $l2tp_user and $l2tp_secret ) {
    #    error( "Couldn't load l2tp auth $config{l2tp_auth}" );
    #    return( 0 );
    #}

    # Setup template data.
    $tpt_data{'modem_init'}       = $config{'modem_init'} or "ATZ";
    $tpt_data{'modem_post_init'}  = $config{'modem_post_init'};
    $tpt_data{'modem_ans_init'}   = $config{'modem_answer_init'};
    $tpt_data{'dial_string'}      = $config{'modem_dial_mode'} . $config{'modem_dial_number'};
    $tpt_data{'ppp_user'}         = $config{'att_auth_user'};
    $tpt_data{'att_pap_auth'}     = "$config{att_wuth_user} * $config{att_auth_pw} *";

    $tpt_data{'vpn_ip'}           = $config{'vpn_server'};
    $tpt_data{'vpn_secret'}       = $config{'vpn_auth_psk'};

    $tpt_data{'incoming_calls'}   = $config{'l2tp_incoming_calls'} ? "yes" : "no";
    $tpt_data{'outgoing_calls'}   = $config{'l2tp_outgoing_calls'} ? "yes" : "no";
    $tpt_data{'primary_number'}   = ($config{'att_number_primary'} or "none");
    $tpt_data{'backup_number'}    = ($config{'att_number_backup'} or "none");
    $tpt_data{'dial_mode'}        = $config{'modem_dial_mode'};
    $tpt_data{'l2tp_user'}        = $config{'l2tp_auth_user'};
    $tpt_data{'l2tp_chap_auth'}   = "$config{l2tp_auth_user} * $config{l2tp_auth_pw} $config{vpn_server}";
    $tpt_data{'remote_ip'}        = $config{'l2tp_remote_host'};

    $tpt_data{'rs_auth_user'}     = $config{'rs_ppp_user'};
    $tpt_data{'rs_chap_auth'}     = "$config{rs_ppp_user} * $config{rs_ppp_pw} *";
    $tpt_data{'rs_hmc_ip'}        = "169.254.2." . hex( $config{'rs_hmc_station'} );
    $tpt_data{'rs_remote_ip'}     = "169.254.0.1";

    # If there is a second init string, we must modify it for the chat script.
    if( $config{'modem_post_init'} ) {
        $tpt_data{'chat_post_init'} = "\"OK\" \"$config{modem_post_init}\"";
    } else {
        $tpt_data{'chat_post_init'} = "";
    }

    $MODEM_ANSWER_PARMS = "--device /dev/modem --baud_rate 57600 --reset $tpt_data{modem_init} --answer $config{modem_answer_init} --script $config{run_as_root}/rcsRemoteService.sh";
    $MONITOR_MODEM_PARMS = "--device /dev/modem --baud_rate 57600 --reset $tpt_data{modem_init} --hangup --script $config{run_as_root}/rcsUnattendedService.sh  --stopprogram $STOP_MODEM";   # (-03)

    return( 1 );
}

##############################################################################
#####                           SCRIPT CODE                              #####
##############################################################################

trace( "!!! starting $dir/$me (@ARGV)" );

unless( $> == 0 ) {
    error( "You must be root." );
}

# Check to see if there is REALLY a daemon in memory.
unless( daemon_exists() )
{
    unlink( $PID_FILE );
    unlink( $FIFO_DAEMON );
}

my $action = "";
my %mm;
GetOptions(
    'checkdaemon'      => sub { $action = 'check_daemon' },

    'data=s'           => \$config{'data'},
    'status'           => sub { $action = 'daemon_status' },
    'quit'             => sub { $action = 'daemon_shutdown' },

    'startdial'        => sub { $action = 'dial_start' },
    'stopdial'         => sub { $action = 'dial_stop' },

    'startremdial'     => sub { $action = 'rem_dial_start' },
    'stopremdial'      => sub { $action = 'rem_dial_stop' },

    'startvpn'         => sub { $action = 'vpn_start' },
    'stopvpn'          => sub { $action = 'vpn_stop' },

    'startremvpn'      => sub { $action = 'rem_vpn_start' },
    'stopremvpn'       => sub { $action = 'rem_vpn_stop' },

    'startremvpnbkp'   => sub { $action = 'rem_vpn_bkp_start' },
    'stopremvpnbkp'    => sub { $action = 'rem_vpn_bkp_stop' },

    'startservicevpn'  => sub { $action = 'service_vpn_start' },
    'stopservicevpn'   => sub { $action = 'service_vpn_stop' },
    'servicestatus'    => sub { $action = 'service_status' },

    'startservicedial' => sub { $action = 'service_dial_start' },
    'setupservicedial' => sub { $action = 'service_dial_setup' },
    'stopservicedial'  => sub { $action = 'service_dial_stop' },

    'runmodemanswer'   => sub { $action = 'modem_answer_run' },
    'stopmodemanswer'  => sub { $action = 'modem_answer_stop' },

    'updateconfig'     => sub { $action = 'update_config' },

    # Added for MegaMouth port forwarding
    'portForwardToLpar=s' => \$mm{'fwd_mtms'},
    'portForwardToIp=s'   => \$mm{'fwd_ip'},
    'hmcPort=s'           => \$mm{'hmc_port'},
    'lparPort=s'          => \$mm{'fwd_port'},
);

unless( $config{'data'} ) {
    $config{'data'} = $ENV{'CONSOLE_PATH'};
}

if( $action eq "check_daemon" )
{
    exit( daemon_exists() ? 0 : $RC_GENERAL_ERR );
}

#if( $help ) {
#    usage();
#    exit;
#}

# Pull the megamouth parms and construct and action string to send to the
# daemon.
if( $mm{'fwd_mtms'} ) {
    $action = "mmPortForwardToLpar $mm{'fwd_mtms'} $mm{'hmc_port'} $mm{'fwd_port'}";
print "DEBUG: fwd_mtms\n";
}
elsif( $mm{'fwd_ip'} ) {
    $action = "mmPortForwardToIp $mm{'fwd_ip'} $mm{'hmc_port'} $mm{'fwd_port'}";
print "DEBUG: fwd_ip ($action)\n";
}

# If an action was requested, send it up to the daemon.
my ($ret, $msg);
if( $action ) {
    ( $ret, $msg ) = daemon_command( $action );
}

if( defined $ret )
{
    unless( $ret )
    {
        if( $msg ) { print( "$msg\n" ); }
        exit( 0 );
    }
    print( STDERR "$msg\n" );
    exit( $ret );
}

# Nothing specified, so let's try to start the daemon.
unless( daemon_exists() )
{
    unless( $config{'data'} ) {
        print( STDERR "Config data directory not specified.\n" );

        if( $ENV{'CONSOLE_PATH'} ) {
            $config{'data'} = "$ENV{CONSOLE_PATH}/data/rcs";
        }
        else {
            $config{'data'} = "data/rcs";
        }

        print( STDERR "Using default: $config{data}\n" );
    }
    unless( -d $config{'data'} ) {
        print( STDERR "Config data directory invalid: $config{data}\n" );
        exit( $RC_GENERAL_ERR );
    }

    daemon_init();
    daemon_reset();

    unless( daemonize() ) {
        print( STDERR "Failed to spawn daemon.\n" );
        exit( $RC_GENERAL_ERR );
    }

    print( "$me spawning...\n" );
    pause( 15 );
    exit( 0 );
} else {
    print( "$me already appears to be running.\n" );
}

trace( "!!! finishing $me" );
exit;
