#!/bin/bash

# usbmemformat.sh
#
# Usage: usbmemformat.sh <device file> <filesystem> <log file> <label>
#
# device file: The device file for the USB flash memory drive, usually given as /dev/sda
# filesystem: The integer representing the filesystem type, either 1 for vfat, 4 for ext2,
#             or 5 for ext3 
# log file: The log file for recording the output of the USB flash memory drive format script
# label: The optionally passed label string used to format the USB flash memory drive with a label
#
# Return Codes:
# 2 - Error formating the USB flash memory drive
#
# Module History
#  00  10/07/2004  P. Provost - Initial release
#  01  03/01/2005  P. Provost - Put double quotes around $LABEL in case the given label contains spaces
#  02  03/17/2005  P. Provost - Fixed tracing
#  03  07/18/2006  J. Kunz    - use mkdosfs -I option to force format on unpartitioned memory keys

DEVICEFILE=$1
FILESYSTEM=$2
LOGFILE=$3
LABEL=$4

echo "-> usbmemformat.sh" >> $LOGFILE
echo "usbmemformat.sh: Formatting USB flash memory drive at $DEVICEFILE for `date`" >> $LOGFILE

actzTrace "XMDA   T: -> usbmemformat.sh"
actzTrace "XMDA   F: usbmemformat.sh: Formatting the USB flash memory drive at $DEVICEFILE for `date`" 

# Determine what filesystem is to be used to format the USB flash memory drive
case "$FILESYSTEM" in
    # If the USB flash memory drive is to be formatted with the vfat filesystem 
    1)
        # Check to see if the label argument exists
        if [ -z "$LABEL" ]
        then
            # Format the USB flash memory drive in vfat with no label
            echo "usbmemformat.sh: Formatting the USB flash memory drive in vfat with -I and no label." >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the USB flash memory drive in vfat with -I and no label."
            
            if ! /sbin/mkdosfs -I -F 32 -v $DEVICEFILE >> $LOGFILE 2>&1; then
                
                echo "usbmemformat.sh: Couldn't format the USB flash memory drive...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format a USB flash memory drive...exiting." 

                echo "<- usbmemformat.sh for `date`" >> $LOGFILE
                echo "" >> $LOGFILE
                
                actzTrace "XMDA   T: <- usbmemformat.sh for `date`"
                
                exit 2
            fi
        else
            # Format the USB flash memory drive in vfat with a label
            echo "usbmemformat.sh: Formatting the USB flash memory drive in vfat with -I and the label: $LABEL" >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the USB flash memory drive in vfat with -I and the label: $LABEL"
            
            if ! /sbin/mkdosfs -I -F 32 -v -n "$LABEL" $DEVICEFILE >> $LOGFILE 2>&1; then
                
                echo "usbmemformat.sh: Couldn't format the USB flash memory drive with a label...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format a USB flash memory drive in vfat with a label...exiting."

                echo "<- usbmemformat.sh for `date`" >> $LOGFILE
                echo "" >> $LOGFILE
                
                actzTrace "XMDA   T: <- usbmemformat.sh for `date`"
                
                exit 2
            fi
        fi
    ;;
    # If the USB flash memory drive is to be formatted with the ext2 filesystem
    4)
        # Check to see if the label argument exists
        if [ -z "$LABEL" ]
        then
            # Format the USB flash memory drive in ext2 with no label
            echo "usbmemformat.sh: Formatting the USB flash memory drive in ext2 with no label." >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the USB memory drive in ext2 with no label."
            
            if ! /sbin/mkfs -t ext2 -v $DEVICEFILE >> $LOGFILE 2>&1; then
                
                echo "usbmemformat.sh: Couldn't format the USB flash memory drive in ext2...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format the USB memory drive in ext2...exiting."

                echo "<- usbmemformat.sh for `date`" >> $LOGFILE
                echo "" >> $LOGFILE
                
                actzTrace "XMDA   T: <- usbmemformat.sh for `date`"
                
                exit 2
            fi
        else
            # Format the USB flash memory drive in ext2 with a label
            echo "usbmemformat.sh: Formatting the USB flash memory drive in ext2 with the label: $LABEL" >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the USB flash memory drive in ext2 with the label: $LABEL" 
            
            if ! /sbin/mkfs -t ext2 -v -L "$LABEL" $DEVICEFILE >> $LOGFILE 2>&1; then
                
                echo "usbmemformat.sh: Couldn't format the USB flash memory drive in ext2 with a label...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format the USB flash memory drive in ext2 with a label...exiting."
                
                echo "<- usbmemformat.sh for `date`" >> $LOGFILE
                echo "" >> $LOGFILE
                
                actzTrace "XMDA   T: <- usbmemformat.sh for `date`"

                exit 2
            fi
        fi
    ;;
    # If the USB flash memory drive is to be formatted with the ext3 filesystem
    5)
        # Check to see if the label argument exists
        if [ -z "$LABEL" ]
        then
            # Format the USB flash memory drive in ext3 with no label
            echo "usbmemformat.sh: Formatting the USB flash memory drive in ext3 with no label." >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the USB flash memory drive in ext3 with no label."
            
            if ! /sbin/mkfs -t ext3 -v $DEVICEFILE >> $LOGFILE 2>&1; then
                
                echo "usbmemformat.sh: Couldn't format the USB flash memory drive in ext3...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format the USB flash memory drive in ext3...exiting."
                
                echo "<- usbmemformat.sh for `date`" >> $LOGFILE
                echo "" >> $LOGFILE
                
                actzTrace "XMDA   T: <- usbmemformat.sh for `date`"
                
                exit 2
            fi
        else
            # Format the USB flash memory drive in ext3 with a label
            echo "usbmemformat.sh: Formatting the USB flash memory drive in ext3 with the label: $LABEL" >> $LOGFILE
            actzTrace "XMDA   F: usbmemformat.sh: Formatting the USB flash memory drive in ext3 with the label: $LABEL" 
            
            if ! /sbin/mkfs -t ext3 -v -L "$LABEL" $DEVICEFILE >> $LOGFILE 2>&1; then
                
                echo "usbmemformat.sh: Couldn't format the USB flash memory drive in ext3 with a label...exiting." >> $LOGFILE
                actzTrace "XMDA   F: usbmemformat.sh: Couldn't format the USB flash memory drive in ext3 with a label...exiting."
                
                echo "<- usbmemformat.sh for `date`" >> $LOGFILE
                echo "" >> $LOGFILE
                
                actzTrace "XMDA   T: <- usbmemformat.sh for `date`"
                
                exit 2
            fi
        fi
    ;;
esac

echo "<- usbmemformat.sh for `date`" >> $LOGFILE
echo "" >> $LOGFILE

actzTrace "XMDA   T: <- usbmemformat.sh for `date`"

exit 0

