<!--
/*
Function that handles the 'previous' (down) button being pressed on a spinner

                                                        
Parameters:
spinnerName- The ID on the page of the spinner being modified
min- minimum value of the spinner
max- maximum value of the spinner
increment- how much to modify the spinner by
wrap- Whether to wrap the spinner if the min or max is hit
*/
function prev(spinnerName, min, max, increment, wrap) {
    var spinnerValue = parseInt(spinnerName.value);
    if (isNaN(spinnerValue)) {
        spinnerValue = min;
    }

    // if the spinner is negative, the incrementVal will be negative
    var incrementVal = (spinnerValue % increment);

    //alert("increment: " + increment);
    //alert("spinnerValue: " + spinnerValue);
    //alert("incrementVal: " + incrementVal);

    if (incrementVal == 0) {
        incrementVal = increment;
        if (spinnerValue < 0) {
            incrementVal = incrementVal*-1;
        }
    }

    if (spinnerValue < 0) { // Negative spinner value
        if (((spinnerValue + incrementVal) < min) && wrap) { // Wrap from below min to the maximum
            spinnerValue = max;
        } else if ((spinnerValue + incrementVal) < min) { // Don't wrap, just stay at minimum
            spinnerValue = min;
        } else { // Not  below minimum
            spinnerValue = spinnerValue + incrementVal;
            if (spinnerValue > max) { // In case we don't use an onblur handler, they could type a large number
                spinnerValue = max;
            }
        }
    } else { // Positive spinner value
        if (((spinnerValue - incrementVal) < min) && wrap) { // Wrap from below min to the maximum
            spinnerValue = max;
        } else if ((spinnerValue - incrementVal) < min) { // Don't wrap, just stay at minimum
            spinnerValue = min;
        } else { // Not below minimum
            spinnerValue = spinnerValue - incrementVal;
            if (spinnerValue > max) { // In case we don't use an onblur handler, they could type a large number
                spinnerValue = max;
            }
        }
    }

    spinnerName.value = spinnerValue; // Update the spinner to the new value
}

/*
Function that handles the 'next' (up) button being pressed on a spinner

                                                        
Parameters:
spinnerName- The ID on the page of the spinner being modified
min- minimum value of the spinner
max- maximum value of the spinner
increment- how much to modify the spinner by
wrap- Whether to wrap the spinner if the min or max is hit
*/ 
function next(spinnerName, min, max, increment, wrap) {
    var spinnerValue = parseInt(spinnerName.value);
    if (isNaN(spinnerValue)) {
        spinnerValue = min;
    }

    // if the spinner is negative, the incrementVal will be negative
    var incrementVal = increment - (spinnerValue % increment);

    if (spinnerValue < 0) {
        incrementVal = incrementVal*-1;
    }

    if (spinnerValue < 0) { // Negative spinner value
        if (((spinnerValue - incrementVal) > max) && wrap) { // Wrap from above maximum to minimum
            spinnerValue = min;
        } else if ((spinnerValue - incrementVal) > max) { // Don't wrap, just stay at maximum
            spinnerValue = max;
        } else {
            spinnerValue = spinnerValue - incrementVal;
            if (spinnerValue < min) { // In case we don't use an onblur handler, they could enter a number smaller than the minimum
                spinnerValue = min;
            }
        }
    } else { // Positive spinner value
        if (((spinnerValue + incrementVal) > max) && wrap) { // Wrap from above maximum to minimum
            spinnerValue = min;
        } else if ((spinnerValue + incrementVal) > max) { // Don't wrap, just stay at the max
            spinnerValue = max;
        } else {
            spinnerValue = spinnerValue + incrementVal;
            if (spinnerValue < min) { // In case we don't use an onblur handler, they could enter a number smaller than the minimum
                spinnerValue = min;
            }
        }
    }
    spinnerName.value = spinnerValue;
}

/*
Function that handles the 'onblur' handler for a single spinner

                                                        
Parameters:
spinnerName- The ID on the page of the spinner being modified
min- minimum value of the spinner
max- maximum value of the spinner
increment- how much to modify the spinner by
wrap- Whether to wrap the spinner if the min or max is hit
customJSName- the name of a javascript function to invoke after the spinner has been modified
*/
function blurHandler(spinnerName, min, max, increment, wrap, customJSName) {
    var spinnerValue = parseInt(spinnerName.value);
    if (spinnerValue != spinnerName.value || isNaN(spinnerValue) ) {
        spinnerValue = min;
        spinnerName.value = spinnerValue;
    }

    if (spinnerValue < min) { // Convert a value lower than the minimum to the min
        spinnerValue = min;
        spinnerName.value = spinnerValue;
    } else if (spinnerValue > max) { // Convert a value greater than the maximum to the max
        spinnerValue = max;
        spinnerName.value = spinnerValue;
    } else {
        // if the spinner is negative, the incrementVal will be negative
        var incrementVal = (spinnerValue % increment);

        var halfway = increment / 2;

        if (incrementVal == 0) {
            // do nothing, they're on an increment
        } else if (incrementVal >= halfway ) { // Closer to the next one up, hit the 'next' button
            next(spinnerName, min, max, increment, wrap);
        } else if (incrementVal < halfway) { // closer to the previous one, hit the 'prev' button
            prev(spinnerName, min, max, increment, wrap);
        }
    }  
    if (customJSName != null) {
        customJSName();
    }

}

/*
Function that handles the 'onblur' handler for a pair of inter-related spinners

                                                        
Parameters:
spinnerName- The ID on the page of the first spinner being modified
min- minimum value of the first spinner
max- maximum value of the first spinner
increment- how much to modify the first spinner by
wrap- Whether to wrap the first spinner if the min or max is hit
spinner2Name- The ID on the page of the second spinner being modified
min2- minimum value of the second spinner
max2- maximum value of the second spinner
increment2- how much to modify the second spinner by
customJSName- the name of a javascript function to invoke after the spinners have been modified
*/

function blurHandler2(spinnerName, min, max, increment, wrap, spinnerName2, min2, max2, increment2, customJSName) {
    var spinnerValue = parseInt(spinnerName.value);
    if (spinnerValue != spinnerName.value || isNaN(spinnerValue) ) {
        spinnerValue = min;
        spinnerName.value = spinnerValue;
    }

    if (spinnerValue < min) {
        spinnerValue = min;
        spinnerName.value = spinnerValue;
        spinnerName2.value = max2;
    } else if (spinnerValue > max) {
        spinnerValue = max;
        spinnerName.value = spinnerValue;
        spinnerName2.value = min2;
    } else {
        var amountGained = spinnerValue - min;

        // number of increments
        var numIncrements = (amountGained / increment);
        numIncrements = parseInt( numIncrements ); // Make a whole number

        // maximum number of increments
        var maxNumIncrements = ((max - min) / increment);
        maxNumIncrements = parseInt( maxNumIncrements); // Make a whole number

        //alert("numIncrements: " + numIncrements);

        // if the spinner is negative, the incrementVal will be negative
        var incrementVal = (spinnerValue % increment);

        var halfway = increment / 2;

        if (incrementVal == 0) {
            // do nothing for spinner 1

            spinnerName2.value = max2;

            var i = 0;
            while (i < numIncrements) {
                prev(spinnerName2, min2, max2, increment2, wrap);
                i = i + 1;
            }
        } else if (incrementVal >= halfway) {
            next(spinnerName, min, max, increment);

            spinnerName2.value = max2;

            var i = 0;
            while (i < numIncrements+1) {
                prev(spinnerName2, min2, max2, increment2, wrap);
                i = i + 1;
            }
        } else if (incrementVal < halfway) {
            prev(spinnerName, min, max, increment);

            spinnerName2.value = max2;
            var i = 0;
            while (i < numIncrements) {
                prev(spinnerName2, min2, max2, increment2, wrap);
                i = i + 1;
            }
        }
    }  
    if (customJSName != null) {
        customJSName();
    }
}

/* Global timer variables */
// Timer config items
var delay = 500; // .5 sec
var timerRunning = false;

var timerFunction = null; // prev or next

// Details on how the timer is related to the first spinner
var timerMin = 0;
var timerMax = 0;
var timerIncrement = 0;
var timerWrap = false;
var timerSpinnerName = null;

// Details on how the timer is related to the second spinner
var timerMin2 = 0;
var timerMax2 = 0;
var timerIncrement2 = 0;
var timerSpinnerName2 = null;

var timerCustomJSName = null;

/*
Function that starts the timer off for a single spinner.

Parameters-
spinnerName- The name of the spinner to modify
min- The minimum value of this spinner
max- The maximum value of this spinner
increment- The amount to increase or decrease the spinner by.
wrap- Whether to wrap the spinner around once the maximum or minimum is hit
functionName- Whether to go forward or backward. Should be 'nextFunction' or 'prevFunction' accordingly. 
customJSName- the name of a javascript function to invoke after the spinner has been modified
*/
function startTimer(spinnerName, min, max, increment, wrap, functionName, customJSName) {
    //alert("startTimer");
    timerFunction = functionName;

    timerMin = min;
    timerMax = max;
    timerIncrement = increment;
    timerWrap = wrap;
    timerSpinnerName = spinnerName;
    timerCustomJSName = customJSName;

    timerRunning = true;

    self.setTimeout("timerTick()", 0)
}

/*
Function that starts the timer off for a pair of inter-related spinners.

Parameters-
spinnerName- The name of the first spinner to modify
min- The minimum value of the first spinner
max- The maximum value of the first spinner
increment- The amount to increase or decrease the first spinner by.
spinnerName2- The name of the second spinner to modify
min2- The minimum value of the second spinner
max2- The maximum value of the second spinner
increment2- The amount to increase or decrease the second spinner by.
wrap- Whether to wrap the second spinner around once the maximum or minimum is hit
functionName- Whether to go forward or backward on the first spinner. Should be 'nextFunction' or 'prevFunction' accordingly. 
customJSName- the name of a javascript function to invoke after the spinners have been modified
*/
function startTimer2(spinnerName, min, max, increment, spinnerName2, min2, max2, increment2, wrap, functionName, customJSName) {
    //alert("startTimer2");
    timerFunction = functionName;

    timerMin = min;
    timerMax = max;
    timerIncrement = increment;
    timerSpinnerName = spinnerName;

    timerMin2 = min2;
    timerMax2 = max2;
    timerIncrement2 = increment2;
    timerSpinnerName2 = spinnerName2;

    timerCustomJSName = customJSName;
    //alert ("customJSName: " + timerCustomJSName);

    timerWrap = wrap;
    timerRunning = true;

    self.setTimeout("timerTick2()", 0);
}


/*
Function that starts the timer off for a triplet of inter-related spinners.

Parameters-
spinnerName- The name of the first spinner to modify
min- The minimum value of the first spinner
max- The maximum value of the first spinner
increment- The amount to increase or decrease the first spinner by.
spinnerName2- The name of the second spinner to modify
min2- The minimum value of the second spinner
max2- The maximum value of the second spinner
increment2- The amount to increase or decrease the second spinner by.
wrap- Whether to wrap the second spinner around once the maximum or minimum is hit
functionName- Whether to go forward or backward on the first spinner. Should be 'nextFunction' or 'prevFunction' accordingly. 
customJSName- the name of a javascript function to invoke after the spinners have been modified
*/
function startTimer3(spinnerName, min, max, increment, spinnerName2, min2, max2, increment2, functionName, customJSName) {
    //alert("startTimer3");
    timerFunction = functionName;

    timerMin = min;
    timerMax = max;
    timerIncrement = increment;
    timerSpinnerName = spinnerName;

    timerMin2 = min2;
    timerMax2 = max2;
    timerIncrement2 = increment2;
    timerSpinnerName2 = spinnerName2;

    timerCustomJSName = customJSName;

    timerRunning = true;

    timerWrap = false; // no wrap in this flavor

    self.setTimeout("timerTick3()", 0);
}



/*
Method called after 'delay' seconds have passed.  This is only used for a single spinner.

If the function is the 'prevFunction', we call prev() on the spinner.

We then restart the timer to pop in 'delay' seconds again.
*/
function timerTick() {
    //alert("timerTick");
    if (timerRunning) {
        if (timerFunction == "prevFunction") {
            //alert("Prev");
            prev(timerSpinnerName, timerMin, timerMax, timerIncrement, timerWrap);

            if (timerCustomJSName != null) {
                timerCustomJSName();
            }
            self.setTimeout("timerTick()", delay); // Restart timer
        } else if (timerFunction == "nextFunction") {
            //alert("Next");
            next(timerSpinnerName, timerMin, timerMax, timerIncrement, timerWrap);

            if (timerCustomJSName != null) {
                timerCustomJSName();
            }
            self.setTimeout("timerTick()", delay); // Restart timer
        }  else {
            //alert ("Hmm: " + timerFunction);
        }
    }
}

/*
Method called after 'delay' seconds have passed.  This is used for a pair of spinners.

If the function is the 'prevFunction', we call prev() on the first timer.  
The second spinner has the next() function called, provided we aren't at the limit and are non-wrapping.

We then restart the timer to pop in 'delay' seconds again.
*/
function timerTick2() {
    //alert("timerTick2");
    if (timerRunning) {
        if (timerFunction == "prevFunction") {
            oldTimer2Value = timerSpinnerName2.value;

            if ((oldTimer2Value != timerMax2) || (timerWrap)) {
                //alert("Prev");
                oldTimerValue = timerSpinnerName.value;
                prev(timerSpinnerName, timerMin, timerMax, timerIncrement, timerWrap);
                newTimerValue = timerSpinnerName.value;
                if (oldTimerValue != newTimerValue) {
                    next(timerSpinnerName2, timerMin2, timerMax2, timerIncrement2, timerWrap);
                }
            }

            if (timerCustomJSName != null) {
                timerCustomJSName();
            }
            self.setTimeout("timerTick2()", delay); // Restart timer
        } else if (timerFunction == "nextFunction") {
            //alert("Next");
            oldTimer2Value = timerSpinnerName2.value;

            if ((oldTimer2Value != timerMin2) || (timerWrap)) {
                oldTimerValue = timerSpinnerName.value;
                next(timerSpinnerName, timerMin, timerMax, timerIncrement, timerWrap);
                newTimerValue = timerSpinnerName.value;
                if (oldTimerValue != newTimerValue) {
                    prev(timerSpinnerName2, timerMin2, timerMax2, timerIncrement2, timerWrap);
                }
            }

            if (timerCustomJSName != null) {
                timerCustomJSName();
            }
            self.setTimeout("timerTick2()", delay); // Restart timer
        }  else {
            //alert ("Hmm: " + timerFunction);
        }
        
    }
}

/*
Method called after 'delay' seconds have passed.  This is used for a pair of non wrapping spinners

We then restart the timer to pop in 'delay' seconds again.
*/
function timerTick3() {
    //alert("timerTick23);
    if (timerRunning) {
        if (timerFunction == "prevFunction") {
            //alert("prev");
            oldTimer1Value = timerSpinnerName.value;
            oldTimer2Value = timerSpinnerName2.value;

            if (parseInt(oldTimer2Value) + timerIncrement2 <= timerMax2) {

                prev(timerSpinnerName, timerMin, timerMax, timerIncrement, false);
                newTimerValue = timerSpinnerName.value;
                if (oldTimer1Value != newTimerValue) {
                    next(timerSpinnerName2, timerMin2, timerMax2, timerIncrement2, false);
                }
            }

            if (timerCustomJSName != null) {
                timerCustomJSName();
            }
            self.setTimeout("timerTick3()", delay); // Restart timer

        } else if (timerFunction == "nextFunction") {

            //alert("next");

            var oldTimer1Value = timerSpinnerName.value;
            var oldTimer2Value = timerSpinnerName2.value;

            if (oldTimer2Value - timerIncrement2 >= timerMin2) {
                next(timerSpinnerName, timerMin, timerMax, timerIncrement, timerWrap);
                newTimerValue = timerSpinnerName.value;
                if (oldTimer1Value != newTimerValue) {
                    prev(timerSpinnerName2, timerMin2, timerMax2, timerIncrement2, timerWrap);
                }
            }

            if (timerCustomJSName != null) {
                timerCustomJSName();
            }
            self.setTimeout("timerTick3()", delay); // Restart timer
        } else {
            //alert ("Hmm: " + timerFunction);
        }
    }
}

/*
Method that stops the timer from processing.

Most importantly, the 'timerRunning' variable is turned off.
All other timer variables are nulled out.
*/
function stopTimer() {
    //alert("stopTimer");
    timerRunning = false;

    if (timerCustomJSName != null) {
        timerCustomJSName();
    }

    timerMin = 0;
    timerMax = 0;
    timerIncrement = 0;
    timerFunction = null;
    timerSpinnerName = null;

    timerMin2 = 0;
    timerMax2 = 0;
    timerIncrement2 = 0;
    timerSpinnerName2 = null;

    timerCustomJSName = null;

    timerWrap = false;
}
//-->