/**
 * This file contains the initial script for the hidden window which is connected to
 * the Task Controller servlet. This is a collection of functions used to implement
 * the methods in the TaskControllerQueue class, as well as a global error handler
 * and a heartbeat mechanism to reload the frame should the request be aborted.
 *
 * Please note that several of the function and variable names and values must
 * correspond with names and values in TaskControllerQueue and TaskControllerServlet.
 */

// This is how often we verify that we've received some packet from the server.
// The interval should be larger than the server's heartbeat interval, plus
// some to allow for network delays.  The server's heartbeat interval is
// TaskControllerServlet.DEFAULT_HEARTBEAT_WAIT.
var connectionCheckInterval = 45 * 1000;      // 45 seconds

// Indicates whether the client browser is still connected to the
// TaskControllerServlet.  Do not change, remove or rename this variable
// without making the corresponding change to
// TaskControllerServlet.HEARTBEAT_SCRIPT, which sets this variable from the
// servlet.
var connectedToServer = true;

// Simple debugging support.
// value of "console" presents debug info in a seperate console window
// value of "alert" presents debug info in alerts
// any other value, no debugging presented
var tcdebug = "";

// are we using the applet version of the task controller
var usingApplet = false;
// did the applet load
var appletLoaded = false; // set true by the applet itself

// count of bad connection checks
var checkConnectionCount = 0;
// max reconnect failures before we put up the alert window
var MAX_RECONNECT_FAILURES=1;
// current reconnect failures
var current_reconnect_failures=0;
// msg to the user when connectivity was lost, *should* be overwritten in initialize routines
if (top.TCwinMsgGone==null) {
	top.TCwinMsgGone ="<HTML><HEAD><TITLE>Console Connectivity Problem</TITLE></HEAD><BODY>";
	top.TCwinMsgGone+="<p><b><font color=\"red\">Connectivity to the console has been lost. Do not attempt to launch any new tasks or interact with any existing tasks until connectivity is restored.</font></b>";
	top.TCwinMsgGone+="<BR><BR>This window will close automatically when connectivity is restored, otherwise close the console windows by clicking the X in the upper right hand corner of the windows and try to relogon to the console.";
}

// alert window handle
var alertWinRef=null;

/**
 * Perform some initialization.  Some of this was previously done outside
 * any function in this file, but some browsers (IE) weren't loading the file
 * until long after the page was delivered, and therefore not starting the
 * timer for a while.  There's now an explicit call to the initialize()
 * method after including TCStart.js in order to force the browser to read the
 * file immediately.
 * */
function initialize(msg, jsdebug) {
   tcdebug=jsdebug;
   alertIfDebug("initialize()");
   top.TCwinMsgGone=msg;

   // Install the onload handler.
   window.onload=loaded;

   // Install the window's error handler.
   //window.onerror=handleError;
   setUpJSError();

   window.onunload=closeAlertWindow;

   // Start checking to make sure data is still being received from the pending
   // HTTP request.  We use setTimeout() instead of setInterval() because
   // setInterval() doesn't seem to react well to clock changes.  We therefore
   // schedule each timer pop one at a time.
   window.setTimeout( "checkConnection()", connectionCheckInterval );
}

/**
 *
 * Initialization for the applet flavor
 *
 */
function initialize2(msg, jsdebug) {
   tcdebug=jsdebug;
   alertIfDebug("initialize2()");
   top.TCwinMsgGone=msg;   

   // Install the window's error handler.
   //window.onerror=handleError;
   setUpJSError();

   window.onunload = handleUnload;

   usingApplet = true;

   // Start checking to make sure data is still being received from the pending
   // HTTP request.  We use setTimeout() instead of setInterval() because
   // setInterval() doesn't seem to react well to clock changes.  We therefore
   // schedule each timer pop one at a time.
   window.setTimeout( "checkConnection()", connectionCheckInterval );
}

/*
 * Handle the unload of the task controller frame
 * for the applet flavor
 *
 */
function handleUnload() {
   alertIfDebug("handleUnload()");
   try {
      if ("appletReady" in window) {
         tcapplet = window.document.applets[0];
         if (tcapplet != null) {
            tcapplet.setShowLogoff(false);
         }
      }
   }
   catch (ex) {
      window.alert("exception in unload: "+ex);
   }
   closeAlertWindow();
}

/*
 * For debug messages
 *
 */
function alertIfDebug( msg )
{
  if (tcdebug=="console" || top.scriptWinRef)
  {
	if (!top.scriptWinRef || (top.scriptWinRef && top.scriptWinRef.closed) ) {
		top.scriptWinRef= window.open("","tcscript_"+top.window.name, "toolbar=no,location=no,directories=no,status=no,menubar=no,resizable=yes,scrollbars=yes,width=600,height=400");
		top.scriptWinRef.document.open();
		top.scriptWinRef.document.write("<HTML><HEAD><TITLE>HMC Script Console</TITLE></HEAD><BODY><DIV ID='OUTPUT' ></DIV></BODY></HTML>");
		top.scriptWinRef.document.close();
	}

	el = top.scriptWinRef.document.getElementById("OUTPUT");
	el.appendChild(top.scriptWinRef.document.createTextNode(new Date().toLocaleString()+": "+msg+"\n"));  	
	el.appendChild(top.scriptWinRef.document.createElement("br"));
	//alert("elCH="+el.clientHeight+",SH="+el.scrollHeight);
	//if (!isNaN(el.clientHeight) && el.clientHeight>0) {
	top.scriptWinRef.scrollTo(0,el.scrollHeight);
	// 
  } else if (tcdebug=="alert") {
    alert( msg );
  }
  
}

/**
 * Window scope error handler.  We don't want to show javascript errors
 * to the user, but we don't want to completely ignore them either.  What we'll
 * do is reload the page, passing the error information in the query part of the
 * url.  Not much we can really do to fix things, but the servlet can at least
 * log the problem and we can be sure that the javascript error can't mess up
 * future scripts sent to this frame.
 *
 * @param text
 * @param url
 * @param line
 *
 *
 * NOTE: Currently defunct with the use of SetUpJSError();
 */
function handleError( text, url, line )
{
  alertIfDebug( "handleError(): errorText="+text+"\nerrorURL="+url+"\nerrorLine="+line );
  closeAlertWindow(); // about to reload

  window.location.replace(window.location.pathname+"?reason=scriptError&errorText="+
                          encodeURIComponent(text)+"&errorURL="+encodeURIComponent(url)+
                          "&errorLine="+encodeURIComponent(line));
  return true;
}

/**
 * Reload the frame in which this javascript is running.
 *
 * Do not change, remove or rename this function without also changing
 * TaskControllerServlet.RELOAD_SCRIPT, which invokes this function from the servlet.
 */
function reloadHiddenFrame()
{
  alertIfDebug( "reloadHiddenFrame(): Reloading by request of server" );
  closeAlertWindow(); // about to reload
  window.location.replace(window.location.pathname+"?reason=serverInitiated");
}

/**
 * The following code handles restarting the task controller connection if
 * the pending request should somehow abort.
 *
 * The idea is that every piece of script sent from the server will set
 * connectedToServer to true.  The checkConnection function is set to be
 * called every connectionCheckInterval msecs.  The connectedToServer variable
 * is only set to true by scripts arriving from the server, so if the checkConnection
 * function finds this true, the server must still be there.  The function then
 * sets connectedToServer to be false.  The ratio of the
 * connectionCheckInterval to the heartbeat interval at the server should be set so
 * that new packets arrive more frequently than they are checked.  Therefore, if
 * the checkConnection function ever finds that connectedToServer is still false,
 * the server must not still be connected, so
 * the task controller hidden frame is reloaded.
 */
function checkConnection()
{
  alertIfDebug( "checkConnection() connectedToServer = " + connectedToServer + ", checkConnectionCount = " + checkConnectionCount +", current_reconnect_failures = "+ current_reconnect_failures  );

  window.setTimeout( "checkConnection()", connectionCheckInterval );  

  if ( connectedToServer )
  {
    connectedToServer = false;
    checkConnectionCount = 0;
    current_reconnect_failures=0;
   	closeAlertWindow(); // if its still there
  }
  else
  {
    doit = true;
    if (usingApplet && appletLoaded) {
       // Time changes can cause this timer to pop incorrectly, which causes the
       // frame to get reloaded. To get around this we will need to get 2 failed
       // checks in a row to consider the connection lost.
       if (checkConnectionCount < 2) { // keep from incrementing forever
	       checkConnectionCount++;
       }
       if (checkConnectionCount < 2) {
          doit = false;
       }
    }
    if (doit) {
 		
 	   // increment the connect check retry count, then try to async check if the server is there   
 	   // to reload the window location. If we don't reload then alert the user.
  	   if (current_reconnect_failures < MAX_RECONNECT_FAILURES) {
          current_reconnect_failures++;
       }
       alertIfDebug("current reconnect failures: "+current_reconnect_failures);
       
       try {
        testURL = "/js/taskcontroller/TCping.js?time="+new Date().getTime();
        alertIfDebug(testURL);
	    new AsyncJson( testURL,
               function(json) {
                alertIfDebug("entered my json function");
                if (json==null) {
                	alertIfDebug("JSON object is null");
			  	    // alert the user we lost connectivity if we are at the max retry count 
				 	if (current_reconnect_failures >= MAX_RECONNECT_FAILURES) {
				 		openAlertWindow(top.TCwinMsgGone);
				    }      	 	
                } else {
                   // server responded, ok to reload the frame url
                   	
			       closeAlertWindow(); // if its still there
				    
			       if (usingApplet && !appletLoaded) {
			       	   // applet failed to load or connect
			       	   // reload the frame with the appletLoadFailed reason so tc servlet will force
			       	   // disconnect the user
				      if ("appletReady" in window) {
				        // applet loaded, so couldn't connect to HMC
			       	   alertIfDebug("Applet failed to connect, force disconnect the user");
			    	   window.location.replace(window.location.pathname+"?reason=appletConnectFailed");       	   
				      } else {
			       	   alertIfDebug("Applet failed to load, force disconnect the user");
			    	   window.location.replace(window.location.pathname+"?reason=appletLoadFailed");       	   
			    	  }
			       } else {
				       alertIfDebug( "Reloading since haven't heard from server in a while" );
			    	   window.location.replace(window.location.pathname+"?reason=lostServer");
			       }
               	}
               	return true;                     
               }
	    );
	 } catch( e ) {
	 	//couldn't get XMLHttpRequest stuff going
	 	alertIfDebug("JSON exception: "+e);
  	    // alert the user we lost connectivity if we are at the max retry count
	 	if (current_reconnect_failures >= MAX_RECONNECT_FAILURES) {
	 		openAlertWindow(top.TCwinMsgGone);
	    }      	 	
	 }
   }
  }
}

/**
 *
 * Open the alert window if the window handle is null, or the window was closed by hand
 *
 */
function openAlertWindow(msg) {
	if (!alertWinRef || (alertWinRef && alertWinRef.closed) ) {
		alertIfDebug("openAlertWindow() open the window");
		alertWinRef= window.open("","tcalert_"+top.window.name, "toolbar=0,location=0,directories=0,status=0,menubar=0,resizable=1,scrollbars=1,width=300,height=300");
		alertWinRef.document.open();
		alertWinRef.document.write(msg);
		alertWinRef.document.close();
		alertWinRef.focus();
	}
}

/**
 *
 * Close the alert window if the window handle is not null, the window.open function exists and not its closed already 
 *
 */
function closeAlertWindow() {
	if (alertWinRef && alertWinRef.open && !alertWinRef.closed) {
		try {
			alertIfDebug("closeAlertWindow() close the window");
			alertWinRef.close();
		} catch (e) {
		}
	}
	alertWinRef=null;
}

/**
 * Open a window.
 *
 * @param url     the url for the new window's content
 * @param name    the name of the new window
 * @param options the JavaScript options string describing the window
 * @param relativeWidth the percentage of the screen used for the new window's width
 * @param relativeHeight the percentage of the screen used for the new window's height
 */
function openWindow( url, name, options, relativeWidth, relativeHeight )
{
  alertIfDebug("openWindow() url="+url+", name="+name);
  if ( relativeWidth > 0 )
  {
    newWidth = window.screen.width * (relativeWidth / 100);
    options += ",width=" + Math.round( newWidth );
  }

  if ( relativeHeight > 0 )
  {
    newHeight = window.screen.height * (relativeHeight / 100);
    options += ",height=" + Math.round( newHeight );
  }
  return window.open( url, name, options )
}

/**
 * Refresh a window.  This means that if the window exists, its contents
 * should be reloaded.  If the window does not exist, it should be
 * created just like in the openWindow method above.
 *
 * @param name    the name of the new window
 * @param url     the url for the new window's content
 * @param options the JavaScript options string describing the window
 * @param relativeWidth the percentage of the screen used for the new window's width
 * @param relativeHeight the percentage of the screen used for the new window's height
 * @param getFocus whether the refreshed window should receive focus
 */
function refreshWindow( name, url, options, relativeWidth, relativeHeight, getFocus )
{
  alertIfDebug("refreshWindow() url="+url+", name="+name);
  if ( getFocus == "true" ) {
     theWindow = openWindow( "", name, options, relativeWidth, relativeHeight );
     // If the hmcTaskId property is present, we didn't just open the window,
     // so it's safe just to give it focus.  Otherwise we must reload it.
     if ("hmcTaskId" in theWindow) {
        theWindow.focus();
        return;
     }
  }

  theWindow = openWindow( url, name, options, relativeWidth, relativeHeight );
  if ( getFocus == "true" )
  {
    theWindow.focus();
  }
}

/**
 * This frame should never completely load.  If it does, we'll immediately
 * reload it.
 */
function loaded()
{
  alertIfDebug( "loaded() Reloading since page onload event handler driven" );
  closeAlertWindow(); // about to reload
  window.location.replace(window.location.pathname+"?reason=connectionClosed");
}

/**
 * Log the task controller exception
 *
 */
function logTCException(e, callerInfo) {

	text = e['name'] + ': ' + e['message'];
	url = "TBD";
	line="0";
	
	for (var p in e) {
		if (p == "lineNumber") line = e[p];
		if (p == "fileName") url = e[p];
		if (p == "stack") text += ", stack: " + e[p];		
	}

    alertIfDebug("logTCException: "+text +"\nurl="+url+"\nline="+line);
	
	start = window.location.href.indexOf(window.location.pathname);
	reloadURL = window.location.href.substr(start); 
	start = reloadURL.indexOf("?");
	if (start>-1) {
		reloadURL = reloadURL.substr(0,start);
	}
	
    var o = (window.XMLHttpRequest) ? new XMLHttpRequest() : new ActiveXObject('Microsoft.XMLHTTP');
    if (o)
    {
       o.open("POST", "/hmc/javascriptError?errorText="+encodeURIComponent(text)+"&errorURL="+encodeURIComponent(url)+"&errorLine="+encodeURIComponent(line)+"&reloadURL="+encodeURIComponent(reloadURL)+"&callerInfo="+encodeURIComponent(callerInfo)+"&windowName="+encodeURIComponent(window.name), true);
       o.send("");
    }     
	
}
