package OSInstall::Client;

use strict;
use warnings;
use Socket;
use Carp;
use OSInstall::Common;
use OSInstall::Platform_Tools;
use OSInstall::Control_Host;
use OSInstall::OS_Resource;
use OSInstall::AIX_Resource;
use OSInstall::Linux_Resource;
our $AUTOLOAD;

our $VERSION = '1.0';

my %modifiable_fields= (
		name => "",
		ip_addr => "",
		gateway => "",
		mac_addr => "",
		speed => "",
		duplex => "",
		subnet_mask => "",
		target_disk => "",
		control_host => "",
		lpar => "",
		profile => "",
		managed_sys => "",
		hostname => "",
		config_ip => "",
		allocation => "",
		boot_params => "",
		config_file => "",
	);

my %constant_fields= ();

sub new
{
	my $class_name= shift;
	my $this; #what we will bless into an object
	
	#the first six args are required
	confess "usage OSInstall::Client->new(name, ip_addr, gateway, mac_addr, speed, duplex[, subnet_mask, disks_array_ref, control_host_ref lpar, profile, managed_sys, allocation])\n" unless (@_ >= 6);

	$modifiable_fields{name}= shift;
	$modifiable_fields{ip_addr}= shift;
	$modifiable_fields{gateway}= shift;
	$modifiable_fields{mac_addr}= shift;
	$modifiable_fields{speed}= shift if (@_);
	$modifiable_fields{duplex}= shift if (@_);
	$modifiable_fields{subnet_mask}= shift if (@_);
	$modifiable_fields{target_disk}= shift if (@_);  #array ref to installable disks
	$modifiable_fields{control_host}= shift if (@_); #ref to control_host object
	$modifiable_fields{lpar}= shift if (@_);
	$modifiable_fields{profile}= shift if (@_);
	$modifiable_fields{managed_sys}= shift if (@_);
	$modifiable_fields{allocation}= shift if (@_);

	$modifiable_fields{hostname}= (gethostbyaddr(inet_aton($modifiable_fields{ip_addr}), AF_INET))[0];
	
	unless ($modifiable_fields{hostname})
	{
		$modifiable_fields{hostname}= $modifiable_fields{ip_addr};
	}

	unless ($modifiable_fields{speed})
	{
		$modifiable_fields{speed}= "auto";
	}
	unless ($modifiable_fields{duplex})
	{
		$modifiable_fields{duplex}= "auto";
	}

	# insert :'s into the mac address if it doesn't have them already
	unless ($modifiable_fields{mac_addr} =~ /([0-9a-fA-F][0-9a-fA-F]:){5}([0-9a-fA-F][0-9a-fA-F])/)
	{
		$modifiable_fields{mac_addr} =~ s/([0-9a-fA-F]{2})/$1:/g;
		chop $modifiable_fields{mac_addr};
	}

	$modifiable_fields{config_ip}= 1;

	# see AUTOLOAD below for what _modifiable does- it's not meant for direct external use
	$this= {
		_modifiable => \%modifiable_fields,
		%modifiable_fields,
		%constant_fields,
	};

	bless $this, $class_name;
	return $this;
}

sub AUTOLOAD
{
	my $this= shift; #ref to the class
	my $class_name= ref($this);
	my $param= shift; 

	#must take a reference to this class
	croak "Invalid reference type\n" unless ($class_name =~ /^(?:OSInstall::)?Client$/);

	my $attr= $AUTOLOAD; #the field we're autoloading
	$attr =~ s/.*://; #strip out full package name

	#error if we don't have this field
	croak "The field \"$attr\" does not exist\n" unless (exists $this->{$attr});

	# if we have a parameter to set the attr to, make sure it's a modifiable attr
	if ($param)
	{
		# croak if this field is not modifiable
		croak "This field cannot be modified\n" unless (exists $this->{_modifiable}->{$attr});

		$this->{$attr}= $param; #damage is done
	}

	return $this->{$attr};
}

sub netboot
{
	# shift args
	my $this= shift;

	# report error if we don't ahve a control host
	unless ($this->control_host)
	{
		log_entry(__("The client %s cannot be netbooted because it does not have a hardware controlling host associated with it\n", $this->name), 1);
		return 0;
	}

	$this->control_host->netboot($this);
}

sub monitor_installation
{
	# shift args
	my $this= shift;

	# local data
	my $prev_status;
	my $new_status= [];
	my $each_line;
	my $rc= 1;

	# first, read and ignore all status for this client that was previously there
	$prev_status= $this->{allocation}->get_install_status($this, []);

	# loop perpetually until the install finishes or times out, sleep a few seconds each time
	while (! grep /FINISHED/, @$new_status)
	{
		$new_status= $this->allocation->get_install_status($this, $prev_status);

		foreach $each_line (@$new_status)
		{
			log_entry($each_line) unless ($each_line =~ /^FINISHED/);
		}

		push @$prev_status, @$new_status;
		
		sleep 5;
	}

	log_entry(__("Installation of client %s has completed\n", $this->name));

	$rc;
}

1;

