package OSInstall::Control_Host;

use strict;
use warnings;
use IO::Socket;
use Sys::Hostname;
use OSInstall::Client;
use OSInstall::Common;
use Carp;
our $AUTOLOAD;

our $VERSION = '1.0';

my %modifiable_fields= (
		name => "",
		communication_method => "ssh",
		host_name => "",
		type => "",
	);

my %constant_fields= ();

sub new
{
	my $class_name= shift;
	my $this; #what we will bless into an object
	
	#The first 3 args are required
	confess "usage OSInstall::Control_Host->new(name, host_name, type [,communication_method])\n" unless (@_ >= 3);

	$modifiable_fields{name}= shift;
	$modifiable_fields{host_name}= shift;
	$modifiable_fields{type}= shift;
	$modifiable_fields{communication_method}= shift if (@_);

	# turn type and comm_method into lowercase
	$modifiable_fields{type}= lc $modifiable_fields{type};
	$modifiable_fields{communication_method}= lc $modifiable_fields{communication_method};

	# see AUTOLOAD below for what _modifiable does- it's not meant for direct external use
	$this= {
		_modifiable => \%modifiable_fields,
		%modifiable_fields,
		%constant_fields,
	};

	bless $this, $class_name;
	return $this;
}

sub AUTOLOAD
{
	my $this= shift; #ref to the class
	my $class_name= ref($this);
	my $param= shift; 

	#must take a reference to this class
	croak "Invalid reference type\n" unless ($class_name =~ /^(?:OSInstall::)?Control_Host$/);

	my $attr= $AUTOLOAD; #the field we're autoloading
	$attr =~ s/.*://; #strip out full package name

	#error if we don't have this field
	croak "The field \"$attr\" does not exist\n" unless (exists $this->{$attr});

	# if we have a parameter to set the attr to, make sure it's a modifiable attr
	if ($param)
	{
		# croak if this field is not modifiable
		croak "This field cannot be modified\n" unless ($this->{_modifiable}->{$attr});

		$this->{$attr}= $param; #damage is done
	}

	return $this->{$attr};
}

sub netboot
{
	# shift args
	my $this= shift;
	my $client_ref= shift;

	# local data
	my $rc= 1;
	my $lpar_netboot_cmd= 'lpar_netboot [arg1] [arg11] -t ent -m [arg2] -D -s [arg3] -d [arg4] -S [arg5] -G [arg6] -C [arg7] "[arg8]" "[arg9]" "[arg10]"';
	my @data_params= qw(debug mac_addr speed duplex server_addr gateway client_addr lpar profile managed_sys boot_params);
	my $lpar_netboot_debug= $ENV{OSINSTALL_BOOT_DEBUG} ? '-v -x' : '';  # lpar_netboot debug mode (lots of extra output)
	my $mac_addr_param= $client_ref->mac_addr;
	my %data_hash= ();
	my $boot_params= $client_ref->boot_params ? "-g \"".$client_ref->boot_params."\"" : "";

	# commands
	my $SSH= find_command('ssh');


	return 0 unless (ref($client_ref) =~ /^(?:OSInstall::)Client$/);

	# strip out any :'s from the MAC address
	$mac_addr_param =~ s/://g;

	# make sure this client has p5 lpar attributes set
	unless ($client_ref->lpar && $client_ref->profile && $client_ref->managed_sys)
	{
		log_entry(__("The client %s does not appear to be a p5 LPAR and cannot be netbooted\n", $client_ref->name), 1);
		return 0;
	}

	# check for net adapter information
	unless ($client_ref->speed && $client_ref->duplex)
	{
		log_entry(__("The client %s cannot be netbooted because its network adapter setings have not been configured\n", $client_ref->name), 1);
		return 0;
	}

	# fill in the hash containing the arg data, then build the cmd-line string
	%data_hash= (
		debug => $lpar_netboot_debug,
		mac_addr => $mac_addr_param,
		speed=> $client_ref->speed,
		duplex => $client_ref->duplex,
		# FIXME: server_addr should be ip of resource server, not control host
		# setting value to hostname of machine that osinstall is running on
		server_addr => inet_ntoa(inet_aton(hostname)),
		gateway => $client_ref->gateway,
		client_addr => $client_ref->ip_addr,
		lpar => $client_ref->lpar,
		profile => $client_ref->profile,
		managed_sys => $client_ref->managed_sys,
		boot_params => $boot_params,
	);

	$lpar_netboot_cmd= fill_in_args($lpar_netboot_cmd, build_param_list(\%data_hash, \@data_params));

	unless ($this->host_name =~ /^localhost/)
	{
		# enclose everything in '' in case any wierd characters come in somehow (like through boot_params)
		$lpar_netboot_cmd= "'".$lpar_netboot_cmd."'";
		$lpar_netboot_cmd= "$SSH hscroot@".$this->host_name.' '.$lpar_netboot_cmd;
	}
	else
	{
		# if localhost is the Control_Host, then we are running on an HMC
		# lpar_netboot does not support IVM yet
		# enclose everything in '' in case any wierd characters come in somehow (like through boot_params)
		$lpar_netboot_cmd= "/opt/hsc/bin/".$lpar_netboot_cmd;
	}
	

	$rc= log_cmd($lpar_netboot_cmd);

	$rc;
}

1;

