package OSInstall::OS_Resource;

use strict;
use warnings;
use Carp;
use OSInstall::Common;
use OSInstall::Platform_Tools;
our $AUTOLOAD;

our $VERSION = '1.0';

my %modifiable_fields= (
		name => "",
		image_location => "",
		config_file => "",
		resource_server => "",
		clients_allocated => {}, # empty hash ref
		distribution => "",
                version => "",
		ready => "",
	);

my %constant_fields= (
		image_source => "",
		type => "",
	);

sub new
{
	my $class_name= shift;
	my $this; #empty % ref that we will bless into an object
	
	#the first six args are required
	confess "usage OSInstall::OS_Resource->new(name, image_source, type, version[, image_location, config_file, resource_server, distribution, clients_allocated])\n" unless (@_ >= 4);

	$modifiable_fields{name}= shift;
	$constant_fields{image_source}= shift;
	$constant_fields{type}= shift;
	$constant_fields{version}= shift;
	$modifiable_fields{image_location}= shift if (@_);
	$modifiable_fields{config_file}= shift if (@_);
	$modifiable_fields{resource_server}= shift if (@_);
	$modifiable_fields{distribution}= shift if (@_);
	$modifiable_fields{clients_allocated} = shift if (@_);

	$modifiable_fields{ready} = shift if (@_);
	
	# see AUTOLOAD below for what _modifiable does- it's not meant for direct external use
	$this= {
		_modifiable => \%modifiable_fields,
		_constant => \%constant_fields,
		%modifiable_fields,
		%constant_fields,
	};

	bless $this, $class_name;
	return $this;
}

sub AUTOLOAD
{
	my $this= shift; #ref to the class
	my $class_name= ref($this);
	my $param= shift; 

	#must take a reference to this class
	croak "Invalid reference type\n" unless ($class_name =~ /^(?:OSInstall::)?\w+_Resource$/);

	my $attr= $AUTOLOAD; #the field we're autoloading
	$attr =~ s/.*://; #strip out full package name

	#error if we don't have this field
	croak "The field \"$attr\" does not exist\n" unless (exists $this->{$attr});

	# if we have a parameter to set the attr to, make sure it's a modifiable attr
	if ($param)
	{
		# croak if this field is not modifiable
		croak "This field cannot be modified\n" unless (exists $this->{_modifiable}->{$attr});

		$this->{$attr}= $param; #damage is done
	}

	return $this->{$attr};
}

sub allocate
{
	# shift args
	my $this= shift;
	my $client_ref= shift;
	my $config_file= shift;

	$client_ref->config_file($config_file) if ($config_file);

	# local data
	my $rc= 1;

	# make sure our argument is a reference to the right type
	unless (ref($client_ref) =~ /^(?:OSInstall::)?Client$/)
	{
		log_entry(__("Invalid reference type: %s  Expected: %s\n", ref($client_ref), 'OSInstall::Client'), 1);
		$this->deallocate($client_ref);
		return 0;
	}

	unless (OSInstall::Platform_Tools::flush_firewall_rules($client_ref->hostname))
	{
		log_entry(__("Could not flush firewall rules for client %s\n", $client_ref->name));
	}

	# start the daemons if they are not already active
	unless (OSInstall::Platform_Tools::start_daemons())
	{
		log_entry(__("Failed to start network installation services\n"), 1);
		$this->deallocate($client_ref);
		return 0;
	}

	# configure the install server to reply to the BOOTP request from the client
	unless (OSInstall::Platform_Tools::config_bootp($this, $client_ref))
	{
		log_entry(__("Failed to configure the %s service\n", 'bootp'), 1);
		$this->deallocate($client_ref);
		return 0;
	}

	# add the client to the list of allocated clients, if it's not there already
	$this->clients_allocated->{$client_ref->{name}}= $client_ref;
	
	# NFS-export the OS_Resource's directory to the client
	unless (OSInstall::Platform_Tools::export_resource($this, $client_ref, values %{$this->clients_allocated}))
	{
		log_entry(__("Failed to configure the %s service\n", 'nfs'), 1);
		$this->deallocate($client_ref);
		return 0;
	}

	# create the boot images for this client
	unless ($this->copy_bootfiles($client_ref))
	{
		log_entry(__("Failed to copy boot images for client %s\n", $client_ref->name), 1);
		$this->deallocate($client_ref);
		return 0;
	}

	# refresh the daemons so they can pickup their new config
	unless (OSInstall::Platform_Tools::refresh_daemons())
	{
		log_entry(__("Failed to refresh network installation services"), 1);
		$this->deallocate($client_ref);
		return 0;
	}

	$client_ref->allocation($this) if ($rc);

	$rc;
}

sub deallocate
{
	# shift args
	my $this= shift;
	my $client_ref= shift;
	my %tmp_clients_allocated= ();

	#local data
	my $rc= 1;

	# make sure our argument is a reference to the right type
	unless (ref($client_ref) =~ /^(?:OSInstall::)?Client$/)
	{
		log_entry(__("Invalid reference type: %s  Expected: %s\n", ref($client_ref), 'OSInstall::Client'), 1);
		return 0;
	}

	# remove the client's BOOTP configuration from the install server's BOOTP config
	OSInstall::Platform_Tools::unconfig_bootp($this, $client_ref) or $rc= 0;

	# remove the client from the list of allocated clients
#	%tmp_clients_allocated= %{$this->clients_allocated};
#	delete $tmp_clients_allocated{$client_ref->name};
#	$this->clients_allocated(\%tmp_clients_allocated);
	
	# unexport the directory containing the resources from the client
	OSInstall::Platform_Tools::unexport_resource($this, $client_ref, values %{$this->clients_allocated}) or $rc= 0;

	# remove the boot images copied for this client
	$this->rm_bootfiles($client_ref) or $rc= 0;

	# refresh the daemons so they can pickup their new config
	OSInstall::Platform_Tools::refresh_daemons() or $rc= 0;

	# restore the firewall rules for this client (if we're on an HMC)
	OSInstall::Platform_Tools::enforce_firewall_rules($client_ref->hostname) or $rc= 0;

	# clear out client config file
	$client_ref->config_file("");

	$rc;
}

#########################################################################
# The functions below will be overridden by the implementing subclasses	#
# but will be empty here (kind of like pure virtual functions)		#
#########################################################################

sub copy_bootfiles
{
	1;
}

sub rm_bootfiles
{
	1;
}

sub copy_images
{
	1;
}

sub rm_images
{
	1;
}

sub get_install_status
{
	1;
}

1;

